<?php
/**
 * Common class.
 *
 * @since 1.0.0
 *
 * @package Envira_Dynamic
 * @author  Envira Team
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Common class.
 *
 * @since 1.0.0
 *
 * @package Envira_Dynamic
 * @author  Envira Team
 */
class Envira_Dynamic_Metaboxes {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * Path to the file.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $file = __FILE__;

	/**
	 * Holds the base class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public $base;

	/**
	 * Holds the Envira Gallery Dynamic ID.
	 *
	 * @since 1.0.0
	 *
	 * @var int
	 */
	public $gallery_dynamic_id;

	/**
	 * Holds the Envira Album Dynamic ID.
	 *
	 * @since 1.0.0
	 *
	 * @var int
	 */
	public $album_dynamic_id;

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Load the base class object.
		$this->base = Envira_Dynamic::get_instance();

		// Get Envira Gallery and Albums Dynamic ID.
		$this->gallery_dynamic_id = get_option( 'envira_dynamic_gallery' );
		$this->album_dynamic_id   = get_option( 'envira_dynamic_album' );

		// Hide Slug Box.
		add_filter( 'envira_gallery_metabox_styles', array( $this, 'maybe_hide_slug_box' ) );
		add_filter( 'envira_album_metabox_styles', array( $this, 'maybe_hide_slug_box' ) );

		// Actions and Filters: Galleries.
		add_filter( 'envira_gallery_types', array( $this, 'add_dynamic_type' ), 9999, 2 );
		add_action( 'envira_gallery_display_dynamic', array( $this, 'images_display' ) );
		add_action( 'envira_gallery_include_justified_config_box', array( $this, 'config_box' ), 10, 2 );
		add_filter( 'envira_gallery_save_settings', array( $this, 'gallery_settings_save' ), 10, 2 );

		// Actions and Filters: Albums.
		add_filter( 'envira_albums_types', array( $this, 'add_dynamic_type' ), 9999, 2 );
		add_action( 'envira_albums_display_dynamic', array( $this, 'images_display' ) );
		add_filter( 'envira_albums_save_settings', array( $this, 'albums_settings_save' ), 10, 2 );

	}

	/**
	 * Removes the slug metabox if we are on a Dynamic Gallery or Album
	 *
	 * @since 1.0.0
	 */
	public function maybe_hide_slug_box() {

		if ( ! isset( $_GET['post'] ) ) { // @codingStandardsIgnoreLine - add nonce?
			return;
		}

		// Check if we are viewing a Dynamic Gallery or Album.
		if ( $_GET['post'] !== $this->gallery_dynamic_id && $_GET['post'] !== $this->album_dynamic_id ) { // @codingStandardsIgnoreLine - add nonce?
			return;
		}

		?>
		<style type="text/css"> #edit-slug-box, #post-preview { display: none; } #delete-action, .misc-pub-section.misc-pub-post-status, .misc-pub-section.misc-pub-visibility, .misc-pub-section.misc-pub-curtime { display: none !important;} </style>
		<?php

	}

	/**
	 * Changes the available Gallery Type to Dynamic if the user is editing
	 * the Envira Dynamic Post
	 *
	 * @since 1.0.0
	 *
	 * @param array   $types Gallery Types.
	 * @param WP_Post $post WordPress Post.
	 * @return array Gallery Types
	 */
	public function add_dynamic_type( $types, $post ) {

		// Check Post = Dynamic.
		switch ( get_post_type( $post ) ) {
			case 'envira':
				if ( $post->ID !== $this->gallery_dynamic_id ) {
					return $types;
				}
				break;
			case 'envira_album':
				if ( $post->ID !== $this->album_dynamic_id ) {
					return $types;
				}
				break;
			default:
				// Not an Envira CPT.
				return $types;
		}

		// Change Types = Dynamic only.
		$types = array(
			'dynamic' => __( 'Dynamic', 'envira-dynamic' ),
		);

		return $types;

	}

	/**
	 * Display output for the Images Tab
	 *
	 * @since 1.0.0
	 * @param WP_Post $post WordPress Post.
	 */
	public function images_display( $post ) {

		?>
		<div id="envira-dynamic">
			<p class="envira-intro">
				<?php
				switch ( get_post_type( $post ) ) {
					case 'envira':
						esc_html_e( 'Dynamic Gallery Settings', 'envira-dynamic' );
						break;
					case 'envira_album':
						esc_html_e( 'Dynamic Album Settings', 'envira-dynamic' );
						break;
				}
				?>
			</p>
			<p>
				<?php
				switch ( get_post_type( $post ) ) {
					case 'envira':
						esc_html_e( 'This gallery and its settings will be used as defaults for any dynamic gallery you create on this site. Any of these settings can be overwritten on an individual gallery basis via template tag arguments or shortcode parameters.', 'envira-dynamic' );
						break;
					case 'envira_album':
						esc_html_e( 'This album and its settings will be used as defaults for any dynamic album you create on this site. Any of these settings can be overwritten on an individual album basis via template tag arguments or shortcode parameters.', 'envira-dynamic' );
						break;
				}
				?>
			</p>

			<div class="envira-video-help">
				<iframe src="https://www.youtube.com/embed/Gu0cwpGyEuA?rel=0" width="600" height="338" frameborder="0" allowfullscreen></iframe>
			</div>

			<p>
				<a href="http://enviragallery.com/docs/dynamic-addon/" title="Click here for Dynamic Addon documentation." target="_blank" class="button button-primary envira-button-primary">
					<?php esc_html_e( 'Click here for Dynamic Addon Documentation', 'envira-dynamic' ); ?>
				</a>
			</p>
		</div>
		<?php

	}

	/**
	 * Adds options to the Config tab of the Dynamic Gallery
	 *
	 * @since 1.1.6
	 *
	 * @param   WP_Post $post   Gallery Post.
	 * @param   WP_Post $gallery_data   Gallery Data.
	 */
	public function config_box( $post, $gallery_data ) {

		if ( Envira_Dynamic_Common::get_instance()->get_gallery_dynamic_id() !== $post->ID ) {
			return;
		}
		$data = envira_get_gallery( $post->ID );

		?>
		<tr id="envira-config-dynamic-gallery-override-box">
			<th scope="row">
				<label for="envira-config-dynamic-gallery-override"><?php esc_html_e( 'Render all WordPress Galleries using Envira?', 'envira-dynamic' ); ?></label>
			</th>
			<td>
				<input id="envira-config-dynamic-gallery-override" type="checkbox" name="_envira_gallery[native_gallery_override]" value="1" <?php checked( envira_get_config( 'native_gallery_override', $data, envira_get_config_default( 'native_gallery_override' ) ), 1 ); ?> />
				<span class="description"><?php esc_html_e( 'If enabled, every WordPress Gallery which uses the [gallery] shortcode will be rendered using Envira Gallery.', 'envira-dynamic' ); ?></span>
			</td>
		</tr>

		<!-- Sorting -->
		<tr id="envira-config-sorting-box">
				<th scope="row">
					<label for="envira-config-sorting"><?php esc_html_e( 'Sorting', 'envira-gallery' ); ?></label>
				</th>
				<td>
					<select id="envira-config-sorting" name="_envira_gallery[random]">
						<?php
						foreach ( (array) envira_get_sorting_options() as $i => $data ) {
							?>
							<option value="<?php echo esc_html( $data['value'] ); ?>"<?php selected( $data['value'], envira_get_config( 'random', $gallery_data, envira_get_config_default( 'random' ) ) ); ?>><?php echo esc_html( $data['name'] ); ?></option>
							<?php
						}
						?>
					</select>
				<p class="description"><?php esc_html_e( 'Choose to sort the images in a different order than displayed on the Images tab.', 'envira-gallery' ); ?></p>
				</td>
			</tr>
<tr id="envira-config-sorting-direction-box">
				<th scope="row">
				<label for="envira-config-sorting-direction"><?php esc_html_e( 'Direction', 'envira-gallery' ); ?></label>
				</th>
				<td>
					<select id="envira-config-sorting-direction" name="_envira_gallery[sorting_direction]">
						<?php
						foreach ( (array) envira_get_sorting_directions() as $i => $data ) {
							?>
								<option value="<?php echo esc_html( $data['value'] ); ?>"<?php selected( $data['value'], envira_get_config( 'sorting_direction', $gallery_data ) ); ?>><?php echo esc_html( $data['name'] ); ?></option>
								<?php
						}
						?>
					</select>
				</td>
			</tr>

		<?php // } ?>

		<?php

	}

	/**
	 * Save album settings.
	 *
	 * @since 1.1.6
	 *
	 * @param   array $settings  Settings.
	 * @param   int   $post_id   Post ID.
	 */
	public function albums_settings_save( $settings, $post_id ) {

		if ( ! empty( $this->album_default_id ) && $post_id === $this->album_default_id ) {
			$settings['config']['type'] = 'dynamic';

		}

		return $settings;

	}

	/**
	 * Save gallery settings.
	 *
	 * @since 1.1.6
	 *
	 * @param   array $settings  Settings.
	 * @param   int   $post_id   Post ID.
	 */
	public function gallery_settings_save( $settings, $post_id ) {

		// Settings.
		$settings['config']['native_gallery_override'] = ( isset( $_POST['_envira_gallery']['native_gallery_override'] ) ? 1 : 0 ); // @codingStandardsIgnoreLine - add nonce?

		if ( $post_id === $this->gallery_dynamic_id ) {
			$settings['config']['type'] = 'dynamic';
		}

		return $settings;

	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Envira_Dynamic_Metaboxes object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Envira_Dynamic_Metaboxes ) ) {
			self::$instance = new Envira_Dynamic_Metaboxes();
		}

		return self::$instance;

	}

}

// Load the metaboxes class.
$envira_dynamic_metaboxes = Envira_Dynamic_Metaboxes::get_instance();
