<?php
/**
 * Handles all admin ajax interactions for the Envira Albums plugin.
 *
 * @since 1.0.0
 *
 * @package Envira_Nextgen_Importer
 * @author  Envira Team
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Imports a gallery from NextGEN into Envira
 *
 * @since 1.0.0
 * @param int    $gallery_id Gallery ID.
 * @param string $return_json JSON.
 */
function envira_nextgen_importer_import_gallery( $gallery_id = '', $return_json = true ) {

	global $wpdb;

	// Set max execution time so we don't timeout.
	ini_set( 'max_execution_time', 0 ); // @codingStandardsIgnoreLine
	set_time_limit( 0 );

	// If no gallery ID, get from AJAX request.
	if ( empty( $gallery_id ) ) {

		// Run a security check first.
		check_ajax_referer( 'envira-nextgen-importer', 'nonce' );

		// Define ABSPATH.
		if ( ! defined( 'ABSPATH' ) ) {
			define( 'ABSPATH', dirname( __FILE__ ) . '/' );
		}

		// Check variables exist.
		if ( ! isset( $_POST['id'] ) ) {
			return_result( false, __( 'No gallery was selected', 'envira-nextgen-importer' ) );
		}

		// Prepare variables.
		$gallery_id = absint( $_POST['id'] );

	}

	// Get Envira Common base for default config settings.
	$instance = Envira_Gallery_Common::get_instance();

	// Get image path.
	$gallery = $wpdb->get_row( // @codingStandardsIgnoreLine
		$wpdb->prepare(
			'SELECT path, title, galdesc, pageid
                            FROM ' . $wpdb->prefix . 'ngg_gallery
                            WHERE gid = %d
                            LIMIT 1',
			$gallery_id
		)
	);

	// Get images, in order, from this NextGEN Gallery.
	$images = $wpdb->get_results( // @codingStandardsIgnoreLine
		$wpdb->prepare(
			'SELECT * FROM ' . $wpdb->prefix . 'ngg_pictures
                            WHERE galleryid = %d
                            ORDER BY sortorder ASC,
                            imagedate ASC',
			$gallery_id
		)
	);

	// Check gallery has images.
	$attachments = array();
	if ( is_array( $images ) && count( $images ) > 0 ) {
		// Add each image to Media Library.
		foreach ( $images as $image ) {
			// Store image in WordPress Media Library.
			$attachment = envira_nextgen_importer_add_image_to_library( $gallery->path, $image->filename, $image->description, $image->alttext );

			if ( false !== $attachment ) {
				// Import tags.
				$tags = wp_get_object_terms( $image->pid, 'ngg_tag', 'fields=names' );

				// Add to new attachment.
				wp_set_object_terms( $attachment['ID'], $tags, 'envira-tag' );

				// Add to array of attachments.
				$attachments[] = $attachment;
			}
		}
	}

	// If image(s) were added to the Media Library, create a new Envira CPT containing these images.
	if ( count( $attachments ) === 0 ) {
		return_result( false, __( 'No images found in gallery. Skipping...', 'envira-nextgen-importer' ) );
	}

	// Get attachment data.
	$attachment_data = wp_get_attachment_metadata( $attachment['ID'] );

	// Build Envira Gallery metadata.
	$meta            = array();
	$meta['gallery'] = array();
	foreach ( $attachments as $attachment ) {
		$meta['gallery'][ $attachment['ID'] ] = array(
			'status'  => 'active',
			'src'     => $attachment['src'],
			'title'   => $attachment['title'],
			'link'    => $attachment['src'],
			'width'   => $attachment_data['width'],
			'height'  => $attachment_data['height'],
			'alt'     => $attachment['alt'],
			'caption' => $attachment['caption'],
			'thumb'   => '',
		);

		if ( defined( 'ENVIRA_DEBUG' ) && ENVIRA_DEBUG ) {
			error_log( 'attachment meta' ); // @codingStandardsIgnoreLine
			error_log( print_r( $meta['gallery'][ $attachment['ID'] ], true ) ); // @codingStandardsIgnoreLine
		}
	}

	// Build Envira Config metadata.
	$meta['config'] = array();

	// Grab defaults.
	$config_defaults = envira_get_config_defaults( null );
	$config_keys     = array_keys( $config_defaults );
	foreach ( $config_keys as $key ) {
		$meta['config'][ $key ] = $config_defaults[ $key ];
	}

	// Create Envira CPT.
	$envira_gallery_id = wp_insert_post(
		array(
			'post_type'   => 'envira',
			'post_status' => 'publish',
			'post_title'  => $gallery->title,
		)
	);

	// Assign the meta key/value for the NextGen Gallery.
	add_post_meta( $envira_gallery_id, 'envira_ng_gallery_id', $gallery_id, true );

	// Read Envira CPT so we can set title and slug metadata.
	$envira_gallery = get_post( $envira_gallery_id );

	// Manually set post id, title and slug.
	$meta['id']              = $envira_gallery_id;
	$meta['config']['title'] = trim( wp_strip_all_tags( $envira_gallery->post_title ) );
	$meta['config']['slug']  = sanitize_text_field( $envira_gallery->post_name );

	// Attach meta to Envira CPT.
	update_post_meta( $envira_gallery_id, '_eg_gallery_data', $meta );

	// Mark this NextGEN gallery as imported in option data.
	$settings = get_option( 'envira_nextgen_importer' );
	if ( ! isset( $settings['galleries'] ) ) {
		$settings['galleries'] = array();
	}

	// Store NextGEN Gallery ID => Envira Gallery ID in options data, so we can show user in the Import UI
	// that this gallery has been imported.
	$settings['galleries'][ $gallery_id ] = $envira_gallery_id;
	update_option( 'envira_nextgen_importer', $settings );

	// NextGEN and Envira Shortcodes.
	$next_gen_shortcode = '[nggallery id=' . $gallery_id . ']';
	$envira_shortcode   = '[envira-gallery id="' . $envira_gallery_id . '"]';

	// Raw query to replace NextGEN shortcode with Envira Shortcode in Posts, Pages + CPTs.
	$wpdb->query( // @codingStandardsIgnoreLine
		$wpdb->prepare(
			'UPDATE ' . $wpdb->prefix . 'posts SET post_content = REPLACE(post_content, %s, %s)',
			"'" . $next_gen_shortcode . "'",
			"'" . $envira_shortcode . "'"
		)
	);

	// Send back the response.
	if ( $return_json ) {
		// Return JSON for AJAX request.
		return_result( true, __( 'Imported!', 'envira-nextgen-importer' ) );
	} else {
		// Return Envira Gallery ID for envira_nextgen_importer_import_album().
		return $envira_gallery_id;
	}
}
add_action( 'wp_ajax_envira_nextgen_importer_import_gallery', 'envira_nextgen_importer_import_gallery' );

/**
 * Imports an album from NextGEN into Envira
 *
 * @since 1.0.0
 */
function envira_nextgen_importer_import_album() {

	global $wpdb;

	// Set max execution time so we don't timeout.
	ini_set( 'max_execution_time', 0 ); // @codingStandardsIgnoreLine
	set_time_limit( 0 );

	// Run a security check first.
	check_ajax_referer( 'envira-nextgen-importer', 'nonce' );

	// Define ABSPATH.
	if ( ! defined( 'ABSPATH' ) ) {
		define( 'ABSPATH', dirname( __FILE__ ) . '/' );
	}

	// Get Envira Common base for default config settings.
	$instance = Envira_Albums_Common::get_instance();

	// Check variables exist.
	if ( ! isset( $_POST['id'] ) ) {
		return_result( false, __( 'No album was selected', 'envira-nextgen-importer' ) );
	}

	// Prepare variables..
	$album_id = absint( $_POST['id'] );

	// Get album name.
	$album            = $wpdb->get_row( // @codingStandardsIgnoreLine
		$wpdb->prepare(
			'SELECT name, sortorder, albumdesc
						FROM ' . $wpdb->prefix . 'ngg_album
						WHERE id = %d
						LIMIT 1',
			$album_id
		)
	);
	$album->galleries = nextgen_unserialize( $album->sortorder );

	// Check album has galleries.
	if ( ! is_array( $album->galleries ) ) {
		return_result( false, __( 'No galleries found in album. Skipping...', 'envira-nextgen-importer' ) );
	}

	// Build Envira Album metadata.
	$meta                = array();
	$meta['gallery_ids'] = array();
	$meta['galleries']   = array();

	// Get settings.
	$settings = get_option( 'envira_nextgen_importer' );

	// Iterate through galleries.
	foreach ( $album->galleries as $gallery_id ) {

		// got the Envira ID from the $gallery_id, which is the NextGen gallery ID and not Envira's.
		$args  = array(
			'post_type'  => 'envira',
			'meta_query' => array( // @codingStandardsIgnoreLine
				array(
					'key'     => 'envira_ng_gallery_id',
					'value'   => $gallery_id,
					'compare' => '=',
				),
			),
		);
		$query = new WP_Query( $args );

		if ( empty( $query->post->ID ) ) {
			continue;
		}

		$envira_gallery_id = $query->post->ID;

		// Check if this NextGEN gallery has already been imported into Envira and still exists as a published Envira CPT
		// If so, we don't need to import it again.
		if ( isset( $settings['galleries'] ) && array_key_exists( $envira_gallery_id, $settings['galleries'] ) ) {
			// Gallery imported
			// Get existing Envira Gallery.
			$envira_galleries = new WP_Query(
				array(
					'post_type'      => 'envira',
					'post_status'    => 'publish',
					'p'              => $settings['galleries'][ $envira_gallery_id ], // value of array key = Envira Gallery ID.
					'posts_per_page' => 1,
				)
			);
		} else {
			// Gallery not imported
			// Import.
			$result = envira_nextgen_importer_import_gallery( $gallery_id, false );

			// Get existing Envira Gallery.
			$envira_galleries = new WP_Query(
				array(
					'post_type'      => 'envira',
					'post_status'    => 'publish',
					'p'              => $result,
					'posts_per_page' => 1,
				)
			);
		}

		// Check Envira Gallery Exists.
		if ( $envira_galleries->have_posts() ) {
			// Get Envira Gallery and Meta.
			$envira_gallery      = $envira_galleries->posts[0];
			$envira_gallery_meta = get_post_meta( $envira_gallery->ID, '_eg_gallery_data', true );

			// Get cover image from first gallery image.
			reset( $envira_gallery_meta['gallery'] );
			$image_id = key( $envira_gallery_meta['gallery'] );

			// Gallery exists in Envira - no need to import
			// Just add to metadata.
			$meta['galleryIDs'][]                     = $envira_gallery->ID;
			$meta['galleries'][ $envira_gallery->ID ] = array(
				'title'          => $envira_gallery->post_title,
				'alt'            => $envira_gallery->post_title,
				'cover_image_id' => $image_id,
			);
		}
	}

	// All galleries imported
	// Build Envira Album Config metadata.
	$meta['config'] = array();
	$keys           = array(
		'columns',
		'album_theme',
		'gutter',
		'margin',
		'classes',
		'rtl',
	);
	foreach ( $keys as $key ) {
		$meta['config'][ $key ] = $instance->get_config_default( $key );
	}

	// Set description.
	$meta['config']['description'] = $album->albumdesc;

	// Create Envira Album CPT.
	$envira_album_id = wp_insert_post(
		array(
			'post_type'   => 'envira_album',
			'post_status' => 'publish',
			'post_title'  => $album->name,
		)
	);

	// Read Envira Album CPT so we can set title and slug metadata.
	$envira_album = get_post( $envira_album_id );

	// Manually set post id, title and slug.
	$meta['id']              = $envira_album_id;
	$meta['config']['title'] = trim( wp_strip_all_tags( $envira_album->post_title ) );
	$meta['config']['slug']  = sanitize_text_field( $envira_album->post_name );

	// Attach meta to Envira Album CPT.
	update_post_meta( $envira_album_id, '_eg_album_data', $meta );

	// Mark this NextGEN album as imported in option data.
	$settings = get_option( 'envira_nextgen_importer' );
	if ( ! isset( $settings['albums'] ) ) {
		$settings['albums'] = array();
	}

	// Store NextGEN Gallery ID => Envira Gallery ID in options data, so we can show user in the Import UI
	// that this gallery has been imported.
	$settings['albums'][ $album_id ] = $envira_album_id;
	update_option( 'envira_nextgen_importer', $settings );

	// Send back the response.
	return_result( true, __( 'Imported!', 'envira-nextgen-importer' ) );

}
add_action( 'wp_ajax_envira_nextgen_importer_import_album', 'envira_nextgen_importer_import_album' );

/**
 * Unserialize NextGEN data
 *
 * @param string $value  Serialized Data.
 * @return array         Unserialized Data
 */
function nextgen_unserialize( $value ) {

	$retval = null;

	if ( is_string( $value ) ) {
		$retval = stripcslashes( $value );

		if ( strlen( $value ) > 1 ) {
			// We can't always rely on base64_decode() or json_decode() to return FALSE as their documentation
			// claims so check if $retval begins with a: as that indicates we have a serialized PHP object.
			if ( strpos( $retval, 'a:' ) === 0 ) {
				$er     = error_reporting( 0 ); // @codingStandardsIgnoreLine
				$retval = unserialize( $value ); // @codingStandardsIgnoreLine - JSON?
				error_reporting( $er ); // @codingStandardsIgnoreLine
			} else {
				// We use json_decode() here because PHP's unserialize() is not Unicode safe.
				$retval = json_decode( base64_decode( $retval ), true );
			}
		}
	}

	return $retval;

}

/**
 * Adds a server side image to the WordPress Media Library
 *
 * @param string $source_path Source Path (e.g. /wp-content/gallery/gallery-name) - no trailing slash.
 * @param string $source_file Source Image (e.g. my-image.jpg).
 * @param string $description Image Description.
 * @param string $alt Image Alt Text.
 * @return mixed Image ID | false
 *
 * @since 1.0.0
 */
function envira_nextgen_importer_add_image_to_library( $source_path, $source_file, $description, $alt ) {

	// Get full path and filename.
	$source_file_and_path = ABSPATH . $source_path . '/' . $source_file;

	// Get WP upload dir.
	$upload_dir = wp_upload_dir();

	// Generate a unique filename so we don't overwrite an existing WordPress Media Library image
	// Create our destination file paths and URLs.
	$destination_file          = wp_unique_filename( $upload_dir['path'], $source_file );
	$destination_file_and_path = $upload_dir['path'] . '/' . $destination_file;
	$destination_url           = $upload_dir['url'] . '/' . $destination_file;

	// Check file is valid.
	$wp_filetype = wp_check_filetype( $source_file, null );
	extract( $wp_filetype ); // @codingStandardsIgnoreLine - ????
	if ( ( ! $type || ! $ext ) && ! current_user_can( 'unfiltered_upload' ) ) {
		// Invalid file type - skip.
		return false;
	}

	// Copy the file to the WordPress uploads dir.
	$result = copy( $source_file_and_path, $destination_file_and_path );
	if ( ! $result ) {
		// Could not copy image.
		return false;
	}

	// Set correct file permissions, as NextGEN can set these wrong.
	$stat  = stat( $destination_file_and_path );
	$perms = $stat['mode'] & 0000666;
	chmod( $destination_file_and_path, $perms );

	// Apply upload filters.
	$return = apply_filters(
		'wp_handle_upload',
		array(
			'file' => $destination_file_and_path,
			'url'  => $destination_url,
			'type' => $type,
		)
	);

	// Construct the attachment array.
	$attachment = array(
		'post_mime_type' => $type,
		'guid'           => $destination_url,
		'post_title'     => $alt,
		'post_name'      => $alt,
		'post_content'   => $description,
	);

	// Save as attachment.
	$attachment_id = wp_insert_attachment( $attachment, $destination_file_and_path );

	// Update attachment metadata.
	if ( ! is_wp_error( $attachment_id ) ) {
		$metadata = wp_generate_attachment_metadata( $attachment_id, $destination_file_and_path );
		wp_update_attachment_metadata( $attachment_id, wp_generate_attachment_metadata( $attachment_id, $destination_file_and_path ) );
	}

	// Force alt and caption.
	update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt );
	$attachment               = get_post( $attachment_id );
	$attachment->post_excerpt = $description;
	wp_update_post( $attachment );

	// Return attachment data.
	return array(
		'ID'      => $attachment_id,
		'src'     => $destination_url,
		'title'   => $alt,
		'alt'     => $alt,
		'caption' => $description,
	);

}


/**
 * Returns a JSON encoded success or error flag with message string
 *
 * @param bool   $success Success (true|false).
 * @param string $message Message.
 *
 * @since 1.0.0
 */
function return_result( $success, $message ) {
	echo wp_json_encode(
		array(
			'success' => (bool) $success,
			'message' => (string) $message,
		)
	);
	die;
}
