<?php
/**
 * Metabox class.
 *
 * @since 1.0.0
 *
 * @package Envira_WooCommerce
 * @author  Envira Team
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Metabox class.
 *
 * @since 1.0.0
 *
 * @package Envira_WooCommerce
 * @author  Envira Team
 */
class Envira_WooCommerce_Metaboxes {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * Path to the file.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $file = __FILE__;

	/**
	 * Holds the base class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public $base;

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Load the base class object.
		$this->base = Envira_WooCommerce::get_instance();

		// Gallery.
		add_action( 'envira_gallery_metabox_scripts', array( $this, 'scripts' ) );

		add_filter( 'envira_gallery_tab_nav', array( $this, 'register_tabs' ) );
		add_action( 'envira_gallery_tab_woocommerce', array( $this, 'woocommerce_tab' ) );
		add_filter( 'envira_gallery_save_settings', array( $this, 'save_gallery' ), 10, 2 );
		add_action( 'envira_gallery_mobile_box', array( $this, 'mobile_screen' ) );

		// Gallery: Individual Image Settings.
		add_action( 'print_media_templates', array( $this, 'meta_settings' ), 10, 3 );

		// Album.
		add_action( 'envira_gallery_mobile_box', array( $this, 'mobile_screen' ) );

	}

	/**
	 * Enqueues the Media Editor script, which is used when editing a gallery image
	 * This outputs the WooCommerce settings for each individual image
	 *
	 * @since 1.0.4
	 */
	public function scripts() {

		wp_enqueue_script( $this->base->plugin_slug . '-media-edit', plugins_url( 'assets/js/media-edit.js', $this->base->file ), array( 'jquery' ), $this->base->version, true );
		wp_localize_script(
			$this->base->plugin_slug . '-media-edit',
			'envira_woocommerce',
			array(
				'get_product_attributes_nonce' => wp_create_nonce( 'envira-woocommerce-get-product-attributes' ),
			)
		);
	}

	/**
	 * Registers tab(s) for this Addon in the Settings screen
	 *
	 * @since 1.0.0
	 *
	 * @param   array $tabs   Tabs.
	 * @return  array           Tabs
	 */
	public function register_tabs( $tabs ) {

		$tabs['woocommerce'] = __( 'WooCommerce', 'envira-woocommerce' );
		return $tabs;

	}

	/**
	 * Adds addon settings UI to the WooCommerce tab
	 *
	 * @since 1.0.0
	 *
	 * @param object $post The current post object.
	 */
	public function woocommerce_tab( $post ) {

		wp_nonce_field( 'envira_woocommerce_save_settings', 'envira_woocommerce_nonce' );

		// Get post type so we load the correct metabox instance and define the input field names
		// Input field names vary depending on whether we are editing a Gallery or Album.
		$post_type = get_post_type( $post );
		switch ( $post_type ) {
			/**
			* Gallery
			*/
			case 'envira':
				$instance = Envira_Gallery_Metaboxes::get_instance();
				$key      = '_envira_gallery';
				break;

			/**
			* Album
			*/
			case 'envira_album':
				$instance = Envira_Albums_Metaboxes::get_instance();
				$key      = '_eg_album_data[config]';
				break;
		}
		if ( ! class_exists( 'WooCommerce' ) ) : ?>
				<div class="envira-external-req">
					<h2><?php esc_html_e( 'Missing WooCommerce Plugin', 'envira-woocommerce' ); ?></h2>
					<p><?php esc_html_e( 'In order to use WooCommerce in galleries, you must first enable and setup WooCommerce.', 'envira-woocommerce' ); ?></p>
					<p><a href="<?php echo esc_html( admin_url( 'plugin-install.php?tab=search&s=woocommerce' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Click here to download WooCommerce', 'envira-woocommerce' ); ?></a></p>
				</div>
			<?php
			else :
				// Gallery options only apply to Galleries, not Albums.
				if ( 'envira' === $post_type ) {
					?>
			<p class="envira-intro">
						<?php esc_html_e( 'WooCommerce Gallery Options.', 'envira-woocommerce' ); ?>
				<small>
					<?php esc_html_e( 'The settings below adjust the WooCommerce options for the gallery.', 'envira-woocommerce' ); ?>
					<br/>
					<?php esc_html_e( 'Need some help?', 'envira-woocommerce' ); ?>
					<a href="http://enviragallery.com/docs/woocommerce-addon/" class="envira-doc" target="_blank">
					<?php esc_html_e( 'Read the Documentation', 'envira-woocommerce' ); ?>
					</a>
					or
					<a href="https://www.youtube.com/embed/H5zAzLUJZTg/?rel=0" class="envira-video" target="_blank">
					<?php esc_html_e( 'Watch a Video', 'envira-woocommerce' ); ?>
					</a>
				</small>
			</p>
			<table class="form-table">
				<tbody>
					<tr id="envira-config-woocommerce-box">
						<th scope="row">
							<label for="envira-config-woocommerce"><?php esc_html_e( 'Enable WooCommerce?', 'envira-woocommerce' ); ?></label>
						</th>
						<td>
							<input id="envira-config-woocommerce" type="checkbox" name="<?php echo esc_html( $key ); ?>[woocommerce]" value="1"<?php checked( $instance->get_config( 'woocommerce', $instance->get_config_default( 'woocommerce' ) ), 1 ); ?> />
							<span class="description"><?php esc_html_e( 'Enables WooCommerce Add to Cart functionality for each image in the gallery grid, if the image is assigned to a WooCommerce Product.', 'envira-woocommerce' ); ?></span>
						</td>
					</tr>
				</tbody>
			</table>
					<?php
				}

				// Lightbox Options.
				?>
		<p class="envira-intro">
				<?php esc_html_e( 'WooCommerce Lightbox Options.', 'envira-woocommerce' ); ?>
		</p>
		<table class="form-table">
			<tbody>
				<tr id="envira-config-woocommerce-box">
					<th scope="row">
						<label for="envira-config-lightbox-woocommerce"><?php esc_html_e( 'Enable WooCommerce?', 'envira-woocommerce' ); ?></label>
					</th>
					<td>
						<input id="envira-config-lightbox-woocommerce" type="checkbox" name="<?php echo esc_html( $key ); ?>[lightbox_woocommerce]" value="1"<?php checked( $instance->get_config( 'lightbox_woocommerce', $instance->get_config_default( 'lightbox_woocommerce' ) ), 1 ); ?> />
						<span class="description"><?php esc_html_e( 'Enables WooCommerce Add to Cart functionality for each image in the Lightbox view, if the image is assigned to a WooCommerce Product.', 'envira-woocommerce' ); ?></span>
					</td>
				</tr>
			</tbody>
		</table>
				<?php
		endif;
	}

	/**
	 * Saves the addon's settings for Galleries.
	 *
	 * @since 1.0.0
	 *
	 * @param array $settings  Array of settings to be saved.
	 * @param int   $post_id     The current post ID.
	 * @return array $settings Amended array of settings to be saved.
	 */
	public function save_gallery( $settings, $post_id ) {

		if (
			! isset( $_POST['_envira_gallery'], $_POST['envira_woocommerce_nonce'] )
			|| ! wp_verify_nonce( sanitize_key( $_POST['envira_woocommerce_nonce'] ), 'envira_woocommerce_save_settings' )
		) {
			return $settings;
		}

		$settings['config']['woocommerce']                  = ( isset( $_POST['_envira_gallery']['woocommerce'] ) ? 1 : 0 );
		$settings['config']['lightbox_woocommerce']         = ( isset( $_POST['_envira_gallery']['lightbox_woocommerce'] ) ? 1 : 0 );
		$settings['config']['mobile_disable_woo_galleries'] = ( isset( $_POST['_envira_gallery']['mobile_disable_woo_galleries'] ) ? 1 : 0 );

		return $settings;

	}

	/**
	 * Saves the addon's settings for Albums.
	 *
	 * @since 1.0.9
	 *
	 * @param array $settings  Array of settings to be saved.
	 * @param int   $post_id     The current post ID.
	 * @return array $settings Amended array of settings to be saved.
	 */
	public function save_album( $settings, $post_id ) {

		if (
			! isset( $_POST['_eg_album_data'], $_POST['envira_woocommerce_nonce'] )
			|| ! wp_verify_nonce( sanitize_key( $_POST['envira_woocommerce_nonce'] ), 'envira_woocommerce_save_settings' )
		) {
			return $settings;
		}

		$settings['config']['lightbox_woocommerce']         = ( isset( $_POST['_eg_album_data']['config']['lightbox_woocommerce'] ) ? 1 : 0 );
		$settings['config']['mobile_disable_woo_galleries'] = ( isset( $_POST['_envira_gallery']['mobile_disable_woo_galleries'] ) ? 1 : 0 );

		return $settings;

	}

	/**
	 * Outputs fields in the modal window when editing an existing image,
	 * allowing the user to choose which WooCommerce Product/Variant to
	 * link the image to.
	 *
	 * @since 1.0.0
	 *
	 * @param int $post_id The current post ID.
	 */
	public function meta_settings( $post_id ) {

		// Get WooCommerce Products.
		$args     = array(
			'post_type'      => 'product',
			'posts_per_page' => -1,
		);
		$products = new WP_Query( $args );

		// WooCommerce Meta Editor
		// Use: wp.media.template( 'envira-meta-editor-woocommerce' ).
		?>
		<script type="text/html" id="tmpl-envira-meta-editor-woocommerce">
			<label class="setting">
				<span class="name"><?php esc_html_e( 'WooCommerce Product', 'envira-woocommerce' ); ?></span>
				<select name="woocommerce_product" size="1">
					<option value="0"><?php esc_html_e( '(No Product)', 'envira-woocommerce' ); ?></option>
					<?php
					if ( $products->have_posts() ) {
						foreach ( $products->posts as $product ) {
							?>
							<option value="<?php echo esc_html( $product->ID ); ?>"><?php echo esc_html( $product->post_title ); ?></option>
							<?php
						}
					}
					?>
				</select>
				<span class="description">
					<?php esc_html_e( 'Choose a WooCommerce Product which relates to this image.  Visitors will then be able to order the selected product in the gallery and/or lightbox views.', 'envira-woocommerce' ); ?>
				</span>
			</label>
		</script>
		<?php

	}

	/**
	 * Adds addon settings UI to the Mobile tab
	 *
	 * @since 1.0.0
	 *
	 * @param object $post The current post object.
	 */
	public function mobile_screen( $post ) {

		// Get post type so we load the correct metabox instance and define the input field names
		// Input field names vary depending on whether we are editing a Gallery or Album.
		$post_type = get_post_type( $post );
		switch ( $post_type ) {
			/**
			* Gallery
			*/
			case 'envira':
				$instance = Envira_Gallery_Metaboxes::get_instance();
				$key      = '_envira_gallery';
				break;

			/**
			* Album
			*/
			case 'envira_album':
				$instance = Envira_Albums_Metaboxes::get_instance();
				$key      = '_eg_album_data[config]';
				break;
		}
		?>
		<tr id="envira-config-woo-galleries-mobile-box">
			<th scope="row">
				<label for="envira-config-woo-galleries-mobile"><?php esc_html_e( 'Disable WooCommerce In Galleries On Mobile?', 'envira-woocommerce' ); ?></label>
			</th>
			<td>
				<input id="envira-config-woo-galleries-mobile" type="checkbox" name="<?php echo esc_html( $key ); ?>[mobile_disable_woo_galleries]" value="1" <?php checked( $instance->get_config( 'mobile_disable_woo_galleries', $instance->get_config_default( 'mobile_disable_woo_galleries' ) ), 1 ); ?> />
				<span class="description"><?php esc_html_e( 'If enabled, no WooCommerce functionality or buttons will be displayed in galleries or lightboxes on mobile.', 'envira-woocommerce' ); ?></span>
			</td>
		</tr>
		<?php

	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Envira_WooCommerce_Metaboxes object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Envira_WooCommerce_Metaboxes ) ) {
			self::$instance = new Envira_WooCommerce_Metaboxes();
		}

		return self::$instance;

	}

}

// Load the metabox class.
$envira_woocommerce_metaboxes = Envira_WooCommerce_Metaboxes::get_instance();
