<?php

if (!defined('ABSPATH')) exit;


if (!class_exists('ElfsightContactFormMail')) {
    class ElfsightContactFormMail {
        private $pluginName;
        private $pluginSlug;
        private $recaptchaSecretCheckbox;
        private $recaptchaSecretInvisible;

        public function __construct($config) {
            $this->pluginName = $config['plugin_name'];
            $this->pluginSlug = $config['plugin_slug'];
            $this->recaptchaSecretCheckbox = $config['recaptcha_secret_checkbox'];
            $this->recaptchaSecretInvisible = $config['recaptcha_secret_invisible'];

            add_action('wp_ajax_nopriv_' . str_replace('-', '_', $this->pluginSlug) . '_mail', array($this, 'mail'));
            add_action('wp_ajax_' . str_replace('-', '_', $this->pluginSlug) . '_mail', array($this, 'mail'));
        }

        public function mail() {
            $mail_to = $this->input('mailTo');
            $fieldsNames = $this->input('fieldsNames');
            $fieldsValues = $this->input('fieldsValues');
            $widget_id = $this->input('widgetId');

            if (!$this->handle_recaptcha()) {
                return $this->response('WRONG_RECAPTCHA', 400);
            }

            if (empty($mail_to) && !empty($widget_id)) {
                $widget_options = $this->get_widget_settings($widget_id);

                if (is_array($widget_options)) {
                    $mail_to = $widget_options['mailTo'];
                }
            }

            $mail_to = explode(',', $mail_to);

            if (empty($mail_to)) {
                return $this->response('MAILTO_IS_NOT_SPECIFIED', 400);
            }

            $fields = $this->format_fields($fieldsNames, $fieldsValues);

            if (empty($fields)) {
                return $this->response('EMPTY_FIELDS', 400);
            }

            $mail = $this->build_message($fields, $mail_to);
            $files = $this->prepare_files();

            $status = $this->send_message($mail, $mail_to, $files);

            $this->clean_files($files);

            if (!empty($widget_id)) {
                $this->save_mail($fields, $widget_id);
            }

            if ($status) {
                return $this->response('OK', 200);
            } else {
                return $this->response('FAIL', 400);
            }
        }

        public function clean_files($files) {
            if (empty($files)) {
                return;
            }

            foreach ($files as $file) {
                wp_delete_file($file);
            }
        }

        public function prepare_files() {
            $files = array();

            if (!empty($_FILES) && !empty($_FILES['files'])) {
                $tmp_files = $_FILES['files'];

                foreach ($tmp_files['name'] as $i => $value) {
                    if ($tmp_files['name'][$i]) {
                        $file = array(
                            'name'     => $tmp_files['name'][$i],
                            'type'     => $tmp_files['type'][$i],
                            'tmp_name' => $tmp_files['tmp_name'][$i],
                            'error'    => $tmp_files['error'][$i],
                            'size'     => $tmp_files['size'][$i]
                        );

                        $upl_file = wp_handle_upload($file, array('test_form' => false));
                        if (isset($upl_file['file'])) {
                            $files[] = $upl_file['file'];
                        }
                    }
                }
            }

            return $files;
        }

        public function format_fields($names, $values) {
            $fields = array();

            foreach ($names as $key => $name) {
                $fields[] = array(
                    'name' => $names[$key],
                    'value' => $values[$key]
                );
            }

            return $fields;
        }

        public function build_message($fields, $mail_to) {
            $eol = "\r\n";
            $url = $this->input('url');
            $from_name = $_SERVER['SERVER_NAME'];
            $from_email = 'wordpress@' . $_SERVER['SERVER_NAME'];

            $subject = 'New message from your website ' . $_SERVER['SERVER_NAME'];

            $text = '
            <p>
                This message was sent from page ' . urldecode($url) . '<br>
                See it below:
            </p>';

            $text .= '<ul>';
            foreach($fields as $field) {
                $text .= '<li><b>' . $field['name'] . '</b>: ' . $field['value'] . '</li>';
            }
            $text .= '</ul>';

            $headers = 'From: ' . $from_name . ' <' . $from_email .'>' . $eol;
            $headers .= 'MIME-Version: 1.0' . $eol;
            $headers .= 'Content-type: text/html; charset=UTF-8' . $eol;

            return array(
                'subject' => $subject,
                'text' => $text,
                'headers' => $headers
            );
        }

        public function send_message($mail, $mail_to, $files) {
            return wp_mail($mail_to, $mail['subject'], $mail['text'], $mail['headers'], $files);
        }

        private function get_widget_settings($widget_id) {
            global $wpdb;

            $table_name = $this->get_table_name('widgets');

            $results = $wpdb->get_results('SELECT * FROM `' . esc_sql($table_name) . '` WHERE `id` = ' . esc_sql($widget_id));

            if (is_array($results) && !empty($results[0])) {
                return json_decode($results[0]->options, true);
            }
            else {
                return false;
            }
        }

        private function get_table_name($name) {
            global $wpdb;

            return $wpdb->prefix . str_replace('-', '_', $this->pluginSlug) . '_' . $name;
        }

        private function create_table() {
            $table_name = $this->get_table_name('mails');

            if (!function_exists('dbDelta')) {
                require(ABSPATH . 'wp-admin/includes/upgrade.php');
            }

            dbDelta(
                'CREATE TABLE `' . esc_sql($table_name) . '` (
                    `id` int(11) unsigned NOT NULL AUTO_INCREMENT,
                    `data` text NOT NULL,
                    `widget_id` int(11) NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=MyISAM DEFAULT CHARSET=utf8 AUTO_INCREMENT=1;'
            );
        }

        private function table_exist() {
            global $wpdb;

            $table_name = $this->get_table_name('mails');

            return !!$wpdb->get_var('SHOW TABLES LIKE "' . esc_sql($table_name) . '"');
        }

        private function save_mail($fields, $widget_id) {
            global $wpdb;

            $table_name = $this->get_table_name('mails');
            $data = json_encode($fields);

            if (!$this->table_exist()) {
                $this->create_table();
            }

            $wpdb->insert($table_name, array(
                'data' => $data,
                'widget_id' => $widget_id
            ));
        }

        public function handle_recaptcha() {
            $recaptcha_type = $this->input('recaptcha_type');
            $recaptcha_changed = true;

            if (!empty($_SESSION['recaptcha_token']) && $_SESSION['recaptcha_token'] === $this->input('recaptcha_token')) {
                $recaptcha_changed = false;
            }

            $_SESSION['recaptcha_token'] = $recaptcha_token = $this->input('recaptcha_token');

            if (!$recaptcha_changed) {
                return true;
            }

            if (!$recaptcha_token) {
                $this->response('Recaptcha token is missing', 400);
            }
            if (empty($recaptcha_type)) {
                $this->response('Recaptcha type is not defined', 400);
            }

            if ($recaptcha_type === 'none') {
                return true;
            }

            if ($recaptcha_type === 'checkbox') {
                $recaptcha_secret = $this->recaptchaSecretCheckbox;
            } elseif ($recaptcha_type === 'invisible') {
                $recaptcha_secret = $this->recaptchaSecretInvisible;
            }

            if (empty($recaptcha_secret)) {
                $this->response('No Recaptcha secret has been specified', 400);
            }

            $response = wp_remote_get('https://www.google.com/recaptcha/api/siteverify?secret=' . $recaptcha_secret . '&response=' . $recaptcha_token);
            $result = json_decode($response['body'], true);

            if (!isset($result['success']) || $result['success'] !== true) {
                return false;
            }

            return true;
        }

        public function response($status, $code) {
            $response = array($code, $status);

            header('Content-type: application/json; charset=utf-8');
            exit(json_encode($response));
        }

        public function input($name, $default = null) {
            return isset($_REQUEST[$name]) ? $_REQUEST[$name] : $default;
        }
    }
}

?>