<?php
namespace ElementorExtras\Modules\Image\Widgets;

use ElementorExtras\Base\Extras_Widget;

// Elementor Classes
use Elementor\Controls_Manager;
use Elementor\Control_Media;
use Elementor\Utils;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Background;
use Elementor\Scheme_Typography;
use Elementor\Scheme_Color;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Random_Image
 *
 * @since 2.0.0
 */
class Random_Image extends Extras_Widget {

	public function get_name() {
		return 'ee-random-image';
	}

	public function get_title() {
		return __( 'Random Image', 'elementor-extras' );
	}

	public function get_icon() {
		return 'nicon nicon-random';
	}

	public function get_categories() {
		return [ 'elementor-extras' ];
	}

	/**
	 * A list of scripts that the widgets is depended in
	 * @since 0.1.0
	 **/
	public function get_script_depends() {
		return [
			'magnific-popup',
		];
	}

	protected function _register_controls() {
		$this->start_controls_section(
			'section_images',
			[
				'label' => __( 'Images', 'elementor-extras' ),
			]
		);

			$this->add_control(
				'wp_gallery',
				[
					'label' 	=> __( 'Add Images', 'elementor-extras' ),
					'type' 		=> Controls_Manager::GALLERY,
					'frontend_available' => true,
					'dynamic'	=> [
						'active' => true,
					],
				]
			);

			$this->add_group_control(
				Group_Control_Image_Size::get_type(),
				[
					'name' 		=> 'image', // Actually its `image_size`.
					'label' 	=> __( 'Image Size', 'elementor-extras' ),
					'default' 	=> 'full',
				]
			);

			$this->add_responsive_control(
				'align',
				[
					'label' 	=> __( 'Alignment', 'elementor-extras' ),
					'type' 		=> Controls_Manager::CHOOSE,
					'options' 	=> [
						'left' 	=> [
							'title' 	=> __( 'Left', 'elementor-extras' ),
							'icon' 		=> 'fa fa-align-left',
						],
						'center' 	=> [
							'title' 	=> __( 'Center', 'elementor-extras' ),
							'icon' 		=> 'fa fa-align-center',
						],
						'right' 	=> [
							'title' 	=> __( 'Right', 'elementor-extras' ),
							'icon' 		=> 'fa fa-align-right',
						],
					],
					'selectors' 	=> [
						'{{WRAPPER}}' => 'text-align: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'caption',
				[
					'label' 		=> __( 'Caption', 'elementor-extras' ),
					'type' 			=> Controls_Manager::SWITCHER,
					'default' 		=> '',
					'label_on' 		=> __( 'Show', 'elementor-extras' ),
					'label_off' 	=> __( 'Hide', 'elementor-extras' ),
					'return_value' 	=> 'yes',
				]
			);

			$this->add_control(
				'link_to',
				[
					'label' => __( 'Link to', 'elementor-extras' ),
					'type' => Controls_Manager::SELECT,
					'default' => 'none',
					'options' => [
						'none' => __( 'None', 'elementor-extras' ),
						'file' => __( 'Media File', 'elementor-extras' ),
						'custom' => __( 'Custom URL', 'elementor-extras' ),
					],
				]
			);

			$this->add_control(
				'link',
				[
					'label' => __( 'Link to', 'elementor-extras' ),
					'type' => Controls_Manager::URL,
					'dynamic' => [
						'active' => true,
					],
					'placeholder' => __( 'https://your-link.com', 'elementor-extras' ),
					'condition' => [
						'link_to' => 'custom',
					],
					'show_label' => false,
				]
			);

			$this->add_control(
				'open_lightbox',
				[
					'label' => __( 'Lightbox', 'elementor-extras' ),
					'type' => Controls_Manager::SELECT,
					'default' => 'default',
					'options' => [
						'default' => __( 'Default', 'elementor-extras' ),
						'yes' => __( 'Yes', 'elementor-extras' ),
						'no' => __( 'No', 'elementor-extras' ),
					],
					'condition' => [
						'link_to' => 'file',
					],
				]
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_image',
			[
				'label' => __( 'Image', 'elementor-extras' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

			$this->add_responsive_control(
				'width',
				[
					'label' => __( 'Width', 'elementor-extras' ),
					'type' => Controls_Manager::SLIDER,
					'default' => [
						'unit' => '%',
					],
					'tablet_default' => [
						'unit' => '%',
					],
					'mobile_default' => [
						'unit' => '%',
					],
					'size_units' => [ '%', 'px', 'vw' ],
					'range' => [
						'%' => [
							'min' => 1,
							'max' => 100,
						],
						'px' => [
							'min' => 1,
							'max' => 1000,
						],
						'vw' => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .ee-random-image__image' => 'width: {{SIZE}}{{UNIT}};',
					],
				]
			);

			$this->add_responsive_control(
				'space',
				[
					'label' => __( 'Max Width', 'elementor-extras' ) . ' (%)',
					'type' => Controls_Manager::SLIDER,
					'default' => [
						'unit' => '%',
					],
					'tablet_default' => [
						'unit' => '%',
					],
					'mobile_default' => [
						'unit' => '%',
					],
					'size_units' => [ '%' ],
					'range' => [
						'%' => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .ee-random-image__image' => 'max-width: {{SIZE}}{{UNIT}};',
					],
				]
			);

			$this->add_control(
				'opacity',
				[
					'label' => __( 'Opacity', 'elementor-extras' ),
					'type' => Controls_Manager::SLIDER,
					'range' => [
						'px' => [
							'max' => 1,
							'min' => 0.10,
							'step' => 0.01,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .ee-random-image__image' => 'opacity: {{SIZE}};',
					],
				]
			);

			$this->add_control(
				'hover_animation',
				[
					'label' => __( 'Hover Animation', 'elementor-extras' ),
					'type' => Controls_Manager::HOVER_ANIMATION,
				]
			);

			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name' => 'image_border',
					'selector' => '{{WRAPPER}} .ee-random-image__image',
					'separator' => 'before',
				]
			);

			$this->add_responsive_control(
				'image_border_radius',
				[
					'label' => __( 'Border Radius', 'elementor-extras' ),
					'type' => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors' => [
						'{{WRAPPER}} .ee-random-image__image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'image_box_shadow',
					'exclude' => [
						'box_shadow_position',
					],
					'selector' => '{{WRAPPER}} .ee-random-image__image',
				]
			);

			$this->end_controls_section();

			$this->start_controls_section(
				'section_style_caption',
				[
					'label' => __( 'Caption', 'elementor-extras' ),
					'tab'   => Controls_Manager::TAB_STYLE,
					'condition' => [
						'caption!' => '',
					],
				]
			);

			$this->add_control(
				'caption_align',
				[
					'label' => __( 'Alignment', 'elementor-extras' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'left' => [
							'title' => __( 'Left', 'elementor-extras' ),
							'icon' => 'fa fa-align-left',
						],
						'center' => [
							'title' => __( 'Center', 'elementor-extras' ),
							'icon' => 'fa fa-align-center',
						],
						'right' => [
							'title' => __( 'Right', 'elementor-extras' ),
							'icon' => 'fa fa-align-right',
						],
						'justify' => [
							'title' => __( 'Justified', 'elementor-extras' ),
							'icon' => 'fa fa-align-justify',
						],
					],
					'default' => '',
					'selectors' => [
						'{{WRAPPER}} .widget-image-caption' => 'text-align: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'text_color',
				[
					'label' => __( 'Text Color', 'elementor-extras' ),
					'type' => Controls_Manager::COLOR,
					'default' => '',
					'selectors' => [
						'{{WRAPPER}} .widget-image-caption' => 'color: {{VALUE}};',
					],
					'scheme' => [
						'type' => Scheme_Color::get_type(),
						'value' => Scheme_Color::COLOR_3,
					],
				]
			);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' => 'caption_typography',
					'selector' => '{{WRAPPER}} .widget-image-caption',
					'scheme' => Scheme_Typography::TYPOGRAPHY_3,
				]
			);

			$this->add_responsive_control(
				'caption_space',
				[
					'label' => __( 'Spacing', 'elementor-extras' ),
					'type' => Controls_Manager::SLIDER,
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .widget-image-caption' => 'margin-top: {{SIZE}}{{UNIT}};',
					],
				]
			);

		$this->end_controls_section();
	}

	protected function render() {
		$settings 		= $this->get_settings_for_display();

		if ( ! $settings['wp_gallery'] ) {
			echo $this->render_placeholder( [
				'body' => __( 'No images selected.', 'elementor-extras' ),
			] );
			return;
		}

		$count 			= count( $settings['wp_gallery'] );
		$_index 		= ( $count > 1 ) ? rand( 0, $count - 1 ) : $count;
		$id 			= $settings['wp_gallery'][$_index]['id'];
		$has_caption 	= 'yes' === $settings['caption'];
		$link 			= $this->get_link_url( $settings, $_index );

		$this->add_render_attribute( [
			'wrapper' => [
				'class' => 'ee-random-image',
			],
			'figure' => [
				'class' => [
					'wp-caption',
					'ee-random-image__figure'
				],
			],
			'image' => [
				'class' => 'elementor-image ee-random-image__image',
				'src' => Group_Control_Image_Size::get_attachment_image_src( $id, 'image', $settings ),
				'alt' => esc_attr( Control_Media::get_image_alt( $id ) ),
			],
			'caption' => [
				'class' => [
					'widget-image-caption',
					'wp-caption-text',
					'ee-random-image__caption',
				],
			],
		] );

		if ( '' !== $settings['hover_animation'] ) {
			$this->add_render_attribute( 'image', 	'class', 'elementor-animation-' . $settings['hover_animation'] );
		}

		if ( $link ) {
			$this->add_render_attribute( 'link', [
				'href' => $link['url'],
				'data-elementor-open-lightbox' => $settings['open_lightbox'],
			] );

			if ( $this->is_edit_mode ) {
				$this->add_render_attribute( 'link', [
					'class' => 'elementor-clickable',
				] );
			}

			if ( ! empty( $link['is_external'] ) ) {
				$this->add_render_attribute( 'link', 'target', '_blank' );
			}

			if ( ! empty( $link['nofollow'] ) ) {
				$this->add_render_attribute( 'link', 'rel', 'nofollow' );
			}
		}

		?><div <?php echo $this->get_render_attribute_string( 'wrapper' ); ?>>
			<?php if ( $has_caption ) { ?>
			<figure <?php echo $this->get_render_attribute_string( 'figure' ); ?>>
			<?php } ?>

			<?php if ( $link ) { ?>
				<a <?php echo $this->get_render_attribute_string( 'link' ); ?>>
			<?php } ?>
					<img <?php echo $this->get_render_attribute_string( 'image' ); ?> />
			<?php if ( $link ) { ?>
				</a>
			<?php } ?>

			<?php if ( $has_caption ) { ?>
				<figcaption <?php echo $this->get_render_attribute_string( 'caption' ); ?>>
					asdasd
				</figcaption>
			</figure>
			<?php } ?>
		</div><?php
	}

	protected function _content_template() { ?><#
		if ( ! settings.wp_gallery.length ) {
			return;
		}

		var count = settings.wp_gallery.length,
			index = ( count > 1 ) ? Math.floor( ( Math.random() * count ) ) : count,
			has_caption = 'yes' === settings.caption,
			link_url;

		var image = settings.wp_gallery[index],
			_image = {
				id: image.id,
				url: image.url,
				size: settings.image_size,
				dimension: settings.image_custom_dimension,
				model: view.getEditModel()
			};

		if ( 'custom' === settings.link_to ) {
			link_url = settings.link.url;
		}

		if ( 'file' === settings.link_to ) {
			link_url = settings.wp_gallery[index].url;
		}

		view.addRenderAttribute( {
			'wrapper' : {
				'class' : 'ee-random-image',
			},
			'figure' : {
				'class' : 'wp-caption ee-random-image__figure',
			},
			'image' : {
				'src' : elementor.imagesManager.getImageUrl( _image ),
				'class' : 'ee-random-image__image',
			},
			'caption' : {
				'class' : [
					'widget-image-caption',
					'wp-caption-text',
					'ee-random-image__caption',
				],
			},
		} );

		if ( '' !== settings.hover_animation ) {
			view.addRenderAttribute( 'image', 	'src', 'elementor-animation-' + settings.hover_animation );
		}
	
	#><div {{{ view.getRenderAttributeString( 'wrapper' ) }}}>
		<# if ( has_caption ) { #>
		<figure {{{ view.getRenderAttributeString( 'figure' ) }}}>
		<# }

			if ( link_url ) { #><a class="elementor-clickable" data-elementor-open-lightbox="{{ settings.open_lightbox }}" href="{{ link_url }}"><# }
				#><img {{{ view.getRenderAttributeString( 'image' ) }}} /><#
			if ( link_url ) { #></a><# }

		if ( has_caption ) { #>
			<figcaption {{{ view.getRenderAttributeString( 'caption' ) }}}>
				asdasd
			</figcaption>
		</figure>
		<# } #>
	</div><?php
	}

	/**
	 * Retrieve image widget link URL.
	 *
	 * @since 2.0.0
	 * @access private
	 *
	 * @param array $settings
	 *
	 * @return array|string|false An array/string containing the link URL, or false if no link.
	 */
	private function get_link_url( $settings, $index ) {
		if ( 'none' === $settings['link_to'] ) {
			return false;
		}

		if ( 'custom' === $settings['link_to'] ) {
			if ( empty( $settings['link']['url'] ) ) {
				return false;
			}
			return $settings['link'];
		}

		return [
			'url' => $settings['wp_gallery'][$index]['url'],
		];
	}
}
