<?php
namespace ElementorExtras\Modules\Breadcrumbs\Widgets;

// Elementor Extras Classes
use ElementorExtras\Base\Extras_Widget;
use ElementorExtras\Utils;

// Elementor Classes
use Elementor\Icons_Manager;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Scheme_Color;
use Elementor\Scheme_Typography;
use Elementor\Modules\DynamicTags\Module as TagsModule;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Breadcrumbs
 *
 * @since 1.2.0
 */
class Breadcrumbs extends Extras_Widget {

	/**
	 * Query
	 *
	 * @since  1.2.0
	 * @var    \WP_Query
	 */
	private $_query = null;

	/**
	 * Separator
	 *
	 * @since  1.2.0
	 * @var    string
	 */
	private $_separator = null;

	/**
	 * Get Name
	 * 
	 * Get the name of the widget
	 *
	 * @since  1.2.0
	 * @return string
	 */
	public function get_name() {
		return 'ee-breadcrumbs';
	}

	/**
	 * Get Title
	 * 
	 * Get the title of the widget
	 *
	 * @since  1.2.0
	 * @return string
	 */
	public function get_title() {
		return __( 'Breadcrumbs', 'elementor-extras' );
	}

	/**
	 * Get Icon
	 * 
	 * Get the icon of the widget
	 *
	 * @since  1.2.0
	 * @return string
	 */
	public function get_icon() {
		return 'nicon nicon-breadcrumbs';
	}

	/**
	 * Register Widget Controls
	 *
	 * @since  1.2.0
	 * @return void
	 */
	protected function _register_controls() {
		
		$this->start_controls_section(
			'section_content',
			[
				'label' => __( 'Display', 'elementor-extras' ),
			]
		);

			$this->add_control(
				'source',
				[
					'label' 	=> __( 'Source', 'elementor-extras' ),
					'type'		=> Controls_Manager::SELECT,
					'default'	=> '',
					'options'	=> [
						''		=> __( 'Current page', 'elementor-extras' ),
						'id'	=> __( 'Specific page', 'elementor-extras' ),
					]
				]
			);

			$this->add_control(
				'source_id',
				[
					'label' 		=> __( 'ID', 'elementor-extras' ),
					'type'			=> Controls_Manager::NUMBER,
					'min' 			=> 0,
					'placeholder' 	=> '15',
					'condition'		=> [
						'source'	=> 'id',
					]
				]
			);

			$this->add_control(
				'show_home',
				[
					'label' 		=> __( 'Show Home', 'elementor-extras' ),
					'type' 			=> Controls_Manager::SWITCHER,
					'default' 		=> 'yes',
					'label_on' 		=> __( 'Yes', 'elementor-extras' ),
					'label_off' 	=> __( 'No', 'elementor-extras' ),
					'return_value' 	=> 'yes',
				]
			);

			$this->add_control(
				'show_current',
				[
					'label' 		=> __( 'Show Current', 'elementor-extras' ),
					'type' 			=> Controls_Manager::SWITCHER,
					'default' 		=> 'yes',
					'label_on' 		=> __( 'Yes', 'elementor-extras' ),
					'label_off' 	=> __( 'No', 'elementor-extras' ),
					'return_value' 	=> 'yes',
				]
			);

			$this->add_control(
				'cpt_crumbs',
				[
					'label' 		=> __( 'CPT Crumbs', 'elementor-extras' ),
					'type' 			=> Controls_Manager::SELECT,
					'default' 		=> '',
					'options'		=> [
						'' 			=> __( 'CPT Name', 'elementor-extras' ),
						'terms' 	=> __( 'Taxonomy Terms', 'elementor-extras' ),
						'both' 		=> __( 'Both', 'elementor-extras' ),
					],
				]
			);

			$this->add_control(
				'home_text',
				[
					'label' 		=> __( 'Home Text', 'elementor-extras' ),
					'type' 			=> Controls_Manager::TEXT,
					'default' 		=> __( 'Homepage', 'elementor-extras' ),
					'dynamic'		=> [
						'active'	=> true,
						'categories' => [ TagsModule::POST_META_CATEGORY ]
					],
					'condition'		=> [
						'show_home' => 'yes'
					],
				]
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_separator',
			[
				'label' => __( 'Separator', 'elementor-extras' ),
			]
		);

			$this->add_control(
				'separator_type',
				[
					'label'		=> __( 'Type', 'elementor-extras' ),
					'type' 		=> Controls_Manager::SELECT,
					'default' 	=> 'icon',
					'options' 	=> [
						'text' 		=> __( 'Text', 'elementor-extras' ),
						'icon' 		=> __( 'Icon', 'elementor-extras' ),
					],
				]
			);

			$this->add_control(
				'separator_text',
				[
					'label' 		=> __( 'Text', 'elementor-extras' ),
					'type' 			=> Controls_Manager::TEXT,
					'default' 		=> __( '>', 'elementor-extras' ),
					'condition'		=> [
						'separator_type' => 'text'
					],
				]
			);

			$this->add_control(
				'selected_separator_icon',
				[
					'label' => __( 'Icon', 'elementor-extras' ),
					'type' => Controls_Manager::ICONS,
					'label_block' => true,
					'fa4compatibility' => 'separator_icon',
					'condition'		=> [
						'separator_type' => 'icon'
					],
					'default' => [
						'value' => 'fas fa-angle-right',
						'library' => 'fa-solid',
					],
				]
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_item_style',
			[
				'label' 	=> __( 'Crumbs', 'elementor-extras' ),
				'tab' 		=> Controls_Manager::TAB_STYLE,
			]
		);

			$this->add_responsive_control(
				'items_align',
				[
					'label' 		=> __( 'Align Crumbs', 'elementor-extras' ),
					'type' 			=> Controls_Manager::CHOOSE,
					'default' 		=> '',
					'options' 		=> [
						'left' 			=> [
							'title' 	=> __( 'Left', 'elementor-extras' ),
							'icon' 		=> 'eicon-h-align-left',
						],
						'center' 		=> [
							'title' 	=> __( 'Center', 'elementor-extras' ),
							'icon' 		=> 'eicon-h-align-center',
						],
						'right' 		=> [
							'title' 	=> __( 'Right', 'elementor-extras' ),
							'icon' 		=> 'eicon-h-align-right',
						],
						'stretch' 		=> [
							'title' 	=> __( 'Stretch', 'elementor-extras' ),
							'icon' 		=> 'eicon-h-align-stretch',
						],
					],
					'prefix_class' 	=> 'ee-breadcrumbs-align%s-',
				]
			);

			$this->add_responsive_control(
				'items_text_align',
				[
					'label' 		=> __( 'Align Text', 'elementor-extras' ),
					'type' 			=> Controls_Manager::CHOOSE,
					'default' 		=> '',
					'options' 		=> [
						'left' 			=> [
							'title' 	=> __( 'Left', 'elementor-extras' ),
							'icon' 		=> 'fa fa-align-left',
						],
						'center' 		=> [
							'title' 	=> __( 'Center', 'elementor-extras' ),
							'icon' 		=> 'fa fa-align-center',
						],
						'right' 		=> [
							'title' 	=> __( 'Right', 'elementor-extras' ),
							'icon' 		=> 'fa fa-align-right',
						],
					],
					'selectors' => [
						'{{WRAPPER}} .ee-breadcrumbs' => 'text-align: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'item_spacing',
				[
					'label' 	=> __( 'Spacing', 'elementor-extras' ),
					'type' 		=> Controls_Manager::SLIDER,
					'default'	=> [
						'size'	=> 12
					],
					'range' 	=> [
						'px' 	=> [
							'max' => 36,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .ee-breadcrumbs' => 'margin-left: -{{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .ee-breadcrumbs__item' => 'margin-left: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .ee-breadcrumbs__separator' => 'margin-left: {{SIZE}}{{UNIT}};',
					],
				]
			);

			$this->add_control(
				'item_padding',
				[
					'label' 		=> __( 'Padding', 'elementor-extras' ),
					'type' 			=> Controls_Manager::DIMENSIONS,
					'size_units' 	=> [ 'px', 'em', '%' ],
					'selectors' 	=> [
						'{{WRAPPER}} .ee-breadcrumbs__item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'allowed_dimensions' => [ 'right', 'left' ],
				]
			);

			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name' 		=> 'item_border',
					'label' 	=> __( 'Border', 'elementor-extras' ),
					'selector' 	=> '{{WRAPPER}} .ee-breadcrumbs__item',
				]
			);

			$this->add_control(
				'item_border_radius',
				[
					'label' 		=> __( 'Border Radius', 'elementor-extras' ),
					'type' 			=> Controls_Manager::DIMENSIONS,
					'size_units' 	=> [ 'px', '%' ],
					'selectors' 	=> [
						'{{WRAPPER}} .ee-breadcrumbs__item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' 		=> 'item_typography',
					'label' 	=> __( 'Typography', 'elementor-extras' ),
					'scheme' 	=> Scheme_Typography::TYPOGRAPHY_4,
					'selector' 	=> '{{WRAPPER}} .ee-breadcrumbs__text',
				]
			);

			$this->start_controls_tabs( 'crumb_style' );

			$this->start_controls_tab( 'crumb_default', [ 'label' => __( 'Default', 'elementor-extras' ) ] );

				$this->add_control(
					'item_background_color',
					[
						'label' 	=> __( 'Background Color', 'elementor-extras' ),
						'type' 		=> Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .ee-breadcrumbs__item' => 'background-color: {{VALUE}};',
						],
					]
				);

				$this->add_control(
					'item_color',
					[
						'label' 	=> __( 'Color', 'elementor-extras' ),
						'type' 		=> Controls_Manager::COLOR,
						'default'	=> '',
						'selectors' => [
							'{{WRAPPER}} .ee-breadcrumbs__item' => 'color: {{VALUE}};',
							'{{WRAPPER}} .ee-breadcrumbs__item a' => 'color: {{VALUE}};',
						],
					]
				);

			$this->end_controls_tab();

			$this->start_controls_tab( 'crumb_hover', [ 'label' => __( 'Hover', 'elementor-extras' ) ] );

				$this->add_control(
					'item_background_color_hover',
					[
						'label' 	=> __( 'Background Color', 'elementor-extras' ),
						'type' 		=> Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .ee-breadcrumbs__item:hover' => 'background-color: {{VALUE}};',
						],
					]
				);

				$this->add_control(
					'item_color_hover',
					[
						'label' 	=> __( 'Color', 'elementor-extras' ),
						'type' 		=> Controls_Manager::COLOR,
						'default'	=> '',
						'selectors' => [
							'{{WRAPPER}} .ee-breadcrumbs__item:hover' => 'color: {{VALUE}};',
							'{{WRAPPER}} .ee-breadcrumbs__item:hover a' => 'color: {{VALUE}};',
						],
					]
				);
			
			$this->end_controls_tab();

			$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_separator_style',
			[
				'label' 	=> __( 'Separators', 'elementor-extras' ),
				'tab' 		=> Controls_Manager::TAB_STYLE,
			]
		);

			$this->add_control(
				'separator_padding',
				[
					'label' 		=> __( 'Padding', 'elementor-extras' ),
					'type' 			=> Controls_Manager::DIMENSIONS,
					'size_units' 	=> [ 'px', 'em', '%' ],
					'selectors' 	=> [
						'{{WRAPPER}} .ee-breadcrumbs__separator' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'allowed_dimensions' => [ 'right', 'left' ],
				]
			);

			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name' 		=> 'separator_border',
					'label' 	=> __( 'Border', 'elementor-extras' ),
					'selector' 	=> '{{WRAPPER}} .ee-breadcrumbs__separator',
				]
			);

			$this->add_control(
				'separator_border_radius',
				[
					'label' 		=> __( 'Border Radius', 'elementor-extras' ),
					'type' 			=> Controls_Manager::DIMENSIONS,
					'size_units' 	=> [ 'px', '%' ],
					'selectors' 	=> [
						'{{WRAPPER}} .ee-breadcrumbs__separator' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->add_control(
				'separator_background_color',
				[
					'label' 	=> __( 'Background Color', 'elementor-extras' ),
					'type' 		=> Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .ee-breadcrumbs__separator' => 'background-color: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'separator_color',
				[
					'label' 	=> __( 'Color', 'elementor-extras' ),
					'type' 		=> Controls_Manager::COLOR,
					'default'	=> '',
					'selectors' => [
						'{{WRAPPER}} .ee-breadcrumbs__separator' => 'color: {{VALUE}};',
					],
				]
			);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' 		=> 'separator_typography',
					'label' 	=> __( 'Typography', 'elementor-extras' ),
					'scheme' 	=> Scheme_Typography::TYPOGRAPHY_4,
					'selector' 	=> '{{WRAPPER}} .ee-breadcrumbs__separator',
				]
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_current_style',
			[
				'label' 	=> __( 'Current', 'elementor-extras' ),
				'tab' 		=> Controls_Manager::TAB_STYLE,
			]
		);

			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name' 		=> 'current_border',
					'label' 	=> __( 'Border', 'elementor-extras' ),
					'selector' 	=> '{{WRAPPER}} .ee-breadcrumbs__item--current',
				]
			);

			$this->add_control(
				'current_border_radius',
				[
					'label' 		=> __( 'Border Radius', 'elementor-extras' ),
					'type' 			=> Controls_Manager::DIMENSIONS,
					'size_units' 	=> [ 'px', '%' ],
					'selectors' 	=> [
						'{{WRAPPER}} .ee-breadcrumbs__item--current' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->add_control(
				'current_background_color',
				[
					'label' 	=> __( 'Background Color', 'elementor-extras' ),
					'type' 		=> Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .ee-breadcrumbs__item--current' => 'background-color: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'current_color',
				[
					'label' 	=> __( 'Color', 'elementor-extras' ),
					'type' 		=> Controls_Manager::COLOR,
					'default'	=> '',
					'selectors' => [
						'{{WRAPPER}} .ee-breadcrumbs__item--current' => 'color: {{VALUE}};',
					],
				]
			);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' 		=> 'current_typography',
					'label' 	=> __( 'Typography', 'elementor-extras' ),
					'scheme' 	=> Scheme_Typography::TYPOGRAPHY_4,
					'selector' 	=> '{{WRAPPER}} .ee-breadcrumbs__item--current .ee-breadcrumbs__text',
				]
			);

		$this->end_controls_section();

	}

	/**
	 * Get Query
	 *
	 * @since  1.2.0
	 * @return \WP_Query|bool
	 */
	protected function get_query() {
		global $post;

		$settings 	= $this->get_settings_for_display();
		$_id 		= null;
		$_post_type = 'post';

		if ( 'id' === $settings['source'] && '' !== $settings['source_id'] ) {

			$_id = $settings['source_id'];
			$_post_type = 'any';

			$_args = array(
				'p' 		=> $_id,
				'post_type' => $_post_type,
			);

			// Create custom query
			$_post_query = new \WP_Query( $_args );

			return $_post_query;
		}

		return false;
	}

	/**
	 * Set Separator
	 *
	 * Sets the markup for the breadcrumbs separator
	 *
	 * @since  1.2.0
	 * @return string
	 */
	protected function set_separator() {

		$settings = $this->get_settings_for_display();
		$separator = '';
		$has_icon = ! empty( $settings['separator_icon'] ) || ! empty( $settings['selected_separator_icon']['value'] );

		if ( 'icon' === $settings['separator_type'] && $has_icon ) {
			$migrated = isset( $settings['__fa4_migrated']['selected_separator_icon'] );
			$is_new = empty( $settings['icon'] ) && Icons_Manager::is_migration_allowed();

			$this->add_render_attribute( 'icon-wrapper', 'class', [
				'ee-icon',
				'ee-icon-support--svg',
			] );

			$separator .= '<span ' . $this->get_render_attribute_string( 'icon-wrapper' ) . '>';
			
			if ( $is_new || $migrated ) {
				ob_start();
				Icons_Manager::render_icon( $settings['selected_separator_icon'], [ 'aria-hidden' => 'true' ] );
				$separator .= ob_get_clean();
			} else {
				$this->add_render_attribute( 'icon', [
					'class' => $settings['separator_icon'],
					'aria-hidden' => 'true',
				] );
				$separator .= '<i '. $this->get_render_attribute_string('icon') . '></i>';
			}

			$separator .= '</span>';
		} else {
			$this->add_inline_editing_attributes( 'separator_text' );
			$this->add_render_attribute( 'separator_text', 'class', 'ee-breadcrumbs__separator__text' );

			$separator_text = $settings['separator_text'];

			/**
			 * Separator Text filter
			 *
			 * Filters the separator text if the separator is set as text
			 * in the widget settings
			 *
			 * @since 2.2.0
			 * @param string 			$separator_text 	The separator text setting
			 */
			$separator_text = apply_filters( 'elementor_extras/widgets/breadcrumbs/separator/text', $separator );
			$separator = '<span ' . $this->get_render_attribute_string( 'separator_text' ) . '>' . $separator_text . '</span>';
		}

		$this->_separator = $separator;
	}

	/**
	 * Get Separator
	 *
	 * @since  1.2.0
	 * @return var\string
	 */
	protected function get_separator() {
		return $this->_separator;
	}

	/**
	 * Render
	 * 
	 * Render widget contents on frontend
	 *
	 * @since  1.2.0
	 * @return void
	 */
	protected function render() {

		$settings 	= $this->get_settings_for_display();
		$_query 	= $this->get_query();

		$this->set_separator();
		$this->add_render_attribute( 'breadcrumbs', [
			'class' => 'ee-breadcrumbs',
			'itemscope' => "",
			'itemtype' => "http://schema.org/BreadcrumbList",
		]);

		if ( $_query ) {
			if ( $_query->have_posts() ) {

				// Setup post
				$_query->the_post();

				/**
				 * Post Query Filter
				 *
				 * Filters the post query when breadcrumbs are set for a specific page
				 *
				 * @since 2.2.0
				 * @param WP_Query 	$_query 	The current query
				 */
				$_query = apply_filters( 'elementor_extras/widgets/breadcrumbs/query', $_query );

				// Render using the new query
				$this->render_breadcrumbs( $_query );

				// Reset post data to original query
				wp_reset_postdata();
				wp_reset_query();

			} else {

				_e( 'Post or page not found', 'elementor-extras' );

			}
		} else {
			// Render using the original query
			$this->render_breadcrumbs();
		}
	}

	/**
	 * Render Home Link
	 * 
	 * The markup for the home link crumb
	 *
	 * @since  1.2.0
	 * @return void
	 */
	protected function render_home_link() {
		$settings = $this->get_settings_for_display();

		$this->add_item_render_attribute( 'home-item', 0 );
		$this->add_render_attribute( 'home-item', [
			'class' => 'ee-breadcrumbs__item--home',
		] );

		$this->add_link_render_attribute( 'home-link' );
		$this->add_render_attribute( 'home-link', [
			'class' => [
				'ee-breadcrumbs__crumb--link',
				'ee-breadcrumbs__crumb--home'
			],
			/**
			 * Home Link Filter
			 *
			 * Filters the home breadcrumb link
			 *
			 * @since 2.2.4
			 * @param string 	$link 	The link, defaults to site home url
			 */
			'href' 	=> apply_filters( 'elementor_extras/widgets/breadcrumbs/home/link', get_home_url() ),
			'title' => $settings['home_text'],
		] );

		$this->add_render_attribute( 'home-text', [
			'itemprop' 	=> 'name',
			'class' 	=> 'ee-breadcrumbs__text',
		] );

		?><li <?php echo $this->get_render_attribute_string( 'home-item' ); ?>><?php

			/**
			 * Before Home Link
			 *
			 * Fires right before output of the home link
			 *
			 * @since 2.2.0
			 * @param Extras_Widget 	$this 	The current widget instance
			 */
			do_action( 'elementor_extras/widgets/breadcrumbs/before_home_link', $this );

			?><a <?php echo $this->get_render_attribute_string( 'home-link' ); ?>>
				<span <?php echo $this->get_render_attribute_string( 'home-text' ); ?>>
					<?php echo $settings['home_text']; ?>
				</span>
			</a><?php

			/**
			 * After Home Link
			 *
			 * Fires right after output of the home link
			 *
			 * @since 2.2.0
			 * @param Extras_Widget 	$this 	The current widget instance
			 */
			do_action( 'elementor_extras/widgets/breadcrumbs/after_home_link', $this );

		?></li><?php

		$this->render_separator();

	}

	/**
	 * Render Separator
	 * 
	 * The markup for the separator item
	 *
	 * @since  1.2.0
	 * @return void
	 */
	protected function render_separator( $output = true ) {

		$this->add_render_attribute( 'separator', [
			'class' => [
				'ee-breadcrumbs__separator',
			],
		] );

		$separator = $this->get_separator();

		/**
		 * Separator filter
		 *
		 * Filters the separator
		 *
		 * @since 2.2.0
		 * @param string 			$post_title 	The markup for the separator
		 */
		$separator = apply_filters( 'elementor_extras/widgets/breadcrumbs/separator', $separator );
		$markup = sprintf( '<li %1$s>%2$s</li>', $this->get_render_attribute_string( 'separator' ), $separator );

		if ( $output === true ) {
			echo $markup;
		} else {
			return $markup;
		}
	}

	/**
	 * Render Breadcrumbs
	 * 
	 * Identifies and outputs all the breadcrumbs
	 *
	 * @since  1.2.0
	 * @return void
	 */
	protected function render_breadcrumbs( $query = false ) {

		global $post, $wp_query;

		if ( $query === false ) {

			// Reset post data to parent query
			$wp_query->reset_postdata();

			// Set active query to native query
			$query = $wp_query;
		}

		$settings = $this->get_settings_for_display();
		$separator = $this->get_separator();

		$custom_taxonomy = 'product_cat';

		if ( $query->is_front_page() || $query->is_home() ) {
			return;
		}

		?><ul <?php echo $this->get_render_attribute_string( 'breadcrumbs' ); ?>><?php

			if ( 'yes' === $settings['show_home'] ) {
				$this->render_home_link();
			}

			if ( $query->is_archive() && ! $query->is_tax() && ! $query->is_category() && ! $query->is_tag() && ! $query->is_date() && ! $query->is_author() ) {

				$this->render_custom_archive();
				
			} else if ( $query->is_archive() && $query->is_tax() && ! $query->is_category() && ! $query->is_tag() ) {

				$this->render_custom_taxonomy_archive();

			} else if ( $query->is_post_type_archive() ) {

				$this->render_post_type_archive();
				
			} else if ( $query->is_single() ) {

				$this->render_single( $post, $query );
				
			} else if ( $query->is_category() ) {

				$this->render_category();
				
			} else if ( $query->is_page() ) {
				
				$this->render_page( $post );
				
			} else if ( $query->is_tag() ) {
				
				$this->render_tag();
			
			} else if ( $query->is_day() ) {

				$this->render_day();
				
			} else if ( $query->is_month() ) {

				$this->render_month();
				
			} else if ( $query->is_year() ) {

				$this->render_year();
				
			} else if ( $query->is_author() ) {
				
				$this->render_author();
				
			} else if ( $query->is_search() ) {

				$this->render_search();
			
			} elseif ( $query->is_404() ) {

				$this->render_404();

			}

		?></ul><?php
	}

	/**
	 * Render Custom Archive
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_custom_archive() {
		$this->render_item( 'archive', [
			'index'		=> 1,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'archive',
			'content' 	=> post_type_archive_title( '', false ),
		] );
	}

	/**
	 * Render Custom Taxonomy Archive
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_custom_taxonomy_archive() {
		$term 		= get_queried_object();
		$term_name 	= $term->name;
		$taxonomy 	= $term->taxonomy;
		$parents 	= get_ancestors( $term->term_id, $taxonomy );

		$post_type 	= get_post_type();
		$post_type_object = get_post_type_object( $post_type );

		$this->render_item( 'post-type-archive', [
			'index'		=> 1,
			'current' 	=> false,
			'separator'	=> true,
			'key' 		=> 'post-type-archive',
			'ids' 		=> [ $post_type ],
			'content' 	=> $post_type_object->labels->name,
			'link'		=> get_post_type_archive_link( $post_type ),
		] );

		if ( $parents )  {
			$parent_terms = get_terms( [
				'taxonomy' 	=> $taxonomy,
				'include' 	=> $parents,
			] );

			$parent_terms = array_reverse( $parent_terms );

			$counter = 2;
			foreach ( $parent_terms as $parent_term ) {
				$term_slug = $term->slug;

				/**
				 * Before Taxonomy Term
				 *
				 * Fires right before a custom taxonomy term that is not
				 * the current or last one.
				 * 
				 * The dynamic portion of the hook name, `$term_slug`, refers to the slug of the term.
				 *
				 * @since 2.2.0
				 * @param WP_Term 			$term 	The queried term
				 * @param Extras_Widget 	$this 	The current widget instance
				 */
				do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/term/{$term_slug}", $term, $this );

				$this->render_item( 'custom-tax-archive-parents', [
					'index'		=> $counter,
					'current' 	=> false,
					'separator'	=> true,
					'key' 		=> 'custom-tax-archive-' . $parent_term->term_id,
					'ids' 		=> [ $parent_term->term_id, $parent_term->slug ],
					'content' 	=> $parent_term->name,
					'link'		=> get_term_link( $parent_term ),
				] );
				$counter++;
			}
		}

		/**
		 * Before Last Taxonomy Term
		 *
		 * Fires right before the last custom taxonomy archive crumb
		 *
		 * @since 2.2.0
		 * @param WP_Term 			$term 	The queried term
		 * @param Extras_Widget 	$this 	The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/term/last", $term, $this );

		/**
		 * Before Specific Last Taxonomy Term
		 *
		 * Fires right before the last specific custom taxonomy archive crumb
		 *
		 * The dynamic portion of the hook name, `$term_name`, refers to the slug of the term.
		 *
		 * @since 2.2.0
		 * @param WP_Term 			$term 	The last term
		 * @param Extras_Widget 	$this 	The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/term/last/{$term_name}", $term, $this );

		$this->render_item( 'custom-tax-archive', [
			'index'		=> $counter,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'custom-tax-archive',
			'ids' 		=> [ $post_type ],
			'content' 	=> $term->name,
			'link'		=> '',
		] );
	}

	/**
	 * Render Post Type Archive
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_post_type_archive( $post = false, $current = true ) {
		$post_type 			= false === $post ? get_post_type() : get_post_type( $post );
		$post_type_object 	= get_post_type_object( $post_type );

		/**
		 * Before Post Type Archive
		 *
		 * Fires right before a custom post type archive crumb
		 *
		 * @since 2.2.0
		 * @param object 			$post_type_object 	The queried post type object
		 * @param Extras_Widget 	$this 				The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/post_type/{$post_type}", $post_type_object, $this );

		$this->render_item( 'post-type-archive', [
			'index'		=> 1,
			'current' 	=> $current,
			'key' 		=> 'post-type-archive',
			'ids' 		=> [ $post_type ],
			'content' 	=> $post_type_object->labels->name,
			'link'		=> get_post_type_archive_link( $post_type ),
		] );
	}

	/**
	 * Render Single Non Post
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_single_non_post( $post, $query ) {
		$settings 	= $this->get_settings();
		$counter 	= 1;

		if ( '' === $settings['cpt_crumbs'] || 'both' === $settings['cpt_crumbs'] ) {
			$this->render_post_type_archive( $post, false );

			$counter++;
		}

		if ( in_array( $settings['cpt_crumbs'], [ 'terms', 'both' ] ) ) {
			$item_content = 'terms';

			$terms = Utils::get_parent_terms_highest( $post->ID );

			if ( $terms ) { $counter = 1;
				foreach ( $terms as $term ) {
					$term_slug = $term->slug;

					/**
					 * Before Taxonomy Term
					 *
					 * Fires right before a custom taxonomy term that is not
					 * the current or last one.
					 * 
					 * The dynamic portion of the hook name, `$term_slug`, refers to the slug of the term.
					 *
					 * @since 2.2.0
					 * @param WP_Term 			$term 	The queried term
					 * @param Extras_Widget 	$this 	The current widget instance
					 */
					do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/term/{$term_slug}", $term, $this );

					$this->render_item( 'post-type-terms', [
						'index'		=> $counter,
						'current' 	=> false,
						'separator'	=> true,
						'key' 		=> 'terms-' . $term->term_id,
						'ids' 		=> [ $term->term_id, $term_slug ],
						'content' 	=> $term->name,
						'link'		=> get_term_link( $term ),
					] );

					$counter++;
				}
			}

		}
	}

	/**
	 * Render Single
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_single( $post, $query ) {
		$settings = $this->get_settings();
		$post_type = get_post_type();

		/**
		 * Before Single
		 *
		 * Fires right before any single crumb
		 *
		 * @since 2.2.0
		 * @param WP_Post 			$post 		The queried single post
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( 'elementor_extras/widgets/breadcrumbs/before/crumb/single', $post, $this );

		if ( $post_type !== 'post' ) {
			$this->render_single_non_post( $post, $query );
		} else {

			$posts_page_id = get_option( 'page_for_posts' );

			if ( $posts_page_id ) {
				$blog = get_post( $posts_page_id );

				/**
				 * Before Blog Crumb
				 *
				 * Fires right before a blog main page crumb is rendered
				 *
				 * @since 2.2.0
				 * @param WP_Post 			$blog 	The blog page
				 * @param Extras_Widget 	$this 	The current widget instance
				 */
				do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/single/blog", $blog, $this );

				$this->render_item( 'blog', [
					'index'		=> 1,
					'current' 	=> false,
					'separator'	=> true,
					'key' 		=> 'blog',
					'ids' 		=> [ $blog->ID ],
					'content' 	=> $blog->post_title,
					'link'		=> get_permalink( $blog->ID ),
				] );
			}
		}

		$category = get_the_category();
		$last_category = null;

		if( ! empty( $category ) ) {
			$cat_display 	= '';
			$values 		= array_values( $category );
			$last_category 	= get_term( Utils::get_most_parents_category( $category ) );
			$cat_parents 	= array_reverse( get_ancestors( $last_category->term_id, 'category' ) );
		}

		if ( ! empty( $last_category ) ) {
			$counter = 1;

			foreach ( $cat_parents as $parent ) {
				$parent = get_term( $parent );
				$parent_slug = $parent->slug;

				if ( has_category( $parent->term_id, $post ) ) {

					/**
					 * Before Parent Category
					 *
					 * Fires right before a parent category that is not the current or last one.
					 * 
					 * The dynamic portion of the hook name, `$parent_slug`, refers to the slug of the parent category.
					 *
					 * @since 2.2.0
					 * @param WP_Term 			$parent 	The parent category
					 * @param Extras_Widget 	$this 		The current widget instance
					 */
					do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/single/category/{$parent_slug}", $parent, $this );

					$this->render_item( 'category', [
						'index'		=> $counter,
						'current' 	=> false,
						'separator'	=> true,
						'key' 		=> 'category-' . $parent->term_id,
						'ids' 		=> [ $parent->term_id, $parent_slug ],
						'content' 	=> $parent->name,
						'link'		=> get_term_link( $parent ),
					] );

					$counter++;
				}
			}

			/**
			 * Before Last Category
			 *
			 * Fires right before the last category crumb.
			 *
			 * @since 2.2.0
			 * @param WP_Term 			$last_category 	The last category in the list
			 * @param Extras_Widget 	$this 			The current widget instance
			 */
			do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/single/category/last", $last_category, $this );

			$this->render_item( 'category', [
				'index'		=> $counter,
				'current' 	=> false,
				'separator'	=> true,
				'key' 		=> 'category' . $last_category->term_id,
				'ids' 		=> [ $last_category->term_id, $last_category->slug ],
				'content' 	=> $last_category->name,
				'link'		=> get_term_link( $last_category ),
			] );

			/**
			 * Before Single
			 *
			 * Fires right before any single crumb
			 *
			 * @since 2.2.0
			 * @param WP_Post 			$post 		The queried single post
			 * @param Extras_Widget 	$this 		The current widget instance
			 */
			do_action( 'elementor_extras/widgets/breadcrumbs/before/crumb/single', $post, $this );

			$this->render_item( 'single', [
				'index'		=> $counter++,
				'current' 	=> true,
				'separator'	=> false,
				'key' 		=> 'single',
				'ids' 		=> [ $post->ID ],
				'content' 	=> get_the_title(),
			] );

		} else {
			
			/**
			 * Before Single
			 *
			 * Fires right before any single crumb
			 *
			 * @since 2.2.0
			 * @param WP_Post 			$post 		The queried single post
			 * @param Extras_Widget 	$this 		The current widget instance
			 */
			do_action( 'elementor_extras/widgets/breadcrumbs/before/crumb/single', $post, $this );

			$this->render_item( 'single', [
				'index'		=> 1,
				'current' 	=> true,
				'separator'	=> false,
				'key' 		=> 'single',
				'ids' 		=> [ $post->ID ],
				'content' 	=> get_the_title(),
			] );

		}
	}

	/**
	 * Render Category
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_category() {
		$cat_id 		= get_query_var( 'cat' );
		$category 		= get_category( $cat_id );
		$cat_parents 	= array_reverse( get_ancestors( $cat_id, 'category' ) );
		$counter 		= 1;

		foreach ( $cat_parents as $parent ) {
			$parent = get_term( $parent );
			$parent_slug = $parent->slug;

			/**
			 * Before Parent Category
			 *
			 * Fires right before a parent category that is not the current or last one.
			 * 
			 * The dynamic portion of the hook name, `$parent_slug`, refers to the slug of the parent category.
			 *
			 * @since 2.2.0
			 * @param WP_Term 			$parent 	The parent category
			 * @param Extras_Widget 	$this 		The current widget instance
			 */
			do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/category/{$parent_slug}", $parent, $this );

			$this->render_item( 'category', [
				'index'		=> $counter,
				'current' 	=> false,
				'separator'	=> true,
				'key' 		=> 'category-' . $parent->term_id,
				'ids' 		=> [ $parent->term_id, $parent_slug ],
				'content' 	=> $parent->name,
				'link'		=> get_term_link( $parent ),
			] );
			$counter++;
		}

		/**
		 * Before Current Category
		 *
		 * Fires right before the current category crumb.
		 *
		 * @since 2.2.0
		 * @param WP_Term 			$category 	The last category in the list
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/category/current", $category, $this );

		$this->render_item( 'category', [
			'index'		=> $counter,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'category',
			'ids' 		=> [ $cat_id, $category->slug ],
			'content' 	=> single_cat_title( '', false ),
		] );
	}

	/**
	 * Render Page
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_page( $post ) {
		$counter = 1;

		if ( $post->post_parent ) {
				
			$parents = get_post_ancestors( $post->ID );
			$parents = array_reverse( $parents );
				
			if ( ! isset( $parents ) ) $parents = null;

			$counter = 1;

			foreach ( $parents as $parent ) {

				/**
				 * Before Page Crumb
				 *
				 * Fires right before any page crumb.
				 *
				 * @since 2.2.0
				 * @param WP_Post 			$parent 	The page object
				 * @param Extras_Widget 	$this 		The current widget instance
				 */
				do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/page", $parent, $this );

				$this->render_item( 'ancestor', [
					'index'		=> $counter,
					'current' 	=> false,
					'separator'	=> true,
					'key' 		=> 'ancestor-' . $parent,
					'ids' 		=> [ $parent ],
					'content' 	=> get_the_title( $parent ),
					'link'		=> get_permalink( $parent ),
				] );

				$counter++;
			}
		}

		/**
		 * Before Current Page Crumb
		 *
		 * Fires right before the current page crumb.
		 *
		 * @since 2.2.0
		 * @param WP_Post 			$post 		The page object
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/page/current", $post, $this );

		$this->render_item( 'page', [
			'index'		=> $counter,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'page',
			'ids' 		=> [ $post->ID ],
			'content' 	=> get_the_title(),
		] );
	}

	/**
	 * Render Tag
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_tag() {
		$term_id 		= get_query_var('tag_id');
		$taxonomy 		= 'post_tag';
		$args 			= 'include=' . $term_id;
		$terms 			= get_terms( $taxonomy, $args );
		$tag 			= $terms[0];

		/**
		 * Before Tag Crumb
		 *
		 * Fires right before a tag crumb.
		 *
		 * @since 2.2.0
		 * @param WP_Post 			$post 		The page object
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/tag", $tag, $this );

		$this->render_item( 'tag', [
			'index'		=> 1,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'tag',
			'ids' 		=> [ $tag->term_id, $tag->slug ],
			'content' 	=> sprintf( __( 'Tag: %s', 'elementor-extras' ), $tag->name ),
		] );
	}

	/**
	 * Render Day
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_day() {
		
		$this->render_year( false );
		$this->render_month( false );

		/**
		 * Before Day Crumb
		 *
		 * Fires right before a day crumb.
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/day", $this );

		$this->render_item( 'day', [
			'index'		=> 3,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'day',
			'ids' 		=> [ get_the_time('j') ],
			'content' 	=> sprintf( __( '%1$s %2$s Archives', 'elementor-extras' ), get_the_time('F'), get_the_time('jS') ),
		] );
	}

	/**
	 * Render Month
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_month() {
		
		$this->render_year( false );

		/**
		 * Before Month Crumb
		 *
		 * Fires right before a month crumb.
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/month", $this );

		$this->render_item( 'month', [
			'index'		=> 2,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'month',
			'ids' 		=> [ get_the_time('m') ],
			'content' 	=> sprintf( __( '%s Archives', 'elementor-extras' ), get_the_time('F') ),
		] );
	}

	/**
	 * Render Year
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_year( $current = true ) {
		/**
		 * Before Year Crumb
		 *
		 * Fires right before a year crumb.
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/year", $this );

		$this->render_item( 'year', [
			'index'		=> 1,
			'current' 	=> $current,
			'separator'	=> false,
			'key' 		=> 'year',
			'ids' 		=> [ get_the_time('Y') ],
			'content' 	=> sprintf( __( '%s Archives', 'elementor-extras' ), get_the_time('Y') ),
		] );
	}

	/**
	 * Render Author
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_author() {
		global $author;

		$userdata = get_userdata( $author );

		/**
		 * Before Author Crumb
		 *
		 * Fires right before an author crumb.
		 *
		 * @since 2.2.0
		 * @param WP_User 			$post 		The queried author
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/author", $author, $this );

		$this->render_item( 'author', [
			'index'		=> 1,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'author',
			'ids' 		=> [ $userdata->user_nicename ],
			'content' 	=> sprintf( __( 'Author: %s', 'elementor-extras' ), $userdata->display_name ),
		] );
	}

	/**
	 * Render Search
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_search() {
		/**
		 * Before Search Query Crumb
		 *
		 * Fires right before an author crumb.
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/search", $this );

		$this->render_item( 'search', [
			'index'		=> 1,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> 'search',
			'content' 	=> sprintf( __( 'Search results for: %s', 'elementor-extras' ), get_search_query() ),
		] );
	}

	/**
	 * Render 404
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function render_404() {
		/**
		 * Before 404 Crumb
		 *
		 * Fires right before a 404 crumb.
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb/404", $this );

		$this->render_item( '404', [
			'index'		=> 1,
			'current' 	=> true,
			'separator'	=> false,
			'key' 		=> '404',
			'content' 	=> __( 'Page not found', 'elementor-extras' ),
		] );
	}

	/**
	 * Render Item
	 * 
	 * Gets the markup for a breadcrumb item
	 *
	 * @since  1.2.0
	 * @param  slug|string
	 * @param  args|array
	 * @return void
	 */
	protected function render_item( $slug, $args ) {

		$defaults = [
			'current' 		=> false,
			'key' 			=> false,
			'ids'			=> [],
			'content'		=> '',
			'index'			=> false,
			'link'			=> false,
		];

		$args = wp_parse_args( $args, $defaults );

		if ( $args['current'] && '' === $this->get_settings( 'show_current' ) ) {
			return;
		}

		$item_key 	= $args['key'] . '-item';
		$text_key 	= $args['key'] . '-text';
		$link_key 	= ( ! $args['current'] ) ? '-link' : '-current';
		$link_key 	= $args['key'] . $link_key;
		$link_tag 	= ( ! $args['current'] ) ? 'a' : 'strong';
		$link 		= ( ! $args['current'] ) ? ' href="' . $args['link'] .'" ' : ' ';
		$classes 	= [];

		if ( $args['current'] ) {
			$classes[] = 'ee-breadcrumbs__item--current';
		} else {
			$classes[] = 'ee-breadcrumbs__item--parent';
		}

		if ( $slug )
			$classes[] = 'ee-breadcrumbs__item--' . $slug;

		if ( $args['ids'] ) {
			foreach( $args['ids'] as $id ) {
				if ( $slug ) {
					$classes[] = 'ee-breadcrumbs__item--' . $slug . '-' . $id;
				} else { $classes[] = 'ee-breadcrumbs__item--' . $id; }
			}
		}

		$this->add_item_render_attribute( $item_key, $args['index'] );
		$this->add_render_attribute( $item_key, [
			'class' => $classes,
		] );

		$this->add_link_render_attribute( $link_key );
		$this->add_render_attribute( $text_key, [
			'itemprop' 	=> 'name',
			'class' 	=> 'ee-breadcrumbs__text',
		] );

		/**
		 * Before Crumb
		 *
		 * Fires right before any breadcrumb
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/crumb", $this );

		?><li <?php echo $this->get_render_attribute_string( $item_key ); ?>>
			<<?php echo $link_tag; ?><?php echo $link; ?><?php echo $this->get_render_attribute_string( $link_key ); ?>>
				<span <?php echo $this->get_render_attribute_string( $text_key ); ?>>
					<?php echo $args['content']; ?>
				</span>
			</<?php echo $link_tag; ?>>
		</li><?php

		if ( ! $args['current'] )
		/**
		 * After Crumb
		 *
		 * Fires right after any breadcrumb
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/after/crumb", $this );

		/**
		 * Before Separator
		 *
		 * Fires right before any separator
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/before/separator", $this );

		if ( $args['separator'] || false === $args['current'] )
			$this->render_separator();

		/**
		 * After Separator
		 *
		 * Fires right after any separator
		 *
		 * @since 2.2.0
		 * @param Extras_Widget 	$this 		The current widget instance
		 */
		do_action( "elementor_extras/widgets/breadcrumbs/after/separator", $this );
	}

	/**
	 * Add Item Render Attribute
	 * 
	 * Adds the render attributes for a specified item
	 *
	 * @since  1.2.0
	 * @param  key|string 	The render attribute key for the item
	 * @param. index|int 	The index of the item. Defaults to 0 
	 * @return void
	 */
	protected function add_item_render_attribute( $key, $index = 0 ) {

		$this->add_render_attribute( $key, [
			'class' => [
				'ee-breadcrumbs__item',
			],
			'itemprop' 	=> 'itemListElement',
			'position' 	=> $index,
			'itemscope' => '',
			'itemtype' 	=> 'http://schema.org/ListItem',
		] );
	}

	/**
	 * Add Link Render Attribute
	 * 
	 * Adds the render attributes for the item link
	 *
	 * @since  1.2.0
	 * @param  key|string 	The render attribute key for the item
	 * @return void
	 */
	protected function add_link_render_attribute( $key ) {
		$this->add_render_attribute( $key, [
			'class' => [
				'ee-breadcrumbs__crumb',
			],
			'itemprop' 	=> 'item',
			'rel' 		=> 'v:url',
			'property' 	=> 'v:title',
		] );
	}

	/**
	 * Content Template
	 * 
	 * Javascript content template for quick rendering. None in this case
	 *
	 * @since  1.2.0
	 * @return void
	 */
	protected function _content_template() {}
}
