<?php
/**
 * Template Functions
 *
 * @package   edd-invoices
 * @copyright Copyright (c) 2021, Sandhills Development, LLC
 * @license   GPL2+
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Adds the EDD Invoices templates directory to the stack.
 *
 * @param array $paths Array of template directories.
 *
 * @since 1.2
 * @return array
 */
function edd_invoices_add_template_stack( $paths ) {
	$paths[ 60 ] = EDD_INVOICES_PLUGIN_DIR . 'templates/';

	return $paths;
}
add_filter( 'edd_template_paths', 'edd_invoices_add_template_stack' );

/**
 * Outputs the invoice title.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return void
 */
function edd_invoices_do_invoice_title( $order ) {
	?>
	<!-- Title -->
	<?php
	printf(
		'<title>%s</title>',
		sprintf(
			/* translators: the invoice number */
			esc_html__( 'Invoice #%s', 'edd-invoices' ),
			esc_html( edd_get_payment_number( $order->ID ) )
		)
	);
}
add_action( 'edd_invoices_invoice_head', 'edd_invoices_do_invoice_title' );

/**
 * Loads the invoice stylesheet.
 *
 * @since 1.2
 * @return void
 */
function edd_invoices_load_stylesheet() {
	wp_enqueue_style( 'edd-invoices', EDD_INVOICES_PLUGIN_URL . 'css/standard.css', array(), EDD_INVOICES_VERSION, 'all' );
	$color = edd_get_option( 'edd-invoices-color' );
	if ( $color ) {
		$css = sprintf(
			':root{--accent-color:%1$s;--accent-color-rgba:rgba(%2$s,.2);--border-color:%1$s;}',
			$color,
			edd_invoices_convert_hex_to_rgb( $color )
		);
		wp_add_inline_style( 'edd-invoices', $css );
	}
	wp_print_styles( 'edd-invoices' );
}
add_action( 'edd_invoices_invoice_head', 'edd_invoices_load_stylesheet', 15 );

/**
 * Converts a hex color to RGB for use with rgba in CSS.
 *
 * @since 1.3
 * @param string $color
 * @return string
 */
function edd_invoices_convert_hex_to_rgb( $color ) {
	$hexcolor = str_replace( '#', '', $color );
	$rgb      = array();
	$rgb[]    = hexdec( substr( $hexcolor, 0, 2 ) );
	$rgb[]    = hexdec( substr( $hexcolor, 2, 2 ) );
	$rgb[]    = hexdec( substr( $hexcolor, 4, 2 ) );

	return implode( ',', $rgb );
}

/**
 * Outputs the invoice logo, if set.
 *
 * @since 1.2
 * @return void
 */
function edd_invoices_do_invoice_logo() {
	$logo = edd_get_option( 'edd-invoices-logo' );
	if ( ! $logo ) {
		return;
	}
	?>
	<!-- Logo -->
	<div class="logo">
		<img src="<?php echo esc_url( $logo ); ?>" alt="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>"/>
	</div>
	<?php
}
add_action( 'edd_invoices_invoice_header', 'edd_invoices_do_invoice_logo' );

/**
 * Outputs the invoice number as the heading.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return void
 */
function edd_invoices_do_invoice_order_heading( $order ) {
	?>
	<!-- Invoice Details -->
	<div class="invoice-details">
		<div class="order-number">
			<?php
			printf(
				'<h1>%s</h1>',
				sprintf(
					/* translators: the invoice number */
					esc_html__( 'Invoice %s', 'edd-invoices' ),
					esc_html( edd_get_payment_number( $order->ID ) )
				)
			);
			?>

			<?php
			switch ( $order->status ) {
				case 'publish':
				case 'complete':
				case 'edd_subscription':
					$status_label = __( 'Paid', 'edd-invoices' );
					break;

				case 'refunded':
					$status_label = __( 'Refunded', 'edd-invoices' );
					break;

				case 'partially_refunded':
					$status_label = __( 'Partially Refunded', 'edd-invoices' );
					break;

				default:
					$status_label = false;
			}

			if ( $status_label ) {
				printf(
					'<div class="payment-status-badge payment-%s">%s</div>',
					esc_attr( $order->status ),
					esc_html( $status_label )
				);
			}
			?>
		</div>

		<?php
		$date = edd_invoices_get_order_date( $order );
		if ( $date ) {
			?>
			<div class="date">
				<!-- Purchase Date -->
				<?php
				$date_format   = get_option( 'date_format' );
				$purchase_date = date_i18n( $date_format, strtotime( $date ) );

				/* Translators: %s - Date of purchase */
				printf( esc_html__( 'Purchase Date: %s', 'edd-invoices' ), esc_html( $purchase_date ) );
				?>
			</div>
			<?php
		}
		?>
	</div>
	<?php
}
add_action( 'edd_invoices_invoice_header', 'edd_invoices_do_invoice_order_heading', 11 );

/**
 * Outputs the storefront and customer contact information.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return void
 */
function edd_invoices_do_invoice_contacts( $order ) {
	edd_get_template_part( 'invoice-contacts' );
}
add_action( 'edd_invoices_invoice_contacts', 'edd_invoices_do_invoice_contacts' );

/**
 * Outputs the invoice items table.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return void
 */
function edd_invoices_do_invoice_items_table( $order ) {
	edd_get_template_part( 'invoice-table' );
}
add_action( 'edd_invoices_invoice_items_table', 'edd_invoices_do_invoice_items_table' );

/**
 * Outputs the additional info section for the invoice.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return void
 */
function edd_invoices_invoice_do_additional_info( $order ) {
	edd_get_template_part( 'invoice-additional-info' );
}
add_action( 'edd_invoices_invoice_additional_info', 'edd_invoices_invoice_do_additional_info' );

/**
 * Outputs action buttons (print, back) for the invoice.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return void
 */
function edd_invoices_do_invoice_action_buttons( $order ) {
	$home_button = ! empty( $_GET['home'] );
	?>
	<div class="invoice-actions" data-html2canvas-ignore>
		<?php
		if ( $home_button ) {
			?>
			<a class="button home" href="<?php echo esc_url( home_url() ); ?>"><?php esc_html_e( 'Home', 'edd-invoices' ); ?></a>
			<?php
		} else {
			?>
			<button class="button back" onclick="window.history.back()"><?php esc_html_e( 'Back', 'edd-invoices' ); ?></button>
			<?php
		}
		?>
		<button class="button print" onclick="window.print()"><?php esc_html_e( 'Print', 'edd-invoices' ); ?></button>
		<button class="button pdf" data-name="<?php echo esc_attr( edd_invoices_get_pdf_filename( $order->ID ) ); ?>"><?php esc_html_e( 'Download PDF', 'edd-invoices' ); ?></button>
	</div>
	<?php
}
add_action( 'edd_invoices_invoice_footer', 'edd_invoices_do_invoice_action_buttons', 20 );

/**
 * If the current user can edit the invoice, output an edit button.
 *
 * @since 1.3.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return void
 */
function edd_invoices_do_invoice_edit_button( $order ) {
	if ( ! edd_invoices_can_user_edit_invoice_data( get_current_user_id() ) ) {
		return;
	}
	?>
	<a href="<?php echo esc_url( edd_invoices_get_invoice_form_url( $order->ID ) ); ?>" class="button hide-on-print" data-html2canvas-ignore>
		<?php
		printf(
		/* Translators: %1$s - opening <span> tag; %2$s - closing </span> tag */
			__( 'Update %1$sbilling information%2$s', 'edd-invoices' ),
			'<span class="screen-reader-text">',
			'</span>'
		);
		?>
	</a>
	<?php
}
