<?php
/**
 * Invoice Functions
 *
 * @package   easy-digital-downloads
 * @copyright Copyright (c) 2021, Sandhills Development, LLC
 * @license   GPL2+
 * @since     1.2
 */

/**
 * Retrieves the invoice URL for a given order.
 *
 * @param int    $order_id ID of the order.
 * @param bool   $new_tab  Whether the link is opening in a new tab (admin links, emails).
 *                         If true, a "Home" link will be added to the invoice, replacing the "Back" button.
 *
 * @since 1.2
 * @return string
 */
function edd_invoices_get_invoice_url( $order_id, $new_tab = false ) {
	$args = array(
		'edd_action' => 'view_invoice',
		'payment_id' => urlencode( $order_id ),
	);
	$key   = false;
	if ( function_exists( 'edd_get_order' ) ) {
		$order = edd_get_order( intval( $order_id ) );
		if ( ! empty( $order->payment_key ) ) {
			$key = $order->payment_key;
		}
	} else {
		$order = edd_get_payment( intval( $order_id ) );
		if ( ! empty( $order->payment_meta['key'] ) ) {
			$key = $order->payment_meta['key'];
		}
	}
	if ( $key && ! empty( $order->email ) ) {
		$args['invoice'] = urlencode( md5( $order_id . $key . $order->email ) );
	}
	if ( $new_tab ) {
		$args['home'] = true;
	}

	return add_query_arg(
		$args,
		trailingslashit( home_url() )
	);
}

/**
 * Retrieves the URL to edit the data associated with an invoice.
 *
 * @param int $order_id ID of the order.
 *
 * @since 1.3
 * @return string Returns an empty string if the invoice page is not set.
 */
function edd_invoices_get_invoice_form_url( $order_id ) {
	if ( ! edd_get_option( 'edd-invoices-page' ) ) {
		return '';
	}

	$args  = array(
		'payment_id' => urlencode( $order_id ),
	);
	$key   = false;
	if ( function_exists( 'edd_get_order' ) ) {
		$order = edd_get_order( intval( $order_id ) );
		if ( ! empty( $order->payment_key ) ) {
			$key = $order->payment_key;
		}
	} else {
		$order = edd_get_payment( intval( $order_id ) );
		if ( ! empty( $order->payment_meta['key'] ) ) {
			$key = $order->payment_meta['key'];
		}
	}
	if ( $key && ! empty( $order->email ) ) {
		$args['invoice'] = urlencode( md5( $order_id . $key . $order->email ) );
	}

	return add_query_arg(
		$args,
		get_permalink( edd_get_option( 'edd-invoices-page' ) )
	);
}

/**
 * Returns an array of order statuses that support invoice generation.
 *
 * @since 1.2
 * @return array
 */
function edd_invoices_get_invoiceable_order_statuses() {
	$statuses = array(
		'publish',
		'complete',
		'revoked',
		'refunded',
		'partially_refunded',
	);

	/**
	 * Filters the order statuses that support invoice generation.
	 *
	 * @param array $statuses
	 */
	return apply_filters( 'edd_invoices_acceptable_payment_statuses', $statuses );
}

/**
 * Gets the order billing address.
 *
 * @param EDD_Payment $payment EDD_Payment object.
 * @return array
 * @since 1.2
 */
function edd_invoices_get_order_address( $payment ) {
	$address = array_fill_keys(
		array( 'name', 'line1', 'line2', 'city', 'state', 'zip', 'country' ),
		''
	);

	if ( function_exists( 'edd_get_order_addresses' ) ) {
		$addresses = edd_get_order_addresses(
			array(
				'order_id' => $payment->ID,
				'type'     => 'billing',
				'number'   => 1,
			)
		);
		if ( ! empty( $addresses[0] ) ) {
			$address = reset( $addresses );

			return array(
				'id'      => $address->id,
				'name'    => $address->name,
				'line1'   => $address->address,
				'line2'   => $address->address2,
				'city'    => $address->city,
				'state'   => $address->region,
				'zip'     => $address->postal_code,
				'country' => $address->country,
			);
		}

		// We can at least attempt to get the name from the customer record.
		$customer        = edd_get_customer( $payment->customer_id );
		$address['name'] = $customer->name;
	} else {
		$address = $payment->address;
		if ( ! empty( $payment->user_info['first_name'] ) ) {
			$address['name'] = $payment->user_info['first_name'];
			$payment_meta    = get_post_meta( $payment->ID, '_edd_payment_meta', true );
			if ( ! empty( $payment_meta['user_info']['last_name'] ) ) {
				$address['name'] .= ' ' . $payment_meta['user_info']['last_name'];
			}
		} else {
			$customer        = new EDD_Customer( $payment->customer_id );
			$address['name'] = $customer->name;
		}
	}

	return $address;
}

/**
 * Gets the order custom meta.
 *
 * @param EDD_Payment $payment EDD_Payment object.
 * @param string $key     The meta key to retrieve.
 * @return bool|string
 * @since 1.2
 */
function edd_invoices_get_custom_order_meta( $payment, $key ) {
	if ( function_exists( 'edd_get_order_meta' ) ) {
		return edd_get_order_meta( $payment->ID, $key, true );
	}

	$key = str_replace( 'invoices_', '', $key );

	return ! empty( $payment->user_info['address'][ $key ] ) ? $payment->user_info ['address'][ $key ] : '';
}

/**
 * Gets the order items data specific to the invoice.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return array
 */
function edd_invoices_get_order_items( $order ) {
	$invoice_items = array();

	// EDD 3.0
	if ( function_exists( 'edd_get_order_items' ) ) {
		$items = edd_get_order_items(
			array(
				'order_id' => $order->id,
			)
		);
		foreach ( $items as $key => $item ) {
			$invoice_items[ $key ] = array(
				'item_id'  => intval( $item->product_id ),
				'price'    => $item->amount,
				'name'     => $item->product_name,
				'price_id' => intval( $item->price_id ),
			);
			if ( edd_get_order_meta( $order->id, '_edd_sl_is_renewal', true ) && $item->discount > 0 ) {
				$invoice_items[ $key ]['name'] .= '<div class="renewal-discount">' . __( 'License Renewal Discount:', 'edd-invoices' ) . ' ' . edd_currency_filter( edd_format_amount( edd_negate_amount( $item->discount ) ) ) . '</div>';
			}
		}
	} else {

		// EDD 2.x
		foreach ( $order->cart_details as $key => $item ) {
			$name     = ! empty( $item['name'] ) ? $item['name'] : __( 'Product', 'edd-invoices' );
			$item_id  = ! empty( $item['id'] ) ? $item['id'] : 0;
			$price_id = null;
			if ( ! empty( $item['item_number']['options']['price_id'] ) ) {
				$price_id = $item['item_number']['options']['price_id'];
			}
			if ( null !== $price_id ) {
				$name .= ' — ' . edd_get_price_option_name( $item_id, $price_id );
			}
			$invoice_items[ $key ] = array(
				'item_id'  => $item_id,
				'price'    => isset( $item['price'] ) ? $item['price'] : '0.00',
				'name'     => $name,
				'price_id' => $price_id,
			);
			if ( edd_get_payment_meta( $order->ID, '_edd_sl_is_renewal', true ) && $item['discount'] > 0 ) {
				$invoice_items[ $key ]['name'] .= '<div class="renewal-discount">' . __( 'License Renewal Discount:', 'edd-invoices' ) . ' ' . edd_currency_filter( edd_format_amount( $item['discount'] ) ) . '</div>';
			}
		}
	}

	return $invoice_items;
}

/**
 * Gets the order discounts.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return bool|array
 */
function edd_invoices_get_order_discounts( $order ) {
	// EDD 3.0
	if ( function_exists( 'edd_get_order_adjustments' ) ) {
		$discounts = edd_get_order_adjustments(
			array(
				'object_id'   => $order->id,
				'object_type' => 'order',
				'type'        => 'discount',
			)
		);
		if ( ! $discounts ) {
			return false;
		}
		$output = array();
		foreach ( $discounts as $discount ) {
			$name = $discount->description;
			if ( 'percent' === edd_get_discount_type( $discount->type_id ) ) {
				$rate  = edd_format_discount_rate( 'percent', edd_get_discount_amount( $discount->type_id ) );
				$name .= "&nbsp;({$rate})";
			}
			$output[ $discount->id ] = array(
				'name'   => __( 'Discount', 'edd-invoices' ) . ' &mdash; ' . $name,
				'amount' => edd_currency_filter( edd_format_amount( edd_negate_amount( $discount->total ) ) ),
			);
		}

		return $output;
	}

	// EDD 2.x
	if ( empty( $order->discounts ) || 'none' === $order->discounts ) {
		return false;
	}
	$discounts = explode( ',', $order->discounts );

	return array(
		array(
			'name'   => _n( 'Discount', 'Discounts', count( $discounts ), 'edd-invoices' ),
			'amount' => implode( ', ', $discounts ),
		),
	);
}

/**
 * Gets the order date for the invoice.
 *
 * @since 1.2
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return string|bool
 */
function edd_invoices_get_order_date( $order ) {
	// EDD 3.0 `date_completed`.
	if ( ! empty( $order->date_completed ) ) {
		return $order->date_completed;
	}

	// EDD 2.x uses `completed_date`.
	if ( ! empty( $order->completed_date ) ) {
		return $order->completed_date;
	}

	// Subscriptions don't have a completed date in 2.x.
	if ( ! empty( $order->date ) ) {
		return $order->date;
	}

	return false;
}

/**
 * Gets the tax rate for the order.
 *
 * @since 1.2.1
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return float|bool
 */
function edd_invoices_get_tax_rate( $order ) {
	if ( method_exists( $order, 'get_tax_rate' ) ) {
		$rate = $order->get_tax_rate();
	} else {
		$rate = $order->tax_rate * 100;
	}
	return ! empty( $rate ) ? floatval( $rate ) : false;
}

/**
 * Gets the filename for the generated PDF.
 *
 * @since  1.3
 * @param int $order_id The order ID.
 * @return string
 */
function edd_invoices_get_pdf_filename( $order_id ) {
	/**
	 * Optionally filter the file name. `.pdf` will be appended after the filter is applied, so should not be included in the filter.
	 *
	 * @since 1.3
	 * @param $filename The name of the file. Default is `invoice-payment_number`.
	 * @param $order_id The order ID.
	 */
	$filename = apply_filters( 'edd_invoices_pdf_filename', 'invoice-' . edd_get_payment_number( $order_id ), $order_id );

	return $filename . '.pdf';
}

/*
 * Gets the company address.
 * Returns the full company address if set; otherwise, compiles the original
 * address settings and creates the full address, and deletes the old settings.
 *
 * @since 1.3
 * @return string
 */
function edd_invoices_get_company_address() {
	$address = edd_get_option( 'edd-invoices-company-address' );
	if ( $address ) {
		return $address;
	}
	$address = array();
	$keys    = array( 'address', 'address2', 'city', 'zipcode', 'country' );
	foreach ( $keys as $key ) {
		$line = edd_get_option( 'edd-invoices-' . $key );
		if ( $line ) {
			$address[] = $line;
			edd_delete_option( "edd-invoices-{$key}" );
		}
	}
	if ( ! empty( $address ) ) {
		$address = implode( "\n", $address );
		edd_update_option( 'edd-invoices-company-address', $address );

		return $address;
	}

	return '';
}

/**
 * Gets the body class attribute for the invoices.
 *
 * @since 1.3
 * @param \EDD\Orders\Order|EDD_Payment $order The order/payment object.
 * @return string
 */
function edd_invoices_get_invoice_body_class( $order ) {
	$body_classes = array(
		strtolower( $order->status ),
		edd_get_option( 'edd-invoices-style' ),
	);

	return implode( ' ', array_map( 'sanitize_html_class', array_filter( $body_classes ) ) );
}

add_action( 'edd_30_migrate_order', 'edd_invoices_30_migration', 10, 3 );
/**
 * During the EDD 3.0 migration, copies the custom invoices meta to the new order meta table.
 *
 * @since 1.2
 * @param int   $order_id      The new order ID.
 * @param array $payment_meta  The original payment meta.
 * @param array $name          The original post meta.
 * @return void
 */
function edd_invoices_30_migration( $order_id, $payment_meta, $meta ) {

	if ( empty( $meta['_edd_payment_meta'] ) ) {
		return;
	}
	$meta      = maybe_unserialize( $meta['_edd_payment_meta'][0] );
	$user_info = $meta['user_info'];
	$meta_keys = array( 'vat', 'notes', 'company' );
	foreach ( $meta_keys as $key ) {
		if ( ! empty( $user_info['address'][ $key ] ) ) {
			edd_update_order_meta( $order_id, "invoices_{$key}", sanitize_text_field( $user_info['address'][ $key ] ) );
		}
	}
}

/**
 * Whether the current user is permitted to edit the customer billing information on the invoice.
 *
 * Warning: This does not check whether the user has permission to even view the invoice in question,
 * so that permission check should be handled separately.
 *
 * @see EDD_Invoice_Generator::current_user_can_view_invoice
 * @see EDD_Invoice_Generator::validate_request (calls the above)
 *
 * The general assumption here is that the user ID passed in is either that of an
 * administrator, or that of the customer who "owns" the invoice.
 *
 * @link https://github.com/easydigitaldownloads/edd-invoices/issues/120
 *
 * @since 1.3
 *
 * @param int $user_id ID of the user to check.
 *
 * @return bool
 */
function edd_invoices_can_user_edit_invoice_data( $user_id ) {
	// In the event that a `0` user ID is passed.
	if ( $user_id <= 0 ) {
		return false;
	}

	// If there's no invoice page, no one can edit -- not even admins.
	if ( ! edd_get_option( 'edd-invoices-page' ) ) {
		return false;
	}

	// If editing isn't disabled, everyone can edit.
	if ( ! edd_get_option( 'edd-invoices-disable-editing' ) ) {
		return true;
	}

	// If we're here, that means editing has been disabled, so only high privileged users can edit.
	return user_can( $user_id, 'manage_shop_settings' );
}
