<?php
/**
 * Invoice Generator
 *
 * @package   edd-invoice
 * @copyright Copyright (c) 2021, Sandhills Development, LLC
 * @license   GPL2+
 * @since     1.2
 */

class EDD_Invoice_Generator {

	/**
	 * Holds any error messages.
	 *
	 * @var WP_Error
	 * @since 1.2
	 */
	protected $errors;

	/**
	 * Payment object for the current invoice.
	 *
	 * @var \EDD\Orders\Order|EDD_Payment|false
	 * @since 1.2
	 */
	public $order;

	/**
	 * EDD_Invoice_Generator constructor.
	 *
	 * @since 1.2
	 */
	public function __construct() {
		$this->errors = new WP_Error();
	}

	/**
	 * Adds a new error.
	 *
	 * @param string $message Error message to display to the user.
	 * @param string $code    Internal error ID.
	 *
	 * @since 1.2
	 * @return void
	 */
	protected function add_error( $message, $code = 'invalid_request' ) {
		$this->errors->add( $code, $message );
	}

	/**
	 * Returns an array of error messages.
	 *
	 * @since 1.2
	 * @return array
	 */
	public function get_error_messages() {
		return $this->errors->get_error_messages();
	}

	/**
	 * Validates the current request.
	 *
	 * @since 1.2
	 * @return void
	 */
	public function validate_request() {
		try {
			$this->get_payment_from_request();
		} catch ( Exception $e ) {
			$this->add_error( __( 'Invalid payment ID specified.', 'edd-invoices' ), 'invalid_payment_id' );

			return;
		}

		if ( ! $this->current_user_can_view_invoice() ) {
			$this->add_error( __( 'You do not have permission to view this invoice.', 'edd-invoices' ), 'permission_denied' );
		}
	}

	/**
	 * Whether or not the request is valid.
	 *
	 * If true, the user can proceed to view the invoice.
	 * If false, the user either doesn't have permission
	 *
	 * @since 1.2
	 * @return bool
	 */
	public function is_valid_request() {
		// WP_Error::has_errors() was introducted in WordPress 5.1
		if ( method_exists( $this->errors, 'has_errors' ) ) {
			return ! $this->errors->has_errors();
		}

		return empty( $this->errors->errors );
	}

	/**
	 * Attempts to parse the payment ID from the URL.
	 *
	 * @since 1.2
	 * @return void
	 * @throws Exception
	 */
	private function get_payment_from_request() {
		$payment_id = false;

		if ( isset( $_GET['payment_key'] ) ) {
			$payment_id = edd_get_purchase_id_by_key( urldecode( $_GET['payment_key'] ) );
		}

		if ( isset( $_GET['payment_id'] ) ) {
			$payment_id = urldecode( $_GET['payment_id'] );
		}

		if ( empty( $payment_id ) ) {
			throw new Exception();
		}

		// Now get the actual order (EDD 3.0) or payment (EDD 2.x) object.
		if ( function_exists( 'edd_get_order' ) ) {
			$this->order = edd_get_order( intval( $payment_id ) );
		} else {
			$this->order = edd_get_payment( intval( $payment_id ) );
		}

		if ( empty( $this->order->ID ) || $this->order->ID != $payment_id ) {
			throw new Exception();
		}
	}

	/**
	 * Determines whether or not the current user can view the current invoice.
	 *
	 * @since 1.2
	 * @return bool
	 */
	private function current_user_can_view_invoice() {
		$user_id       = $this->order->user_id;
		$user_can_view = false;

		if ( is_user_logged_in() && $user_id == get_current_user_id() ) {
			// Current user is the owner of the payment.
			$user_can_view = true;
		} elseif ( ( $user_id == 0 || $user_id == '-1' ) && ! is_user_logged_in() && edd_get_purchase_session() ) {
			// User is not logged in but they have session data available.
			$user_can_view = true;
		} elseif ( current_user_can( 'view_shop_sensitive_data' ) ) {
			// User can view sensitive shop data. (admins, etc)
			$user_can_view = true;
		}

		// User is not logged in, but is using the link with the correct ID and hashed invoice key.
		if ( ! is_user_logged_in() ) {
			$id         = (int) filter_input( INPUT_GET, 'payment_id', FILTER_SANITIZE_NUMBER_INT );
			$order_hash = ! empty( $_GET['invoice'] ) ? sanitize_text_field( $_GET['invoice'] ) : false;
			if ( $id === (int) $this->order->ID && $order_hash ) {
				$key           = ! empty( $this->order->payment_key ) ? $this->order->payment_key : $this->order->payment_meta['key'];
				$user_can_view = hash_equals( md5( $this->order->ID . $key . $this->order->email ), $order_hash );
			}
		}

		return $user_can_view;
	}

	/**
	 * Saves billing information from the request.
	 *
	 * @todo Improve support for EDD 3.0
	 *
	 * @since 1.2
	 * @return void
	 * @throws Exception
	 */
	public function save_billing_information() {
		if ( empty( $this->order->ID ) ) {
			throw new Exception( __( 'Missing payment ID.', 'edd-invoices' ) );
		}
		$address = array_map( 'sanitize_text_field', $_POST['edd-payment-address'] );
		$name    = ! empty( $_POST['edd-payment-user-name'] ) ? sanitize_text_field( $_POST['edd-payment-user-name'] ) : '';
		$vat     = ! empty( $_POST['edd-payment-user-name'] ) ? sanitize_text_field( $_POST['edd-payment-vat'] ) : '';
		$notes   = ! empty( $_POST['edd-payment-user-name'] ) ? sanitize_text_field( $_POST['edd-payment-notes'] ) : '';
		$company = ! empty( $_POST['edd-payment-user-name'] ) ? sanitize_text_field( $_POST['edd-payment-company'] ) : '';

		if ( function_exists( 'edd_update_order_address' ) ) {
			$address['name']        = $name;
			$address['address']     = $address['line1'];
			$address['address2']    = $address['line2'];
			$address['region']      = $address['state'];
			$address['postal_code'] = $address['zip'];
			if ( ! empty( $_POST['address-id'] ) ) {
				edd_update_order_address( absint( $_POST['address-id'] ), $address );
			} else {
				$address['order_id'] = $this->order->ID;
				edd_add_order_address( $address );
			}
		} else {
			$meta                            = edd_get_payment_meta( $this->order->ID );
			$user_info                       = edd_get_payment_meta_user_info( $this->order->ID );
			$user_info['first_name']         = $name;
			$user_info['last_name']          = ''; // clearing because `first_name` is now the full name.
			$user_info['address']            = $address;
			$user_info['address']['vat']     = $vat;
			$user_info['address']['notes']   = $notes;
			$user_info['address']['company'] = $company;
			$meta['user_info']               = $user_info;

			edd_update_payment_meta( $this->order->ID, '_edd_payment_meta', $meta );

			// Refresh the payment object so we have fresh information to display in the invoice.
			wp_cache_delete( md5( 'edd_payment' . $this->order->ID ), 'payments' );
			$this->order = edd_get_payment( $this->order->ID );
		}

		if ( function_exists( 'edd_update_order_meta' ) ) {
			edd_update_order_meta( $this->order->ID, 'invoices_vat', sanitize_text_field( $vat ) );
			edd_update_order_meta( $this->order->ID, 'invoices_notes', sanitize_text_field( $notes ) );
			edd_update_order_meta( $this->order->ID, 'invoices_company', sanitize_text_field( $company ) );
		}
	}

}
