<?php
/**
 * Settings
 *
 * @package   edd-invoices
 * @copyright Copyright (c) 2021, Sandhills Development, LLC
 * @license   GPL2+
 * @since     1.2
 */

/**
 * Registers a new settings section for "Invoices".
 *
 * @param array $sections
 *
 * @since 1.2
 * @return array
 */
function edd_invoices_register_settings_section( $sections ) {
	$sections['edd-invoices'] = __( 'Invoices', 'edd-invoices' );

	return $sections;
}
$settings_tab = version_compare( EDD_VERSION, '2.11.3', '<' ) ? 'extensions' : 'gateways';
add_filter( "edd_settings_sections_{$settings_tab}", 'edd_invoices_register_settings_section', 10 );

/**
 * Registers Invoices settings.
 *
 * @param array $settings An array of EDD Plugin and extension settings.
 *
 * @since 1.2
 * @return array
 */
function edd_invoices_register_settings( $settings ) {
	$invoice_settings = array(
		'edd-invoices-disable-editing' => array(
			'id'   => 'edd-invoices-disable-editing',
			'name' => __( 'Disable Address Editing?', 'edd-invoices' ),
			'desc' => sprintf(
				__( 'If checked, customers will not be permitted to change their billing information (name, address) that\'s displayed on the invoice. Users with the %s capability can always edit this information.', 'edd-invoices' ),
				'<code>manage_shop_settings</code>'
			),
			'type' => 'checkbox',
		)
	);

	$invoice_settings['edd-invoices-page']            =
		array(
			'id'      => 'edd-invoices-page',
			'name'    => __( 'Invoice Page', 'edd-invoices' ),
			'desc'    => __( 'Which Page contains the [edd_invoices] shortcode? This page is used to display the form customers can use to update how their billing information appears on the invoice.', 'edd-invoices' ),
			'type'    => 'select',
			'options' => edd_get_pages(),
		);
	$invoice_settings['edd-invoices-logo']            =
		array(
			'id'   => 'edd-invoices-logo',
			'name' => __( 'Logo URL', 'edd-invoices' ),
			'type' => 'upload',
			'size' => 'regular',
		);
	$invoice_settings['edd-invoices-styles']          =
		array(
			'id'   => 'edd-invoices-styles',
			'name' => __( 'Invoice Style', 'edd-invoices' ),
			'type' => 'invoices_styles',
		);
	$invoice_settings['edd-invoices-company-name']    =
		array(
			'id'   => 'edd-invoices-company-name',
			'name' => __( 'Company Name', 'edd-invoices' ),
			'desc' => __( 'Company Name shown on Invoices', 'edd-invoices' ),
			'type' => 'text',
			'size' => 'regular',
		);
	$invoice_settings['edd-invoices-company-address'] =
		array(
			'id'   => 'edd-invoices-company-address',
			'name' => __( 'Company Address', 'edd-invoices' ),
			'desc' => __( 'This address block will display on your invoices.', 'edd-invoices' ),
			'type' => 'textarea',
			'std'  => edd_invoices_get_company_address(),
		);
	$invoice_settings['edd-invoices-number']          =
		array(
			'id'   => 'edd-invoices-number',
			'name' => __( 'Registration Number', 'edd-invoices' ),
			'desc' => __( 'Company Registration Number', 'edd-invoices' ),
			'type' => 'text',
			'size' => 'regular',
		);
	$invoice_settings['edd-invoices-vat']             =
		array(
			'id'   => 'edd-invoices-tax',
			'name' => __( 'Tax/VAT Number', 'edd-invoices' ),
			'desc' => __( 'Company Tax/VAT Number', 'edd-invoices' ),
			'type' => 'text',
			'size' => 'regular',
		);
	$invoice_settings['edd-invoices-additional-text'] =
		array(
			'id'   => 'edd-invoices-additional-text',
			'name' => __( 'Additional Text', 'edd-invoices' ),
			'desc' => __( 'Any text entered here will appear at the bottom of each invoice', 'edd-invoices' ),
			'type' => 'rich_editor',
			'size' => '10',
		);

	if ( version_compare( EDD_VERSION, 2.5, '>=' ) ) {
		$invoice_settings = array( 'edd-invoices' => $invoice_settings );
	}

	return array_merge( $settings, $invoice_settings );
}
add_filter( "edd_settings_{$settings_tab}", 'edd_invoices_register_settings', 1 );

/**
 * Displays the combined settings for the invoice style selector and color picker.
 *
 * @since 1.3
 * @param array $args
 * @return void
 */
function edd_invoices_styles_callback( $args ) {
	?>
	<div class="edd-invoices-styles" style="display:flex;gap:12px;">
		<div class="edd-invoices-styles__style">
		<?php
		edd_select_callback(
			array(
				'id'          => 'edd-invoices-style',
				'name'        => __( 'Invoice Style', 'edd-invoices' ),
				'type'        => 'select',
				'multiple'    => false,
				'options'     => array(
					''        => __( 'Default', 'edd-invoices' ),
					'minimal' => __( 'Minimal', 'edd-invoices' ),
					'lines'   => __( 'Lines', 'edd-invoices' ),
				),
				'desc'        => '',
				'field_class' => 'edd-invoices-style',
			)
		);
		?>
		</div>
		<div class="edd-invoices-styles__colors" style="display:none;">
		<?php
		edd_color_callback(
			array(
				'id'          => 'edd-invoices-color',
				'name'        => __( 'Accent Color', 'edd-invoices' ),
				'type'        => 'color',
				'desc'        => __( 'The accent color is not available for the minimal invoice style.', 'edd-invoices' ),
				'field_class' => '',
			)
		);
		?>
		</div>
	</div>
	<?php
}

add_action( 'admin_enqueue_scripts', 'edd_invoices_settings_script', 100 );
/**
 * Adds a small inline script to dynamically enable/disable the color picker based on the invoice style.
 *
 * @since 1.3
 * @return void
 */
function edd_invoices_settings_script() {
	$screen = get_current_screen();
	if ( 'download_page_edd-settings' !== $screen->id ) {
		return;
	}
	$script = "jQuery(document).ready(function($) {
		var select = $( 'select.edd-invoices-style' ),
		    accentWrapper = $( '.edd-invoices-styles__colors' ),
		    button = accentWrapper.find( 'button.wp-color-result' );

		edd_invoices_color_picker( select.val() );
		$( select ).on( 'change', function() {
			edd_invoices_color_picker( $( this ).val() );
		} );

		function edd_invoices_color_picker( value ) {
			if ( 'minimal' === value ) {
				accentWrapper.fadeOut();
				button.attr( 'disabled', true );
			} else {
				accentWrapper.fadeIn();
				button.attr( 'disabled', false );
			}
		}
	} );";
	if ( function_exists( 'wp_add_inline_script' ) ) {
		$handle = function_exists( 'edd_get_orders' ) ? 'edd-admin-settings' : 'edd-admin-scripts';
		wp_add_inline_script( $handle, $script );
	} else {
		wp_print_scripts( 'jquery' );
		echo "<script>{$script}</script>";
	}
}
