<?php
/**
 * Actions
 *
 * @package   edd-invoices
 * @copyright Copyright (c) 2021, Sandhills Development, LLC
 * @license   GPL2+
 * @since     1.2
 */

/**
 * Generates, saves user details from the form, and displays the invoice to the user.
 *
 * @since 1.2
 */
function edd_invoices_generate_invoice() {
	// Check invoice generation was requested
	if ( ! isset( $_REQUEST['edd-invoices-nonce'] ) ) {
		return;
	}

	if ( ! wp_verify_nonce( $_REQUEST['edd-invoices-nonce'], 'edd-invoices-generate-invoice' ) ) {
		return;
	}

	if ( ! edd_invoices_can_user_edit_invoice_data( get_current_user_id() ) ) {
		return;
	}

	$generator = new EDD_Invoice_Generator();

	$generator->validate_request();

	if ( ! $generator->is_valid_request() ) {
		wp_die( implode( '<br>', $generator->get_error_messages() ) );
	}

	// Save user details from POST.
	try {
		$generator->save_billing_information();
	} catch ( Exception $e ) {
		wp_die( esc_html( $e->getMessage() ) );
	}

	// Redirect to viewing the invoice.
	wp_safe_redirect( edd_invoices_get_invoice_url( $generator->order->ID ) );
	exit;
}

add_action( 'init', 'edd_invoices_generate_invoice' );

/**
 * Displays the invoice.
 *
 * @since 1.3
 */
function edd_invoices_display_invoice() {
	$generator = new EDD_Invoice_Generator();

	$generator->validate_request();

	if ( ! $generator->is_valid_request() ) {
		wp_die( implode( '<br>', $generator->get_error_messages() ) );
	}

	$order = $generator->order;

	// Generate HTML
	set_query_var( 'order', $order );

	edd_get_template_part( 'invoice' );
	die();
}
add_action( 'edd_view_invoice', 'edd_invoices_display_invoice' );

/**
 * Adds an "Invoice" header to the order history table.
 *
 * @since 1.2
 */
function edd_invoices_order_history_header() {
	if ( ! edd_get_option( 'edd-invoices-page' ) ) {
		return;
	}
	?>
	<th class="edd_invoice"><?php _e( 'Invoice', 'edd-invoices' ); ?></th>
	<?php
}

add_action( 'edd_purchase_history_header_after', 'edd_invoices_order_history_header', 1 );

/**
 * Displays a "View Invoice" link for applicable orders.
 *
 * @param \EDD\Orders\Order|int   $order_or_order_id In EDD 3.0, this is the order object; in 2.x, it is the payment ID.
 * @param array                   $purchase_data     The array of purchase data (not used in EDD 3.0).
 *
 * @since 1.2
 */
function edd_invoices_maybe_display_generate_invoice_link( $order_or_order_id, $purchase_data = array() ) {
	if ( ! edd_get_option( 'edd-invoices-page' ) ) {
		return;
	}

	if ( $order_or_order_id instanceof \EDD\Orders\Order ) {
		$order_id = $order_or_order_id->id;
		remove_action( 'edd_purchase_history_row_end', 'edd_invoices_maybe_display_generate_invoice_link', 1, 2 );
	} else {
		$order_id = $order_or_order_id;
	}

	if ( ! in_array( edd_get_payment_status( $order_id ), edd_invoices_get_invoiceable_order_statuses() ) ) {
		?>
		<td class="edd_invoice"></td>
		<?php
		return;
	}
	?>
	<td class="edd_invoice">
		<a href="<?php echo esc_url( edd_invoices_get_invoice_url( $order_id ) ); ?>"><?php esc_html_e( 'View Invoice', 'edd-invoices' ); ?></a>
	</td>
	<?php
}
add_action( 'edd_order_history_row_end', 'edd_invoices_maybe_display_generate_invoice_link', 1 );
add_action( 'edd_purchase_history_row_end', 'edd_invoices_maybe_display_generate_invoice_link', 1, 2 );

/**
 * Adds a "View Invoice" link to the end of the purchase receipt table.
 *
 * @param \EDD\Orders\Order|EDD_Payment $order_or_payment Order (EDD 3.0) or payment (2.x) object.
 * @param array                         $args             Receipt arguments.
 *
 * @since 1.2
 * @return void
 */
function edd_invoices_maybe_add_receipt_invoice_link( $order_or_payment, $args ) {
	if ( ! edd_get_option( 'edd-invoices-page' ) ) {
		return;
	}
	if ( $order_or_payment instanceof \EDD\Orders\Order ) {
		$order_id = $order_or_payment->id;
	} else {
		$order_id = $order_or_payment->ID;
	}

	if ( empty( $order_id ) || ! in_array( edd_get_payment_status( $order_id ), edd_invoices_get_invoiceable_order_statuses() ) ) {
		return;
	}
	?>
	<tr>
		<td><strong><?php esc_html_e( 'Invoice:', 'edd-invoices' ); ?></strong></td>
		<td>
			<a href="<?php echo esc_url( edd_invoices_get_invoice_url( $order_id ) ); ?>"><?php esc_html_e( 'View Invoice', 'edd-invoices' ); ?></a>
		</td>
	</tr>
	<?php
}
$hook = 'edd_payment_receipt_after';
if ( function_exists( 'edd_get_orders' ) ) {
	$hook = 'edd_order_receipt_after';
}
add_action( $hook, 'edd_invoices_maybe_add_receipt_invoice_link', 10, 2 );
