<?php
/**
 * Plugin Name: Easy Digital Downloads - Invoices
 * Plugin URI: https://persianscript.ir
 * Version: 1.3.2
 * Author: پرشین اسکریپت
 * Author URI: https://persianscript.ir
 * Description: Display HTML Invoices for EDD
 * License: GPL2
 */

/*  Copyright 2021 Easy Digital Downloads (email : support@easydigitaldownloads.com)
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2, as
    published by the Free Software Foundation.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * EDD Invoices Class
 *
 * @package    EDD
 * @subpackage EDD Invoices
 * @author     Easy Digital Downloads
 * @copyright  Easy Digital Downloads
 */
class EDDInvoices {

	/**
	 * Object containing meta information about the plugin (slug, name, version).
	 * @see EDDInvoices::get_plugin_details()
	 *
	 * @deprecated 1.2 In favour of constants.
	 *
	 * @var stdClass
	 */
	private $plugin;

	/**
	 * Array of EDD settings.
	 *
	 * @deprecated 1.2 In favour of `edd_get_option()`
	 *
	 * @var array|false
	 */
	private $settings;

	/**
	 * Constructor.
	 */
	function __construct() {
		$this->setup_constants();
		$this->includes();

		// Updater
		if ( class_exists( '\\EDD\\Extensions\\ExtensionRegistry' ) ) {
			add_action( 'edd_extension_license_init', function( \EDD\Extensions\ExtensionRegistry $registry ) {
				$registry->addExtension(
					__FILE__,
					'Invoices',
					375153,
					EDD_INVOICES_VERSION
				);
			} );
		} elseif ( class_exists( 'EDD_License' ) ) {
			$license = new EDD_License( __FILE__, 'Invoices', EDD_INVOICES_VERSION, 'Easy Digital Downloads', null, null, 375153 );
		}

		add_action( 'plugins_loaded', array( $this, 'loadLanguageFiles' ) );

	}

	/**
	 * Magic getter to support calls to `$plugin`.
	 *
	 * @param string $key Property name.
	 *
	 * @since 1.2
	 * @return mixed
	 */
	public function __get( $key ) {
		if ( 'plugin' === $key ) {
			return $this->get_plugin_details();
		} elseif ( 'settings' === $key ) {
			return edd_get_settings();
		} elseif ( property_exists( $this, $key ) ) {
			return $this->{$key};
		}

		return null;
	}

	/**
	 * Set up constants.
	 *
	 * @since 1.2
	 */
	protected function setup_constants() {
		if ( ! defined( 'EDD_INVOICES_VERSION' ) ) {
			define( 'EDD_INVOICES_VERSION', '1.3.2' );
		}

		if ( ! defined( 'EDD_INVOICES_PLUGIN_DIR' ) ) {
			define( 'EDD_INVOICES_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
		}

		if ( ! defined( 'EDD_INVOICES_PLUGIN_URL' ) ) {
			define( 'EDD_INVOICES_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
		}
	}

	/**
	 * Returns information about the plugin.
	 *
	 * This is used when accessing the `$plugin` property.
	 *
	 * @since 1.2
	 * @return stdClass
	 */
	private function get_plugin_details() {
		$plugin              = new stdClass();
		$plugin->name        = 'edd-invoices';
		$plugin->displayName = 'Invoices';
		$plugin->version     = EDD_INVOICES_VERSION;
		$plugin->folder      = EDD_INVOICES_PLUGIN_DIR;
		$plugin->dirname     = EDD_INVOICES_PLUGIN_DIR;
		$plugin->url         = EDD_INVOICES_PLUGIN_URL;

		return $plugin;
	}

	/**
	 * Include required files.
	 *
	 * @since 1.2
	 */
	protected function includes() {
		require_once EDD_INVOICES_PLUGIN_DIR . 'includes/admin/settings.php';
		require_once EDD_INVOICES_PLUGIN_DIR . 'includes/actions.php';
		require_once EDD_INVOICES_PLUGIN_DIR . 'includes/class-invoice-generator.php';
		require_once EDD_INVOICES_PLUGIN_DIR . 'includes/functions.php';
		require_once EDD_INVOICES_PLUGIN_DIR . 'includes/shortcodes.php';
		require_once EDD_INVOICES_PLUGIN_DIR . 'includes/template-functions.php';
		require_once EDD_INVOICES_PLUGIN_DIR . 'includes/emails.php';
		require_once EDD_INVOICES_PLUGIN_DIR . 'includes/pdf.php';

		if ( is_admin() ) {
			require_once EDD_INVOICES_PLUGIN_DIR . 'includes/admin/actions.php';
		}
	}

	/**
	 * Loads plugin textdomain
	 */
	function loadLanguageFiles() {
		// Set filter for language directory
		$lang_dir = EDD_INVOICES_PLUGIN_DIR . '/languages/';
		$lang_dir = apply_filters( 'edd-invoices' . '_languages_directory', $lang_dir );

		// Traditional WordPress plugin locale filter
		$locale = apply_filters( 'plugin_locale', get_locale(), 'edd-invoices' );
		$mofile = sprintf( '%1$s-%2$s.mo', 'edd-invoices', $locale );

		// Setup paths to current locale file
		$mofile_local  = $lang_dir . $mofile;
		$mofile_global = WP_LANG_DIR . '/' . 'edd-invoices' . '/' . $mofile;

		if ( file_exists( $mofile_global ) ) {
			// Look in global /wp-content/languages/edd-plugin-name/ folder
			load_textdomain( 'edd-invoices', $mofile_global );
		} elseif ( file_exists( $mofile_local ) ) {
			// Look in local /wp-content/plugins/edd-plugin-name/languages/ folder
			load_textdomain( 'edd-invoices', $mofile_local );
		} else {
			// Load the default language files
			load_plugin_textdomain( 'edd-invoices', false, $lang_dir );
		}
	}
}

require_once dirname( __FILE__ ) . '/vendor/autoload.php';
\EDD\ExtensionUtils\v1\ExtensionLoader::loadOrQuit(
	__FILE__,
	function() {
		new EDDInvoices();
	},
	array(
		'php'                    => '5.3',
		'easy-digital-downloads' => '2.5',
	)
);


/**
 * Create the page contianing the edd_invoices page upon activation and define the setting
 */
function edd_invoices_activation() {

	// if our page isn't set
	if ( function_exists( 'edd_get_option' ) && ! edd_get_option( 'edd-invoices-page', false ) ) {
		// make page
		$page = wp_insert_post(
			array(
				'post_title'     => __( 'Invoice', 'edd-invoices' ),
				'post_content'   => '[edd_invoices]',
				'post_status'    => 'publish',
				'post_author'    => get_current_user_id(),
				'post_type'      => 'page',
				'post_parent'    => edd_get_option( 'purchase_history_page', false ),
				'comment_status' => 'closed',
			)
		);
		edd_update_option( 'edd-invoices-page', $page );
	}
}
register_activation_hook( __FILE__, 'edd_invoices_activation' );
