<?php
namespace SabaiApps\Directories\Component\Location;

use SabaiApps\Directories\Component\AbstractComponent;
use SabaiApps\Directories\Component\Form;
use SabaiApps\Directories\Component\Field;
use SabaiApps\Directories\Component\Entity;
use SabaiApps\Directories\Component\View;
use SabaiApps\Directories\Component\Search;
use SabaiApps\Directories\Component\CSV;
use SabaiApps\Directories\Application;

class LocationComponent extends AbstractComponent implements
    Form\IFields,
    Field\ITypes,
    Field\IWidgets,
    Field\IRenderers,
    Field\IFilters,
    Entity\IBundleTypes,
    CSV\IExporters,
    CSV\IImporters,
    Search\IFields
{
    const VERSION = '1.1.24', PACKAGE = 'directories-pro';

    public static function interfaces()
    {
        return array(
            'Payment\IFeatures',
            'Faker\IGenerators',
        );
    }

    public static function description()
    {
        return 'Group and filter content by location and display them on a map.';
    }

    public function onCoreComponentsLoaded()
    {
        $this->_application->setHelper('Location_Hierarchy', array(__CLASS__, 'hierarchyHelper'));
    }

    public static function hierarchyHelper(Application $application, Entity\Model\Bundle $bundle = null)
    {
        if (!isset($bundle)
            || empty($bundle->info['location_hierarchy_custom'])
            || !isset($bundle->info['location_hierarchy'])
        ) {
            return [
                'country' => __('Country', 'directories-pro'),
                'province' => __('State / Province / Region', 'directories-pro'),
                'city' => __('City', 'directories-pro'),
            ];
        }
        return $bundle->info['location_hierarchy'];
    }

    public function formGetFieldTypes()
    {
        return array('location_address', 'location_text');
    }

    public function formGetField($name)
    {
        switch ($name) {
            case 'location_address':
                return new FormField\AddressFormField($this->_application, $name);
            case 'location_text':
                return new FormField\TextFormField($this->_application, $name);
        }
    }

    public function fieldGetTypeNames()
    {
        return array('location_address');
    }

    public function fieldGetType($name)
    {
        switch ($name) {
            case 'location_address':
                return new FieldType\AddressFieldType($this->_application, $name);
        }
    }

    public function fieldGetFilterNames()
    {
        return array('location_address');
    }

    public function fieldGetFilter($name)
    {
        switch ($name) {
            case 'location_address':
                return new FieldFilter\AddressFieldFilter($this->_application, $name);
        }
    }

    public function fieldGetRendererNames()
    {
        return array('location_address');
    }

    public function fieldGetRenderer($name)
    {
        switch ($name) {
            case 'location_address':
                return new FieldRenderer\AddressFieldRenderer($this->_application, $name);
        }
    }

    public function fieldGetWidgetNames()
    {
        return array('location_address');
    }

    public function fieldGetWidget($name)
    {
        switch ($name) {
            case 'location_address':
                return new FieldWidget\AddressFieldWidget($this->_application, $name);
        }
    }

    public function csvGetImporterNames()
    {
        return array('location_address');
    }

    public function csvGetImporter($name)
    {
        return new CSVImporter\LocationCSVImporter($this->_application, $name);
    }

    public function csvGetExporterNames()
    {
        return array('location_address');
    }

    public function csvGetExporter($name)
    {
        return new CSVExporter\LocationCSVExporter($this->_application, $name);
    }

    public function fakerGetGeneratorNames()
    {
        return array('location_address');
    }

    public function fakerGetGenerator($name)
    {
        return new FakerGenerator\LocationFakerGenerator($this->_application, $name);
    }

    public function onViewEntitiesSettingsFilter(&$settings, $bundle, $view)
    {
        if (!empty($bundle->info['is_taxonomy'])
            || empty($bundle->info['location_enable'])
        ) return;

        $assets_dir = $this->_application->getPlatform()->getAssetsDir('directories-pro');
        if (!$view->viewModeInfo('mapable')
            || empty($settings['map']['show'])
        ) return;

        $settings['map']['template'] = $settings['template'];
        $settings['template'] = $assets_dir . '/templates/location_entities';

        $settings['ajax_container_template'] = $assets_dir . '/templates/location_entities_container';
        if (empty($settings['map']['style'])) unset($settings['map']['style']); // required to have the default setting override
        $settings['map'] += $this->_application->getComponent('Map')->getConfig('map');

        $this->_application->Location_Api_load(array(
            'map' => true,
            'style' => empty($settings['map']['style']) ? false : $settings['map']['style'],
            'markerclusterer' => $load_marker_clusterer = !empty($settings['map']['marker_clusters']),
            'markercluster_color' => $load_marker_clusterer ? $settings['map']['marker_cluster_color'] : null,
            'sticky' => !empty($settings['map']['sticky'])
        ));
    }

    public function onViewEntitiesSortsFilter(&$sorts, $bundle, $request, $settings)
    {
        if (empty($bundle->info['location_enable'])
            || !isset($sorts['location_address'])
            || !isset($settings['query']['fields']['location_address'])
        ) return;

        // Pass query args as 2nd sort arg
        $sorts['location_address']['args'] = array(
            'asc',
            explode(',', $settings['query']['fields']['location_address'])
        );
    }

    public function searchGetFieldNames()
    {
        return array('location_address');
    }

    public function searchGetField($name)
    {
        return new SearchField\AddressSearchField($this->_application, $name);
    }

    public function onViewModeSettingsFormFilter(&$form, View\Mode\IMode $mode, Entity\Model\Bundle $bundle, $settings, $parents, $submitValues)
    {
        if ($bundle->info['is_taxonomy']
            || empty($bundle->info['location_enable'])
            || !$mode->viewModeInfo('mapable')
        ) return;

        //$form['map']['geolocate'] = array(
        //    '#type' => 'checkbox',
        //    '#title' => __('Enable geolocation', 'directories-pro'),
        //    '#default_value' => !empty($settings['map']['geolocate']),
        //    '#horizontal' => true,
        //    '#weight' => 15,
        //);

        $field_name = $this->_application->Form_FieldName(array_merge($parents, array('map')));
        $states = array(
            sprintf('input[name="%s[show]"]', $field_name) => array('type' => 'checked', 'value' => true),
        );
        $form['map'] = array(
            'show' => array(
                '#type' => 'checkbox',
                '#title' => __('Show map', 'directories-pro'),
                '#default_value' => !empty($settings['map']['show']),
                '#horizontal' => true,
                '#weight' => 1,
            ),
            'position' => array(
                '#type' => 'select',
                '#title' => __('Map position', 'directories-pro'),
                '#options' => array('side' => __('Side', 'directories-pro'), 'top' => __('Top', 'directories-pro')),
                '#default_value' => isset($settings['map']['position']) ? $settings['map']['position'] : 'side',
                '#states' => array('visible' => $states),
                '#horizontal' => true,
                '#weight' => 5,
            ),
            'span' => array(
                '#type' => 'slider',
                '#integer' => true,
                '#min_value' => 1,
                '#max_value' => 11,
                '#title' => __('Map width', 'directories-pro'),
                '#description' => __('The horizontal display ratio (12 being 100% wide) of the map', 'directories-pro'),
                '#default_value' => isset($settings['map']['span']) ? $settings['map']['span'] : 4,
                '#states' => array('visible' => $states + array(
                    sprintf('select[name="%s[position]"]', $field_name) => array('value' => 'side'),
                )),
                '#horizontal' => true,
                '#weight' => 9,
            ),
            'height' => array(
                '#type' => 'slider',
                '#title' => __('Map height', 'directories-pro'),
                '#default_value' => isset($settings['map']['height']) ? $settings['map']['height'] : 400,
                '#min_value' => 100,
                '#max_value' => 1000,
                '#step' => 10,
                '#integer' => true,
                '#field_suffix' => 'px',
                '#horizontal' => true,
                '#weight' => 10,
                '#states' => array('visible' => $states),
            ),
            'style' => array(
                '#type' => 'select',
                '#options' => array('' => __('Default map style', 'directories-pro')) + $this->_application->Map_Api_mapStyle(),
                '#title' => __('Map style', 'directories-pro'),
                '#default_value' => isset($settings['map']['style']) ? $settings['map']['style'] : null,
                '#horizontal' => true,
                '#weight' => 12,
                '#states' => array('visible' => $states),
            ),
            'sticky' => array(
                '#type' => 'checkbox',
                '#title' => __('Enable sticky map', 'directories-pro'),
                '#default_value' => !isset($settings['map']['sticky']) || !empty($settings['map']['sticky']),
                '#horizontal' => true,
                '#weight' => 14,
                '#states' => array('visible' => $states),
            ),
            'scroll_to_item' => array(
                '#type' => 'checkbox',
                '#title' => __('Scroll to item on marker click', 'directories-pro'),
                '#default_value' => !isset($settings['map']['scroll_to_item']) || !empty($settings['map']['scroll_to_item']),
                '#horizontal' => true,
                '#weight' => 13,
                '#states' => array('visible' => $states),
            ),
            'fullscreen' => array(
                '#type' => 'checkbox',
                '#title' => __('Enable full screen mode', 'directories-pro'),
                '#default_value' => !isset($settings['map']['fullscreen']) || !empty($settings['map']['fullscreen']),
                '#horizontal' => true,
                '#weight' => 15,
                '#states' => array('visible' => $states),
            ),
            'fullscreen_span' => array(
                '#type' => 'slider',
                '#integer' => true,
                '#min_value' => 1,
                '#max_value' => 11,
                '#title' => __('Full screen map width', 'directories-pro'),
                '#description' => __('The horizontal display ratio (12 being 100% wide) of the map', 'directories-pro'),
                '#default_value' => isset($settings['map']['fullscreen_span']) ? $settings['map']['fullscreen_span'] : 6,
                '#states' => array(
                    'visible' => $states + array(
                        sprintf('input[name="%s[fullscreen]"]', $field_name) => array('value' => 1),
                    ),
                ),
                '#horizontal' => true,
                '#weight' => 16,
            ),
            'infobox' => array(
                '#type' => 'checkbox',
                '#title' => __('Enable map infobox', 'directories-pro'),
                '#default_value' => !isset($settings['map']['infobox']) || !empty($settings['map']['infobox']),
                '#weight' => 20,
                '#horizontal' => true,
                '#states' => array('visible' => $states),
            ),
            'infobox_width' => array(
                '#type' => 'slider',
                '#min_value' => 100,
                '#max_value' => 500,
                '#step' => 10,
                '#default_value' => isset($settings['map']['infobox_width']) ? $settings['map']['infobox_width'] : 240,
                '#title' => __('Map infobox width', 'directories-pro'),
                '#field_suffix' => 'px',
                '#integer' => true,
                '#states' => array(
                    'visible' => $states + array(
                        sprintf('input[name="%s[infobox]"]', $field_name) => array('type' => 'checked', 'value' => true),
                    ),
                ),
                '#weight' => 21,
                '#horizontal' => true,
            ),
            'trigger_infobox' => array(
                '#type' => 'checkbox',
                '#title' => __('Open infobox on item hover', 'directories-pro'),
                '#default_value' => !empty($settings['map']['trigger_infobox']),
                '#states' => array(
                    'visible' => $states + array(
                        sprintf('input[name="%s[infobox]"]', $field_name) => array('type' => 'checked', 'value' => true),
                    ),
                ),
                '#weight' => 22,
                '#horizontal' => true,
            ),
        );

        return $form;
    }

    public function entityGetBundleTypeNames()
    {
        return array('location_location');
    }

    public function entityGetBundleType($name)
    {
        return new EntityBundleType\LocationEntityBundleType($this->_application, $name);
    }

    public function onDirectoryContentTypeSettingsFormFilter(&$form, $directoryType, $contentType, $info, $settings, $parents, $submitValues)
    {
        if (empty($info['location_enable'])
            || !empty($info['is_taxonomy'])
            || !empty($info['parent'])
        ) return;

        $form['location_enable'] = array(
            '#type' => 'checkbox',
            '#title' => __('Enable locations', 'directories-pro'),
            '#default_value' => !empty($settings['location_enable']) || is_null($settings),
            '#horizontal' => true,
        );
    }

    public function onDirectoryContentTypeInfoFilter(&$info, $contentType, $settings = null)
    {
        if (!isset($info['location_enable'])) return;

        if (!empty($info['is_taxonomy'])
            || !empty($info['parent'])
        ) {
            unset($info['location_enable']);
        }

        if (isset($settings['location_enable']) && !$settings['location_enable']) {
            $info['location_enable'] = false;
        }
    }

    public function onEntityBundlesInfoFilter(&$bundles, $componentName, $group)
    {
        foreach (array_keys($bundles) as $bundle_type) {
            $info =& $bundles[$bundle_type];

            if (empty($info['location_enable'])
                || !empty($info['is_taxonomy'])
                || !empty($info['parent'])
            ) continue;

            // Add location_location bundle
            if (!isset($bundles['location_location'])) { // may already set if updating or importing
                $bundles['location_location'] = [];
            }
            $bundles['location_location'] += $this->entityGetBundleType('location_location')->entityBundleTypeInfo();

            return; // there should be only one bundle with location enabled
        }

        // No bundle with locations enabled found, so make sure the location_location bundle is not assigned
        unset($bundles['location_location']);
    }

    public function onEntityBundleInfoFilter(&$info, $componentName, $group)
    {
        if (empty($info['location_enable'])
            || !empty($info['is_taxonomy'])
            || !empty($info['parent'])
        ) return;

        // Associate location bundle
        $info['taxonomies']['location_location'] = array(
            'weight' => 10,
        );
        // Add location address field
        if (!isset($info['fields']['location_address'])) { // may already be set
            $info['fields']['location_address'] = [];
        }
        if (!empty($info['location_field'])) {
            $info['fields']['location_address'] += $info['location_field'];
        }
        $info['fields']['location_address'] += array(
            'type' => 'location_address',
            'settings' => [],
            'max_num_items' => 1,
            'weight' => 7,
            'label' => __('Location', 'directories-pro'),
            'required' => false,
        );
    }

    public function onEntityBundleInfoKeysFilter(&$keys)
    {
        $keys[] = 'location_enable';
        $keys[] = 'location_marker_taxonomy';
    }

    public function onEntityBundleInfoUserKeysFilter(&$keys)
    {
        $keys[] = 'location_hierarchy';
        $keys[] = 'location_hierarchy_custom';
    }

    public function onViewCurrentTermFilter(&$termId, $bundle)
    {
        if (empty($termId)
            && $bundle->type === 'location_location'
            && ($term_id = $this->_application->Location_IsSearchRequested())
        ) {
            $termId = $term_id;
        }
        return $termId;
    }

    public function onEntityBeforeCreateEntity($bundle, &$values)
    {
        $this->_setTermValuesFromLocation($bundle, $values);
    }

    public function onEntityBeforeUpdateEntity($bundle, Entity\Type\IEntity $entity, &$values)
    {
        $this->_setTermValuesFromLocation($bundle, $values);
    }

    /**
     * Fill taxonomy term values when submitting a post from the backend
     */
    protected function _setTermValuesFromLocation($bundle, &$values)
    {
        if (empty($bundle->info['location_enable'])) return;

        if (isset($values['location_address'])) {
            // Extract term ID from location field and copy to taxonomy term field
            $values['location_location'] = [];
            foreach ($values['location_address'] as $field_value) {
                if (!empty($field_value['term_id'])) {
                    $values['location_location'][] = $field_value['term_id'];
                }
            }
        }
    }

    public function paymentGetFeatureNames()
    {
        return array('location_locations');
    }

    public function paymentGetFeature($name)
    {
        return new PaymentFeature\LocationsPaymentFeature($this->_application, $name);
    }

    public function onCorePlatformWordPressInit()
    {
        if (is_admin()) {
            add_action('admin_head' , array($this, 'adminHeadAction'));
        }
    }

    public function adminHeadAction()
    {
        if (empty($GLOBALS['typenow'])
            || (!$bundle = $this->_application->Entity_Bundle($GLOBALS['typenow']))
            || empty($bundle->info['location_enable'])
            || empty($bundle->info['taxonomies']['location_location'])
        ) return;

        remove_meta_box($bundle->info['taxonomies']['location_location'] . 'div', $bundle->name, 'side');
    }

    public function onEntityFieldValuesLoaded(Entity\Type\IEntity $entity, Entity\Model\Bundle $bundle, array $fields, $cache)
    {
        if (!$cache
            || !$this->_application->isComponentLoaded('Payment')
            || empty($bundle->info['location_enable'])
            || empty($bundle->info['payment_enable'])
        ) return;

        $features = $this->_application->Payment_Plan_features($entity);

        if (!empty($features[0]['location_locations']['unlimited'])) return;

        if (!isset($features[0]['location_locations']['num'])) {
            $max_num_allowed = 1;
        } else {
            $max_num_allowed = empty($features[0]['location_locations']['num']) ? 0 : $features[0]['location_locations']['num'];
        }
        if (!empty($features[1]['location_locations']['num'])) { // any additional num of photos allowed?
            $max_num_allowed += $features[1]['location_locations']['num'];
        }

        // Limit both taxonomy term and address fields
        foreach (array('location_location', 'location_address') as $field_name) {
            if ((!$field_values = $entity->getFieldValue($field_name))
                || count($field_values) <= $max_num_allowed
            ) continue;

            $entity->setFieldValue($field_name, array_slice($field_values, 0, $max_num_allowed));
        }
    }

     public function onSystemAdminSystemToolsFilter(&$tools)
    {
        $tools['location_geocode'] = [
            'label' => __('Load geolocation data', 'directories-pro'),
            'description' => __('This tool will load geolocation data (lat/lng/timezone) of each content item if currently empty.', 'directories-pro'),
            'with_progress' => true,
            'weight' => 80,
        ];
    }

    public function onSystemAdminRunTool($tool, $progress, $values)
    {
        switch ($tool) {
            case 'location_geocode':
                $this->_application->Location_Tools_geocode($progress);
                break;
        }
    }
}
