'use strict';

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function(obj) {
  return typeof obj;
} : function(obj) {
  return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
};

(function($) {
  DRTS.Map = DRTS.Map || {};
  DRTS.Map.styles = {};
  DRTS.Map.map = DRTS.Map.map || function(container, options) {
    if (container instanceof jQuery) {
      this.container = container;
      this.containerSelector = '#' + container.attr('id');
    } else {
      this.container = $(container);
      this.containerSelector = container;
    }
    this.options = options;
    this.markers = [];
    this.currentMarker = null;
    this.markerClusterer = null;
    this.popover = null;
    this.overlay = null;
    this.currentCircle = null;
    var self = this;
    var $map = this.container.find('.drts-map-map').addClass('drts-popover-ignore-click').outerHeight(this.options.height);
    var mapTypeIds = [];
    for (var mapType in google.maps.MapTypeId) {
      mapTypeIds.push(google.maps.MapTypeId[mapType]);
    }
    mapTypeIds.push('osm');
    this.options.map_type_control = typeof this.options.map_type_control === 'undefined' || this.options.map_type_control ? true : false;
    this.options.zoom_control = typeof this.options.zoom_control === 'undefined' || this.options.zoom_control ? true : false;
    this.options.infobox = typeof this.options.infobox === 'undefined' || this.options.infobox ? true : false;
    this.options.infobox_event = this.options.infobox_event || 'click';
    var settings = {
      mapTypeId: $.inArray(this.options.type, mapTypeIds) !== -1 ? this.options.type : google.maps.MapTypeId.ROADMAP,
      mapTypeControl: this.options.map_type_control,
      zoomControl: this.options.zoom_control,
      streetViewControl: false,
      scaleControl: false,
      rotateControl: false,
      fullscreenControl: this.options.fullscreen_control || false,
      center: new google.maps.LatLng(this.options.default_location.lat || 40.69847, this.options.default_location.lng || -73.95144),
      scrollwheel: this.options.scrollwheel || false,
      styles: this.options.style && DRTS.Map.styles[this.options.style] ? DRTS.Map.styles[this.options.style] : [{
        'featureType': 'poi',
        'stylers': [{
          'visibility': 'off'
        }]
      }],
      zoom: this.options.default_zoom = parseInt(this.options.default_zoom, 10) || 10
    };
    if (settings.mapTypeControl) {
      settings.mapTypeControlOptions = {
        style: google.maps.MapTypeControlStyle.DROPDOWN_MENU,
        mapTypeIds: mapTypeIds,
        position: google.maps.ControlPosition.TOP_RIGHT
      };
    }
    this.map = new google.maps.Map($map.get(0), settings);
    this.map.mapTypes.set('osm', new google.maps.ImageMapType({
      getTileUrl: function getTileUrl(coord, zoom) {
        return '//a.tile.openstreetmap.org/' + zoom + '/' + coord.x + '/' + coord.y + '.png';
      },
      tileSize: new google.maps.Size(256, 256),
      isPng: true,
      maxZoom: 19,
      minZoom: 0,
      name: 'OSM'
    }));

    // Add marker clusterer?
    if (this.options.marker_clusters) {
      var marker_clusterer_options = {
        maxZoom: 15
      };
      if (this.options.marker_cluster_imgurl) {
        marker_clusterer_options.imagePath = this.options.marker_cluster_imgurl + '/m';
      }
      this.markerClusterer = new MarkerClusterer(this.map, [], marker_clusterer_options);
    }

    // Enable popover
    if (this.options.infobox) {
      this.getPopover();
      this.getOverlay();
      google.maps.event.addListener(this.map, 'dragstart', function() {
        self.getPopover().sabaiPopover('hide');
        self.setCurrentMarker(null);
      });
      google.maps.event.addListener(this.map, 'zoom_changed', function() {
        self.getPopover().sabaiPopover('hide');
        self.setCurrentMarker(null);
      });
    }

    // Init street view panorama
    this.map.getStreetView().setOptions({
      disableDefaultUI: true,
      enableCloseButton: false,
      zoomControl: this.options.zoom_control,
      visible: false
    });
  };

  DRTS.Map.map.prototype.getMap = function() {
    return this.map;
  };

  DRTS.Map.map.prototype.getContainer = function() {
    return this.container;
  };

  DRTS.Map.map.prototype.getContainerSelector = function() {
    return this.containerSelector;
  };

  DRTS.Map.map.prototype.getOptions = function() {
    return this.options;
  };

  DRTS.Map.map.prototype.draw = function(options) {
    options = options || {};
    this.setCurrentMarker(null);
    if (this.currentCircle) {
      this.currentCircle.setMap(null);
    }

    if (this.markers.length > 0) {
      var fit_bounds, bounds;
      fit_bounds = typeof options.fit_bounds === 'undefined' ? this.options.fit_bounds : options.fit_bounds;

      if (fit_bounds && this.markers.length > 1) {
        bounds = new google.maps.LatLngBounds();
      }

      for (var i = 0; i < this.markers.length; i++) {
        if (!this.markerClusterer) {
          // will add markers in bulk later if marker cluster exists
          this.markers[i].setMap(this.map);
        }
        if (bounds) {
          var pos = this.markers[i].getPosition();
          bounds.extend(pos);
          if (options.center) {
            // Extend bound to include the point opposite the marker so the center stays the same
            bounds.extend(new google.maps.LatLng(options.center[0] * 2 - pos.lat(), options.center[1] * 2 - pos.lng()));
          }
        }
        google.maps.event.addListener(this.markers[i], this.options.infobox_event, function(map, marker) {
          return function(e) {
            DRTS.Map.map.clickMarker(map, marker, e);
          };
        }(this, this.markers[i]));

        if (this.markers.length <= 100) {
          // Bounce on display
          this.markers[i].setAnimation(google.maps.Animation.BOUNCE);
          setTimeout(function(marker) {
            return function() {
              marker.setAnimation(null);
            };
          }(this.markers[i]), 500);
        }
      }

      if (this.markerClusterer) {
        this.markerClusterer.addMarkers(this.markers);
      }

      if (bounds) {
        this.map.fitBounds(bounds);
      } else {
        // Center position required if no automattic bounding
        if (!options.center) {
          var pos = this.markers[0].getPosition();
          options.center = [pos.lat(), pos.lng()];
        }
      }

      if (options.street_view) {
        this.drawStreetView(_typeof(options.street_view) === 'object' ? options.street_view : this.markers[0]);
      }
    }

    if (options.center) {
      var center = new google.maps.LatLng(options.center[0], options.center[1]);
      this.map.setCenter(center);
      this.map.setZoom(options.zoom || this.options.default_zoom || 10);
      if (options.circle && parseInt(options.circle.draw)) {
        this.currentCircle = new google.maps.Circle({
          strokeColor: options.circle.stroke_color || '#99f',
          strokeOpacity: 0.8,
          strokeWeight: 1,
          fillColor: options.circle.fill_color || '#99f',
          fillOpacity: 0.3,
          map: this.map,
          center: center,
          radius: options.circle.radius
        });
      }
    }

    return this;
  };

  DRTS.Map.map.prototype.drawStreetView = function(position, radius, notify) {
    var sv = new google.maps.StreetViewService(),
      map = this.map;
    if (position.setMap) {
      var marker = position;
      position = position.getPosition();
    }
    sv.getPanorama({
      location: position,
      radius: radius || 50
    }, function(data, status) {
      if (status === google.maps.StreetViewStatus.OK) {
        var pano = map.getStreetView();
        pano.setPosition(data.location.latLng);
        if (marker) {
          var heading = google.maps.geometry.spherical.computeHeading(data.location.latLng, position);
          pano.setPov({
            heading: heading,
            pitch: 0,
            zoom: 1
          });
          marker.setMap(pano);
        }
        pano.setVisible(true);
      } else {
        if (notify) {
          alert('No street map view is available for this location.');
        }
        console.log(status);
      }
    });
    return this;
  };

  DRTS.Map.map.prototype.setMarkers = function(markers) {
    for (var i = 0; i < markers.length; i++) {
      this.addMarker(markers[i]);
    }
    return this;
  };
  DRTS.Map.map.prototype.getMarker = function(index) {
    return this.markers[index];
  };
  DRTS.Map.map.prototype.addMarker = function(marker) {
    var _marker;
    var defaultMarkerIconOptions;
    if (this.options.marker_custom) {
      if (marker.icon) {
        switch (marker.icon.type) {
          case 'file':
            if (marker.icon.url && /^https?:\/\//i.test(marker.icon.url)) {
              _marker = new google.maps.Marker();
              var markerUrlOptions = {
                url: marker.icon.url
              };
              var icon_width = marker.icon.width ? parseInt(marker.icon.width) : 0;
              var icon_height = marker.icon.height ? parseInt(marker.icon.height) : 0;
              if (icon_width && icon_height) {
                markerUrlOptions.size = markerUrlOptions.scaledSize = new google.maps.Size(icon_width, icon_height);
                _marker.set('marker_height', icon_height);
              }
              _marker.setIcon(markerUrlOptions);
            }
            break;
          default:
            var markerIconOptions = {
              html: marker.icon.url ? $('<img/>').attr('src', marker.icon.url)[0].outerHTML : null,
              icon: marker.icon.icon || this.options.marker_icon,
              icon_color: marker.icon.icon_color || this.options.marker_icon_color,
              size: marker.icon.size || this.options.marker_size,
              color: marker.icon.color || this.options.marker_color || '#fff',
              event: this.options.infobox_event
            };
            _marker = new DRTS.Map.map.marker(markerIconOptions);
            break;
        }
      }
      if (!_marker) {
        if (typeof defaultMarkerIconOptions === 'undefined') {
          defaultMarkerIconOptions = {
            icon: this.options.marker_icon || 'fas fa-dot-circle',
            icon_color: this.options.marker_icon_color,
            size: this.options.marker_size,
            color: this.options.marker_color || '#fff',
            event: this.options.infobox_event
          };
          _marker = new DRTS.Map.map.marker(defaultMarkerIconOptions);
        }
      }
    }
    if (!_marker) {
      _marker = new google.maps.Marker();
    }
    _marker.setPosition(new google.maps.LatLng(marker.lat, marker.lng));

    var marker_infobox_event = this.options.infobox_event;
    if (marker.trigger) {
      var marker_trigger = $(marker.trigger);
      if (marker_trigger.length) {
        if (marker.triggerEvent) {
          marker_trigger.on(marker.triggerEvent, function(marker) {
            return function(e) {
              e.preventDefault();
              google.maps.event.trigger(marker, marker_infobox_event, {
                triggered: true
              });
            };
          }(_marker));
        } else {
          marker_trigger.hoverIntent(function(marker) {
            return function(e) {
              e.preventDefault();
              google.maps.event.trigger(marker, marker_infobox_event, {
                triggered: true
              });
            };
          }(_marker), function() {});
        }
      }
      _marker.set('trigger', marker.trigger);
      _marker.set('index', marker.index);
    }
    _marker.set('content', marker.content);
    _marker.set('entity_url', marker.entity_url);
    _marker.set('id', this.markers.length);

    if (marker.default) {
      setTimeout(function(marker) {
        return function() {
          google.maps.event.trigger(marker, marker_infobox_event, {
            init: true
          });
        };
      }(_marker), 1500);
    }

    this.markers.push(_marker);
    return this;
  };

  DRTS.Map.map.prototype.setCurrentMarker = function(marker) {
    this.currentMarker = marker;
    return this;
  };
  DRTS.Map.map.prototype.getCurrentMarker = function() {
    return this.currentMarker;
  };
  DRTS.Map.map.prototype.getCurrentCircle = function() {
    return this.currentCircle;
  };
  DRTS.Map.map.prototype.getMarkerClusterer = function() {
    return this.markerClusterer;
  };
  DRTS.Map.map.prototype.getPopover = function(pixel, markerHeight, content) {
    if (!this.options.infobox) return;

    if (!this.popover) {
      var popover = this.container.find('span.drts-map-popover');
      if (!popover.length) {
        popover = $('<span style="position:absolute;" class="drts-map-popover"></span>').prependTo(this.container.find('.drts-map-container'));
      }
      DRTS.popover(popover.removeClass('drts-popover-processed'), {
        html: true,
        template: '<div class="' + DRTS.bsPrefix + 'popover drts-map-popover ' + DRTS.bsPrefix + 'p-0" style="width:' + (this.options.infobox_width || 240) + 'px">' + '<div class="' + DRTS.bsPrefix + 'arrow"></div><div class="' + DRTS.bsPrefix + 'close ' + DRTS.bsPrefix + 'p-1 drts-map-popover-close" aria-label="Close"><i aria-hidden="true" class="fas fa-times"></i></div>' + '<div class="' + DRTS.bsPrefix + 'popover-body ' + DRTS.bsPrefix + 'p-0"></div></div>',
        placement: 'top',
        container: this.container
      });
      var self = this;
      popover.on('hidden.bs.popover', function() {
        self.setCurrentMarker(null);
      });
      this.popover = popover;
    }
    if (pixel) {
      var _top = pixel.y;
      if (markerHeight) {
        _top -= markerHeight;
      }
      this.popover.css({
        left: pixel.x + 'px',
        top: _top + 'px'
      });
    }
    if (content) {
      this.popover.attr('data-content', content);
    }
    return this.popover;
  };
  DRTS.Map.map.prototype.getOverlay = function(create) {
    if (!this.overlay || create) {
      this.overlay = new google.maps.OverlayView();
      this.overlay.draw = function() {};
      this.overlay.setMap(this.map);
    }
    return this.overlay;
  };
  DRTS.Map.map.prototype.clearMarkers = function() {
    for (var i = 0; i < this.markers.length; i++) {
      this.markers[i].setMap(null);
    }
    this.markers = [];
    var marker_clusterer = this.getMarkerClusterer();
    if (marker_clusterer) {
      marker_clusterer.clearMarkers();
    }
    return this;
  };
  DRTS.Map.map.prototype.enableDirections = function() {
    // Reset panel and form
    this.container.find('.drts-map-directions-panel').html('').hide().end().find('.drts-map-directions-input').val('').end().find('.drts-map-directions-destination').val(0);
    if (DRTS.Map.autocomplete) {
      DRTS.Map.autocomplete(this.container.find('.drts-map-directions-input'));
    }
    if (DRTS.Map.geolocate) {
      DRTS.Map.geolocate(this.container.find('.drts-map-directions-input'));
    }

    var self = this;
    this.container.find('.drts-map-directions-trigger').on('click', function(e) {
      var $this = $(this),
        markerIndex = self.container.find('.drts-map-directions-destination').val(),
        marker,
        origin,
        panel = self.container.find('.drts-map-directions-panel').html('');

      e.preventDefault();
      self.clearCurrentDirections();

      if (markerIndex === '') {
        panel.html('No destination selected');
        return this;
      }

      marker = self.getMarker(markerIndex);
      if (!marker) {
        panel.html('Invalid destination');
        return this;
      }

      origin = self.container.find('.drts-map-directions-input').val();
      if (!origin) {
        panel.html('No origin entered');
        return this;
      }

      $this.addClass(DRTS.bsPrefix + 'disabled');
      DRTS.Map.map.drawDirections(self, panel, origin, marker.getPosition(), $this.data('travel-mode') || 'DRIVING');

      DRTS.scrollTo(self.container);
      $this.removeClass(DRTS.bsPrefix + 'disabled');
    });

    return this;
  };
  DRTS.Map.map.prototype.clearCurrentDirections = function() {
    if (this.currentDirections) {
      this.currentDirections.setMap(null);
    }
    this.currentDirections = null;
    return this.currentDirections;
  };
  DRTS.Map.map.prototype.setCurrentDirections = function(directions) {
    this.currentDirections = directions;
    return this;
  };

  DRTS.Map.map.drawDirections = function(map, panel, origin, destination, mode) {
    var service = new google.maps.DirectionsService(),
      request = {
        origin: origin,
        destination: new google.maps.LatLng(destination.lat(), destination.lng()),
        travelMode: google.maps.TravelMode[mode || 'DRIVING']
      };
    panel.slideUp();
    service.route(request, function(response, status) {
      if (status === google.maps.DirectionsStatus.OK) {
        map.setCurrentDirections(new google.maps.DirectionsRenderer({
          map: map.getMap(),
          draggable: true,
          panel: panel.get(0),
          directions: response
        }));
        panel.slideDown().find('img.adp-marker').hide();
      } else {
        panel.html('<p>No directions found: ' + status + '</p>').slideDown();
      }
    });
  };

  DRTS.Map.map.clickMarker = function(map, marker, e) {
    var current = map.getCurrentMarker(),
      triggered = e && e.triggered;

    if (current && current.get('id') === marker.get('id')) {
      DRTS.Map.map.showMarkerContent(map, marker, e);
      if (!triggered) {
        // make sure manually clicked
        $(DRTS).trigger('map_marker_clicked.sabai', {
          map: map,
          marker: marker
        });
      }
      return;
    }

    if (triggered && map.getMarkerClusterer()) {
      // Add back previously removed marker
      if (current) {
        map.getMarkerClusterer().addMarker(current);
      }
      // Remove marker from cluster for better view of the marker
      map.getMarkerClusterer().removeMarker(marker);
      marker.setMap(map.getMap());
    }

    if (map.getMap().getBounds() && !map.getMap().getBounds().contains(marker.getPosition())) {
      map.getMap().panTo(marker.getPosition());
    }

    if (current) {
      //current.setAnimation(null);
      current.setZIndex(0);
    }
    marker.setZIndex(1);

    DRTS.Map.map.showMarkerContent(map, marker, e);

    map.setCurrentMarker(marker);

    if (!triggered) {
      // make sure manually clicked
      $(DRTS).trigger('map_marker_clicked.sabai', {
        map: map,
        marker: marker
      });
    }
  };

  DRTS.Map.map.showMarkerContent = function(map, marker, e) {
    var popover = map.getPopover(),
      triggered = e && e.triggered;

    // Close if popover is currently open
    if (popover && marker.index === 0 // do not close if sub-location since this will close the popver for the main location
    ) {
      popover.sabaiPopover('hide');
    }

    // Animate marker if triggered, or manually clicked and no infobox
    if (triggered || !map.getOptions().infobox) {
      marker.setAnimation(google.maps.Animation.BOUNCE);
      setTimeout(function() {
        marker.setAnimation(null);
      }, 1000);
    }

    if (triggered && !map.getOptions().trigger_infobox || // trigger infobox disabled
      !popover // No overlay or is not ready
      ||
      !map.getOverlay() || !map.getOverlay().getProjection()) {
      return;
    }

    if (triggered) {
      if (marker.index !== 0) return; // do no show content for sub-locations if triggered
    }

    if (!marker.get('content')) return;

    popover = map.getPopover(map.getOverlay().getProjection().fromLatLngToContainerPixel(marker.getPosition()), marker.get('marker_height') || 38, marker.get('content')).sabaiPopover('show');
  };
  DRTS.Map.map.markerHeight = function(height) {
    var diagnol = height * Math.sqrt(2);
    var trim = (diagnol - height) / 2;
    return diagnol - trim;
  };
  DRTS.Map.map.marker = DRTS.Map.map.marker || function(options) {
    this.options = options || {};
    this.visible = true;
    this.classes = ['drts-map-marker'];
    this.div = null;
  };
  DRTS.Map.map.marker.prototype = new google.maps.OverlayView();
  DRTS.Map.map.marker.prototype.onAdd = function() {
    var size = this.options.size || 38,
      marker;
    this.div = document.createElement('div');
    this.div.className = this.classes.join(' ');
    this.div.style.width = size + 'px';
    this.div.style.height = size + 'px';
    this.div.style.marginTop = '-' + (size * Math.sqrt(2) - DRTS.Map.map.markerHeight(size)) + 'px';
    marker = document.createElement('div');
    if (this.options.color) {
      this.div.style.backgroundColor = this.div.style.color = marker.style.borderColor = this.options.color;
    }
    if (this.options.data) {
      this.div.dataset = this.options.data;
    }
    if (this.options.html) {
      marker.innerHTML = this.options.html;
      //this.div.innerHTML = '<div style="border-color:' + this.options.color + ';">' + this.options.html + '</div>';
    } else if (this.options.icon) {
      marker.innerHTML = '<i class="' + this.options.icon + '"></i>';
      if (this.options.icon_color) marker.style.backgroundColor = this.options.icon_color;
      //this.div.innerHTML = '<div style="border-color:' + this.options.color + ';">'+ '<i class="' + this.options.icon + '" style="'
      //    + (this.options.icon_color ? 'color:' + this.options.icon_color : '') + ';"></i></div>';
    }
    this.div.appendChild(marker);
    this.getPanes().overlayImage.appendChild(this.div);
    var self = this,
      ev = this.options.event;
    google.maps.event.addDomListener(this.div, ev, function(event) {
      google.maps.event.trigger(self, ev);
    });
    this.setPosition(this.position);
    this.set('marker_height', DRTS.Map.map.markerHeight(size));
  };
  DRTS.Map.map.marker.prototype.draw = function() {
    this.setPosition(this.position);
  };
  DRTS.Map.map.marker.prototype.setPosition = function(position) {
    this.position = position;
    if (this.div) {
      var point = this.getProjection().fromLatLngToDivPixel(this.position);
      if (point) {
        this.div.style.left = point.x + 'px';
        this.div.style.top = point.y + 'px';
      }
    }
  };
  DRTS.Map.map.marker.prototype.onRemove = function() {
    if (this.div) {
      this.div.parentNode.removeChild(this.div);
    }
    this.div = null;
  };
  DRTS.Map.map.marker.prototype.getPosition = function() {
    return this.position;
  };
  DRTS.Map.map.marker.prototype.setDraggable = function(draggable) {
    this.draggable = draggable;
  };
  DRTS.Map.map.marker.prototype.getDraggable = function() {
    this.draggable;
  };
  DRTS.Map.map.marker.prototype.getVisible = function() {
    return this.visible;
  };
  DRTS.Map.map.marker.prototype.setVisible = function(visible) {
    if (this.div) {
      this.div.style.display = visible ? 'inline-block' : 'none';
    }
    this.visible = visible;
  };
  DRTS.Map.map.marker.prototype.getDraggable = function() {
    return this.draggable;
  };
  DRTS.Map.map.marker.prototype.setDraggable = function(draggable) {
    this.draggable = draggable;
  };
  DRTS.Map.map.marker.prototype.setZIndex = function(zIndex) {
    this.zIndex = zIndex;
    if (this.div) {
      this.div.style.zIndex = this.zIndex;
    }
  };
  DRTS.Map.map.marker.prototype.setAnimation = function(animation) {
    var class_name = 'drts-map-marker-bounce';
    if (animation) {
      if (this.classes.indexOf(class_name) === -1) {
        this.classes.push(class_name);
      }
    } else {
      var index = this.classes.indexOf(class_name);
      if (index > -1) {
        this.classes.splice(index, 1);
      }
    }
    if (this.div) {
      this.div.className = this.classes.join(' ');
    }
  };
})(jQuery);