<?php
namespace SabaiApps\Directories\Component\Location\FieldType;

use SabaiApps\Directories\Component\Field;
use SabaiApps\Directories\Component\Entity;
use SabaiApps\Directories\Exception;
use SabaiApps\Directories\Application;

class AddressFieldType extends Field\Type\AbstractType implements
    Field\Type\ISortable,
    Field\Type\ISchemable,
    Field\Type\IQueryable,
    Field\Type\IOpenGraph,
    Field\Type\IHumanReadable,
    Field\Type\IConditionable
{
    protected function _fieldTypeInfo()
    {
        return array(
            'label' => __('Location', 'directories-pro'),
            'default_widget' => $this->_name,
            'default_renderer' => $this->_name,
            'icon' => 'fas fa-map-marker-alt',
        );
    }

    public function fieldTypeSchema()
    {
        return array(
            'columns' => array(
                'address' => array(
                    'type' => Application::COLUMN_VARCHAR,
                    'length' => 255,
                    'notnull' => true,
                    'was' => 'address',
                    'default' => '',
                ),
                'street' => array(
                    'type' => Application::COLUMN_VARCHAR,
                    'length' => 255,
                    'notnull' => true,
                    'was' => 'street',
                    'default' => '',
                ),
                'street2' => array(
                    'type' => Application::COLUMN_VARCHAR,
                    'length' => 255,
                    'notnull' => true,
                    'was' => 'street2',
                    'default' => '',
                ),
                'city' => array(
                    'type' => Application::COLUMN_VARCHAR,
                    'length' => 100,
                    'notnull' => true,
                    'was' => 'city',
                    'default' => '',
                ),
                'province' => array(
                    'type' => Application::COLUMN_VARCHAR,
                    'length' => 100,
                    'notnull' => true,
                    'was' => 'state',
                    'default' => '',
                ),
                'zip' => array(
                    'type' => Application::COLUMN_VARCHAR,
                    'length' => 30,
                    'notnull' => true,
                    'was' => 'zip',
                    'default' => '',
                ),
                'country' => array(
                    'type' => Application::COLUMN_VARCHAR,
                    'length' => 50,
                    'notnull' => true,
                    'was' => 'country',
                    'default' => '',
                ),
                'timezone' => array(
                    'type' => Application::COLUMN_VARCHAR,
                    'length' => 50,
                    'notnull' => true,
                    'was' => 'timezone',
                    'default' => '',
                ),
                'zoom' => array(
                    'type' => Application::COLUMN_INTEGER,
                    'unsigned' => true,
                    'notnull' => true,
                    'length' => 2,
                    'was' => 'zoom',
                    'default' => 10,
                ),
                'lat' => array(
                    'type' => Application::COLUMN_DECIMAL,
                    'length' => 9,
                    'scale' => 6,
                    'notnull' => true,
                    'unsigned' => false,
                    'was' => 'lat',
                    'default' => 0,
                ),
                'lng' => array(
                    'type' => Application::COLUMN_DECIMAL,
                    'length' => 9,
                    'scale' => 6,
                    'notnull' => true,
                    'unsigned' => false,
                    'was' => 'lng',
                    'default' => 0,
                ),
                'term_id' => array(
                    'type' => Application::COLUMN_INTEGER,
                    'unsigned' => true,
                    'notnull' => true,
                    'length' => 10,
                    'was' => 'term_id',
                    'default' => 0,
                ),
            ),
            'indexes' => array(
                'lat_lng' => array(
                    'fields' => array(
                        'lat' => array('sorting' => 'ascending'),
                        'lng' => array('sorting' => 'ascending'),
                    ),
                    'was' => 'lat_lng',
                ),
            ),
        );
    }

    public function fieldTypeOnSave(Field\IField $field, array $values)
    {
        $ret = [];
        foreach ($values as $weight => $value) {
            if (!is_array($value)) continue;

            foreach (array('city', 'zip', 'country', 'province') as $key) {
                if (isset($value[$key])
                    && is_array($value[$key])
                ) {
                    $value[$key] = trim((string)array_shift($value[$key]));
                }
            }
            if (isset($value['lat'])
                && ($lat = trim($value['lat']))
                && ($lat = (float)$lat)
            ) {
                $value['lat'] = $lat;
                if (isset($value['lng'])
                    && ($lng = trim($value['lng']))
                    && ($lng = (float)$lng)
                ) {
                    $value['lng'] = $lng;
                } else {
                    unset($value['lat'], $value['lng'], $value['zoom']);
                }
            } else {
                unset($value['lat'], $value['lng'], $value['zoom']);
            }
            if ($value = array_filter($value)) {
                $ret[] = $value;
            }
        }

        return $ret;
    }

    public function fieldTypeOnLoad(Field\IField $field, array &$values, Entity\Type\IEntity $entity)
    {
        foreach (array_keys($values) as $key) {
            settype($values[$key]['lat'], 'float');
            settype($values[$key]['lng'], 'float');
        }
    }
    
    public function fieldSortableOptions(Field\IField $field)
    {
        return array(
            array('label' => __('Distance', 'directories-pro')),
        );
    }
    
    public function fieldSortableSort(Field\Query $query, $fieldName, array $args = null)
    {
        $config = $this->_application->getComponent('Map')->getConfig('map');
        $order = isset($args[0]) && $args[0] === 'desc' ? 'DESC' : 'ASC';
        $lat = $config['default_location']['lat'];
        $lng = $config['default_location']['lng'];
        if (isset($args[1])) {
            if (is_array($args[1]) && !empty($args[1])) {
                // Args passed from query settings of view
                switch (count($args[1])) {
                    case 1:
                    case 2:
                        if ($args[1][0] === '_current_') {
                            if (isset($GLOBALS['drts_entity'])
                                && ($location = $GLOBALS['drts_entity']->getSingleFieldValue($fieldName))
                                && !empty($location['lat'])
                                && !empty($location['lng'])
                            ) {
                                $lat = $location['lat'];
                                $lng = $location['lng'];
                            } else {
                                $this->_application->logError('Faield fetching current entity lat/lng for sorting by distance.');
                            }
                        } else {
                            try {
                                $geo = $this->_application->Map_Api_geocode($args[1][0], false);
                                $lat = $geo['lat'];
                                $lng = $geo['lng'];
                            } catch (Exception\IException $e) {
                                $this->_application->logError('Faield fetching lat/lng of ' . $args[1][0] . ' for sorting by distance. Geocode error: ' . $e);
                            }
                        }
                        break;
                    default:
                        $lat = $args[1][0];
                        $lng = $args[1][1];
                }
            } else {
                if (isset($args[2])) {
                    $lat = $args[1];
                    $lng = $args[2];
                }
            }
        }
        if (empty($lat) || empty($lng)) return;
        
        $query->sortByExtraField('distance', $order)->addExtraField(
            'distance',
            $fieldName,
            sprintf(
                '(%1$d * acos(cos(radians(%3$.6F)) * cos(radians(%2$s.lat)) * cos(radians(%2$s.lng) - radians(%4$.6F)) + sin(radians(%3$.6F)) * sin(radians(%2$s.lat))))',
                $config['distance_unit'] === 'mi' ? 3959 : 6371,
                $fieldName,
                $lat,
                $lng
            ),
            true
        );
    }
    
    public function fieldSchemaProperties()
    {
        return array('address', 'geo', 'location');
    }
    
    public function fieldSchemaRenderProperty(Field\IField $field, $property, Entity\Type\IEntity $entity)
    {
        if (!$value = $entity->getFieldValue($field->getFieldName())) return;
     
        $ret = [];
        switch ($property) {
            case 'address':
            case 'location':
                foreach ($value as $_value) {
                    $ret[] = array(
                        '@type' => 'PostalAddress',
                        'addressCountry' => $_value['country'],
                        'addressLocality' => $_value['city'],
                        'addressRegion' => $_value['province'],
                        'postalCode' => $_value['zip'],
                        'streetAddress' => $_value['street'],
                    );
                }
                break;
            case 'geo':
                foreach ($value as $_value) {
                    $ret[] = array(
                        '@type' => 'GeoCoordinates',
                        'latitude' => $_value['lat'],
                        'longitude' => $_value['lng'],
                    );
                }
                break;
        }
        return $ret;
    }
    
    public function fieldQueryableInfo(Field\IField $field)
    {
        return array(
            'example' => 'New York USA,10',
            'tip' => __('Enter an address (no commas) to query by address. Enter two values (address, radius) separated with a comma to specify a search radius. Enter three values (latitude, longitude, radius) separated with commas to query by coordinates. Enter "_current_" for the address of the current post if any.', 'directories-pro'),
        );
    }
    
    public function fieldQueryableQuery(Field\Query $query, $fieldName, $paramStr, Entity\Model\Bundle $bundle = null)
    {
        if (!$field = $this->_application->Entity_Field($bundle, $fieldName)) return;
        
        if (!$params = $this->_queryableParams($paramStr)) return;
                
        switch (count($params)) {
            case 1:
                if ($params[0] === '_current_') {
                    if (!isset($GLOBALS['drts_entity'])
                        || (!$location = $GLOBALS['drts_entity']->getSingleFieldValue($fieldName))
                        || empty($location['address'])
                    ) return;
                        
                    $params[0] = $location['address'];
                }
                $geo = $this->_application->Map_Api_geocode($params[0], false);
                $query->fieldIsOrGreaterThan($fieldName, $geo['viewport'][0], 'lat')
                    ->fieldIsOrSmallerThan($fieldName, $geo['viewport'][2], 'lat')
                    ->fieldIsOrGreaterThan($fieldName, $geo['viewport'][1], 'lng')
                    ->fieldIsOrSmallerThan($fieldName, $geo['viewport'][3], 'lng');
                return;
            case 2:
                if ($params[0] === '_current_') {
                    if (!isset($GLOBALS['drts_entity'])
                        || (!$location = $GLOBALS['drts_entity']->getSingleFieldValue($fieldName))
                        || empty($location['lat'])
                        || empty($location['lng'])
                    ) return;
                        
                    $lat = $location['lat'];
                    $lng = $location['lng'];
                } else {
                    $geo = $this->_application->Map_Api_geocode($params[0], false);
                    $lat = $geo['lat'];
                    $lng = $geo['lng'];
                }
                $radius = (int)$params[1];
                break;
            default:
                $lat = $params[0];
                $lng = $params[1];
                $radius = (int)$params[2];
                break;
        }
        $query->addCriteria($this->_application->Map_IsNearbyCriteria($lat, $lng, $field, $radius));
    }
    
    public function fieldOpenGraphProperties()
    {
        return array('business:contact_data', 'place:location');
    }
    
    public function fieldOpenGraphRenderProperty(Field\IField $field, $property, Entity\Type\IEntity $entity)
    {
        if (!$value = $entity->getSingleFieldValue($field->getFieldName())) return;
        
        switch ($property) {
            case 'business:contact_data':
                return array(
                    'business:contact_data:street_address' => $value['street'],
                    'business:contact_data:locality' => $value['city'],
                    'business:contact_data:region' => $value['province'],
                    'business:contact_data:postal_code' => $value['zip'],
                    'business:contact_data:country_name' => $value['country'],
                );
            case 'place:location':
                return array(
                    'place:location:latitude' => $value['lat'],
                    'place:location:longitude' => $value['lng'],
                );
        }
    }
    
    public function fieldHumanReadableText(Field\IField $field, Entity\Type\IEntity $entity, $separator = null, $key = null)
    {
        if (!$values = $entity->getFieldValue($field->getFieldName())) return '';
        
        $ret = [];
        foreach ($values as $value) {
            $ret[] = $value['address'];
        }
        return implode(isset($separator) ? $separator : PHP_EOL, $ret);
    }
    
    public function fieldConditionableInfo(Field\IField $field)
    {
        if (!$location_bundle = $this->_application->Entity_Bundle('location_location', $field->Bundle->component, $field->Bundle->group)) return;
        
        return [
            'term_id' => [
                'compare' => ['value', '!value', 'one', 'empty', 'filled'],
                'tip' => __('Enter taxonomy term IDs and/or slugs separated with commas.', 'directories-pro'),
                'example' => '1,5,new-york',
                //'label' => $location_bundle->getLabel('singular'),
            ],
        ];
    }
    
    public function fieldConditionableRule(Field\IField $field, $compare, $value = null, $name = '')
    {
        switch ($compare) {
            case 'value':
            case '!value':
            case 'one':
                $value = trim($value);
                if (strpos($value, ',')) {
                    if (!$value = explode(',', $value)) return;
                    
                    $value = array_map('trim', $value);
                }
                return ['type' => $compare, 'value' => $value, 'target' => '.drts-map-location-address-component'];
            case 'empty':
                return ['type' => 'filled', 'value' => false, 'target' => '.drts-map-location-address-component'];
            case 'filled':
                return ['type' => 'empty', 'value' => false, 'target' => '.drts-map-location-address-component'];
            default:
                return;
        }
    }
}