<?php
namespace SabaiApps\Directories\Component\Map;

use SabaiApps\Directories\Component\AbstractComponent;
use SabaiApps\Directories\Component\Form;
use SabaiApps\Directories\Component\Field;
use SabaiApps\Directories\Component\Entity;
use SabaiApps\Directories\Application;
use SabaiApps\Directories\Context;

class MapComponent extends AbstractComponent implements 
    Form\IFields,
    Field\ITypes,
    Field\IWidgets,
    Field\IRenderers
{
    const VERSION = '1.1.13', PACKAGE = 'directories',
        LAT_REGEX = '/^-?([1-8]?[1-9]|[1-9]?0)\.{1}\d{5,}/',
        LNG_REGEX = '/^-?((([1]?[0-7][0-9]|[1-9]?[0-9])\.{1}\d{5,}$)|[1]?[1-8][0]\.{1}0{5,}$)/';
    
    public static function description()
    {
        return 'Adds geographic metadata and features to content.';
    }
    
    public static function interfaces()
    {
        return array(
            'Faker\IGenerators',
        );
    }

    public function formGetFieldTypes()
    {
        return array('map_map', 'map_latlng');
    }

    public function formGetField($name)
    {
        switch ($name) {
            case 'map_map':
                return new FormField\MapFormField($this->_application, $name);
            case 'map_latlng':
                return new FormField\LatLngFormField($this->_application, $name);
        }
    }

    public function fieldGetTypeNames()
    {
        return array('map_map');
    }

    public function fieldGetType($name)
    {
        switch ($name) {
            case 'map_map':
                return new FieldType\MapFieldType($this->_application, $name);
        }
    }
    
    public function fieldGetRendererNames()
    {
        return array('map_map', 'map_street_view');
    }

    public function fieldGetRenderer($name)
    {
        switch ($name) {
            case 'map_map':
                return new FieldRenderer\MapFieldRenderer($this->_application, $name);
            case 'map_street_view':
                return new FieldRenderer\StreetViewFieldRenderer($this->_application, $name);
        }
    }

    public function fieldGetWidgetNames()
    {
        return array('map_map');
    }

    public function fieldGetWidget($name)
    {
        switch ($name) {
            case 'map_map':
                return new FieldWidget\MapFieldWidget($this->_application, $name);
        }
    }
    
    public function getDefaultConfig()
    {
        return [
            'map' => [
                'type' => 'roadmap',
                'distance_unit' => 'km',
                'style' => '',
                'default_zoom' => 9,
                'default_location' => array('lat' => 40.69847, 'lng' => -73.95144),
                'scrollwheel' => false,
                'marker_custom' => true,
                'marker_color' => null,
                'marker_size' => 38,
                'marker_icon' => 'fas fa-dot-circle',
                'marker_icon_color' => null,
                'marker_clusters' => false,
                'marker_cluster_color' => null,
                'fit_bounds' => true,
            ],
            'api' => [
                'key' => null,
            ],
        ];
    }
    
    public function onFormScripts($options)
    {
        if (empty($options) || in_array('map_markermap', $options)) {
            $this->_application->Map_Api_load(array('markermap' => true));
        }
    }
    
    public function fakerGetGeneratorNames()
    {
        return array('map_map');
    }
    
    public function fakerGetGenerator($name)
    {
        return new FakerGenerator\MapFakerGenerator($this->_application, $name);
    }
    
    public function onDirectoryAdminSettingsFormFilter(&$form)
    {
        $map = $this->_config['map'];
        $form['#tabs'][$this->_name] = array(
            '#title' => __('Map', 'directories'),
            '#weight' => 5,
        );
        $form[$this->_name] = array(
            '#tree' => true,
            '#component' => $this->_name,
            '#tab' => $this->_name,
            'map' => array(
                '#title' => __('Map Settings', 'directories'),
                'type' => array(
                    '#type' => 'select',
                    '#title' => __('Default map type', 'directories'),
                    '#options' => array(
                        'roadmap' => __('Google (roadmap)', 'directories'),
                        'satellite' => __('Google (satellite)', 'directories'),
                        'hybrid' => __('Google (hybrid)', 'directories'),
                        'osm' => __('OpenStreetMap', 'directories'),
                    ),
                    '#default_value' => isset($map['type']) ? $map['type'] : null,
                    '#horizontal' => true,
                ),
                'distance_unit' => array(
                    '#type' => 'select',
                    '#title' => __('Distance unit', 'directories'),
                    '#options' => array('km' => __('Kilometers', 'directories'), 'mi' => __('Miles', 'directories')),
                    '#default_value' => isset($map['distance_unit']) ? $map['distance_unit'] : null,
                    '#horizontal' => true,
                ),
                'style' => array(
                    '#type' => 'select',
                    '#options' => array('' => 'Google Maps') + $this->_application->Map_Api_mapStyle(),
                    '#title' => __('Default map style', 'directories'),
                    '#default_value' => isset($map['style']) ? $map['style'] : null,
                    '#horizontal' => true,
                ),
                'default_zoom' => array(
                    '#type' => 'slider',
                    '#min_value' => 0,
                    '#max_value' => 19,
                    '#integer' => true,
                    '#title' => __('Default zoom level', 'directories'),
                    '#default_value' => isset($map['default_zoom']) ? $map['default_zoom'] : null,
                    '#horizontal' => true,
                ),
                'default_location' => array(
                    '#type' => 'map_map',
                    '#title' => __('Default location', 'directories'),
                    '#map_height' => 200,
                    '#default_value' => isset($map['default_location']) ? $map['default_location'] : null,
                    '#horizontal' => true,
                ),
                'scrollwheel' => array(
                    '#type' => 'checkbox',
                    '#default_value' => !empty($map['scrollwheel']),
                    '#title' => __('Enable scrollwheel zooming', 'directories'),
                    '#horizontal' => true,
                ),
                'marker_custom' => array(
                    '#type' => 'checkbox',
                    '#default_value' => !empty($map['marker_custom']),
                    '#title' => __('Enable custom markers', 'directories'),
                    '#horizontal' => true,
                ),
                'marker_color' => array(
                    '#type' => 'colorpicker',
                    '#title' => __('Custom marker color', 'directories'),
                    '#default_value' => isset($map['marker_color']) ? $map['marker_color'] : null,
                    '#horizontal' => true,
                    '#states' => array(
                        'visible' => array(
                            'input[name="' . $this->_name . '[map][marker_custom]"]' => array('type' => 'checked', 'value' => true),
                        ), 
                    ),
                ),
                'marker_size' => array(
                    '#type' => 'slider',
                    '#title' => __('Custom marker size', 'directories'),
                    '#default_value' => isset($map['marker_size']) ? $map['marker_size'] : 38,
                    '#min_value' => 32,
                    '#max_value' => 80,
                    '#integer' => true,
                    '#field_suffix' => 'px',
                    '#horizontal' => true,
                    '#states' => array(
                        'visible' => array(
                            'input[name="' . $this->_name . '[map][marker_custom]"]' => array('type' => 'checked', 'value' => true),
                        ), 
                    ),
                ),
                'marker_icon' => array(
                    '#type' => 'icon',
                    '#title' => __('Custom marker icon', 'directories'),
                    '#default_value' => empty($map['marker_icon']) ? 'fa-dot-circle' : $map['marker_icon'],
                    '#horizontal' => true,
                    '#states' => array(
                        'visible' => array(
                            'input[name="' . $this->_name . '[map][marker_custom]"]' => array('type' => 'checked', 'value' => true),
                        ), 
                    ),
                ),
                'marker_icon_color' => array(
                    '#type' => 'colorpicker',
                    '#title' => __('Custom marker icon color', 'directories'),
                    '#default_value' => isset($map['marker_icon_color']) ? $map['marker_icon_color'] : null,
                    '#horizontal' => true,
                    '#states' => array(
                        'visible' => array(
                            'input[name="' . $this->_name . '[map][marker_custom]"]' => array('type' => 'checked', 'value' => true),
                        ), 
                    ),
                ),
                'marker_clusters' => array(
                    '#type' => 'checkbox',
                    '#default_value' => !empty($map['marker_clusters']),
                    '#title' => __('Enable marker clusters', 'directories'),
                    '#horizontal' => true,
                ),
                'marker_cluster_color' => array(
                    '#type' => 'colorpicker',
                    '#title' => __('Marker cluster color', 'directories'),
                    '#default_value' => isset($map['marker_cluster_color']) ? $map['marker_cluster_color'] : null,
                    '#states' => array(
                        'visible' => array(
                            'input[name="' . $this->_name . '[map][marker_clusters]"]' => array('type' => 'checked', 'value' => true),
                        ), 
                    ),
                    '#horizontal' => true,
                ),
                'fit_bounds' => array(
                    '#type' => 'checkbox',
                    '#title' => __('Fit all markers inside map', 'directories'),
                    '#default_value' => !empty($map['fit_bounds']),
                    '#horizontal' => true,
                ),
            ),
            'api' => array(
                '#title' => __('Google Maps API Settings', 'directories'),
                'key' => array(
                    '#type' => 'textfield',
                    '#title' => __('Google Maps API key', 'directories'),
                    '#default_value' => isset($this->_config['api']['key']) ? $this->_config['api']['key'] : null,
                    '#horizontal' => true,
                ),
                'no' => array(
                    '#type' => 'checkbox',
                    '#title' => __('Do not load Google Maps API', 'directories'),
                    '#default_value' => !empty($this->_config['api']['no']),
                    '#horizontal' => true,
                ),
                'no_admin' => array(
                    '#type' => 'checkbox',
                    '#title' => __('Do not load Google Maps API in admin dashboard', 'directories'),
                    '#default_value' => !empty($this->_config['api']['no_admin']),
                    '#horizontal' => true,
                ),
            ),
        );
        if (empty($this->_config['api']['key'])
            || !empty($this->_config['api']['no_admin'])
        ) {
            $form[$this->_name] ['map']['default_location']['#type'] = 'fieldset';
            $form[$this->_name] ['map']['default_location']['#tree'] = true;
            $form[$this->_name] ['map']['default_location']['lat'] = array(
                '#type' => 'textfield',
                '#maxlength' => 20,
                '#field_prefix' => $this->_application->H(__('Latitude', 'directories')),
                '#default_value' => isset($map['default_location']['lat']) ? $map['default_location']['lat'] : 40.69847,
                '#regex' => self::LAT_REGEX,
                '#numeric' => true,
            );
            $form[$this->_name] ['map']['default_location']['lng'] = array(
                '#type' => 'textfield',
                '#maxlength' => 20,
                '#field_prefix' => $this->_application->H(__('Longitude', 'directories')),
                '#default_value' => isset($map['default_location']['lng']) ? $map['default_location']['lng'] : -73.95144,
                '#regex' => self::LNG_REGEX,
                '#numeric' => true,
            );
        }
    }
}