<?php
namespace SabaiApps\Directories\Component\Map\Helper;

use SabaiApps\Directories\Application;
use SabaiApps\Directories\Exception;

class ApiHelper
{
    /**
     * Loads GoogleMaps API
     * @param Application $application
     * @param array $options
     */
    public function load(Application $application, array $options = [])
    {
        $config = $application->getComponent('Map')->getConfig();
        $platform = $application->getPlatform()
            ->addCssFile('map-map.min.css', 'drts-map-map', array('drts'), 'directories');
        if (empty($config[$platform->isAdmin() ? 'no_admin' : 'no'])) {
            $platform->addJsFile($this->_getApiUrl($application, '/js', array('libraries' => 'places', 'language' => $this->_getLanguage($application)), true, '//'), 'google-maps', null, false);
        }   
        if (!empty($options['map'])) {
            $platform->addJsFile('map-map.min.js', 'drts-map-map', array('google-maps', 'drts'), 'directories', true)
                ->addJsFile('map-util.min.js', 'drts-map-util', array('google-maps', 'drts'), 'directories', true);
        }
        if (!empty($options['style'])) {
            $platform->addJsFile($this->mapStyle($application, $options['style'], true), 'drts-map-style-' . str_replace(' ', '-', strtolower($options['style'])), 'google-maps', false, true);
        } 
        if (!empty($options['autocomplete'])) {
            $platform->addJsFile('map-util.min.js', 'drts-map-util', array('google-maps', 'drts'), 'directories', true);
        }
        if (!empty($options['markermap'])) {
            $platform->addJsFile('map-markermap.min.js', 'drts-map-markermap', array('google-maps', 'drts'), 'directories', true)
                ->addJsFile('map-util.min.js', 'drts-map-util', array('google-maps', 'drts'), 'directories', true)
                ->addJsFile('jquery.fitmaps.min.js', 'jquery-fitmaps', 'jquery', 'directories', true, true);
        }
        if (!empty($options['markerclusterer'])) {  
            $platform->addJsFile('markerclusterer.min.js', 'markerclusterer', 'google-maps', 'directories', true, true);
            if (!empty($options['markercluster_color'])) {
                $platform->addCss(
                    '.drts-map-map .cluster {background-color: rgba(' . implode(',', sscanf($options['markercluster_color'], '#%02x%02x%02x')) . ',0.5) !important;}
.drts-map-map .cluster > div {background-color: ' . $options['markercluster_color'] . ' !important;}',
                    'drts-map-map'
                );
            }
        }
    }
    
    /**
     * @param Application $application
     * @param string $query
     * @param bool $cache
     * @param bool $latlng
     * @param array $params
     * @throw Exception\RuntimeException
     */
    public function geocode(Application $application, $query, $cache = true, $latlng = false, array $params = [])
    {
        $query = trim($query);
        if (!$cache
            || ($hash = md5(serialize(array($query, $latlng, $params))))
            || (!$data = $application->getPlatform()->getCache('map_api_geocode'))
            || !isset($data[$hash])
        ) {
            $geocoded = $this->_doGeocode($application, $query, $latlng, $params);
            if (!$cache) return $geocoded;
            
            // Init cache
            if (!is_array($data)) {
                $data = [];
            } else {
                if (count($data) > 100) {
                    array_shift($data);
                }
            }
            // Append to cache
            $data[$hash] = $geocoded;
            $application->getPlatform()->setCache($data, 'map_api_geocode');
        }
        return $data[$hash];
    }
    
    /**
     * @param Application $application
     * @param string $query
     * @param bool $latlng
     * @throw Exception\RuntimeException
     */
    protected function _doGeocode(Application $application, $query, $isLatLng = false, array $params = [])
    {
        $url = $this->_getApiUrl($application, '/geocode/json', $params + [
            'language' => $this->_getLanguage($application),
            $isLatLng ? 'latlng' : 'address' => $query,
        ]);
        $geocode = $this->_sendRequest($application, $url);
        $geometry = $geocode->results[0]->geometry;
        return [
            'lat' => $geometry->location->lat,
            'lng' => $geometry->location->lng,
            'address' => $geocode->results[0]->formatted_address,
            'viewport' => [
                $geometry->viewport->southwest->lat,
                $geometry->viewport->southwest->lng, 
                $geometry->viewport->northeast->lat,
                $geometry->viewport->northeast->lng,
            ],
        ] + $this->_getAddressComponents($geocode->results[0]->address_components);
    }
    
    public function timezone(Application $application, $lat, $lng)
    {
        $url = $this->_getApiUrl($application, '/timezone/json', [
            'timestamp' => time(),
            'location' => $lat . ',' . $lng,
        ]);
        return $this->_sendRequest($application, $url)->timeZoneId;
    }
    
    public function streetviewMeta(Application $application, $lat, $lng)
    {
        $url = $this->_getApiUrl($application, '/streetview/metadata', [
            'location' => $lat . ',' . $lng,
        ]);
        return $this->_sendRequest($application, $url)->timeZoneId;
    }
    
    protected function _sendRequest(Application $application, $url, $decodeJson = true)
    {
        $result = $application->getPlatform()->remoteGet($url);
        if ($decodeJson) {
            if (!$result = json_decode($result)) {
                throw new Exception\RuntimeException('Failed parsing result returned from URL: ' . $url);
            }
            if ($result->status !== 'OK') {
                $error = isset($result->error_message) ? $result->error_message : 'An error occurred while querying Google Maps API.';
                throw new Exception\RuntimeException($error . ' Requested URL: ' . $url . '; Returned status: ' . $result->status);
            }
        }
        return $result;
    }
    
    protected function _getAddressComponents(array $components)
    {
        $ret = array('street_numer' => '', 'street' => '', 'city' => '', 'province' => '', 'zip' => '', 'country' => '');
        foreach ($components as $component) {
            switch ($component->types[0]) {
                case 'country':
                    $ret['country'] = $component->short_name;
                    continue;
                case 'postal_code':
                    $ret['zip'] = $component->long_name;
                    continue;
                case 'administrative_area_level_1':
                    $ret['province'] = $component->long_name;
                    continue;
                case 'locality':
                case 'sublocality':
                    $ret['city'] = $component->long_name;
                    continue;
                case 'route':
                    $ret['street'] = $component->long_name;
                    continue;
                case 'street_number':
                    $ret['street_number'] = $component->long_name;
                    continue;
            }
        }
        return $ret;
    }
    
    protected function _getApiUrl(Application $application, $path, array $params, $includeKey = true, $protocol = 'https://')
    {
        foreach ($params as $key => $value) {
            $params[$key] = $key . '=' . urlencode($value);
        }
        $path = rtrim($path, '?') . '?' . implode('&', $params);
        if ($includeKey
            && ($api_key = $application->getComponent('Map')->getConfig('api', 'key'))
        ) {
            $path .= '&key=' . urlencode($api_key);
        }
        return $protocol . 'maps.googleapis.com/maps/api' . $path;
    }
    
    private static $_lang, $_languages = array(
        'ar', 'bg', 'bn', 'ca', 'cs', 'da', 'de', 'el', 'en', 'en-AU', 'en-GB', 'es', 'eu',
        'fa', 'fi', 'fil', 'fr', 'gl', 'gu', 'he', 'hi', 'hr', 'hu', 'id', 'it', 'iw', 'ja', 'kn',
        'ko', 'lt', 'lv', 'ml', 'mr', 'nl', 'nn', 'no', 'or', 'pl', 'pt', 'pt-BR', 'pt-PT',
        'rm', 'ro', 'ru', 'sk', 'sl', 'sr', 'sv', 'tl', 'ta', 'te', 'th', 'tr', 'uk', 'vi',
        'zh-CN', 'zh-TW'
    );

    protected function _getLanguage(Application $application, $reset = false)
    {
        $language = $application->getPlatform()->getLocale();
        if (!isset(self::$_lang) || $reset) {
            if (strpos($language, '_')) {
                $langs = array_reverse(explode('_', $language));
                $langs[0] = strtolower($langs[1]) . '-' . strtoupper($langs[0]);
            } elseif (strpos($language, '-')) {
                $langs = array_reverse(explode('-', $language));
                $langs[0] = strtolower($langs[1]) . '-' . strtoupper($langs[0]);
            } else {
                $langs = array(strtolower($language));
            }
            self::$_lang = 'en';
            foreach ($langs as $lang) {
                if (in_array($lang, self::$_languages)) {
                    self::$_lang = $lang;
                    break;
                }
            }
        }
        return self::$_lang;
    }
    
    public function mapStyle(Application $application, $style = null, $url = false)
    {
        if (!$styles = $application->getPlatform()->getCache('map_api_styles')) {
            $styles = array('Red' => null, 'Blue' => null, 'Greyscale' => null, 'Night' => null, 'Sepia' => null, 'Chilled' => null,
                'Mixed' => null, 'Pale Dawn' => null, 'Apple Maps-esque' => null, 'Paper' => null, 'Hot Pink' => null, 'Flat Map' => null,
                'Subtle' => null, 'Light Monochrome' => null, 'Bright and Bubbly' => null, 'Clean Grey' => null, 'Subtle Greyscale' => null,
                'Light Dream' => null,
            );
            $styles = $application->Filter('map_api_styles', $styles);
            ksort($styles);
            $application->getPlatform()->setCache($styles, 'map_api_styles');
        }

        if (!isset($style)) return array_combine(array_keys($styles), array_keys($styles));
            
        $file = isset($styles[$style]) ? $styles[$style] : 'map-style-' . str_replace(' ', '-', strtolower($style)) . '.min.js';

        if (!$url) return $file;
        
        return strpos($file, 'http') === 0 ? $file : $application->JsUrl($file, 'directories');
    }
}