<?php
namespace SabaiApps\Directories\Component\Field\Renderer;

use SabaiApps\Directories\Component\Entity;
use SabaiApps\Directories\Component\Field\IField;

class ImageRenderer extends AbstractRenderer
{
    protected $_fieldTypes;
    
    protected function _fieldRendererInfo()
    {
        return array(
            'field_types' => array('wp_image', 'file_image'),
            'default_settings' => array(
                'size' => 'thumbnail',
                'width' => 100,
                'height' => 0,
                'cols' => 4,
                'link' => 'photo',
                'link_image_size' => 'large',
            ),
            'separatable' => false,
            'no_imageable' => true,
        );
    }

    protected function _fieldRendererSettingsForm(IField $field, array $settings, array $parents = [])
    {
        $form = array(
            'size' => array(
                '#title' => __('Image size', 'directories'),
                '#type' => 'select',
                '#options' => $this->_getImageSizeOptions(),
                '#default_value' => $settings['size'],
                '#weight' => 1,
            ),
            'width' => array(
                '#title' => __('Image width', 'directories'),
                '#type' => 'slider',
                '#min_value' => 1,
                '#max_value' => 100,
                '#integer' => true,
                '#default_value' => $settings['width'],
                '#weight' => 2,
                '#field_suffix' => '%',
            ),
            'height' => array(
                '#title' => __('Image height', 'directories'),
                '#type' => 'slider',
                '#min_value' => 0,
                '#min_text' => 'auto',
                '#max_value' => 100,
                '#integer' => true,
                '#default_value' => $settings['height'],
                '#weight' => 2,
                '#field_suffix' => 'px',
            ),
            'link' => array(
                '#type' => 'select',
                '#title' => __('Link image to', 'directories'),
                '#options' => $this->_getImageLinkTypeOptions(),
                '#inline' => true,
                '#default_value' => $settings['link'],
                '#weight' => 5,
            ),
            'link_image_size' => array(
                '#title' => __('Linked image size', 'directories'),
                '#type' => 'select',
                '#options' => $this->_getLinkedImageSizeOptions(),
                '#inline' => true,
                '#default_value' => $settings['link_image_size'],
                '#states' => array(
                    'visible' => array(
                        sprintf('select[name="%s[link]"]', $this->_application->Form_FieldName($parents)) => array('value' => 'photo'),
                    ),
                ),
                '#weight' => 6,
            ),
        );           
        if ($field->getFieldMaxNumItems() !== 1) {
            $form['cols'] = array(
                '#title' => __('Number of columns', 'directories'),
                '#type' => 'select',
                '#options' => array(1 => 1, 2 => 2, 3 => 3, 4 => 4, 6 => 6, 12 => 12),
                '#default_value' => $settings['cols'],
                '#weight' => 3,
            );
        }
        return $form;
    }
    
    protected function _getImageLinkTypeOptions()
    {
        return [
            'none' => __('Do not link', 'directories'),
            'page' => __('Link to page', 'directories'),
            'photo' => __('Single image', 'directories'),
        ];
    }
    
    protected function _getLinkedImageSizeOptions()
    {
        return [
            'medium' => __('Medium size', 'directories'),
            'large' => __('Large size', 'directories'),
            '' => __('Original size', 'directories'),
        ];
    }

    protected function _fieldRendererRenderField(IField $field, array &$settings, Entity\Type\IEntity $entity, array $values, $more = 0)
    {
        $permalink_url = $settings['link'] === 'page' ? $this->_application->Entity_PermalinkUrl($entity) : null;

        if (empty($values)
            || (!$field_type_impl = $this->_application->Field_Type($field->getFieldType(), true))
        ) {
            $settings['_render_background'] = true;
            unset($settings['_hover_zoom'], $settings['_hover_brighten']); // disable hover effects
            return ['url' => $permalink_url];
        }
        
        // Return image and link URLs only for rendering field as background image
        if (!empty($settings['_render_background'])) {
            return [
                'html' => $field_type_impl->fieldImageGetUrl($values[0], $settings['size']),
                'url' => $settings['link'] === 'page'
                    ? $permalink_url
                    : ($settings['link'] !== 'none' ? $field_type_impl->fieldImageGetUrl($values[0], $settings['link_image_size']) : null)
            ];
        }
        
        if ($field->getFieldMaxNumItems() !== 1) {
            $col_sm = $col = 12 / $settings['cols'];
            if ($col_sm < 6) {
                $col = 6;
            }
        } else {
            $col_sm = $col = 12;
        }
        if ($col_sm === 12 && count($values) === 1) {
            return $this->_getImage($settings, $values[0], $field_type_impl, $permalink_url);
        }
        
        unset($settings['_hover_zoom'], $settings['_hover_brighten']); // disable hover effects if multiple images
        $ret = array('<div class="' . DRTS_BS_PREFIX . 'row ' . DRTS_BS_PREFIX . 'no-gutters">');
        foreach ($values as $value) {
            $image = $this->_getImage($settings, $value, $field_type_impl, $permalink_url);
            $ret[] = sprintf(
                '<div class="%1$scol-sm-%2$d %1$scol-%3$d">%4$s</div>',
                DRTS_BS_PREFIX,
                $col_sm,
                $col,
                isset($image['url']) ? '<a href="' . $image['url'] . '">' . $image['html'] . '</a>' : $image['html']
            );
        }
        $ret[] = '</div>';
        return implode(PHP_EOL, $ret);
    }
    
    protected function _getImage(array $settings, $value, $fieldType, $permalinkUrl, $image = null)
    {
        $ret = [
            'html' => sprintf(
                '<img src="%s" data-full-image="%s" title="%s" alt="" style="width:%d%%;height:%s" />',
                $url = $fieldType->fieldImageGetUrl($value, $settings['size']),
                $fieldType->fieldImageGetUrl($value, 'full'),
                $this->_application->H($fieldType->fieldImageGetTitle($value)),
                $settings['width'],
                empty($settings['height']) ? 'auto' : intval($settings['height']) . 'px'
            ),
        ];
        if ($settings['link'] === 'page') {
            $ret['url'] = $permalinkUrl;
        } elseif ($settings['link'] !== 'none') {
            $ret['url'] = $settings['size'] == $settings['link_image_size'] ? $url : $fieldType->fieldImageGetUrl($value, $settings['link_image_size']);
        }
        
        return $ret;
    }
    
    public function fieldRendererGetBackgroundUrl(IField $field, $value, $size)
    {
        if (!$field_type = $this->_application->Field_Type($field->getFieldType(), true)) return;
        
        return $field_type->fieldImageGetUrl($value, $size);
    }
    
    protected function _fieldRendererReadableSettings(IField $field, array $settings)
    {
        $ret = [
            'size' => [
                'label' => __('Image size', 'directories'),
                'value' => $this->_getImageSizeOptions()[$settings['size']],
            ],
            'width' => [
                'label' => __('Image width', 'directories'),
                'value' => $settings['width'] . '%',
            ],
            'height' => [
                'label' => __('Image height', 'directories'),
                'value' => empty($settings['height']) ? 'auto' : $settings['height'] . 'px',
            ],
            'link' => [
                'label' => __('Link image to', 'directories'),
                'value' => $this->_getImageLinkTypeOptions()[$settings['link']],
            ],
        ];
        if ($settings['link'] === 'photo') {
            $ret['link_image_size'] = [
                'label' => __('Linked image size', 'directories'),
                'value' => $this->_getLinkedImageSizeOptions()[$settings['link_image_size']],
            ];
        }          
        if ($field->getFieldMaxNumItems() !== 1) {
            $ret['cols'] = [
                'label' => __('Number of columns', 'directories'),
                'value' => $settings['cols'],
            ];
        }
        
        return $ret;
    }
}