<?php
/**
 * DeBlocker
 * Most effective way to detect ad blockers. Ask the visitors to disable their ad blockers.
 * Exclusively on https://1.envato.market/deblocker
 *
 * @encoding        UTF-8
 * @version         3.3.3
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 **/

namespace Merkulove\Deblocker;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

use Merkulove\Deblocker\Unity\Plugin;
use Merkulove\Deblocker\Unity\Settings;
use Merkulove\Deblocker\Unity\TabGeneral;

final class Config {

	/**
	 * The one true Settings.
	 *
     * @since 1.0.0
     * @access private
	 * @var Config
	 **/
	private static $instance;

	/**
	 * Divider unique id
	 * @var int
	 */
	private static $divider = 0;

    /**
     * Prepare plugin settings by modifying the default one.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function prepare_settings() {

        /** Get default plugin settings. */
        $tabs = Plugin::get_tabs();

        /** Change General tab title. */
        $tabs['general']['title'] = esc_html__( 'DeBlocker Settings', 'deblocker' );

        /** Set System Requirements. */
	    $tabs['status']['reports']['server']['bcmath_installed'] = false;

        # Algorithm
        $tabs['general']['fields']['algorithm'] = [
            'type'              => 'select',
            'label'             => esc_html__( 'Algorithm:', 'deblocker' ),
            'placeholder'       => esc_html__( 'Algorithm', 'deblocker' ),
            'description'       => esc_html__('The DeBlocker supports several algorithms. Choose the most suitable for your needs. ', 'deblocker' ) .
                esc_html__('Read more ', 'deblocker' ) .
                '<a href="https://docs.merkulov.design/algorithms-of-the-deblocker-wordpress-plugin/" target="_blank" rel="noopener">' .
                esc_html__('about algorithms', 'deblocker' ) .
                '</a>' .
                esc_html__(' in the documentation.', 'deblocker' ),
            'default'           => 'inline',
            'options'           => [
                'inline'        => esc_html__( 'Inline', 'deblocker' ),
                'random-folder' => esc_html__( 'Random Folder', 'deblocker' ),
                'proxy'         => esc_html__( 'Script Proxy', 'deblocker' ),
            ]
        ];

		# Random folder lifetime
	    $tabs['general']['fields']['lifetime'] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Lifetime:', 'deblocker' ),
			'description'       => esc_html__( 'Random folder lifetime ', 'deblocker' ) .
			                       ' <strong>' .
			                       esc_html( ( ! empty( Settings::get_instance()->options['lifetime'] ) ) ? Settings::get_instance()->options['lifetime'] : '1' ) .
			                       '</strong>' . esc_html__( ' days.', 'deblocker' ),
			'min'               => 1,
			'max'               => 365,
			'step'              => 1,
			'default'           => 1,
			'discrete'          => true,
		];

        # Modal Style
        $tabs['general']['fields']['style'] = [
            'type'              => 'select',
            'label'             => esc_html__( 'Modal Style:', 'deblocker' ),
            'placeholder'       => esc_html__( 'Modal Style', 'deblocker' ),
            'description'       => esc_html__('Deblocker modal window style.', 'deblocker' ),
            'default'           => 'compact',
            'options'           => [
                'compact'               => esc_html__( 'Compact', 'deblocker' ),
                'compact-right-top'     => esc_html__( 'Compact: Upper Right Corner', 'deblocker' ),
                'compact-left-top'      => esc_html__( 'Compact: Upper Left Corner', 'deblocker' ),
                'compact-right-bottom'  => esc_html__( 'Compact: Bottom Right Corner', 'deblocker' ),
                'compact-left-bottom'   => esc_html__( 'Compact: Bottom Left Corner', 'deblocker' ),
                'full'                  => esc_html__( 'Full Screen', 'deblocker' ),
            ]
        ];

        # Title
        $tabs['general']['fields']['title'] = [
            'type'              => 'text',
            'label'             => esc_html__( 'Title:', 'deblocker' ),
            'placeholder'       => esc_html__( 'Title', 'deblocker' ),
            'description'       => esc_html__( 'Modal window title.', 'deblocker' ),
            'default'           => esc_html__( 'It Looks Like You Have AdBlocker Enabled', 'deblocker' ),
            'attr'              => [
                'maxlength' => '1000'
            ]
        ];

        # Content
        $tabs['general']['fields']['content'] = [
            'type'              => 'editor',
            'label'             => esc_html__( 'Content:', 'deblocker' ),
            'placeholder'       => esc_html__( 'Please disable AdBlock to proceed to the destination page.', 'deblocker' ),
            'description'       => esc_html__( 'Please disable AdBlock to proceed to the destination page.', 'deblocker' ),
            'default'           => esc_html__( 'Please disable AdBlock to proceed to the destination page.', 'deblocker' ),
            'attr'              => [
                'textarea_rows' => '3'
            ]
        ];

	    $tabs['general']['fields'][ 'divider_' . $this::$divider ] = ['type' => 'divider', 'default' => ''];
	    $this::$divider++;

        # JavaScript Required
        $tabs['general']['fields']['javascript'] = [
            'type'              => 'switcher',
            'label'             => esc_html__( 'JavaScript Required:', 'deblocker' ),
            'placeholder'       => esc_html__( 'Protect if JavaScript is Disabled', 'deblocker' ),
            'description'       => esc_html__( 'Block page content if JS is disabled.', 'deblocker' ),
            'default'           => 'on',
        ];

        # JavaScript Message
        $tabs['general']['fields']['javascript_msg'] = [
            'type'              => 'editor',
            'label'             => esc_html__( 'JavaScript Message:', 'deblocker' ),
            'description'       => esc_html__( 'Message to show if JavaScript is Disabled.', 'deblocker' ),
            'default'           => '<h3>' . esc_html__( 'Please Enable JavaScript in your Browser to Visit this Site.', 'deblocker' ) . '<h3>',
            'attr'              => [
                'textarea_rows' => '3'
            ]
        ];

	    $tabs['general']['fields'][ 'divider_' . $this::$divider ] = ['type' => 'divider', 'default' => ''];
	    $this::$divider++;

		$tabs['general']['fields']['check_url'] = [
			'type' => 'text',
			'label' => esc_html__( 'Check URL:', 'deblocker' ),
			'placeholder' => esc_html__( 'Check URL', 'deblocker' ),
			'description' => esc_html__( 'URL to check if AdBlock is enabled.', 'deblocker' ),
			'default' => 'https://static.doubleclick.net/instream/ad_status.js',
		];

        /** Behaviour Tab */
	    $tabs = $this->tab_behaviour( $tabs );
	    $tabs = $this->refresh_settings( $tabs );
	    $tabs = $this->tab_behaviour( $tabs );

	    /** Style Tab */
	    $tabs = $this->tab_style( $tabs );
	    $tabs = $this->refresh_settings( $tabs );
	    $tabs = $this->tab_style( $tabs );

	    /** Translations Tab */
	    $tabs = $this->tab_translations( $tabs );
	    $tabs = $this->refresh_settings( $tabs );
	    $tabs = $this->tab_translations( $tabs );

        /** Set updated tabs. */
        Plugin::set_tabs( $tabs );

        /** Refresh settings. */
        Settings::get_instance()->get_options();

    }

	/**
	 * Refresh settings
	 *
	 * @param $tabs
	 *
	 * @return array
	 */
	private function refresh_settings( $tabs ) {

		/** Set updated tabs. */
		Plugin::set_tabs( $tabs );

		/** Refresh settings. */
		Settings::get_instance()->get_options();

		/** Get default plugin settings. */
		return Plugin::get_tabs();

	}

	/**
	 * Create Behaviour tab.
	 *
	 * @param $tabs - List of tabs with all settings and fields.
	 *
	 * @return array - List of tabs with all settings and fields.
	 */
	private function tab_behaviour( $tabs ): array {

		/** Shorthand access to plugin settings. */
		$options = Settings::get_instance()->options;

		/** Add Behaviour tab after General. */
		$offset = 1; // Position for new tab.
		$tabs = array_slice( $tabs, 0, $offset, true ) +
		        ['behaviour' => [
			        'enabled'       => true,
			        'class'         => TabGeneral::class,
			        'label'         => esc_html__( 'Behaviour', 'readabler' ),
			        'title'         => esc_html__( 'Behaviour Settings', 'readabler' ),
			        'show_title'    => true,
			        'icon'          => 'track_changes',
			        'fields'        => []
		        ] ] +
		        array_slice( $tabs, $offset, NULL, true );

		$tabs['behaviour']['fields']['timeout'] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Delay:', 'deblocker' ),
			'description'       => esc_html__( 'Modal window or redirect act after ', 'deblocker' ) .
			                       ' <strong>' .
			                       esc_html( ( ! empty( Settings::get_instance()->options['timeout'] ) ) ? Settings::get_instance()->options['timeout'] : '0' ) .
			                       '</strong>' . esc_html__( ' milliseconds.', 'deblocker' ),
			'min'               => 0,
			'max'               => 10000,
			'step'              => 100,
			'default'           => 0,
			'discrete'          => true,
		];

		$tabs['behaviour']['fields']['cross'] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Cross button', 'deblocker' ),
			'placeholder'       => esc_html__( 'Cross button', 'deblocker' ),
			'description'       => esc_html__( 'Add a X button to close the popup in the top corner of the popup content.', 'deblocker' ),
			'default'           => 'on',
		];

		$tabs['behaviour']['fields']['cross_delay'] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Close button delay', 'deblocker' ) . ':',
			'description'       => esc_html__( 'Delay until the close button appear ', 'deblocker' ) .
			                       ' <strong>' .
			                       esc_attr( ( ! empty( Settings::get_instance()->options['cross_delay'] ) ) ? Settings::get_instance()->options['cross_delay'] : '0' ) .
			                       '</strong>' . esc_html__( ' milliseconds.', 'deblocker' ),
			'min'               => 0,
			'max'               => 10000,
			'step'              => 500,
			'default'           => 1000,
			'discrete'          => true,
		];

		$tabs['behaviour']['fields']['button'] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Text button', 'deblocker' ),
			'placeholder'       => esc_html__( 'Text button', 'deblocker' ),
			'description'       => esc_html__( 'Add a text button to close the popup at the end of the popup content', 'deblocker' ),
			'default'           => 'off',
		];

		$tabs['behaviour']['fields']['button_caption'] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Button caption:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Button caption', 'deblocker' ),
			'description'       => esc_html__( 'Caption for close button', 'deblocker' ),
			'default'           => esc_html__( 'I have disabled AdBlocker', 'deblocker' ),
			'attr'              => [
				'maxlength' => '1000'
			]
		];

		$tabs['behaviour']['fields']['guide'] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Enable guide', 'deblocker' ),
			'placeholder'       => esc_html__( 'Guide', 'deblocker' ),
			'description'       => esc_html__( 'Display additional information about how to disable the AdBlocker in browser', 'deblocker' ),
			'default'           => 'off',
		];

		$tabs['behaviour']['fields']['blur'] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Blur Content:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Blur Content', 'deblocker' ),
			'description'       => esc_html__( 'Effects like blur or color shifting on an element\'s rendering before the element is displayed.', 'deblocker' ),
			'default'           => 'on',
		];

		$tabs['behaviour']['fields']['is_loop'] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Popup loop:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Loop', 'deblocker' ),
			'description'       => esc_html__( 'Show popup until blocker is disabled', 'deblocker' ),
			'default'           => 'off',
		];

		$tabs['behaviour']['fields']['loop_timeout'] = [
			'type'              => 'slider',
			'label'             => esc_html__( 'Popup loop delay:', 'deblocker' ),
			'description'       => esc_html__( 'Interval between popup openings in a loop is ', 'deblocker' ) .
			                       ' <strong>' .
			                       esc_html( ( ! empty( Settings::get_instance()->options['loop_timeout'] ) ) ? Settings::get_instance()->options['loop_timeout'] : '0' ) .
			                       '</strong>' . esc_html__( ' seconds.', 'deblocker' ),
			'min'               => 1,
			'max'               => 30,
			'step'              => 1,
			'default'           => 5,
			'discrete'          => true,
		];

		$tabs['behaviour']['fields']['is_redirect'] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Redirect', 'deblocker' ),
			'placeholder'       => esc_html__( 'Redirect', 'deblocker' ),
			'description'       => esc_html__( 'Redirect to another URL instead of a display of popup', 'deblocker' ),
			'default'           => 'off',
		];

		$tabs['behaviour']['fields']['redirect'] = [
			'type'              => 'text',
			'label'             => esc_html__( 'Redirect URL:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Redirect URL', 'deblocker' ),
			'description'       => esc_html__( 'Redirect link', 'deblocker' ),
			'default'           => '',
		];

		/** If no options on this tab - disable it. */
		if ( ! count( $tabs['behaviour']['fields'] ) ) {
			$tabs['behaviour']['enabled'] = false;
		}

		return $tabs;

	}

	/**
	 * Translation tab
	 * @param $tabs
	 *
	 * @return array|array[]
	 */
	private function tab_translations( $tabs ): array {

		/** Shorthand access to plugin settings. */
		$options = Settings::get_instance()->options;

		/** Add Behaviour tab after General. */
		$offset = 3; // Position for new tab.
		$tabs = array_slice( $tabs, 0, $offset, true ) +
		        [ 'translations' => [
			        'enabled'       => true,
			        'class'         => TabGeneral::class,
			        'label'         => esc_html__( 'Translations', 'readabler' ),
			        'title'         => esc_html__( 'Multilingual content', 'readabler' ),
			        'show_title'    => true,
			        'icon'          => 'translate',
			        'fields'        => []
		        ] ] +
		        array_slice( $tabs, $offset, NULL, true );

		$tabs['translations']['fields']['translations'] = [
			'type'              => 'switcher',
			'label'             => esc_html__( 'Advanced translations:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Advanced translations', 'deblocker' ),
			'description'       => esc_html__( 'The option enables advanced translation strings. It allows to show content for different languages on the frontend. By turning it off you can use the standard translation via the .pot file, with default plain texts.', 'deblocker' ),
			'default'           => 'off',
		];

		for ( $i = 1; $i <= 10; $i++ ) {

			$tabs['translations']['fields'][ 'divider_' . $this::$divider ] = ['type' => 'divider', 'default' => ''];
			$this::$divider++;

			$tabs['translations']['fields'][ 'translate_locale_' . $i ] = [
				'type'              => 'text',
				'label'             => esc_html__( 'Locale:', 'deblocker' ),
				'placeholder'       => esc_html__( 'Locale', 'deblocker' ),
				'description'       => esc_html__( 'Language code, for example', 'deblocker' ) . ' <code>en</code>, ' . esc_html__( 'or', 'deblocker' ) . ' <code> en-US</code>.',
				'default'           => '',
				'attr'              => [
					'class'     => 'translations-locale',
				]
			];

			$tabs['translations']['fields'][ 'translate_title_' . $i ] = [
				'type'              => 'text',
				'label'             => esc_html__( 'Title:', 'deblocker' ),
				'placeholder'       => esc_html__( 'Title', 'deblocker' ),
				'description'       => esc_html__( 'Modal window content', 'deblocker' ),
				'default'           => esc_html__( 'It Looks Like You Have AdBlocker Enabled', 'deblocker' ),
			];

			$tabs['translations']['fields'][ 'translate_content_' . $i ] = [
				'type'              => 'editor',
				'label'             => esc_html__( 'Content:', 'deblocker' ),
				'description'       => '',
				'default'           => esc_html__( 'Please disable AdBlock to proceed to the destination page.', 'deblocker' ),
				'attr'              => [
					'textarea_rows' => '3'
				]
			];

			if ( $options[ 'button' ] === 'on' ) {

				$tabs['translations']['fields'][ 'translate_button_caption_' . $i ] = [
					'type'              => 'text',
					'label'             => esc_html__( 'Button caption:', 'deblocker' ),
					'placeholder'       => esc_html__( 'Button caption', 'deblocker' ),
					'description'       => esc_html__( 'Caption for close button', 'deblocker' ),
					'default'           => esc_html__( 'I have disabled AdBlocker', 'deblocker' ),
				];

			}

		}

		$tabs['translations']['fields']['translate_add' ] = [
			'type'              => 'button',
			'placeholder'       => esc_html__( 'Add translation', 'deblocker' ),
			'default'           => '',
			'icon'              => 'add',
			'attr'              => [
				'class'     => 'mdc-button--outlined mdp-add-translation',
				'id'        => 'mdp-add-translation'
			]
		];

		$tabs['translations']['fields']['translate_remove' ] = [
			'type'              => 'button',
			'placeholder'       => esc_html__( 'Remove translation', 'deblocker' ),
			'default'           => '',
			'icon'              => 'remove',
			'attr'              => [
				'class'     => 'mdc-button--outlined mdp-remove-translation',
				'id'        => 'mdp-remove-translation'
			]
		];

		return $tabs;

	}

	/**
	 * Style tab
	 * @param $tabs
	 *
	 * @return array|array[]
	 */
	private function tab_style( $tabs ): array {

		/** Shorthand access to plugin settings. */
		$options = Settings::get_instance()->options;

		/** Add Behaviour tab after General. */
		$offset = 2; // Position for new tab.
		$tabs = array_slice( $tabs, 0, $offset, true ) +
		        [ 'style' => [
			        'enabled'       => true,
			        'class'         => TabGeneral::class,
			        'label'         => esc_html__( 'Style', 'readabler' ),
			        'title'         => esc_html__( 'Style Settings', 'readabler' ),
			        'show_title'    => true,
			        'icon'          => 'format_paint',
			        'fields'        => []
		        ] ] +
		        array_slice( $tabs, $offset, NULL, true );

		$tabs['style']['fields']['align'] = [
			'type'              => 'select',
			'label'             => esc_html__( 'Content align:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Content align', 'deblocker' ),
			'description'       => esc_html__('Choose the content alignment inside the popup', 'deblocker' ),
			'default'           => 'left',
			'options'           => [
				'left'          => esc_html__( 'Left', 'deblocker' ),
				'center' => esc_html__( 'Center', 'deblocker' ),
				'right'         => esc_html__( 'Right', 'deblocker' ),
			]
		];

		$tabs['style']['fields'][ 'divider_' . $this::$divider ] = ['type' => 'divider', 'default' => ''];
		$this::$divider++;

		# Overlay Color
		$tabs['style']['fields']['bg_color'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Overlay Color:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Overlay Color', 'deblocker' ),
			'description'       => esc_html__( 'Page overlay Background Color.', 'deblocker' ),
			'default'           => 'rgba(255,0,0,0.75)',
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		# Modal Color
		$tabs['style']['fields']['modal_color'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Modal Color:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Modal Color', 'deblocker' ),
			'description'       => esc_html__( 'Modal window Background Color.', 'deblocker' ),
			'default'           => 'rgba(255,255,255,1)',
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$tabs['style']['fields']['close_color'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Close Button Color:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Close Button Color', 'deblocker' ),
			'description'       => esc_html__( 'Close button color inside a modal window.', 'deblocker' ),
			'default'           => '#23282d',
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];

		$tabs['style']['fields']['text_color'] = [
			'type'              => 'colorpicker',
			'label'             => esc_html__( 'Text Color:', 'deblocker' ),
			'placeholder'       => esc_html__( 'Text Color', 'deblocker' ),
			'description'       => esc_html__( 'Text color inside a modal window.', 'deblocker' ),
			'default'           => '#23282d',
			'attr'              => [
				'readonly'      => 'readonly',
			]
		];


		if ( $options[ 'button' ] === 'on' ) {

			$tabs['style']['fields'][ 'divider_' . $this::$divider ] = ['type' => 'divider', 'default' => ''];
			$this::$divider++;

			$tabs['style']['fields']['button_style'] = [
				'type'              => 'switcher',
				'label'             => esc_html__( 'Button style:', 'deblocker' ),
				'placeholder'       => esc_html__( 'Custom button styles', 'deblocker' ),
				'description'       => esc_html__( 'Enable or disable theme styles for the button', 'deblocker' ),
				'default'           => 'off',
			];

			$tabs['style']['fields']['button_bg'] = [
				'type'              => 'colorpicker',
				'label'             => esc_html__( 'Button Background Color:', 'deblocker' ),
				'placeholder'       => esc_html__( 'Background Color', 'deblocker' ),
				'description'       => esc_html__( 'Background color of the close button', 'deblocker' ),
				'default'           => 'rgba(255,0,0,0.75)',
				'attr'              => [
					'readonly'      => 'readonly',
				]
			];

			$tabs['style']['fields']['button_text'] = [
				'type'              => 'colorpicker',
				'label'             => esc_html__( 'Button text color:', 'deblocker' ),
				'placeholder'       => esc_html__( 'Text Color', 'deblocker' ),
				'description'       => esc_html__( 'Text color of the close button', 'deblocker' ),
				'default'           => 'rgba(255,255,255,1)',
				'attr'              => [
					'readonly'      => 'readonly',
				]
			];

			$tabs['style']['fields']['button_hover_bg'] = [
				'type'              => 'colorpicker',
				'label'             => esc_html__( 'Button hover background:', 'deblocker' ),
				'placeholder'       => esc_html__( 'Text Color', 'deblocker' ),
				'description'       => esc_html__( 'Background color of the close button in hover state', 'deblocker' ),
				'default'           => 'rgba(255,0,0,1)',
				'attr'              => [
					'readonly'      => 'readonly',
				]
			];

			$tabs['style']['fields']['button_hover_text'] = [
				'type'              => 'colorpicker',
				'label'             => esc_html__( 'Button hover text:', 'deblocker' ),
				'placeholder'       => esc_html__( 'Text Color', 'deblocker' ),
				'description'       => esc_html__( 'Text color of the close button in hover state', 'deblocker' ),
				'default'           => 'rgba(255,255,255,1)',
				'attr'              => [
					'readonly'      => 'readonly',
				]
			];

		}

		return $tabs;

	}

	/**
	 * Main Settings Instance.
	 * Insures that only one instance of Settings exists in memory at any one time.
	 *
	 * @static
     * @since 1.0.0
     * @access public
     *
	 * @return Config
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
