<?php
/**
 * DeBlocker
 * Most effective way to detect ad blockers. Ask the visitors to disable their ad blockers.
 * Exclusively on https://1.envato.market/deblocker
 *
 * @encoding        UTF-8
 * @version         3.3.3
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Alexander Khmelnitskiy (info@alexander.khmelnitskiy.ua), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 **/

namespace Merkulove\Deblocker;

use RuntimeException;
use Merkulove\Deblocker\Unity\Plugin;
use Merkulove\Deblocker\Unity\Helper;
use Merkulove\Deblocker\Unity\Settings;
use Merkulove\Deblocker\Unity\TabAssignments;
use Merkulove\Deblocker\Unity\TabActivation;
use Merkulove\Deblocker\Unity\UI;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

final class Caster {

	/**
	 * The one true Caster.
	 *
     * @since 1.0.0
     * @access private
	 * @var Caster
	 **/
	private static $instance;

    /**
     * Set up the plugin.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function setup() {

        /** Define hooks that runs on both the front-end and the dashboard. */
        $this->both_hooks();

        /** Define public hooks. */
        $this->public_hooks();

        /** Define admin hooks. */
        $this->admin_hooks();

    }

    /**
     * Define hooks that runs on both the front-end and the dashboard.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function both_hooks() { }

    /**
     * Register all the hooks related to the public-facing functionality.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function public_hooks() {

        /** Work only on frontend area. */
        if ( is_admin() ) { return; }

        /** Load JavaScript for Frontend Area. */
        add_action( 'wp_enqueue_scripts', [$this, 'scripts'] ); // JS.

        /** JavaScript Required. */
        add_action( 'wp_footer', [$this, 'javascript_required'] );

        /** We need Sessions for Script Proxy algorithm */
        if ( Settings::get_instance()->options['algorithm'] === 'proxy' ) {

	        add_action( 'init', [ Helper::class, 'start_session' ], 1 );

	        if ( ! defined( 'HMW_VERSION' ) ) { // Add session only if HideMyWP is not active
		        add_action( 'wp_logout', [ Helper::class, 'end_session' ] );
		        add_action( 'wp_login', [ Helper::class, 'end_session' ] );
	        }

        }

        /** Run one of selected algorithm. */
        $this->run_algorithm();

        /** Add cloudflare data-attribute for script tag in random folder algorithm */
	    add_filter( 'script_loader_tag', [ $this, 'add_cloudflare_data_attribute' ], 10, 2);

    }

    /**
     * Register all the hooks related to the admin area functionality.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function admin_hooks() {

        /** Work only in admin area. */
        if ( ! is_admin() ) { return; }

        /**
         * The adBlock extensions blocks us in admin area too, so we add scripts and styles inline.
         **/

        /** Remove Unity css and JS. */
//        add_action( 'admin_enqueue_scripts', [$this, 'dequeue_unity_assets'], 1000 );

        /** Add inline JS and CSS for Backend Area. */
//        add_action( 'admin_footer', [ $this, 'admin_styles' ], 1000 );
//        add_action( 'admin_footer', [ $this, 'admin_scripts' ], 1000 );

	    /** Show activation warning */
	    add_action( 'admin_footer', [ $this, 'not_activated_notice' ] );

	    /** Add not-activated class to the admin body */
	    add_filter( 'admin_body_class', [ $this, 'not_activated_class' ] );

        /** Change Custom CSS description */
        add_filter( 'deblocker_custom_css_description', function ( $description ) {

            return esc_html__( 'Add custom css for frontend rendering. Use', 'deblocker' ) . ' <code>{{deblocker}}</code> ' .  esc_html__( 'to replace the randomly generated CSS prefix.', 'deblocker' );

        } );

        /** Recreate random folder */
        add_filter( 'admin_menu', [ $this, 'recreate_random_folder' ] );

    }

	/**
     * Re-create random folder with new settings
	 * @return void
	 */
	public function recreate_random_folder() {

        if ( ! isset( $_GET['settings-updated'] ) ) { return; }

		$algorithm = Settings::get_instance()->options['algorithm'];
        if ( 'random-folder' !== $algorithm ) { return; }

		/** Exit if settings saving was not successful. */
        if ( ! isset( $_GET[ 'tab' ] ) ) { return; }
		if ( 'style' !== $_GET['tab'] && 'behaviour' !== $_GET['tab'] ) { return; }
		if ( 'true' !== $_GET['settings-updated'] ) { return; }

		/** Delete random folder generation time transient */
        delete_transient( 'mdp_deblocker_random_folder_generated' );

		add_filter( 'admin_footer_text', function () {

			UI::get_instance()->render_snackbar(
				esc_html__( 'Random folder has been regenerated with new style settings', 'deblocker' )
			);

        } );

	}

    /**
     * Protect site if JavaScript is Disabled.
     *
     * @since 2.0.0
     * @access public
     **/
    public function javascript_required() {

        /** Custom JavaScript is not allowed in AMP. */
        if ( function_exists( 'is_amp_endpoint' ) && is_amp_endpoint() ) { return; }

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return; }

        /** Get plugin settings */
        $options = Settings::get_instance()->options;

        if ( 'on' !== $options['javascript'] ) { return; }

        ob_start();
        ?>
        <noscript>
            <div id='mdp-deblocker-js-disabled'>
                <div><?php echo wp_kses_post( $options['javascript_msg'] ); ?></div>
            </div>
            <style>
                #mdp-deblocker-js-disabled {
                    position: fixed;
                    top: 0;
                    left: 0;
                    height: 100%;
                    width: 100%;
                    z-index: 999999;
                    text-align: center;
                    background-color: #FFFFFF;
                    color: #000000;
                    font-size: 40px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }
            </style>
        </noscript>
        <?php
        $result = ob_get_clean();

        echo $result;

    }

    /**
     * Add JavaScript for the public-facing side of the site.
     *
     * @return void
     * @since 1.0.0
     **/
    public function scripts() {

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return; }

	    $ads_handle = get_option( 'mdp_deblocker_random_folder_fake_prefix', 'mdp-deblocker-ads' );
        wp_enqueue_script( $ads_handle, Plugin::get_url() . 'js/ads' . Plugin::get_suffix() . '.js', [], false, false );

    }

    /**
     * Remove Unity css and JS.
     *
     * @since 3.0.0
     * @access private
     *
     * @return void
     **/
    public function dequeue_unity_assets() {

        /** Dequeue CSS. */
        wp_dequeue_style( 'mdp-plugins' );
        wp_dequeue_style( 'mdp-deblocker-ui' );
        wp_dequeue_style( 'mdp-deblocker-unity-admin' );
        wp_dequeue_style( 'mdp-deblocker-admin' );
        wp_dequeue_style( 'mdp-deblocker-plugin-install' );

        /** Dequeue JS. */
        wp_dequeue_script( 'mdp-deblocker-ui' );
        wp_dequeue_script( 'mdp-deblocker-unity-admin' );
        wp_dequeue_script( 'mdp-deblocker-admin' );
        wp_dequeue_script( 'mdp-plugins' );
        wp_dequeue_script( 'mdp-deblocker-assignments' );

    }

    /**
     * Add JS for admin area.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
    public function admin_scripts() {

        /** Get current screen to add styles on specific pages. */
        $screen = get_current_screen();
        if ( null === $screen ) { return; }

        /** Plugin Settings Page. */
        if ( in_array( $screen->base, Plugin::get_menu_bases(), true ) ) {

            ?>
            <script>
                window.mdpDeblocker = {
                    "ajaxURL":"<?php echo esc_url( admin_url('admin-ajax.php') ); ?>",
                    "nonce":"<?php esc_attr_e( wp_create_nonce( 'deblocker-unity' ) ); ?>"
                };
                <?php echo file_get_contents( Plugin::get_path() . 'src/Merkulove/Unity/assets/js/merkulov-ui' . Plugin::get_suffix() . '.js' ); ?>
                <?php echo file_get_contents( Plugin::get_path() . 'src/Merkulove/Unity/assets/js/assignments' . Plugin::get_suffix() . '.js' ); ?>
                <?php echo file_get_contents( Plugin::get_path() . 'src/Merkulove/Unity/assets/js/admin' . Plugin::get_suffix() . '.js' ); ?>
                <?php echo file_get_contents( Plugin::get_path() . 'js/admin' . Plugin::get_suffix() . '.js' ); ?>
            </script>
            <?php

        }

        /** Add script only on WP Plugins page. */
        if ( 'plugins' === $screen->base ) {

            ?>
            <script>
                <?php echo file_get_contents( Plugin::get_path() . 'src/Merkulove/Unity/assets/js/plugins' . Plugin::get_suffix() . '.js' ); ?>
            </script>
            <?php

         }

    }

    /**
     * Add CSS for admin area.
     *
     * @since 3.0.0
     * @access public
     *
     * @return void
     **/
    public function admin_styles() {

        /** Get current screen to add styles on specific pages. */
        $screen = get_current_screen();
        if ( null === $screen ) { return; }

        /** Plugin Settings Page. */
        if ( in_array( $screen->base, Plugin::get_menu_bases(), true ) ) {

            ?>
            <style>
                <?php echo file_get_contents( Plugin::get_path() . 'src/Merkulove/Unity/assets/css/merkulov-ui' . Plugin::get_suffix() . '.css' ); ?>
                <?php echo file_get_contents( Plugin::get_path() . 'src/Merkulove/Unity/assets/css/admin' . Plugin::get_suffix() . '.css' ); ?>
                <?php echo file_get_contents( Plugin::get_path() . 'css/admin' . Plugin::get_suffix() . '.css' ); ?>
            </style>
            <?php

        }

        /** Plugin popup on update. Styles only for our plugin. */
        if ( ( 'plugin-install' === $screen->base ) && isset( $_GET[ 'plugin' ] ) && $_GET[ 'plugin' ] === 'deblocker' ) {

            ?><style><?php echo file_get_contents( Plugin::get_path() . 'src/Merkulove/Unity/assets/css/plugin-install' . Plugin::get_suffix() . '.css' ); ?></style><?php

        }

        /** Add styles only on WP Plugins page. */
        if ( 'plugins' === $screen->base ) {

            ?><style><?php echo file_get_contents( Plugin::get_path() . 'src/Merkulove/Unity/assets/css/plugins' . Plugin::get_suffix() . '.css' ); ?></style><?php

        }

    }

    /**
     * Run one of selected algorithm.
     *
     * @since  2.0.1
     * @access private
     *
     * @return void
     **/
    private function run_algorithm() {

        /** Get algorithm from plugin settings. */
        $algorithm = Settings::get_instance()->options['algorithm'];

        if ( 'inline' === $algorithm ) {

            $this->inline_algorithm();

        } elseif ( 'random-folder' === $algorithm ) {

            $this->random_folder_algorithm();

            /** Proxies all scripts. */
        } elseif ( 'proxy' === $algorithm && TabActivation::get_instance()->is_activated() ) {

            $this->proxy_algorithm();

        }

    }

    /**
     * Pass variables to JS.
     *
     * @return void
     * @since 2.0.2
     **/
    public function localize_deblocker( $script_handle ) {

	    $fake_folder = get_option( 'mdp_deblocker_random_folder_fake_folder', 'mdp-deblocker' );
	    $object_name = get_option( 'mdp_deblocker_random_folder_fake_prefix', 'mdpDeblocker' );

        wp_localize_script( $script_handle, $object_name, $this->deblocker_localization( $fake_folder ) );

    }

    public function deblocker_localization( $fake_folder ) {

	    /** Get plugin settings. */
	    $options = Settings::get_instance()->options;

	    $content = $options['content'] !== strip_tags( $options['content'] ) ?
		    wp_kses_post( $options['content'] ): // HTML
		    esc_html__( $options['content'], 'deblocker' ); // Plain text

	    // Basic strings
	    $js = array(
		    'title' => esc_html__( $options['title'], 'deblocker' ),
		    'content' => $content,
		    'folder' => $fake_folder
	    );

	    // Button Strings
	    if ( $options[ 'button' ] === 'on' ) {

		    $button = array(
			    'button_caption' => esc_html__( $options['button_caption'], 'deblocker' ),
		    );
		    $js = array_merge( $js, $button );

	    }

	    // Guide strings
	    if ( $options[ 'guide' ] === 'on' ) {

		    $guide_list         = '<ol><li>';
		    $guide_list         .= esc_html__( 'Right click on the ad blocker extension icon at the top right corner of your browser', 'deblocker' );
		    $guide_list         .= '</li><li>';
		    $guide_list         .= esc_html__( 'From the menu choose', 'deblocker' ) . ' <b>' . esc_html__( '"Disable on this site"', 'deblocker' ) . '</b> ' . esc_html__( 'or', 'deblocker' ) . ' <b>' . esc_html__( '"Pause on this site"', 'deblocker' ) . ' </b>';
		    $guide_list         .= '</li><li>';
		    $guide_list         .= esc_html__( 'Refresh the page if not automatically refreshed', 'deblocker' ) ;
		    $guide_list         .= '</li></ol>';

		    $guide = array(
			    'guide_trigger'      => esc_html__( 'How do I disable my ad blocker?', 'deblocker' ),
			    'guide_trigger_ok'   => esc_html__( 'OK. I understand.', 'deblocker' ),
			    'guide_title'        => esc_html__( 'To disable ad blocker on this site:', 'deblocker' ),
			    'guide_list'         => $guide_list
		    );

		    $js = array_merge( $js, $guide );

	    }

        return $js;

    }

    /**
     * Add obfuscated inline script on page.
     * Very fast. But low reliability.
     *
     * @since  2.0.1
     * @access private
     * @return void
     **/
    private function inline_algorithm() {

        /** Load inline JavaScript for Frontend Area. */
        add_action( 'wp_footer', [$this, 'footer_scripts'], mt_rand( 1, 60 ) ); // JS.

    }

    /**
     * Load inline JavaScript for Frontend Area.
     *
     * @return void
     * @since 1.0.0
     **/
    public function footer_scripts() {

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return; }

        /**
         * Get Randomized Script.
         * @noinspection PhpIncludeInspection
         **/
        $js = require Plugin::get_path() . 'src/Merkulove/Deblocker/DeBlockerJS.php';

        ?><script><?php echo self::obfuscate( $js ); ?></script><?php

    }

    /**
     * Obfuscate JS code.
     *
     * @param $js
     *
     * @since  2.0.0
     * @access public
     * @return string
     **/
    public static function obfuscate( $js ) {

        $hunter = new Obfuscator( $js );

        $parse = parse_url( site_url() );
        $domain_name = $parse['host'];

        $hunter->addDomainName( $domain_name );
        $hunter->setExpiration('+3 day'); // Expires after 3 days

        return $hunter->Obfuscate();

    }

    /**
     * Create a random folder once at day. A quick, fairly reliable way to bypass ad blockers.
     *
     * @since  2.0.1
     * @access private
     * @return void
     **/
    private function random_folder_algorithm() {

        /** Is the plugins folder writable? */
        if ( ! is_writable( WP_PLUGIN_DIR ) ) {

            /** Switch to inline algorithm, as the safest in this case. */
            $this->inline_algorithm();

            return;
        }

        /** Is it time to generate a new folder? */
        $opt_name = 'mdp_deblocker_random_folder_generated';
        $generated = get_transient( $opt_name );

        /** Generate new random folder. */
        if ( false === $generated ) {

            $options = Settings::get_instance()->options;

            /** Get the name of the old folder. */
            $fake_folder = get_option( 'mdp_deblocker_random_folder_fake_folder', 'deblocker' );

            /** Remove old folder. */
            if ( $fake_folder ) {
                Helper::get_instance()->remove_directory( WP_PLUGIN_DIR . '/' . $fake_folder );
            }

            /** Create new folder with scripts. */
            $this->random_folder_create_fake_folder();

            /** Regenerate after 24 hours. */
            set_transient( $opt_name, 'true', intval( $options[ 'lifetime' ] ) * DAY_IN_SECONDS ); // 24 hours

        }

        /** Load JavaScript for Frontend Area. */
        add_action( 'wp_enqueue_scripts', [$this, 'random_folder_algorithm_scripts'] ); // JS.

    }

    /**
     * Add JavaScript for the public-facing side of the site.
     * For random folder algorithm.
     *
     * @return void
     * @since 2.0.1
     **/
    public function random_folder_algorithm_scripts() {

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return; }

        $folder_name = get_option( 'mdp_deblocker_random_folder_fake_folder', 'deblocker' );
        $file_name = get_option( 'mdp_deblocker_random_folder_fake_file', 'deblocker' );

        /** If script file not exist. */
        if ( ! file_exists( WP_PLUGIN_DIR . '/' . $folder_name . '/' . $file_name ) ) {
            $this->random_folder_create_fake_folder(); // Create new folder with scripts
        } else {
            $script_handle = $folder_name ?? 'mdp-deblocker';
	        $script_url = WP_PLUGIN_URL . '/' . $folder_name . '/' . $file_name;
	        wp_enqueue_script( esc_attr( $script_handle ), esc_url( $script_url ), [], false, true );
	        $this->localize_deblocker( $script_handle );
        }

    }

	/**
     * Filter handler for script tag
	 * @param $tag
	 * @param $handle
	 *
	 * @return array|mixed|string|string[]
	 */
    public function add_cloudflare_data_attribute( $tag, $handle ) {

	    $ads_handle = get_option( 'mdp_deblocker_random_folder_fake_prefix', 'mdp-deblocker-ads' );
        if ( ! in_array( $handle, [ 'mdp-deblocker', $ads_handle ] ) )
            return $tag;

        return str_replace( ' src', ' data-cfasync=false src', $tag );

    }

    /**
     * Create random folder and random script file.
     *
     * @since  2.0.1
     * @access private
     * @return void
     **/
    private function random_folder_create_fake_folder() {

        /** Create Random Names. */
        $dir = $this->generate_random_name();
        $file = $this->generate_random_name() . '.js';

        /** Create and store prefix */
        $prefix = $this->generate_random_name( 'safe' );
	    update_option( 'mdp_deblocker_random_folder_fake_prefix', $prefix );

        /** Create Folder. */
        $path = WP_PLUGIN_DIR . '/' . $dir;
        if ( ! $this->make_random_directory( $path ) ) { throw new RuntimeException( sprintf( 'Directory "%s" was not created', $path ) ); }

        /**
         * Get Randomized Script.
         * @noinspection PhpIncludeInspection
         **/
        $deblocker_script = require Plugin::get_path() . 'src/Merkulove/Deblocker/DeBlockerJS.php';
        file_put_contents( $path . '/' . $file, $deblocker_script );

        /** Remember folder and script name. */
        update_option( 'mdp_deblocker_random_folder_fake_folder', $dir );
        update_option( 'mdp_deblocker_random_folder_fake_file', $file );

    }

	/**
     * Make readabler directory
	 * @param $directory
	 *
	 * @return bool
	 */
    private function make_random_directory( $directory ) {

	    return mkdir( $directory );

    }

    /**
     * Return random alphanumeric name.
     *
     * @since  2.0.1
     * @access private
     * @return string
     **/
    public function generate_random_name( $case = '' ) {

        $permitted_chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        if ( $case === '' ) {
	        $permitted_chars .= '-_!@()+';
        }

        /** Prepare random parts. */
        $part_1 = substr( str_shuffle( $permitted_chars ), 0, mt_rand( 4, 16 ) );
        $part_2 = substr( str_shuffle( $permitted_chars ), 0, mt_rand( 4, 16 ) );
        $part_3 = substr( str_shuffle( '0123456789' ), 0, mt_rand( 0, 2 ) );

        /** Add random dash. */
        $dash = '';
        if ( $case === '' && mt_rand( 0, 1 ) ) {
            $dash = '-';
        }

        /** Add random wp. */
        $wp = '';
        if ( mt_rand( 0, 1 ) ) {
            $wp = 'wp';
        }

        return $wp . $part_1 . $dash . $part_2 . $dash . $part_3;

    }

    /**
     * Most powerful algorithm. Proxies all scripts and is randomly added self to the end of someone.
     * The disadvantages include a slight slowdown in loading and unstable operation with some caching systems.
     *
     * @since  2.0.1
     * @access private
     * @return void
     **/
    private function proxy_algorithm() {

        if ( ! TabActivation::get_instance()->is_activated() ) { return; }

        /** Let's try to saddle any other script to avoid blocking. */
        add_action( 'wp_print_scripts', [$this, 'list_scripts'], PHP_INT_MAX );

        /** Return proxied scripts. */
        add_action( 'template_redirect', [$this, 'do_stuff_on_404'] );

    }

    /**
     * Let's try to saddle any other script to avoid blocking.
     *
     * @since 2.0.0
     * @access private
     * @return void
     **/
    public function list_scripts() {

        global $wp_scripts;

        /** Checks if plugin should work on this page. */
        if ( ! TabAssignments::get_instance()->display() ) { return; }

        /** Prepare relative paths to plugins and themes. */
        $rel_plugin_path = str_replace( ABSPATH, '', WP_PLUGIN_DIR );
        $rel_theme_path = str_replace( ABSPATH, '', get_theme_root() );

        /** Create MD5 hashes. */
        $md5_plugin_path = md5( $rel_plugin_path );
        $md5_theme_path = md5( $rel_theme_path );

        /** Select random script. */
        $victim = $this->get_random_script_number( $wp_scripts->queue );

        /** Exit if no one victim is in queue */
        if ( $victim === 0 ) { return; }

        /** Replace paths to MD5 hashes. */
        foreach( $wp_scripts->queue as $key => $handle ) {

            /** Remember victims. */
            if ( $victim === $key ) {
                $_SESSION['mdb_deblocker_victim'] = $wp_scripts->registered[$handle]->src;
            }

            /** We send all scripts through our handler. */
            $wp_scripts->registered[$handle]->src = str_replace( $rel_plugin_path, $md5_plugin_path, $wp_scripts->registered[$handle]->src );
            $wp_scripts->registered[$handle]->src = str_replace( $rel_theme_path, $md5_theme_path, $wp_scripts->registered[$handle]->src );

        }

    }

	/**
	 * @param $queue
	 *
	 * @return false|int|string
	 */
    private function get_random_script_number( $queue ) {

        global $wp_scripts;

        if ( ! is_array( $queue ) ) { return 0; }

        /** Let's remove our own script. */
	    $ads_handle = get_option( 'mdp_deblocker_random_folder_fake_prefix', 'mdp-deblocker-ads' );
        if ( ( $key = array_search( $ads_handle, $queue, true ) ) !== false ) {
            unset( $queue[ $key ] );
        }

        /** Exit if no one script is in queue */
	    if ( empty( $queue ) ) { return 0; }

        /** Name of victim. */
        $victim_name = $queue[array_rand( $queue )];

        /** Key of victim. */
        $key =  array_search( $victim_name, $wp_scripts->queue, true );

        return $key;

    }

    /**
     * Return proxied scripts.
     *
     * @since 2.0.0
     * @access private
     *
     * @return void
     **/
    public function do_stuff_on_404() {

        global $wp;

        /** We are interested in requests for nonexistent files. */
        if ( ! is_404() ) { return; }

        /** We are interested in js files. */
        if ( 'js' !== strtolower( pathinfo( $wp->request, PATHINFO_EXTENSION ) ) ) { return; }

        /** Prepare relative paths to plugins and themes. */
        $rel_plugin_path = str_replace( ABSPATH, '', WP_PLUGIN_DIR );
        $rel_theme_path = str_replace( ABSPATH, '', get_theme_root() );

        /** Create MD5 hashes. */
        $md5_plugin_path = md5( $rel_plugin_path );
        $md5_theme_path = md5( $rel_theme_path );

        /** Reverse replace MD5 to path. */
        $url = $wp->request;

        $url = str_replace( [$md5_plugin_path, $md5_theme_path], [$rel_plugin_path, $rel_theme_path], $url );

        /** Path to script. */
        $script_path = ABSPATH . $url;

        $add = false;
        if ( isset( $_SESSION['mdb_deblocker_victim'] ) ) {
            $victim = $_SESSION['mdb_deblocker_victim'];
            if ( strpos( $victim, $url ) !== false ) {
                $add = true;
            }
        }

        /** Return script. */
        if ( file_exists( $script_path ) ) {
            header( 'HTTP/1.1 200 OK' );
            header( 'Content-Type: application/javascript' );

            echo self::get_js_contents( $script_path, $add );
            die();

        }

    }

    /**
     * Return js file and add deblocker scripts.
     *
     * @param      $path
     * @param bool $add
     *
     * @since  2.0.0
     * @access public
     * @return false|string
     **/
    public static function get_js_contents( $path, $add = false ) {

        $js = file_get_contents( $path );

        if ( $add ) {

            /**
             * Get Randomized Script.
             * @noinspection PhpIncludeInspection
             **/
            $d_js = require Plugin::get_path() . 'src/Merkulove/Deblocker/DeBlockerJS.php';

            $js .= self::obfuscate( $d_js );

        }

        return $js;

    }

    /**
     * This method used in register_activation_hook
     * Everything written here will be done during plugin activation.
     *
     * @since 1.0.0
     * @access public
     */
    public function activation_hook() {

        /** Activation hook */

    }

	/**
	 * Render Activation message.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @return void
	 **/
	public function not_activated_notice() {

		/** Get current screen. */
		$screen = get_current_screen();
		if ( null === $screen ) { return; }

		/** Readabler Settings Page. */
		if ( in_array( $screen->base ,Plugin::get_menu_bases() ) && ! TabActivation::get_instance()->is_activated() ) {

			/** Render "Before you start" message. */
			UI::get_instance()->render_snackbar(
				esc_html__( 'Activate your copy of the DeBlocker to enable additional blocking algorithms', 'deblocker' ),
				'info', // Type
				-1, // Timeout
				true, // Is Closable
				[ [ 'caption' => 'Activate', 'link' => get_admin_url('admin', 'admin.php?page=mdp_deblocker_settings&tab=activation' ) ] ] // Buttons
			);

		}

	}

	/**
	 * @param $classes
	 *
	 * @return string
	 */
	public function not_activated_class( $classes ) {

		if ( TabActivation::get_instance()->is_activated() ) { return $classes; }

		$my_class = 'mdp-readabler-not-activated';

		return $classes ? $classes . ' ' . $my_class : $my_class;

	}

	/**
	 * Main Caster Instance.
	 * Insures that only one instance of Caster exists in memory at any one time.
	 *
	 * @static
     * @since 1.0.0
     * @access public
     *
	 * @return Caster
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
