<?php

class Product_Protector_Frontend {

  function __construct() {

    if (get_option('product_protector_hide_price') === 'yes') {

      add_filter('woocommerce_get_price_html', array($this, 'product_protector_product_price_html'), 10, 2);

      add_action( 'woocommerce_show_variation_price', array($this, 'show_product_variation_price'), 10, 1);
    }

    add_action('woocommerce_before_add_to_cart_button', array($this, 'product_protector_before_add_to_cart_button'));

    add_filter('woocommerce_add_to_cart_validation', array($this, 'check_product_password'), 10, 3);
  }

  function show_product_variation_price($show) {
    // if single product page
    if (is_product()) {
      global $product;

      // if product is secured and not in cart yet
      if (
        self::is_secured_product($product->get_id())
        && !self::product_in_cart($product->get_id())
      ) {
        $show = false;
      }
    }

    return $show;
  }

  /**
   * Remove product price for protected products
   * @param $price
   * @param $product
   * @return mixed|string|void
   */
  function product_protector_product_price_html($price, $product) {

    // if product is secured and not in cart yet
    if (
      self::is_secured_product($product->get_id())
      && !self::product_in_cart($product->get_id())
    ) {
      $price = __('Price is hidden', PRODUCT_PROTECTOR_TD);
      $price = apply_filters('product_protector_hidden_price_text', $price);
    }

    return $price;
  }

  /**
   * Check if product is secured by password
   * @param $product_id
   * @return bool
   */
  public static function is_secured_product($product_id) {

    $product_secured = get_post_meta($product_id, 'product_protector_password_enabled_checkbox', true) === 'yes' ? true : false;

    if ($product_secured) return true;

    // let's try to locate category password for product..

    $categories_info = unserialize(get_option('product_protector_categories_info')); // get categories passwords

    $terms = wp_get_post_terms($product_id, 'product_cat');
    foreach ($terms as $term) {
      $product_categories[] = $term->term_id;
    }

    if (!empty($categories_info) && !empty($product_categories)) {

      foreach ($categories_info as $key => $value) {

        if (in_array($key, $product_categories) && !empty($value)) {

          return true;
        }
      }
    }

    return false;
  }

  /**
   * Get WooCommerce Version
   */
  public static function product_protector_get_woocommerce_version_number() {

    // If get_plugins() isn't available, require it
    if (!function_exists('get_plugins'))
      require_once(ABSPATH . 'wp-admin/includes/plugin.php');

    // Create the plugins folder and file variables
    $plugin_folder = get_plugins('/' . 'woocommerce');
    $plugin_file = 'woocommerce.php';

    // If the plugin version number is set, return it
    if (isset($plugin_folder[$plugin_file]['Version'])) {
      return $plugin_folder[$plugin_file]['Version'];
    } else {
      // Otherwise return null
      return NULL;
    }
  }

  /**
   * Get passwords of protected categories
   * @param $product_id
   * @return array
   */
  private static function get_protected_categories_passwords($product_id) {

    $categories_passwords = array();

    $categories_info = unserialize(get_option('product_protector_categories_info')); // get categories passwords

    $terms = wp_get_post_terms($product_id, 'product_cat');
    foreach ($terms as $term) {

      if (isset($term->term_id)) {
        $product_categories[] = $term->term_id;
      }
    }

    if (!empty($categories_info) && !empty($product_categories)) {

      ksort($categories_info);

      foreach ($categories_info as $key => $value) {

        if (in_array($key, $product_categories) && $value != "") {

          $categories_passwords[] = $value;
        }
      }
    }

    return $categories_passwords;
  }

  

  /**
   * Function of checking password for product while adding to cart
   * @param $valid
   * @param $product_id
   * @return bool
   */
  function check_product_password($valid, $product_id, $quantity) {

    // Priority: Global password > Category password > Individual

    global $woocommerce, $post;

    $global_product_pass = get_option('product_protector'); // get global password

    // get protected categories passwords
    $categories_passwords = self::get_protected_categories_passwords($product_id);

    // get individual password
    $protected_product = get_post_meta($product_id, 'product_protector_password_enabled_checkbox', true) === 'yes' ? true : false;
    $individual_product_pass = get_post_meta($product_id, 'product_protector_individual_password', true);

    $product_password = isset($_POST['product_password']) ? sanitize_title($_POST['product_password']) : '';

    if (self::is_secured_product($product_id) && !self::product_in_cart($product_id)) {

      // if individual password protection
      if (
        $protected_product
        && strlen($individual_product_pass) > 0
      ) {

        if ($product_password === $individual_product_pass) {
          return true;
        } else {
          $this->return_woocommerce_error(__('Wrong product password.', PRODUCT_PROTECTOR_TD));
          return false;
        }
      }

      // if category password exist
      if (!empty($categories_passwords)) {

        if (in_array($product_password, $categories_passwords)) {
          return true;
        } else {

          $this->return_woocommerce_error(__('Wrong product password.', PRODUCT_PROTECTOR_TD));
          return false;
        }
      }

      // if global password protection
      if ($product_password === $global_product_pass && $protected_product) {
        return true;
      } else {
        $this->return_woocommerce_error(__('Wrong product password.', PRODUCT_PROTECTOR_TD));
        return false;
      }
    }

    return true;
  }

  private function return_woocommerce_error($message) {

    global $woocommerce;

    $woocommerce_version = self::product_protector_get_woocommerce_version_number();

    if ($woocommerce_version >= 2.1) {
      wc_add_notice(sprintf($message), 'error');
    } else if ($woocommerce_version < 2.1) {
      $woocommerce->add_error($message);
    }
  }

  /**
   * Check if product is in cart
   * @param $product_id
   * @return bool
   */
  public static function product_in_cart($product_id) {
    global $woocommerce;

    return in_array( $product_id, array_column( $woocommerce->cart->get_cart(), 'product_id' ) );
  }

  /**
   * Add password input to the product page form.
   */
  function product_protector_before_add_to_cart_button() {

    global $post;
    $product_id = $post->ID;

    $product_password_label = __('Product password (required)', PRODUCT_PROTECTOR_TD);

    $product_password_label = apply_filters('product_protector_password_label', $product_password_label);

    if (self::is_secured_product($product_id) && !self::product_in_cart($product_id)) :
?>
      <div class="product-protector-row">
        <label class="product-protector-label"><?php echo $product_password_label ?></label>
        <input class="input-text product-protector-input" name="product_password" type="text" value="">
      </div>
      <br />
<?php
    endif;
  }
}
new Product_Protector_Frontend();
