<?php

class Product_Protector_Admin {

  function __construct() {

    add_action('woocommerce_product_options_general_product_data', array($this, 'product_protector_add_custom_general_fields'));
    add_action('woocommerce_process_product_meta', array($this, 'product_protector_add_custom_general_fields_save'));
    add_action('admin_menu', array($this, 'product_protector_add_menu'));
    add_action('admin_enqueue_scripts', array($this, 'product_protector_ajax_load_scripts'));
    add_action('wp_ajax_product-protector-update-password', array($this, 'product_protector_ajax_process_request'));
  }

  /**
   * Enqueue admin scripts
   */
  function product_protector_ajax_load_scripts() {
    // load our jquery file that sends the $.post request
    wp_enqueue_script("product-protector-admin-js", PRODUCT_PROTECTOR_URL . 'product-protector-admin-script.js', array('jquery'), PRODUCT_PROTECTOR_VERSION_NUM);

    // make the ajaxurl var available to the above script
    wp_localize_script("product-protector-admin-js", 'product_protector_vars', array(
      'ajaxurl' => admin_url('admin-ajax.php'),
      'errorMsg' => __('Some error occured.', PRODUCT_PROTECTOR_TD),
      'passwordDeletedMsg' => __('Password has been deleted.', PRODUCT_PROTECTOR_TD),
      'passwordUpdatedMsg' => __('Password has been updated.', PRODUCT_PROTECTOR_TD),
    ));
  }

  /**
   * Custom fields will be created here...
   */
  function product_protector_add_custom_general_fields() {

    global $woocommerce, $post;

    echo '<div class="options_group">';

    woocommerce_wp_checkbox(
      array(
        'id'            => 'product_protector_password_enabled_checkbox',
        'value'         => get_post_meta( get_the_ID(), 'product_protector_password_enabled_checkbox', true ),
        // 'wrapper_class' => 'show_if_simple',
        'label'         => __('Enable product protection (and override other passwords if any)', PRODUCT_PROTECTOR_TD),
        'description'   => __('Check me to enable request password for this product', PRODUCT_PROTECTOR_TD),
        'desc_tip'      => true
      )
    );

    woocommerce_wp_text_input(
      array(
        'id'          => 'product_protector_individual_password',
        'label'       => __('Set individual password', PRODUCT_PROTECTOR_TD),
        'description' => __('Note: This will override your global products password. To disable this - just leave field empty', PRODUCT_PROTECTOR_TD),
        'desc_tip'    => true
      )
    );

    echo '</div>';
  }

  /**
   * Custom fields will be saved here...
   */
  function product_protector_add_custom_general_fields_save($post_id) {

    $woocommerce_checkbox = isset($_POST['product_protector_password_enabled_checkbox']) ? 'yes' : 'no';
    update_post_meta($post_id, 'product_protector_password_enabled_checkbox', $woocommerce_checkbox);
    update_post_meta($post_id, 'product_protector_individual_password', sanitize_title($_POST['product_protector_individual_password']));
  }

  /**
   * Add menu page to admin panel
   */
  function product_protector_add_menu() {
    add_menu_page('Product protector', 'Protector', 'manage_options', 'product-protector', array($this, 'settings_page'), 'dashicons-lock');
  }

  /**
   * Settings page code
   */
  function settings_page() {

    $info_message = '';

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {

      // check nonce field first
      if (
        isset($_POST['product_protector_nonce_field'])
        && wp_verify_nonce($_POST['product_protector_nonce_field'], 'action')
      ) {

        if (isset($_POST['password'])) {

          $safe_new_password = sanitize_title($_POST['password']);

          if (strlen($safe_new_password) > 0) {
            update_option('product_protector_global_password', $safe_new_password);
            $info_message = __('Settings saved!', PRODUCT_PROTECTOR_TD);
          } else {
            $info_message = __('Password can\'t be empty!', PRODUCT_PROTECTOR_TD);
          }
        }

        if (isset($_POST['hide_prices'])) {
          update_option('product_protector_hide_price', 'yes');
        } else {
          update_option('product_protector_hide_price', 'no');
        }
      } else {

        $info_message = __('Can\'t verify nonce field!', PRODUCT_PROTECTOR_TD);
      }
    }

?>

    <div class="wrap">
      <h2><?php _e('Product protector settings', PRODUCT_PROTECTOR_TD) ?></h2>

      <?php if ($info_message) : ?>
        <div class="updated inline">
          <p><strong><?php echo $info_message ?></strong></p>
        </div>
      <?php endif; ?>

      <h4><?php _e('Protection priority (from highest to lowest):', PRODUCT_PROTECTOR_TD) ?></h4>

      <ul style="list-style-type: decimal; margin-left: 20px;">
        <li><?php _e('Individual Password (!!!)', PRODUCT_PROTECTOR_TD) ?></li>
        <li><?php _e('Category Password (!!)', PRODUCT_PROTECTOR_TD) ?></li>
        <li><?php _e('Global Password (!)', PRODUCT_PROTECTOR_TD) ?></li>
      </ul>
      <small><?php _e('!!! - highest priority, ! - lowest priority', PRODUCT_PROTECTOR_TD) ?></small>

      <form action="" method="post">
        <table class="form-table">
          <tbody>
            <tr valign="top">
              <th scope="row">
                <label><?php _e('Global password', PRODUCT_PROTECTOR_TD) ?></label>
              </th>
              <td>
                <input type="text" name="password" value="<?php echo get_option('product_protector_global_password'); ?>" placeholder="<?php _e('Password here', PRODUCT_PROTECTOR_TD) ?>" class="regular-text code">
              </td>
            </tr>
            <tr valign="top">
              <th scope="row">
                <label><?php _e('Hide price on protected products', PRODUCT_PROTECTOR_TD) ?></label>
              </th>
              <td>
                <input type="checkbox" name="hide_prices" <?php echo (get_option('product_protector_hide_price') === 'yes') ? 'checked' : ''; ?>>
              </td>
            </tr>
            <tr valign="top">
              <th scope="row">
                <?php wp_nonce_field('action', 'product_protector_nonce_field'); ?>
                <input type="submit" class="button button-primary button-large" value="<?php _e('Save settings', PRODUCT_PROTECTOR_TD) ?>" />
              </th>
            </tr>
          </tbody>
        </table>
      </form>

      <h3><?php _e('Password By Category', PRODUCT_PROTECTOR_TD) ?></h3>

      <table class="widefat">
        <thead>
          <tr>
            <th><?php _e('Category', PRODUCT_PROTECTOR_TD); ?></th>
            <th><?php _e('Password', PRODUCT_PROTECTOR_TD); ?></th>
          </tr>
        </thead>
        <tfoot>
          <tr>
            <th><?php _e('Category', PRODUCT_PROTECTOR_TD); ?></th>
            <th><?php _e('Password', PRODUCT_PROTECTOR_TD); ?></th>
          </tr>
        </tfoot>
        <tbody>

          <?php
          // get categories passwords

          $categories_info = unserialize(get_option('product_protector_categories_info'));

          // get all product categories
          $all_categories = get_categories('taxonomy=product_cat&hide_empty=0&hierarchical=1');
          foreach ($all_categories as $cat) {

            $category_id = $cat->term_id;
            $category_password = __('<i>no password selected</i>', PRODUCT_PROTECTOR_TD);

            if (
              is_array($categories_info)
              && array_key_exists($category_id, $categories_info)
              && $categories_info[$category_id]
            ) {

              $category_password = $categories_info[$category_id];
            }
          ?>
            <tr>
              <td class="post-title page-title column-title">
                <strong><a class="row-title" href="javascript://" title="Edit this item"><?php echo $cat->name ?></a></strong>
                <div class="row-actions">
                  <span class="edit">
                    <a href="javascript://" onclick="showHiddenBox(this);" title="<?php _e('Edit', 'woocommerce') ?>"><?php _e('Edit', 'woocommerce') ?></a> |
                  </span>
                  <span class="trash">
                    <a class="submitdelete" title="<?php _e('Delete password for this category', PRODUCT_PROTECTOR_TD) ?>" href="javascript://" onclick="deletePassword(<?php echo $category_id ?>);"><?php _e('Delete Password', PRODUCT_PROTECTOR_TD) ?></a>
                  </span>
                </div>
                <div class="hidden-field">
                  <input class="new-password" type="text" value="" />
                  <input name="save" type="button" onclick="updatePassword(this, <?php echo $category_id ?>);" class="button button-primary button-large" value="<?php _e('Update', PRODUCT_PROTECTOR_TD) ?>">
                </div>
              </td>
              <td><?php echo $category_password ?></td>
            </tr>
          <?php
          }
          ?>

        </tbody>
      </table>

      <p><?php _e('Plugin Version:', PRODUCT_PROTECTOR_TD) ?> <?php echo PRODUCT_PROTECTOR_VERSION_NUM ?></p>

      <p>
        <?php
        $url = 'https://codecanyon.net/item/product-password-protector-woocommerce/7481489?ref=alex91ckua';
        $new_version = sprintf(wp_kses(__('Check new version <a target="_blank" href="%s">here</a>', PRODUCT_PROTECTOR_TD), array('a' => array('href' => array(), 'target' => array()))), esc_url($url));
        echo $new_version;
        ?>
      </p>

    </div>
<?php

  }

  /**
   * Save category password through ajax
   */
  function product_protector_ajax_process_request() {

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {

      if (isset($_POST['categoryId']) && isset($_POST['newPassword'])) {

        $category_id = sanitize_title($_POST['categoryId']);
        $categories_info = unserialize(get_option('product_protector_categories_info'));
        $categories_info[$category_id] = sanitize_title($_POST['newPassword']);

        update_option('product_protector_categories_info', serialize($categories_info));
      }
    }
  }
}

new Product_Protector_Admin();
