<?php
if (!defined('ABSPATH')) {
    exit;
}

use Amazon\Affiliate\Admin\AutomationLogs;

// Get available filter options
$automation_types = AutomationLogs::get_automation_types();
$processes = AutomationLogs::get_processes();
?>

<?php include "common-header.php"; ?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <!-- Main Logs Card -->
            <div class="card shadow-sm ams-common-card-fix">
                <div class="card-header bg-light border-bottom">
                    <div class="d-flex align-items-center justify-content-between">
                        <h5 class="mb-0 fw-bold text-dark">
                            <i class="bi bi-file-text text-primary me-2"></i>
                            <?= esc_html__('Automation System Logs', 'ams-wc-amazon'); ?>
                        </h5>
                        <div class="d-flex gap-2">
                            <div class="dropdown">
                                <button class="btn btn-outline-secondary btn-sm dropdown-toggle" type="button" id="logsActions" data-bs-toggle="dropdown">
                                    <i class="bi bi-gear me-1"></i><?= esc_html__('Actions', 'ams-wc-amazon'); ?>
                                </button>
                                <ul class="dropdown-menu">
                                    <li><a class="dropdown-item" href="#" id="exportLogs">
                                        <i class="bi bi-download me-2"></i><?= esc_html__('Export Logs', 'ams-wc-amazon'); ?>
                                    </a></li>
                                    <li><hr class="dropdown-divider"></li>
                                    <li><a class="dropdown-item text-warning" href="#" id="clearOldLogs">
                                        <i class="bi bi-brush me-2"></i><?= esc_html__('Clear Old Logs (30+ days)', 'ams-wc-amazon'); ?>
                                    </a></li>
                                    <li><a class="dropdown-item text-danger" href="#" id="clearAllLogs">
                                        <i class="bi bi-trash me-2"></i><?= esc_html__('Clear All Logs', 'ams-wc-amazon'); ?>
                                    </a></li>
                                </ul>
                            </div>
                            <a href="<?= admin_url('admin.php?page=automation'); ?>" class="btn btn-outline-primary btn-sm">
                                <i class="bi bi-arrow-left me-1"></i><?= esc_html__('Back to Automation', 'ams-wc-amazon'); ?>
                            </a>
                        </div>
                    </div>
                </div>
                <div class="card-body">

            <!-- Information Panel -->
            <div class="alert alert-info border-0 mb-4" role="alert">
                <div class="d-flex">
                    <div class="flex-shrink-0">
                        <i class="bi bi-info-circle fa-lg text-info"></i>
                    </div>
                    <div class="flex-grow-1 ms-3">
                        <h6 class="alert-heading mb-2"><?= esc_html__('How Automation Logs Work', 'ams-wc-amazon'); ?></h6>
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="d-flex align-items-start">
                                    <i class="bi bi-database text-primary me-2 mt-1"></i>
                                    <div>
                                        <strong class="small"><?= esc_html__('Storage Location', 'ams-wc-amazon'); ?></strong>
                                        <p class="small text-muted mb-0"><?= esc_html__('Logs are stored in your WordPress database table', 'ams-wc-amazon'); ?> <code>wp_ams_automation_logs</code></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="d-flex align-items-start">
                                    <i class="bi bi-brush text-warning me-2 mt-1"></i>
                                    <div>
                                        <strong class="small"><?= esc_html__('Auto Cleanup', 'ams-wc-amazon'); ?></strong>
                                        <p class="small text-muted mb-0"><?= esc_html__('System automatically keeps only the latest 1,000 log entries to prevent database bloat', 'ams-wc-amazon'); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Filters -->
            <div class="card shadow-sm mb-4">
                <div class="card-body">
                    <form id="logsFilterForm" class="row g-3 align-items-end">
                        <div class="col-md-2">
                            <label class="form-label small text-muted"><?= esc_html__('Automation Type', 'ams-wc-amazon'); ?></label>
                            <select name="automation_type" class="form-select form-select-sm">
                                <option value="all"><?= esc_html__('All Types', 'ams-wc-amazon'); ?></option>
                                <?php foreach ($automation_types as $type): ?>
                                    <option value="<?= esc_attr(strtolower($type)); ?>">
                                        <?= esc_html(ucwords(str_replace('_', ' ', strtolower($type)))); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label small text-muted"><?= esc_html__('Log Level', 'ams-wc-amazon'); ?></label>
                            <select name="level" class="form-select form-select-sm">
                                <option value="all"><?= esc_html__('All Levels', 'ams-wc-amazon'); ?></option>
                                <option value="success"><?= esc_html__('Success', 'ams-wc-amazon'); ?></option>
                                <option value="error"><?= esc_html__('Error', 'ams-wc-amazon'); ?></option>
                                <option value="warning"><?= esc_html__('Warning', 'ams-wc-amazon'); ?></option>
                                <option value="info"><?= esc_html__('Info', 'ams-wc-amazon'); ?></option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label small text-muted"><?= esc_html__('Process', 'ams-wc-amazon'); ?></label>
                            <select name="process" class="form-select form-select-sm">
                                <option value="all"><?= esc_html__('All Processes', 'ams-wc-amazon'); ?></option>
                                <?php foreach ($processes as $process): ?>
                                    <option value="<?= esc_attr(strtolower($process)); ?>">
                                        <?= esc_html(ucwords(str_replace('_', ' ', strtolower($process)))); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label small text-muted"><?= esc_html__('Date Range', 'ams-wc-amazon'); ?></label>
                            <select name="date_range" class="form-select form-select-sm">
                                <option value="1"><?= esc_html__('Today', 'ams-wc-amazon'); ?></option>
                                <option value="7" selected><?= esc_html__('Last 7 Days', 'ams-wc-amazon'); ?></option>
                                <option value="30"><?= esc_html__('Last 30 Days', 'ams-wc-amazon'); ?></option>
                                <option value="all"><?= esc_html__('All Time', 'ams-wc-amazon'); ?></option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label small text-muted"><?= esc_html__('Per Page', 'ams-wc-amazon'); ?></label>
                            <select name="per_page" class="form-select form-select-sm">
                                <option value="25">25</option>
                                <option value="50" selected>50</option>
                                <option value="100">100</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label small text-muted"><?= esc_html__('Search', 'ams-wc-amazon'); ?></label>
                            <input type="text" name="search" class="form-control form-control-sm" placeholder="<?= esc_attr__('Search in messages...', 'ams-wc-amazon'); ?>">
                        </div>
                        <div class="col-md-1">
                            <button type="submit" class="btn btn-primary btn-sm w-100">
                                <i class="bi bi-search"></i>
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Logs Table -->
            <div class="card shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <div class="d-flex align-items-center justify-content-between">
                        <h6 class="mb-0 fw-semibold text-dark">
                            <i class="bi bi-list text-secondary me-2"></i>
                            <?= esc_html__('Log Entries', 'ams-wc-amazon'); ?>
                        </h6>
                        <div class="d-flex gap-2 align-items-center">
                            <small class="text-muted" id="logsCount"><?= esc_html__('Loading...', 'ams-wc-amazon'); ?></small>
                            <button type="button" id="refreshLogs" class="btn btn-outline-primary btn-sm">
                                <i class="bi bi-arrow-clockwise me-1"></i><?= esc_html__('Refresh', 'ams-wc-amazon'); ?>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body p-0">
                    <!-- Loading State -->
                    <div id="logsLoading" class="text-center py-5">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden"><?= esc_html__('Loading...', 'ams-wc-amazon'); ?></span>
                        </div>
                        <div class="mt-2 text-muted"><?= esc_html__('Loading logs...', 'ams-wc-amazon'); ?></div>
                    </div>

                    <!-- Logs Table -->
                    <div id="logsTableContainer" class="d-none">
                        <div class="table-responsive">
                            <table class="table table-sm table-hover mb-0 monospace" style="font-size: 13px;">
                                <thead class="table-light">
                                    <tr>
                                        <th style="width: 80px;"><?= esc_html__('Time', 'ams-wc-amazon'); ?></th>
                                        <th style="width: 70px;"><?= esc_html__('Level', 'ams-wc-amazon'); ?></th>
                                        <th style="width: 120px;"><?= esc_html__('Type', 'ams-wc-amazon'); ?></th>
                                        <th style="width: 100px;"><?= esc_html__('Process', 'ams-wc-amazon'); ?></th>
                                        <th><?= esc_html__('Message', 'ams-wc-amazon'); ?></th>
                                        <th style="width: 40px;"></th>
                                    </tr>
                                </thead>
                                <tbody id="logsTableBody">
                                    <!-- Logs will be loaded here -->
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- Empty State -->
                    <div id="logsEmpty" class="text-center py-5 d-none">
                        <i class="bi bi-file-text fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted"><?= esc_html__('No logs found', 'ams-wc-amazon'); ?></h5>
                        <p class="text-muted"><?= esc_html__('No automation logs match your current filters.', 'ams-wc-amazon'); ?></p>
                    </div>
                </div>

                <!-- Pagination -->
                <div class="card-footer bg-white border-top" id="logsPagination" style="display: none;">
                    <div class="d-flex justify-content-between align-items-center">
                        <div class="text-muted small" id="paginationInfo"></div>
                        <nav>
                            <ul class="pagination pagination-sm mb-0" id="paginationControls"></ul>
                        </nav>
                    </div>
                </div>
            </div>
            <!-- Close outer card-body and outer card -->
            </div>
        </div>
    </div>
</div>
</div>

<!-- Log Details Modal -->
<div class="modal fade" id="logDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><?= esc_html__('Log Entry Details', 'ams-wc-amazon'); ?></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div id="logDetailsContent" class="monospace" style="font-size: 13px; white-space: pre-wrap;"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-outline-secondary btn-sm" id="copyLogDetails">
                    <i class="bi bi-clipboard me-1"></i><?= esc_html__('Copy', 'ams-wc-amazon'); ?>
                </button>
                <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal">
                    <?= esc_html__('Close', 'ams-wc-amazon'); ?>
                </button>
            </div>
        </div>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    let currentPage = 1;
    
    // Initialize logs
    loadLogs();
    
    // Filter form submission
    $('#logsFilterForm').on('submit', function(e) {
        e.preventDefault();
        currentPage = 1;
        loadLogs();
    });
    

    
    // Refresh button
    $('#refreshLogs').on('click', function() {
        const $btn = $(this);
        const originalHtml = $btn.html();
        
        // Show loading state
        $btn.prop('disabled', true).html('<i class="bi bi-arrow-clockwise me-1"></i><?= esc_js(__('Refreshing...', 'ams-wc-amazon')); ?>');
        
        // Load logs and restore button
        loadLogs();
        
        // Restore button after a short delay
        setTimeout(function() {
            $btn.prop('disabled', false).html(originalHtml);
        }, 1000);
    });
    
    // Export logs
    $('#exportLogs').on('click', function(e) {
        e.preventDefault();
        window.open('<?= admin_url('admin-ajax.php?action=ams_export_automation_logs'); ?>', '_blank');
    });
    
    // Clear logs actions
    $('#clearOldLogs').on('click', function(e) {
        e.preventDefault();
        if (confirm('<?= esc_js(__('Clear logs older than 30 days?', 'ams-wc-amazon')); ?>')) {
            clearLogs('old');
        }
    });
    
    $('#clearAllLogs').on('click', function(e) {
        e.preventDefault();
        if (confirm('<?= esc_js(__('Clear ALL logs? This cannot be undone.', 'ams-wc-amazon')); ?>')) {
            clearLogs('all');
        }
    });
    
    // Load logs function
    function loadLogs() {
        $('#logsLoading').removeClass('d-none');
        $('#logsTableContainer, #logsEmpty').addClass('d-none');
        
        const formData = {
            action: 'ams_get_automation_logs',
            page: currentPage,
            ...Object.fromEntries(new FormData($('#logsFilterForm')[0]))
        };
        
        $.post(ajaxurl, formData, function(response) {
            if (response.success) {
                displayLogs(response.data);
            } else {
                console.error('Failed to load logs:', response.data);
            }
        }).always(function() {
            $('#logsLoading').addClass('d-none');
        });
    }
    
    // Display logs function
    function displayLogs(data) {
        const { logs, total, page, per_page, total_pages } = data;
        
        if (logs.length === 0) {
            $('#logsEmpty').removeClass('d-none');
            $('#logsPagination').hide();
            $('#logsCount').text('<?= esc_js(__('No logs found', 'ams-wc-amazon')); ?>');
            return;
        }
        
        // Update count
        $('#logsCount').text(`${total} <?= esc_js(__('entries found', 'ams-wc-amazon')); ?>`);
        
        // Build table rows
        let tableHtml = '';
        logs.forEach(log => {
            const levelBadge = getLevelBadge(log.level);
            const hasDetails = log.details && log.details.trim() !== '';
            const enhancedMessage = enhanceLogMessage(log.message, log.product_id);
            
            tableHtml += `
                <tr class="log-entry" data-log-id="${log.id}">
                    <td class="text-muted" title="${log.date_formatted}">${log.timestamp_formatted}</td>
                    <td>${levelBadge}</td>
                    <td class="text-muted">${log.type_display}</td>
                    <td class="text-muted">${log.process_display}</td>
                    <td>${enhancedMessage}</td>
                    <td class="text-center">
                        ${hasDetails ? '<i class="bi bi-info-circle text-info cursor-pointer" title="<?= esc_attr__('View Details', 'ams-wc-amazon'); ?>"></i>' : ''}
                    </td>
                </tr>
            `;
        });
        
        $('#logsTableBody').html(tableHtml);
        $('#logsTableContainer').removeClass('d-none');
        
        // Setup pagination
        if (total_pages > 1) {
            setupPagination(page, total_pages, total, per_page);
            $('#logsPagination').show();
        } else {
            $('#logsPagination').hide();
        }
        
        // Setup click handlers for details
        $('.log-entry .bi-info-circle').on('click', function() {
            const logId = $(this).closest('.log-entry').data('log-id');
            const log = logs.find(l => l.id == logId);
            showLogDetails(log);
        });
    }
    
    // Get level badge HTML
    function getLevelBadge(level) {
        const badges = {
            'SUCCESS': '<span class="badge bg-success">SUCCESS</span>',
            'ERROR': '<span class="badge bg-danger">ERROR</span>',
            'WARNING': '<span class="badge bg-warning text-dark">WARNING</span>',
            'INFO': '<span class="badge bg-info">INFO</span>'
        };
        return badges[level] || `<span class="badge bg-secondary">${level}</span>`;
    }
    
    // Enhance log message with product links
    function enhanceLogMessage(message, productId) {
        // First, strip any HTML tags from the message (for existing logs)
        let cleanMessage = message.replace(/<[^>]*>/g, '');
        let enhancedMessage = escapeHtml(cleanMessage);
        
        // Extract product ID from message if not provided
        if (!productId || productId <= 0) {
            // Try different patterns to extract product ID
            const patterns = [
                /product (\d+)/i,           // "product 12345"
                /product_id[:\s]+(\d+)/i,   // "product_id: 12345"
                /ID[:\s]+(\d+)/i,           // "ID: 12345"
                /\bproduct\s*#?(\d+)/i      // "product #12345" or "product12345"
            ];
            
            for (const pattern of patterns) {
                const match = message.match(pattern);
                if (match) {
                    productId = parseInt(match[1]);
                    break;
                }
            }
        }
        
        // If there's a product ID, add links for both success and failed products
        if (productId && productId > 0) {
            // Add product edit link
            const productEditLink = `<?= admin_url('post.php?post='); ?>${productId}&action=edit`;
            enhancedMessage += ` <a href="${productEditLink}" class="text-decoration-none" target="_blank" title="<?= esc_attr__('Edit Product', 'ams-wc-amazon'); ?>">
                <i class="bi bi-pencil-square text-primary"></i>
            </a>`;
            
            // Add placeholder for Amazon link (will be filled by AJAX)
            enhancedMessage += ` <span class="amazon-link-placeholder" data-product-id="${productId}">
                <i class="bi bi-hourglass-split text-muted" title="<?= esc_attr__('Loading Amazon link...', 'ams-wc-amazon'); ?>"></i>
            </span>`;
            
            // Try to get Amazon URL via AJAX and add Amazon link
            getAmazonUrlForProduct(productId).then(amazonUrl => {
                if (amazonUrl) {
                    const amazonLink = `<a href="${amazonUrl}" class="text-decoration-none" target="_blank" title="<?= esc_attr__('View on Amazon', 'ams-wc-amazon'); ?>">
                        <i class="bi bi-box-arrow-up-right text-warning"></i>
                    </a>`;
                    
                    // Replace the placeholder with the actual link
                    $(`.amazon-link-placeholder[data-product-id="${productId}"]`).html(amazonLink);
                } else {
                    // Remove placeholder if no Amazon URL found
                    $(`.amazon-link-placeholder[data-product-id="${productId}"]`).remove();
                }
            }).catch(() => {
                // Remove placeholder on error
                $(`.amazon-link-placeholder[data-product-id="${productId}"]`).remove();
            });
        }
        
        return enhancedMessage;
    }
    
    // Get Amazon URL for a product
    async function getAmazonUrlForProduct(productId) {
        try {
            const response = await $.post(ajaxurl, {
                action: 'ams_get_product_amazon_url',
                product_id: productId,
                nonce: '<?= wp_create_nonce('ams_get_product_url'); ?>'
            });
            
            if (response.success && response.data.amazon_url) {
                return response.data.amazon_url;
            }
        } catch (error) {
            console.log('Could not fetch Amazon URL for product:', productId);
        }
        return null;
    }
    
    // Setup pagination
    function setupPagination(page, totalPages, total, perPage) {
        const start = (page - 1) * perPage + 1;
        const end = Math.min(page * perPage, total);
        
        $('#paginationInfo').text(`<?= esc_js(__('Showing', 'ams-wc-amazon')); ?> ${start}-${end} <?= esc_js(__('of', 'ams-wc-amazon')); ?> ${total}`);
        
        let paginationHtml = '';
        
        // Previous button
        if (page > 1) {
            paginationHtml += `<li class="page-item"><a class="page-link" href="#" data-page="${page - 1}">‹</a></li>`;
        }
        
        // Page numbers
        const startPage = Math.max(1, page - 2);
        const endPage = Math.min(totalPages, page + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const active = i === page ? 'active' : '';
            paginationHtml += `<li class="page-item ${active}"><a class="page-link" href="#" data-page="${i}">${i}</a></li>`;
        }
        
        // Next button
        if (page < totalPages) {
            paginationHtml += `<li class="page-item"><a class="page-link" href="#" data-page="${page + 1}">›</a></li>`;
        }
        
        $('#paginationControls').html(paginationHtml);
        
        // Pagination click handlers
        $('#paginationControls a').on('click', function(e) {
            e.preventDefault();
            currentPage = parseInt($(this).data('page'));
            loadLogs();
        });
    }
    
    // Show log details modal
    function showLogDetails(log) {
        let details = `Timestamp: ${log.timestamp}\n`;
        details += `Level: ${log.level}\n`;
        details += `Type: ${log.type_display}\n`;
        details += `Process: ${log.process_display}\n`;
        details += `Message: ${log.message}\n`;
        
        if (log.details) {
            details += `\nDetails:\n${log.details}`;
        }
        
        if (log.product_id) {
            details += `\nProduct ID: ${log.product_id}`;
        }
        
        $('#logDetailsContent').text(details);
        $('#logDetailsModal').modal('show');
    }
    
    // Copy log details
    $('#copyLogDetails').on('click', function() {
        const text = $('#logDetailsContent').text();
        navigator.clipboard.writeText(text).then(function() {
            $(this).html('<i class="bi bi-check me-1"></i><?= esc_js(__('Copied!', 'ams-wc-amazon')); ?>');
            setTimeout(() => {
                $(this).html('<i class="bi bi-clipboard me-1"></i><?= esc_js(__('Copy', 'ams-wc-amazon')); ?>');
            }, 2000);
        }.bind(this));
    });
    
    // Clear logs function
    function clearLogs(type) {
        $.post(ajaxurl, {
            action: 'ams_clear_automation_logs',
            clear_type: type,
            nonce: '<?= wp_create_nonce('ams_automation_nonce'); ?>'
        }, function(response) {
            if (response.success) {
                loadLogs();
                alert('<?= esc_js(__('Logs cleared successfully', 'ams-wc-amazon')); ?>');
            } else {
                alert('<?= esc_js(__('Failed to clear logs', 'ams-wc-amazon')); ?>: ' + response.data);
            }
        });
    }
    
    // Utility function to escape HTML
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    

});
</script>

<?php include "common-footer.php"; ?>