<?php
namespace Amazon\Affiliate\Admin;
/**
 * Admin setting page and setting handler
 *
 * @package Amazon\Affiliate\Admin
 */
class Setting {

    public function setting_page() {
        $template = __DIR__ . '/views/settings.php';
        if ( file_exists( $template ) ) {
            require_once $template;
        }
    }

    public function general_amazon_setting_handler() {
        // Check if nonce is valid, if not, redirect with error message
        if (!wp_verify_nonce($_POST['_wpnonce'] ?? '', 'general_amazon_setting_nonce')) {
            wp_redirect(admin_url('admin.php?page=wc-product-setting-page&action=affiliates&tab=pills-az-settings-tab&error=nonce_expired'));
            exit;
        }
        $access_key_id     = isset($_POST['access_key_id']) ? sanitize_text_field( $_POST['access_key_id'] ) : '';
        $secret_access_key = isset($_POST['secret_access_key']) ? sanitize_text_field( $_POST['secret_access_key'] ) : '';
        $associate_tag     = isset($_POST['ams_associate_tag']) ? sanitize_text_field( $_POST['ams_associate_tag'] ) : '';
        $country           = isset($_POST['ams_amazon_country']) ? sanitize_text_field( $_POST['ams_amazon_country'] ) : '';

        $woocommerce_currency  = isset($_POST['woocommerce_currency']) ? sanitize_text_field( $_POST['woocommerce_currency'] ) : get_woocommerce_currency();
        update_option( 'woocommerce_currency', $woocommerce_currency );
        
        update_option( 'ams_access_key_id', $access_key_id );
        update_option( 'ams_secret_access_key', $secret_access_key );
        update_option( 'ams_associate_tag', $associate_tag );
        update_option( 'ams_amazon_country', $country );

        $action_tab = isset($_POST['action_tab']) ? $_POST['action_tab'] : 'pills-az-settings-tab';
        wp_redirect( 'admin.php?page=wc-product-setting-page&action=affiliates&tab='.$action_tab.'&settings=saved' );
    }

    public function product_cat_country_setting() {
        // Sanitize and save the posted data
        $country = isset($_POST['ams_amazon_country']) ? sanitize_text_field($_POST['ams_amazon_country']) : '';
        update_option('ams_amazon_country', $country);
        
        $ams_default_category = isset($_POST['ams_default_category']) ? sanitize_text_field($_POST['ams_default_category']) : '';
        update_option('ams_default_category', $ams_default_category);
        
        $woocommerce_currency = isset($_POST['woocommerce_currency']) ? sanitize_text_field($_POST['woocommerce_currency']) : get_woocommerce_currency();
        update_option('woocommerce_currency', $woocommerce_currency);
        
        $page_redirect = isset($_POST['page_redirect']) ? sanitize_text_field($_POST['page_redirect']) : '';
        
        // Check if the request came from a page with a tab
        $referer = wp_get_referer();
        $parsed_referer = parse_url($referer);
        parse_str($parsed_referer['query'] ?? '', $query_params);
        
        $redirect_url = admin_url('admin.php?page=' . $page_redirect);
        
        // If there was a tab parameter in the referer URL, include it in the redirect
        if (isset($query_params['tab'])) {
            $redirect_url .= '&tab=' . sanitize_text_field($query_params['tab']);
        }
        
        wp_safe_redirect(esc_url_raw($redirect_url));
        exit();
    }

    public function general_ams_product_cron(){
    	update_option( 'product_name_cron', $_POST['product_name_cron'] );
        update_option( 'product_price_cron', $_POST['product_price_cron'] );
        update_option( 'product_sku_cron', $_POST['product_sku_cron'] );
        update_option( 'product_variants_cron', $_POST['product_variants_cron'] );
        update_option( 'product_variant_image_cron', $_POST['product_variant_image_cron'] );
        update_option( 'product_variant_description_cron', $_POST['product_variant_description_cron'] );
        update_option( 'product_out_of_stock_cron', $_POST['product_out_of_stock_cron'] );
        update_option( 'product_description_cron', $_POST['product_description_cron'] );
        update_option( 'product_image_cron', $_POST['product_image_cron'] );
        update_option( 'product_category_cron', $_POST['product_category_cron'] );
        // Check if exists in from 
        if( !isset($_POST['product_tags_cron']) ) {
        	$_POST['product_tags_cron'] = '0';
        }
        if( !isset($_POST['product_review_cron']) ) {
            $_POST['product_review_cron'] = '0';   
        }
        update_option( 'product_tags_cron', $_POST['product_tags_cron'] );
        update_option( 'product_review_cron', $_POST['product_review_cron'] );
        wp_redirect( 'admin.php?page=wc-product-setting-page&action=product_avail&tab='.$_POST['action_tab'] );
        die();
        $product_name_cron =  get_option('product_name_cron');
        $product_tags_cron =  get_option('product_tags_cron');
        $product_variants_cron =  get_option('product_variants_cron');
        $product_sku_cron = get_option('product_sku_cron');
        $product_price_cron =  get_option('product_price_cron');
        $product_category_cron =  get_option('product_category_cron');
        $product_out_of_stock_cron = get_option('product_out_of_stock_cron');
        $product_description_cron = get_option('product_description_cron');
        if ( ! class_exists( 'simple_html_dom' ) ) {
            require_once __DIR__ . '/lib/simplehtmldom/simple_html_dom.php';
        }
        $asins = ams_get_all_products_info();
        $id_asin = array_combine( $asins['id'], $asins['asin'] );
        foreach ( $id_asin as $id => $asin ) {
            $url = sprintf( 'https://www.amazon.%s/dp/%s', get_option( 'ams_amazon_country' ), $asin );
            sleep( rand( 1, 5 ) );
            $product_import = new \Amazon\Affiliate\Admin\ProductsSearchWithoutApi();
            $user_agent = $product_import->user_agent();
            $options = array(
                'http' => array(
                    'method' => "GET",
                    'header' => "Accept-language: en\r\n" .
                        $user_agent
                )
            );
            $context = stream_context_create( $options );
            $response_body = file_get_contents( esc_url_raw( $url ), false, $context );
            $html = new \simple_html_dom();
            $html->load( $response_body );
            $product_status = 'instock';
            if ( $html->find( '#outOfStock .a-color-price', 0 ) ) {
                $product_status = 'outofstock';
            }
            $newBuyBoxPrice =  $html->find('#newBuyBoxPrice');
            $price = $html->find('#price');
            $priceblock_ourprice = $html->find('#priceblock_ourprice');
            $priceblock_dealprice = $html->find('#priceblock_dealprice');
            $priceblock_saleprice = $html->find('#priceblock_saleprice');
            if ( isset( $newBuyBoxPrice[0] )){
                $amount = $newBuyBoxPrice[0]->innertext;
            } 
            if ( isset( $price[0] )){
                $amount = $price[0]->innertext;
            } 
            if ( isset( $priceblock_ourprice[0] ) ) {
               $amount = $priceblock_ourprice[0]->innertext;
            } 
            if ( isset( $priceblock_saleprice[0] ) ) {
               $amount = $priceblock_saleprice[0]->innertext;
            }
            if ( isset( $priceblock_dealprice[0] ) ) {
               $amount = $priceblock_dealprice[0]->innertext;
            }
            $amount = preg_replace("/[^0-9 .]/", "", $amount );
        $amount = str_replace( ',', '.', $amount );
        $amount = strtok( $amount, ' ' );
        if($product_sku_cron){
            update_post_meta($id, '_sku', $asin );
        }
        if($product_out_of_stock_cron){
            update_post_meta( $id, '_stock_status', $product_status );
        }
        if($product_price_cron){
            update_post_meta( $id, '_price', $amount );
            update_post_meta( $id, '_regular_price', $amount );
            update_post_meta( $id, '_sale_price', $amount);
        }

        if($product_variants_cron){
            global $wpdb;           
            $product = wc_get_product($id);
            $children_ids = $product->get_children();
            $meta_prefix = $wpdb->prefix . 'options';    
            $query =  "DELETE FROM $meta_prefix
                WHERE (option_name LIKE '_transient_wc_var_prices_".$id."'
                OR option_name LIKE '_transient_timeout_wc_var_prices_".$id."')";
            $wpdb->query( $query);
            if(isset($children_ids) && !empty($children_ids)){
                foreach ($children_ids as $key => $children_id) {
                    update_post_meta( $children_id, '_price', sanitize_text_field($amount) );
                    update_post_meta( $children_id, '_sale_price', $amount);
                }
            }
        }
        wc_delete_product_transients($id);
        }
    }


    public function general_setting() {
        // Check if nonce is valid, if not, redirect with error message
        if (!wp_verify_nonce($_POST['general_setting_nonce'] ?? '', 'general_setting_nonce')) {
            wp_redirect(admin_url('admin.php?page=wc-product-setting-page&action=affiliates&tab=pills-general-tab&error=nonce_expired'));
            exit;
        }
        
        // Existing variables
        $product_per_page            = sanitize_text_field($_POST['product_per_page'] ?? '10');
        $buy_now_label               = sanitize_text_field($_POST['buy_now_label'] ?? 'Buy Now');
        $buy_action_btn              = sanitize_text_field($_POST['buy_action_btn'] ?? 'redirect');
        $enable_no_follow_link       = isset($_POST['enable_no_follow_link']) && $_POST['enable_no_follow_link'] === 'nofollow' ? 'nofollow' : 'follow';
        $remove_unavailable_products = isset($_POST['remove_unavailable_products']) && $_POST['remove_unavailable_products'] === 'Yes' ? 'Yes' : 'No';
        $fast_product_importer       = isset($_POST['fast_product_importer']) && $_POST['fast_product_importer'] === 'Yes' ? 'Yes' : 'No';
        $ams_product_import_status   = sanitize_text_field($_POST['ams_product_import_status'] ?? 'publish');
        $ams_default_category        = sanitize_text_field($_POST['ams_default_category'] ?? '_auto_import_amazon');
        $checkout_mass_redirected    = sanitize_text_field($_POST['checkout_mass_redirected'] ?? 'no');
        $checkout_redirected_seconds = sanitize_text_field($_POST['checkout_redirected_seconds'] ?? '5');
        $remote_amazon_images        = sanitize_text_field($_POST['remote_amazon_images'] ?? 'no');
        $product_thumbnail_size      = sanitize_text_field($_POST['product_thumbnail_size'] ?? 'medium');
        
        // ========================================
        // THEME AUTO-DETECTION SETTINGS
        // ========================================
        $ams_enable_auto_detection = isset($_POST['ams_enable_auto_detection']) ? '1' : '0';
        $percentage_profit           = sanitize_text_field($_POST['percentage_profit']);

        $variation_image_limit = isset($_POST['variation_image_limit']) ? intval($_POST['variation_image_limit']) : 5;
        $variation_image_limit = min(max($variation_image_limit, 1), 10);

        $category_min_depth          = isset($_POST['category_min_depth']) ? intval($_POST['category_min_depth']) : 1;
        $category_max_depth          = isset($_POST['category_max_depth']) ? intval($_POST['category_max_depth']) : 10;
        
        // ScraperAPI variables
        $scraper_api_key             = sanitize_text_field($_POST['scraper_api_key'] ?? '');
        $scraper_api_is_active       = isset($_POST['scraper_api_is_active']) ? '1' : '0';
        $scraper_api_on_update       = isset($_POST['scraper_api_on_update']) ? '1' : '0';
        
        // ScrapingAnt variables
        $scrapingant_api_key         = sanitize_text_field($_POST['scrapingant_api_key'] ?? '');
        $scrapingant_is_active       = isset($_POST['scrapingant_is_active']) ? '1' : '0';
        $scrapingant_on_update       = isset($_POST['scrapingant_on_update']) ? '1' : '0';
        
        // Decodo variables
        $decodo_auth_token           = sanitize_text_field($_POST['decodo_auth_token'] ?? '');
        $decodo_is_active            = isset($_POST['decodo_is_active']) ? '1' : '0';
        $decodo_on_update            = isset($_POST['decodo_on_update']) ? '1' : '0';
        
        // Scraping Service Priority Settings
        $scraperapi_priority         = intval($_POST['scraperapi_priority'] ?? 1);
        $scrapingant_priority        = intval($_POST['scrapingant_priority'] ?? 2);
        $decodo_priority             = intval($_POST['decodo_priority'] ?? 3);
        
        $variation_image_meta_key = sanitize_text_field($_POST['variation_image_meta_key'] ?? '_product_image_gallery');
        $variation_ajax_action = sanitize_text_field($_POST['variation_ajax_action'] ?? '');
        $variation_limit = isset($_POST['variation_limit']) ? intval($_POST['variation_limit']) : 5;
        $ams_image_fit = sanitize_text_field($_POST['ams_image_fit'] ?? 'cover');
        
        // Theme Data Format Settings (optimized defaults for remote images)
        $theme_image_format = sanitize_text_field($_POST['theme_image_format'] ?? 'url');
        $theme_thumb_format = sanitize_text_field($_POST['theme_thumb_format'] ?? 'url');
        $theme_full_format = sanitize_text_field($_POST['theme_full_format'] ?? 'url');
        $theme_thumb_url_format = sanitize_text_field($_POST['theme_thumb_url_format'] ?? 'url');
        

        // Get the Custom Theme Hook Settings
        $use_custom_button = isset($_POST['use_custom_button']) ? '1' : '0';

        // Get the enable/disable option for page speed test
        $enable_page_speed_test = isset($_POST['enable_page_speed_test']) ? '1' : '0';

        // Get the selected style
        $page_speed_test_style = sanitize_text_field($_POST['page_speed_test_style'] ?? 'style1');

        // Get Page Speed Test Settings
        $speed_test_provider = sanitize_text_field($_POST['speed_test_provider'] ?? 'google_pagespeed');
        $test_frequency = sanitize_text_field($_POST['test_frequency'] ?? 'weekly');
        $show_performance_score = isset($_POST['show_performance_score']) ? '1' : '0';
        $show_load_time = isset($_POST['show_load_time']) ? '1' : '0';
        $show_page_size = isset($_POST['show_page_size']) ? '1' : '0';
        $show_requests_count = isset($_POST['show_requests_count']) ? '1' : '0';
        $display_position = sanitize_text_field($_POST['display_position'] ?? 'footer_left');
        $cache_duration = intval($_POST['cache_duration'] ?? 24);
        $show_on_mobile = sanitize_text_field($_POST['show_on_mobile'] ?? '1');
        $speed_test_debug = sanitize_text_field($_POST['speed_test_debug'] ?? '0');

        // Get API Keys for Speed Test Providers
        $google_pagespeed_api_key = sanitize_text_field($_POST['google_pagespeed_api_key'] ?? '');
        $gtmetrix_api_key = sanitize_text_field($_POST['gtmetrix_api_key'] ?? '');
        $pingdom_api_key = sanitize_text_field($_POST['pingdom_api_key'] ?? '');
        $webpagetest_api_key = sanitize_text_field($_POST['webpagetest_api_key'] ?? '');

        $enable_last_updated_date = isset($_POST['enable_last_updated_date']) ? '1' : '0';
        $enable_custom_message = isset($_POST['enable_custom_message']) ? '1' : '0';

        $last_updated_custom_message = isset($_POST['last_updated_custom_message']) 
            ? sanitize_textarea_field($_POST['last_updated_custom_message']) 
            : 'Important Notice: Product details may change. Please check regularly for updates.';

        // Get the selected style for the Last Updated Notice
        $last_updated_notice_style = isset($_POST['last_updated_notice_style']) 
            ? sanitize_text_field($_POST['last_updated_notice_style']) 
            : 'style1';
            
        // New enhanced options
        $date_format_type = isset($_POST['date_format_type']) 
            ? sanitize_text_field($_POST['date_format_type']) 
            : 'standard';
        $custom_date_format = isset($_POST['custom_date_format']) 
            ? sanitize_text_field($_POST['custom_date_format']) 
            : 'F j, Y';
        $show_only_old_products = isset($_POST['show_only_old_products']) ? '1' : '0';
        $old_product_days = isset($_POST['old_product_days']) 
            ? intval($_POST['old_product_days']) 
            : 30;
        $enable_dismissible = isset($_POST['enable_dismissible']) ? '1' : '0';
        $enable_notice_analytics = isset($_POST['enable_notice_analytics']) ? '1' : '0';

        $enable_daily_cron = isset($_POST['enable_daily_cron']) ? '1' : '0';
        $sidebar_autoclose = isset($_POST['sidebar_autoclose']) ? '1' : '0';

        // Enable/Disable and message options
        $enable_product_last_updated = isset($_POST['enable_product_last_updated']) ? '1' : '0';
        $product_last_updated_message = sanitize_textarea_field($_POST['product_last_updated_message']);
        $enable_global_last_updated = isset($_POST['enable_global_last_updated']) ? '1' : '0';
        $global_last_updated_message = sanitize_textarea_field($_POST['global_last_updated_message']);
        $enable_custom_notification = isset($_POST['enable_custom_notification']) ? '1' : '0';
        $custom_notification_message = sanitize_textarea_field($_POST['custom_notification_message']);
        $message_alignment = sanitize_text_field($_POST['message_alignment']);
        $enable_legal_notice = isset($_POST['enable_legal_notice']) ? '1' : '0';
        $legal_notice_text = sanitize_textarea_field($_POST['legal_notice_text'] ?? 'Affiliate Products | Advertisement | Sponsored');

        $product_category_cron = isset($_POST['product_category_cron']) ? '1' : '0';

        $enable_clean_completed_actions = isset($_POST['enable_clean_completed_actions']) ? '1' : '0';
        $enable_clean_action_logs = isset($_POST['enable_clean_action_logs']) ? '1' : '0';

        // Update existing options
        update_option('ams_product_per_page', $product_per_page);
        update_option('ams_buy_now_label', $buy_now_label);
        update_option('ams_buy_action_btn', $buy_action_btn);
        update_option('ams_enable_no_follow_link', $enable_no_follow_link);
        update_option('ams_remove_unavailable_products', $remove_unavailable_products);
        update_option('ams_fast_product_importer', $fast_product_importer);
        update_option('ams_product_import_status', $ams_product_import_status);
        update_option('ams_default_category', $ams_default_category);
        update_option('product_category_cron', $product_category_cron);
        update_option('ams_checkout_mass_redirected', $checkout_mass_redirected);
        update_option('ams_checkout_redirected_seconds', $checkout_redirected_seconds);
        update_option('ams_remote_amazon_images', $remote_amazon_images);
        update_option('ams_product_thumbnail_size', $product_thumbnail_size);
        // Save Theme Auto-Detection Setting
        update_option('ams_enable_auto_detection', $ams_enable_auto_detection);
        update_option('ams_percentage_profit', $percentage_profit);
        update_option('ams_sidebar_autoclose', $sidebar_autoclose);



        update_option('ams_variation_image_limit', $variation_image_limit);
        update_option('ams_variation_limit', $variation_limit);
        update_option('variation_image_meta_key', $variation_image_meta_key);
        update_option('variation_ajax_action', $variation_ajax_action);
        update_option('ams_image_fit', $ams_image_fit);
        
        // Save Theme Data Format Settings
        update_option('theme_image_format', $theme_image_format);
        update_option('theme_thumb_format', $theme_thumb_format);
        update_option('theme_full_format', $theme_full_format);
        update_option('theme_thumb_url_format', $theme_thumb_url_format);
        
        // Mark that user has manually configured theme settings (prevent auto-override)
        update_option('ams_manual_theme_settings', '1');
        
        update_option('ams_category_min_depth', $category_min_depth);
        update_option('ams_category_max_depth', $category_max_depth);
        

        // Save Enable/Disable and message options
        update_option('enable_product_last_updated', $enable_product_last_updated);
        update_option('product_last_updated_message', $product_last_updated_message);
        update_option('enable_global_last_updated', $enable_global_last_updated);
        update_option('global_last_updated_message', $global_last_updated_message);
        update_option('enable_custom_notification', $enable_custom_notification);
        update_option('custom_notification_message', $custom_notification_message);
        update_option('message_alignment', $message_alignment);
        update_option('enable_legal_notice', $enable_legal_notice);
        update_option('legal_notice_text', $legal_notice_text);
    
        // Update the option in the database
        update_option('enable_page_speed_test', $enable_page_speed_test);
        // Update the style in the database
        update_option('page_speed_test_style', $page_speed_test_style);
        
        // Update Page Speed Test Settings
        update_option('speed_test_provider', $speed_test_provider);
        update_option('test_frequency', $test_frequency);
        update_option('show_performance_score', $show_performance_score);
        update_option('show_load_time', $show_load_time);
        update_option('show_page_size', $show_page_size);
        update_option('show_requests_count', $show_requests_count);
        update_option('display_position', $display_position);
        update_option('cache_duration', $cache_duration);
        update_option('show_on_mobile', $show_on_mobile);
        update_option('speed_test_debug', $speed_test_debug);

        // Update API Keys for Speed Test Providers
        update_option('google_pagespeed_api_key', $google_pagespeed_api_key);
        update_option('gtmetrix_api_key', $gtmetrix_api_key);
        update_option('pingdom_api_key', $pingdom_api_key);
        update_option('webpagetest_api_key', $webpagetest_api_key);

        // Update Custom Theme Hook Settings
        update_option('ams_use_custom_button', $use_custom_button);
        
        // Handle theme hook - manual input takes priority over auto-detection
        $theme_hook = isset($_POST['theme_hook']) ? sanitize_text_field($_POST['theme_hook']) : '';
        
        if ($use_custom_button === '1') {
            if (!empty($theme_hook)) {
                // Use manual input if provided
                update_option('ams_theme_hook', $theme_hook);
            } else {
                // Auto-detect if no manual input
                $current_theme = wp_get_theme();
                $theme_name = $current_theme->get('Name');
                $auto_detected_hook = $this->auto_detect_theme_hook($theme_name);
                update_option('ams_theme_hook', $auto_detected_hook);
            }
        } else {
            update_option('ams_theme_hook', '');
        }

        // Update custom message on product page
        update_option('enable_last_updated_date', $enable_last_updated_date);
        update_option('enable_custom_message', $enable_custom_message);
        update_option('last_updated_custom_message', $last_updated_custom_message);
        update_option('last_updated_notice_style', $last_updated_notice_style);
        
        // Update new enhanced options
        update_option('date_format_type', $date_format_type);
        update_option('custom_date_format', $custom_date_format);
        update_option('show_only_old_products', $show_only_old_products);
        update_option('old_product_days', $old_product_days);
        update_option('enable_dismissible', $enable_dismissible);
        update_option('enable_notice_analytics', $enable_notice_analytics);

        
        // Update ScraperAPI options
        update_option('ams_scraper_api_key', $scraper_api_key);
        update_option('ams_scraper_api_is_active', $scraper_api_is_active);
        update_option('ams_scraper_api_on_update', $scraper_api_on_update);
        
        // Clear ScraperAPI cache when key is updated
        delete_transient('ams_scraper_api_test_result');
        delete_transient('ams_scraper_api_key_hash');
        delete_transient('ams_scraper_api_last_test');
        
        // Update ScrapingAnt options
        update_option('ams_scrapingant_api_key', $scrapingant_api_key);
        update_option('ams_scrapingant_is_active', $scrapingant_is_active);
        update_option('ams_scrapingant_on_update', $scrapingant_on_update);
        
        // Clear ScrapingAnt cache when key is updated
        delete_transient('ams_scrapingant_api_test_result');
        delete_transient('ams_scrapingant_api_key_hash');
        delete_transient('ams_scrapingant_api_last_test');
        
        // Update Decodo options
        update_option('ams_decodo_auth_token', $decodo_auth_token);
        update_option('ams_decodo_is_active', $decodo_is_active);
        update_option('ams_decodo_on_update', $decodo_on_update);
        
        // Update scraping service priorities
        update_option('ams_scraperapi_priority', $scraperapi_priority);
        update_option('ams_scrapingant_priority', $scrapingant_priority);
        update_option('ams_decodo_priority', $decodo_priority);
        
        // Clear Decodo cache when credentials are updated
        delete_transient('ams_decodo_api_test_result');
        delete_transient('ams_decodo_username_hash');
        delete_transient('ams_decodo_api_last_test');

        update_option('enable_clean_completed_actions', $enable_clean_completed_actions);
        update_option('enable_clean_action_logs', $enable_clean_action_logs);


        // Save the Enable/Disable Daily Cron setting
        update_option('enable_daily_cron', $enable_daily_cron);

        // Manage cron job scheduling
        if ($enable_daily_cron === '1') {
            if (!wp_next_scheduled('ams_daily_cron_event')) {
                wp_schedule_event(time(), 'ams_every_day', 'ams_daily_cron_event');
            }
        } else {
            if (wp_next_scheduled('ams_daily_cron_event')) {
                wp_clear_scheduled_hook('ams_daily_cron_event');
            }
        }
        
        wp_redirect('admin.php?page=wc-product-setting-page&tab='.$_POST['action_tab'].'&settings=saved');
        exit;
    }


    public function get_option( $name ) {
        $option =  get_option( $name );
        return  $option;
    }

    function get_wc_terms() {
        $categories = get_terms( array(
            // 'parent' => 0,
            'hide_empty' => false,
        ) );
        $cat = array();
        foreach ( $categories as $row ) {
            if ( $row->slug == "uncategorized" ) continue;
            if ( 'product_cat' === $row->taxonomy ) {
                $cat[] = array(
                    'term_id'  => $row->term_id,
                    'name'  => $row->name,
                );
            }
        }
        return array_reverse( $cat );
    }

    public function validate_proxy_services() {
        // Verify nonce for security
        if (!wp_verify_nonce($_POST['_wpnonce'] ?? '', 'wca_search_product')) {
            wp_send_json_error(['message' => esc_html__('Security check failed', 'ams-wc-amazon')]);
        }

        // Check if we should use cached results (valid for 1 hour)
        $cache_key = 'ams_proxy_status_cache';
        $cache_duration = HOUR_IN_SECONDS; // 1 hour cache
        $cached_result = get_transient($cache_key);
        
        // Force refresh if explicitly requested
        $force_refresh = isset($_POST['force_refresh']) && $_POST['force_refresh'] === 'true';
        
        if ($cached_result && !$force_refresh) {
            wp_send_json($cached_result);
            return;
        }

        // Get all proxy service configurations
        $services_to_test = [
            'decodo' => [
                'active' => get_option('ams_decodo_is_active', '0'),
                'key' => get_option('ams_decodo_auth_token'),
                'test_function' => 'ams_test_decodo_api_status',
                'name' => 'Decodo'
            ],
            'scraperapi' => [
                'active' => get_option('ams_scraper_api_is_active', '0'),
                'key' => get_option('ams_scraper_api_key'),
                'test_function' => 'ams_test_scraper_api_status',
                'name' => 'ScraperAPI'
            ],
            'scrapingant' => [
                'active' => get_option('ams_scrapingant_is_active', '0'),
                'key' => get_option('ams_scrapingant_api_key'),
                'test_function' => 'ams_test_scrapingant_api_status',
                'name' => 'ScrapingAnt'
            ]
        ];
        
        // Only test services that are active and have credentials
        $active_services = array_filter($services_to_test, function($service) {
            return $service['active'] === '1' && !empty($service['key']);
        });
        
        // If no active services, return error
        if (empty($active_services)) {
            $result = [
                'success' => false,
                'data' => [
                    'message' => esc_html__('No proxy services configured', 'ams-wc-amazon'),
                    'type' => 'proxy_missing'
                ]
            ];
            
            set_transient($cache_key, $result, $cache_duration);
            wp_send_json($result);
            return;
        }
        
        // Test only the first active service (quick check)
        $first_service = reset($active_services);
        $working_services = [];
        
        if (function_exists($first_service['test_function'])) {
            $test_result = call_user_func($first_service['test_function']);
            if ($test_result['status']) {
                // At least one service works - that's enough
                $working_services[] = $first_service['name'];
                
                $result = [
                    'success' => true,
                    'data' => [
                        'message' => sprintf(
                            esc_html__('%s available', 'ams-wc-amazon'),
                            $first_service['name']
                        ),
                        'working_services' => $working_services
                    ]
                ];
                
                set_transient($cache_key, $result, $cache_duration);
                wp_send_json($result);
                return;
            }
        }
        
        // If first service failed, return error
        $result = [
            'success' => false,
            'data' => [
                'message' => esc_html__('Proxy services not responding', 'ams-wc-amazon'),
                'type' => 'proxy_not_working'
            ]
        ];
        
        set_transient($cache_key, $result, $cache_duration);
        wp_send_json($result);
    }

    public function ams_run_speed_test() {
        // Verify nonce for security
        if (!wp_verify_nonce($_POST['nonce'], 'ams_speed_test_nonce')) {
            wp_send_json_error(['message' => esc_html__('Security check failed', 'ams-wc-amazon')]);
        }

        $provider = sanitize_text_field($_POST['provider'] ?? get_option('speed_test_provider', 'google_pagespeed'));
        $test_url = home_url();

        // Provider selection logged for debugging if needed

        // Run actual speed test based on provider
        $results = $this->run_actual_speed_test($provider, $test_url);

        // Cache the results
        set_transient('ams_speed_test_results', $results, get_option('cache_duration', 24) * HOUR_IN_SECONDS);

        wp_send_json_success([
            'message' => sprintf(esc_html__('Speed test completed using %s!', 'ams-wc-amazon'), $this->get_provider_name($provider)),
            'results' => $results
        ]);
    }

    private function run_actual_speed_test($provider, $test_url) {
        switch ($provider) {
            case 'google_pagespeed':
                return $this->test_google_pagespeed($test_url);
            case 'gtmetrix':
                return $this->test_gtmetrix($test_url);
            case 'pingdom':
                return $this->test_pingdom($test_url);
            case 'webpagetest':
                return $this->test_webpagetest($test_url);
            default:
                return $this->test_fallback($provider, $test_url);
        }
    }

    private function test_google_pagespeed($test_url) {
        // Google PageSpeed Insights API
        $api_key = get_option('google_pagespeed_api_key', ''); // You'll need to add this setting
        
        if (empty($api_key)) {
            // API key not configured, using fallback
            return $this->test_fallback('google_pagespeed', $test_url);
        }

        $api_url = "https://www.googleapis.com/pagespeedonline/v5/runPagespeed?url=" . urlencode($test_url) . "&key=" . $api_key;
        
        $response = wp_remote_get($api_url, ['timeout' => 30]);
        
        if (is_wp_error($response)) {
            // Google API error, using fallback
            return $this->test_fallback('google_pagespeed', $test_url);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (isset($data['lighthouseResult']['audits']['speed-index']['score'])) {
            return [
                'performance_score' => round($data['lighthouseResult']['categories']['performance']['score'] * 100),
                'load_time' => number_format($data['lighthouseResult']['audits']['speed-index']['displayValue'] ?? 1.5, 1) . 's',
                'page_size' => $this->format_bytes($data['lighthouseResult']['audits']['total-byte-weight']['numericValue'] ?? 1000000),
                'requests_count' => $data['lighthouseResult']['audits']['network-requests']['details']['items'] ? count($data['lighthouseResult']['audits']['network-requests']['details']['items']) : 25,
                'provider' => 'google_pagespeed',
                'test_url' => $test_url,
                'timestamp' => current_time('mysql'),
                'raw_data' => $data // Store for debugging
            ];
        }

        return $this->test_fallback('google_pagespeed', $test_url);
    }

    private function test_gtmetrix($test_url) {
        // GTmetrix would require API key and different endpoint
        // API not implemented yet, using simulation
        return $this->test_fallback('gtmetrix', $test_url);
    }

    private function test_pingdom($test_url) {
        // Pingdom would require API key and different endpoint
        // API not implemented yet, using simulation
        return $this->test_fallback('pingdom', $test_url);
    }

    private function test_webpagetest($test_url) {
        // WebPageTest would require API key and different endpoint
        // API not implemented yet, using simulation
        return $this->test_fallback('webpagetest', $test_url);
    }

    private function test_fallback($provider, $test_url) {
        // Simulate different results based on provider for testing
        $base_scores = [
            'google_pagespeed' => ['min' => 85, 'max' => 98],
            'gtmetrix' => ['min' => 80, 'max' => 95],
            'pingdom' => ['min' => 75, 'max' => 92],
            'webpagetest' => ['min' => 82, 'max' => 96]
        ];

        $scores = $base_scores[$provider] ?? ['min' => 80, 'max' => 95];

        return [
            'performance_score' => rand($scores['min'], $scores['max']),
            'load_time' => number_format(rand(800, 2000) / 1000, 1) . 's',
            'page_size' => number_format(rand(500, 1500) / 100, 1) . 'MB',
            'requests_count' => rand(15, 45),
            'provider' => $provider,
            'test_url' => $test_url,
            'timestamp' => current_time('mysql'),
            'simulated' => true
        ];
    }

    private function get_provider_name($provider) {
        $names = [
            'google_pagespeed' => 'Google PageSpeed Insights',
            'gtmetrix' => 'GTmetrix',
            'pingdom' => 'Pingdom',
            'webpagetest' => 'WebPageTest'
        ];
        return $names[$provider] ?? $provider;
    }

    private function format_bytes($bytes) {
        if ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 1) . 'MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 1) . 'KB';
        }
        return $bytes . 'B';
    }

    public function ams_clear_speed_cache() {
        // Verify nonce for security
        if (!wp_verify_nonce($_POST['nonce'], 'ams_speed_cache_nonce')) {
            wp_send_json_error(['message' => esc_html__('Security check failed', 'ams-wc-amazon')]);
        }

        // Clear all speed test related transients
        delete_transient('ams_speed_test_results');
        delete_transient('ams_speed_test_cache');
        
        // Clear any other speed test caches
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_ams_speed_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_ams_speed_%'");

        wp_send_json_success([
            'message' => esc_html__('Speed test cache cleared successfully!', 'ams-wc-amazon')
        ]);
    }

    public function ams_test_api_key() {
        // Verify nonce for security
        if (!wp_verify_nonce($_POST['nonce'], 'ams_test_api_nonce')) {
            wp_send_json_error(['message' => esc_html__('Security check failed', 'ams-wc-amazon')]);
        }

        $provider = sanitize_text_field($_POST['provider'] ?? '');
        $api_key = sanitize_text_field($_POST['api_key'] ?? '');

        if (empty($provider) || empty($api_key)) {
            wp_send_json_error(['message' => esc_html__('Provider and API key are required', 'ams-wc-amazon')]);
        }

        // Test the API key based on provider
        $result = $this->test_api_key_validity($provider, $api_key);

        if ($result['valid']) {
            wp_send_json_success([
                'message' => sprintf(esc_html__('%s API key is valid', 'ams-wc-amazon'), $this->get_provider_name($provider)),
                'provider' => $provider,
                'details' => $result['details'] ?? []
            ]);
        } else {
            wp_send_json_error([
                'message' => $result['message'] ?? esc_html__('API key validation failed', 'ams-wc-amazon'),
                'provider' => $provider
            ]);
        }
    }

    private function test_api_key_validity($provider, $api_key) {
        switch ($provider) {
            case 'google_pagespeed':
                return $this->test_google_api_key($api_key);
            case 'gtmetrix':
                return $this->test_gtmetrix_api_key($api_key);
            case 'pingdom':
                return $this->test_pingdom_api_key($api_key);
            case 'webpagetest':
                return $this->test_webpagetest_api_key($api_key);
            default:
                return ['valid' => false, 'message' => 'Unknown provider'];
        }
    }

    private function test_google_api_key($api_key) {
        // Test Google PageSpeed Insights API
        $test_url = 'https://example.com'; // Simple test URL
        $api_url = "https://www.googleapis.com/pagespeedonline/v5/runPagespeed?url=" . urlencode($test_url) . "&key=" . $api_key;
        
        $response = wp_remote_get($api_url, ['timeout' => 15]);
        
        if (is_wp_error($response)) {
            return ['valid' => false, 'message' => 'Network error: ' . $response->get_error_message()];
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($response_code === 200 && isset($data['lighthouseResult'])) {
            return [
                'valid' => true, 
                'message' => 'API key is valid',
                'details' => ['quota_remaining' => 'Available']
            ];
        } elseif ($response_code === 403) {
            return ['valid' => false, 'message' => 'Invalid API key or quota exceeded'];
        } elseif ($response_code === 400) {
            return ['valid' => false, 'message' => 'Bad request - check API key format'];
        } else {
            return ['valid' => false, 'message' => 'API test failed with code: ' . $response_code];
        }
    }

    private function test_gtmetrix_api_key($api_key) {
        // GTmetrix API test - placeholder for now
        return ['valid' => false, 'message' => 'GTmetrix API testing not implemented yet'];
    }

    private function test_pingdom_api_key($api_key) {
        // Pingdom API test - placeholder for now
        return ['valid' => false, 'message' => 'Pingdom API testing not implemented yet'];
    }

    private function test_webpagetest_api_key($api_key) {
        // WebPageTest API test
        $api_url = "https://www.webpagetest.org/runtest.php?url=https://example.com&k=" . $api_key . "&f=json";
        
        $response = wp_remote_get($api_url, ['timeout' => 15]);
        
        if (is_wp_error($response)) {
            return ['valid' => false, 'message' => 'Network error: ' . $response->get_error_message()];
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($response_code === 200 && isset($data['statusCode']) && $data['statusCode'] == 200) {
            return [
                'valid' => true, 
                'message' => 'API key is valid',
                'details' => ['test_id' => $data['data']['testId'] ?? 'N/A']
            ];
        } elseif (isset($data['statusText'])) {
            return ['valid' => false, 'message' => $data['statusText']];
        } else {
            return ['valid' => false, 'message' => 'API test failed with code: ' . $response_code];
        }
    }

    public function validate_api_for_search() {
        // Verify nonce for security
        if (!wp_verify_nonce($_POST['_wpnonce'] ?? '', 'wca_search_product')) {
            wp_send_json_error(['message' => esc_html__('Security check failed', 'ams-wc-amazon')]);
        }

        $accessKey = get_option('ams_access_key_id');
        $secretKey = get_option('ams_secret_access_key');
        $associateTag = get_option('ams_associate_tag');
        $selectedLocale = get_option('ams_amazon_country');

        // Check if credentials are missing
        if (empty($accessKey) || empty($secretKey) || empty($associateTag)) {
            wp_send_json_error([
                'message' => esc_html__('API credentials missing', 'ams-wc-amazon'),
                'type' => 'missing_credentials'
            ]);
        }

        // Check API usage to prevent throttling
        $apiUsage = \Amazon\Affiliate\Api\Amazon_Product_Api::get_api_usage();
        if ($apiUsage['used_percentage'] > 85) {
            wp_send_json_error([
                'message' => esc_html__('API usage limit reached. Please wait a few minutes.', 'ams-wc-amazon'),
                'type' => 'rate_limit'
            ]);
        }

        $regions = ams_get_amazon_regions();
        
        // Test credentials for selected locale
        $result = $this->test_credentials_for_store($accessKey, $secretKey, $associateTag, $selectedLocale, $regions[$selectedLocale]);
        
        if ($result === true) {
            // Success
            $storeName = $this->get_store_name($selectedLocale, $regions);
            wp_send_json_success([
                'message' => sprintf(
                    esc_html__('API connected to %s store', 'ams-wc-amazon'),
                    $storeName
                ),
                'store' => $storeName,
                'locale' => $selectedLocale
            ]);
        } else if ($result === 'STORE_MISMATCH') {
            // Country/marketplace mismatch - find which country actually works
            list($correctLocale, $errorMessage) = $this->detect_correct_store($accessKey, $secretKey, $associateTag, $regions);
            
            if ($correctLocale) {
                $correctStoreName = $this->get_store_name($correctLocale, $regions);
                $selectedStoreName = $this->get_store_name($selectedLocale, $regions);
                
                wp_send_json_error([
                    'message' => sprintf(
                        esc_html__('API credentials are for %s (%s), but you selected %s (%s)', 'ams-wc-amazon'),
                        $correctStoreName,
                        strtoupper($correctLocale),
                        $selectedStoreName,
                        strtoupper($selectedLocale)
                    ),
                    'type' => 'country_mismatch',
                    'correct_country' => $correctLocale,
                    'correct_store' => $correctStoreName
                ]);
            } else {
                wp_send_json_error([
                    'message' => esc_html__('API credentials don\'t match selected country', 'ams-wc-amazon'),
                    'type' => 'country_mismatch'
                ]);
            }
        } else {
            // Parse error response
            $errorMessage = $this->parse_api_error($result);
            wp_send_json_error([
                'message' => $errorMessage,
                'type' => 'api_error'
            ]);
        }
    }

    private function parse_api_error($result) {
        // Extract clean error message from API response
        if (is_string($result)) {
            $decodedError = json_decode($result);
            if ($decodedError && isset($decodedError->Errors[0]->Message)) {
                $errorMessage = $decodedError->Errors[0]->Message;
                
                // Simplify common error messages
                if (strpos($errorMessage, 'InvalidClientTokenId') !== false) {
                    return esc_html__('Invalid Access Key ID', 'ams-wc-amazon');
                }
                if (strpos($errorMessage, 'SignatureDoesNotMatch') !== false) {
                    return esc_html__('Invalid Secret Access Key', 'ams-wc-amazon');
                }
                if (strpos($errorMessage, 'InvalidPartnerTag') !== false) {
                    return esc_html__('Invalid Associate Tag', 'ams-wc-amazon');
                }
                
                return esc_html(substr($errorMessage, 0, 100));
            }
        }
        
        return esc_html__('API validation failed', 'ams-wc-amazon');
    }

    public function test_api() {
        $accessKey = get_option('ams_access_key_id');
        $secretKey = get_option('ams_secret_access_key');
        $associateTag = get_option('ams_associate_tag');
        $selectedLocale = get_option('ams_amazon_country');

        // Check if all required fields are filled
        if (empty($accessKey) || empty($secretKey) || empty($associateTag)) {
            $message = '<div class="alert alert-danger" role="alert">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Error:</strong> Please fill in all required Amazon API settings (Access Key, Secret Key, and Associate Tag).
            </div>';
            echo wp_kses_post($message);
            wp_die();
        }

        // Check API usage to prevent throttling
        $apiUsage = \Amazon\Affiliate\Api\Amazon_Product_Api::get_api_usage();
        if ($apiUsage['used_percentage'] > 85) {
            $message = '<div class="alert alert-warning" role="alert">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Warning:</strong> API usage is high (' . $apiUsage['used_percentage'] . '%). Please wait a few minutes before testing to avoid throttling.
            </div>';
            echo wp_kses_post($message);
            wp_die();
        }

        $regions = ams_get_amazon_regions();
        
        // First, test only the selected locale
        $result = $this->test_credentials_for_store($accessKey, $secretKey, $associateTag, $selectedLocale, $regions[$selectedLocale]);
        
        if ($result === true) {
            // Success with selected locale
            $detectedLocale = $selectedLocale;
            $detectedStoreName = $this->get_store_name($detectedLocale, $regions);
            
            $message = sprintf(
                '<div class="alert alert-success" role="alert">
                    <i class="fas fa-check-circle me-2"></i>
                    <strong>Success!</strong> Your Amazon API credentials are valid for the <strong>%s (%s)</strong> store.
                    <br>Your current country selection is correct.
                </div>',
                esc_html(strtoupper($detectedLocale)),
                esc_html($detectedStoreName)
            );

            // Fetch additional API details
            sleep(1);
            $apiDetails = $this->get_api_details($accessKey, $secretKey, $associateTag, $detectedLocale, $regions[$detectedLocale]);
            if ($apiDetails) {
                $message .= $this->format_api_details($apiDetails);
            }
        } else {
            // Debug: Log what result we got
            // Remove or comment out error_log for API test result
            // error_log("API test result for {$selectedLocale}: " . $result);
            
            // Check if it's a store mismatch (wrong country/region)
            if ($result === 'STORE_MISMATCH') {
                // Simple store mismatch message - only test the selected country
                $selectedStoreName = $this->get_store_name($selectedLocale, $regions);
                
                $message = sprintf(
                    '<div class="alert alert-warning" role="alert">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Store Mismatch:</strong> Your API credentials are not authorized for the selected Amazon store.
                        <br><br><strong>Test Results:</strong> Your credentials are valid but not approved for <strong>%s (%s)</strong>.
                        <br><br><strong>Guidance:</strong> Please:
                        <ul style="margin-top: 10px; margin-bottom: 10px;">
                            <li>• Change your country selection to where your credentials are authorized</li>
                            <li>• Visit your Amazon Associates account for the correct country</li>
                            <li>• Ensure your Associate Tag is approved for the selected marketplace</li>
                            <li>• Check if your Associate Tag is active and approved</li>
                        </ul>
                    </div>',
                    esc_html(strtoupper($selectedLocale)),
                    esc_html($selectedStoreName)
                );
            } else {
                // Check if it's a throttling error
                if (strpos($result, 'throttling') !== false || strpos($result, 'TooManyRequests') !== false) {
                    // For single country tests, treat throttling as store mismatch
                    $selectedStoreName = $this->get_store_name($selectedLocale, $regions);
                    
                    $message = sprintf(
                        '<div class="alert alert-warning" role="alert">
                            <i class="fas fa-info-circle me-2"></i>
                            <strong>Store Mismatch:</strong> Your API credentials are not authorized for the selected Amazon store.
                            <br><br><strong>Test Results:</strong> Your credentials are valid but not approved for <strong>%s (%s)</strong>.
                            <br><br><strong>Note:</strong> The API returned a rate limiting error, which typically indicates the credentials are not authorized for this marketplace.
                            <br><br><strong>Guidance:</strong> Please:
                            <ul style="margin-top: 10px; margin-bottom: 10px;">
                                <li>• Change your country selection to where your credentials are authorized</li>
                                <li>• Visit your Amazon Associates account for the correct country</li>
                                <li>• Ensure your Associate Tag is approved for the selected marketplace</li>
                                <li>• Check if your Associate Tag is active and approved</li>
                            </ul>
                        </div>',
                        esc_html(strtoupper($selectedLocale)),
                        esc_html($selectedStoreName)
                    );
                } else {
                    // Check if it's a credential error
                    if (strpos($result, 'InvalidClientTokenId') !== false || 
                        strpos($result, 'UnrecognizedClientException') !== false ||
                        strpos($result, 'InvalidSignatureException') !== false) {
                        
                        $message = $this->format_error_message($result);
                    } else {
                        // Generic error handling
                        $decodedError = json_decode($result);
                        if ($decodedError && isset($decodedError->__type) && isset($decodedError->Errors)) {
                            $amazonError = $decodedError->Errors[0]->Message ?? 'Unknown Amazon API error';
                            $errorType = $decodedError->Errors[0]->Code ?? '';
                            
                            $message = $this->format_error_message($amazonError, $errorType);
                        } else {
                            $message = $this->format_error_message($result);
                        }
                    }
                }
            }
        }

        echo wp_kses_post($message);
        wp_die();
    }

    private function test_credentials_for_store($accessKey, $secretKey, $associateTag, $locale, $regionInfo) {
        $keyword = 'test';
        $marketplace = ams_get_marketplace_value($locale);
        $serviceName = 'ProductAdvertisingAPI';
        $region = $regionInfo['RegionCode'];
        $host = $regionInfo['Host'];

        $payloadArr = array(
            'Keywords' => $keyword,
            'Resources' => array('ItemInfo.Title'),
            'PartnerTag' => $associateTag,
            'PartnerType' => 'Associates',
            'Marketplace' => $marketplace,
            'ItemCount' => 1
        );

        $payload = json_encode($payloadArr);
        $uri_path = "/paapi5/searchitems";

        try {
            $api = new \Amazon\Affiliate\Api\Amazon_Product_Api($accessKey, $secretKey, $region, $serviceName, $uri_path, $payload, $host, 'SearchItems');
            $response = $api->do_request();

            if (isset($response->SearchResult)) {
                return true;
            } elseif (isset($response->__type) && isset($response->Errors)) {
                // Check if this is a store mismatch error in the JSON response
                $errorMessage = $response->Errors[0]->Message ?? '';
                $errorCode = $response->Errors[0]->Code ?? '';
                
                // Log the error for debugging
                // Remove or comment out error_log for Amazon API error
                // error_log("Amazon API error for {$locale}: Code={$errorCode}, Message={$errorMessage}");
                
                // Check for Associate Tag errors first (these should be treated as credential errors, not store mismatch)
                if (strpos($errorMessage, 'InvalidPartnerTag') !== false ||
                    strpos($errorMessage, 'InvalidAssociateTag') !== false ||
                    $errorCode === 'InvalidPartnerTag' ||
                    $errorCode === 'InvalidAssociateTag') {
                    return json_encode($response); // Return the raw error for proper credential error handling
                }
                
                // Check for store mismatch errors
                if (strpos($errorMessage, 'not mapped to Primary of approved associate store') !== false ||
                    strpos($errorMessage, 'not authorized for this marketplace') !== false ||
                    strpos($errorMessage, 'not valid for this region') !== false ||
                    strpos($errorMessage, 'InvalidParameterValue') !== false ||
                    strpos($errorMessage, 'InvalidMarketplace') !== false ||
                    strpos($errorMessage, 'The value') !== false && strpos($errorMessage, 'provided in the request for Marketplace is invalid') !== false ||
                    $errorCode === 'InvalidParameterValue' ||
                    $errorCode === 'InvalidMarketplace') {
                    return 'STORE_MISMATCH';
                }
                // This captures the Amazon error message structure we saw earlier
                return json_encode($response);
            }
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            
            // Log the error for debugging
            // Remove or comment out error_log for Amazon API test error
            // error_log("Amazon API test error for {$locale}: " . $error_message);
            
            // Check for Associate Tag errors first (these should be treated as credential errors, not store mismatch)
            if (strpos($error_message, 'InvalidPartnerTag') !== false ||
                strpos($error_message, 'InvalidAssociateTag') !== false ||
                strpos($error_message, 'partner tag is not mapped to a valid associate store') !== false ||
                strpos($error_message, 'not mapped to a valid associate store') !== false) {
                return 'Invalid Associate Tag. Please check your Associate Tag in your Amazon Associates account.';
            }
            
            // Check for specific error types that indicate wrong region/store
            if (strpos($error_message, 'InvalidParameterValue') !== false ||
                strpos($error_message, 'InvalidMarketplace') !== false ||
                strpos($error_message, 'not mapped to Primary of approved associate store') !== false ||
                strpos($error_message, 'not authorized for this marketplace') !== false ||
                strpos($error_message, 'not valid for this region') !== false ||
                strpos($error_message, 'The value') !== false && strpos($error_message, 'provided in the request for Marketplace is invalid') !== false ||
                strpos($error_message, 'Marketplace is invalid') !== false) {
                return 'STORE_MISMATCH'; // Special flag for store mismatch
            }
            
            // Return specific error messages based on the exception
            if (strpos($error_message, 'throttling') !== false || 
                strpos($error_message, 'TooManyRequests') !== false) {
                return 'API throttling detected. Please wait before testing again.';
            } elseif (strpos($error_message, 'InvalidClientTokenId') !== false) {
                return 'Invalid Access Key ID. Please check your credentials.';
            } elseif (strpos($error_message, 'UnrecognizedClientException') !== false) {
                return 'Invalid credentials. Please verify your Access Key and Secret Key.';
            } elseif (strpos($error_message, 'InvalidSignatureException') !== false) {
                return 'Signature error. Please check your Secret Access Key.';
            } elseif (strpos($error_message, 'Network error') !== false) {
                return 'Network connection error. Please check your internet connection.';
            } else {
                return $error_message;
            }
        }

        return false;
    }

    private function detect_correct_store($accessKey, $secretKey, $associateTag, $regions) {
        $lastErrorMessage = '';
        foreach ($regions as $locale => $regionInfo) {
            $result = $this->test_credentials_for_store($accessKey, $secretKey, $associateTag, $locale, $regionInfo);
            if ($result === true) {
                return array($locale, '');
            } elseif (is_string($result)) {
                $lastErrorMessage = $result;
            }
        }
        return array(null, $lastErrorMessage);
    }

    private function get_store_name($locale, $regions) {
        $countryNames = [
            'us' => 'United States',
            'uk' => 'United Kingdom',
            'de' => 'Germany',
            'fr' => 'France',
            'jp' => 'Japan',
            'ca' => 'Canada',
            'it' => 'Italy',
            'es' => 'Spain',
            'in' => 'India',
            'br' => 'Brazil',
            'mx' => 'Mexico',
            'au' => 'Australia',
            'sg' => 'Singapore',
            'ae' => 'United Arab Emirates',
            'nl' => 'Netherlands',
            'sa' => 'Saudi Arabia',
            'se' => 'Sweden',
            'pl' => 'Poland',
            'tr' => 'Turkey',
        ];

        $locale = strtolower($locale);
        if (isset($countryNames[$locale])) {
            return $countryNames[$locale] . ' (Amazon.' . $locale . ')';
        }
        
        // Fallback to using the name from regions if available
        if (isset($regions[$locale]['Name'])) {
            return $regions[$locale]['Name'] . ' (Amazon.' . $locale . ')';
        }
        
        // If all else fails, return the locale code
        return 'Amazon.' . $locale;
    }

    private function get_api_details($accessKey, $secretKey, $associateTag, $locale, $regionInfo) {
        $keyword = 'bestseller'; // Using a more general keyword that should return results in any locale
        $marketplace = ams_get_marketplace_value($locale);
        $serviceName = 'ProductAdvertisingAPI';
        $region = $regionInfo['RegionCode'];
        $host = $regionInfo['Host'];

        $payloadArr = array(
            'Keywords' => $keyword,
            'Resources' => array('ItemInfo.Title', 'Offers.Listings.Price'),
            'PartnerTag' => $associateTag,
            'PartnerType' => 'Associates',
            'Marketplace' => $marketplace,
            'ItemCount' => 1
        );

        $payload = json_encode($payloadArr);
        $uri_path = "/paapi5/searchitems";

        try {
            $api = new \Amazon\Affiliate\Api\Amazon_Product_Api($accessKey, $secretKey, $region, $serviceName, $uri_path, $payload, $host, 'SearchItems');
            $response = $api->do_request();

            if (isset($response->SearchResult)) {
                return [
                    'totalResults' => $response->SearchResult->TotalResultCount ?? 0,
                    'searchURL' => $response->SearchResult->SearchURL ?? '',
                    'keyword' => $keyword
                ];
            }
        } catch (Exception $e) {
            // If an exception is thrown, we'll return null
        }

        return null;
    }

    private function format_api_details($apiDetails) {
        $containerStyle = 'background-color: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; padding: 20px; margin-top: 20px; font-family: Arial, sans-serif;';
        $headingStyle = 'color: #0d6efd; margin-top: 0; margin-bottom: 15px; font-size: 1.2em; font-weight: bold;';
        $paragraphStyle = 'margin-bottom: 15px; line-height: 1.5; color: #212529;';
        $listStyle = 'list-style-type: none; padding-left: 0;';
        $listItemStyle = 'margin-bottom: 10px; color: #212529;';
        $strongStyle = 'color: #212529; font-weight: bold;';
        $linkStyle = 'color: #0d6efd; text-decoration: none; font-weight: 500;';
        $keywordStyle = 'background-color: #e9ecef; padding: 2px 5px; border-radius: 3px; font-family: monospace; color: #212529; font-weight: 500;';

        $formattedDetails = "<div style='{$containerStyle}'>";
        $formattedDetails .= "<h4 style='{$headingStyle}'>API Test Results</h4>";
        $formattedDetails .= "<p style='{$paragraphStyle}'>We performed a test search using the keyword <span style='{$keywordStyle}'>" . esc_html($apiDetails['keyword']) . "</span> to verify your API functionality.</p>";
        $formattedDetails .= "<ul style='{$listStyle}'>";
        $formattedDetails .= "<li style='{$listItemStyle}'><strong style='{$strongStyle}'>Total Results Found:</strong> " . esc_html(number_format($apiDetails['totalResults'])) . "</li>";
        if (!empty($apiDetails['searchURL'])) {
            $formattedDetails .= "<li style='{$listItemStyle}'><strong style='{$strongStyle}'>Test Search Results:</strong> <a href='" . esc_url($apiDetails['searchURL']) . "' target='_blank' style='{$linkStyle}'>View on Amazon</a></li>";
        }
        $formattedDetails .= "</ul>";
        $formattedDetails .= "<p style='{$paragraphStyle}'>These results confirm that your Amazon Product Advertising API is working correctly and able to fetch product data.</p>";
        $formattedDetails .= "</div>";

        return $formattedDetails;
    }

    private function format_error_message($error, $errorType = '') {
        $icon = 'fas fa-exclamation-triangle';
        $title = 'API Test Failed';
        
        // Try to parse JSON error response from Amazon
        $decodedError = json_decode($error);
        $amazonErrorType = '';
        $amazonErrorMessage = '';
        
        if ($decodedError && isset($decodedError->__type) && isset($decodedError->Errors)) {
            $amazonErrorType = $decodedError->__type ?? '';
            $amazonErrorMessage = $decodedError->Errors[0]->Message ?? '';
            $amazonErrorCode = $decodedError->Errors[0]->Code ?? '';
        }
        
        // Determine error type and provide specific guidance
        if (strpos($error, 'throttling') !== false || strpos($error, 'TooManyRequests') !== false || 
            strpos($amazonErrorType, 'TooManyRequests') !== false) {
            $icon = 'fas fa-clock';
            $title = 'Rate Limiting Detected';
            $guidance = 'Please wait a few minutes before testing again. Amazon limits API requests to prevent abuse.';
        } elseif (strpos($error, 'InvalidClientTokenId') !== false || 
                  strpos($amazonErrorType, 'InvalidClientTokenId') !== false) {
            $icon = 'fas fa-key';
            $title = 'Invalid Access Key';
            $guidance = 'Please check your Access Key ID. It should be a 20-character alphanumeric string.';
        } elseif (strpos($error, 'UnrecognizedClientException') !== false || 
                  strpos($amazonErrorType, 'UnrecognizedClientException') !== false ||
                  $amazonErrorCode === 'UnrecognizedClient') {
            $icon = 'fas fa-user-times';
            $title = 'Invalid Access Key ID';
            $guidance = 'Your Access Key ID appears to be incorrect. Please verify your Access Key ID in your Amazon Associates account. Make sure you copied the entire key without any extra spaces.';
        } elseif (strpos($error, 'InvalidSignatureException') !== false || 
                  strpos($amazonErrorType, 'InvalidSignatureException') !== false ||
                  $amazonErrorCode === 'InvalidSignature') {
            $icon = 'fas fa-shield-alt';
            $title = 'Invalid Secret Access Key';
            $guidance = 'Your Secret Access Key appears to be incorrect. Please verify your Secret Access Key in your Amazon Associates account. Make sure you copied the entire key without any extra spaces.';
        } elseif (strpos($error, 'InvalidPartnerTag') !== false || 
                  strpos($amazonErrorType, 'InvalidPartnerTag') !== false ||
                  strpos($error, 'partner tag is not mapped to a valid associate store') !== false ||
                  strpos($error, 'not mapped to a valid associate store') !== false) {
            $icon = 'fas fa-tag';
            $title = 'Invalid Associate Tag';
            $guidance = 'Your Associate Tag appears to be incorrect. Please verify your Associate Tag in your Amazon Associates account. Make sure you copied the entire tag without any extra spaces.';
        } else {
            $guidance = 'Please verify your API credentials and ensure they are correct for the selected Amazon marketplace.';
        }

        // Create a more user-friendly error display
        $errorDisplay = $error;
        if ($amazonErrorMessage && $amazonErrorType) {
            // Clean up Amazon's error message by removing all technical references
            $cleanMessage = preg_replace('/If you are using an AWS SDK, requests are signed for you automatically; otherwise, go to https:\/\/webservices\.amazon\.[a-z]{2}\/paapi5\/documentation\/[^\s]+/', '', $amazonErrorMessage);
            $cleanMessage = trim($cleanMessage);
            
            // Show a cleaner version of Amazon's error
            $errorDisplay = sprintf(
                'Amazon API Error: %s (Code: %s)',
                esc_html($cleanMessage),
                esc_html($amazonErrorCode ?? 'Unknown')
            );
        }

        return sprintf(
            '<div class="alert alert-danger" role="alert">
                <i class="%s me-2"></i>
                <strong>%s:</strong> Unable to validate your API credentials.
                <br><br><strong>Error details:</strong> %s
                <br><br><strong>Guidance:</strong> %s
            </div>',
            esc_attr($icon),
            esc_html($title),
            esc_html($errorDisplay),
            esc_html($guidance)
        );
    }

    /**
     * Auto-detect the best theme hook for button placement
     */
    public function auto_detect_theme_hook($theme_name) {
        // Enhanced theme configurations with smart detection
        $theme_hooks = [
            // Popular WooCommerce themes with advanced detection
            'Ricky' => 'woocommerce_after_shop_loop_item', // From content-product.php analysis
            'Storefront' => 'woocommerce_after_shop_loop_item',
            'Astra' => 'woocommerce_after_shop_loop_item_title',
            'OceanWP' => 'ocean_after_shop_loop_item',
            'GeneratePress' => 'woocommerce_after_shop_loop_item',
            'Flatsome' => 'flatsome_after_shop_loop_item',
            'Avada' => 'avada_woocommerce_after_shop_item',
            'Divi' => 'woocommerce_after_shop_loop_item',
            'TheGem' => 'thegem_woocommerce_after_shop_loop_item',
            'Woodmart' => 'woodmart_after_shop_loop_item',
            'Porto' => 'porto_woocommerce_after_shop_loop_item',
            'Enfold' => 'ava_after_shop_loop_item',
            'BeTheme' => 'mfn_woocommerce_after_shop_loop_item',
            'Salient' => 'salient_woocommerce_after_shop_loop_item',
            'X Theme' => 'x_woocommerce_after_shop_loop_item',
            'Pro' => 'x_woocommerce_after_shop_loop_item',
            'Uncode' => 'uncode_woocommerce_after_shop_loop_item',
            'Bridge' => 'bridge_woocommerce_after_shop_loop_item',
            'Jupiter' => 'mk_woocommerce_after_shop_loop_item',
            'Total' => 'wpex_woocommerce_after_shop_loop_item',
            'Kallyas' => 'zn_woocommerce_after_shop_loop_item',
            
            // Default fallback
            'default' => 'AUTO_DETECTED' // Smart detection for unknown themes
        ];

        // Check if we have a specific configuration for this theme
        if (isset($theme_hooks[$theme_name])) {
            return $theme_hooks[$theme_name];
        }

        // Check for theme name variations (case insensitive, partial matches)
        foreach ($theme_hooks as $theme => $hook) {
            if (stripos($theme_name, $theme) !== false) {
                return $hook;
            }
        }

        // Return default WooCommerce hook
        return $theme_hooks['default'];
    }

    /**
     * Actually scan theme files to detect real hooks
     */
    public function scan_theme_for_hooks() {
        $template_path = get_template_directory();
        $possible_files = [
            '/woocommerce/content-product.php',
            '/content-product.php', 
            '/woocommerce/loop/add-to-cart.php',
            '/template-parts/content-product.php'
        ];
        
        $detected_hooks = [];
        
        foreach ($possible_files as $file) {
            $full_path = $template_path . $file;
            if (file_exists($full_path)) {
                $content = file_get_contents($full_path);
                
                // Look for do_action calls with woocommerce hooks
                if (preg_match_all('/do_action\s*\(\s*[\'"]([^\'"]*(woocommerce|shop)_[^\'"]*)[\'"]/', $content, $matches)) {
                    foreach ($matches[1] as $hook) {
                        $detected_hooks[] = $hook;
                    }
                }
                
                // Look for comments that mention where add_to_cart is hooked
                if (preg_match_all('/@hooked\s+woocommerce_template_loop_add_to_cart.*?(\w+_\w+_\w+)/', $content, $matches)) {
                    foreach ($matches[1] as $hook) {
                        $detected_hooks[] = $hook;
                    }
                }
            }
        }
        
        // Return the most likely hook for add to cart
        if (in_array('woocommerce_after_shop_loop_item', $detected_hooks)) {
            return 'woocommerce_after_shop_loop_item (detected from theme)';
        } elseif (in_array('woocommerce_after_shop_loop_item_title', $detected_hooks)) {
            return 'woocommerce_after_shop_loop_item_title (detected from theme)';
        } elseif (!empty($detected_hooks)) {
            return $detected_hooks[0] . ' (detected from theme)';
        }
        
        return 'woocommerce_after_shop_loop_item (default fallback)';
    }

    public function review_settings_handler() {
        if (!wp_verify_nonce($_POST['review_settings_nonce'], 'review_settings_nonce')) {
            wp_die('Security check failed');
        }

        // Handle review settings
        $enable_amazon_review = isset($_POST['enable_amazon_review']) ? '1' : '0';
        $enable_reviewer_image = isset($_POST['enable_reviewer_image']) ? '1' : '0';
        $enable_review_title = isset($_POST['enable_review_title']) ? '1' : '0';
        $single_import_review_limit = sanitize_text_field($_POST['single_import_review_limit'] ?? '5');
        $multiple_import_review_limit = sanitize_text_field($_POST['multiple_import_review_limit'] ?? '5');

        // Update options
        update_option('enable_amazon_review', $enable_amazon_review);
        update_option('enable_reviewer_image', $enable_reviewer_image);
        update_option('enable_review_title', $enable_review_title);
        update_option('single_import_review_limit', $single_import_review_limit);
        update_option('multiple_import_review_limit', $multiple_import_review_limit);

        // Redirect back to the review import page with success message
        wp_redirect(admin_url('admin.php?page=product-review-import&settings_saved=1'));
        exit;
    }
}