<?php
namespace Amazon\Affiliate\Admin;

class ProductReviewImport extends ProductsSearchWithoutApi
{
    public function __construct()
    {
        // Constructor logic (if any) goes here
    }

    public function product_review_page()
    {
        if ('1' != get_option('enable_amazon_review', true)) {
            wp_redirect('admin.php?page=wc-product-setting-page&tab=pills-general-tab');
        }
        $this->get_amazon_cat();
        $this->get_wc_terms();
        $template = __DIR__ . '/views/product-review-import.php';
        if (file_exists($template)) {
            require_once $template;
        }
    }

    public function product_review_import($return_data = false) {
        // Check if user has proper capabilities
        if (!current_user_can('manage_woocommerce')) {
            $error_response = [
                'message' => esc_html__('Insufficient permissions.', 'ams-wc-amazon')
            ];
            
            if ($return_data) {
                return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
            }
            
            wp_send_json_error($error_response);
        }

        // Verify nonce (skip for automation calls)
        if (!$return_data) {
            $nonce = sanitize_text_field($_POST['_wpnonce'] ?? '');
        if (!wp_verify_nonce($nonce, 'ams_product_review_import_nonce')) {
                wp_send_json_error([
                    'message' => esc_html__('Security verification failed.', 'ams-wc-amazon')
                ]);
            }
        }
        
        // Validate and sanitize input
        $ams_product = sanitize_text_field($_POST['ams_product'] ?? '');
        if (empty($ams_product) || !is_numeric($ams_product)) {
            $error_response = [
                'message' => esc_html__('Invalid product ID provided.', 'ams-wc-amazon')
            ];
            
            if ($return_data) {
                return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
            }
            
            wp_send_json_error($error_response);
        }  
        // Get and validate product
        $product = wc_get_product(intval($ams_product));
        if (!$product || !$product->exists()) {
            $error_response = [
                'status' => false, 
                'message' => '<div class="alert alert-danger w-100">' . esc_html__('Please select a valid product!', 'ams-wc-amazon') . '</div>'
            ];
            
            if ($return_data) {
                return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
            }
            
            wp_send_json($error_response);
        }

        // Get and validate product URL
        $product_url = get_post_meta($product->get_id(), '_ams_product_url', true);
        if (empty($product_url) || !filter_var($product_url, FILTER_VALIDATE_URL)) {
            $error_response = [
                'status' => false,
                'message' => '<div class="alert alert-danger w-100">' . esc_html__('Product does not have a valid Amazon URL.', 'ams-wc-amazon') . '</div>'
            ];
            
            if ($return_data) {
                return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
            }
            
            wp_send_json($error_response);
        }
        
        // Get and validate review limit from form input or settings
        $limit = isset($_POST['review_limit']) ? absint($_POST['review_limit']) : absint(get_option('single_import_review_limit', 10));
        $limit = max(1, min($limit, 50)); // Ensure limit is between 1 and 50
        
        // Get minimum rating filter from form input
        $min_rating = isset($_POST['min_rating']) ? absint($_POST['min_rating']) : 1;
        $min_rating = max(1, min($min_rating, 5)); // Ensure rating is between 1 and 5
        
        // Get advanced filters from form input
        $min_length = isset($_POST['min_length']) ? absint($_POST['min_length']) : 0;
        $review_age = isset($_POST['review_age']) ? absint($_POST['review_age']) : 0;
        $verified_only = isset($_POST['verified_only']) && $_POST['verified_only'] === 'on';
        


        // Get user agent
        $user_agent = $this->user_agent();
        
        // Get product data using existing method with error handling
        try {
        $response_body = fetchAndValidateProductData($product_url, $user_agent, false);
        if (!is_string($response_body) || empty($response_body)) {

                $error_response = [
                    'status' => false,
                    'message' => '<div class="alert alert-danger w-100">' . esc_html__('Failed to fetch product data from Amazon. Please try again later.', 'ams-wc-amazon') . '</div>'
                ];
                
                if ($return_data) {
                    return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
                }
                
                wp_send_json($error_response);
            }
        } catch (Exception $e) {

            $error_response = [
                'status' => false,
                'message' => '<div class="alert alert-danger w-100">' . esc_html__('An error occurred while fetching product data.', 'ams-wc-amazon') . '</div>'
            ];
            
            if ($return_data) {
                return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
            }
            
            wp_send_json($error_response);
        }

        // Create HTML object with error handling
        if (!class_exists('simple_html_dom')) {
            require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
        }
        
        try {
        $html = new \simple_html_dom();
            if (!$html->load($response_body)) {

                $error_response = [
                    'status' => false,
                    'message' => '<div class="alert alert-danger w-100">' . esc_html__('Failed to parse Amazon page content.', 'ams-wc-amazon') . '</div>'
                ];
                
                if ($return_data) {
                    return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
                }
                
                wp_send_json($error_response);
            }
        } catch (Exception $e) {

            $error_response = [
                'status' => false,
                'message' => '<div class="alert alert-danger w-100">' . esc_html__('Error parsing Amazon page content.', 'ams-wc-amazon') . '</div>'
            ];
            
            if ($return_data) {
                return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
            }
            
            wp_send_json($error_response);
        }

        // Check for broken page
        $message = check_for_broken_page($response_body, $html);
        if ($message !== null) {
            $error_response = [
                'status' => false,
                'message' => '<div class="alert alert-danger w-100">' . $message . '</div>'
            ];
            
            if ($return_data) {
                return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
            }
            
            wp_send_json($error_response);
        }

        // Use existing scrape function to get reviews from the main product page HTML
        try {
            // Get more reviews than needed to allow for filtering
            $scrape_limit = min($limit * 3, 50); // Get up to 3x the requested amount for filtering
            $all_reviews = scrape_amazon_reviews($html, $scrape_limit);
            
            if (empty($all_reviews)) {

                $error_response = [
                    'status' => false,
                    'message' => '<div class="alert alert-info w-100">' . esc_html__('No reviews found for this product on Amazon.', 'ams-wc-amazon') . '</div>'
                ];
                
                if ($return_data) {
                    return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
                }
                
                wp_send_json($error_response);
            }
            

            
            // Apply all filters and limit
            $reviews = [];
            $count = 0;
            $filtered_out = ['rating' => 0, 'length' => 0, 'age' => 0, 'verified' => 0];
            
            foreach ($all_reviews as $hash => $review) {
                $skip_review = false;
                $skip_reason = '';
                
                // Check minimum rating requirement
                $review_rating = isset($review['rating']) ? floatval($review['rating']) : 0;
                if ($review_rating < $min_rating) {
                    $filtered_out['rating']++;
                    $skip_review = true;
                    $skip_reason = 'rating';
                }
                
                // Check minimum length requirement
                if (!$skip_review && $min_length > 0) {
                    $review_text = isset($review['text']) ? $review['text'] : '';
                    if (strlen(trim($review_text)) < $min_length) {
                        $filtered_out['length']++;
                        $skip_review = true;
                        $skip_reason = 'length';
                    }
                }
                
                // Check review age requirement
                if (!$skip_review && $review_age > 0) {
                    $review_date = isset($review['date']) ? $review['date'] : '';
                    if (!empty($review_date)) {
                        $review_timestamp = strtotime($review_date);
                        $cutoff_timestamp = strtotime("-{$review_age} days");
                        if ($review_timestamp < $cutoff_timestamp) {
                            $filtered_out['age']++;
                            $skip_review = true;
                            $skip_reason = 'age';
                        }
                    }
                }
                
                // Check verified purchase requirement
                if (!$skip_review && $verified_only) {
                    $is_verified = isset($review['verified']) ? $review['verified'] : false;
                    if (!$is_verified) {
                        $filtered_out['verified']++;
                        $skip_review = true;
                        $skip_reason = 'verified';
                    }
                }
                
                // If review passes all filters, add it
                if (!$skip_review) {
                    $reviews[$hash] = $review;
                    $count++;
                    
                    // Stop when we reach the requested limit
                    if ($count >= $limit) {
                        break;
                    }
                }
            }
            
            // Log filtering results
            $total_filtered = array_sum($filtered_out);

        
        if (empty($reviews)) {
                // Create detailed error message based on what was filtered out
                $filter_reasons = [];
                if ($filtered_out['rating'] > 0) $filter_reasons[] = sprintf(__('%d reviews below %d stars', 'ams-wc-amazon'), $filtered_out['rating'], $min_rating);
                if ($filtered_out['length'] > 0) $filter_reasons[] = sprintf(__('%d reviews too short (<%d chars)', 'ams-wc-amazon'), $filtered_out['length'], $min_length);
                if ($filtered_out['age'] > 0) $filter_reasons[] = sprintf(__('%d reviews too old (>%d days)', 'ams-wc-amazon'), $filtered_out['age'], $review_age);
                if ($filtered_out['verified'] > 0) $filter_reasons[] = sprintf(__('%d non-verified reviews', 'ams-wc-amazon'), $filtered_out['verified']);
                
                $filter_message = !empty($filter_reasons) ? 
                    __('All reviews were filtered out: ', 'ams-wc-amazon') . implode(', ', $filter_reasons) . '.' :
                    __('No reviews found matching your criteria.', 'ams-wc-amazon');
                
                $filter_error = [
                    'status' => false,
                    'message' => '<div class="alert alert-info w-100">' . $filter_message . ' ' . 
                                esc_html__('Try adjusting your filters to be less restrictive.', 'ams-wc-amazon') . '</div>'
                ];
                
                if ($return_data) {
                    return [
                        'success' => false,
                        'message' => $filter_message . ' ' . esc_html__('Try adjusting your filters to be less restrictive.', 'ams-wc-amazon'),
                        'reviews_imported' => 0
                    ];
                }
                
                wp_send_json($filter_error);
            }
            
        } catch (Exception $e) {

            $error_response = [
                'status' => false,
                'message' => '<div class="alert alert-danger w-100">' . esc_html__('Error extracting reviews from Amazon page.', 'ams-wc-amazon') . '</div>'
            ];
            
            if ($return_data) {
                return ['success' => false, 'message' => $error_response['message'], 'reviews_imported' => 0];
            }
            
            wp_send_json($error_response);
        }

        // Get existing reviews
        $existing_reviews = get_comments([
            'post_id' => $product->get_id(),
            'type' => 'review',
            'status' => 'approve'
        ]);
        
        $existing_count = count($existing_reviews);
        
        // Create array of existing review hashes
        $existing_hashes = [];
        foreach ($existing_reviews as $existing_review) {
            $existing_hash = get_comment_meta($existing_review->comment_ID, 'review_hash', true);
            if (!empty($existing_hash)) {
                $existing_hashes[$existing_hash] = $existing_review->comment_ID;
            }
        }

        // Check if all reviews already exist
        $new_reviews_count = 0;
        foreach ($reviews as $review_hash => $review) {
            if (!isset($existing_hashes[$review_hash])) {
                $new_reviews_count++;
            }
        }

        if ($new_reviews_count === 0) {
            $no_new_response = [
                'status' => true,
                'data' => [
                    $product_url,
                    $limit,
                    [
                        'totalRating' => 0,
                        'totalReview' => 0,
                        'existingReviews' => $existing_count,
                        'result' => []
                    ]
                ],
                'message' => '<div class="alert alert-info text-center w-100">' . 
                    sprintf(__('No new reviews to import. Product already has %d reviews.', 'ams-wc-amazon'), $existing_count) . 
                '</div>'
            ];
            
            if ($return_data) {
                return [
                    'success' => true,
                    'message' => sprintf(__('No new reviews to import. Product already has %d reviews.', 'ams-wc-amazon'), $existing_count),
                    'reviews_imported' => 0,
                    'data' => $no_new_response['data'][2]
                ];
            }
            
            wp_send_json($no_new_response);
            return;
        }

        // Initialize rating totals
        $rating_sum = 0;
        $rating_count = 0;
        $review_results = [];

        // Process each review
        foreach ($reviews as $review_hash => $review) {
            // Skip if review already exists
            if (isset($existing_hashes[$review_hash])) {
                continue;
            }

            // Prepare comment data
            $commentdata = [
                'comment_post_ID' => $product->get_id(),
                'comment_author' => $review['reviewer_name'],
                'comment_content' => $review['text'],
                'comment_date' => $review['date'],
                'comment_date_gmt' => get_gmt_from_date($review['date']),
                'comment_approved' => 1,
                'comment_type' => 'review',
                'user_id' => 0
            ];

            // Insert the comment with error handling
            $comment_id = wp_insert_comment($commentdata);

            if ($comment_id && !is_wp_error($comment_id)) {
                // Add all the comment meta
                add_comment_meta($comment_id, 'rating', sanitize_text_field($review['rating']));
                add_comment_meta($comment_id, 'review_hash', sanitize_text_field($review_hash));
                add_comment_meta($comment_id, 'verified', 1);
                add_comment_meta($comment_id, 'title', sanitize_text_field($review['title']));

                if (!empty($review['reviewer_image']) && filter_var($review['reviewer_image'], FILTER_VALIDATE_URL)) {
                    add_comment_meta($comment_id, 'reviewer_image', esc_url_raw($review['reviewer_image']));
                }

                $rating_sum += floatval($review['rating']);
                $rating_count++;

                // Store for response
                $review_results[] = [
                    'title' => sanitize_text_field($review['title']),
                    'user' => sanitize_text_field($review['reviewer_name']),
                    'rating' => floatval($review['rating']),
                    'content' => sanitize_textarea_field($review['text'])
                ];
            } else {

            }
        }

        // Update product rating if we added any new reviews
        if ($rating_count > 0) {
            $product = wc_get_product($product->get_id());
            if ($product) {
                // Get actual count of approved reviews
                $actual_review_count = get_comments([
                    'post_id' => $product->get_id(),
                    'type' => 'review',
                    'status' => 'approve',
                    'count' => true
                ]);

                // Calculate actual rating sum
                $actual_rating_sum = 0;
                $product_reviews = get_comments([
                    'post_id' => $product->get_id(),
                    'type' => 'review',
                    'status' => 'approve'
                ]);

                foreach ($product_reviews as $review) {
                    $rating = get_comment_meta($review->comment_ID, 'rating', true);
                    $actual_rating_sum += floatval($rating);
                }

                // Calculate new average
                $new_average = $actual_rating_sum / $actual_review_count;

                // Update all rating meta
                update_post_meta($product->get_id(), '_wc_average_rating', round($new_average, 2));
                update_post_meta($product->get_id(), '_wc_rating_count', $actual_review_count);
                update_post_meta($product->get_id(), '_wc_review_count', $actual_review_count);
                update_post_meta($product->get_id(), '_wc_rating_sum', $actual_rating_sum);

                // Clear all relevant caches
                delete_transient('wc_product_reviews_' . $product->get_id());
                delete_transient('wc_average_rating_' . $product->get_id());
                wp_cache_delete($product->get_id(), 'product');
                
                if (function_exists('wc_delete_product_transients')) {
                    wc_delete_product_transients($product->get_id());
                }
            }
        }

        // Prepare response results
        $results = [
            'totalRating' => $rating_sum,
            'totalReview' => $rating_count,
            'existingReviews' => $existing_count,
            'result' => $review_results
        ];



        // Send response
        $success_response = [
            'status' => true,
            'data' => [$product_url, $limit, $results],
            'message' => '<div class="alert alert-success text-center w-100">' . 
                sprintf(esc_html__('Successfully imported %d new reviews. Total reviews: %d', 'ams-wc-amazon'), 
                    $rating_count, ($existing_count + $rating_count)) . 
            '</div>'
        ];
        
        if ($return_data) {
            return [
                'success' => true,
                'message' => sprintf(esc_html__('Successfully imported %d new reviews', 'ams-wc-amazon'), $rating_count),
                'reviews_imported' => $rating_count,
                'data' => $results
            ];
        }
        
        wp_send_json($success_response);
    }



    /**
     * Recalculate product rating after review deletion
     */
    private function recalculate_product_rating($product_id) {
        // Get remaining approved reviews
        $remaining_reviews = get_comments([
            'post_id' => $product_id,
            'type' => 'review',
            'status' => 'approve'
        ]);

        if (empty($remaining_reviews)) {
            // No reviews left - reset rating
            update_post_meta($product_id, '_wc_average_rating', 0);
            update_post_meta($product_id, '_wc_rating_count', 0);
            update_post_meta($product_id, '_wc_review_count', 0);
            update_post_meta($product_id, '_wc_rating_sum', 0);
        } else {
            // Recalculate rating
            $rating_sum = 0;
            $rating_count = count($remaining_reviews);

            foreach ($remaining_reviews as $review) {
                $rating = get_comment_meta($review->comment_ID, 'rating', true);
                $rating_sum += floatval($rating);
            }

            $average_rating = $rating_sum / $rating_count;

            update_post_meta($product_id, '_wc_average_rating', round($average_rating, 2));
            update_post_meta($product_id, '_wc_rating_count', $rating_count);
            update_post_meta($product_id, '_wc_review_count', $rating_count);
            update_post_meta($product_id, '_wc_rating_sum', $rating_sum);
        }

        // Clear caches
        delete_transient('wc_product_reviews_' . $product_id);
        delete_transient('wc_average_rating_' . $product_id);
        wp_cache_delete($product_id, 'product');
        
        if (function_exists('wc_delete_product_transients')) {
            wc_delete_product_transients($product_id);
        }
    }


    /**
     * Delete single review
     */
    public function delete_single_review() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_delete_review_nonce')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $review_id = intval($_POST['review_id']);
        if (!$review_id) {
            wp_send_json_error(array('message' => __('Invalid review ID.', 'ams-wc-amazon')));
            return;
        }

        // Get the comment to verify it's an imported review
        $comment = get_comment($review_id);
        if (!$comment) {
            wp_send_json_error(array('message' => __('Review not found.', 'ams-wc-amazon')));
            return;
        }

        // Check if it's an imported review
        $review_hash = get_comment_meta($review_id, 'review_hash', true);
        if (!$review_hash) {
            wp_send_json_error(array('message' => __('This is not an imported review.', 'ams-wc-amazon')));
            return;
        }

        $product_id = $comment->comment_post_ID;

        // Delete the review
        $deleted = wp_delete_comment($review_id, true);
        
        if ($deleted) {
            // Recalculate product rating
            $this->recalculate_product_rating($product_id);
            
            wp_send_json_success(array('message' => __('Review deleted successfully.', 'ams-wc-amazon')));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete review.', 'ams-wc-amazon')));
        }
    }





    /**
     * Bulk delete reviews from specific product
     */
    public function bulk_delete_product_reviews() {
        // Security check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_bulk_delete_nonce')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $product_id = intval($_POST['product_id']);
        if (!$product_id) {
            wp_send_json_error(array('message' => __('Invalid product ID.', 'ams-wc-amazon')));
            return;
        }

        global $wpdb;

        try {
            // Get all imported reviews for this specific product
            $product_reviews = $wpdb->get_results($wpdb->prepare("
                SELECT c.comment_ID 
                FROM {$wpdb->comments} c
                INNER JOIN {$wpdb->commentmeta} cm ON c.comment_ID = cm.comment_id
                WHERE c.comment_type = 'review' 
                AND c.comment_post_ID = %d
                AND cm.meta_key = 'review_hash'
            ", $product_id));

            if (empty($product_reviews)) {
                wp_send_json_success(array(
                    'message' => __('No imported reviews found for this product.', 'ams-wc-amazon'),
                    'deleted_count' => 0
                ));
                return;
            }

            $deleted_count = 0;
            $failed_count = 0;

            foreach ($product_reviews as $review) {
                if (wp_delete_comment($review->comment_ID, true)) {
                    $deleted_count++;
                } else {
                    $failed_count++;

                }
            }

            // Recalculate product rating
            $this->recalculate_product_rating($product_id);


            if ($deleted_count > 0) {
                wp_send_json_success(array(
                    'message' => sprintf(__('Deleted %d reviews from product.', 'ams-wc-amazon'), $deleted_count),
                    'deleted_count' => $deleted_count,
                    'failed_count' => $failed_count
                ));
            } else {
                wp_send_json_error(array(
                    'message' => __('No reviews could be deleted from this product.', 'ams-wc-amazon')
                ));
            }

        } catch (Exception $e) {

            wp_send_json_error(array('message' => __('Error deleting reviews from product.', 'ams-wc-amazon')));
        }
    }
}