<?php
namespace Amazon\Affiliate\Admin;

class LogsPage {
    private $log_file;

    public function __construct() {
        $this->log_file = plugin_dir_path(dirname(__FILE__)) . 'import_verification.log';
        add_action('wp_ajax_ams_logs_pagination', array($this, 'ajax_logs_pagination'));
    }

    public function render_logs_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }

        // Remove WordPress admin footer text from plugin pages
        add_filter('admin_footer_text', '__return_empty_string', 11);
        add_filter('update_footer', '__return_empty_string', 11);

        // Handle clear logs action
        if (isset($_POST['action']) && $_POST['action'] == 'clear_logs') {
            $this->handle_clear_logs_request();
        }

        // Handle export logs action
        if (isset($_POST['action']) && $_POST['action'] == 'export_logs') {
            $this->export_logs();
        }

        $log_contents = $this->get_log_contents();
        include dirname(__FILE__) . '/views/view-logs.php';
    }

    public function get_log_contents() {
        if (!file_exists($this->log_file)) {
            return '';
        }
        
        return file_get_contents($this->log_file);
    }

    public function clear_logs() {
        if (file_exists($this->log_file)) {
            $result = file_put_contents($this->log_file, '');
            if ($result !== false) {
                return true;
            }
        }
        return false;
    }

    private function handle_clear_logs_request() {
        // Simple security check - verify user capability
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to perform this action.'));
        }
        
        $result = $this->clear_logs();
        if ($result) {
            $this->logs_cleared_notice();
        } else {
            $this->logs_clear_failed_notice();
        }
        
        // Redirect to prevent form resubmission
        wp_safe_redirect(add_query_arg('page', 'view-logs', admin_url('admin.php')));
        exit;
    }

    public function logs_cleared_notice() {
        add_settings_error('ams_logs', 'logs_cleared', __('Logs have been cleared successfully.', 'ams-wc-amazon'), 'success');
    }

    public function logs_clear_failed_notice() {
        add_settings_error('ams_logs', 'logs_clear_failed', __('Failed to clear logs. Please check file permissions.', 'ams-wc-amazon'), 'error');
    }

    public function logs_file_not_found_notice() {
        add_settings_error('ams_logs', 'logs_file_not_found', __('Log file not found. It may have been already deleted.', 'ams-wc-amazon'), 'warning');
    }

    private function export_logs() {
        // Simple security check - verify user capability
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to perform this action.'));
        }

        if (!file_exists($this->log_file)) {
            wp_die(__('Log file not found.', 'ams-wc-amazon'));
        }

        $log_contents = file_get_contents($this->log_file);

        if ($log_contents === false) {
            wp_die(__('Failed to read log file.', 'ams-wc-amazon'));
        }

        $filename = 'import_verification_logs_' . date('Y-m-d_H-i-s') . '.txt';

        // Disable output buffering
        if (ob_get_level()) {
            ob_end_clean();
        }

        // Set headers for file download
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($log_contents));
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Output file contents
        echo $log_contents;
        exit;
    }

    public function paginate_logs($log_contents, $page, $per_page) {
        $lines = explode("\n", $log_contents);
        $offset = ($page - 1) * $per_page;
        return implode("\n", array_slice($lines, $offset, $per_page));
    }

    public function filter_logs($log_contents, $filter) {
        if (empty($filter)) return $log_contents;
        $lines = explode("\n", $log_contents);
        $filtered = array_filter($lines, function($line) use ($filter) {
            return stripos($line, $filter) !== false;
        });
        return implode("\n", $filtered);
    }

    public function ajax_logs_pagination() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied.', 'ams-wc-amazon'));
        }
        
        $filter = isset($_POST['filter']) ? sanitize_text_field($_POST['filter']) : '';
        $paged = isset($_POST['paged']) ? intval($_POST['paged']) : 1;
        
        $logs_per_page = 100;
        $log_contents = $this->get_log_contents();
        $filtered_logs = $this->filter_logs($log_contents, $filter);
        $paginated_logs = $this->paginate_logs($filtered_logs, $paged, $logs_per_page);
        $total_pages = ceil(substr_count($filtered_logs, "\n") / $logs_per_page);
        
        ob_start();
        $lines = explode("\n", $paginated_logs);
        foreach ($lines as $i => $line) {
            $line_num = $logs_per_page * ($paged - 1) + $i + 1;
            $highlight_class = '';
            if (stripos($line, 'error') !== false) $highlight_class = 'text-danger fw-semibold';
            elseif (stripos($line, 'success') !== false) $highlight_class = 'text-success fw-semibold';
            elseif (stripos($line, 'warning') !== false) $highlight_class = 'text-warning fw-semibold';
            elseif (stripos($line, 'info') !== false) $highlight_class = 'text-info fw-semibold';
            echo '<span class="text-muted">' . str_pad($line_num, 4, ' ', STR_PAD_LEFT) . ' | </span>';
            echo '<span class="' . $highlight_class . '">' . esc_html($line) . "</span>\n";
        }
        $log_html = ob_get_clean();
        
        // Pagination links
        ob_start();
        if ($total_pages > 1) {
            $pagination_args = array(
                'base' => '#',
                'format' => '',
                'total' => $total_pages,
                'current' => $paged,
                'show_all' => false,
                'end_size' => 1,
                'mid_size' => 2,
                'prev_next' => true,
                'prev_text' => __('‹'),
                'next_text' => __('›'),
                'type' => 'array',
            );
            $page_links = paginate_links($pagination_args);
            if (is_array($page_links)) {
                echo '<ul class="pagination pagination-sm mb-0">';
                foreach ($page_links as $link) {
                    if (strpos($link, 'current') !== false) {
                        echo '<li class="page-item active"><span class="page-link">' . strip_tags($link) . '</span></li>';
                    } else {
                        // Extract page number from the link
                        $page_num = 1;
                        if (preg_match('/paged=(\d+)/', $link, $matches)) {
                            $page_num = $matches[1];
                        } elseif (preg_match('/>(\d+)</', $link, $matches)) {
                            $page_num = $matches[1];
                        } elseif (strpos($link, '‹') !== false) {
                            $page_num = max(1, $paged - 1);
                        } elseif (strpos($link, '›') !== false) {
                            $page_num = min($total_pages, $paged + 1);
                        }
                        
                        $text = strip_tags($link);
                        echo '<li class="page-item"><a class="page-link ams-logs-page-link" href="#" data-page="' . $page_num . '">' . $text . '</a></li>';
                    }
                }
                echo '</ul>';
            }
        }
        $pagination_html = ob_get_clean();
        
        wp_send_json_success([
            'log_html' => $log_html,
            'pagination_html' => $pagination_html,
            'total_entries' => strlen($filtered_logs) ? substr_count($filtered_logs, "\n") : 0,
        ]);
    }
}