<?php
namespace Amazon\Affiliate\Admin;

/**
 * Class LicenseActivation
 *
 * @package Amazon\Affiliate\Admin
 */
class LicenseActivation {
    private $api_url = 'https://buy.affiliatepro.org/';
    private $product_id = '29955429';

    /**
     * License activation handler
     */
    public function license_activation() {
        $url = home_url('/');
        $purchase_code = sanitize_text_field($_POST['purchase_code']);

        $response = $this->call_api($this->api_url . 'license/activate', [
            'purchase_code' => $purchase_code,
            'url' => $url,
            'product_id' => $this->product_id,
        ]);

        $this->process_response($response);
    }

    /**
     * Deactivate license plugin
     */
    public function deactivated_license_plugin() {
        // Check for nonce security
        $nonce = sanitize_text_field($_POST['nonce']);
        if (!wp_verify_nonce($nonce, 'ams_de_activated')) {
            echo esc_html__('Busted!', 'ams-wc-amazon');
            wp_die();
        }

        $purchase_code = get_option('ams_activated_license');
        $url = home_url('/');

        $response = $this->call_api($this->api_url . 'license/deactivate', [
            'purchase_code' => $purchase_code,
            'url' => $url,
            'product_id' => $this->product_id,
        ]);

        $this->process_response($response, true);
    }

    /**
     * Call API helper function
     */
    private function call_api($url, $body, $method = 'POST') {
        $args = [
            'body' => $body,
            'timeout' => 30,
        ];

        if ($method === 'GET') {
            $url = add_query_arg($body, $url);
            $response = wp_remote_get($url, $args);
        } else {
            $response = wp_remote_post($url, $args);
        }

        if (is_wp_error($response)) {
            return false;
        }

        return json_decode(wp_remote_retrieve_body($response), true);
    }

    /**
     * Process API response
     */
    private function process_response($response, $is_deactivation = false) {
        $info = [];

        if ($response === false) {
            $info['status'] = 'error';
            $info['message'] = 'Unable to connect to the license server.';
        } elseif (isset($response['status']) && strtolower($response['status']) === 'success') {
            if ($is_deactivation) {
                update_option('ams_activated_status', 'failed');
                update_option('ams_activated_license', '');
                $info['license_status'] = '<span class="wca-warning">' . esc_html__('Not Activated', 'ams-wc-amazon') . '</span>';
            } else {
                update_option('ams_activated_status', 'success');
                update_option('ams_activated_license', $_POST['purchase_code']);
                update_option('ams_activated_site', $_SERVER['HTTP_HOST']);
                $info['license_status'] = '<span class="wca-success">' . esc_html__('Activated', 'ams-wc-amazon') . '</span>';
            }
            $info['status'] = 'success';
            $info['message'] = isset($response['message']) ? esc_html($response['message']) : '';
        } else {
            update_option('ams_activated_status', 'failed');
            $info['status'] = 'error';
            $info['license_status'] = '<span class="wca-warning">' . esc_html__('Not Activated', 'ams-wc-amazon') . '</span>';
            $info['message'] = isset($response['message']) ? esc_html($response['message']) : 'Activation failed';
        }

        echo wp_json_encode($info);
        wp_die();
    }
}