<?php
namespace Amazon\Affiliate\Admin;

/**
 * Product import handler class
 *
 * @package Amazon\Affiliate\Admin
 */
class ImportProducts {

    public function create_product_variation($post_id,$data){

        ## ---------------------- VARIATION ATTRIBUTES ---------------------- ##
        $product_attributes = array();
        foreach( $data['attributes'] as $key => $terms ){
            $taxonomy = wc_attribute_taxonomy_name($key); // The taxonomy slug
            $attr_label = ucfirst($key); // attribute label name
            $attr_name = ( wc_sanitize_taxonomy_name($key)); // attribute slug
            // NEW Attributes: Register and save them
            if( ! taxonomy_exists( $taxonomy ) )
                $this->save_product_attribute_from_name( $attr_name, $attr_label );
            $product_attributes[$taxonomy] = array (
                'name'         => $taxonomy,
                'value'        => '',
                'position'     => '',
                'is_visible'   => 0,
                'is_variation' => 1,
                'is_taxonomy'  => 1
            );
            foreach( $terms as $value ){
                $term_name = ucfirst($value);
                $term_slug = sanitize_title($value);
                // Check if the Term name exist and if not we create it.
                if( ! term_exists( $value, $taxonomy ) )
                    wp_insert_term( $term_name, $taxonomy, array('slug' => $term_slug ) ); // Create the term
                // Set attribute values
                wp_set_post_terms( $post_id, $term_name, $taxonomy, true );
            }
        }
        update_post_meta( $post_id, '_product_attributes', $product_attributes );
    }

    public function save_product_attribute_from_name( $name, $label='', $set=true ){
        if( ! function_exists ('get_attribute_id_from_name') ) return;
        global $wpdb;
        $label = $label == '' ? ucfirst($name) : $label;
        $attribute_id = $this->get_attribute_id_from_name( $name );
        if( empty($attribute_id) ){
            $attribute_id = NULL;
        } else {
            $set = false;
        }
        $args = array(
            'attribute_id'      => $attribute_id,
            'attribute_name'    => $name,
            'attribute_label'   => $label,
            'attribute_type'    => 'select',
            'attribute_orderby' => 'menu_order',
            'attribute_public'  => 0,
        );
        if( empty($attribute_id) ) {
            $wpdb->insert(  "{$wpdb->prefix}woocommerce_attribute_taxonomies", $args );
            set_transient( 'wc_attribute_taxonomies', false );
        }
        if( $set ){
            $attributes = wc_get_attribute_taxonomies();
            $args['attribute_id'] = $this->get_attribute_id_from_name( $name );
            $attributes[] = (object) $args;
            //print_r($attributes);
            set_transient( 'wc_attribute_taxonomies', $attributes );
        } else {
            return;
        }
    }

    public function get_attribute_id_from_name( $name ){
        global $wpdb;
        $attribute_id = $wpdb->get_col($wpdb->prepare("SELECT attribute_id
        FROM {$wpdb->prefix}woocommerce_attribute_taxonomies
        WHERE attribute_name LIKE %s", $name));
        return reset($attribute_id);
    }

// API Import
public function product_import() {
    $__import_log_file = plugin_dir_path(__FILE__) . '../import_verification.log';
    try {
        @$product_region = $_POST['region'];
        @$nonce = $_POST['nonce'];
        if(!isset($_POST['is_cron']) || $_POST['is_cron'] != 1){
            if (!wp_verify_nonce( $nonce, 'ams_product_import' ) ) {
                die('Busted!');
            }
        }

        // Check License
        if (ams_plugin_license_status() === false) {
            echo "<script>console.log('Plugin license not activated');</script>";
            $license = sprintf(esc_html__('Activate License!','ams-wc-amazon'));
            echo wp_kses_post($license);
            wp_die();
        }

        $asin = sanitize_text_field( $_POST['asin'] );
        
        // Validate ASIN format
        if (empty($asin) || !preg_match('/^[A-Z0-9]{10}$/', $asin)) {
            $message = esc_html__('Invalid ASIN format. ASIN must be 10 characters long and contain only letters and numbers.', 'ams-wc-amazon');
            echo wp_kses_post($message);
            wp_die();
        }
        $locale = get_option( 'ams_amazon_country' );
        @file_put_contents($__import_log_file, '[' . date('Y-m-d H:i:s') . '] Import Start: func=ImportProducts::product_import ASIN=' . $asin . "\n", FILE_APPEND);
        $regions = ams_get_amazon_regions();
        $marketplace = ams_get_marketplace_value($locale);
        $service_name = 'ProductAdvertisingAPI';
        if (!isset($_POST['is_cron']) || $_POST['is_cron'] != 1) {
            $region = $regions[ $locale ]['RegionCode'];
        } else {
            $region = isset($_POST['region']) ? $_POST['region'] : "";
        }
        $access_key = get_option( 'ams_access_key_id' );
        $secret_key = get_option( 'ams_secret_access_key' );
        
        // Validate API credentials
        if (empty($access_key) || empty($secret_key)) {
            $message = esc_html__('Amazon API credentials are missing. Please configure your Access Key ID and Secret Access Key in the plugin settings.', 'ams-wc-amazon');
            echo wp_kses_post($message);
            wp_die();
        }
        
        $payload_arr = array();
        $payload_arr['ItemIds'] = array( $asin );

        $payload_arr['Resources'] = array( 
            "BrowseNodeInfo.BrowseNodes.Ancestor",
            "ParentASIN", "Images.Primary.Small", "Images.Primary.Medium", "Images.Primary.Large", 
            "Images.Variants.Small", "Images.Variants.Medium", "Images.Variants.Large", 
            "ItemInfo.ByLineInfo", "ItemInfo.ContentInfo", "ItemInfo.ContentRating", 
            "ItemInfo.Classifications", "ItemInfo.ExternalIds", "ItemInfo.Features", 
            "ItemInfo.ManufactureInfo", "ItemInfo.ProductInfo", "ItemInfo.TechnicalInfo", 
            "ItemInfo.Title", "ItemInfo.TradeInInfo", "Offers.Listings.Availability.MaxOrderQuantity", 
            "Offers.Listings.Availability.Message", "Offers.Listings.Availability.MinOrderQuantity", 
            "Offers.Listings.Availability.Type", "Offers.Listings.Condition", 
            "Offers.Listings.Condition.ConditionNote", "Offers.Listings.Condition.SubCondition", 
            "Offers.Listings.DeliveryInfo.IsAmazonFulfilled", "Offers.Listings.DeliveryInfo.IsFreeShippingEligible", 
            "Offers.Listings.DeliveryInfo.IsPrimeEligible", "Offers.Listings.DeliveryInfo.ShippingCharges", 
            "Offers.Listings.IsBuyBoxWinner", "Offers.Listings.LoyaltyPoints.Points", 
            "Offers.Listings.MerchantInfo", "Offers.Listings.Price", 
            "Offers.Listings.ProgramEligibility.IsPrimeExclusive", 
            "Offers.Listings.ProgramEligibility.IsPrimePantry", "Offers.Listings.Promotions", 
            "Offers.Listings.SavingBasis", "Offers.Summaries.HighestPrice", 
            "Offers.Summaries.LowestPrice", "Offers.Summaries.OfferCount"
        );

        $payload_arr['PartnerTag'] = get_option( 'ams_associate_tag' );
        $payload_arr['PartnerType'] = 'Associates';
        $payload_arr['Marketplace'] = $marketplace;
        $payload_arr['Operation'] = 'GetItems';
        $payload = wp_json_encode( $payload_arr );
        $host = $regions[ $locale ]['Host'];
        $uri_path = "/paapi5/getitems";
        
        $cache_key = 'ams_product_' . md5($asin . $region . $payload);
        $cached_response = get_transient($cache_key);
        if ($cached_response !== false && !isset($_POST['is_cron'])) {
            $response = $cached_response;
        } else {
        $api = new \Amazon\Affiliate\Api\Amazon_Product_Api ( $access_key, $secret_key, $region, $service_name, $uri_path, $payload, $host, 'GetItems' );
        $response = $api->do_request();
        

        
            if (isset($response->ItemsResult) && isset($response->ItemsResult->Items)) {
                set_transient($cache_key, $response, 1800);
            }
        }

        // Check if ItemsResult is set and not null before accessing Items
        if (isset($response->ItemsResult) && isset($response->ItemsResult->Items)) {
            $results = $response->ItemsResult->Items;
        } else {
            $results = null;
        }

        if (empty($results)) {
            sleep(2);
            $apiRetried = new \Amazon\Affiliate\Api\Amazon_Product_Api($access_key, $secret_key, $region, $service_name, $uri_path, $payload, $host, 'GetItems');
            $response = $apiRetried->do_request();
        
            // Check if ItemsResult is set and not null before accessing Items
            if (isset($response->ItemsResult) && isset($response->ItemsResult->Items)) {
                $results = $response->ItemsResult->Items;
            } else {
                $results = null;
            }
        
            if (empty($results)) {
                // Make sure $post_id is defined before using it
                if (isset($post_id)) {
                    update_post_meta($post_id, 'ams_last_cron_update', date('Y-m-d H:i:s'));
                    update_post_meta($post_id, 'ams_last_cron_status', 1);
                }
        
                if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                    $message = esc_html__('API request failed during cron job. Please check your Amazon API credentials (Access Key, Secret Key, Associate Tag) and ensure they are valid for the selected region.', 'ams-wc-amazon');
                    echo wp_kses_post($message);
                } else {
                    $message = esc_html__('Product import failed. Please verify your Amazon API credentials and try again. If the problem persists, check if the ASIN exists in the selected Amazon region.', 'ams-wc-amazon');
                    echo wp_kses_post($message);
                }
                wp_die();
            }
        }

        $ams_all_asin = ams_get_all_products_info_with_parent();
        foreach ($results as $row) {
            $thumbnail_size  = get_option('ams_product_thumbnail_size');
            $asin            = $row->ASIN;
            $parentASIN      = isset($row->ParentASIN) ? $row->ParentASIN : null;
            $detail_page_url = $row->DetailPageURL;

            // If ParentASIN is null, we treat this as a standalone product
            $parentASIN = $parentASIN !== null ? $parentASIN : $asin;

            if (!isset($_POST['prod_id'])) {
                if (!empty($ams_all_asin) && !empty($ams_all_asin['asin'])) {
                    if ($parentASIN !== null && in_array($parentASIN, $ams_all_asin['asin'])) {
                        $message = esc_html__('Parent ASIN already imported', 'ams-wc-amazon');
                        echo wp_kses_post($message);
                        wp_die();
                    } elseif (in_array($asin, $ams_all_asin['asin'])) {
                        $message = esc_html__('ASIN already imported', 'ams-wc-amazon');
                        echo wp_kses_post($message);
                        wp_die();
                    }
                }
            }

            // Check if specific thumbnail size exists, otherwise get the first key
            if(isset($row->Images->Primary->{$thumbnail_size})) {
                $image = $row->Images->Primary->{$thumbnail_size}->URL;
            } else {
                $primaryImages = (array) $row->Images->Primary;
                $key = key($primaryImages);
                $image = $row->Images->Primary->{$key}->URL;
            }
        
            $gallery = isset($row->Images->Variants) ? $row->Images->Variants : [];
        
            // Safe check for Amount
            if (isset($row->Offers->Listings[0]->Price->Amount)) {
                $amount = $row->Offers->Listings[0]->Price->Amount;
            } else {
                $amount = null; // or some default value
            }
        
            // Safe check for SavingBasis Amount
            if (isset($row->Offers->Listings[0]->SavingBasis) && isset($row->Offers->Listings[0]->SavingBasis->Amount)) {
                $saving_amount = $row->Offers->Listings[0]->SavingBasis->Amount;
            } else {
                $saving_amount = null; // or some default value
            }
        
            // Safe check for DisplayAmount
            if (isset($row->Offers->Listings[0]->Price->DisplayAmount)) {
                $DisplayAmount = $row->Offers->Listings[0]->Price->DisplayAmount;
            } else {
                $DisplayAmount = null; // or some default value
            }
        
            // Safe check for SavingBasis Amount
            if (isset($row->Offers->Listings[0]->SavingBasis) && isset($row->Offers->Listings[0]->SavingBasis->Amount)) {
                $saving_amount = $row->Offers->Listings[0]->SavingBasis->Amount;
            } else {
                $saving_amount = null; // or some default value
            }

            $product_status = isset($row->Offers->Listings[0]->Availability->Message) ? $row->Offers->Listings[0]->Availability->Message : '';
            $product_status = !empty($product_status) ? 'instock' : 'outofstock';
            $title = $row->ItemInfo->Title->DisplayValue;



            // Safe check for Features DisplayValues
            if (isset($row->ItemInfo->Features) && isset($row->ItemInfo->Features->DisplayValues)) {
                $features = $row->ItemInfo->Features->DisplayValues;
            } else {
                $features = []; // Assign an empty array or some default value if Features or DisplayValues are not set
            }

            // Import Product Faster if setting is Enabled
            if ((!isset($_POST['prod_id']) && empty($_POST['prod_id'])) && 'Yes' == get_option('ams_fast_product_importer')) {
                $productData = [];
                $productData['asin'] = $asin;
                $productData['title'] = $title;
                $productData['region'] = $region;
                $productData['parentSku'] = $parentASIN;
                $productData['product_url'] = $detail_page_url;
                $productData['import_method'] = '1'; // Set manually
                $default_message = '<span class="dashicons dashicons-saved"></span> ' . esc_html__('Success', 'ams-wc-amazon');
                $productData['default_message'] = $default_message; // Set manually
                $productData['attributes'] = [];
                $productData['sale_price'] = $amount;
                $productData['regular_price'] = !empty($saving_amount) ? $saving_amount : $amount;
                
                // FIXED: Add product description to Fast Import
                $fast_content = '';
                

                
                // Add Features if available - try both sources
                $features_to_use = [];
                if (isset($features) && is_array($features) && !empty($features)) {
                    $features_to_use = $features;
                } elseif (isset($row->ItemInfo->Features->DisplayValues) && is_array($row->ItemInfo->Features->DisplayValues)) {
                    $features_to_use = $row->ItemInfo->Features->DisplayValues;
                }
                
                if (!empty($features_to_use)) {
                    foreach ($features_to_use as $feature) {
                        $fast_content .= '<ul><li>' . $feature . '</li></ul>';
                    }
                }
                
                // Add basic product info if no features
                if (empty($fast_content)) {
                    $basic_info = [];
                    
                    // Add brand if available
                    if (isset($row->ItemInfo->ByLineInfo->Brand->DisplayValue)) {
                        $basic_info[] = '<p><strong>Brand:</strong> ' . $row->ItemInfo->ByLineInfo->Brand->DisplayValue . '</p>';
                    }
                    
                    // Add model if available
                    if (isset($row->ItemInfo->ManufactureInfo->Model->DisplayValue)) {
                        $basic_info[] = '<p><strong>Model:</strong> ' . $row->ItemInfo->ManufactureInfo->Model->DisplayValue . '</p>';
                    }
                    
                    // Add color if available
                    if (isset($row->ItemInfo->ProductInfo->Color->DisplayValue)) {
                        $basic_info[] = '<p><strong>Color:</strong> ' . $row->ItemInfo->ProductInfo->Color->DisplayValue . '</p>';
                    }
                    
                    // Add size if available
                    if (isset($row->ItemInfo->ProductInfo->Size->DisplayValue)) {
                        $basic_info[] = '<p><strong>Size:</strong> ' . $row->ItemInfo->ProductInfo->Size->DisplayValue . '</p>';
                    }
                    
                    if (!empty($basic_info)) {
                        $fast_content = implode('', $basic_info);
                    }
                }
                
                // Set fallback if still empty
                if (empty($fast_content)) {
                    $fast_content = '<p>This product is available on Amazon. Please check the product page for detailed specifications and features.</p>';
                }
                
                $productData['content'] = $fast_content;
                
                /**
                 * Import Product Faster
                 * 
                 * @param array $productData
                 * 
                 * @return string
                 */
                advancedProductImporter($productData);
                wp_die();
            }

            $payload_arr2 = array();
            $payload_arr2['ASIN'] = $asin;
            $payload_arr2['Resources'] = array("ParentASIN", "ItemInfo.Title", "Offers.Listings.Price", "Offers.Listings.ProgramEligibility.IsPrimeExclusive", "Offers.Listings.ProgramEligibility.IsPrimePantry", "Offers.Listings.Promotions", "Offers.Listings.SavingBasis", "Offers.Listings.Availability.Message", "Offers.Summaries.HighestPrice", "Offers.Summaries.LowestPrice", "VariationSummary.Price.HighestPrice", "VariationSummary.Price.LowestPrice", "VariationSummary.VariationDimension", "Images.Primary.Small", "Images.Primary.Medium", "Images.Primary.Large", "Images.Variants.Small", "Images.Variants.Medium", "Images.Variants.Large");
            $payload_arr2['PartnerTag'] = get_option('ams_associate_tag');
            $payload_arr2['PartnerType'] = 'Associates';
            $payload_arr2['Marketplace'] = $marketplace;
            $payload_arr2['Operation'] = 'GetVariations';
            $payload2 = json_encode($payload_arr2);
            $host = $regions[$locale]['Host'];
            $uri_path = "/paapi5/getvariations";
            $api2 = new \Amazon\Affiliate\Api\Amazon_Product_Api($access_key, $secret_key, $region, $service_name, $uri_path, $payload2, $host, 'getVariation');
            $response2 = $api2->do_request();

            $variations = isset($response2->VariationsResult->VariationSummary) ? $response2->VariationsResult->VariationSummary : null;
            $attributes = isset($response2->VariationsResult->Items) ? $response2->VariationsResult->Items : [];
            $VariationPage = 2;
            $Variationlist = [];

            if (isset($variations->PageCount) && $variations->PageCount >= 1) {
                foreach ($response2->VariationsResult->Items as $item) {
                    $VariationAttribute = [];
                    foreach ($item->VariationAttributes as $ItemVariationAttribute) {
                        $VariationAttribute[$ItemVariationAttribute->Name] = trim($ItemVariationAttribute->Value);
                    }

                    // Safe check for Amount
                    $amount = isset($item->Offers->Listings[0]->Price->Amount) ? $item->Offers->Listings[0]->Price->Amount : 0;

                    // Safe check for SavingBasis Amount
                    $saving_amount = isset($item->Offers->Listings[0]->SavingBasis->Amount) ? $item->Offers->Listings[0]->SavingBasis->Amount : 0;

                    // Stock status
                    $product_stock = isset($item->Offers->Listings[0]->Availability->Message) ? $item->Offers->Listings[0]->Availability->Message : '';
                    $stock_status = !empty($product_stock) ? 'instock' : 'outofstock';

                    if (empty($saving_amount)) {
                        $sale_price = $amount;
                        $regular_price = $amount;
                    } else {
                        $sale_price = $amount;
                        $regular_price = $saving_amount;
                    }

                    // v_gallery loop
                    $v_gallery = [];
                    // Check for variant images and add them to the gallery
                    if (isset($item->Images->Variants)) {
                        foreach ($item->Images->Variants as $variant_image) {
                            if (isset($variant_image->Large->URL)) {
                                $v_gallery[] = $variant_image->Large->URL;
                            }
                        }
                    }
                    // Fallback to primary image if no variant images are found
                    if (empty($v_gallery) && isset($item->Images->Primary->Large->URL)) {
                        $v_gallery[] = $item->Images->Primary->Large->URL;
                    }
                    // v_gallery loop

                    $Variationlist[] = array(
                        'post_title' => $item->ItemInfo->Title->DisplayValue,
                        'attributes' => $VariationAttribute,
                        'sku' => $item->ASIN,
                        'regular_price' => floatval($regular_price),
                        'sale_price' => floatval($sale_price),
                        'stock_status' => $stock_status,
                        'product_image_gallery' => $v_gallery,
                    );
                }
                while ($VariationPage <= $variations->PageCount) {
                    $payload_arr2['VariationPage'] = $VariationPage;
                    $payload3 = json_encode($payload_arr2);
                    $api3 = new \Amazon\Affiliate\Api\Amazon_Product_Api(
                        $access_key, $secret_key, $region, $service_name, $uri_path, $payload3, $host, 'getVariation'
                    );
                    $response3 = $api3->do_request();

                    foreach (isset($response3->VariationsResult->Items) ? $response3->VariationsResult->Items : [] as $item) {
                        $VariationAttribute = [];
                        foreach ($item->VariationAttributes as $ItemVariationAttribute) {
                            $VariationAttribute[$ItemVariationAttribute->Name] = trim($ItemVariationAttribute->Value);
                        }
                        
                        $amount = isset($item->Offers->Listings[0]->Price->Amount) 
                            ? $item->Offers->Listings[0]->Price->Amount 
                            : 0;
                        
                        $saving_amount = isset($item->Offers->Listings[0]->SavingBasis->Amount) 
                            ? $item->Offers->Listings[0]->SavingBasis->Amount 
                            : 0;

                        $product_stock = isset($item->Offers->Listings[0]->Availability->Message) 
                            ? $item->Offers->Listings[0]->Availability->Message 
                            : '';
                        
                        $stock_status = !empty($product_stock) ? 'instock' : 'outofstock';

                        if ($amount == 0 && $saving_amount == 0) {
                            $sale_price = 0;
                            $regular_price = 0;
                        } elseif ($saving_amount == 0 || $amount >= $saving_amount) {
                            $sale_price = $amount;
                            $regular_price = $amount;
                        } else {
                            $sale_price = $amount;
                            $regular_price = $saving_amount;
                        }

                        // v_gallery loop
                        $v_gallery = [];
                        // Check for variant images and add them to the gallery
                        if (isset($item->Images->Variants)) {
                            foreach ($item->Images->Variants as $variant_image) {
                                if (isset($variant_image->Large->URL)) {
                                    $v_gallery[] = $variant_image->Large->URL;
                                }
                            }
                        }
                        // Fallback to primary image if no variant images are found
                        if (empty($v_gallery) && isset($item->Images->Primary->Large->URL)) {
                            $v_gallery[] = $item->Images->Primary->Large->URL;
                        }
                        // v_gallery loop

                        $Variationlist[] = array(
                            'post_title' => $item->ItemInfo->Title->DisplayValue,
                            'attributes' => $VariationAttribute,
                            'sku' => $item->ASIN,
                            'regular_price' => floatval($regular_price),
                            'sale_price' => floatval($sale_price),
                            'stock_status' => $stock_status,
                            'product_image_gallery' => $v_gallery,
                        );
                    }
                    $VariationPage++;
                }
            }


            // Product Description
            $content = '';
            // Step 1: Populate content from `$features` array, if available
            if (isset($features) && is_array($features)) {
                foreach ($features as $feature) {
                    $content .= '<ul><li>' . $feature . '</li></ul>';
                }
            }
            // Step 2: If content is empty, check cron settings and other API fields
            if (empty($content)) {
                $run_content = 1;
                if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1 && get_option('product_description_cron') != 1) {
                    $run_content = 0;
                }



                // Step 3: If cron settings allow, continue populating content
                if ($run_content) {
                    $content = '';

                    // Method 1: Populate content with `Features.DisplayValues` if available
                    if (isset($row->ItemInfo->Features->DisplayValues) && is_array($row->ItemInfo->Features->DisplayValues)) {
                        foreach ($row->ItemInfo->Features->DisplayValues as $feature) {
                            $content .= '<ul><li>' . $feature . '</li></ul>';
                        }
                    }

                    // Method 2: Try Product Description from `ProductInfo.ProductDescription`
                    if (empty($content) && isset($row->ItemInfo->ProductInfo->ProductDescription)) {
                        $content .= '<p>' . $row->ItemInfo->ProductInfo->ProductDescription . '</p>';
                    }

                    // Method 3: Try ContentInfo fields
                    if (empty($content) && isset($row->ItemInfo->ContentInfo)) {
                        if (isset($row->ItemInfo->ContentInfo->Edition->DisplayValue)) {
                            $content .= '<p><strong>Edition:</strong> ' . $row->ItemInfo->ContentInfo->Edition->DisplayValue . '</p>';
                        }
                        if (isset($row->ItemInfo->ContentInfo->PagesCount->DisplayValue)) {
                            $content .= '<p><strong>Pages:</strong> ' . $row->ItemInfo->ContentInfo->PagesCount->DisplayValue . '</p>';
                        }
                        if (isset($row->ItemInfo->ContentInfo->PublicationDate->DisplayValue)) {
                            $content .= '<p><strong>Publication Date:</strong> ' . $row->ItemInfo->ContentInfo->PublicationDate->DisplayValue . '</p>';
                        }
                    }

                    // Method 4: Try TechnicalInfo fields
                    if (empty($content) && isset($row->ItemInfo->TechnicalInfo)) {
                        if (isset($row->ItemInfo->TechnicalInfo->Formats->DisplayValues) && is_array($row->ItemInfo->TechnicalInfo->Formats->DisplayValues)) {
                            $content .= '<p><strong>Formats:</strong> ' . implode(', ', $row->ItemInfo->TechnicalInfo->Formats->DisplayValues) . '</p>';
                        }
                    }

                    // Method 5: Try ManufactureInfo
                    if (empty($content) && isset($row->ItemInfo->ManufactureInfo)) {
                        if (isset($row->ItemInfo->ManufactureInfo->ItemPartNumber->DisplayValue)) {
                            $content .= '<p><strong>Part Number:</strong> ' . $row->ItemInfo->ManufactureInfo->ItemPartNumber->DisplayValue . '</p>';
                        }
                        if (isset($row->ItemInfo->ManufactureInfo->Model->DisplayValue)) {
                            $content .= '<p><strong>Model:</strong> ' . $row->ItemInfo->ManufactureInfo->Model->DisplayValue . '</p>';
                        }
                        if (isset($row->ItemInfo->ManufactureInfo->Warranty->DisplayValue)) {
                            $content .= '<p><strong>Warranty:</strong> ' . $row->ItemInfo->ManufactureInfo->Warranty->DisplayValue . '</p>';
                        }
                    }

                    // Method 6: Create enhanced content from available product info
                    if (empty($content) && isset($row->ItemInfo->Title->DisplayValue)) {
                        $enhanced_info = [];
                        
                        // Create a more descriptive introduction
                        $title = $row->ItemInfo->Title->DisplayValue;
                        $enhanced_info[] = '<p>This is the <strong>' . $title . '</strong> available on Amazon.</p>';
                        
                        // Add brand if available
                        if (isset($row->ItemInfo->ByLineInfo->Brand->DisplayValue)) {
                            $brand = $row->ItemInfo->ByLineInfo->Brand->DisplayValue;
                            $enhanced_info[] = '<p><strong>Brand:</strong> ' . $brand . '</p>';
                        }
                        
                        // Add manufacturer if available and different from brand
                        if (isset($row->ItemInfo->ByLineInfo->Manufacturer->DisplayValue)) {
                            $manufacturer = $row->ItemInfo->ByLineInfo->Manufacturer->DisplayValue;
                            if (!isset($brand) || $manufacturer !== $brand) {
                                $enhanced_info[] = '<p><strong>Manufacturer:</strong> ' . $manufacturer . '</p>';
                            }
                        }
                        
                        // Add product classification info
                        if (isset($row->ItemInfo->Classifications->ProductGroup->DisplayValue)) {
                            $enhanced_info[] = '<p><strong>Product Category:</strong> ' . $row->ItemInfo->Classifications->ProductGroup->DisplayValue . '</p>';
                        }
                        
                        if (isset($row->ItemInfo->Classifications->Binding->DisplayValue)) {
                            $enhanced_info[] = '<p><strong>Type:</strong> ' . $row->ItemInfo->Classifications->Binding->DisplayValue . '</p>';
                        }
                        
                        // Add a call-to-action
                        $enhanced_info[] = '<p>For complete product specifications, features, and customer reviews, please visit the product page on Amazon.</p>';
                        
                        if (!empty($enhanced_info)) {
                            $content = implode('', $enhanced_info);
                        }
                    }

                    // Final fallback: Set default message if content is still empty
                    if (empty($content)) {
                        $content = '<p>This product is available on Amazon. Please check the product page for detailed specifications and features.</p>';
                    }
                    

                }
            }
            $user_id = get_current_user();
            if(isset($_POST['prod_id']) && !empty($_POST['prod_id'])){
                $post_id = $_POST['prod_id'];
                $postData= array('ID' => $post_id);

                $run_content = 1;
                if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                    if (get_option('product_description_cron') != 1) {
                        $run_content = 0;
                    }
                    
                    if( 1 == get_option('product_name_cron', true) ) {
                        $postData['post_title'] = stripslashes($title);
                        $postData['post_name'] = sanitize_title( $title );
                    }
                }

                if(!empty($run_content)) {
                     $postData['post_content'] = $content;
                }
                
                wp_update_post($postData); 
            }else{
                // Get status settings
                $importStatus = get_option( 'ams_product_import_status', true );

                $post_id = wp_insert_post(array(
                    'post_author'  => $user_id,
                    'post_title'   => stripslashes($title),
                    'post_content' => $content,
                    'post_status'  => $importStatus,
                    'post_type'    => "product",
                    'post_parent'  => '',
                ));
            }

            if(!isset($variations->VariationDimensions) || empty($variations->VariationDimensions)){
                wp_set_object_terms( $post_id, 'simple', 'product_type');
                $product = wc_get_product( $post_id );
                $product->save(); // Update
                if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                    if (get_option('product_sku_cron') == 1) {
                        update_post_meta( $post_id, '_sku', $asin );
                    }
                }else {
                    update_post_meta( $post_id, '_sku', $asin );
                }
            } else {
                wp_set_object_terms( $post_id, 'variable', 'product_type');
                $product = wc_get_product( $post_id );
                $product->save(); // Update
                if($parentASIN){
                    if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                        if (get_option('product_sku_cron') == 1) {
                            update_post_meta( $post_id, '_sku', $parentASIN );
                        }
                    }else {
                        update_post_meta( $post_id, '_sku', $parentASIN );
                    }
                }
            }

            // Import/Update categories
            if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
               // Category cron
               if (get_option('product_category_cron', true) == 1) {
                   // Get default category setting
                   $ams_default_category = get_option('ams_default_category');
                   
                   if (!empty($ams_default_category)) {
                       if ($ams_default_category == '_auto_import_amazon') {
                           $product_category = '';

                           // Check if BrowseNodes exist in the API response
                           if (isset($row->BrowseNodeInfo->BrowseNodes)) {
                               $category_paths = [];

                               // Fetch the dynamically configured min and max depth from the settings
                               $min_depth = intval(get_option('ams_category_min_depth', 1));
                               $max_depth = intval(get_option('ams_category_max_depth', 6));

                               $browse_nodes = $row->BrowseNodeInfo->BrowseNodes;

                               // Loop through each BrowseNode to gather category path and ancestors
                               foreach ($browse_nodes as $node) {
                                   $current_node = $node;
                                   $node_parts = [];

                                   // Traverse the ancestors to get the full category hierarchy for this node
                                   while (isset($current_node->Ancestor)) {
                                       array_unshift($node_parts, $current_node->Ancestor->DisplayName);
                                       $current_node = $current_node->Ancestor;
                                   }

                                   array_push($node_parts, $node->DisplayName);

                                   $cleaned_path = array_filter($node_parts, function ($category) {
                                       return !in_array($category, ['Categories', 'Home', 'Root']);
                                   });

                                   // Ensure we get paths between min_depth and max_depth
                                   if (count($cleaned_path) >= $min_depth) {
                                       $trimmed_path = array_slice($cleaned_path, 0, $max_depth);
                                       $category_paths[] = implode(' > ', $trimmed_path);
                                   }
                               }

                               $product_category = !empty($category_paths) ? $category_paths[0] : '';
                           }

                           if (empty($product_category)) {
                               if (isset($row->ItemInfo->Classifications->Binding->DisplayValue) &&
                                   strtolower($row->ItemInfo->Classifications->Binding->DisplayValue) !== 'unknown binding') {
                                   $product_category = $row->ItemInfo->Classifications->Binding->DisplayValue;
                               } elseif (isset($row->ItemInfo->Classifications->ProductGroup->DisplayValue)) {
                                   $product_category = $row->ItemInfo->Classifications->ProductGroup->DisplayValue;
                               }
                           }

                           if (empty($product_category)) {
                               $product_category = 'Uncategorized';
                               // logImportVerification('No valid category found, setting to Uncategorized');
                           }

                           if (!empty($product_category)) {
                               $categories = explode(' > ', $product_category);
                               $parent_id = 0;
                               $category_ids = [];

                               foreach ($categories as $category_name) {
                                   $category_name = trim($category_name);

                                   // Skip invalid UUID-like categories
                                   if (preg_match('/[a-f0-9\-]{36,}/', $category_name)) {
                                       continue;
                                   }

                                   // Check if the category exists under the parent
                                   $existing_category = term_exists($category_name, 'product_cat', $parent_id);

                                   if (!$existing_category) {
                                       // Create the category if it doesn't exist, with the correct parent
                                       $new_category = wp_insert_term($category_name, 'product_cat', array(
                                           'description' => $category_name,
                                           'parent' => $parent_id
                                       ));
                                       if (!is_wp_error($new_category)) {
                                           $parent_id = $new_category['term_id'];
                                           $category_ids[] = $parent_id;
                                           // logImportVerification('Created new category: ' . $category_name);
                                       }
                                   } else {
                                       // If the category already exists, fetch its ID
                                       $term = get_term_by('name', esc_attr($category_name), 'product_cat');
                                       if ($term) {
                                           $parent_id = $term->term_id;
                                           $category_ids[] = $parent_id;
                                       } else {
                                           $parent_id = $existing_category['term_id'];
                                           $category_ids[] = $parent_id;
                                       }
                                   }
                               }

                               // Assign ALL categories in the hierarchy to the product
                               if (!empty($category_ids)) {
                                   update_post_meta($post_id, '_product_categories_hierarchy', $product_category);
                                   wp_set_object_terms($post_id, $category_ids, 'product_cat');
                                   // logImportVerification('Updated product categories: ' . $product_category);
                               }
                           }
                       } else {
                           // Manual category assignment
                           $term = null;
                           if (is_numeric($ams_default_category)) {
                               $term = get_term($ams_default_category, 'product_cat');
                           }
                           if (!$term) {
                               $term = get_term_by('id', $ams_default_category, 'product_cat');
                           }
                           if (!$term) {
                               $term = get_term_by('slug', $ams_default_category, 'product_cat');
                           }
                           if (!$term) {
                               $term = get_term_by('name', $ams_default_category, 'product_cat');
                           }

                           // Set the category
                           if ($term && !is_wp_error($term)) {
                               wp_set_object_terms($post_id, $term->term_id, 'product_cat');
                               // logImportVerification('Category Updated: ' . $term->name);
                           } else {
                               wp_set_object_terms($post_id, 'Uncategorized', 'product_cat');
                               // logImportVerification('Category set to Uncategorized (no valid term found)');
                           }
                       }
                   }
               }
            } else {
               // Initial import category handling
               $ams_default_category = get_option('ams_default_category');
               if (!empty($ams_default_category)) {
                   if ($ams_default_category == '_auto_import_amazon') {
                       $product_category = '';

                       // Check if BrowseNodes exist in the API response
                       if (isset($row->BrowseNodeInfo->BrowseNodes)) {
                           $category_paths = [];

                           // Fetch the dynamically configured min and max depth from the settings
                           $min_depth = intval(get_option('ams_category_min_depth', 1));
                           $max_depth = intval(get_option('ams_category_max_depth', 6));

                           $browse_nodes = $row->BrowseNodeInfo->BrowseNodes;

                           foreach ($browse_nodes as $node) {
                               $current_node = $node;
                               $node_parts = [];

                               while (isset($current_node->Ancestor)) {
                                   array_unshift($node_parts, $current_node->Ancestor->DisplayName);
                                   $current_node = $current_node->Ancestor;
                               }

                               array_push($node_parts, $node->DisplayName);

                               $cleaned_path = array_filter($node_parts, function ($category) {
                                   return !in_array($category, ['Categories', 'Home', 'Root']);
                               });

                               if (count($cleaned_path) >= $min_depth) {
                                   $trimmed_path = array_slice($cleaned_path, 0, $max_depth);
                                   $category_paths[] = implode(' > ', $trimmed_path);
                               }
                           }

                           $product_category = !empty($category_paths) ? $category_paths[0] : '';
                       }

                       if (empty($product_category)) {
                           if (isset($row->ItemInfo->Classifications->Binding->DisplayValue) &&
                               strtolower($row->ItemInfo->Classifications->Binding->DisplayValue) !== 'unknown binding') {
                               $product_category = $row->ItemInfo->Classifications->Binding->DisplayValue;
                           } elseif (isset($row->ItemInfo->Classifications->ProductGroup->DisplayValue)) {
                               $product_category = $row->ItemInfo->Classifications->ProductGroup->DisplayValue;
                           }
                       }

                       if (empty($product_category)) {
                           $product_category = 'Uncategorized';
                           // logImportVerification('No valid category found during import, setting to Uncategorized');
                       }

                       if (!empty($product_category)) {
                           $categories = explode(' > ', $product_category);
                           $parent_id = 0;
                           $category_ids = [];

                           foreach ($categories as $category_name) {
                               $category_name = trim($category_name);

                               if (preg_match('/[a-f0-9\-]{36,}/', $category_name)) {
                                   continue;
                               }

                               $existing_category = term_exists($category_name, 'product_cat', $parent_id);

                               if (!$existing_category) {
                                   $new_category = wp_insert_term($category_name, 'product_cat', array(
                                       'description' => $category_name,
                                       'parent' => $parent_id
                                   ));
                                   if (!is_wp_error($new_category)) {
                                       $parent_id = $new_category['term_id'];
                                       $category_ids[] = $parent_id;
                                       // logImportVerification('Created new category during import: ' . $category_name);
                                   }
                               } else {
                                   $term = get_term_by('name', esc_attr($category_name), 'product_cat');
                                   if ($term) {
                                       $parent_id = $term->term_id;
                                       $category_ids[] = $parent_id;
                                   } else {
                                       $parent_id = $existing_category['term_id'];
                                       $category_ids[] = $parent_id;
                                   }
                               }
                           }

                           if (!empty($category_ids)) {
                               update_post_meta($post_id, '_product_categories_hierarchy', $product_category);
                               wp_set_object_terms($post_id, $category_ids, 'product_cat');
                               // logImportVerification('Set initial product categories: ' . $product_category);
                           }
                       }
                   } else {
                       // Manual category assignment during import
                       $term = null;
                       if (is_numeric($ams_default_category)) {
                           $term = get_term($ams_default_category, 'product_cat');
                       }
                       if (!$term) {
                           $term = get_term_by('id', $ams_default_category, 'product_cat');
                       }
                       if (!$term) {
                           $term = get_term_by('slug', $ams_default_category, 'product_cat');
                       }
                       if (!$term) {
                           $term = get_term_by('name', $ams_default_category, 'product_cat');
                       }

                       if ($term && !is_wp_error($term)) {
                           wp_set_object_terms($post_id, $term->term_id, 'product_cat');
                           // logImportVerification('Set initial category: ' . $term->name);
                       } else {
                           wp_set_object_terms($post_id, 'Uncategorized', 'product_cat');
                           // logImportVerification('Set initial category to Uncategorized (no valid term found)');
                       }
                   }
               }
            }


            // Import Brand Name
            if (isset($row->ItemInfo->ByLineInfo->Brand->DisplayValue)) {
                $brandName = trim($row->ItemInfo->ByLineInfo->Brand->DisplayValue);

                // logImportVerification('Brand: ' . $brandName);

                // Check if the brand already exists as a term
                $brandTerm = term_exists($brandName, 'product_brand');

                if (!$brandTerm) {
                    // If the brand doesn't exist, create it
                    $brandTerm = wp_insert_term($brandName, 'product_brand');
                }

                if (is_wp_error($brandTerm)) {
                    // logImportVerification('Error creating brand term: ' . $brandTerm->get_error_message());
                } else {
                    // Get the term ID
                    $brandTermId = isset($brandTerm['term_id']) ? $brandTerm['term_id'] : $brandTerm;

                    // Assign the brand to the product
                    wp_set_object_terms($post_id, intval($brandTermId), 'product_brand');

                    // Store the brand in post meta for further use
                    update_post_meta($post_id, '_product_brand', $brandName);

                    // logImportVerification('Brand assigned to product successfully.');
                }
            } else {
                // logImportVerification('Brand not found in the API response.');
            }
            // Import Brand Name

            update_post_meta( $post_id, '_visibility', 'visible' );
            update_post_meta( $post_id, 'total_sales', '0' );
            update_post_meta( $post_id, '_downloadable', 'no' );
            update_post_meta( $post_id, '_virtual', 'yes' );

            if (!isset($variations->VariationDimensions) || empty($variations->VariationDimensions)) {
                if (empty($saving_amount)) {
                    $price_to_set = $this->parsePriceStringnew($amount);
                    if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                        if (get_option('product_price_cron') == 1) {
                            update_post_meta($post_id, '_regular_price', $price_to_set);
                            update_post_meta($post_id, '_sale_price', $price_to_set);
                            update_post_meta($post_id, '_price', $price_to_set);
                        }
                    } else {
                        update_post_meta($post_id, '_regular_price', $price_to_set);
                        update_post_meta($post_id, '_sale_price', $price_to_set);
                        update_post_meta($post_id, '_price', $price_to_set);
                    }
                } else {
                    $regular_price = $this->parsePriceStringnew($saving_amount);
                    $sale_price = $this->parsePriceStringnew($amount);
                    if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                        if (get_option('product_price_cron') == 1) {
                            update_post_meta($post_id, '_regular_price', $regular_price);
                            update_post_meta($post_id, '_sale_price', $sale_price);
                            update_post_meta($post_id, '_price', $sale_price);
                        }
                    } else {
                        update_post_meta($post_id, '_regular_price', $regular_price);
                        update_post_meta($post_id, '_sale_price', $sale_price);
                        update_post_meta($post_id, '_price', $sale_price);
                    }
                }
            } else {
                wp_set_object_terms( $post_id, 'variable', 'product_type');
            }

            // Get the image and variation limits from settings
            $image_limit = get_option('ams_variation_image_limit', 5);
            $variation_limit = get_option('ams_variation_limit', 5);

            // Import/Update Images
            if ($image || $gallery) {
                // Remove existing images and URLs
                delete_product_images($post_id);
                reset_product_thumbnail_url($post_id, $flag = 0); // Reset featured image URL
                delete_product_gallery_images($post_id);
                reset_product_thumbnail_url($post_id, $flag = 1); // Reset gallery URL

                $gallery_url = [];
                $gallery = is_array($gallery) ? $gallery : [];

                if ('Yes' === get_option('ams_remote_amazon_images')) {
                    // Set featured image URL (remote)
                    attach_product_thumbnail_url_api($post_id, $image, 0);

                    // Process and set gallery images (remote)
                    $gallery_url = [];
                    foreach ($gallery as $img) {
                        if (isset($img->{$thumbnail_size})) {
                            $gallery_url[] = $img->{$thumbnail_size}->URL;
                        } else {
                            // Fallback to any available size if specific size is missing
                            $imageArray = (array) $img;
                            $key = key($imageArray);
                            $gallery_url[] = $img->{$key}->URL;
                        }
                    }

                    if (!empty($gallery_url)) {
                        attach_product_thumbnail_url_api($post_id, $gallery_url, 1);
                    }

                } else {
                    // Download and set the featured image locally
                    attach_product_thumbnail_api($post_id, $image, 0);

                    // Download and set the gallery images locally
                    if (count($gallery) > 0) {
                        foreach ($gallery as $img) {
                            if (isset($img->{$thumbnail_size})) {
                                attach_product_thumbnail_api($post_id, $img->{$thumbnail_size}->URL, 1);
                            } else {
                                $imageArray = (array) $img;
                                $key = key($imageArray);
                                attach_product_thumbnail_api($post_id, $img->{$key}->URL, 1);
                            }
                        }
                    }
                }


            }
            // Import/Update reviews
            if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                // Review cron
                if ('1' == get_option('enable_amazon_review', true) && '1' == get_option('product_review_cron', true)) {
                    // Get product URL
                    $product_url = esc_url_raw($row->DetailPageURL);
                    
                    // Get user agent
                    $user_agent = $this->user_agent();
                    
                    // Get product data
                    $response_body = fetchAndValidateProductData($product_url, $user_agent, false);
                    if (is_string($response_body) && !empty($response_body)) {
                        if (!class_exists('simple_html_dom')) {
                            require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
                        }
                        
                        $html = new \simple_html_dom();
                        $html->load($response_body);
                        // Check for broken page
                        $message = check_for_broken_page($response_body, $html);
                        if ($message === null) {
                            // Get review limit from settings
                            $review_limit = get_option('multiple_import_review_limit', 10);
                            
                            // Scrape the reviews
                            $reviews = scrape_amazon_reviews($html, $review_limit);
                            
                            // logImportVerification("Processed " . count($reviews) . " reviews for product ID: " . $post_id);

                            if (!empty($reviews)) {
                                // Get existing reviews
                                $existing_reviews = get_comments([
                                    'post_id' => $post_id,
                                    'type' => 'review',
                                    'status' => 'approve'
                                ]);
                                
                                // Create array of existing review hashes
                                $existing_hashes = [];
                                foreach ($existing_reviews as $existing_review) {
                                    $existing_hash = get_comment_meta($existing_review->comment_ID, 'review_hash', true);
                                    if (!empty($existing_hash)) {
                                        $existing_hashes[$existing_hash] = $existing_review->comment_ID;
                                    }
                                }

                                // Initialize rating totals
                                $rating_sum = 0;
                                $rating_count = 0;

                                // Process each review
                                foreach ($reviews as $review_hash => $review) {
                                    // Skip if review already exists
                                    if (isset($existing_hashes[$review_hash])) {
                                        // logImportVerification("Skipping duplicate review: " . $review['title']);
                                        continue;
                                    }

                                    // Prepare comment data
                                    $commentdata = [
                                        'comment_post_ID' => $post_id,
                                        'comment_author' => $review['reviewer_name'],
                                        'comment_content' => $review['text'],
                                        'comment_date' => $review['date'],
                                        'comment_date_gmt' => get_gmt_from_date($review['date']),
                                        'comment_approved' => 1,
                                        'comment_type' => 'review',
                                        'user_id' => 0
                                    ];

                                    // Insert the comment
                                    $comment_id = wp_insert_comment($commentdata);

                                    if ($comment_id) {
                                        // Add all the comment meta
                                        add_comment_meta($comment_id, 'rating', $review['rating']);
                                        add_comment_meta($comment_id, 'review_hash', $review_hash);
                                        add_comment_meta($comment_id, 'verified', 1);
                                        add_comment_meta($comment_id, 'title', $review['title']);

                                        if (!empty($review['reviewer_image'])) {
                                            add_comment_meta($comment_id, 'reviewer_image', $review['reviewer_image']);
                                        }

                                        $rating_sum += floatval($review['rating']);
                                        $rating_count++;

                                        // logImportVerification("Added review: " . $review['title'] . " with ID: " . $comment_id);
                                    }
                                }

                                // Update product rating if we added any new reviews
                                if ($rating_count > 0) {
                                    $product = wc_get_product($post_id);
                                    if ($product) {
                                        // Get actual count of approved reviews
                                        $actual_review_count = get_comments([
                                            'post_id' => $post_id,
                                            'type' => 'review',
                                            'status' => 'approve',
                                            'count' => true
                                        ]);

                                        // Calculate actual rating sum
                                        $actual_rating_sum = 0;
                                        $product_reviews = get_comments([
                                            'post_id' => $post_id,
                                            'type' => 'review',
                                            'status' => 'approve'
                                        ]);

                                        foreach ($product_reviews as $review) {
                                            $rating = get_comment_meta($review->comment_ID, 'rating', true);
                                            $actual_rating_sum += floatval($rating);
                                        }

                                        // Calculate new average
                                        $new_average = $actual_rating_sum / $actual_review_count;

                                        // Update all rating meta
                                        update_post_meta($post_id, '_wc_average_rating', round($new_average, 2));
                                        update_post_meta($post_id, '_wc_rating_count', $actual_review_count);
                                        update_post_meta($post_id, '_wc_review_count', $actual_review_count);
                                        update_post_meta($post_id, '_wc_rating_sum', $actual_rating_sum);

                                        // Clear all relevant caches
                                        delete_transient('wc_product_reviews_' . $post_id);
                                        delete_transient('wc_average_rating_' . $post_id);
                                        wp_cache_delete($post_id, 'product');
                                        
                                        if (function_exists('wc_delete_product_transients')) {
                                            wc_delete_product_transients($post_id);
                                        }

                                        // logImportVerification("Updated product rating. New average: " . round($new_average, 2));
                                    }
                                }

                                // logImportVerification("Completed review import. Added " . $rating_count . " new reviews");
                            }
                        } else {
                            // logImportVerification("Broken page detected: " . $message);
                        }
                    }
                }
            }
            
            update_post_meta( $post_id, '_purchase_note', '' );
            update_post_meta( $post_id, '_featured', 'no' );
            update_post_meta( $post_id, '_weight', '' );
            update_post_meta( $post_id, '_length', '' );
            update_post_meta( $post_id, '_width', '' );
            update_post_meta( $post_id, '_height', '' );
            
            update_post_meta( $post_id, '_sale_price_dates_from', '' );
            update_post_meta( $post_id, '_sale_price_dates_to', '' );
            update_post_meta( $post_id, '_sold_individually', '' );
            update_post_meta( $post_id, '_manage_stock', 'no' );
            update_post_meta( $post_id, '_backorders', 'no' );
            update_post_meta( $post_id, '_stock', '' );
            update_post_meta( $post_id, '_wca_amazon_affiliate_asin', $asin );
            update_post_meta( $post_id, '_wca_amazon_affiliate_parent_asin', $parentASIN );
            
            // Amazon product URL
            update_post_meta( $post_id, '_ams_product_url', $detail_page_url );
            update_post_meta( $post_id, '_detail_page_url', $detail_page_url );
            update_post_meta( $post_id, '_region', $locale );
            update_post_meta( $post_id, '_import_method', '1' );


            ########### Create product Variations ###########
            // Get the limits from settings
            $variation_limit = (int) get_option('ams_variation_limit', 5);
            $image_limit = (int) get_option('ams_variation_image_limit', 5);
            $remove_unavailable = get_option('ams_remove_unavailable_products') === 'Yes';

            if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                if (get_option('product_variants_cron') == 1) {
                    if (isset($variations->VariationDimensions) && !empty($variations->VariationDimensions)) {
                        $attributeChecks = [];
                        $attributes_data = [];

                        foreach ($variations->VariationDimensions as $attribute => $term_name ) {
                            $attr_label = $term_name->DisplayName;
                            $values     = $term_name->Values;
                            $values_array = implode('|', $values);
                            $attr_slug  = sanitize_title($term_name->Name);

                            $values = array_map('trim', $values);
                            $attributeChecks[$attr_slug] = sanitize_title($attr_label);

                            $attributes_data[] = array(
                                'name'      => $attr_label,
                                'slug'      => $attr_slug,
                                'options'   => $values,
                                'visible'   => 1,
                                'variation' => 1
                            );
                        }

                        wc_update_product_attributes($post_id, $attributes_data);

                        // Load the product object
                        $product = wc_get_product($post_id);

                        // Filter out unavailable variations if setting is enabled
                        if ($remove_unavailable) {
                            $Variationlist = array_filter($Variationlist, function($variation) {
                                return !empty($variation['stock_status']) &&
                                       ($variation['regular_price'] > 0 || $variation['sale_price'] > 0);
                            });
                        }

                        // Apply the limit to the Variationlist
                        $limited_variations = array_slice($Variationlist, 0, $variation_limit);

                        // Collect SKUs/ASINs for the limited variations
                        $limited_variation_skus = array();
                        foreach ($limited_variations as $variation) {
                            $limited_variation_skus[] = $variation['sku'];
                        }

                        // Get existing variations
                        $existing_variations = $product->get_children();

                        // Remove variations not in the limited set
                        foreach ($existing_variations as $variation_id) {
                            $variation_sku = get_post_meta($variation_id, '_sku', true);
                            if (!in_array($variation_sku, $limited_variation_skus)) {
                                wp_delete_post($variation_id, true); // Permanently delete
                            }
                        }

                        // Update Variationlist to use only limited variations
                        $Variationlist = $limited_variations;

                        foreach ($Variationlist as $SingleVariation) {
                            // Skip if variation is unavailable and setting is enabled
                            if ($remove_unavailable &&
                                (empty($SingleVariation['stock_status']) ||
                                 ($SingleVariation['regular_price'] == 0 && $SingleVariation['sale_price'] == 0))) {
                                $existing_variation = get_product_by_sku($SingleVariation['sku']);
                                if ($existing_variation !== null) {
                                    wp_delete_post($existing_variation->get_id(), true);
                                }
                                continue;
                            }

                            $variation_post = array(
                                'post_title'  => $SingleVariation['post_title'],
                                'post_name'   => 'product-'.$post_id.'-variation-'.$SingleVariation['sku'],
                                'post_status' => 'publish',
                                'post_parent' => $post_id,
                                'post_type'   => 'product_variation',
                                'guid'        => $product->get_permalink()
                            );

                            $existing_variation = get_product_by_sku($SingleVariation['sku']);
                            if ($existing_variation !== null) {
                                $variation_post['ID'] = $variation_id = $existing_variation->get_id();
                                wp_update_post($variation_post);
                            } else {
                                $variation_id = wp_insert_post($variation_post);
                            }

                            $variation_obj = new \WC_Product_Variation($variation_id);

                            // Assign attributes
                            if (count($SingleVariation['attributes']) > 0) {
                                foreach ($SingleVariation['attributes'] as $attribute => $term_name) {
                                    $taxonomy   = 'pa_' . $attribute;
                                    $term_name  = esc_attr($term_name);

                                    // 1) If the taxonomy doesn't exist, skip
                                    if (!taxonomy_exists($taxonomy)) {
                                        continue;
                                    }

                                    // 2) If term doesn't exist, create it
                                    $maybe_term = term_exists($term_name, $taxonomy);
                                    if (!$maybe_term) {
                                        $inserted = wp_insert_term($term_name, $taxonomy);
                                        if (is_wp_error($inserted)) {
                                            continue;
                                        }
                                        $maybe_term = $inserted;
                                    }

                                    // 3) Fetch the term object
                                    $term_obj = get_term_by('name', $term_name, $taxonomy);
                                    if (!$term_obj) {
                                        continue;
                                    }

                                    // 4) Assign this term to the parent product
                                    wp_set_object_terms($post_id, [$term_obj->term_id], $taxonomy, true);

                                    // 5) Update variation meta
                                    update_post_meta($variation_id, 'attribute_' . $taxonomy, $term_obj->slug);
                                }
                            }

                            // Set SKU
                            if (!empty($SingleVariation['sku'])) {
                                $current_variation_sku = $variation_obj->get_sku();
                                if ($current_variation_sku !== $SingleVariation['sku']) {
                                    $existing_product = wc_get_product_id_by_sku($SingleVariation['sku']);
                                    if (!$existing_product || $existing_product == $variation_obj->get_id()) {
                                $variation_obj->set_sku($SingleVariation['sku']);
                                    }
                                }
                            }

                            // Set prices
                            if (empty($SingleVariation['sale_price'])) {
                                $variation_obj->set_price($SingleVariation['regular_price']);
                            } else {
                                $variation_obj->set_price($SingleVariation['sale_price']);
                                $variation_obj->set_sale_price($SingleVariation['sale_price']);
                            }
                            $variation_obj->set_regular_price($SingleVariation['regular_price']);

                            if (isset($SingleVariation['regular_price']) && !empty($SingleVariation['regular_price'])) {
                                update_post_meta($variation_id, 'regular_price', $SingleVariation['regular_price']);
                            }
                            if (isset($SingleVariation['sale_price']) && !empty($SingleVariation['sale_price'])) {
                                update_post_meta($variation_id, 'sale_price', $SingleVariation['sale_price']);
                            }

                            // Stock status
                            if (!empty($data['stock_status'])) {
                                $variation_obj->set_stock_status($data['stock_status']);
                            } else {
                                $variation_obj->set_manage_stock(false);
                            }
                            // Variant images
                            if (!empty($SingleVariation['product_image_gallery'])) {
                                // Remove existing thumbnail
                                $existing_thumbnail_id = get_post_thumbnail_id($variation_id);
                                if ($existing_thumbnail_id) {
                                    delete_post_thumbnail($variation_id);
                                    wp_delete_attachment($existing_thumbnail_id, true);
                                }

                                // Remove existing gallery
                                $existing_gallery = get_post_meta($variation_id, '_product_image_gallery', true);
                                if ($existing_gallery) {
                                    $existing_gallery_ids = explode(',', $existing_gallery);
                                    foreach ($existing_gallery_ids as $gallery_image_id) {
                                        wp_delete_attachment($gallery_image_id, true);
                                    }
                                    delete_post_meta($variation_id, '_product_image_gallery');
                                }

                                // Reset Product Gallery Meta
                                delete_post_meta($variation_id, '_product_image_gallery');

                                // Apply image limit
                                $limited_gallery = array_slice($SingleVariation['product_image_gallery'], 0, $image_limit);

                                if ('Yes' === get_option('ams_remote_amazon_images')) {
                                    $gallery_image_urls = [];
                                    
                                    // Process gallery images
                                    foreach ($limited_gallery as $attachment) {
                                        $gallery_image_urls[] = esc_url($attachment);
                                    }

                                    // Set gallery images
                                    if (!empty($gallery_image_urls)) {
                                        // Set gallery
                                        attach_product_thumbnail_url_api($variation_id, $gallery_image_urls, 1);
                                        
                                        // Set featured image
                                        if (!empty($gallery_image_urls[0])) {
                                            attach_product_thumbnail_url_api($variation_id, $gallery_image_urls[0], 0);
                                        }
                                    }
                                } else {
                                    $gallery_image_ids = [];
                                    
                                    // Process gallery images
                                    foreach ($limited_gallery as $attachment) {
                                        $attachment_id = attach_product_thumbnail_api($variation_id, $attachment, 1);
                                        if ($attachment_id) {
                                            $gallery_image_ids[] = $attachment_id;
                                        }
                                    }

                                    // Set gallery images
                                    if (!empty($gallery_image_ids)) {
                                        update_post_meta($variation_id, '_product_image_gallery', implode(',', $gallery_image_ids));
                                        
                                        // Set featured image
                                        if (!empty($gallery_image_ids[0])) {
                                            set_post_thumbnail($variation_id, $gallery_image_ids[0]);
                                        }
                                    }
                                }
                            }

                            $variation_obj->set_weight('');
                            $variation_obj->save();
                        }
                    }
                }
            } else {
                // NOT CRON
                if (isset($variations->VariationDimensions) && !empty($variations->VariationDimensions)) {
                    $attributeChecks = [];
                    $attributes_data = [];
                    foreach ($variations->VariationDimensions as $attribute => $term_name ) {
                        $attr_label = $term_name->DisplayName;
                        $values     = $term_name->Values;
                        $values_array = implode('|', $values);
                        $attr_slug  = sanitize_title($term_name->Name);

                        $values = array_map('trim', $values);
                        $attributeChecks[$attr_slug] = sanitize_title($attr_label);

                        $attributes_data[] = array(
                            'name'      => $attr_label,
                            'slug'      => $attr_slug,
                            'options'   => $values,
                            'visible'   => 1,
                            'variation' => 1
                        );
                    }

                    wc_update_product_attributes($post_id, $attributes_data);

                    $product = wc_get_product($post_id);

                    // Filter out unavailable variations if setting is enabled
                    if ($remove_unavailable) {
                        $Variationlist = array_filter($Variationlist, function($variation) {
                            return !empty($variation['stock_status']) &&
                                ($variation['regular_price'] > 0 || $variation['sale_price'] > 0);
                        });
                        // Reindex
                        $Variationlist = array_values($Variationlist);
                    }

                    // Apply variation limit
                    $Variationlist = array_slice($Variationlist, 0, $variation_limit);

                    foreach ($Variationlist as $SingleVariation) {
                        // Skip if unavailable
                        if ($remove_unavailable &&
                            (empty($SingleVariation['stock_status']) ||
                             ($SingleVariation['regular_price'] == 0 && $SingleVariation['sale_price'] == 0))) {
                            $existing_variation = get_product_by_sku($SingleVariation['sku']);
                            if ($existing_variation !== null) {
                                wp_delete_post($existing_variation->get_id(), true);
                            }
                            continue;
                        }

                        $variation_post = array(
                            'post_title'  => $SingleVariation['post_title'],
                            'post_name'   => 'product-'.$post_id.'-variation-'.$SingleVariation['sku'],
                            'post_status' => 'publish',
                            'post_parent' => $post_id,
                            'post_type'   => 'product_variation',
                            'guid'        => $product->get_permalink()
                        );

                        $existing_variation = get_product_by_sku($SingleVariation['sku']);
                        if ($existing_variation !== null) {
                            $variation_post['ID'] = $variation_id = $existing_variation->get_id();
                            wp_update_post($variation_post);
                        } else {
                            $variation_id = wp_insert_post($variation_post);
                        }

                        $variation_obj = new \WC_Product_Variation($variation_id);

                        if (count($SingleVariation['attributes']) > 0) {
                            foreach ($SingleVariation['attributes'] as $attribute => $term_name) {
                                $taxonomy  = 'pa_' . $attribute;
                                $term_name = esc_attr($term_name);

                                // 1) If the taxonomy doesn't exist, skip
                                if (!taxonomy_exists($taxonomy)) {
                                    continue;
                                }

                                // 2) If term doesn't exist, create it
                                $maybe_term = term_exists($term_name, $taxonomy);
                                if (!$maybe_term) {
                                    $inserted = wp_insert_term($term_name, $taxonomy);
                                    if (is_wp_error($inserted)) {
                                        continue;
                                    }
                                    $maybe_term = $inserted;
                                }

                                // 3) Get term object
                                $term_obj = get_term_by('name', $term_name, $taxonomy);
                                if (!$term_obj) {
                                    continue;
                                }

                                // 4) Assign this term to the parent
                                wp_set_object_terms($post_id, [$term_obj->term_id], $taxonomy, true);

                                // 5) Assign to variation
                                update_post_meta($variation_id, 'attribute_'.$taxonomy, $term_obj->slug);
                            }
                        }

                        // Set SKU
                        if (!empty($SingleVariation['sku'])) {
                            $current_variation_sku = $variation_obj->get_sku();
                            if ($current_variation_sku !== $SingleVariation['sku']) {
                                $existing_product = wc_get_product_id_by_sku($SingleVariation['sku']);
                                if (!$existing_product || $existing_product == $variation_obj->get_id()) {
                            $variation_obj->set_sku($SingleVariation['sku']);
                                }
                            }
                        }

                        // Set prices
                        if (empty($SingleVariation['sale_price'])) {
                            $variation_obj->set_price($SingleVariation['regular_price']);
                        } else {
                            $variation_obj->set_price($SingleVariation['sale_price']);
                            $variation_obj->set_sale_price($SingleVariation['sale_price']);
                        }
                        $variation_obj->set_regular_price($SingleVariation['regular_price']);

                        if (isset($SingleVariation['regular_price']) && !empty($SingleVariation['regular_price'])) {
                            update_post_meta($variation_id, 'regular_price', $SingleVariation['regular_price']);
                        }
                        if (isset($SingleVariation['sale_price']) && !empty($SingleVariation['sale_price'])) {
                            update_post_meta($variation_id, 'sale_price', $SingleVariation['sale_price']);
                        }

                        // Stock
                        if (!empty($data['stock_status'])) {
                            $variation_obj->set_stock_status($data['stock_status']);
                        } else {
                            $variation_obj->set_manage_stock(false);
                        }

                        // Variation images
                        if (!empty($SingleVariation['product_image_gallery'])) {
                            // Remove existing thumbnail
                            $existing_thumbnail_id = get_post_thumbnail_id($variation_id);
                            if ($existing_thumbnail_id) {
                                delete_post_thumbnail($variation_id);
                                wp_delete_attachment($existing_thumbnail_id, true);
                            }

                            // Remove existing gallery
                            $existing_gallery = get_post_meta($variation_id, '_product_image_gallery', true);
                            if ($existing_gallery) {
                                $existing_gallery_ids = explode(',', $existing_gallery);
                                foreach ($existing_gallery_ids as $gallery_image_id) {
                                    wp_delete_attachment($gallery_image_id, true);
                                }
                                delete_post_meta($variation_id, '_product_image_gallery');
                            }

                            // Reset Product Gallery Meta
                            delete_post_meta($variation_id, '_product_image_gallery');

                            // Apply image limit
                            $limited_gallery = array_slice($SingleVariation['product_image_gallery'], 0, $image_limit);

                            if ('Yes' === get_option('ams_remote_amazon_images')) {
                                $gallery_image_urls = [];
                                foreach ($limited_gallery as $attachment) {
                                    $gallery_image_urls[] = esc_url($attachment);
                                }
                                if (!empty($gallery_image_urls)) {
                                    attach_product_thumbnail_url_api($variation_id, $gallery_image_urls, 1);
                                    if (!empty($gallery_image_urls[0])) {
                                        attach_product_thumbnail_url_api($variation_id, $gallery_image_urls[0], 0);
                                    }
                                }
                            } else {
                                $gallery_image_ids = [];
                                foreach ($limited_gallery as $attachment) {
                                    $attachment_id = attach_product_thumbnail_api($variation_id, $attachment, 1);
                                    if ($attachment_id) {
                                        $gallery_image_ids[] = $attachment_id;
                                    }
                                }
                                if (!empty($gallery_image_ids)) {
                                    update_post_meta($variation_id, '_product_image_gallery', implode(',', $gallery_image_ids));
                                    if (!empty($gallery_image_ids[0])) {
                                        set_post_thumbnail($variation_id, $gallery_image_ids[0]);
                                    }
                                }
                            }
                        }

                        $variation_obj->set_weight('');
                        $variation_obj->save();
                    }
                }
            }
            ########### End Create product Variations ###########


            $Current = get_post_meta( $post_id, '_stock_status',true);
            if(!isset($_POST['is_cron']) || $_POST['is_cron'] != 1){ 
                update_post_meta( $post_id, 'ams_last_cron_update',date('Y-m-d H:i:s'));
                update_post_meta( $post_id, 'ams_last_cron_status',0);
                update_post_meta( $post_id, '_stock_status', $product_status );
                $message = "Success";
            } else {
                update_post_meta( $post_id, 'ams_last_cron_update',date('Y-m-d H:i:s'));
                update_post_meta( $post_id, 'ams_last_cron_status',1);
                
                if (get_option('product_out_of_stock_cron') == 1) {
                    update_post_meta( $post_id, '_stock_status', $product_status );
                } else {
                    // If stock management is disabled, still update stock status if product is available (has valid status)
                    if ($product_status === 'instock') {
                        update_post_meta( $post_id, '_stock_status', 'instock' );
                    } else {
                    update_post_meta( $post_id, '_stock_status', $Current );
                    }
                }
                
                $short_title = wp_trim_words($title, 3, '...');
                $short_url = (strlen($detail_page_url) > 25) ? substr($detail_page_url, 0, 22) . '...' : $detail_page_url;
                $message = sprintf(
                    '%s %s <a href="%s" target="_blank" class="text-white">%s</a>',
                    esc_html__('Updated:', 'ams-wc-amazon'),
                    esc_html($short_title),
                    esc_url($detail_page_url),
                    esc_html($short_url)
                );
            }
            echo wp_kses_post($message);
        }
    } catch (Throwable $th) {
        update_post_meta( $post_id, 'ams_last_cron_update',date('Y-m-d H:i:s'));
        update_post_meta( $post_id, 'ams_last_cron_status',0);
        echo wp_kses_post($th->getMessage());
    }
    wp_die();
}

// Public API Update for Cron
public function ams_product_api_update_cron($post_id, $asin) {
    $__log_file = plugin_dir_path(__FILE__) . '../import_verification.log';
    @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] Update Start (Cron): func=ImportProducts::ams_product_api_update_cron POST_ID=' . $post_id . '; ASIN=' . $asin . "\n", FILE_APPEND);
    $result = $this->ams_product_api_update($post_id, $asin, true);
    @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] Update End (Cron): func=ImportProducts::ams_product_api_update_cron POST_ID=' . $post_id . '; ASIN=' . $asin . '; status=' . ($result ? 'success' : 'failure') . "\n", FILE_APPEND);
    return $result;
}
// API Update
private function ams_product_api_update($post_id, $asin, $is_cron = false) {
    $__log_file = plugin_dir_path(__FILE__) . '../import_verification.log';
    @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] Update Start (API): func=ImportProducts::ams_product_api_update POST_ID=' . $post_id . '; ASIN=' . $asin . '; cron=' . ($is_cron ? '1' : '0') . "\n", FILE_APPEND);
    try {
        // Check License
        if (ams_plugin_license_status() === false) {
            if ($is_cron) {

                return false;
            } else {
            echo "<script>console.log('Plugin license not activated');</script>";
            $license = sprintf(esc_html__('Activate License!','ams-wc-amazon'));
            echo wp_kses_post($license);
            wp_die();
            }
        }

        $product_region = get_post_meta( $post_id, '_region', true );
        $locale = !empty($product_region) ? $product_region : get_option( 'ams_amazon_country' );
        $regions = ams_get_amazon_regions();
        $marketplace = ams_get_marketplace_value($locale);
        $service_name = 'ProductAdvertisingAPI';
        $region = $regions[ $locale ]['RegionCode'];
        $access_key = get_option( 'ams_access_key_id' );
        $secret_key = get_option( 'ams_secret_access_key' );
        $payload_arr = array();
        $payload_arr['ItemIds'] = array( $asin );

        $payload_arr['Resources'] = array(
            // Category and Relationships
            "BrowseNodeInfo.BrowseNodes.Ancestor",  // Include category information
            "ParentASIN",

            // Images
            "Images.Primary.Small",
            "Images.Primary.Medium",
            "Images.Primary.Large",
            "Images.Variants.Small",
            "Images.Variants.Medium",
            "Images.Variants.Large",

            // Basic Product Information
            "ItemInfo.ByLineInfo",
            "ItemInfo.ContentInfo",
            "ItemInfo.ContentRating",
            "ItemInfo.Classifications",
            "ItemInfo.ExternalIds",
            "ItemInfo.Features",
            "ItemInfo.ManufactureInfo",
            "ItemInfo.ProductInfo",
            "ItemInfo.TechnicalInfo",
            "ItemInfo.Title",
            "ItemInfo.TradeInInfo",

            // Offers and Availability
            "Offers.Listings.Availability.MaxOrderQuantity",
            "Offers.Listings.Availability.Message",
            "Offers.Listings.Availability.MinOrderQuantity",
            "Offers.Listings.Availability.Type",
            "Offers.Listings.Condition",
            "Offers.Listings.Condition.ConditionNote",
            "Offers.Listings.Condition.SubCondition",
            "Offers.Listings.DeliveryInfo.IsAmazonFulfilled",
            "Offers.Listings.DeliveryInfo.IsFreeShippingEligible",
            "Offers.Listings.DeliveryInfo.IsPrimeEligible",
            "Offers.Listings.DeliveryInfo.ShippingCharges",
            "Offers.Listings.IsBuyBoxWinner",
            "Offers.Listings.LoyaltyPoints.Points",
            "Offers.Listings.MerchantInfo",
            "Offers.Listings.Price",
            "Offers.Listings.ProgramEligibility.IsPrimeExclusive",
            "Offers.Listings.ProgramEligibility.IsPrimePantry",
            "Offers.Listings.Promotions",
            "Offers.Listings.SavingBasis",

            // Offer Summaries
            "Offers.Summaries.HighestPrice",
            "Offers.Summaries.LowestPrice",
            "Offers.Summaries.OfferCount"
        );

        $payload_arr['PartnerTag'] = get_option( 'ams_associate_tag' );
        $payload_arr['PartnerType'] = 'Associates';
        $payload_arr['Marketplace'] = $marketplace;
        $payload_arr['Operation'] = 'GetItems';
        $payload = wp_json_encode( $payload_arr );
        $host = $regions[ $locale ]['Host'];
        $uri_path = "/paapi5/getitems";
        $api = new \Amazon\Affiliate\Api\Amazon_Product_Api ( $access_key, $secret_key, $region, $service_name, $uri_path, $payload, $host, 'GetItems' );
        $response = $api->do_request();

        // Check if ItemsResult and Items are set before accessing them
        if (isset($response->ItemsResult) && isset($response->ItemsResult->Items)) {
            $results = $response->ItemsResult->Items;
        } else {
            $results = null;
        }

        if(empty($results)) {
            sleep(3);
            $apiRetried = new \Amazon\Affiliate\Api\Amazon_Product_Api($access_key, $secret_key, $region, $service_name, $uri_path, $payload, $host, 'GetItems');
            $response = $apiRetried->do_request();
            
            // Check if ItemsResult and Items are set before accessing them
            if (isset($response->ItemsResult) && isset($response->ItemsResult->Items)) {
                $results = $response->ItemsResult->Items;
            } else {
                $results = null;
            }

            if(empty($results)) {
                update_post_meta( $post_id, 'ams_last_cron_update',date('Y-m-d H:i:s'));
                update_post_meta( $post_id, 'ams_last_cron_status', 0 );
                @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] Update End (API): func=ImportProducts::ams_product_api_update POST_ID=' . $post_id . '; ASIN=' . $asin . '; status=failure (no results)' . "\n", FILE_APPEND);
                return FALSE;
            }
        }
        foreach ($results as $row) {
            $thumbnail_size  = get_option( 'ams_product_thumbnail_size' );
            $asin            = $row->ASIN;
            $parentASIN = isset($row->ParentASIN) ? $row->ParentASIN : null;

            $detail_page_url = $row->DetailPageURL;
            if(isset($row->Images->Primary->{$thumbnail_size})) {
                $image = $row->Images->Primary->{$thumbnail_size}->URL;
            } else {
                $key = key((array)$row->Images->Primary);
                $image = $row->Images->Primary->{$key}->URL;
            }
            $gallery = isset($row->Images->Variants) ? $row->Images->Variants : null;

            $amount = isset($row->Offers->Listings[0]->Price->Amount) ? $row->Offers->Listings[0]->Price->Amount : 0;
            $saving_amount = isset($row->Offers->Listings[0]->SavingBasis->Amount) ? $row->Offers->Listings[0]->SavingBasis->Amount : 0;
            $DisplayAmount = isset($row->Offers->Listings[0]->Price->DisplayAmount) ? $row->Offers->Listings[0]->Price->DisplayAmount : '';
            $SavingDisplayAmount = isset($row->Offers->Listings[0]->SavingBasis->DisplayAmount) ? $row->Offers->Listings[0]->SavingBasis->DisplayAmount : null;
            $product_status = isset($row->Offers->Listings[0]->Availability->Message) ? $row->Offers->Listings[0]->Availability->Message : '';
            $product_status  = !empty($product_status) ? 'instock' : 'outofstock';
            
            // If API shows outofstock but we have prices from scraping, override to instock
            if ($product_status === 'outofstock' && ($amount > 0 || $saving_amount > 0)) {
                $product_status = 'instock';
                // logImportVerification('API showed outofstock but has prices, changed to instock');
            }
            $title           = $row->ItemInfo->Title->DisplayValue;
            $features = isset($row->ItemInfo->Features->DisplayValues) ? $row->ItemInfo->Features->DisplayValues : null;
            $payload_arr2 = array();
            $payload_arr2['ASIN']     = $asin;//'B00T0C9XRK';

            $payload_arr2['Resources'] = array(
                "ParentASIN", "ItemInfo.Title", "Offers.Listings.Price",
                "Offers.Listings.ProgramEligibility.IsPrimeExclusive", 
                "Offers.Listings.ProgramEligibility.IsPrimePantry", 
                "Offers.Listings.Promotions", 
                "Offers.Listings.SavingBasis", 
                "Offers.Listings.Availability.Message", 
                "Offers.Summaries.HighestPrice", 
                "Offers.Summaries.LowestPrice", 
                "VariationSummary.Price.HighestPrice", 
                "VariationSummary.Price.LowestPrice", 
                "VariationSummary.VariationDimension",
                "Images.Variants.Large",
                "Images.Variants.Medium",
                "Images.Variants.Small"
            );

            $payload_arr2['PartnerTag']  = get_option( 'ams_associate_tag' );
            $payload_arr2['PartnerType'] = 'Associates';
            $payload_arr2['Marketplace'] = $marketplace; //'www.amazon.com';
            $payload_arr2['Operation']   = 'GetVariations';
            $payload2                   = json_encode($payload_arr2);
            $host                      = $regions[ $locale ]['Host'];
            $uri_path                  = "/paapi5/getvariations";
            $api2                      = new  \Amazon\Affiliate\Api\Amazon_Product_Api ($access_key, $secret_key,$region, $service_name, $uri_path, $payload2, $host, 'getVariation' );

            $response2 = $api2->do_request();

            if (isset($response2->VariationsResult)) {
                $variations = $response2->VariationsResult->VariationSummary;
                $attributes = $response2->VariationsResult->Items;
            } else {
                $variations = null;
                $attributes = null;
            }

            $VariationPage = 2;
            $Variationlist = [];

            // Variants
            $Variationlist = array(); // Initialize empty array
            $variation_limit = get_option('ams_variation_limit', 5);
            $image_limit = get_option('ams_variation_image_limit', 5);
            $remove_unavailable = get_option('ams_remove_unavailable_products') === 'Yes';

            if (isset($variations->PageCount) && $variations->PageCount >= 1) {
                // Load the product object using the post_id
                $product = wc_get_product($post_id);

                // Check if the product exists and is valid
                if (!$product) {
                    throw new Exception("Product not found or invalid for post_id: " . $post_id);
                }

                // Apply the limit to the list of variations
                $limited_variations = array_slice($response2->VariationsResult->Items, 0, $variation_limit);

                // Collect ASINs for the limited variations
                $limited_variation_asins = array();
                foreach ($limited_variations as $item) {
                    // Check availability
                    $amount = isset($item->Offers->Listings[0]->Price->Amount) ? $item->Offers->Listings[0]->Price->Amount : 0;
                    $saving_amount = isset($item->Offers->Listings[0]->SavingBasis->Amount) ? $item->Offers->Listings[0]->SavingBasis->Amount : 0;
                    $product_stock = isset($item->Offers->Listings[0]->Availability->Message) ? $item->Offers->Listings[0]->Availability->Message : '';

                    // Skip if unavailable and setting is enabled
                    if ($remove_unavailable && (empty($product_stock) || ($amount == 0 && $saving_amount == 0))) {
                        continue;
                    }

                    $limited_variation_asins[] = $item->ASIN;
                }

                // Fetch existing variations for the product
                $existing_variations = $product->get_children();

                // Loop through the existing variations and remove those not in the limited set
                foreach ($existing_variations as $variation_id) {
                    $variation_sku = get_post_meta($variation_id, '_sku', true); // Assuming ASIN is stored as SKU

                    // If the existing variation SKU is not in the limited set, delete it
                    if (!in_array($variation_sku, $limited_variation_asins)) {
                        wp_delete_post($variation_id, true); // Permanently delete the variation
                    }
                }

                // Process the limited variations
                foreach ($limited_variations as $item) {
                    if (!isset($item->ASIN) || empty($item->ASIN)) {
                        continue; // Skip if ASIN is missing
                    }

                    // Check if variation should be skipped due to unavailability
                    $amount = isset($item->Offers->Listings[0]->Price->Amount) ? $item->Offers->Listings[0]->Price->Amount : 0;
                    $DisplayAmount = isset($item->Offers->Listings[0]->Price->DisplayAmount) ? $item->Offers->Listings[0]->Price->DisplayAmount : null;
                    $saving_amount = isset($item->Offers->Listings[0]->SavingBasis->Amount) ? $item->Offers->Listings[0]->SavingBasis->Amount : 0;
                    $SavingDisplayAmount = isset($item->Offers->Listings[0]->SavingBasis->DisplayAmount) ? $item->Offers->Listings[0]->SavingBasis->DisplayAmount : null;
                    $product_stock = isset($item->Offers->Listings[0]->Availability->Message) ? $item->Offers->Listings[0]->Availability->Message : '';
                    $stock_status = !empty($product_stock) ? 'instock' : 'outofstock';

                    // Skip unavailable variations if setting is enabled
                    if ($remove_unavailable && (empty($product_stock) || ($amount == 0 && $saving_amount == 0))) {
                        // Remove if it exists
                        $existing_variation = get_product_by_sku($item->ASIN);
                        if ($existing_variation !== null) {
                            wp_delete_post($existing_variation->get_id(), true);
                        }
                        continue;
                    }

                    $VariationAttribute = [];
                    if (isset($item->VariationAttributes)) {
                        foreach ($item->VariationAttributes as $ItemVariationAttribute) {
                            if (isset($ItemVariationAttribute->Name) && isset($ItemVariationAttribute->Value)) {
                                $VariationAttribute[$ItemVariationAttribute->Name] = trim($ItemVariationAttribute->Value);
                            }
                        }
                    }

                    // Set prices
                    if (empty($saving_amount)) {
                        $sale_price = $amount;
                        $regular_price = $amount;
                    } else {
                        $sale_price = $amount;
                        $regular_price = $saving_amount;
                    }

                    // Process variant images
                    $v_gallery = [];
                    
                    // Check for variant images
                    if (isset($item->Images->Variants)) {
                        foreach ($item->Images->Variants as $variant_image) {
                            if (isset($variant_image->Large->URL)) {
                                $v_gallery[] = $variant_image->Large->URL;
                            }
                        }
                    }
                    
                    // Fallback to primary image if no variant images
                    if (empty($v_gallery) && isset($item->Images->Primary->Large->URL)) {
                        $v_gallery[] = $item->Images->Primary->Large->URL;
                    }

                    // Apply image limit
                    if ($image_limit > 0) {
                        $v_gallery = array_slice($v_gallery, 0, $image_limit);
                    }

                    // Only add to Variationlist if we have required fields
                    if (isset($item->ItemInfo->Title->DisplayValue)) {
                        $Variationlist[] = array(
                            'post_title' => $item->ItemInfo->Title->DisplayValue,
                            'attributes' => $VariationAttribute,
                            'sku' => $item->ASIN,
                            'regular_price' => floatval($regular_price),
                            'sale_price' => floatval($sale_price),
                            'stock_status' => $stock_status,
                            'product_image_gallery' => $v_gallery,
                        );
                    }
                }
            }
            // Process additional variation pages
            while ($VariationPage <= ($variations->PageCount ?? 0)) {
                // Skip if we've already reached our variation limit
                if (count($Variationlist) >= $variation_limit) {
                    break;
                }

                $payload_arr2['VariationPage'] = $VariationPage;
                $payload3 = json_encode($payload_arr2);
                $api3 = new \Amazon\Affiliate\Api\Amazon_Product_Api(
                    $access_key, $secret_key, $region, $service_name, $uri_path, $payload3, $host, 'getVariation'
                );
                $response3 = $api3->do_request();

                if (isset($response3->VariationsResult) && isset($response3->VariationsResult->Items)) {
                    foreach ($response3->VariationsResult->Items as $item) {
                        // Break if we've reached the variation limit
                        if (count($Variationlist) >= $variation_limit) {
                            break 2; // Break both foreach and while loops
                        }

                        if (!isset($item->ASIN) || empty($item->ASIN)) {
                            continue; // Skip if ASIN is missing
                        }

                        // Check if variation should be skipped due to unavailability
                        $amount = isset($item->Offers->Listings[0]->Price->Amount) ? $item->Offers->Listings[0]->Price->Amount : 0;
                        $DisplayAmount = isset($item->Offers->Listings[0]->Price->DisplayAmount) ? $item->Offers->Listings[0]->Price->DisplayAmount : null;
                        $saving_amount = isset($item->Offers->Listings[0]->SavingBasis->Amount) ? $item->Offers->Listings[0]->SavingBasis->Amount : 0;
                        $SavingDisplayAmount = isset($item->Offers->Listings[0]->SavingBasis->DisplayAmount) ? $item->Offers->Listings[0]->SavingBasis->DisplayAmount : null;
                        $product_stock = isset($item->Offers->Listings[0]->Availability->Message) ? $item->Offers->Listings[0]->Availability->Message : '';
                        $stock_status = !empty($product_stock) ? 'instock' : 'outofstock';

                        // Skip unavailable variations if setting is enabled
                        if ($remove_unavailable && (empty($product_stock) || ($amount == 0 && $saving_amount == 0))) {
                            // Remove if it exists
                            $existing_variation = get_product_by_sku($item->ASIN);
                            if ($existing_variation !== null) {
                                wp_delete_post($existing_variation->get_id(), true);
                            }
                            continue;
                        }

                        $VariationAttribute = [];
                        if (isset($item->VariationAttributes)) {
                            foreach ($item->VariationAttributes as $ItemVariationAttribute) {
                                if (isset($ItemVariationAttribute->Name) && isset($ItemVariationAttribute->Value)) {
                                    $VariationAttribute[$ItemVariationAttribute->Name] = trim($ItemVariationAttribute->Value);
                                }
                            }
                        }

                        // Set prices
                        if (empty($saving_amount)) {
                            $sale_price = $amount;
                            $regular_price = $amount;
                        } else {
                            $sale_price = $amount;
                            $regular_price = $saving_amount;
                        }

                        // Process variant images
                        $v_gallery = [];
                        if (isset($item->Images->Variants)) {
                            foreach ($item->Images->Variants as $variant_image) {
                                if (isset($variant_image->Large->URL)) {
                                    $v_gallery[] = $variant_image->Large->URL;
                                }
                            }
                        }
                        if (empty($v_gallery) && isset($item->Images->Primary->Large->URL)) {
                            $v_gallery[] = $item->Images->Primary->Large->URL;
                        }

                        // Apply image limit
                        if ($image_limit > 0) {
                            $v_gallery = array_slice($v_gallery, 0, $image_limit);
                        }

                        // Only add to Variationlist if we have required fields
                        if (isset($item->ItemInfo->Title->DisplayValue)) {
                            $Variationlist[] = array(
                                'post_title' => $item->ItemInfo->Title->DisplayValue,
                                'attributes' => $VariationAttribute,
                                'sku' => $item->ASIN,
                                'regular_price' => floatval($regular_price),
                                'sale_price' => floatval($sale_price),
                                'stock_status' => $stock_status,
                                'product_image_gallery' => $v_gallery,
                            );
                        }
                    }
                }

                $VariationPage++;
            }

            // Check if $variations is a valid object and if VariationDimensions exists and is not empty
            if (!is_object($variations) || !isset($variations->VariationDimensions) || empty($variations->VariationDimensions)) {
                // Set product type to 'simple' if no variations are available
                wp_set_object_terms($post_id, 'simple', 'product_type');
            } else {
                // Set product type to 'variable' if variations are available
                wp_set_object_terms($post_id, 'variable', 'product_type');
            }

            // Main product SKU update
            if (get_option('product_sku_cron', true)) {
                // Check if ASIN exists to update the main product SKU
                if (!empty($asin)) {
                    $current_sku = get_post_meta($post_id, '_sku', true);
                    if ($current_sku !== $asin) {
                        // Check if this ASIN is already used by another product
                        $existing_product = wc_get_product_id_by_sku($asin);
                        if (!$existing_product || $existing_product == $post_id) {
                    update_post_meta($post_id, '_sku', $asin);

                        }
                    }
                } else {
                    //error_log("Main product SKU not available for post ID: " . $post_id);
                }
            }

            // Product name update
            if (get_option('product_name_cron', true) == 1) {
                $postData = array('ID' => $post_id);
                $postData['post_title'] = stripslashes($title);
                wp_update_post($postData);
            }


            // Product description update
            if (get_option('product_description_cron', true) == 1) {
                $content = '';

                // Check if `Features.DisplayValues` has content and add it to the description
                if (isset($row->ItemInfo->Features->DisplayValues) && is_array($row->ItemInfo->Features->DisplayValues)) {
                    foreach ($row->ItemInfo->Features->DisplayValues as $feature) {
                        $content .= '<ul><li>' . $feature . '</li></ul>';
                    }
                }

                // Update post content if not empty
                if (!empty($content)) {
                    $postData = array('ID' => $post_id, 'post_content' => $content);
                    wp_update_post($postData);
                }
            }


            // Product category update
            if (get_option('product_category_cron', true) == 1) {
                $ams_default_category = get_option('ams_default_category', true);

                // If auto-import from Amazon is selected
                if (!empty($ams_default_category) && $ams_default_category == '_auto_import_amazon') {
                    $category_paths = [];
                    $min_depth = intval(get_option('ams_category_min_depth', 1));
                    $max_depth = intval(get_option('ams_category_max_depth', 6));

                    // Check if BrowseNodes exist in the API response
                    if (isset($row->BrowseNodeInfo->BrowseNodes)) {
                        $browse_nodes = $row->BrowseNodeInfo->BrowseNodes;

                        foreach ($browse_nodes as $node) {
                            $current_node = $node;
                            $node_parts = [];

                            while (isset($current_node->Ancestor)) {
                                array_unshift($node_parts, $current_node->Ancestor->DisplayName);
                                $current_node = $current_node->Ancestor;
                            }

                            array_push($node_parts, $node->DisplayName);

                            $cleaned_path = array_filter($node_parts, function ($category) {
                                return !in_array($category, ['Categories', 'Home', 'Root']);
                            });

                            // Ensure we get paths between min_depth and max_depth
                            if (count($cleaned_path) >= $min_depth) {
                                // Trim the cleaned path to match the max depth if necessary
                                $trimmed_path = array_slice($cleaned_path, 0, $max_depth);
                                // Store paths with '>' separator
                                $category_paths[] = implode(' > ', $trimmed_path);
                            }
                        }

                        if (!empty($category_paths)) {
                            $category_path = $category_paths[0];
                            $categories = explode(' > ', $category_path);
                            $parent_id = 0;
                            $category_ids = array();

                            foreach ($categories as $category_name) {
                                $category_name = trim($category_name);

                                if (preg_match('/[a-f0-9\-]{36,}/', $category_name)) {
                                    continue;
                                }

                                $existing_category = term_exists($category_name, 'product_cat', $parent_id);

                                if (!$existing_category) {
                                    $new_category = wp_insert_term($category_name, 'product_cat', array(
                                        'description' => $category_name,
                                        'parent' => $parent_id
                                    ));
                                    if (!is_wp_error($new_category)) {
                                        $parent_id = $new_category['term_id'];
                                        $category_ids[] = $parent_id;
                                    }
                                } else {
                                    $term = get_term_by('name', esc_attr($category_name), 'product_cat');
                                    if ($term) {
                                        $parent_id = $term->term_id;
                                        $category_ids[] = $parent_id;
                                    } else {
                                        $parent_id = $existing_category['term_id'];
                                        $category_ids[] = $parent_id;
                                    }
                                }
                            }

                            if (!empty($category_ids)) {
                                update_post_meta($post_id, '_product_categories_hierarchy', $category_path);
                                wp_set_object_terms($post_id, $category_ids, 'product_cat');
                            }
                        }
                    }
                } else {
                    // If a specific category is selected (not auto-import)
                    if (!empty($ams_default_category)) {
                        // Try to get the term
                        $term = null;
                        if (is_numeric($ams_default_category)) {
                            $term = get_term($ams_default_category, 'product_cat');
                        }
                        if (!$term) {
                            $term = get_term_by('id', $ams_default_category, 'product_cat');
                        }
                        if (!$term) {
                            $term = get_term_by('slug', $ams_default_category, 'product_cat');
                        }
                        if (!$term) {
                            $term = get_term_by('name', $ams_default_category, 'product_cat');
                        }

                        // Set the category
                        if ($term && !is_wp_error($term)) {
                            wp_set_object_terms($post_id, $term->term_id, 'product_cat');
                            //error_log('AMS Debug: Setting category to ' . $term->name . ' (ID: ' . $term->term_id . ')');
                        } else {
                            //error_log('AMS Debug: Could not find category for ' . print_r($ams_default_category, true));
                        }
                    }
                }

                // Clear the term cache for this post
                clean_post_cache($post_id);
                wp_cache_delete($post_id, 'product_cat_relationships');
            }

            // Update the brand name
            if (isset($row->ItemInfo->ByLineInfo->Brand->DisplayValue)) {
                $brandName = trim($row->ItemInfo->ByLineInfo->Brand->DisplayValue);

                // logImportVerification('Brand: ' . $brandName);

                // Check if the brand already exists as a term
                $brandTerm = term_exists($brandName, 'product_brand');

                if (!$brandTerm) {
                    // If the brand doesn't exist, create it
                    $brandTerm = wp_insert_term($brandName, 'product_brand');
                }

                if (is_wp_error($brandTerm)) {
                    // logImportVerification('Error creating brand term: ' . $brandTerm->get_error_message());
                } else {
                    // Get the term ID
                    $brandTermId = isset($brandTerm['term_id']) ? $brandTerm['term_id'] : $brandTerm;

                    // Assign the brand to the product
                    wp_set_object_terms($post_id, intval($brandTermId), 'product_brand');

                    // Store the brand in post meta for further use
                    update_post_meta($post_id, '_product_brand', $brandName);

                    // logImportVerification('Brand assigned to product successfully.');
                }
            } else {
                // logImportVerification('Brand not found in the API response.');
            }
            // Update the brand name
            // Update images
            if ('1' == get_option('product_image_cron', true)) {
                if ($image || $gallery) {
                    // Remove existing images and URLs
                    delete_product_images($post_id);
                    reset_product_thumbnail_url($post_id, $flag = 0); // Reset featured image URL
                    delete_product_gallery_images($post_id);
                    reset_product_thumbnail_url($post_id, $flag = 1); // Reset gallery URL

                    $gallery_url = [];
                    $gallery = is_array($gallery) ? $gallery : [];

                    if ('Yes' === get_option('ams_remote_amazon_images')) {
                        // Set featured image URL (remote)
                        attach_product_thumbnail_url_api($post_id, $image, 0);

                        // Process and set gallery images (remote)
                        foreach ($gallery as $image) {
                            if (isset($image->{$thumbnail_size})) {
                                $gallery_url[] = $image->{$thumbnail_size}->URL;
                            } else {
                                // Fallback to any available size if specific size is missing
                                $imageArray = (array) $image;
                                $key = key($imageArray);
                                $gallery_url[] = $image->{$key}->URL;
                            }
                        }

                        // Apply the gallery image limit from plugin settings
                        $image_limit = get_option('ams_variation_image_limit', 5);
                        if ($image_limit > 0) {
                            $gallery_url = array_slice($gallery_url, 0, $image_limit);
                        }

                        if (!empty($gallery_url)) {
                            attach_product_thumbnail_url_api($post_id, $gallery_url, 1);
                        }

                    } else {
                        // Download and set the featured image locally
                        attach_product_thumbnail_api($post_id, $image, 0);

                        // Download and set the gallery images locally with image limit applied
                        if (count($gallery) > 0) {
                            $image_limit = get_option('ams_variation_image_limit', 5);
                            $limited_gallery = array_slice($gallery, 0, $image_limit);
                            foreach ($limited_gallery as $image) {
                                if (isset($image->{$thumbnail_size})) {
                                    attach_product_thumbnail_api($post_id, $image->{$thumbnail_size}->URL, 1);
                                } else {
                                    $imageArray = (array) $image;
                                    $key = key($imageArray);
                                    attach_product_thumbnail_api($post_id, $image->{$key}->URL, 1);
                                }
                            }
                        }
                    }
                }
            }


            $user_id = get_current_user();
            if(!isset($variations->VariationDimensions) || empty($variations->VariationDimensions)){
                wp_set_object_terms( $post_id, 'simple', 'product_type');
                if( get_option('product_sku_cron', true) == 1 ) {
                    $current_sku = get_post_meta($post_id, '_sku', true);
                    if ($current_sku !== $asin) {
                        $existing_product = wc_get_product_id_by_sku($asin);
                        if (!$existing_product || $existing_product == $post_id) {
                    update_post_meta( $post_id, '_sku', $asin );
                        }
                    }
                }
                if( get_option('product_price_cron', true) == 1 ) {
                    $final_regular_price = 0;
                    $final_sale_price = 0;
                    
                    if( empty($saving_amount) ) {
                        $final_regular_price = $this->parsePriceStringnew($amount);
                        $final_sale_price = $final_regular_price;
                    } else {
                        $final_sale_price = $this->parsePriceStringnew($amount);
                        $final_regular_price = $this->parsePriceStringnew($saving_amount);
                    }
                    
                    // Fallback to scraping if API prices are 0
                    if ($final_regular_price == 0 && $final_sale_price == 0) {
                        $product_url = get_post_meta($post_id, '_detail_page_url', true);
                        if (!empty($product_url)) {
                            $content = $this->getContentUsingCurl($product_url, true);
                            if ($content) {
                                if (!class_exists('simple_html_dom')) {
                                    require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
                                }
                                $html = new \simple_html_dom();
                                $html->load($content);
                                
                                $productPrice = $this->fetchPriceFromHtml($html);
                                $scraped_regular = isset($productPrice['regular_price']) ? $productPrice['regular_price'] : 0;
                                $scraped_sale = isset($productPrice['sale_price']) ? $productPrice['sale_price'] : 0;
                                
                                if ($scraped_regular > 0 || $scraped_sale > 0) {
                                    $final_regular_price = $scraped_regular;
                                    $final_sale_price = $scraped_sale;
                                    // Also update stock status since we found prices via scraping
                                    $product_status = 'instock';
                                    // logImportVerification('API prices were 0, used scraped prices: Regular=' . $scraped_regular . ', Sale=' . $scraped_sale . ', Status=instock');
                                }
                                
                                $html->clear();
                            }
                        }
                    }
                    
                    update_post_meta( $post_id, '_regular_price', $final_regular_price );
                    if ($final_sale_price > 0 && $final_sale_price < $final_regular_price) {
                        update_post_meta( $post_id, '_sale_price', $final_sale_price );
                        update_post_meta( $post_id, '_price', $final_sale_price );
                    } else {
                        delete_post_meta( $post_id, '_sale_price' );
                        update_post_meta( $post_id, '_price', $final_regular_price );
                    }
                }
            } else {
                // SKU
                if(get_option('product_sku_cron', true) == 1 && $parentASIN) {
                    $current_sku = get_post_meta($post_id, '_sku', true);
                    if ($current_sku !== $parentASIN) {
                        $existing_product = wc_get_product_id_by_sku($parentASIN);
                        if (!$existing_product || $existing_product == $post_id) {
                    update_post_meta( $post_id, '_sku', $parentASIN );
                        }
                    }
                }
            }
            ########### Update product Variations ###########        
            if (get_option('product_variants_cron', true) == 1) {
                if (isset($variations->VariationDimensions) && !empty($variations->VariationDimensions)) {
                    wp_set_object_terms($post_id, 'variable', 'product_type');

                    $postData = [
                      'ID'           => $post_id,
                      'post_excerpt' => '',
                    ];
                    wp_update_post($postData);



                    $attributeChecks = [];
                    $attributes_data = [];
            
                    foreach ($variations->VariationDimensions as $attribute => $term_name) {
                        if (isset($term_name->Values, $term_name->DisplayName, $term_name->Name) && is_array($term_name->Values)) {
                            $values = array_map('trim', $term_name->Values);
                            $attr_label = $term_name->DisplayName;
                            $attr_slug = sanitize_title($term_name->Name);
            
                            $attributeChecks[$attr_slug] = sanitize_title($attr_label);
            
                            $attributes_data[] = array(
                                'name' => $attr_label, 
                                'slug' => $attr_slug, 
                                'options' => $values, 
                                'visible' => 1, 
                                'variation' => 1 
                            );
                        }
                    }
            
                    wc_update_product_attributes($post_id, $attributes_data);
            
                    $product = wc_get_product($post_id);

                    // Get the settings
                    $variation_limit = (int) get_option('ams_variation_limit', 5);
                    $remove_unavailable = get_option('ams_remove_unavailable_products') === 'Yes';

                    // Filter unavailable variations if setting is enabled
                    if ($remove_unavailable) {
                        $Variationlist = array_filter($Variationlist, function($variation) {
                            return isset($variation['stock_status']) && 
                                   $variation['stock_status'] !== 'outofstock' && 
                                   ($variation['regular_price'] > 0 || $variation['sale_price'] > 0);
                        });
                        // Reindex array after filtering
                        $Variationlist = array_values($Variationlist);
                    }

                    // Apply variation limit
                    $Variationlist = array_slice($Variationlist, 0, $variation_limit);

                    // Get existing variations to handle cleanup
                    $existing_variations = $product->get_children();
                    
                    // Collect SKUs of variations we'll keep
                    $valid_variation_skus = array_map(function($variation) {
                        return $variation['sku'];
                    }, $Variationlist);

                    // Remove variations that are not in our final list
                    foreach ($existing_variations as $variation_id) {
                        $variation_sku = get_post_meta($variation_id, '_sku', true);
                        if (!in_array($variation_sku, $valid_variation_skus)) {
                            wp_delete_post($variation_id, true);
                        }
                    }

                    foreach ($Variationlist as $SingleVariation) {
                        // Skip variations that don't meet our criteria
                        if (!isset($SingleVariation['sku']) || 
                            empty($SingleVariation['attributes']) || 
                            ($remove_unavailable && (
                                $SingleVariation['stock_status'] == 'outofstock' || 
                                ($SingleVariation['regular_price'] == 0 && $SingleVariation['sale_price'] == 0)
                            ))) {
                            continue;
                        }
            
                        $variation_post = array(
                            'post_title'  => $SingleVariation['post_title'],
                            'post_name'   => 'product-' . $post_id . '-variation-' . $SingleVariation['sku'],
                            'post_status' => 'publish',
                            'post_parent' => $post_id,
                            'post_type'   => 'product_variation',
                            'guid'        => $product->get_permalink()
                        );

                        $existing_variation = get_product_by_sku($SingleVariation['sku']);
                        if ($existing_variation !== null) {
                            $variation_post['ID'] = $variation_id = $existing_variation->get_id();
                            wp_update_post($variation_post);
                        } else {
                            $variation_id = wp_insert_post($variation_post);
                        }

                        $variation = new \WC_Product_Variation($variation_id);

                        if (count($SingleVariation['attributes']) > 0) {
                            foreach ($SingleVariation['attributes'] as $attribute => $term_name) {
                                $taxonomy = 'pa_' . $attribute;

                                // Ensure the taxonomy exists
                                if (!taxonomy_exists($taxonomy)) {
                                    continue; // Skip if the taxonomy doesn't exist
                                }

                                // If the term doesn't exist, create it
                                $maybe_term = term_exists($term_name, $taxonomy);
                                if (!$maybe_term) {
                                    $inserted = wp_insert_term($term_name, $taxonomy);
                                    if (is_wp_error($inserted)) {
                                        continue; // Skip if we fail to create
                                    }
                                    $maybe_term = $inserted;
                                }

                                // Fetch the term object
                                $term_obj = get_term_by('name', $term_name, $taxonomy);
                                if (!$term_obj) {
                                    continue; // Safety check
                                }

                                // Assign this term to the parent product
                                wp_set_object_terms($post_id, [$term_obj->term_id], $taxonomy, true);

                                // Assign the attribute to the variation
                                update_post_meta($variation_id, 'attribute_' . $taxonomy, $term_obj->slug);
                            }
                        }

                        if (!empty($SingleVariation['sku'])) {
                            $current_variation_sku = $variation->get_sku();
                            if ($current_variation_sku !== $SingleVariation['sku']) {
                                $existing_product = wc_get_product_id_by_sku($SingleVariation['sku']);
                                if (!$existing_product || $existing_product == $variation->get_id()) {
                            $variation->set_sku($SingleVariation['sku']);
                                }
                            }
                        }

                        if (empty($SingleVariation['sale_price'])) {
                            $variation->set_price($SingleVariation['regular_price'] ?? '');
                        } else {
                            $variation->set_price($SingleVariation['sale_price'] ?? '');
                            $variation->set_sale_price($SingleVariation['sale_price'] ?? '');
                        }
                        $variation->set_regular_price($SingleVariation['regular_price'] ?? '');

                        if (!empty($SingleVariation['stock_status'])) {
                            $variation->set_stock_status($SingleVariation['stock_status']);
                        } else {
                            $variation->set_manage_stock(false);
                        }




                        // Product variant image gallery
                        if (!empty($SingleVariation['product_image_gallery'])) {
                            // Remove existing thumbnail
                            $existing_thumbnail_id = get_post_thumbnail_id($variation_id);
                            if ($existing_thumbnail_id) {
                                delete_post_thumbnail($variation_id);
                                wp_delete_attachment($existing_thumbnail_id, true);
                            }

                            // Remove existing gallery
                            $existing_gallery = get_post_meta($variation_id, '_product_image_gallery', true);
                            if ($existing_gallery) {
                                $existing_gallery_ids = explode(',', $existing_gallery);
                                foreach ($existing_gallery_ids as $gallery_image_id) {
                                    wp_delete_attachment($gallery_image_id, true);
                                }
                                delete_post_meta($variation_id, '_product_image_gallery');
                            }

                            // Reset Product Gallery Meta
                            delete_post_meta($variation_id, '_product_image_gallery');
                            // Apply image limit before processing
                            $limited_gallery = array_slice($SingleVariation['product_image_gallery'], 0, $image_limit);

                            if ('Yes' === get_option('ams_remote_amazon_images')) {
                                $gallery_image_urls = [];
                                
                                // Process gallery images
                                foreach ($limited_gallery as $attachment) {
                                    $gallery_image_urls[] = esc_url($attachment);
                                }

                                // Set gallery images
                                if (!empty($gallery_image_urls)) {
                                    // Set gallery
                                    attach_product_thumbnail_url_api($variation_id, $gallery_image_urls, 1);
                                    
                                    // Set featured image
                                    if (!empty($gallery_image_urls[0])) {
                                        attach_product_thumbnail_url_api($variation_id, $gallery_image_urls[0], 0);
                                    }
                                }
                            } else {
                                $gallery_image_ids = [];
                                
                                // Process gallery images using the limited gallery
                                foreach ($limited_gallery as $attachment) {
                                    $attachment_id = attach_product_thumbnail_api($variation_id, $attachment, 1);
                                    if ($attachment_id) {
                                        $gallery_image_ids[] = $attachment_id;
                                    }
                                }

                                // Set gallery images
                                if (!empty($gallery_image_ids)) {
                                    // Set gallery
                                    update_post_meta($variation_id, '_product_image_gallery', implode(',', $gallery_image_ids));
                                    
                                    // Set featured image
                                    if (!empty($gallery_image_ids[0])) {
                                        set_post_thumbnail($variation_id, $gallery_image_ids[0]);
                                    }
                                }
                            }
                        }

                        $variation->set_weight('');
                        $variation->save();
                    }
                }
            }
            ########### Update product Variations ###########


            // Update Product Review
            if ('1' == get_option('enable_amazon_review', true) && '1' == get_option('product_review_cron', true)) {
                // Get product URL
                $product_url = esc_url_raw($row->DetailPageURL);
                
                // Get user agent
                $user_agent = $this->user_agent();
                
                // Get product data
                $response_body = fetchAndValidateProductData($product_url, $user_agent, false);
                if (is_string($response_body) && !empty($response_body)) {
                    if (!class_exists('simple_html_dom')) {
                        require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
                    }
                    
                    $html = new \simple_html_dom();
                    $html->load($response_body);

                    // Check for broken page
                    $message = check_for_broken_page($response_body, $html);
                    if ($message === null) {
                        // Get review limit from settings
                        $review_limit = get_option('multiple_import_review_limit', 10);
                        
                        // Scrape the reviews
                        $reviews = scrape_amazon_reviews($html, $review_limit);

                        if (!empty($reviews)) {
                            // Get existing reviews
                            $existing_reviews = get_comments([
                                'post_id' => $post_id,
                                'type' => 'review',
                                'status' => 'approve'
                            ]);
                            
                            // Create array of existing review hashes
                            $existing_hashes = [];
                            foreach ($existing_reviews as $existing_review) {
                                $existing_hash = get_comment_meta($existing_review->comment_ID, 'review_hash', true);
                                if (!empty($existing_hash)) {
                                    $existing_hashes[$existing_hash] = $existing_review->comment_ID;
                                }
                            }

                            // Process each review
                            foreach ($reviews as $review_hash => $review) {
                                // Check if review exists
                                if (isset($existing_hashes[$review_hash])) {
                                    continue; // Skip existing reviews in cron
                                }

                                // Prepare comment data for new review
                                $commentdata = [
                                    'comment_post_ID' => $post_id,
                                    'comment_author' => $review['reviewer_name'],
                                    'comment_content' => $review['text'],
                                    'comment_date' => $review['date'],
                                    'comment_date_gmt' => get_gmt_from_date($review['date']),
                                    'comment_approved' => 1,
                                    'comment_type' => 'review',
                                    'user_id' => 0,
                                    'comment_author_email' => '',
                                    'comment_author_url' => '',
                                    'comment_author_IP' => '',
                                    'comment_agent' => ''
                                ];

                                // Insert new review
                                $comment_id = wp_insert_comment($commentdata);

                                // Add comment meta if insert successful
                                if ($comment_id) {
                                    add_comment_meta($comment_id, 'rating', $review['rating']);
                                    add_comment_meta($comment_id, 'review_hash', $review_hash);
                                    add_comment_meta($comment_id, 'verified', 1);
                                    add_comment_meta($comment_id, 'title', $review['title']);

                                    if (!empty($review['reviewer_image'])) {
                                        add_comment_meta($comment_id, 'reviewer_image', $review['reviewer_image']);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            // Update Product Review    

            update_post_meta( $post_id, 'ams_last_cron_update',date('Y-m-d H:i:s'));
            update_post_meta( $post_id, 'ams_last_cron_status',1);
            
            $Current = get_post_meta( $post_id, '_stock_status', true );

            if( get_option('product_out_of_stock_cron', true) == 1 ) {
                update_post_meta( $post_id, '_stock_status', $product_status );
            } else {
                // If stock management is disabled, still update stock status if product is available (has valid status)
                if ($product_status === 'instock') {
                    update_post_meta( $post_id, '_stock_status', 'instock' );
            } else {
                update_post_meta( $post_id, '_stock_status', $Current );
                }
            }
        } 

        clean_completed_woocommerce_actions();
        clean_all_actionscheduler_logs();
        
        @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] Update End (API): func=ImportProducts::ams_product_api_update POST_ID=' . $post_id . '; ASIN=' . $asin . '; status=success' . "\n", FILE_APPEND);
        return TRUE;

        @file_put_contents($__import_log_file, '[' . date('Y-m-d H:i:s') . '] Import End: func=ImportProducts::product_import ASIN=' . (isset($asin) ? $asin : '') . '; status=success' . "\n", FILE_APPEND);
        return TRUE;
    } catch (Throwable $th) {
        @file_put_contents($__import_log_file, '[' . date('Y-m-d H:i:s') . '] Import End: func=ImportProducts::product_import ASIN=' . (isset($asin) ? $asin : '') . '; status=failure; error=' . $th->getMessage() . "\n", FILE_APPEND);
        update_post_meta( $post_id, 'ams_last_cron_update',date('Y-m-d H:i:s'));
        update_post_meta( $post_id, 'ams_last_cron_status',0);
        return FALSE;
    }
}
// NO-API Update
public function product_update_request() {
    try {
        if (!wp_verify_nonce($_POST['nonce'], 'product_update_request')) die('Busted!');

        if (ams_plugin_license_status() === false) {
            echo "<script>console.log('Plugin license not activated');</script>";
            $license = sprintf(esc_html__('Activate License!', 'ams-wc-amazon'));
            echo wp_kses_post($license);
            wp_die();
        }

        $__log_file = plugin_dir_path(__FILE__) . '../import_verification.log';
        $post_id = sanitize_text_field($_POST['post_id']);
        $product_url = $_POST['product_url'];
        @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] Start: func=ImportProducts::product_update_request POST_ID=' . $post_id . "\n", FILE_APPEND);
        logImportVerification('Product URL: ', $product_url);

        if ('1' == get_post_meta($post_id, '_import_method', true)) {
            if (ams_plugin_license_status() === false) {
                echo "<script>console.log('Plugin license not activated');</script>";
                $license = sprintf(esc_html__('Activate License!', 'ams-wc-amazon'));
                echo wp_kses_post($license);
                wp_die();
            }
            $sku = $this->getSkuFromUrl($product_url);
            $response = $this->ams_product_api_update($post_id, $sku);

            if ($response) {
                echo 'Success';
                @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=ImportProducts::product_update_request POST_ID=' . $post_id . '; status=success; method=API' . "\n", FILE_APPEND);
            } else {
                echo 'Try again!';
                @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=ImportProducts::product_update_request POST_ID=' . $post_id . '; status=failure; method=API' . "\n", FILE_APPEND);
            }
            wp_die();
        }

        // If Added in CRON            
        $product_sku_cron = get_option('product_sku_cron', true);
        $product_tags_cron = get_option('product_tags_cron', true);
        $product_name_cron = get_option('product_name_cron', true);
        $product_price_cron = get_option('product_price_cron', true);
        $product_image_cron = get_option('product_image_cron', true);
        $product_review_cron = get_option('product_review_cron', true);
        $enable_amazon_review = get_option('enable_amazon_review', true);
        $product_variants_cron = get_option('product_variants_cron', true);
        $product_variant_image_cron = get_option('product_variant_image_cron', true);
        $product_category_cron = get_option('product_category_cron', true);
        $product_description_cron = get_option('product_description_cron', true);
        $product_out_of_stock_cron = get_option('product_out_of_stock_cron', true);

        // Get product data first time
        $user_agent = $this->user_agent();
        $response_body = fetchAndValidateProductData($product_url, $user_agent, false);

        if (is_string($response_body) && strlen($response_body)) {
            if (!class_exists('simple_html_dom')) {
                require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
            }

            $html = new \simple_html_dom();
            $html->load($response_body);

            // Check for broken page
            $message = check_for_broken_page($response_body, $html);
            if ($message !== null) {
                echo wp_kses_post($message);
                logImportVerification($message, null);
                @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=ImportProducts::product_update_request POST_ID=' . $post_id . '; status=failure; error=broken-page' . "\n", FILE_APPEND);
                wp_die();
            }

            // Extract Asin from product_url
            $asin = extractAsin($html, $product_url);
            if (empty($asin)) {
                die(esc_html__('ASIN not found!', 'ams-wc-amazon'));
            }

            // Get Parent ASIN from html
            $parentSku = $this->getParentSkuFromHtml($html);
            if (!empty($parentSku)) {
                logImportVerification('Valid parent SKU found: ', $parentSku);
            } else {
                logImportVerification('Failed to extract valid parent SKU', null);
            }

            // Check if product title exists, else abort
            $productTitle = extractAmazonProductTitle($html);
            if ($productTitle === false) {
                logImportVerification('Product Title Extraction Failed', null);
                @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=ImportProducts::product_update_request POST_ID=' . $post_id . '; status=failure; error=title-extraction' . "\n", FILE_APPEND);
                wp_die();
            }
            $title = html_entity_decode($productTitle, ENT_QUOTES | ENT_HTML5, 'UTF-8');

            // Get Product Category
            $product_category = $this->syncAndGetCategory($html);

            // Get Product Content
            $content = $this->fetchContentFromHtml($html);
            //echo '<pre>'; dd( $content ); echo '</pre>';
            //echo '<pre>'; dd( $content ); echo '</pre>'; exit;

            // Get Product Short Description
            $short_description = $this->fetchShortDescriptionFromHtml($html);

            // Get Product Additional Content
            $additional_description = $this->fetchAdditionalContentFromHtml($html);
            //echo '<pre>'; dd( $additional_description ); echo '</pre>'; exit;

            clean_completed_woocommerce_actions();
            clean_all_actionscheduler_logs();


            // Update Product SKU
            if ($product_sku_cron) {
                $asinElements = $html->find('#ASIN');
                $asin = !empty($asinElements) ? $asinElements[0]->value : '';

                if (empty($asin)) {
                    $elements = $html->find('input[name="ASIN.0"]');
                    $asin = !empty($elements) ? $elements[0]->value : '';
                }

                if (empty($asin)) {
                    $asin = $this->getSkuFromUrl($product_url);
                }
                update_post_meta($post_id, '_sku', $asin);
            }

            // Update Product title
            if ($product_name_cron) {
                $product_update = array(
                    'ID' => $post_id,
                    'post_title' => $title,
                    'post_name' => sanitize_title($title)
                );
                wp_update_post($product_update);
                logImportVerification('Product Title Updated: ', $title);
            }
            // echo '<pre>'; dd( $title ); echo '</pre>';exit;

            // Update Product main content
            if ($product_description_cron) {
                if (!empty(trim($content))) {
                    $product_update = array(
                        'ID' => $post_id,
                        'post_content' => $content
                    );
                    wp_update_post($product_update);
                    logImportVerification('Product Content Updated!', null);
                }
            }

            // Update Product Category
            if ($product_category_cron) {
                $product_category = $this->syncAndGetCategory($html);
                if (!empty($product_category)) {
                    wp_set_object_terms($post_id, $product_category, 'product_cat');
                    logImportVerification('Product Category Updated: ', $product_category);
                }
            }


            // Update brand name
            $brandElement = $html->find('a#bylineInfo', 0) 
                ?: $html->find('div#bylineInfo_feature_div', 0) 
                ?: $html->find('div#bondByLine_feature_div', 0);

            if ($brandElement) {
                $rawBrandName = trim($brandElement->plaintext);

                $brandName = str_replace(array('Visit the', 'Store'), '', $rawBrandName);

                // Trim any extra spaces
                $brandName = trim($brandName);

                logImportVerification('Brand: ' . $brandName);

                $brandTerm = term_exists($brandName, 'product_brand');

                if (!$brandTerm) {
                    // If the brand doesn't exist, create it
                    $brandTerm = wp_insert_term($brandName, 'product_brand');
                }

                if (is_wp_error($brandTerm)) {
                    logImportVerification('Error creating brand term: ' . $brandTerm->get_error_message());
                } else {
                    // Get the term ID
                    $brandTermId = isset($brandTerm['term_id']) ? $brandTerm['term_id'] : $brandTerm;

                    $productId = $post_id;

                    // Assign the brand to the product
                    wp_set_object_terms($productId, intval($brandTermId), 'product_brand');

                    update_post_meta($productId, '_product_brand', $brandName);

                    logImportVerification('Brand assigned to product successfully.');

                    // Add brand as an attribute
                    $product = wc_get_product($productId);
                    if ($product) {
                        $attributes = $product->get_attributes();

                        // Check if the "Brand" attribute already exists
                        $existingAttribute = false;
                        foreach ($attributes as $key => $attribute) {
                            if ($attribute->get_name() === 'Brand') {
                                $attributes[$key]->set_options([$brandName]); // Update the brand value
                                $existingAttribute = true;
                                break;
                            }
                        }

                        // If the attribute does not exist, create it
                        if (!$existingAttribute) {
                            $brandAttribute = new \WC_Product_Attribute();
                            $brandAttribute->set_name('Brand'); // Attribute name
                            $brandAttribute->set_options([$brandName]); // Set the scraped brand name
                            $brandAttribute->set_visible(true); // Make visible on product page
                            $brandAttribute->set_variation(false); // Not used for variations
                            $attributes[] = $brandAttribute; // Add new attribute to the list
                        }

                        // Save the updated attributes back to the product
                        $product->set_attributes($attributes);
                        $product->save(); // Save changes

                        logImportVerification('Brand attribute successfully added to the product.');
                    } else {
                        logImportVerification('Failed to retrieve product for adding brand attribute.');
                    }
                }
            } else {
                logImportVerification('Brand not found in the provided HTML.');
            }
            // Update brand name


            // Update the GTIN, UPC, EAN, or ISBN code
            $upcElement = $html->find('div#productDetails_expanderTables_depthLeftSections', 0);
            if ($upcElement) {
                $upcCode = ''; // Initialize variable

                // Iterate through table rows to find GTIN, UPC, EAN, or ISBN
                foreach ($upcElement->find('table.prodDetTable tr') as $row) {
                    $header = $row->find('th', 0); // Get the header cell
                    $value = $row->find('td', 0); // Get the value cell

                    if ($header && $value) {
                        $headerText = trim($header->plaintext);
                        $valueText = trim($value->plaintext);

                        // Check for GTIN, UPC, EAN, or ISBN
                        if (stripos($headerText, 'UPC') !== false || stripos($headerText, 'GTIN') !== false || stripos($headerText, 'EAN') !== false || stripos($headerText, 'ISBN') !== false) {
                            $upcCode = $valueText; // Extract the value
                            break; // Exit loop once found
                        }
                    }
                }

                if (!empty($upcCode)) {
                    // Save the value to the default WooCommerce GTIN/UPC/EAN/ISBN fields
                    update_post_meta($post_id, '_gtin', $upcCode); // GTIN field
                    update_post_meta($post_id, '_upc', $upcCode);  // UPC field
                    update_post_meta($post_id, '_ean', $upcCode);  // EAN field
                    update_post_meta($post_id, '_isbn', $upcCode); // ISBN field
                }
            }
            // Update the GTIN, UPC, EAN, or ISBN code


            // Check remote amazon images.
            if($product_image_cron) {
                // Set product feature image.
                $gallery = $this->fetchImagesFromHtml($html);

                $image = array_shift($gallery);
                if( $image ) {
                    // Remove featured image and url.
                    delete_product_images($post_id);
                    reset_product_thumbnail_url($post_id, $flag=0);
                }
                
                if( count($gallery) > 0 ) {
                    // Remove product gallery images and url.
                    delete_product_gallery_images($post_id);
                    reset_product_thumbnail_url($post_id, $flag=1);
                }
                
                if ( 'Yes' === get_option( 'ams_remote_amazon_images' ) ) {
                    // Set featured image url
                    if( $image ) {
                        attach_product_thumbnail_url( $post_id, $image, 0 );
                    }
                    // Set featured image gallary
                    if( count($gallery) > 0 ) {
                        attach_product_thumbnail_url( $post_id, $gallery, 1 );
                    }
                } else {
                    // Set featured image url
                    if( $image ) {
                        attach_product_thumbnail($post_id, $image, 0);
                    }
                    // Set featured image gallary
                    if( count($gallery) > 0 ) {
                        foreach( $gallery as $image ) {
                            // Set gallery image.
                            attach_product_thumbnail( $post_id, $image, 1 );
                        }
                    }
                }
            }

            // Update Product Review
            if ($enable_amazon_review && $product_review_cron) {
                // Get review limit from settings
                $review_limit = get_option('multiple_import_review_limit', 10);
                
                // Scrape the reviews
                $reviews = scrape_amazon_reviews($html, $review_limit);
                
                logImportVerification("Processed " . count($reviews) . " reviews");

                if (!empty($reviews) && isset($post_id)) {
                    // Get existing reviews
                    $existing_reviews = get_comments([
                        'post_id' => $post_id,
                        'type' => 'review',
                        'status' => 'approve'
                    ]);
                    
                    // Create array of existing review hashes
                    $existing_hashes = [];
                    foreach ($existing_reviews as $existing_review) {
                        $existing_hash = get_comment_meta($existing_review->comment_ID, 'review_hash', true);
                        if (!empty($existing_hash)) {
                            $existing_hashes[$existing_hash] = $existing_review->comment_ID;
                        }
                    }

                    // Initialize rating totals
                    $rating_sum = 0;
                    $rating_count = 0;

                    // Process each review
                    foreach ($reviews as $review_hash => $review) {
                        // Skip if review already exists
                        if (isset($existing_hashes[$review_hash])) {
                            logImportVerification("Skipping duplicate review: " . $review['title']);
                            continue;
                        }

                        // Prepare comment data
                        $commentdata = [
                            'comment_post_ID' => $post_id,
                            'comment_author' => $review['reviewer_name'],
                            'comment_content' => $review['text'],
                            'comment_date' => $review['date'],
                            'comment_date_gmt' => get_gmt_from_date($review['date']),
                            'comment_approved' => 1,
                            'comment_type' => 'review',
                            'user_id' => 0
                        ];

                        // Insert the comment
                        $comment_id = wp_insert_comment($commentdata);

                        if ($comment_id) {
                            // Add all the comment meta
                            add_comment_meta($comment_id, 'rating', $review['rating']);
                            add_comment_meta($comment_id, 'review_hash', $review_hash);
                            add_comment_meta($comment_id, 'verified', 1);
                            add_comment_meta($comment_id, 'title', $review['title']);

                            if (!empty($review['reviewer_image'])) {
                                add_comment_meta($comment_id, 'reviewer_image', $review['reviewer_image']);
                            }

                            $rating_sum += floatval($review['rating']);
                            $rating_count++;

                            logImportVerification("Added review: " . $review['title'] . " with ID: " . $comment_id);
                        }
                    }

                    // Update product rating if we added any new reviews
                    if ($rating_count > 0) {
                        $product = wc_get_product($post_id);
                        if ($product) {
                            // Get actual count of approved reviews
                            $actual_review_count = get_comments([
                                'post_id' => $post_id,
                                'type' => 'review',
                                'status' => 'approve',
                                'count' => true
                            ]);

                            // Calculate actual rating sum
                            $actual_rating_sum = 0;
                            $product_reviews = get_comments([
                                'post_id' => $post_id,
                                'type' => 'review',
                                'status' => 'approve'
                            ]);

                            foreach ($product_reviews as $review) {
                                $rating = get_comment_meta($review->comment_ID, 'rating', true);
                                $actual_rating_sum += floatval($rating);
                            }

                            // Calculate new average
                            $new_average = $actual_rating_sum / $actual_review_count;

                            // Update all rating meta
                            update_post_meta($post_id, '_wc_average_rating', round($new_average, 2));
                            update_post_meta($post_id, '_wc_rating_count', $actual_review_count);
                            update_post_meta($post_id, '_wc_review_count', $actual_review_count);
                            update_post_meta($post_id, '_wc_rating_sum', $actual_rating_sum);

                            // Clear all relevant caches
                            delete_transient('wc_product_reviews_' . $post_id);
                            delete_transient('wc_average_rating_' . $post_id);
                            wp_cache_delete($post_id, 'product');
                            
                            if (function_exists('wc_delete_product_transients')) {
                                wc_delete_product_transients($post_id);
                            }

                            logImportVerification("Updated product rating. New average: " . round($new_average, 2));
                        }
                    }

                    logImportVerification("Completed review import. Added " . $rating_count . " new reviews");
                }
            }
            // Update Product Review


            // Get Product attributes
            $attributes = $this->getProductAttributeFromHtml($html);
            //echo '<pre>'; dd( $attributes ); echo '</pre>'; exit;


            // Set variable if the product is simple despite attributes
            if (count($attributes) > 0) {
                // Check product type
                $terms = wp_get_object_terms($post_id, 'product_type');
                if (empty($terms) || $terms[0]->slug === 'simple') {
                    // Convert to variable
                    wp_set_object_terms($post_id, 'variable', 'product_type');
                    
                    // Clear cache
                    clean_post_cache($post_id);
                    wc_delete_product_transients($post_id);
                }
            }

            //Run if variable//
            if (count($attributes) > 0) {

                if($parentSku) {
                    update_post_meta($post_id, '_sku', $parentSku);
                }
                //echo '<pre>'; dd( $parentSku ); echo '</pre>'; exit;

                // Delete product short description
                $postData = array(
                    'ID' => $post_id,
                    'post_excerpt' => ''
                );
                wp_update_post($postData);

                // Update Additional Description
                if (!empty($additional_description)) {
                    update_post_meta($post_id, '_ams_additional_information', $additional_description);
                    logImportVerification('Additional description updated.', null);
                }

                $skus = $imported_skus = $product_variations = [];

                // Get all variants based on the SKUs found
                $all_skus = $this->getSkusFromHtml($html);
                //echo '<pre>'; print_r($all_skus); echo '</pre>';

                $variation_ids = $this->getProductFirstVariationFromHtml($html, $parentSku, $product_url, $all_skus);
                //echo '<pre>'; print_r($variation_ids); echo '</pre>';exit;

                // variations to process
                $variation_limit = get_option('ams_variation_limit', 5);

                // Update Variants
                if ($product_variants_cron && count($attributes) > 0) {

                    // Check if there are variation IDs:
                    if (!empty($variation_ids) && count($variation_ids) > 0) {

                    // Apply the dynamic variations to process
                    $variation_ids = array_slice($variation_ids, 0, $variation_limit);
                    //echo '<pre>'; print_r($variation_ids); echo '</pre>';

                    // Determine the preferred URL-generation function based solely on product title
                    $preferred_function = null;
                    $first_variation_processed = false;

                    foreach ($variation_ids as $variation_id) {
                        if (in_array($variation_id, $imported_skus)) {
                            continue;
                        }
                        array_push($imported_skus, $variation_id);

                        // For the first variation, decide which function to use based solely on product title.
                        if (!$first_variation_processed) {
                            // Try using function 1 with regular curl first
                            $test_url = generate_amazon_url_1($product_url, $variation_id);
                            $userAgent = getAlternatingBool();
                            $test_content = $this->getContentUsingCurl($test_url, $userAgent);
                            $test_html = new \simple_html_dom();
                            $test_html->load($test_content);
                            
                            // If regular curl fails or no attributes found, try scraping
                            if (!$test_content || count($this->getProductAttributeFromHtml($test_html)) == 0) {
                                $test_content = executeScrapingService($test_url, true);
                                $test_html = new \simple_html_dom();
                                $test_html->load($test_content);
                            }
                            
                            // Check if product title exists; if not, choose function 2
                            $productTitle = extractAmazonProductTitle($test_html);
                            if ($productTitle === false) {
                                $preferred_function = 2;
                                //echo "<pre>Preferred function set to 2 (function 1 failed to extract product title).</pre>";
                                logImportVerification('function 1 failed to extract product title');
                            } else {
                                $title = html_entity_decode($productTitle, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                                $preferred_function = 1;
                                //echo "<pre>Preferred function set to 1 based on product title: {$title}</pre>";
                                logImportVerification('Preferred function set to 1 based on product title: {$title}');
                            }
                            $test_html->clear();
                            $first_variation_processed = true;
                        }

                        // Use the preferred function to generate the base URL.
                        if ($preferred_function === 1) {
                            $base_url = generate_amazon_url_1($product_url, $variation_id);
                        } else {
                            $base_url = generate_amazon_url_2($product_url, $variation_id);
                        }
                        
                        //echo "<pre>Processing Variation ID: {$variation_id} using function {$preferred_function}. Base URL: {$base_url}</pre>";

                        // First try with regular curl
                        $userAgent = getAlternatingBool();
                        $content = $this->getContentUsingCurl($base_url, $userAgent);
                        $loop_html = new \simple_html_dom();
                        $loop_html->load($content);
                        
                        // Check if we need to use scraping service
                        if (!$content || count($this->getProductAttributeFromHtml($loop_html)) == 0) {
                            $content = executeScrapingService($base_url, true);
                            $loop_html = new \simple_html_dom();
                            $loop_html->load($content);
                        }

                        $productPrice = $this->fetchPriceFromHtml($loop_html);

                        // If no ppd div found, retry twice with proxy
                        $retry = 2;
                        while (isset($productPrice['search_area']) && $productPrice['search_area'] == 'entire HTML' && $retry > 0) {
                            $content = executeScrapingService($base_url, true);
                            $loop_html = new \simple_html_dom();
                            $loop_html->load($content);
                            $productPrice = $this->fetchPriceFromHtml($loop_html);
                            $retry--;
                        }

                        $regular_price = isset($productPrice['final_prices']['regular_price'])
                            ? $productPrice['final_prices']['regular_price']
                            : (isset($productPrice['regular_price']) ? $productPrice['regular_price'] : 0);
                        $sale_price = isset($productPrice['final_prices']['sale_price'])
                            ? $productPrice['final_prices']['sale_price']
                            : (isset($productPrice['sale_price']) ? $productPrice['sale_price'] : 0);

                        // Add ScraperAPI retry if prices are zero
                        $isUpdate = true;
                        if ($regular_price == 0) {
                            $scraperapi = get_scraping_services_config()['scraperapi'];
                            if (!empty($scraperapi['api_key']) && ($isUpdate ? $scraperapi['on_update'] : $scraperapi['is_active'])) {
                                $content = call_user_func($scraperapi['execute'], $base_url, $scraperapi['api_key']);
                                if ($content) {
                                    $loop_html = new \simple_html_dom();
                                    $loop_html->load($content);
                                    $productPrice = $this->fetchPriceFromHtml($loop_html);
                                    
                                    // Update prices with new values
                                    $regular_price = isset($productPrice['final_prices']['regular_price'])
                                        ? $productPrice['final_prices']['regular_price']
                                        : (isset($productPrice['regular_price']) ? $productPrice['regular_price'] : 0);
                                    $sale_price = isset($productPrice['final_prices']['sale_price'])
                                        ? $productPrice['final_prices']['sale_price']
                                        : (isset($productPrice['sale_price']) ? $productPrice['sale_price'] : 0);
                                }
                            } else {
                                //echo '<pre>Warning: ScraperAPI service is disabled. Enable it for better price scraping.</pre>';
                                logImportVerification('Warning: ScraperAPI service is disabled. Enable it for better price scraping.');
                            }
                        }

                        $currency = $this->fetchCurrencyFromHtml($loop_html);
                        logImportVerification('Currency: ', $currency);

                        if ($regular_price > 0 || $sale_price > 0) {
                            $product_status = 'instock';
                        } else {
                            $product_status = check_product_stock_status($loop_html);
                            if ($product_status === 'instock') {
                                $product_status = 'outofstock';
                                logImportVerification('Status changed to outofstock due to zero prices');
                            }
                        }
                        logImportVerification('Final stock status: ', $product_status);

                        $quantity = 0;
                        if ($qty = $loop_html->find('#availability span', 0)) {
                            $quantity = $this->parseNumberFromString($qty->text());
                        }
                        logImportVerification('Quantity: ', $quantity);

                        $short_description = $this->fetchVariationContentFromHtml($loop_html);
                        $additional_description = $this->fetchAdditionalContentFromHtml($loop_html);

                        if ($product_variant_image_cron) {
                            $v_gallery = $this->fetchImagesFromHtml($loop_html);
                            $image_limit = get_option('ams_variation_image_limit', 5);
                            if ($image_limit > 0) {
                                $v_gallery = array_slice($v_gallery, 0, $image_limit);
                            }
                        }

                        $attributes = $this->getProductAttributeFromHtml($loop_html);
                        
                        // Retry logic if attributes are empty (same as import function)
                        if (empty($attributes)) {
                            for ($retry = 1; $retry <= 3; $retry++) {
                                sleep(rand(2, 4)); // Random delay 2-4 seconds
                                
                                // Try with different user agent
                                $userAgent = getAlternatingBool();
                                $retry_content = $this->getContentUsingCurl($base_url, $userAgent);
                                
                                if ($retry_content) {
                                    $retry_html = new \simple_html_dom();
                                    $retry_html->load($retry_content);
                                    $attributes = $this->getProductAttributeFromHtml($retry_html);
                                    $retry_html->clear();
                                    
                                    if (!empty($attributes)) {
                                        $loop_html->clear();
                                        $loop_html = $retry_html;
                                        break;
                                    }
                                }
                            }
                        }
                        
                        // Fix identical variants by assigning different selected values based on variation index
                        static $used_attributes = array();
                        
                        foreach ($attributes as &$attribute) {
                            if (isset($attribute['options']) && is_array($attribute['options']) && count($attribute['options']) > 1) {
                                $attribute_key = $attribute['slug'];
                                
                                // Initialize tracking for this attribute if not exists
                                if (!isset($used_attributes[$attribute_key])) {
                                    $used_attributes[$attribute_key] = array();
                                }
                                
                                // Find the first unused option
                                $selected_option = null;
                                foreach ($attribute['options'] as $option) {
                                    if (!in_array($option, $used_attributes[$attribute_key])) {
                                        $selected_option = $option;
                                        $used_attributes[$attribute_key][] = $option;
                                        break;
                                    }
                                }
                                
                                // If all options are used, reset and use the first one
                                if ($selected_option === null) {
                                    $used_attributes[$attribute_key] = array();
                                    $selected_option = $attribute['options'][0];
                                    $used_attributes[$attribute_key][] = $selected_option;
                                }
                                
                                $attribute['selected'] = $selected_option;
                            }
                        }

                        $product_variations[] = array(
                            'sku'                     => $variation_id,
                            'stock_qty'               => $quantity,
                            'stock_status'            => $product_status,
                            'regular_price'           => $regular_price,
                            'sale_price'              => $sale_price,
                            'attributes'              => $attributes,
                            'description'             => $short_description,
                            'product_image_gallery'   => isset($v_gallery) ? $v_gallery : array(),
                            'additional_description'  => $additional_description,
                        );
                    }
                }

                    //echo '<pre>'; dd( $product_variations ); echo '</pre>'; exit;
                    //echo '<pre>'; dd( $product_variations ); echo '</pre>';
                    if (count($product_variations) > 0) {
                        wc_update_product_variations($post_id, $product_variations, $parentSku);
                    }
                }

                logImportVerification('Variable product updated!', null);
            }

            //Run if simple//
            else {
                // Initialize the $product object
                $product = wc_get_product($post_id);

                // Product price Start
                $productPrice = $this->fetchPriceFromHtml($html);
                $regular_price = isset($productPrice['regular_price']) ? $productPrice['regular_price'] : 0;
                $sale_price = isset($productPrice['sale_price']) ? $productPrice['sale_price'] : 0;
                logImportVerification('Regular price: ', $regular_price);
                logImportVerification('Sale price: ', $sale_price);

                // Currency
                $currency = $this->fetchCurrencyFromHtml($html);
                logImportVerification('Currency: ', $currency);

                // Determine initial stock status based on price availability
                if ($regular_price > 0 || $sale_price > 0) {
                    $product_status = 'instock';
                } else {
                    // If no prices found, proceed with the original stock check
                    $product_status = check_product_stock_status($html);
                }

                logImportVerification('Initial Stock status: ', $product_status);

                // Out Of Stock check moved to the beginning
                if ($product_out_of_stock_cron) {
                    if ($product_status == 'outofstock' && 'Yes' == get_option('ams_remove_unavailable_products')) {
                        removeProductIfNotExists($post_id);
                        logImportVerification('Product removal: ', "Product with ID $post_id has been processed for removal as it is out of stock.");
                        wp_die(
                            esc_html__('OutOfStock!', 'ams-wc-amazon'),
                            ['response' => 200]
                        );
                    }

                    // Quantity
                    $quantity = 0;
                    $qty = $html->find('#availability span', 0);
                    if ($qty) {
                        $quantity = $this->parseNumberFromString($qty->text());
                    }

                    if ($quantity > 0) {
                        update_post_meta($post_id, '_stock', $quantity);
                        update_post_meta($post_id, '_manage_stock', 'yes');
                        update_post_meta($post_id, '_stock_status', $product_status);
                    } else {
                        update_post_meta($post_id, '_stock', '');
                        update_post_meta($post_id, '_manage_stock', 'no');
                        update_post_meta($post_id, '_stock_status', $product_status);
                    }
                }

                // Update price meta
                update_post_meta($post_id, '_regular_price', $regular_price);
                if ($sale_price > 0 && $sale_price < $regular_price) {
                    update_post_meta($post_id, '_sale_price', $sale_price);
                    update_post_meta($post_id, '_price', $sale_price);
                } else {
                    delete_post_meta($post_id, '_sale_price');
                    update_post_meta($post_id, '_price', $regular_price);
                }

                // Update Short Description
                if (!empty($short_description)) {
                    $product->set_short_description($short_description);
                    logImportVerification('Short description updated.', null);
                    $product->save();
                }

                // Update Additional Description
                if (!empty($additional_description)) {
                    update_post_meta($post_id, '_ams_additional_information', $additional_description);
                    logImportVerification('Additional description updated.', null);
                }


                // Product images + feature image
                if ($product_image_cron) {
                    // Get the image limit from plugin settings
                    $image_limit = get_option('ams_variation_image_limit', 5);
                    
                    // Set product feature image.
                    $gallery = $this->fetchImagesFromHtml($html);
                    
                    // Apply the limit to the gallery
                    $gallery = array_slice($gallery, 0, $image_limit);
                    
                    $image = array_shift($gallery);
                    $use_remote_images = ('Yes' === get_option('ams_remote_amazon_images'));
                    
                    // Always remove existing images and URLs
                    if ($image) {
                        // Remove featured image and url.
                        delete_product_images($post_id);
                        reset_product_thumbnail_url($post_id, $flag = 0);
                    }
                    
                    if (count($gallery) > 0) {
                        // Remove product gallery images and url.
                        delete_product_gallery_images($post_id);
                        reset_product_thumbnail_url($post_id, $flag = 1);
                    }
                    
                    if ($use_remote_images) {
                        // Set featured image url
                        if ($image) {
                            attach_product_thumbnail_url($post_id, $image, 0);
                        }
                        // Set featured image gallery
                        if (count($gallery) > 0) {
                            attach_product_thumbnail_url($post_id, $gallery, 1);
                        }
                        // Remove any locally stored images
                        delete_local_product_images($post_id);
                    } else {
                        // Set featured image
                        if ($image) {
                            attach_product_thumbnail($post_id, $image, 0);
                        }
                        // Set featured image gallery
                        foreach ($gallery as $gallery_image) {
                            // Set gallery image.
                            attach_product_thumbnail($post_id, $gallery_image, 1);
                        }
                        // Remove any stored image URLs
                        delete_product_image_urls($post_id);
                    }
                }

                update_post_meta($post_id, '_product_currency', $currency);
                update_post_meta($post_id, 'ams_last_cron_update', date('Y-m-d H:i:s'));
                update_post_meta($post_id, 'ams_last_cron_status', 0);

                logImportVerification('Simple product updated!', null);
            }

            update_post_meta($_POST['post_id'], 'ams_last_cron_update', date('Y-m-d H:i:s'));
            update_post_meta($_POST['post_id'], 'ams_last_cron_status', 1);

            logImportVerification('Product Updated!',null);
            echo esc_html__('Success', 'ams-wc-amazon');
            wc_delete_product_transients($post_id);
            clean_completed_woocommerce_actions();
            clean_all_actionscheduler_logs();
        } else {
            error_log("Failed to fetch product data");
            echo 'Failed to fetch product data';
        }
    } catch (\Throwable $th) {
        error_log("Exception occurred: " . $th->getMessage());
        echo 'Try again!';
        update_post_meta($_POST['post_id'], 'ams_last_cron_update', date('Y-m-d H:i:s'));
        update_post_meta($_POST['post_id'], 'ams_last_cron_status', 0);
    }
    
    @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=ImportProducts::product_update_request POST_ID=' . $_POST['post_id'] . '; status=success; method=NO-API' . "\n", FILE_APPEND);
    wp_die();        
}
    
public function get_string_between($string, $start, $end){
    $string = ' ' . $string;
    $ini = strpos($string, $start);
    if ($ini == 0) return '';
    $ini += strlen($start);
    $len = strpos($string, $end, $ini) - $ini;
    return substr($string, $ini, $len);
}

public function parseNumberFromString($numberString) {
    // Handle multilingual stock status text
    $stockPhrases = [
        // In stock phrases (return -1 as available stock indicator)
        'op voorraad' => -1,           // Dutch: In stock
        'in stock' => -1,             // English: In stock  
        'auf lager' => -1,            // German: In stock
        'en stock' => -1,             // French: In stock
        'en existencias' => -1,       // Spanish: In stock
        'disponibile' => -1,          // Italian: Available
        'em estoque' => -1,           // Portuguese: In stock
        'na skladě' => -1,            // Czech: In stock
        'w magazynie' => -1,          // Polish: In stock
        
        // Out of stock phrases (return 0)
        'niet op voorraad' => 0,       // Dutch: Out of stock
        'tijdelijk niet op voorraad' => 0, // Dutch: Temporarily out of stock
        'momenteel niet beschikbaar' => 0,  // Dutch: Currently unavailable
        'out of stock' => 0,           // English: Out of stock
        'nicht verfügbar' => 0,        // German: Not available
        'rupture de stock' => 0,       // French: Out of stock
        'agotado' => 0,               // Spanish: Sold out
        'non disponibile' => 0,       // Italian: Not available
        'fora de estoque' => 0,       // Portuguese: Out of stock
    ];
    
    $lowerString = strtolower(trim($numberString));
    
    // Check for stock status phrases first
    foreach ($stockPhrases as $phrase => $quantity) {
        if (stripos($lowerString, $phrase) !== false) {
            return $quantity;
        }
    }
    
    // If no stock phrase found, try to extract actual numbers
    $numberString = preg_replace('/\s*/m', '', $numberString);
    $numberString = preg_replace('#[^0-9\.,]#', '', $numberString);
    $numberString = preg_replace('/[^0-9]+/', '', $numberString);
    $number = (int) $numberString;
    
    // If we found a number, return it; otherwise return 0
    return $number > 0 ? $number : 0;
}

// remove currency symbols extra spaces and comma dot from price except decimal separater
public function parsePrice($priceString) {
    // Remove whitespace
    $pattern = '/\s*/m';
    $replace = '';
    $priceString = preg_replace($pattern, $replace, $priceString);
    
    // Handle different currency formats
    $originalString = $priceString;
    
    // Remove currency symbols but keep numbers, dots, and commas
    $priceString = preg_replace('#[^0-9\.,]#', '', $priceString);
    
    if (empty($priceString)) {
        return 0;
    }
    
    // Detect format based on patterns
    $price = 0;
    
    // Brazilian format: R$ 123,45 or 1.234,56
    if (preg_match('/^\d{1,3}(?:\.\d{3})*,\d{2}$/', $priceString)) {
        $priceString = str_replace('.', '', $priceString); // Remove thousands separator
        $priceString = str_replace(',', '.', $priceString); // Convert decimal separator
        $price = (float)$priceString;
    }
    // US format: $123.45 or 1,234.56
    else if (preg_match('/^\d{1,3}(?:,\d{3})*\.\d{2}$/', $priceString)) {
        $priceString = str_replace(',', '', $priceString); // Remove thousands separator
        $price = (float)$priceString;
    }
    // European format: €123,45 or 1.234,56
    else if (preg_match('/^\d+,\d{2}$/', $priceString)) {
        $priceString = str_replace(',', '.', $priceString);
        $price = (float)$priceString;
    }
    // Fallback: original logic (only for strings that might contain prices)
    else {
        // Only use fallback if the string contains some price-like indicators
        if (preg_match('/[\$£€¥₹]/', $originalString) || preg_match('/\d+[.,]\d{2}/', $originalString)) {
    $pattern = '/[.,]\d{1,2}$/';
    $hasDecimal = preg_match_all($pattern, $priceString);
    $priceString = preg_replace('/[^0-9]+/', '', $priceString);
    if($hasDecimal) {
        $priceString = number_format(substr_replace($priceString, '.', -2, 0),2);
                $price = (float)$priceString;
            } else {
        $price = (float)$priceString;
    }
        } else {
            $price = 0;
        }
    }
    return $price;
}
public function parsePricenew($priceString) {
    $pattern = '/\s*/m';
    $replace = '';
    $priceString = preg_replace($pattern, $replace, $priceString);
    $priceString = preg_replace('#[^0-9\.,]#', '', $priceString);
    $pattern = '/[.,]\d{1,2}$/';
    $hasDecimal = preg_match_all($pattern, $priceString);
    $priceString = preg_replace('/[^0-9]+/', '', $priceString);
    if($hasDecimal) {
        $priceString = number_format(substr_replace($priceString, '.', -2, 0),2);
        $price = $priceString;
    }else {
        $price = (float)$priceString;
    }
    //Product will be added to site cart when will checkout than add order list for DropShip
    $percentage_profit = (float) get_option('ams_percentage_profit');
    if (strtolower(get_option('ams_buy_action_btn')) === strtolower('dropship')) {
        if (!empty($price)) {
            $profit = ($price / 100) * $percentage_profit;
            $price = $price + $profit;
        }
    }
    return $price;
}

public function parsePriceStringnew($priceString) {

    $price = (float)$priceString;
    
    //Product will be added to site cart when will checkout than add order list for DropShip
    $percentage_profit = (float) get_option('ams_percentage_profit');
    if (strtolower(get_option('ams_buy_action_btn')) === strtolower('dropship')) {
        if (!empty($price)) {
            $profit = ($price / 100) * $percentage_profit;
            $price = $price + $profit;
        }
    }
    return $price;
}

public static function slugify($text) {
    $text = preg_replace('~[^\pL\d]+~u', '-', $text);
    $text = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $text);
    $text = preg_replace('~[^-\w]+~', '', $text);
    $text = trim($text, '-');
    $text = preg_replace('~-+~', '-', $text);
    $text = strtolower($text);
    if (empty($text)) {
        return 'n-a';
    }
    return $text;
}
    
public function syncAndGetCategory($html) {
    $cat = null;
    $ams_default_category = get_option('ams_default_category');

    $min_depth = intval(get_option('ams_category_min_depth', 1));
    $max_depth = intval(get_option('ams_category_max_depth', 6));

    if ($ams_default_category && $ams_default_category == '_auto_import_amazon') {

        $breadcrumbHtml = $html->find('#wayfinding-breadcrumbs_feature_div ul li span a');

        if (!empty($breadcrumbHtml)) {
            $category_paths = [];
            $last_id = 0;
            $category_ids = []; 

            // Build the category path from breadcrumbs
            foreach ($breadcrumbHtml as $catHtml) {
                $category_name = trim($catHtml->plaintext);

                if (!empty($category_name)) {
                    $category_paths[] = $category_name;
                }
            }

            // Ensure the cleaned path respects min-max depth
            if (count($category_paths) >= $min_depth) {
                // Trim the path to match the max depth if necessary
                $trimmed_path = array_slice($category_paths, 0, $max_depth);

                // Process each category in the trimmed path and create a hierarchy in WordPress
                foreach ($trimmed_path as $category_name) {
                    $category_name = trim($category_name);

                    // Check if the category exists under the parent
                    $existing_category = term_exists($category_name, 'product_cat', $last_id);

                    if (!$existing_category) {
                        // Create the category if it doesn't exist, with the correct parent
                        $new_category = wp_insert_term($category_name, 'product_cat', array(
                            'description' => $category_name,
                            'parent' => $last_id
                        ));

                        if (!is_wp_error($new_category)) {
                            $last_id = $new_category['term_id'];
                            $category_ids[] = $last_id;
                        }
                    } else {
                        // If the category already exists, fetch its ID and continue
                        $term = get_term_by('name', esc_attr($category_name), 'product_cat');
                        if ($term) {
                            $last_id = $term->term_id;
                            $category_ids[] = $last_id;
                        } else {
                            $last_id = $existing_category['term_id'];
                            $category_ids[] = $last_id;
                        }
                    }
                }

                // Assign all categories in the hierarchy to the product
                if (!empty($category_ids)) {
                    return $category_ids;
                }
            }
        }
    } else if ($ams_default_category) {

        return $ams_default_category;
    }

    return $cat;
}
public function fetchImagesFromHtml($html) {
    $gallery = [];
    $ams_product_thumbnail_size = get_option('ams_product_thumbnail_size');
    $az_index = [
        "hd" => 4,
        "extra_large" => 3,
        "Large" => 2,
        "Medium" => 1,
        "Small" => 0,
    ];

    // Primary method: #imageBlock_feature_div
    $imagesHtmlDataArray = $html->find('#imageBlock_feature_div');
    if (!empty($imagesHtmlDataArray)) {
        $imagesHtmlData = $this->get_string_between($imagesHtmlDataArray[0], 'colorImages', 'colorToAsin');
        preg_match_all('/main(.*?)variant/s', $imagesHtmlData, $imagesArray);
        preg_match_all('/hiRes(.*?)thumb/s', $imagesHtmlData, $hiresimagesArray);
        for ($i = 0; $i < sizeof($imagesArray[0]); $i++) {
            $imagesArray[0][$i] .= $hiresimagesArray[0][$i];
            preg_match_all('/https?:\/\/[^"\']+\.(jpg|jpeg|png|webp)/i', $imagesArray[0][$i], $matches);
            
            if ((!isset($matches[0][$az_index[$ams_product_thumbnail_size]]) || $az_index[$ams_product_thumbnail_size] == 4) 
                && isset($matches[0][sizeof($matches[0])-1])) {
                $gallery[] = $matches[0][sizeof($matches[0])-1];
            } elseif (isset($matches[0][$az_index[$ams_product_thumbnail_size]])) {
                $gallery[] = $matches[0][$az_index[$ams_product_thumbnail_size]];
            }
        }
    }

    // Fallback methods
    $methods = [
        '#mainImageContainer',
        '#main-image-container',
        '#imageBlockContainer',
        '#dp-container',
        '#imgTagWrapperId', // Common container for Amazon images
        '.imgTagWrapper',   // Class-based selection
        '#altImages',       // Alternative images container
        '.a-dynamic-image'  // For dynamically loaded images
    ];

    foreach ($methods as $method) {
        if (empty($gallery)) {
            $containerArray = $html->find($method);
            if (!empty($containerArray)) {
                // Try to extract image from a child <img> element first.
                $imgElement = $containerArray[0]->find('img', 0);
                if ($imgElement) {
                    if ($imgElement->getAttribute('data-old-hires')) {
                        $gallery[] = $imgElement->getAttribute('data-old-hires');
                    } elseif ($imgElement->src) {
                        $gallery[] = $imgElement->src;
                    }
                    if (!empty($gallery)) {
                        break;
                    }
                }
                
                // Existing fallback logic: use innertext with regex if no <img> was found.
                $container = $containerArray[0]->innertext;
                if ($method === '#dp-container') {
                    $container = $this->get_string_between($container, '{"landingImageUrl":"', '"}');
                }

                // Check for data-a-dynamic-image attribute if applicable.
                if (strpos($method, 'dynamic-image') !== false) {
                    if (isset($containerArray[0]) && property_exists($containerArray[0], 'data-a-dynamic-image')) {
                        $dataAttr = $containerArray[0]->{'data-a-dynamic-image'};
                        if ($dataAttr) {
                            $jsonData = json_decode($dataAttr, true);
                            if ($jsonData) {
                                $gallery = array_keys($jsonData);
                                break;
                            }
                        }
                    }
                }

                preg_match_all('/https?:\/\/[^"\']+\.(jpg|jpeg|png|webp)/i', $container, $matches);
                if (isset($matches[0][sizeof($matches[0])-1])) {
                    $gallery[] = $matches[0][sizeof($matches[0])-1];
                    break;
                }
            }
        }
    }

    // Last resort: try to find iUrl
    if (empty($gallery)) {
        $iUrl = $this->get_string_between($html, 'var iUrl = "', '";');
        preg_match_all('/https?:\/\/[^"\']+\.(jpg|jpeg|png|webp)/i', $iUrl, $matches);
        if (isset($matches[0][sizeof($matches[0])-1])) {
            $gallery[] = $matches[0][sizeof($matches[0])-1];
        }
    }

    // Additional fallback for JSON-LD structured data
    if (empty($gallery)) {
        $jsonLd = $html->find('script[type="application/ld+json"]');
        if (!empty($jsonLd)) {
            $jsonData = json_decode($jsonLd[0]->innertext, true);
            if (isset($jsonData['image'])) {
                $gallery = is_array($jsonData['image']) ? $jsonData['image'] : [$jsonData['image']];
            }
        }
    }

    return $gallery;
}

public function fetchPriceFromHtml($loop_html, $skus = [], $silent = false) {
    $debug_info = [];
    $sku_prices = [];

    // Function to find the ppd div with multiple attempts
    $findPpdDiv = function($html) use (&$debug_info) {
        $attempts = [
            'div#ppd',
            'div[id="ppd"]',
            'div[id^="ppd"]',
            'div[id*="ppd"]'
        ];

        foreach ($attempts as $attempt) {
            $ppd = $html->find($attempt, 0);
            if ($ppd) {
                $debug_info['ppd_selector'] = $attempt;
                return $ppd;
            }
        }

        return null;
    };

    // Find the relevant section starting from ppd div
    $ppd = $findPpdDiv($loop_html);
    
    if ($ppd) {
        $relevant_section = $ppd;
        $debug_info['search_area'] = 'ppd';
    } else {
        // Search in Northstar-Buybox
        $northstar = $loop_html->find('#Northstar-Buybox', 0);
        if ($northstar) {
            $relevant_section = $northstar;
            $debug_info['search_area'] = 'Northstar-Buybox';
        } else {
            $relevant_section = $loop_html;
            $debug_info['search_area'] = 'entire HTML';
            $debug_info['warning'] = "Could not find ppd div or Northstar-Buybox. Using entire HTML.";
        }
    }

    // Debug: Check what price-related elements exist in the HTML
    $priceElementCounts = [
        '.a-price' => count($loop_html->find('.a-price')),
        '.a-price-whole' => count($loop_html->find('.a-price-whole')),
        '.a-offscreen' => count($loop_html->find('.a-offscreen')),
        'span[aria-hidden="true"]' => count($loop_html->find('span[aria-hidden="true"]')),
        '[id*="price"]' => count($loop_html->find('[id*="price"]')),
        '[class*="price"]' => count($loop_html->find('[class*="price"]')),
    ];
    
    foreach ($priceElementCounts as $selector => $count) {

    }
    
    // Check for product availability indicators
    $htmlText = $loop_html->plaintext;
    
    // Check for non-existent products and CAPTCHA/bot detection
    $nonExistentIndicators = [
        'Clique no botão abaixo para continuar comprando', // Brazil CAPTCHA
        'Click the button below to continue shopping', // English CAPTCHA
        'Continue shopping', // English CAPTCHA (shorter version)
        'Sorry, we couldn\'t find that page', // 404 error
        'Page not found',
        'The page you requested cannot be found',
        'We couldn\'t find the page you were looking for',
        'This page could not be found',
        'Sorry! We couldn\'t find that page',
        'Looking for something?', // Amazon 404 page
        'Dogs of Amazon', // Amazon 404 page with dog images
        
        // Portuguese (Brazil) 404 indicators
        'DESCULPE', // Sorry (Brazil 404 page)
        'não conseguimos encontrar esta página', // we couldn't find this page
        'Pesquise novamente ou volte para', // Search again or return to
        'página inicial da Amazon', // Amazon's home page
        'Assam', // Amazon's 404 dog name (Brazil)
        
        // Spanish 404 indicators (for completeness)
        'Lo sentimos, no pudimos encontrar esa página',
        'Página no encontrada',
        
        // French 404 indicators
        'Désolé, nous n\'avons pas pu trouver cette page',
        'Page introuvable',
        
        // German 404 indicators  
        'Entschuldigung, wir konnten diese Seite nicht finden',
        'Seite nicht gefunden',
    ];
    
    $unavailableIndicators = [
        'Currently unavailable',
        'We don\'t know when or if this item will be back in stock',
        'No featured offers available',
        'Out of stock',
        'Temporarily out of stock',
        'This item is not available'
    ];
    $isUnavailable = false;
    $isNonExistent = false;
    
    // Check for non-existent products first
    foreach ($nonExistentIndicators as $indicator) {
        if (stripos($htmlText, $indicator) !== false) {
            // logImportVerification('🚨 PRODUCT NOT FOUND - Found non-existent indicator: "' . $indicator . '"', null);
            $isNonExistent = true;
            break;
        }
    }
    
    // Check for unavailable products
    if (!$isNonExistent) {
        foreach ($unavailableIndicators as $indicator) {
            if (stripos($htmlText, $indicator) !== false) {
                // logImportVerification('🚨 AVAILABILITY WARNING - Found unavailable indicator: "' . $indicator . '"', null);
                $isUnavailable = true;
            }
        }
    }
    
    // If product is non-existent or unavailable, return zero prices immediately
    if ($isNonExistent) {
        // logImportVerification('🚨 PRODUCT NOT FOUND - Returning zero prices for non-existent product (404/CAPTCHA)', null);
        return [
            'regular_price' => 0,
            'sale_price' => 0,
            'sku_prices' => [],
            'debug_info' => ['non_existent' => true, 'indicators_found' => $nonExistentIndicators]
        ];
    }
    
    // Note: We don't immediately return zero prices for unavailable products anymore
    // Instead, we'll try to extract prices and only return zero if no prices are found
    if ($isUnavailable) {
        // logImportVerification('⚠️ AVAILABILITY WARNING - Product may be unavailable, but continuing price extraction', null);
    }
    
    // If we're using entire HTML and still finding 0 elements, let's check for text patterns
    if ($debug_info['search_area'] === 'entire HTML') {
        $htmlText = $loop_html->plaintext;
        $pricePatterns = [
            '/\$\d+\.\d{2}/' => 'US Dollar format ($12.34)',
            '/CAD\s*\$?\d+\.\d{2}/' => 'Canadian Dollar format',
            '/\$\d{1,3}(?:,\d{3})*\.\d{2}/' => 'Formatted Dollar ($1,234.56)',
        ];
        
        foreach ($pricePatterns as $pattern => $description) {
            if (preg_match_all($pattern, $htmlText, $matches)) {

            }
        }
    }

    // Helper function to find and parse price with enhanced debugging and fallback areas
    $findPrice = function($selectors, $price_type) use ($relevant_section, $loop_html, &$debug_info) {
        
        // Try primary search area first
        $searchAreas = [
            ['area' => $relevant_section, 'name' => 'primary area'],
            ['area' => $loop_html, 'name' => 'full HTML']
        ];
        
        foreach ($searchAreas as $areaInfo) {
            $searchArea = $areaInfo['area'];
            $areaName = $areaInfo['name'];
                        
            foreach ($selectors as $index => $selector) {
                $elements = $searchArea->find($selector);
                
                if (count($elements) > 0) {                    
                    foreach ($elements as $elementIndex => $element) {
                        // Skip price per unit elements
                if (strpos($element->class, 'pricePerUnit') !== false) {
                            continue;
                        }
                        
                        // Skip dropdown elements (quantity selectors, etc.)
                        if (strpos($element->class, 'dropdown') !== false || 
                            strpos($element->class, 'a-button') !== false ||
                            strpos($element->class, 'a-dropdown-prompt') !== false) {
                            continue;
                        }
                        
                        // Skip rating and review elements
                        $parentClass = $element->parent() ? $element->parent()->class : '';
                        if (strpos($element->class, 'rating') !== false || 
                            strpos($element->class, 'review') !== false ||
                            strpos($element->class, 'popover') !== false ||
                            strpos($element->class, 'star') !== false ||
                            strpos($parentClass, 'popover') !== false ||
                            strpos($parentClass, 'rating') !== false ||
                            strpos($parentClass, 'review') !== false) {
                            continue;
                        }
                        
                        // Skip elements that are clearly not prices (single digits without currency, ratings like 4.4)
                        $elementText = trim($element->plaintext);
                        if (preg_match('/^\d{1}$/', $elementText) || 
                            preg_match('/^\d\.\d$/', $elementText) ||
                            preg_match('/^[1-5]\.[0-9]$/', $elementText)) {
                            continue;
                        }
                        
                        // Additional check: Skip if element text looks like a rating (X.X format where X is 1-5)
                        if (preg_match('/^[1-5]\.[0-9]+$/', $elementText)) {
                            continue;
                        }
                        
                        // Skip review counts and other large numbers (typically 4+ digits)
                        if (preg_match('/^\d{4,}$/', str_replace(',', '', $elementText))) {
                            continue;
                        }
                        
                        // Skip ratings (like "4.4 out of 5")
                        if (strpos($elementText, 'out of') !== false) {
                            continue;
                        }
                        
                        // Skip elements that are just numbers without currency (like "31", "4.45" without $)
                        if (preg_match('/^\d+(\.\d+)?$/', trim($elementText)) && !preg_match('/[\$£€¥₹]/', $elementText)) {
                            continue;
                        }
                        
                        // Skip elements in review/rating contexts
                        if (strpos($parentClass, 'review') !== false || 
                            strpos($parentClass, 'rating') !== false ||
                            strpos($element->parent()->parent()->class ?? '', 'review') !== false) {
                    continue;
                }
                
                // Check for the specific price structure
                $price_symbol = $element->find('.a-price-symbol', 0);
                $price_whole = $element->find('.a-price-whole', 0);
                $price_fraction = $element->find('.a-price-fraction', 0);
                
                if ($price_symbol && $price_whole && $price_fraction) {
                    $full_price = $price_symbol->plaintext . $price_whole->plaintext . $price_fraction->plaintext;
                } else {
                    $full_price = $element->plaintext;

                        }
                        
                        // Debug: Show element context
                        $elementId = $element->id ? $element->id : 'no-id';
                        $elementClass = $element->class ? $element->class : 'no-class';
                        $parentElement = $element->parent();
                        $parentInfo = $parentElement ? ($parentElement->id ? $parentElement->id : ($parentElement->class ? $parentElement->class : 'unknown-parent')) : 'no-parent';
                
                $price = $this->parsePrice($full_price);
                $debug_info[$price_type][] = [
                    'selector' => $selector,
                    'found_price' => $full_price,
                            'parsed_price' => $price,
                            'search_area' => $areaName
                ];
                        
                if ($price > 0) {
                            // Special check for suspiciously low prices (likely incorrect elements)
                            if ($price < 5.00) {
                                // Skip very low prices (likely ratings or incorrect elements) unless it's our last resort
                                if ($price < 1.00) {

                                    continue;
                                }
                            }
                            
                            // Skip unreasonably high prices (likely review counts or other numbers)
                            if ($price > 10000) {

                                continue;
                            }
                            
                            // Extra validation for suspicious prices without currency symbols
                            if ($price < 6 && !preg_match('/[\$£€¥₹]/', $full_price)) {

                                continue;
                            }
                    return $price;
                }
            }
        }
            }
            
            // If we found nothing in primary area, continue to full HTML
            if ($areaName === 'primary area') {

            }
        }
        
        return 0;
    };

    // Enhanced price selectors with priority order (most specific first)
    $main_price = $findPrice([
        // HIGH PRIORITY: Main product price selectors
        '#corePrice_feature_div .a-price .a-offscreen',
        '#corePrice_desktop .a-price .a-offscreen', 
        '[data-feature-name="corePrice"] .a-price .a-offscreen',
        '#apex_desktop .a-price .a-offscreen',
        '#buybox .a-price .a-offscreen',
        '#price_inside_buybox .a-offscreen',
        '#corePriceDisplay_desktop_feature_div .a-price .a-offscreen',
        
        // Reinvented price blocks
        'span.reinventPricePriceToPay .a-offscreen',
        'span.reinventPricePriceToPayMargin .a-offscreen',
        'div.reinventPricePriceToPay .a-offscreen',
        'div.reinventPricePriceToPayMargin .a-offscreen',
        
        // BuyBox specific ids
        '#newBuyBoxPrice',
        '#usedBuyBoxPrice',
        '#attach-base-product-price',
        
        // AOD/Offer display fallbacks
        '#aod-asin-price .a-offscreen',
        '#aod-price-1 .a-offscreen',
        
        // Subscribe & Save
        '#sns-base-price',
        '#snsPrice',
        
        // MEDIUM PRIORITY: Standard selectors
        'span.a-price[data-a-size="xl"] span[aria-hidden="true"]',
        'span.a-price[data-a-size="l"] span[aria-hidden="true"]',
        '.priceToPay span[aria-hidden="true"]',
        '.apexPriceToPay span[aria-hidden="true"]',
        '.priceToPay .a-offscreen',
        '.apexPriceToPay .a-offscreen',
        '#priceblock_ourprice',
        '#priceblock_dealprice',
        '#priceblock_saleprice',
        'span.a-price[data-a-size="xl"] .a-offscreen',
        'span.a-price[data-a-size="l"] .a-offscreen',
        'span.a-price:not([data-a-strike="true"]) .a-offscreen',
        '#corePriceDisplay_desktop_feature_div .priceToPay .a-offscreen',
        '.a-section.a-spacing-none.aok-align-center.aok-relative .a-price .a-offscreen',
        'span.a-size-base.a-color-price.a-color-price',
        '#kindle-price',
        '.a-section.a-spacing-none.aok-align-center.aok-relative .a-price.a-text-price span[aria-hidden="true"]',
        '.dimension-slot-info .a-price span[aria-hidden="true"]',
        '.twister_swatch_price .olpWrapper',
        '.apex_on_twister_price .a-price span[aria-hidden="true"]',
        
        // Brazil and international specific selectors
        '.a-price-whole',
        '.a-price .a-price-whole',
        'span.a-price-whole',
        '.a-price-symbol + .a-price-whole',
        '#price_inside_buybox',
        '.a-price[data-a-size="b"] .a-offscreen',
        '.a-price[data-a-size="m"] .a-offscreen',
        '.a-price.a-text-price.a-size-medium.a-color-price .a-offscreen',
        '.a-price.a-text-price.a-size-base.a-color-price .a-offscreen',
        '.a-price.a-text-price.a-size-large.a-color-price .a-offscreen',
        'span.a-price.a-text-price[data-a-size] .a-offscreen',
        
        // Additional fallback selectors for different regions
        '.a-section .a-price .a-offscreen',
        '.a-spacing-none .a-price .a-offscreen',
        'div[data-feature-name="price"] .a-price .a-offscreen',
        'div[data-feature-name="price"] span[aria-hidden="true"]',
        '#buybox .a-price .a-offscreen',
        '#buybox span[aria-hidden="true"]',
        '[itemprop="price"]',
        'span[data-a-color="price"] .a-offscreen',
        '#tp_price_block_total_price_ww .a-offscreen',
        
        // Mobile and responsive selectors
        '.a-price-range .a-price .a-offscreen',
        '.a-price-range span[aria-hidden="true"]',
        
        // Last resort - any price element
        '.a-price .a-offscreen',
        'span[aria-hidden="true"]'
    ], 'main_price');

    $crossed_out_price = $findPrice([
        '.a-text-price .a-offscreen',
        '#listPrice',
        '.priceBlockStrikePriceString .a-text-strike',
        '#corePriceDisplay_desktop_feature_div .a-text-price .a-offscreen',
        'span[data-a-strike="true"] .a-offscreen',
        '.a-section.a-spacing-small.aok-align-center .a-price.a-text-price[data-a-strike="true"] .a-offscreen'
    ], 'crossed_out_price');

    // For variable products, find prices for specific SKUs
    if (!empty($skus) && is_array($skus)) {
        foreach ($skus as $sku) {
            $sku_element = $relevant_section->find("[data-asin='$sku']", 0);
            if ($sku_element) {
                $priceElement = $sku_element->find('.twisterSwatchPrice', 0);
                if ($priceElement && isset($priceElement->plaintext)) {
                    $sku_prices[$sku] = $this->parsePrice($priceElement->plaintext);
                }
            }
        }
    }
    if (!empty($sku_prices)) {
        // For variable products
        $regular_price = max($sku_prices);
        $sale_price = min($sku_prices);
    } else {
        // For simple products or fallback
        if ($crossed_out_price > 0 && $crossed_out_price > $main_price) {
            $regular_price = $crossed_out_price;
            $sale_price = $main_price;
        } else {
            $regular_price = $main_price;
            $sale_price = $main_price;
        }
    }

    // Fallback: JSON-LD structured data (offers.price)
    if (($regular_price == 0 && $sale_price == 0)) {
        $jsonLdScripts = $loop_html->find('script[type="application/ld+json"]');
        foreach ($jsonLdScripts as $script) {
            $json = trim($script->innertext);
            if ($json === '') continue;
            $data = json_decode($json, true);
            if (!$data) continue;
            $candidates = is_array($data) && isset($data[0]) ? $data : [$data];
            foreach ($candidates as $obj) {
                if (isset($obj['offers'])) {
                    $offers = $obj['offers'];
                    if (isset($offers['price'])) {
                        $p = $this->parsePriceToFloat((string)$offers['price']);
                        if ($p > 0) { $regular_price = $sale_price = $p; break 2; }
                    }
                    if (is_array($offers)) {
                        foreach ($offers as $offer) {
                            if (is_array($offer) && isset($offer['price'])) {
                                $p = $this->parsePriceToFloat((string)$offer['price']);
                                if ($p > 0) { $regular_price = $sale_price = $p; break 3; }
                            }
                        }
                    }
                }
            }
        }
    }

    // Fallback: Search any inline script JSON for price fields
    if (($regular_price == 0 && $sale_price == 0)) {
        $scripts = $loop_html->find('script');
        foreach ($scripts as $script) {
            $txt = $script->innertext;
            if (!$txt) continue;
            if (preg_match('/"priceAmount"\s*:\s*"?(\d+(?:\.\d{1,2})?)"?/i', $txt, $m)) {
                $p = $this->parsePriceToFloat($m[1]);
                if ($p > 0) { $regular_price = $sale_price = $p; break; }
            }
            if (preg_match('/"price"\s*:\s*"?(\d+(?:\.\d{1,2})?)"?/i', $txt, $m)) {
                $p = $this->parsePriceToFloat($m[1]);
                if ($p > 0) { $regular_price = $sale_price = $p; break; }
            }
        }
    }

    // Last resort: pick the first .a-offscreen containing a currency symbol anywhere
    if (($regular_price == 0 && $sale_price == 0)) {
        $candidates = $loop_html->find('.a-offscreen');
        foreach ($candidates as $cand) {
            $t = trim($cand->plaintext);
            if (preg_match('/[\$€£¥₹]/', $t)) {
                $p = $this->parsePrice($t);
                if ($p > 0) { $regular_price = $sale_price = $p; break; }
            }
        }
    }

    $debug_info['sku_prices'] = $sku_prices;
    $debug_info['final_prices'] = [
        'regular_price' => $regular_price,
        'sale_price' => $sale_price
    ];

    // Final check: If we detected unavailability and still have zero prices, log it
    if ($isUnavailable && $regular_price == 0 && $sale_price == 0) {
        // logImportVerification('🚨 FINAL CHECK - Product unavailable and no prices found, returning zero prices', null);
        $debug_info['final_unavailable'] = true;
    }

    //print_r($debug_info);

    return [
        'regular_price' => $regular_price,
        'sale_price' => $sale_price,
        'sku_prices' => $sku_prices,
        'debug_info' => $debug_info
    ];
}

private function cleanPriceText($element) {
    if (is_object($element) && property_exists($element, 'plaintext')) {
        return trim(strip_tags($element->plaintext));
    } elseif (is_string($element)) {
        return trim(strip_tags($element));
    } else {
        // Handle unexpected input type
        return '';
    }
}
function parsePriceToFloat($string_price) {
    // Remove any whitespace
    $string_price = trim($string_price);

    // Remove currency symbols and any other non-numeric characters except . and ,
    $cleaned_price = preg_replace('/[^0-9.,]/', '', $string_price);

    // Check for different price formats
    if (preg_match('/^(\d{1,3}(?:,\d{3})*(?:\.\d{2})?)$/', $cleaned_price)) {
        // US format: 1,234.56
        $cleaned_price = str_replace(',', '', $cleaned_price);
    } elseif (preg_match('/^(\d{1,3}(?:.\d{3})*(?:,\d{2})?)$/', $cleaned_price)) {
        // European format: 1.234,56
        $cleaned_price = str_replace('.', '', $cleaned_price);
        $cleaned_price = str_replace(',', '.', $cleaned_price);
    } elseif (preg_match('/^(\d+(?:,\d{2})?)$/', $cleaned_price)) {
        // Format like: 1234,56
        $cleaned_price = str_replace(',', '.', $cleaned_price);
    }

    // Convert to float
    $float_price = (float) $cleaned_price;

    // Round to the number of decimal places specified in WooCommerce
    $precision = function_exists('wc_get_price_decimals') ? wc_get_price_decimals() : 2;
    $float_price = round($float_price, $precision);

    return $float_price;
}

// Group functions to extract and format content
public function fetchContentFromHtml($html) {
    $content = $this->extractContent($html);
    return $this->formatContentForWordPress($content);
}
private function matchesPlaceholder($src, $patterns) {
    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $src)) {
            return true;
        }
    }
    return false;
}
private function removeUnwantedElements($element) {
    // Adjust remove patterns to ensure important content (headings) are not stripped
    $removePatterns = [
        '*' => ['Collapse All', 'Expand All']  // Only removing truly unwanted patterns
    ];

    foreach ($removePatterns as $tag => $patterns) {
        foreach ($element->find($tag) as $subElement) {
            foreach ($patterns as $pattern) {
                if (stripos($subElement->innertext, $pattern) !== false) {
                    $subElement->outertext = ''; // Remove unwanted elements
                }
            }
        }
    }

    // Avoid removing entire important divs like in the old function
    $ignoreDivs = [
        '#productSpecifications_dp_warranty_and_support',
        '#productDetails_feedback_sections',
        '#productDetails_expanderTables_depthRightSections',
    ];

    foreach ($ignoreDivs as $ignoreDiv) {
        foreach ($element->find($ignoreDiv) as $ignored) {
            $ignored->outertext = '';
        }
    }
}
private function handleLazyLoadedImages($element) {
    $lazyAttributes = ['data-a-hires', 'data-src', 'data-lazy-src', 'data-srcset', 'data-original'];
    $placeholderPatterns = [
        '/^data:image\/.*base64/i',
        '/^data:image\/svg\+xml/i',
        '/placeholder\.png$/i',
        '/^about:blank$/i',
        '/grey-pixel\.gif$/i'
    ];

    // First, remove lazy-loaded placeholder images
    foreach ($element->find('img.a-lazy-loaded') as $img) {
        $img->outertext = '';
    }

    // Then handle any remaining images
    foreach ($element->find('img') as $img) {
        $currentSrc = $img->getAttribute('src');
        $isPlaceholder = empty($currentSrc) || $this->matchesPlaceholder($currentSrc, $placeholderPatterns);

        if (!$isPlaceholder) {
            continue;
        }

        foreach ($lazyAttributes as $attr) {
            $lazySource = $img->getAttribute($attr);
            if (!empty($lazySource) && !$this->matchesPlaceholder($lazySource, $placeholderPatterns)) {
                $img->setAttribute('src', $lazySource);
                break;
            }
        }
    }

    // Remove empty div.a-text-center after images
    foreach ($element->find('div.background-image') as $imgDiv) {
        $nextDiv = $imgDiv->next_sibling();
        if ($nextDiv && $nextDiv->class == 'a-section a-text-center' && trim($nextDiv->innertext) === '') {
            $nextDiv->outertext = '';
        }
    }

    // Remove all empty p tags
    foreach ($element->find('p') as $p) {
        if (trim($p->innertext) === '') {
            $p->outertext = '';
        }
    }
}
private function formatContentForWordPress($content) {
    // Remove problematic hover spans before processing
    $content = $this->removeHoverSpans($content);

    // Remove only unsafe or heavy elements
    $content = preg_replace([
        '/<(script|style)\b[^>]*>(.*?)<\/\1>/is', // JS + CSS
        '/<!--(.|\s)*?-->/is',                   // HTML comments
        '/<(iframe|embed|object|area)[^>]*>.*?<\/\1>/is', // media containers
        '/<(input|button|select|textarea|form|option)[^>]*>.*?<\/\1>/is',
        '/<(input|button|select|textarea|form|option)[^>]*>/is'
    ], '', $content);

    // Remove all <hr> elements, regardless of class
    $content = preg_replace('/<hr[^>]*>/', '', $content);

    // Remove empty <div> elements that may cause spacing issues
    $content = preg_replace('/<div class="a-section a-text-center">\s*<\/div>/', '', $content);

    // Remove navigation text artifacts
    $content = str_replace(['Previous page', 'Next page', 'Previous', 'Next'], '', $content);
    
    // Fix broken list items and carousel lists
    $content = preg_replace('/<li>\s*<\/li>/', '', $content);
    $content = preg_replace('/<li[^>]*>\s*<\/li>/', '', $content);
    
    // Extract images from carousel lists before removing the list structure
    $content = preg_replace_callback('/<ol[^>]*class="[^"]*a-carousel[^"]*"[^>]*>(.*?)<\/ol>/is', function($matches) {
        $carouselContent = $matches[1];
        // Extract all images from the carousel
        preg_match_all('/<img[^>]*>/i', $carouselContent, $images);
        return implode('', $images[0]); // Return just the images without list structure
    }, $content);
    
    // Remove pagination dots completely
    $content = preg_replace('/<ul[^>]*class="[^"]*aplus-pagination-dots[^"]*"[^>]*>.*?<\/ul>/is', '', $content);
    
    // Remove any remaining empty <div> or <span> tags
    $content = preg_replace('/<div[^>]*>\s*<\/div>/', '', $content);
    $content = preg_replace('/<span[^>]*>\s*<\/span>/', '', $content);

    // Sanitize with wp_kses_post to retain standard post formatting without stripping images
    $content = wp_kses_post($content);

    // Apply wpautop to ensure consistent paragraph structure
    $content = wpautop($content);

    // Remove any empty <p> tags and normalize spaces
    $content = preg_replace('/<p>\s*<\/p>/', '', $content);
    $content = preg_replace('/\s+/', ' ', $content);
    $content = preg_replace('/\n\s*\n/', "\n", $content);

    // Simple fix: just add inline styles to images in background-image containers
    $content = preg_replace('/<img([^>]*class="[^"]*"[^>]*)>/', '<img$1 style="max-width: 100%; height: auto;">', $content);
    $content = preg_replace('/<img([^>]*(?!class=)[^>]*)>/', '<img$1 style="max-width: 100%; height: auto;">', $content);
    
    return '<div class="woocommerce-product-details">' . $content . '</div>';
}
private function removeHoverSpans($content) {
    // Only remove the specific hover spans, nothing else
    $content = preg_replace(
        '/<span tabindex=0 class="hover-point" data-inline-content="[^"]*" data-position="triggerHorizontal" \/>/is',
        '',
        $content
    );
    
    return $content;
}
private function extractContent($html) {
    $content = '';
    $uniqueTextContent = []; // Track unique text content for better comparison

    // Define a new selector for the "About this Item" section
    $aboutSelectors = [
        'h3.product-facts-title' // Add more selectors if needed
    ];
    // Loop through selectors to scrape content
    foreach ($aboutSelectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Check if the title matches dynamically
            $dynamicTitle = trim($element->plaintext);
            if (stripos($dynamicTitle, 'About this item') !== false) {
                $content .= '<h3>' . htmlspecialchars($dynamicTitle) . '</h3>';
                
                // Locate the unordered list dynamically
                $aboutList = $element->parent()->find('.a-unordered-list', 0); // Adjust parent as needed
                if ($aboutList) {
                    $content .= '<ul>';
                    foreach ($aboutList->find('li') as $item) {
                        $listItem = $item->find('span.a-list-item', 0);
                        if ($listItem) {
                            $itemText = trim($listItem->plaintext);
                            if (!empty($itemText)) {
                                $content .= '<li>' . htmlspecialchars($itemText) . '</li>';
                            }
                        }
                    }
                    $content .= '</ul>';
                }
            }
        }
    }



    // Define a new selector for the "From the Brand" section
    $brandSelectors = [
        '#aplusBrandStory_feature_div' // Add more selectors if needed
    ];
    // Loop through selectors to scrape content
    foreach ($brandSelectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Handle unwanted elements and lazy-loaded images
            $this->removeUnwantedElements($element);
            $this->handleLazyLoadedImages($element);

            // Initialize a flag to check if there's actual content
            $hasBrandContent = false;
            $sectionContent = '';

            // Extract the dynamic title
            $titleElement = $element->find('h2', 0);
            $dynamicTitle = $titleElement ? trim($titleElement->plaintext) : 'Brand Information';

            // Extract the first main image (avoid duplicates)
            $imageHashes = []; // To store hashes of processed images
            $mainImage = $element->find('.apm-brand-story-background-image img', 0);
            if ($mainImage) {
                $imageSrc = $mainImage->getAttribute('data-src') ?: $mainImage->getAttribute('src');
                if (!empty($imageSrc) && !in_array(md5($imageSrc), $imageHashes)) {
                    $imageAlt = $mainImage->getAttribute('alt') ?: 'Brand Image';
                    $sectionContent .= '<div><img src="' . htmlspecialchars($imageSrc) . '" alt="' . htmlspecialchars($imageAlt) . '" style="max-width: 100%;"></div>';
                    $imageHashes[] = md5($imageSrc);
                    $hasBrandContent = true; // Mark that content exists
                }
            }

            // Extract main brand text
            $brandText = $element->find('.apm-brand-story-text-bottom', 0);
            if ($brandText) {
                $sectionContent .= '<p>' . htmlspecialchars(trim($brandText->plaintext)) . '</p>';
                $hasBrandContent = true; // Mark that content exists
            }

            // Only append the title and content if there's actual brand content
            if ($hasBrandContent) {
                $content .= '<h3>' . htmlspecialchars($dynamicTitle) . '</h3>';
                $content .= $sectionContent;
            }
        }
    }




    // Define a new selector for the "From the Manufacturer" section
    $manufacturerSelectors = [
        '#aplus_feature_div' // Add more selectors if needed
    ];
    // Loop through the selectors to scrape content
    foreach ($manufacturerSelectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Handle unwanted elements and lazy-loaded images
            $this->removeUnwantedElements($element);
            $this->handleLazyLoadedImages($element);

            // Pre-clean hover spans before processing
            $element->outertext = $this->removeHoverSpans($element->outertext);

            // Convert h2 to h3 to preserve titles
            $element->outertext = preg_replace('/<h2(.*?)>(.*?)<\/h2>/', '<h3 class="$1">$2</h3>', $element->outertext);

            // Get the HTML and text content
            $elementHtml = trim($element->outertext);
            $elementText = trim($element->plaintext);

            // Create a unique key based on the text content to detect duplicates
            $contentKey = md5($elementText);

            // Check for unique content
            if (!empty($elementText) && !in_array($contentKey, $uniqueTextContent)) {
                $uniqueTextContent[] = $contentKey;
                $content .= $elementHtml;
            }
        }
    }




    // Define a new selector for the "Product Description" section
    $productDescriptionSelectors = [
        '#productDescription_feature_div' // Add more selectors if needed
    ];
    // Initialize an array to track processed content
    $processedSections = [];
    // Loop through selectors to scrape content
    foreach ($productDescriptionSelectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Handle unwanted elements and lazy-loaded images
            $this->removeUnwantedElements($element);
            $this->handleLazyLoadedImages($element);

            // Extract the dynamic title
            $titleElement = $element->find('h2', 0); // Find the title dynamically
            $dynamicTitle = $titleElement ? trim($titleElement->plaintext) : 'Product Description';

            // Extract the product description text
            $descriptionElement = $element->find('#productDescription', 0); // Locate the main description
            $descriptionText = $descriptionElement ? trim($descriptionElement->plaintext) : '';

            // Generate a unique key to track duplicates
            $sectionKey = md5($dynamicTitle . $descriptionText);

            // Check if the section is already processed
            if (!empty($descriptionText) && !isset($processedSections[$sectionKey])) {
                $content .= '<h3>' . htmlspecialchars($dynamicTitle) . '</h3>';
                $content .= '<p>' . htmlspecialchars($descriptionText) . '</p>';

                // Extract the first main image (if any)
                $mainImage = $element->find('img', 0); // Find the first image
                if ($mainImage) {
                    $imageSrc = $mainImage->getAttribute('data-src') ?: $mainImage->getAttribute('src');
                    if (!empty($imageSrc)) {
                        $imageAlt = $mainImage->getAttribute('alt') ?: 'Product Image';
                        $content .= '<div><img src="' . htmlspecialchars($imageSrc) . '" alt="' . htmlspecialchars($imageAlt) . '" style="max-width: 100%;"></div>';
                    }
                }

                // Mark the section as processed
                $processedSections[$sectionKey] = true;
            }
        }
    }




    // Define a new selector for the "Technical Specifications" section
    $technicalSpecificationsSelectors = [
        '#technicalSpecifications_feature_div' // Add more selectors if needed
    ];
    // Initialize an array to track processed technical specifications
    $processedTechnicalSpecifications = [];
    // Loop through selectors to scrape content
    foreach ($technicalSpecificationsSelectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Handle unwanted elements and lazy-loaded images
            $this->removeUnwantedElements($element);
            $this->handleLazyLoadedImages($element);

            // Extract the dynamic title (if any)
            $titleElement = $element->find('h2', 0); // Find the title dynamically
            $dynamicTitle = $titleElement ? trim($titleElement->plaintext) : 'Technical Specifications';

            // Extract the technical specifications table
            $tableElement = $element->find('table', 0); // Locate the table
            $tableHtml = '';
            if ($tableElement) {
                $tableHtml .= '<table>';
                foreach ($tableElement->find('tr') as $row) {
                    $tableHtml .= '<tr>';
                    foreach ($row->find('th, td') as $cell) {
                        $tableHtml .= '<' . $cell->tag . '>' . htmlspecialchars(trim($cell->plaintext)) . '</' . $cell->tag . '>';
                    }
                    $tableHtml .= '</tr>';
                }
                $tableHtml .= '</table>';
            }

            // Generate a unique key to track duplicates
            $sectionKey = md5($dynamicTitle . $tableHtml);

            // Check if the section is already processed
            if (!empty($tableHtml) && !isset($processedTechnicalSpecifications[$sectionKey])) {
                $content .= '<h3>' . htmlspecialchars($dynamicTitle) . '</h3>';
                $content .= $tableHtml;

                // Mark the section as processed
                $processedTechnicalSpecifications[$sectionKey] = true;
            }
        }
    }



    // Define a new selector for the "Detail Bullets" section
    $detailBulletsSelectors = [
        '#detailBulletsWrapper_feature_div' // Add more selectors if needed
    ];
    // Initialize an array to track processed sections
    $processedDetailBullets = [];
    // Loop through selectors to scrape content
    foreach ($detailBulletsSelectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Handle unwanted elements and lazy-loaded images
            $this->removeUnwantedElements($element);
            $this->handleLazyLoadedImages($element);

            // Extract the dynamic title (if any)
            $titleElement = $element->find('h2', 0); // Find the title dynamically
            $dynamicTitle = $titleElement ? trim($titleElement->plaintext) : 'Product Details';

            // Extract the detail bullets list
            $bulletsElement = $element->find('.a-unordered-list', 0); // Locate the unordered list
            $bulletsHtml = '';
            if ($bulletsElement) {
                $bulletsHtml .= '<ul>';
                foreach ($bulletsElement->find('li') as $item) {
                    $itemText = trim($item->plaintext);

                    // Remove or replace special direction marks
                    $itemText = str_replace(['&lrm;', '&rlm;'], '', $itemText);

                    // Clean any extra whitespace
                    $itemText = preg_replace('/\s+/', ' ', $itemText);

                    $bulletsHtml .= '<li>' . htmlspecialchars($itemText) . '</li>';
                }
                $bulletsHtml .= '</ul>';
            }

            // Generate a unique key to track duplicates
            $sectionKey = md5($dynamicTitle . $bulletsHtml);

            // Check if the section is already processed
            if (!empty($bulletsHtml) && !isset($processedDetailBullets[$sectionKey])) {
                $content .= '<h3>' . htmlspecialchars($dynamicTitle) . '</h3>';
                $content .= $bulletsHtml;

                // Mark the section as processed
                $processedDetailBullets[$sectionKey] = true;
            }
        }
    }



    // Define a new selector for the "Legal Disclaimer" section
    $legalSelectors = [
        '#legalEUBtf_feature_div' // Add more selectors if needed
    ];
    // Initialize an array to track processed sections
    $processedLegalSections = [];
    // Loop through selectors to scrape content
    foreach ($legalSelectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Handle unwanted elements and lazy-loaded images
            $this->removeUnwantedElements($element);
            $this->handleLazyLoadedImages($element);

            // Extract the dynamic title (if any)
            $titleElement = $element->find('h2', 0); // Find the title dynamically
            $dynamicTitle = $titleElement ? trim($titleElement->plaintext) : 'Legal Disclaimer';

            // Extract the legal content
            $legalContentElement = $element->find('.a-section', 0); // Locate the content dynamically
            $legalContentHtml = '';
            if ($legalContentElement) {
                // Clean the text content
                $legalContentText = trim($legalContentElement->plaintext);

                // Remove or replace special direction marks
                $legalContentText = str_replace(['&lrm;', '&rlm;'], '', $legalContentText);

                // Clean any extra whitespace
                $legalContentText = preg_replace('/\s+/', ' ', $legalContentText);

                $legalContentHtml = '<p>' . htmlspecialchars($legalContentText) . '</p>';
            }

            // Generate a unique key to track duplicates
            $sectionKey = md5($dynamicTitle . $legalContentHtml);

            // Check if the section is already processed
            if (!empty($legalContentHtml) && !isset($processedLegalSections[$sectionKey])) {
                $content .= '<h3>' . htmlspecialchars($dynamicTitle) . '</h3>';
                $content .= $legalContentHtml;

                // Mark the section as processed
                $processedLegalSections[$sectionKey] = true;
            }
        }
    }



    // Define a new selector for the "Buffet Service Card" section
    $buffetServiceSelectors = [
        '#buffetServiceCard_feature_div' // Add more selectors if needed
    ];
    // Initialize an array to track processed sections
    $processedBuffetServiceSections = [];
    // Loop through selectors to scrape content
    foreach ($buffetServiceSelectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Handle unwanted elements and lazy-loaded images
            $this->removeUnwantedElements($element);
            $this->handleLazyLoadedImages($element);

            // Extract the dynamic title (if any)
            $titleElement = $element->find('h2', 0); // Find the title dynamically
            $dynamicTitle = $titleElement ? trim($titleElement->plaintext) : 'Buffet Service Card';

            // Extract the buffet content
            $buffetContentElement = $element->find('.a-section', 0); // Locate the content dynamically
            $buffetContentHtml = '';
            if ($buffetContentElement) {
                $buffetContentText = trim($buffetContentElement->plaintext);

                // Clean up special characters and extra whitespace
                $buffetContentText = str_replace(['&lrm;', '&rlm;'], '', $buffetContentText);
                $buffetContentText = preg_replace('/\s+/', ' ', $buffetContentText);

                $buffetContentHtml = '<p>' . htmlspecialchars($buffetContentText) . '</p>';
            }

            // Generate a unique key to track duplicates
            $sectionKey = md5($dynamicTitle . $buffetContentHtml);

            // Check if the section is already processed
            if (!empty($buffetContentHtml) && !isset($processedBuffetServiceSections[$sectionKey])) {
                $content .= '<h3>' . htmlspecialchars($dynamicTitle) . '</h3>';
                $content .= $buffetContentHtml;

                // Mark the section as processed
                $processedBuffetServiceSections[$sectionKey] = true;
            }
        }
    }


    // Define all selectors for remaining sections
    $selectors = [
        '#productDetails_techSpec_section_1',
        '#tech',
        '#bondAboutThisItem_feature_div'
    ];

    // Loop through selectors to scrape additional content
    foreach ($selectors as $selector) {
        $elements = $html->find($selector);
        foreach ($elements as $element) {
            // Handle unwanted elements and lazy-loaded images
            $this->removeUnwantedElements($element);
            $this->handleLazyLoadedImages($element);

            // Pre-clean hover spans before processing
            $element->outertext = $this->removeHoverSpans($element->outertext);

            // Convert h2 to h3 to preserve titles
            $element->outertext = preg_replace('/<h2(.*?)>(.*?)<\/h2>/', '<h3 class="$1">$2</h3>', $element->outertext);

            // Get the HTML and text content
            $elementHtml = trim($element->outertext);
            $elementText = trim($element->plaintext);

            // Create a unique key based on the text content to detect duplicates
            $contentKey = md5($elementText);

            // Check for unique content
            if (!empty($elementText) && !in_array($contentKey, $uniqueTextContent)) {
                $uniqueTextContent[] = $contentKey;
                $content .= $elementHtml;
            }
        }
    }

    return $content;
}
// Group functions to extract and format content//


    public function fetchAdditionalContentFromHtml($html) {
        // Start content with WooCommerce standard table class
        $content = '<table class="woocommerce-product-attributes shop_attributes">';

        // First, try to get data from the product overview
        $rows = $html->find('#productOverview_feature_div table tr');
        
        // If no product overview, try the original selectors (fallback)
        if (empty($rows)) {
            $rows = $html->find('#productDetails_detailBullets_sections1 tr, #productDetails_db_sections .prodDetTable tr');
        }

        // Check if the #bond-technical-specfications-desktop section exists and expand it to get the table
        if (empty($rows)) {
            $section = $html->find('#bond-technical-specfications-desktop', 0);
            if ($section) {
                $table = $section->find('.a-expander-content table', 0);
                if ($table) {
                    $rows = $table->find('tr');
                }
            }
        }

        // Check if the #nic-po-expander-section-desktop section exists and expand it to get the table
        if (empty($rows)) {
            $nicSection = $html->find('#nic-po-expander-section-desktop', 0);
            if ($nicSection) {
                $nicTable = $nicSection->find('.a-expander-content table', 0);
                if ($nicTable) {
                    $rows = $nicTable->find('tr');
                }
            }
        }

        // New Section: Check for "Product details" dynamically
        $productDetailsSection = $html->find('#productFactsDesktop_feature_div', 0); // Locate "Product details" section
        if ($productDetailsSection) {
            $content .= '<tr><th colspan="2" class="woocommerce-product-attributes-item__label">Product details</th></tr>'; // Add title dynamically

            // Find all rows of "Product details" dynamically
            $details = $productDetailsSection->find('.a-fixed-left-grid');
            foreach ($details as $detail) {
                $label = $detail->find('.a-col-left span', 0);
                $value = $detail->find('.a-col-right span', 0);

                if ($label && $value) {
                    $labelText = trim($label->plaintext);
                    $valueText = trim($value->plaintext);

                    $content .= '<tr class="woocommerce-product-attributes-item">';
                    $content .= '<th class="woocommerce-product-attributes-item__label">' . esc_html($labelText) . '</th>';
                    $content .= '<td class="woocommerce-product-attributes-item__value">' . esc_html($valueText) . '</td>';
                    $content .= '</tr>';
                }
            }
        }

        // Loop through the rows and scrape data
        foreach ($rows as $row) {
            $label = $row->find('th', 0) ?: $row->find('td.a-span3', 0);
            $value = $row->find('td', 1) ?: $row->find('td.a-span9', 0);
            
            if ($label && $value) {
                $labelText = trim($label->plaintext);
                $valueText = trim($value->plaintext);
                
                // Add scraped data to the content
                $content .= '<tr class="woocommerce-product-attributes-item">';
                $content .= '<th class="woocommerce-product-attributes-item__label">' . esc_html($labelText) . '</th>';
                $content .= '<td class="woocommerce-product-attributes-item__value">' . esc_html($valueText) . '</td>';
                $content .= '</tr>';
            }
        }

        $content .= '</table>';

        return $content;
    }

    public function fetchShortDescriptionFromHtml($html) {
        $content = '';
        $data = $html->find('#feature-bullets ul li .a-list-item');
        if(isset($data[0])){
            $content .= '<ul>';
            foreach ( $data as $element) {
                $content .= '<li>' .  $element->innertext .'</li>';
            }
            $content .= '</ul>';
        }

        $data = $html->find('#bookDescription_feature_div noscript'); 
        if(isset($data[0])){
            $headlines = array();
            foreach($data as $header) {
                $headlines[] = $header->innertext;
                $content .= $header->innertext;
            }
        }

        $data = $html->find('#bookDescription_feature_div .a-expander-collapsed-height .a-expander-content span');
        if(isset($data[0])){
            $headlines = array();
            foreach($data as $header) {
                $headlines[] = $header->innertext;
                $content .= $header->innertext;
            }
        }

        return $content;
    }
    public function fetchVariationContentFromHtml($html) {
        $content = '';
        $data = $html->find('#feature-bullets ul li .a-list-item');
        if($data) {
            $content .= '<ul>';
            foreach ( $data as $element ) {
                $classes = explode(' ', $element->getAttribute('class'));
                if( in_array('aok-hidden', $classes) ) continue;
                $content .= '<li>' .  $element->text() .'</li>';
            }
            $content .= '</ul>';
        }

        $data = $html->find('#productFactsDesktop_feature_div ul li');
        if($data) {
            $content .= '<ul>';
            foreach ( $data as $element ) {
                $content .= '<li>' .  $element->text() .'</li>';
            }
            $content .= '</ul>';
        }

        $data = $html->find('#productFactsDesktop_feature_div .pfDescContent p', 0);
        if($data) {
            $content .= '<p>' .  $data->text() .'</p>';
        }

        return $content;
    }
    
    private function hasVariationAttributesSafe($attributes) {
        if (empty($attributes) || !is_array($attributes)) return false;
        foreach ($attributes as $attr) {
            $isVariation = isset($attr['variation']) && intval($attr['variation']) === 1;
            $hasOptions = isset($attr['options']) && is_array($attr['options']) && count($attr['options']) > 1;
            if ($isVariation || $hasOptions) return true;
        }
        return false;
    }
   
    public function fetchVariationsDetails() {
    }
   
    public function getAmazonPageHtml($url) {
        return wp_remote_get(esc_url_raw($url));
    }
  
    public function fetchCurrencyFromHtml($html, $silent = false) {
        $currency = null;
    
        // Attempt to find the currency symbol using the new selector
        $currencySymbol = $html->find('.a-price-symbol', 0);
        if ($currencySymbol) {
            $currency = $currencySymbol->text() ?: null;
        }
    
        // If the currency symbol is still not found, you can keep the existing logic
        if (null === $currency) {
            $currencySymbol = $html->find('#apex_desktop span.priceToPay span[aria-hidden] .a-price-symbol', 0);
            if ($currencySymbol) {
                $currency = $currencySymbol->text() ?: null;
            }
        }
    
        if (null === $currency) {
            $currencySymbol = $html->find('#dp-container .a-size-medium.a-color-price', 0);
            if ($currencySymbol) {
                $matches = preg_split('/\d|[1-9]/', trim($currencySymbol->text()));
                $currency = (trim($matches[0]) == 'EUR') ? '€' : ($matches[0] ?: null);
            }
        }
    
        if (null === $currency) {
            $currencySymbol = $html->find('#apex_desktop .apexPriceToPay .a-offscreen', 0);
            if ($currencySymbol) {
                $matches = preg_split('/\d|[1-9]/', trim($currencySymbol->text()));
                $currency = $matches[0] ?: null;
            }
        }
    
        if (null === $currency) {
            $currencySymbol = $html->find('#apex_desktop #sns-base-price', 0);
            if ($currencySymbol) {
                if ($currencySymbol->has_child()) {
                    $currencySymbol->removeChild($currencySymbol->find('span', 0));
                }
                $matches = preg_split('/\d|[1-9]/', trim($currencySymbol->text()));
                $currency = end($matches) ?: null;
            }
        }
    
        return $currency;
    }

    public function user_agent() {
        // Modern Chrome/Firefox user agents (2022-2024)
        $user_agents = array(
            // Chrome (Windows)
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36",
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.6167.140 Safari/537.36",
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.6099.110 Safari/537.36",
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.6045.200 Safari/537.36",
            // Chrome (Mac)
            "Mozilla/5.0 (Macintosh; Intel Mac OS X 13_3_1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36",
            "Mozilla/5.0 (Macintosh; Intel Mac OS X 13_2_1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.6167.140 Safari/537.36",
            // Firefox (Windows)
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:123.0) Gecko/20100101 Firefox/123.0",
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:122.0) Gecko/20100101 Firefox/122.0",
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:121.0) Gecko/20100101 Firefox/121.0",
            // Firefox (Mac)
            "Mozilla/5.0 (Macintosh; Intel Mac OS X 13.3; rv:123.0) Gecko/20100101 Firefox/123.0",
            "Mozilla/5.0 (Macintosh; Intel Mac OS X 13.2; rv:122.0) Gecko/20100101 Firefox/122.0",
            // Chrome (Linux)
            "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36",
            "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/121.0.6167.140 Safari/537.36"
        );
    
        $random = rand(0, count($user_agents) - 1);
        return $user_agents[$random];
    }

    /* Helper method to find best matching option for variation attributes */
    private function findBestMatchingOption($variation_title, $options, $variation_id) {
        // Common patterns to look for in variation titles
        $patterns = [
            '/(\d+)\s*Count\s*\(Pack\s*of\s*(\d+)\)/i',  // "60 Count (Pack of 1)"
            '/(\d+)\s*Count/i',                           // "300 Count"
            '/Pack\s*of\s*(\d+)/i',                       // "Pack of 3"
        ];
        
        foreach ($options as $option) {
            // Direct match (case insensitive)
            if (stripos($variation_title, $option) !== false) {
                return $option;
            }
            
            // Extract numbers from both title and option for comparison
            preg_match_all('/\d+/', $variation_title, $title_numbers);
            preg_match_all('/\d+/', $option, $option_numbers);
            
            if (!empty($title_numbers[0]) && !empty($option_numbers[0])) {
                // Check if the main numbers match (e.g., "300" in both)
                $title_main = $title_numbers[0][0];
                $option_main = $option_numbers[0][0];
                
                if ($title_main === $option_main) {
                    return $option;
                }
            }
        }
        
        // Fallback: try to match based on variation ID patterns
        // Some ASINs might have predictable patterns
        // logImportVerification("⚠️ ATTRIBUTE MATCH - No clear match found for {$variation_id}, using first option: " . $options[0]);
        return $options[0]; // Return first option as fallback
    }
    /* For Product Variation */
    public function getProductVariationFromHtml($html, $base_url='') {
        $default_attributes = [];
        $productVariation = $html->find('#twister_feature_div #twisterContainer',0);
        // For single attribute
        if($productVariation) {
            $attributeId = $html->find('#twister_feature_div #twisterContainer form div',0)->getAttribute('id');
            $attributeName = str_replace('variation','',str_replace('_',' ',$attributeId));
            $attributeSlugName = str_replace('variation_','',$attributeId);
            $attributeLabelName = $html->find('#twister_feature_div form div label.a-form-label',0);
            if( $attributeLabelName ) {
                $attributeLabelName = str_replace(':','',trim($attributeLabelName->text()));
            }
            $attributeLabelValue = $html->find('#twister_feature_div form div span.selection',0);
            if( $attributeLabelValue ) {
                $attributeLabelValue = trim($attributeLabelValue->text());
            }

            $variations = $html->find('#twister_feature_div form div ul li');

            if ($variations) {
                $options = $p_variations = [];
                foreach ($variations as $variation) {
                    $name = $variation->find('.twisterImageDiv img',0);
                    $name2 = $variation->find('.a-list-item .twisterTextDiv',0);
                    $name3 = $variation->find('.imgSwatch',0);
                    $url = $variation->getAttribute('data-dp-url');
                    $price = $salePrice = '';
                    if ($name) {
                        $options[] = $name = trim($name->getAttribute('alt'));
                    } elseif ($name2) {
                        $options[] = $name = trim($name2->text());
                    } elseif ($name3) {
                        $options[] = $name = trim($name3->getAttribute('alt'));
                    }
                    $priceString = $variation->find('p.twisterSwatchPrice', 0);
                    if ($priceString) {
                        $price = $this->parsePrice(trim($variation->find('p.twisterSwatchPrice', 0)->text()));
                    }
                    
                    $class_string = $variation->getAttribute('class');
                    $classes = explode(' ', $class_string);
                    if (in_array('selected', $classes) || in_array('swatchSelect', $classes)) {
                        $asin = $variation->getAttribute('data-defaultasin');
                        if (!$asin) {
                            $exp1 = explode("/dp/", $base_url);
                            $exp2 = explode('/', end($exp1));
                            $reverse = array_reverse($exp2);
                            $asin = array_pop($reverse);
                        }
                        $fetchPrice = $this->fetchPriceFromHtml($html);
                        $price = $this->parsePrice($fetchPrice['regular_price']);
                        $salePrice = $this->parsePrice($fetchPrice['amount']);
                        $default_attributes[strtolower($attributeSlugName)] = $name;
                    } else {
                        $url = $variation->getAttribute('data-dp-url');
                        $url = $base_url . '' . $url; 
                        $asin = $variation->getAttribute('data-defaultasin');
                    }
                    $url = $variation->getAttribute('data-dp-url');
                    $p_variations[] = array(
                        'attributes' => array(
                            strtolower($attributeSlugName) => $name,
                        ),
                        'sku' => $asin,
                        'regular_price' => $price,
                        'sale_price' => $salePrice,
                    );
                }
                $attributes_data = array(
                    array('name' => $attributeLabelName, 'slug' => $attributeSlugName, 'options' => $options, 'visible' => 1, 'variation' => 1)
                );
                return [
                    'attributes_data' => $attributes_data,
                    'product_variations' => $p_variations,
                    'default_attributes' => $default_attributes
                ];
            }
        }

        // For multiple attribute => In progress
        $productVariation = $html->find('#twister_feature_div #twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister',0);
            if ($productVariation) {
                $fetchPrice = $this->fetchPriceFromHtml($html);
                $p_variations = $attributes_data = [];
                // Check for single value
                $attributes = $html->find('#twister_feature_div #twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister .inline-twister-singleton-header');
                if ($attributes) {
                    $selected_attributes = [];
                    foreach ($attributes as $attribute) {
                        $attributeId = $attribute->getAttribute('id');
                        $attributeSlugName = str_replace('inline-twister-singleton-header-', '', $attributeId);
                        $attributeName = str_replace('_', ' ', str_replace('inline-twister-singleton-header-', '', $attributeId));
                        $options = [];
                        $title = $name = $attributeLabelName = '';
                        $titleString = $attribute->find('.inline-twister-dim-title', 0);
                        if ($titleString) {
                            $title = trim($titleString->text());
                            $valString = $titleString->nextSibling();
                            if ($valString) {
                                $options[] = $name = trim($valString->text());
                            }
                        }
                        if ($title) {
                            $attributeLabelName = str_replace(':', '', $title);
                        }
                        $selected_attributes[strtolower($attributeSlugName)] = $name;
                        $default_attributes = $selected_attributes;
                        
                        $attributes_data[] = array('name' => $attributeLabelName, 'slug' => $attributeSlugName, 'options' => $options, 'visible' => 1, 'variation' => 1);
                    }
                    $asin = '';
                    $fetchPrice = $this->fetchPriceFromHtml($html);
                    $p_variations[] = array(
                        'attributes' => $selected_attributes,
                        'sku' => $asin,
                        'regular_price' => $this->parsePrice($fetchPrice['regular_price']),
                        'sale_price' => $this->parsePrice($fetchPrice['amount']),
                    );
                }
                // Return
                return [
                    'attributes_data' => $attributes_data,
                    'product_variations' => $p_variations,
                    'default_attributes' => $default_attributes
                ];
            }
        // Return
        return [
            'attributes_data' => [],
            'product_variations' => [],
            'default_attributes' => [],
        ];
    }

    /* For Multi Level Product Variation */
    public function getProductMultiLevelVariationFromHtml($html, $base_url='') {
        $default_attributes = $attributes_data = [];
        $attributes = $html->find('#twister_feature_div #twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister .inline-twister-row');
        if( $attributes ) {
            foreach ($attributes as $key => $attribute) {
                $attributeId = $attribute->getAttribute('id');
                $attributeSlugName = str_replace('inline-twister-row-','',$attributeId);

                $selectedTitleSelector = '#inline-twister-dim-title-' . str_replace('inline-twister-row-','',$attributeId);
                $selectedTitle = $attribute->find($selectedTitleSelector, 0);
                $selectedTitleClassSelector = $attribute->find('.dimension-text', 0);
                if( $selectedTitle ) {
                    $label = $attribute->find($selectedTitleSelector . ' > div > span', 0);
                    $value = $attribute->find($selectedTitleSelector . ' > div > span', 1);
                    if( $label ) {
                        $labelText = $label->text();
                    }
                    $value = $attribute->find('.dimension-text > div > span', 1);
                    if( $value ) {
                        $valueText = $value->text();
                    }
                } else if( $selectedTitleClassSelector ) {
                    $label = $attribute->find('.dimension-text > div > span', 0);
                    if( $label ) {
                        $labelText = $label->text();
                    }
                    $value = $attribute->find('.dimension-text > div > span', 1);
                    if( $value ) {
                        $valueText = $value->text();
                    }
                }

                if( $labelText ) {
                    $attributeLabelName = str_replace(':','',trim($labelText));
                }
                $options = [];
                $variations = $attribute->find('ul li');
                if( $variations ) {
                    foreach ($variations as $key => $variation) {
                        // hidden check
                        $classes = explode(' ', $variation->getAttribute('class'));
                        if( in_array('swatch-prototype', $classes) ) continue;
                        $title = '';
                        $titleString = $variation->find('.swatch-title-text',0);
                        $titleString2 = $variation->find('img.swatch-image',0);
                        if( $titleString ) {
                            $title = $titleString->text();
                        } elseif ($titleString2) {
                            $title = $titleString2->getAttribute('alt');
                        }

                        $options[] = $title;

                        // check if selected
                        $selectedExists = $variation->find('.a-button-selected',0);
                        if( $selectedExists ) {
                            $default_attributes[strtolower($attributeSlugName)] = $title;
                        } else {}
                    }

                    $attributes_data[] = array( 
                        'name' => $attributeLabelName, 
                        'slug' => $attributeSlugName, 
                        'options' => $options, 
                        'visible' => 1, 
                        'variation' => 1 
                    );
                }
            }
        }

        $attributes = $html->find('#softlinesTwister_feature_div #twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister .inline-twister-row');
        if( $attributes ) {
            foreach ($attributes as $key => $attribute) {
                $attributeId = $attribute->getAttribute('id');
                $attributeSlugName = str_replace('inline-twister-row-','',$attributeId);

                $selectedTitleSelector = '#inline-twister-dim-title-' . str_replace('inline-twister-row-','',$attributeId);
                $selectedTitle = $attribute->find($selectedTitleSelector, 0);
                $selectedTitleClassSelector = $attribute->find('.dimension-text', 0);
                if( $selectedTitle ) {
                    $label = $attribute->find($selectedTitleSelector . ' > div > span', 0);
                    $value = $attribute->find($selectedTitleSelector . ' > div > span', 1);
                    if( $label ) {
                        $labelText = $label->text();
                    }
                    $value = $attribute->find('.dimension-text > div > span', 1);
                    if( $value ) {
                        $valueText = $value->text();
                    }
                } else if( $selectedTitleClassSelector ) {
                    $label = $attribute->find('.dimension-text > div > span', 0);
                    if( $label ) {
                        $labelText = $label->text();
                    }
                    $value = $attribute->find('.dimension-text > div > span', 1);
                    if( $value ) {
                        $valueText = $value->text();
                    }
                }

                if( $labelText ) {
                    $attributeLabelName = str_replace(':','',trim($labelText));
                }

                $options = [];
                $variations = $attribute->find('ul li');
                if( $variations ) {
                    foreach ($variations as $key => $variation) {
                        // hidden check
                        $classes = explode(' ', $variation->getAttribute('class'));
                        if( in_array('swatch-prototype', $classes) ) continue;
                        $title = '';
                        $titleString = $variation->find('.swatch-title-text',0);
                        $titleString2 = $variation->find('img.swatch-image',0);
                        if( $titleString ) {
                            $title = $titleString->text();
                        } elseif ($titleString2) {
                            $title = $titleString2->getAttribute('alt');
                        }

                        $options[] = $title;

                        // check if selected
                        $selectedExists = $variation->find('.a-button-selected',0);
                        if( $selectedExists ) {
                            $default_attributes[strtolower($attributeSlugName)] = $title;
                        } else {}
                    }

                    $attributes_data[] = array( 
                        'name' => $attributeLabelName, 
                        'slug' => $attributeSlugName, 
                        'options' => $options, 
                        'visible' => 1, 
                        'variation' => 1 
                    );
                }
            }
        }

        return [
            'attributes_data' => $attributes_data,
            'default_attributes' => $default_attributes
        ];
    }

    public function getSkusFromHtml($html, $skus = []) {
        // Attributes that might contain ASINs
        $attributes_to_check = ['data-asin', 'data-defaultasin', 'data-csa-c-item-id', 'data-asin-id', 'data-sku', 'data-product-id'];
        
        // Extract ASINs from common attributes
        foreach ($attributes_to_check as $attr) {
            $elements = $html->find("[$attr]");
            foreach ($elements as $element) {
                $asin = $element->getAttribute($attr);
                if ($asin && strlen($asin) == 10) {
                    $skus[] = $asin;
                }
            }
        }

        // Check for ASINs in data-dp-url attributes
        $elements = $html->find('[data-dp-url]');
        foreach ($elements as $element) {
            $dp_url = $element->getAttribute('data-dp-url');
            if (preg_match('/\/dp\/([A-Z0-9]{10})/', $dp_url, $matches)) {
                $skus[] = $matches[1];
            }
        }

        // Extract SKUs from embedded JavaScript data
        $scripts = $html->find('script');
        foreach ($scripts as $script) {
            $scriptContent = $script->innertext;
            
            // Extract ASINs from asinVariationValues
            if (preg_match('/var\s+asinVariationValues\s*=\s*({.*?});/s', $scriptContent, $matches)) {
                $asinVariationValues = json_decode($matches[1], true);
                if ($asinVariationValues) {
                    $skus = array_merge($skus, array_keys($asinVariationValues));
                }
            }
            
            // Extract ASINs from dimensionValuesDisplayData
            if (preg_match('/var\s+dimensionValuesDisplayData\s*=\s*({.*?});/s', $scriptContent, $matches)) {
                $dimensionData = json_decode($matches[1], true);
                if ($dimensionData) {
                    foreach ($dimensionData as $dimension) {
                        if (isset($dimension['dimensionValues'])) {
                            foreach ($dimension['dimensionValues'] as $value) {
                                if (isset($value['asin']) && strlen($value['asin']) == 10) {
                                    $skus[] = $value['asin'];
                                }
                            }
                        }
                    }
                }
            }
            
            // Extract ASINs from initial_asins
            if (preg_match('/"initial_asins"\s*:\s*(\[.*?\])/', $scriptContent, $matches)) {
                $initialAsins = json_decode($matches[1], true);
                if ($initialAsins) {
                    $skus = array_merge($skus, $initialAsins);
                }
            }

            // Extract parent ASIN
            if (preg_match('/"parentAsin"\s*:\s*"([A-Z0-9]{10})"/', $scriptContent, $matches)) {
                $skus[] = $matches[1];
            }

            // Extract ASINs from other possible JSON structures
            if (preg_match_all('/"asin"\s*:\s*"([A-Z0-9]{10})"/', $scriptContent, $matches)) {
                $skus = array_merge($skus, $matches[1]);
            }
        }

        // Look for SKUs in twister data
        $twisterData = $html->find('#twister-plus-inline-twister', 0);
        if ($twisterData) {
            $variationItems = $twisterData->find('.a-declarative');
            foreach ($variationItems as $item) {
                $dataAsin = $item->getAttribute('data-asin');
                if ($dataAsin && strlen($dataAsin) == 10) {
                    $skus[] = $dataAsin;
                }
            }
        }

        // Enhanced variation detection for modern Amazon layouts
        // Look for variation buttons and links
        $variationSelectors = [
            'button[data-asin]',
            'a[data-asin]', 
            '.swatch-list-item[data-asin]',
            '.inline-twister-swatch[data-asin]',
            '[data-dp-url*="/dp/"]',
            'input[name*="asin"]',
            '.twister-plus-inline-twister-card [data-asin]',
            '.size-option[data-asin]',
            '.color-option[data-asin]'
        ];

        foreach ($variationSelectors as $selector) {
            $elements = $html->find($selector);
            foreach ($elements as $element) {
                // Try data-asin first
                $asin = $element->getAttribute('data-asin');
                
                // If no data-asin, try extracting from data-dp-url
                if (!$asin) {
                    $dpUrl = $element->getAttribute('data-dp-url');
                    if ($dpUrl && preg_match('/\/dp\/([A-Z0-9]{10})/', $dpUrl, $matches)) {
                        $asin = $matches[1];
                    }
                }
                
                // If still no ASIN, try href attribute
                if (!$asin) {
                    $href = $element->getAttribute('href');
                    if ($href && preg_match('/\/dp\/([A-Z0-9]{10})/', $href, $matches)) {
                        $asin = $matches[1];
                    }
                }
                
                if ($asin && strlen($asin) == 10 && preg_match('/^[A-Z0-9]{10}$/', $asin)) {
                    $skus[] = $asin;
                }
            }
        }

        // Remove duplicates and invalid ASINs
        $skus = array_unique($skus);
        $skus = array_filter($skus, function($asin) {
            return preg_match('/^[A-Z0-9]{10}$/', $asin);
        });

        // Debug: Log found SKUs


        return array_values($skus);
    }

    public function extractAllProductVariants($html, $url = '')
    {
        $variants = [];
        $dimensionValuesDisplayData = [];
        $asinVariationValues = [];
        $dataToReturn = null;

        // Extract script content
        $scripts = $html->find('script');
        foreach ($scripts as $script) {
            $scriptContent = $script->innertext;

            // Look for dimensionValuesDisplayData
            if (preg_match('/var\s+dimensionValuesDisplayData\s*=\s*({.*?});/s', $scriptContent, $matches)) {
                $dimensionValuesDisplayData = json_decode($matches[1], true);
            }

            // Look for asinVariationValues
            if (preg_match('/var\s+asinVariationValues\s*=\s*({.*?});/s', $scriptContent, $matches)) {
                $asinVariationValues = json_decode($matches[1], true);
            }

            // Look for dataToReturn
            if (preg_match('/var\s+dataToReturn\s*=\s*({.*?});/s', $scriptContent, $matches)) {
                $dataToReturn = json_decode($matches[1], true);
            }

            // If we've found all, break the loop
            if (!empty($dimensionValuesDisplayData) && !empty($asinVariationValues) && $dataToReturn !== null) {
                break;
            }
        }

        // Process the variation data
        if (!empty($dataToReturn) && isset($dataToReturn['dimensionValuesData'])) {
            foreach ($dataToReturn['dimensionValuesData'] as $dimension => $values) {
                foreach ($values as $value) {
                    if (isset($value['asin']) && isset($value['dimensionValue'])) {
                        $variants[$value['asin']][$dimension] = $value['dimensionValue'];
                    }
                }
            }
        } elseif (!empty($dimensionValuesDisplayData) && !empty($asinVariationValues)) {
            foreach ($asinVariationValues as $asin => $variationData) {
                $variant = ['asin' => $asin];
                foreach ($variationData as $dimensionIndex => $valueIndex) {
                    $dimensionKey = array_keys($dimensionValuesDisplayData)[$dimensionIndex];
                    $variant[$dimensionKey] = $dimensionValuesDisplayData[$dimensionKey]['displayValuesJson'][$valueIndex];
                }
                $variants[$asin] = $variant;
            }
        }

        // If we still don't have all variants, try to extract from HTML
        if (empty($variants)) {
            $list_items = $html->find('ul[data-action="a-button-group"] li');
            foreach ($list_items as $item) {
                $asin = $item->getAttribute('data-defaultasin') ?: $item->getAttribute('data-csa-c-item-id');
                if ($asin && strlen($asin) == 10) {
                    $variants[$asin] = [
                        'asin' => $asin,
                        'title' => $item->getAttribute('title')
                    ];
                }
            }
        }

        // Add debug logging
        //error_log("Extracted variants: " . print_r($variants, true));

        return $variants;
    }



####################################################################################
/**
 * Retrieves product attribute data from the given HTML.
 * Returns an array of attributes (each with name, slug, selected value, and, if applicable, options).
 */
public function getProductAttributeFromHtml($html) {
    $attributes_data = array();
    $known_size_slugs = array('size', 'tama-o', 'tamaño', 'talla', 'größe');
    $known_color_slugs = array('color', 'colour', 'farbe');

    // Re-ordered list: legacy variation containers first, then inline twister ones.
    $attribute_containers = array(
        '#variation_fit_type',
        '#variation_color_name',
        '#variation_size_name',
        '#twister-plus-inline-twister-card #twister-plus-inline-twister div[id^="inline-twister-row"]',
        '#twister_feature_div #twisterContainer',
        '#twister-plus-inline-twister-card #twister-plus-inline-twister div[id*="inline-twister-row"]',
        '#softlinesTwister_feature_div #twisterContainer',
        '#twister_feature_div #twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister',
        '#softlinesTwister_feature_div #twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister',
        '#twister-plus-inline-twister-card #twister-plus-inline-twister',
        'div[id^="inline-twister-expander-content-"]',
        'ul[data-action="a-button-group"]'
    );

    foreach ($attribute_containers as $container) {
        // For text-based swatch systems using data-a-button-group
        if ($container === 'ul[data-action="a-button-group"]') {
            $swatchLists = $html->find($container);
            foreach ($swatchLists as $swatchList) {
                $attribute = $this->extractTextSwatchAttribute($swatchList);
                if (!empty($attribute)) {
                    // Normalize slug: remove trailing "-name" so that duplicates match.
                    $attribute['slug'] = preg_replace('/-name$/', '', $attribute['slug']);
                    
                    // Standardize common attribute types (size, color)
                    $standardized_slug = $this->standardizeAttributeSlug($attribute['slug'], $known_size_slugs, $known_color_slugs);
                    $attribute['slug'] = $standardized_slug;
                    
                    // Check if we should merge or add this attribute
                    if (!isset($attributes_data[$standardized_slug])) {
                        $attributes_data[$standardized_slug] = $attribute;
                    } else if (isset($attribute['options']) && count($attribute['options']) > 0) {
                        // Keep the attribute with options if one doesn't have them
                        if (!isset($attributes_data[$standardized_slug]['options']) || 
                            count($attributes_data[$standardized_slug]['options']) < count($attribute['options'])) {
                            $attributes_data[$standardized_slug] = $attribute;
                        }
                    }
                }
            }
            continue;
        }
        
        // For inline twister rows and expander content.
        if (strpos($container, 'inline-twister-row') !== false || strpos($container, 'inline-twister-expander-content') !== false) {
            $rows = $html->find($container);
            foreach ($rows as $row) {
                $attribute = $this->extractInlineTwisterAttribute($row);
                if (!empty($attribute)) {
                    // Normalize slug: remove trailing "-name" so that duplicates match.
                    $attribute['slug'] = preg_replace('/-name$/', '', $attribute['slug']);
                    
                    // Standardize common attribute types (size, color)
                    $standardized_slug = $this->standardizeAttributeSlug($attribute['slug'], $known_size_slugs, $known_color_slugs);
                    $attribute['slug'] = $standardized_slug;
                    
                    // Check if we should merge or add this attribute
                    if (!isset($attributes_data[$standardized_slug])) {
                        $attributes_data[$standardized_slug] = $attribute;
                    } else if (isset($attribute['options']) && count($attribute['options']) > 0) {
                        // Keep the attribute with options if one doesn't have them
                        if (!isset($attributes_data[$standardized_slug]['options']) || 
                            count($attributes_data[$standardized_slug]['options']) < count($attribute['options'])) {
                            $attributes_data[$standardized_slug] = $attribute;
                        }
                    }
                }
            }
            continue;
        }

        // For legacy variation containers.
        $productVariation = $html->find($container, 0);
        if ($productVariation) {
            $extracted = $this->extractAttributes($html, $container);
            if (!empty($extracted)) {
                foreach ($extracted as $attribute) {
                    // Clean the name and slug.
                    $attribute['name'] = trim(str_replace(':', '', html_entity_decode($attribute['name'], ENT_QUOTES | ENT_HTML5, 'UTF-8')));
                    $attribute['slug'] = str_replace('-58', '', $attribute['slug']);
                    // Normalize slug: remove trailing "-name"
                    $attribute['slug'] = preg_replace('/-name$/', '', $attribute['slug']);
                    
                    // Standardize common attribute types (size, color)
                    $standardized_slug = $this->standardizeAttributeSlug($attribute['slug'], $known_size_slugs, $known_color_slugs);
                    $attribute['slug'] = $standardized_slug;
                    
                    // Only process if it has a selected value
                    if (isset($attribute['selected']) && !empty($attribute['selected'])) {
                        if (!isset($attributes_data[$standardized_slug])) {
                            $attributes_data[$standardized_slug] = $attribute;
                        } else if (isset($attribute['options']) && count($attribute['options']) > 0) {
                            // Keep the attribute with options if one doesn't have them
                            if (!isset($attributes_data[$standardized_slug]['options']) || 
                                count($attributes_data[$standardized_slug]['options']) < count($attribute['options'])) {
                                // Preserve the name from the existing attribute if it's cleaner
                                if (isset($attributes_data[$standardized_slug]['name']) && 
                                    strpos($attributes_data[$standardized_slug]['name'], 'Name') === false &&
                                    strpos($attribute['name'], 'Name') !== false) {
                                    $attribute['name'] = $attributes_data[$standardized_slug]['name'];
                                }
                                $attributes_data[$standardized_slug] = $attribute;
                            }
                        }
                    }
                }
            }
        }
    }

    // Final deduplication pass - remove duplicates by semantic meaning
    $final_attributes = array();
    
    // Define semantic groups for common attributes in multiple languages
    $semantic_groups = array(
        'size' => array('size', 'maat', 'talla', 'taille', 'größe', 'tamaño', 'tamanho', 'dimensione'),
        'color' => array('color', 'colour', 'kleur', 'farbe', 'couleur', 'colore', 'cor', 'kolor'),
        'brand' => array('brand', 'merk', 'marke', 'marque', 'marca', 'marka'),
        'style' => array('style', 'stijl', 'stil', 'estilo', 'stile'),
        'material' => array('material', 'materiaal', 'matériau', 'materiale', 'material')
    );
    
    $used_semantic_types = array();
    
    foreach ($attributes_data as $attribute) {
        $name_lower = strtolower(trim($attribute['name']));
        $slug_lower = strtolower($attribute['slug']);
        
        // Find which semantic group this attribute belongs to
        $semantic_type = null;
        foreach ($semantic_groups as $group_name => $group_words) {
            foreach ($group_words as $word) {
                if ($name_lower === $word || $slug_lower === $word || 
                    strpos($name_lower, $word) !== false || strpos($slug_lower, $word) !== false) {
                    $semantic_type = $group_name;
                    break 2;
                }
            }
        }
        
        // If no semantic type found, use the slug as unique identifier
        if (!$semantic_type) {
            $semantic_type = $slug_lower;
        }
        
        // Check if we already have this semantic type
        if (isset($used_semantic_types[$semantic_type])) {
            // Keep the one with more options or better name (prefer English)
            $existing = $used_semantic_types[$semantic_type];
            $current_options_count = isset($attribute['options']) ? count($attribute['options']) : 0;
            $existing_options_count = isset($existing['options']) ? count($existing['options']) : 0;
            
            $should_replace = false;
            
            // Prefer the one with more options
            if ($current_options_count > $existing_options_count) {
                $should_replace = true;
            } 
            // If same options count, prefer English names
            else if ($current_options_count === $existing_options_count) {
                $current_is_english = in_array($name_lower, array('size', 'color', 'brand', 'style', 'material'));
                $existing_is_english = in_array(strtolower($existing['name']), array('size', 'color', 'brand', 'style', 'material'));
                
                if ($current_is_english && !$existing_is_english) {
                    $should_replace = true;
                }
            }
            
            if ($should_replace) {
                // Remove existing and add current
                $key_to_remove = array_search($existing, $final_attributes);
                if ($key_to_remove !== false) {
                    unset($final_attributes[$key_to_remove]);
                }
                $final_attributes[] = $attribute;
                $used_semantic_types[$semantic_type] = $attribute;
            }
            // Otherwise keep existing, skip current
        } else {
            // First time seeing this semantic type
            $final_attributes[] = $attribute;
            $used_semantic_types[$semantic_type] = $attribute;
        }
    }
    
    return array_values($final_attributes);
}

/**
 * Extracts the inline twister attribute from a row.
 * Returns an array with the attribute name, slug, selected value, and all available options.
 */
private function extractInlineTwisterAttribute($row) {
    // Try to get the attribute name from the expected span.
    $nameContainer = $row->find('span.a-color-secondary', 0);
    if ($nameContainer) {
        $attributeName = trim(str_replace(':', '', $nameContainer->plaintext));
    } else {
        // Fallback: if the element's id matches the new pattern, extract attribute name.
        if (isset($row->id) && preg_match('/^inline-twister-expander-content-(.+)$/', $row->id, $matches)) {
            $attributeName = ucwords(str_replace('_', ' ', $matches[1]));
        } else {
            return array();
        }
    }
    $slug = $this->generateAttributeSlug($attributeName);

    // Determine the selected value.
    $selected = '';
    $selectedValueEl = $row->find('span.inline-twister-dim-title-value', 0);
    if ($selectedValueEl) {
        $selected = html_entity_decode(trim($selectedValueEl->plaintext), ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }
    if (empty($selected)) {
        $selectedButton = $row->find('span.a-button-selected', 0);
        if ($selectedButton) {
            $titleTextSpan = $selectedButton->find('span.swatch-title-text', 0);
            if ($titleTextSpan) {
                $selected = html_entity_decode(trim($titleTextSpan->plaintext), ENT_QUOTES | ENT_HTML5, 'UTF-8');
            }
        }
    }
    if (empty($selected)) {
        $selectedOption = $row->find('li.swatchSelect', 0);
        if ($selectedOption) {
            $selected = $this->getOptionText($selectedOption);
        }
    }

    // Gather available options.
    $options = array();
    $liElements = $row->find('ul li');
    if ($liElements) {
        foreach ($liElements as $li) {
            $img = $li->find('img', 0);
            $optionText = $img ? trim($img->getAttribute('alt')) : '';
            if (empty($optionText)) {
                $buttonText = $li->find('span.swatch-title-text', 0);
                $optionText = $buttonText ? trim($buttonText->plaintext) : '';
            }
            if (!empty($optionText)) {
                $options[] = $optionText;
            }
            if ($li->getAttribute('data-initiallyselected') === 'true') {
                $selected = $optionText;
            }
        }
    }
    if (empty($options)) {
        $allOptions = $row->find('li.swatch-list-item-text, li.inline-twister-swatch');
        foreach ($allOptions as $option) {
            $titleTextSpan = $option->find('span.swatch-title-text', 0);
            if ($titleTextSpan) {
                $optionText = html_entity_decode(trim($titleTextSpan->plaintext), ENT_QUOTES | ENT_HTML5, 'UTF-8');
                if (!empty($optionText)) {
                    $options[] = $optionText;
                }
            }
        }
    }
    if (empty($options)) {
        $allButtons = $row->find('span.a-button');
        foreach ($allButtons as $button) {
            $titleTextSpan = $button->find('span.swatch-title-text', 0);
            if ($titleTextSpan) {
                $optionText = html_entity_decode(trim($titleTextSpan->plaintext), ENT_QUOTES | ENT_HTML5, 'UTF-8');
                if (!empty($optionText)) {
                    $options[] = $optionText;
                }
            }
        }
    }
    if (empty($selected) && !empty($options)) {
        $selected = $options[0];
    }
    if (empty($selected) && empty($options)) {
        return array();
    }

    return array(
        'name'      => $attributeName,
        'slug'      => $slug,
        'selected'  => $selected,
        'options'   => $options,
        'visible'   => 1,
        'variation' => 1
    );
}

/**
 * Extracts text-based swatch attributes from ul[data-action="a-button-group"] elements.
 * Handles attributes like pattern_name with height values (24"H, 30"H, etc.)
 */
private function extractTextSwatchAttribute($swatchList) {
    // Get attribute name from data-a-button-group
    $buttonGroupData = $swatchList->getAttribute('data-a-button-group');
    if (empty($buttonGroupData)) {
        return array();
    }
    
    // Parse the JSON to get the attribute name
    $buttonGroupJson = html_entity_decode($buttonGroupData, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $buttonGroupArray = json_decode($buttonGroupJson, true);
    if (!isset($buttonGroupArray['name'])) {
        return array();
    }
    
    $attributeKey = $buttonGroupArray['name'];
    $attributeName = ucwords(str_replace('_', ' ', $attributeKey));
    $slug = $this->generateAttributeSlug($attributeName);
    
    // Find selected value and all options
    $selected = '';
    $options = array();
    
    $listItems = $swatchList->find('li');
    foreach ($listItems as $li) {
        // Get the text from the swatch-title-text-display span
        $textSpan = $li->find('span.swatch-title-text-display', 0);
        if ($textSpan) {
            $optionText = html_entity_decode(trim($textSpan->plaintext), ENT_QUOTES | ENT_HTML5, 'UTF-8');
            if (!empty($optionText)) {
                $options[] = $optionText;
                
                // Check if this is the selected option
                $button = $li->find('span.a-button-selected', 0);
                if ($button) {
                    $selected = $optionText;
                }
            }
        }
    }
    
    // If no selected value found, use the first option
    if (empty($selected) && !empty($options)) {
        $selected = $options[0];
    }
    
    // Return empty if we don't have enough data
    if (empty($attributeName) || empty($options)) {
        return array();
    }
    
    return array(
        'name'      => $attributeName,
        'slug'      => $slug,
        'selected'  => $selected,
        'options'   => $options,
        'visible'   => 1,
        'variation' => 1
    );
}

/**
 * Extracts attributes for legacy structures.
 * This function handles both swatch-based attributes (Fit Type, Colour) and dropdown-based attributes (Size).
 * It returns an array of attribute arrays.
 */
private function extractAttributes($html, $container) {
    $attributes = array();

    // For dropdown-based attributes (size, color).
    if ($container === '#variation_size_name' || $container === '#variation_color_name') {
        $singletonDiv = $html->find($container, 0);
        if ($singletonDiv) {
            $attribute_name = $this->getAttributeName($singletonDiv);
            $dropdownData = $this->getSingletonOption($html, $container);
            if (!empty($attribute_name) && !empty($dropdownData['options'])) {
                $slug = $this->generateAttributeSlug($attribute_name);
                // If no explicit selection, choose the first available option.
                if (empty($dropdownData['selected'])) {
                    $dropdownData['selected'] = $dropdownData['options'][0];
                }
                $attributes[] = array(
                    'name'      => $attribute_name,
                    'slug'      => $slug,
                    'selected'  => html_entity_decode($dropdownData['selected'], ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                    'options'   => $dropdownData['options'],
                    'visible'   => 1,
                    'variation' => 1
                );
            }
        }
        return $attributes;
    }

    // For swatch-based attributes (Fit Type, etc.)
    if (strpos($container, 'variation_') !== false && $container !== '#variation_size_name' && $container !== '#variation_color_name') {
        $variationDiv = $html->find($container, 0);
        if ($variationDiv) {
            // Extract attribute name from the label.
            $label = $variationDiv->find('label.a-form-label', 0);
            $attribute_name = $label ? trim(str_replace(':', '', $label->plaintext)) : '';
            
            // Extract the selected value from the span with class "selection".
            $selectedEl = $variationDiv->find('span.selection', 0);
            $selected = $selectedEl ? trim($selectedEl->plaintext) : '';
            
            // Gather available options from the swatch list.
            $options = array();
            $swatchList = $variationDiv->find('ul.swatches, ul.swatchesRectangle, ul.imageSwatches', 0);
            if ($swatchList) {
                foreach ($swatchList->find('li') as $li) {
                    $p = $li->find('p', 0);
                    if ($p) {
                        $optionText = trim($p->plaintext);
                        if (!empty($optionText)) {
                            $options[] = $optionText;
                        }
                    }
                }
            }
            
            // Only add the attribute if we have a name and at least one value.
            if (!empty($attribute_name) && (!empty($selected) || !empty($options))) {
                $slug = $this->generateAttributeSlug($attribute_name);
                $attributes[] = array(
                    'name'      => $attribute_name,
                    'slug'      => $slug,
                    'selected'  => $selected,
                    'options'   => $options,
                    'visible'   => 1,
                    'variation' => 1
                );
            }
        }
    }

    // For legacy structures that use twisterContainer.
    if (strpos($container, 'twisterContainer') !== false) {
        $variationDivs = $html->find($container . ' div[id^="variation_"]');
        foreach ($variationDivs as $variation) {
            $label = $variation->find('label.a-form-label', 0);
            if ($label) {
                $attribute_name = trim(str_replace(':', '', $label->plaintext));
                // Try to get the selected value.
                $selected = '';
                $dropdown = $variation->find('select option[selected]', 0);
                if ($dropdown) {
                    $selected = trim($dropdown->plaintext);
                } else {
                    $selected = $this->getSelectedOption($variation);
                }
                if (!empty($attribute_name) && !empty($selected)) {
                    $slug = $this->generateAttributeSlug($attribute_name);
                    $attributes[] = array(
                        'name'      => $attribute_name,
                        'slug'      => $slug,
                        'selected'  => html_entity_decode($selected, ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                        'visible'   => 1,
                        'variation' => 1
                    );
                }
            }
        }
    }

    return $attributes;
}

/**
 * Helper: For singleton containers (e.g. #variation_size_name) returns an array with:
 * - 'selected': the default selected value (if any; if none, choose the first available)
 * - 'options': an array of all available options (excluding the "Select" placeholder)
 */
private function getSingletonOption($html, $container) {
    $result = array(
        'selected' => '',
        'options'  => array()
    );
    
    $select = $html->find($container . ' select', 0);
    if ($select) {
        foreach ($select->find('option') as $opt) {
            $opt_text = trim($opt->plaintext);
            // Skip the default "Select" prompt.
            if (strcasecmp($opt_text, 'Select') === 0 || empty($opt_text)) {
                continue;
            }
            // Check if the option is available based on its class.
            $class = $opt->getAttribute('class');
            if ($class && strpos($class, 'dropdownAvailable') !== false) {
                $result['options'][] = $opt_text;
            }
        }
        
        // Capture an explicitly selected option, if present.
        $selectedOption = $html->find($container . ' select option[selected]', 0);
        if ($selectedOption) {
            $selectedText = trim($selectedOption->plaintext);
            if (strcasecmp($selectedText, 'Select') === 0) {
                $selectedText = '';
            }
            if (!empty($selectedText)) {
                $result['selected'] = $selectedText;
            }
        }
        
        // If no valid selection is found, default to the first available option.
        if (empty($result['selected']) && !empty($result['options'])) {
            $result['selected'] = $result['options'][0];
        }
    }
    
    return $result;
}

/**
 * Helper: Extracts the attribute name from a variation container.
 */
private function getAttributeName($variation) {
    $label = $variation->find('label.a-form-label', 0);
    if (!$label) {
        $label = $variation->find('span.selection', 0);
    }
    return $label ? trim(html_entity_decode($label->plaintext, ENT_QUOTES | ENT_HTML5, 'UTF-8')) : '';
}

/**
 * Helper: Generates a slug from the attribute name.
 */
private function generateAttributeSlug($text) {
    $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9]+/', '-', $text);
    return trim($text, '-');
}

/**
 * Helper: Standardizes common attribute slugs to prevent duplicates.
 * For example, "size-name", "tama-o", and "talla" should all map to "size".
 */
private function standardizeAttributeSlug($slug, $size_slugs, $color_slugs) {
    // Check if this is a size-related attribute
    foreach ($size_slugs as $size_slug) {
        if (strpos($slug, $size_slug) !== false || levenshtein($slug, $size_slug) <= 2) {
            return 'size';
        }
    }
    
    // Check if this is a color-related attribute
    foreach ($color_slugs as $color_slug) {
        if (strpos($slug, $color_slug) !== false || levenshtein($slug, $color_slug) <= 2) {
            return 'color';
        }
    }
    
    // Remove common suffixes that don't add value
    $slug = preg_replace('/-name$/', '', $slug);
    
    return $slug;
}

/**
 * Helper: Returns the text for an option element (swatch, dropdown, etc.).
 */
private function getOptionText($option) {
    $text = '';
    $img = $option->find('img.imgSwatch', 0);
    if ($img) {
        $text = $img->getAttribute('alt');
    }
    if (!$text) {
        $textDiv = $option->find('div.twisterTextDiv p', 0);
        if ($textDiv) {
            $text = $textDiv->plaintext;
        }
    }
    if (!$text && $option->tag === 'option') {
        $text = $option->plaintext;
    }
    if ($text) {
        $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        $text = trim($text);
    }
    return $text;
}

/**
 * Helper: Returns a selected option for a variation.
 * Tries to find a list item with class "swatchSelect" or a selected dropdown option.
 */
private function getSelectedOption($variation) {
    $selectedOption = $variation->find('li.swatchSelect', 0) ?? $variation->find('option[selected]', 0);
    return $selectedOption ? $this->getOptionText($selectedOption) : '';
}

/**
 * Helper: Cleans an array of options.
 */
private function cleanOptions($options) {
    return array_values(array_filter($options, function($option) {
        return $option !== 'Select' && !empty(trim($option));
    }));
}

/**
 * Extracts a numeric price from a string.
 */
private function extractNumericPrice($price_string) {
    preg_match('/[\d,.]+/', $price_string, $matches);
    return isset($matches[0]) ? floatval(str_replace(',', '', $matches[0])) : null;
}
####################################################################################







    public function createVariants($attributes) {
        $variants = [];
        $base_price = $attributes['price'] ?? 0;
        unset($attributes['price']);

        $combinations = $this->generateAttributeCombinations($attributes);

        foreach ($combinations as $combination) {
            $variant = [
                'attributes' => $combination,
                'price' => $base_price,
                'sku' => $this->generateSKU($combination),
                'stock_quantity' => 100, // Default stock, adjust as needed
                'stock_status' => 'instock'
            ];
            $variants[] = $variant;
        }

        return $variants;
    }

    private function generateAttributeCombinations($attributes, $current = [], $keys = null, $i = 0) {
        if (is_null($keys)) {
            $keys = array_keys($attributes);
        }

        if ($i >= count($keys)) {
            return [$current];
        }

        $key = $keys[$i];
        $combinations = [];

        foreach ($attributes[$key]['options'] as $option) {
            $new = $current;
            $new[$key] = $option;
            $combinations = array_merge($combinations, $this->generateAttributeCombinations($attributes, $new, $keys, $i + 1));
        }

        return $combinations;
    }

    private function generateSKU($combination) {
        $sku_parts = [];
        foreach ($combination as $attribute => $value) {
            $sku_parts[] = substr(preg_replace('/[^a-zA-Z0-9]/', '', $value), 0, 3);
        }
        return strtoupper(implode('-', $sku_parts));
    }

    public function getProductFirstVariationFromHtml($html, $parentSku, $base_url = '', $all_skus = []) {
        $variation_ids = [];
        $extract_ID = $this->getSkuFromUrl($base_url);

        // Add specific selector for digital storage capacity variants
        $storage_selectors = [
            '#inline-twister-row-digital_storage_capacity li[data-asin]',
            '.inline-twister-swatch[data-asin]',
            '.swatch-list-item-text[data-asin]'
        ];

        foreach ($storage_selectors as $selector) {
            $storage_variants = $html->find($selector);
            foreach ($storage_variants as $variant) {
                $asin = $variant->getAttribute('data-asin');
                if ($asin && strlen($asin) == 10 && in_array($asin, $all_skus) && $asin !== $parentSku) {
                    $variation_ids[] = $asin;
                }
            }
        }

        // Original variation selectors
        $variation_selectors = [
            '#variation_size_name ul li', 
            '#variation_color_name ul li',
            '#twister_feature_div #twisterContainer form div ul li',
            '#twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister .inline-twister-row ul li',
            '#softlinesTwister_feature_div #twisterContainer form div ul li',
            '.shelf-item', '.a-carousel-card', '.twisterSlot',
            'select[name^="dropdown_selected_"] option',
            'input[type="hidden"][name="ASIN"]', 
            'input[type="hidden"][name="asin"]'
        ];

        foreach ($variation_selectors as $selector) {
            $variations = $html->find($selector);
            foreach ($variations as $variation) {
                $attributes_to_check = ['data-defaultasin', 'data-asin', 'data-csa-c-item-id', 'data-asin-id', 'value'];
                foreach ($attributes_to_check as $attr) {
                    $sid = trim($variation->getAttribute($attr));
                    if ($sid && strlen($sid) == 10 && in_array($sid, $all_skus) && $sid !== $parentSku) {
                        $variation_ids[] = $sid;
                        break;
                    }
                }
                // Check data-dp-url attribute
                $dp_url = $variation->getAttribute('data-dp-url');
                if ($dp_url && preg_match('/\/dp\/([A-Z0-9]{10})/', $dp_url, $matches)) {
                    $sid = $matches[1];
                    if (in_array($sid, $all_skus) && $sid !== $parentSku) {
                        $variation_ids[] = $sid;
                    }
                }
            }
        }

        // Process JavaScript data for variations
        $scripts = $html->find('script');
        foreach ($scripts as $script) {
            $scriptContent = $script->innertext;
            
            // Extract ASINs from asinVariationValues
            if (preg_match('/var\s+asinVariationValues\s*=\s*({.*?});/s', $scriptContent, $matches)) {
                $asinVariationValues = json_decode($matches[1], true);
                if ($asinVariationValues) {
                    foreach (array_keys($asinVariationValues) as $asin) {
                        if (in_array($asin, $all_skus) && $asin !== $parentSku) {
                            $variation_ids[] = $asin;
                        }
                    }
                }
            }
            
            // Extract ASINs from dimensionValuesDisplayData
            if (preg_match('/var\s+dimensionValuesDisplayData\s*=\s*({.*?});/s', $scriptContent, $matches)) {
                $dimensionData = json_decode($matches[1], true);
                if ($dimensionData) {
                    foreach ($dimensionData as $dimension) {
                        if (isset($dimension['dimensionValues'])) {
                            foreach ($dimension['dimensionValues'] as $value) {
                                if (isset($value['asin']) && in_array($value['asin'], $all_skus) && $value['asin'] !== $parentSku) {
                                    $variation_ids[] = $value['asin'];
                                }
                            }
                        }
                    }
                }
            }

            // Extract ASINs from other possible JSON structures
            if (preg_match_all('/"asin"\s*:\s*"([A-Z0-9]{10})"/', $scriptContent, $matches)) {
                foreach ($matches[1] as $asin) {
                    if (in_array($asin, $all_skus) && $asin !== $parentSku) {
                        $variation_ids[] = $asin;
                    }
                }
            }
        }

        // Remove duplicates and sort
        $variation_ids = array_unique($variation_ids);
        sort($variation_ids);

        return array_values($variation_ids);
    }

    //wf
    public function getSkuFromUrl($url) {
        // Check if URL is empty or null first
        if (empty($url)) {
            return null;
        }
        
        $path = parse_url($url, PHP_URL_PATH);
        if ($path === null || $path === false) {
            return null; // Return null if URL is malformed
        }
        $segments = explode("/dp/", $path);
        if (count($segments) < 2) {
            return null; // Return null if URL does not contain "/dp/"
        }
        $asinSegment = end($segments);
        $asinParts = explode('/', $asinSegment);
        
        return $asinParts[0] ?? null; // Return the ASIN or null if not found
    }
    public function getParentSkuFromHtml($html) {

        if (!$html || !is_object($html)) {
            error_log("getParentSkuFromHtml: Invalid HTML input");
            return null;
        }
        
        // Try to find parentAsin in multiple possible locations
        $searchElements = ['#centerCol', '#dp-container', 'body'];
        
        foreach ($searchElements as $elementSelector) {
            $element = $html->find($elementSelector, 0);
            if ($element) {
                $regex = '/"parentAsin"\s*:\s*"([A-Z0-9]{10})"/i';
                if (preg_match($regex, $element->outertext, $matches)) {
                    $asin = trim($matches[1]);
                    if (!empty($asin)) {
                        return $asin;
                    }
                }
                
                // If not found with first regex, try an alternative
                $altRegex = "/\'parentAsin\'\s*:\s*\'([A-Z0-9]{10})\'/i";
                if (preg_match($altRegex, $element->outertext, $matches)) {
                    $asin = trim($matches[1]);
                    if (!empty($asin)) {
                        return $asin;
                    }
                }
            }
        }
        
        // If we've searched all elements and haven't found a match, try the entire HTML
        $regex = '/"parentAsin"\s*:\s*"([A-Z0-9]{10})"/i';
        if (preg_match($regex, $html->outertext, $matches)) {
            $asin = trim($matches[1]);
            if (!empty($asin)) {
                return $asin;
            }
        }
        
        error_log("getParentSkuFromHtml: No parentAsin found in HTML"); exit;
        return null;
    }

    private function mapAmazonAttributeToWooCommerce($amazon_attribute) {
        $map = [
            'size' => 'pa_size',
            'colour' => 'pa_colour',
            'color' => 'pa_colour',
            'style' => 'pa_style',
            'material' => 'pa_material',
            'fit_type' => 'pa_fit_type',
            'fit_type_options' => 'pa_fit_type',
        ];

        $amazon_attribute = strtolower($amazon_attribute);
        return isset($map[$amazon_attribute]) ? $map[$amazon_attribute] : 'pa_' . $amazon_attribute;
    }

    private function sanitizeAttributeValue($value) {
        if (is_array($value)) {
            return implode('|', array_map(function($v) {
                return $this->sanitizeAttributeValue($v);
            }, $value));
        }
        
        $value = html_entity_decode($value);
        $value = strtolower(str_replace(' ', '-', $value));
        $value = preg_replace('/[^a-z0-9-]/', '', $value);
        return $value;
    }

    public function getAttributeCount($html) {
        $attributes = $html->find('#twister_feature_div #twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister .inline-twister-row');
        if( $attributes ) {
            return count($attributes);
        }

        $attributes = $html->find('#softlinesTwister_feature_div #twister-plus-inline-twister-container #twister-plus-inline-twister-card #twister-plus-inline-twister .inline-twister-row');
        if( $attributes ) {
            return count($attributes);
        }
        return FALSE;
    }

    public function importMultiLevelVariation($product_id, $product_meta_data) {
        $product = wc_get_product($product_id);

        // The attribute data
        $attributes_data = $product_meta_data['attributes_data'];
        wc_update_product_attributes( $product_id, $attributes_data );

        $variations_default_attributes = [];
        foreach ($product_meta_data['default_attributes'] as $attribute => $value) {
            $taxonomy = 'pa_'.$attribute;
            if( $term = get_term_by( 'name', $value, $taxonomy ) ) {
                $variations_default_attributes[$taxonomy] = $term->slug;
            }
        }

        // Save the variation default attributes
        if( !empty($variations_default_attributes) ) {
            update_post_meta( $product_id, '_default_attributes', $variations_default_attributes );
        }
    }

    public function get_parsed_url($url, $part = 'PHP_URL_FULL') {
        $parse = parse_url($url);
        
        switch ($part) {
            case 'PHP_URL_FULL':
                $formatted = $parse['scheme'] . '://' . $parse['host'];
                if (isset($parse['path'])) {
                    $formatted .= $parse['path'];
                }
                if (isset($parse['query'])) {
                    $formatted .= '?' . $parse['query'];
                }
                break;
            case 'PHP_URL_BASE':
                $formatted = $parse['scheme'] . '://' . $parse['host'];
                break;
            default:
                $formatted = '';
        }
        
        // Check for empty string then return false
        if ($formatted == '') return FALSE;
        return $formatted;
    }

    public function fetchReviewFromHtml($html) {
		$reviewData = [];
		$reviewList = $html->find('#cm_cr-review_list [data-hook="review"]');

		foreach ($reviewList as $review) {
			$extract = explode('ASIN=', $review->find('[data-hook="review-title"]',0)->getAttribute('href'));

            // Get ratings
			$rating = 5;
			$star_classes = ['a-star-1','a-star-2','a-star-3','a-star-4','a-star-5'];
			$review_star_rating = $review->find('i.review-rating',0)->getAttribute('class');
			preg_match('/a-star-\d/', $review_star_rating, $match);
			$reviews = isset($match[0]) ? $match[0] : '';
			if( false !== $key = array_search($reviews, $star_classes) ) { $rating = $key + 1; }
			
			$reviewData[] = [
				"id" => $review->getAttribute('id'),
				"asin" => end($extract),
				"user" => $review->find('.a-profile-name',0)->text(),
				"title" => $review->find('[data-hook="review-title"]',0)->text(),
				"content" => $review->find('[data-hook="review-body"]',0)->text(),
				"rating" => $rating
			];
		}

        //echo '<pre>'; dd( $reviewData ); echo '</pre>'; exit;
		return $reviewData;
	}
    //search without-api
    function fetchSearchPriceFromHtml($html) {
        $amount = $regular_price = $sale_price = 0;
        $currency = '';
        
        $price = $html->find('.a-offscreen');
        $priceWhole = $html->find('.a-price-whole');
        $priceFraction = $html->find('.a-price-fraction');
        $aSizeBaseColorPrice = $html->find('.a-size-base.a-color-price.a-color-price');
        $currencySymbol = $html->find('.a-price-symbol');

        $delPrice = $html->find('.a-text-price .a-offscreen', 0);

        // Find sale price
        if (isset($price[0])) {
            $priceText = $price[0]->innertext;
            $currency = preg_replace('/[0-9.,\s]/', '', $priceText);
            $sale_price = preg_replace('/[^0-9.]/', '', $priceText);
        }

        // Find regular price (usually crossed out)
        if ($delPrice) {
            $delPriceText = $delPrice->innertext;
            $currency = preg_replace('/[0-9.,\s]/', '', $delPriceText);
            $regular_price = preg_replace('/[^0-9.]/', '', $delPriceText);
        }

        // If no sale price found, use the regular price
        if ($sale_price == 0 && $regular_price != 0) {
            $sale_price = $regular_price;
        }

        // If no regular price found, use the sale price
        if ($regular_price == 0 && $sale_price != 0) {
            $regular_price = $sale_price;
        }

        $sale_price = $this->parsePrice($sale_price);
        $regular_price = $this->parsePrice($regular_price);
        
        return [
            'sale_price' => $sale_price,
            'regular_price' => $regular_price,
            'currency' => $currency,
            'formatted_sale_price' => $currency . $sale_price,
            'formatted_regular_price' => $currency . $regular_price,
        ];
    }

    function get_search_results($data) {
        $base_url = no_api_active_country_url();
        $ams_keyword = $data['k'] ?? '';
        $ams_amazon_cat = $data['i'] ?? '';
        $ams_amazon_page = $data['page'] ?? 1;

        $results_limit = get_option('ams_results_limit', '50');
        $results_limit = $results_limit ? intval($results_limit) : 50;
        

        
        // Performance optimization: Cache search results
        $cache_key = 'ams_search_' . md5($ams_keyword . $ams_amazon_cat . $ams_amazon_page . $results_limit);
        $cached_results = get_transient($cache_key);
        if ($cached_results !== false) {
            return $cached_results;
        }
        $start_time = microtime(true);
        
        $lists = array();
        $load_more = false;
        $max_pages = 3; // Reduced from 10 to 3 for better performance
        $total_results = 0;
        $pages_processed = 0;

        for ($current_page = $ams_amazon_page; $current_page <= $max_pages; $current_page++) {
            // Build search URL
            $url = $base_url . '/s?' . http_build_query([
                'k' => $ams_keyword,
                'i' => $ams_amazon_cat,
                'page' => $current_page,
                'ref' => 'sr_pg_' . $current_page
            ]);
            

            $page_start = microtime(true);
            
            // Try direct request first, then fallback to scraping service
            $response_body = $this->getContentUsingCurl($url, null, true); // Added fast mode

                        if (empty($response_body)) {
                $response_body = executeScrapingService($url);
            if (empty($response_body)) {
                break;
            }
            }
            
            if (!class_exists('simple_html_dom')) {
                require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
            }
            $html = new \simple_html_dom();
            $html->load($response_body);

            $search_result = $html->find('.s-result-list', 0);
            if (!$search_result) {
                // If no results found with direct request, try scraping service for better HTML
                if (strpos($response_body, 'executeScrapingService') === false) { // Avoid double scraping
                    $scraping_response = executeScrapingService($url);
                    if ($scraping_response) {
                        $html->load($scraping_response);
                $search_result = $html->find('.s-result-list', 0);
                    }
                }
                
                if (!$search_result) {
                    break;
                }
            }
            
            // Parse search results
            $search_items = $search_result->find('.s-result-item');
            
            $page_items = count($search_items);
            $total_results += $page_items;
            $pages_processed++;
            

            

            
            if ($page_items === 0) {


            } else {

            }

            foreach ($search_items as $search_item) {
                $data_asin = $search_item->getAttribute('data-asin') ?? '';
                if ($data_asin) {
                    // Extract title
                    $title_element = $search_item->find('h2 a span, h2 span', 0);
                    
                    if (!$title_element) {
                        continue;
                    }
                    
                    // Extract product link
                    $anchor = $search_item->find('h2 a, .a-link-normal', 0);
                    
                    $detailPageURL = $anchor ? $base_url . $anchor->getAttribute('href') : '';
                    if (strpos($detailPageURL, "/dp/") === false) {
                        continue;
                    }

                    // Optimized price extraction - reduced logging
                    $price = $this->fetchPriceFromHtml($search_item, true); // Added silent mode
                    $regular_price = isset($price['regular_price']) ? $price['regular_price'] : 0;
                    $sale_price = isset($price['sale_price']) ? $price['sale_price'] : 0;

                    // Optimized currency extraction
                    $currency = $this->fetchCurrencyFromHtml($search_item, true); // Added silent mode
                    
                    // Enhanced review extraction - Amazon search often doesn't show ratings
                    $rating = 0;
                    $review_text = '';
                    $review_count_text = '';
                    
                    // Try to find rating elements in search results
                    $rating_found = false;
                    
                    // Look for star rating elements
                    $star_elements = $search_item->find('.a-icon-star-small, .a-icon-star, [class*="star"]');
                    foreach ($star_elements as $star_element) {
                        // Check various attributes for rating text
                        $possible_texts = [
                            $star_element->getAttribute('title'),
                            $star_element->getAttribute('aria-label'),
                            $star_element->plaintext
                        ];
                        
                        // Also check child elements
                        $child_alt = $star_element->find('.a-icon-alt', 0);
                        if ($child_alt) {
                            $possible_texts[] = $child_alt->getAttribute('title');
                            $possible_texts[] = $child_alt->getAttribute('aria-label');
                            $possible_texts[] = $child_alt->innertext;
                        }
                        
                        foreach ($possible_texts as $text) {
                            if (!empty($text)) {
                                $extracted_rating = $this->extract_rating_from_text($text);
                                if ($extracted_rating > 0) {
                                    $rating = $extracted_rating;
                                    $review_text = $text;
                                    $rating_found = true;
                                    break 2;
                                }
                            }
                        }
                    }
                    
                    // If no rating found in search, generate a placeholder rating based on review count
                    if (!$rating_found) {
                        // Look for review count first
                        $review_elements = $search_item->find('a[href*="customerReviews"], span[aria-label*="customer"], .a-size-base');
                        foreach ($review_elements as $review_element) {
                            $text = trim($review_element->plaintext);
                            if (!empty($text) && (preg_match('/[\d,]+/', $text) || strpos($text, '(') !== false)) {
                                $review_count_text = $text;
                                
                                // Generate a realistic rating (4.0-4.5) if we have reviews but no rating
                                $review_number = (int) preg_replace('/[^\d]/', '', $text);
                                if ($review_number > 0) {
                                    $rating = 4.0 + (($review_number % 5) * 0.1); // Generates 4.0-4.4
                                    $review_text = $rating . ' out of 5 stars (estimated)';
                                }
                                break;
                            }
                        }
                    }
                    
                    // Final fallback: if product has reviews but no rating, assign default 4.2
                    if ($rating == 0 && !empty($review_count_text)) {
                        $rating = 4.2;
                        $review_text = '4.2 out of 5 stars (estimated)';
                    }
                    
                    // Enhanced image extraction
                    $image_element = $search_item->find('img', 0);
                    $image_url = '';
                    if ($image_element) {
                        $image_url = $image_element->getAttribute('src') ?: $image_element->getAttribute('data-src') ?: '';
                    }
                    
                    // Extract additional product information
                    
                    // 1. Discount percentage calculation
                    $discount_percentage = 0;
                    if ($regular_price > 0 && $sale_price > 0 && $sale_price < $regular_price) {
                        $discount_percentage = round((($regular_price - $sale_price) / $regular_price) * 100);
                    }
                    
                    // 2. Prime shipping status
                    $is_prime = false;
                    $prime_elements = $search_item->find('[aria-label*="Prime"], .a-icon-prime, [class*="prime"], [data-cy*="prime"]');
                    if (!empty($prime_elements)) {
                        $is_prime = true;
                    }
                    
                    // 3. Brand extraction (Comprehensive)
                    $brand = '';
                    
                    // Try multiple strategies to find brand
                    // Strategy 1: Look for brand in specific Amazon structures
                    $brand_selectors = [
                        'span[data-component-type="s-product-image"] + div span:first-child',
                        '.a-size-base-plus:first-child',
                        '.a-size-base:first-child',
                        'h2 + div span:first-child',
                        '.s-size-mini:first-child',
                        '.a-color-base:first-child',
                        '.a-text-bold:first-child'
                    ];
                    
                    foreach ($brand_selectors as $selector) {
                        $brand_elements = $search_item->find($selector);
                        foreach ($brand_elements as $brand_element) {
                            $brand_text = trim($brand_element->plaintext);
                            if ($this->isValidBrand($brand_text)) {
                                $brand = $brand_text;
                                break 2;
                            }
                        }
                    }
                    
                    // Strategy 2: Extract from product title if no brand found
                    if (empty($brand)) {
                        $title_element = $search_item->find('h2 a span, .a-link-normal span', 0);
                        if ($title_element) {
                            $title = trim($title_element->plaintext);
                            $brand = $this->extractBrandFromTitle($title);
                        }
                    }
                    
                    // 4. Bestseller/Choice badges
                    $badges = [];
                    $badge_elements = $search_item->find('[aria-label*="Best Seller"], [aria-label*="Amazon\'s Choice"], .a-badge, [class*="badge"]');
                    foreach ($badge_elements as $badge_element) {
                        $badge_text = trim($badge_element->plaintext ?: $badge_element->getAttribute('aria-label'));
                        if (!empty($badge_text) && strlen($badge_text) < 30) {
                            $badges[] = $badge_text;
                        }
                    }
                    
                    // 5. Delivery information
                    $delivery_info = '';
                    $delivery_elements = $search_item->find('[aria-label*="delivery"], [aria-label*="shipping"], .a-color-base:contains("delivery")');
                    foreach ($delivery_elements as $delivery_element) {
                        $delivery_text = trim($delivery_element->plaintext ?: $delivery_element->getAttribute('aria-label'));
                        if (!empty($delivery_text) && (stripos($delivery_text, 'delivery') !== false || stripos($delivery_text, 'shipping') !== false)) {
                            $delivery_info = $delivery_text;
                            break;
                        }
                    }
                    
                    // 6. Stock status
                    $stock_status = 'in_stock';
                    $stock_elements = $search_item->find('[aria-label*="stock"], [aria-label*="availability"], .a-color-price, .a-color-secondary, .a-color-state, .a-text-bold');
                    foreach ($stock_elements as $stock_element) {
                        $stock_text = strtolower(trim($stock_element->plaintext));
                        // Check for out of stock indicators
                        if (strpos($stock_text, 'out of stock') !== false || 
                            strpos($stock_text, 'unavailable') !== false ||
                            strpos($stock_text, 'not available') !== false ||
                            strpos($stock_text, 'temporarily unavailable') !== false ||
                            strpos($stock_text, 'currently unavailable') !== false) {
                            $stock_status = 'out_of_stock';
                            break;
                        } 
                        // Check for limited stock indicators
                        elseif (strpos($stock_text, 'limited') !== false || 
                                strpos($stock_text, 'only') !== false ||
                                strpos($stock_text, 'few left') !== false ||
                                strpos($stock_text, 'low stock') !== false ||
                                (strpos($stock_text, 'left in stock') !== false && preg_match('/\d+\s*left/i', $stock_text))) {
                            $stock_status = 'limited_stock';
                            break;
                        }
                    }
                    
                    // 7. Product category (from breadcrumb or title)
                    $category = '';
                    $category_elements = $search_item->find('.a-color-secondary, .a-size-base-plus');
                    foreach ($category_elements as $cat_element) {
                        $cat_text = trim($cat_element->plaintext);
                        if (!empty($cat_text) && strlen($cat_text) < 40 && !preg_match('/\$|€|£|¥/', $cat_text)) {
                            $category = $cat_text;
                            break;
                        }
                    }
                    
                    // 8. Simple variant detection
                    $variants_info = $this->detectSimpleVariants($search_item);
                    

                    
                    $product_data = [
                        'ASIN'              => $data_asin,
                        'Title'             => trim($title_element->plaintext),
                        'Price'             => $currency . ($sale_price ?: $regular_price ?: 0),
                        'SavingBasis'       => $currency . ($regular_price ?: 0),
                        'ImageUrl'          => $image_url,
                        'DetailPageURL'     => $detailPageURL,
                        'Rating'            => $rating,
                        'ReviewText'        => $review_text,
                        'ReviewCount'       => $review_count_text,
                        'RegularPrice'      => $regular_price,
                        'SalePrice'         => $sale_price,
                        // New enhanced data
                        'DiscountPercent'   => $discount_percentage,
                        'IsPrime'           => $is_prime,
                        'Brand'             => $brand,
                        'Badges'            => $badges,
                        'DeliveryInfo'      => $delivery_info,
                        'StockStatus'       => $stock_status,
                        'Category'          => $category,
                        'VariantsInfo'      => $variants_info,
                    ];
                    
                    $lists[] = $product_data;
                }
                
                if (count($lists) >= $results_limit) {
                    $load_more = true;
                    break 2;
                }
            }
            
            $page_time = round(microtime(true) - $page_start, 2);

        }

        $total_time = round(microtime(true) - $start_time, 2);


        $result = [
            'data' => array_slice($lists, 0, $results_limit),
            'load_more' => $load_more,
            'total_results' => $total_results,
            'pages_processed' => $pages_processed,
            'search_time' => $total_time
        ];
        
        // Cache results for 5 minutes
        set_transient($cache_key, $result, 300);
        
        return $result;
    }
    //search without-api


    function extract_rating_from_text($text) {
        // Clean up the text
        $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        // Try multiple rating patterns
        $patterns = [
            '/(\d+(?:\.\d+)?)\s*out of 5 stars?/i',
            '/(\d+(?:\.\d+)?)\s*\/\s*5\s*stars?/i',
            '/(\d+(?:\.\d+)?)\s*stars?\s*out of 5/i',
            '/rating:\s*(\d+(?:\.\d+)?)/i',
            '/(\d+(?:\.\d+)?)\s*star/i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $text, $matches)) {
                $rating = floatval($matches[1]);
                // Validate rating is between 0 and 5
                if ($rating >= 0 && $rating <= 5) {
                    return $rating;
                }
            }
        }
        
        return 0;
    }
    public function getContentUsingCurl($url, $user_agent = false, $fast_mode = false) {
        $max_attempts = $fast_mode ? 1 : 3;
        $attempt = 0;
        
        while ($attempt < $max_attempts) {
            $attempt++;
            
            // Reduced sleep time in fast mode
            if (!$fast_mode) {
                sleep(rand(1, 2)); // Reduced from 2-3 to 1-2
            }
            
            $rand_user_agent = $this->user_agent();
            $timeout = $fast_mode ? 30 : 120; // Faster timeout for search
            
            $options = array(
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_MAXREDIRS => 5, // Reduced redirects
                CURLOPT_CONNECTTIMEOUT => $timeout,
                CURLOPT_TIMEOUT => $timeout,
                CURLOPT_USERAGENT => $user_agent ? $user_agent : $rand_user_agent,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_ENCODING => ""
            );
            
            $ch = curl_init($url);
            curl_setopt_array($ch, $options);
            $contents = curl_exec($ch);
            $response_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if (curl_errno($ch)) {
                $contents = '';
            } else {
                curl_close($ch);
            }
            
            $use_scraping_service = 
                $response_code != 200 || 
                !is_string($contents) || 
                strlen($contents) == 0 || 
                stripos($contents, 'captcha') !== false ||
                stripos($contents, 'Page Not Found') !== false ||
                stripos($contents, 'Dogs of Amazon') !== false ||
                !preg_match('/<title>(.*?)<\/title>/i', $contents) ||
                empty(trim(preg_replace('/<title>(.*?)<\/title>/i', '$1', $contents)));
            
            if (!$use_scraping_service) {
                return $contents;
            }
            // Use scraping service if any of the above conditions are true
            // logImportVerification('Using scraping service', null);
            $scraping_response = executeScrapingService($url);
            //echo '<pre>'; dd( $scraping_response ); echo '</pre>';
            
            if ($scraping_response === false) {
                // logImportVerification('Scraping service failed', null);
                if ($attempt == $max_attempts) {
                    return false;
                }
                continue; // Try again
            }
            
            $contents = is_array($scraping_response) ? $scraping_response['data'] : $scraping_response;
            
            if (is_string($contents) && strlen($contents)) {
                // logImportVerification('Scraping service used successfully', null);
                return $contents;
            }
            
            // logImportVerification('Failed to retrieve content even with scraping service', null);
            
            if ($attempt == $max_attempts) {
                return false;
            }
            // If we reach here, we'll try again
        }
        
        return false;
    }

    public function getRandomUserAgent() {
        $userAgents=array(
        "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/68.0.3440.106 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.113 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/78.0.3904.108 Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/64.0.3282.167 Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.67 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.181 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.3; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/77.0.3865.90 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.3; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/67.0.3396.99 Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/66.0.3359.117 Safari/537.36",
        "Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US) AppleWebKit/533.4 (KHTML, like Gecko) Chrome/5.0.375.99 Safari/533.4",
        "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/73.0.3683.75 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/52.0.2743.116 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/51.0.2704.103 Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/46.0.2490.80 Safari/537.36",
        "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_14_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/74.0.3729.169 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.111 Safari/537.36",
        "Mozilla/5.0 (Linux; Android 9; SM-G960F Build/PPR1.180610.011; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/74.0.3729.157 Mobile Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/56.0.2924.87 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/48.0.2564.109 Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/55.0.2883.87 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/67.0.3396.87 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/61.0.3163.100 Safari/537.36",
        "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_6_8) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/49.0.2623.112 Safari/537.36",
        "BrightSign/8.0.69 (XT1143)Mozilla/5.0 (X11; Linux armv7l) AppleWebKit/537.36 (KHTML, like Gecko) QtWebEngine/5.11.2 Chrome/65.0.3325.230 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.3; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/68.0.3440.106 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.102 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.111 Safari/537.36 MVisionPlayer/1.0.0.0",
        "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/63.0.3239.84 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.0) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/49.0.2623.112 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/36.0.1985.125 Safari/537.36",
        "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/61.0.3163.100 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/51.0.2704.106 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.2; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) QtWebEngine/5.9.4 Chrome/56.0.2924.122 Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.106 Safari/537.36",
        "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.106 Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/64.0.3282.140 Safari/537.36",
        "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/78.0.3904.70 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.3; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.106 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/66.0.3359.117 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.110 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.77 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/63.0.3239.132 Safari/537.36",
        "Mozilla/5.0 (Linux; Android 6.0.1; SM-G532G Build/MMB29T) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/63.0.3239.83 Mobile Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.3; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.181 Safari/537.36",
        "Mozilla/5.0 (Linux; Android 9; SM-G950F Build/PPR1.180610.011; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/74.0.3729.157 Mobile Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.67 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2272.101 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/48.0.2564.116 Safari/537.36",
        "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/32.0.1700.107 Safari/537.36",
        "Mozilla/5.0 (en-us) AppleWebKit/534.14 (KHTML, like Gecko; Google Wireless Transcoder) Chrome/9.0.597 Safari/534.14"
        );
        
        $random = rand(0,count($userAgents)-1);        
        return $userAgents[$random];
    }

    /**
     * Simple variant detection from search results
     * Detects common variant indicators in Amazon search results
     */
    private function detectSimpleVariants($search_item) {
        $variants_info = [
            'has_variants' => false,
            'variant_count' => 0,
            'variant_text' => '',
            'options' => [],
            'price_range' => null,
            'min_price' => null,
            'max_price' => null
        ];
        
        // Look for common variant indicators in search results
        $variant_indicators = [
            '.a-size-base.a-color-secondary' => 'options',
            '.a-row .a-size-base' => 'options',
            '[data-cy="color-variant"]' => 'colors',
            '[data-cy="size-variant"]' => 'sizes',
            '.s-color-swatch-outer-circle' => 'colors',
            '.rush-component .a-size-mini' => 'options'
        ];
        
        foreach ($variant_indicators as $selector => $type) {
            $elements = $search_item->find($selector);
            foreach ($elements as $element) {
                $text = trim($element->plaintext);
                
                // Check for variant count patterns
                if (preg_match('/(\d+)\s*(colors?|sizes?|options?|styles?)/i', $text, $matches)) {
                    $variants_info['has_variants'] = true;
                    $variants_info['variant_count'] = (int)$matches[1];
                    $variants_info['variant_text'] = $matches[1] . ' ' . strtolower($matches[2]);
                    break 2;
                }
                
                // Check for "Available in X colors" patterns
                if (preg_match('/available\s+in\s+(\d+)/i', $text, $matches)) {
                    $variants_info['has_variants'] = true;
                    $variants_info['variant_count'] = (int)$matches[1];
                    $variants_info['variant_text'] = $matches[1] . ' options';
                    break 2;
                }
            }
        }
        
        // Look for color swatches (visual indicators)
        $color_swatches = $search_item->find('.s-color-swatch-outer-circle, .colorsprite, .a-button-thumbnail');
        if (count($color_swatches) > 1) {
            $variants_info['has_variants'] = true;
            $variants_info['variant_count'] = count($color_swatches);
            $variants_info['variant_text'] = count($color_swatches) . ' colors';
        }
        
        // Look for size indicators
        $size_elements = $search_item->find('[data-size], .size-option, .a-size-option');
        if (count($size_elements) > 1) {
            $variants_info['has_variants'] = true;
            if (!$variants_info['variant_count']) {
                $variants_info['variant_count'] = count($size_elements);
                $variants_info['variant_text'] = count($size_elements) . ' sizes';
            }
        }
        
        // Look for price range indicators if variants detected
        if ($variants_info['has_variants']) {
            $this->detectVariantPriceRange($search_item, $variants_info);
        }
        
        return $variants_info;
    }

    /**
     * Detect price range for variant products
     */
    private function detectVariantPriceRange($search_item, &$variants_info) {
        // Simple price range detection - Amazon search results rarely show ranges
        $text_content = $search_item->plaintext;
        
        // Look for explicit "from €X" patterns
        if (preg_match('/(?:from|starting\s+at|ab)\s+([\$€£¥])(\d+(?:[,\.]\d{2})?)/i', $text_content, $matches)) {
            $currency = $matches[1];
            $min_price = floatval(str_replace(',', '.', $matches[2]));
            $variants_info['price_range'] = 'from ' . $currency . number_format($min_price, 2);
            return;
        }
        
        // Look for explicit price ranges like "€25.99 - €45.99"
        if (preg_match('/([\$€£¥])(\d+(?:[,\.]\d{2})?)\s*[-–—to]\s*([\$€£¥])?(\d+(?:[,\.]\d{2})?)/i', $text_content, $matches)) {
            $currency = $matches[1];
            $min_price = floatval(str_replace(',', '.', $matches[2]));
            $max_price = floatval(str_replace(',', '.', $matches[4]));
            $variants_info['price_range'] = $currency . number_format($min_price, 2) . ' - ' . $currency . number_format($max_price, 2);
            return;
        }
        
        // Most Amazon search results don't show price ranges for variants
        // The display will handle this by showing "from €X" with explanation
    }
    
    /**
     * Extract price information from bestseller DOM elements
     */
    private function extractBestsellerPrice($node) {
        $currency = '';
        
        // Currency mapping function
        $map_country_currency = function($country) {
            $map = [
                'com' => 'USD',
                'co.uk' => 'GBP',
                'uk' => 'GBP',
                'nl' => 'EUR',
                'de' => 'EUR',
                'fr' => 'EUR',
                'it' => 'EUR',
                'es' => 'EUR'
            ];
            return isset($map[$country]) ? $map[$country] : 'USD';
        };
        
        // Price parsing function
        $parse_price_text = function($text) use (&$currency, $map_country_currency) {
            $text = trim($text);
            if ($currency === '') {
                if (strpos($text, '€') !== false) $currency = 'EUR';
                elseif (strpos($text, '£') !== false) $currency = 'GBP';
                elseif (strpos($text, '$') !== false) $currency = 'USD';
                else $currency = $map_country_currency(get_option('ams_amazon_country', 'com'));
            }
            
            // Extract numeric value with support for different decimal separators
            if (!preg_match('/(\d{1,3}(?:[\.,]\d{3})*(?:[\.,]\d{2})|\d+(?:[\.,]\d{2})?)/u', $text, $m)) return 0.0;
            
            $raw = $m[1];
            $last_dot = strrpos($raw, '.');
            $last_comma = strrpos($raw, ',');
            
            // Determine decimal separator
            $decimal_sep = ($last_comma !== false && $last_comma > $last_dot) ? ',' : '.';
            
            if ($decimal_sep === ',') {
                $raw = str_replace('.', '', $raw); // Remove thousands separator
                $raw = str_replace(',', '.', $raw); // Convert decimal separator
            } else {
                $raw = str_replace(',', '', $raw); // Remove thousands separator
            }
            
            return floatval($raw);
        };
        
        // Initialize price variables
        $regular_price_text = '';
        $sale_price_text = '';
        
        // Try to find regular price (crossed out price)
        $el = $node->find('.a-text-price .a-offscreen', 0);
        if ($el && trim($el->plaintext)) {
            $regular_price_text = trim($el->plaintext);
        }
        
        // Try to find sale price (current price)
        $el = $node->find('.a-price .a-offscreen', 0);
        if ($el && trim($el->plaintext)) {
            $sale_price_text = trim($el->plaintext);
        }
        
        // Fallback: try any .a-offscreen element
        if ($sale_price_text === '') {
            $el = $node->find('.a-offscreen', 0);
            if ($el && trim($el->plaintext)) {
                $sale_price_text = trim($el->plaintext);
            }
        }
        
        // Parse prices
        $regular_price = 0.0;
        $sale_price = 0.0;
        
        if ($regular_price_text !== '') {
            $regular_price = $parse_price_text($regular_price_text);
        }
        
        if ($sale_price_text !== '') {
            $sale_price = $parse_price_text($sale_price_text);
        }
        
        // If we only have one price, treat it as regular price
        if ($regular_price === 0.0 && $sale_price > 0.0) {
            $regular_price = $sale_price;
            $sale_price = 0.0;
        }
        
        // Set default currency if not detected
        if ($currency === '') {
            $currency = $map_country_currency(get_option('ams_amazon_country', 'com'));
        }
        
        return [
            'regular_price' => $regular_price,
            'sale_price' => $sale_price,
            'currency' => $currency
        ];
    }
    
    /**
     * Smart fallback strategy for "All Categories" bestsellers
     * Tries multiple popular categories to ensure consistent results
     */
    private function getBestsellersWithFallback($base_url, $page, $results_limit, $imported_asins) {
        // Define popular fallback categories by country
        $amazon_country = get_option('ams_amazon_country', 'com');
        $fallback_categories = [
            'com' => ['Electronics', 'HomeAndKitchen', 'Fashion', 'Books', 'Beauty'],
            'nl' => ['Electronics', 'HomeAndKitchen', 'Fashion', 'Books', 'Beauty'],
            'de' => ['Electronics', 'HomeAndKitchen', 'Fashion', 'Books', 'Beauty'],
            'fr' => ['Electronics', 'HomeAndKitchen', 'Fashion', 'Books', 'Beauty'],
            'it' => ['Electronics', 'HomeAndKitchen', 'Fashion', 'Books', 'Beauty'],
            'es' => ['Electronics', 'HomeAndKitchen', 'Fashion', 'Books', 'Beauty'],
            'uk' => ['Electronics', 'HomeAndKitchen', 'Fashion', 'Books', 'Beauty']
        ];
        
        $categories_to_try = $fallback_categories[$amazon_country] ?? $fallback_categories['com'];
        $all_results = [];
        $target_per_category = max(4, intval($results_limit / count($categories_to_try)));
        
        // First, try the general bestsellers page
        $general_results = $this->fetchBestsellersFromUrl($base_url . '/gp/bestsellers/', $page, $imported_asins);
        if (!empty($general_results) && count($general_results) >= ($results_limit * 0.7)) {
            // If general page gives good results (70% of target), use it
            return array_slice($general_results, 0, $results_limit);
        }
        
        // If general page fails or gives poor results, try popular categories
        foreach ($categories_to_try as $fallback_category) {
            if (count($all_results) >= $results_limit) {
                break;
            }
            
            // Use the existing category-specific logic
            $category_data = [
                'category' => $fallback_category,
                'page' => 1, // Always use page 1 for fallback categories
                'results_limit' => $target_per_category
            ];
            
            $category_results = $this->fetchCategoryBestsellers($category_data, $base_url, $imported_asins);
            if (!empty($category_results)) {
                $all_results = array_merge($all_results, array_slice($category_results, 0, $target_per_category));
            }
        }
        
        // Shuffle results to mix categories and return requested amount
        if (!empty($all_results)) {
            shuffle($all_results);
            $final_results = array_slice($all_results, 0, $results_limit);
            
            // Cache the results for 5 minutes
            $cache_key = 'ams_bestsellers_' . md5('' . $page . $results_limit);
            set_transient($cache_key, $final_results, 300);
            
            return $final_results;
        }
        
        return ['error' => 'No bestsellers found. Please try selecting a specific category or check your proxy service configuration.'];
    }
    
    /**
     * Fetch bestsellers from a specific URL with retry logic
     */
    private function fetchBestsellersFromUrl($url, $page, $imported_asins, $max_retries = 2) {
        // Add pagination if needed
        if ($page > 1) {
            $url .= (strpos($url, '?') !== false ? '&' : '?') . 'pg=' . $page;
        }
        
        $attempts = 0;
        $results = [];
        
        while ($attempts < $max_retries && empty($results)) {
            $attempts++;
            
            // Try direct request first, then fallback to scraping service
            $response_body = $this->getContentUsingCurl($url, null, true);
            if (empty($response_body)) {
                $response_body = executeScrapingService($url);
                if (empty($response_body)) {
                    // If both methods fail, wait a bit before retry (except on last attempt)
                    if ($attempts < $max_retries) {
                        sleep(1);
                    }
                    continue;
                }
            }
            
            $results = $this->parseBestsellersHtml($response_body, $imported_asins);
            
            // If we got empty results but valid HTML, try once more with different approach
            if (empty($results) && !empty($response_body) && $attempts < $max_retries) {
                sleep(1);
            }
        }
        
        return $results;
    }
    
    /**
     * Fetch bestsellers for a specific category (reusable method)
     */
    private function fetchCategoryBestsellers($data, $base_url, $imported_asins) {
        $category = $data['category'];
        $page = $data['page'] ?? 1;
        
        // Use existing category mapping logic
        $amazon_country = get_option('ams_amazon_country', 'com');
        $bestsellers_url_maps = [
            'com' => [
                'Electronics' => 'electronics',
                'Computers' => 'pc',
                'Books' => 'books',
                'HomeAndKitchen' => 'home-garden',
                'Fashion' => 'fashion',
                'Beauty' => 'beauty',
                'SportsAndOutdoors' => 'sporting-goods',
                'ToysAndGames' => 'toys-and-games',
                'Baby' => 'baby-products',
                'Automotive' => 'automotive'
            ],
            'nl' => [
                'Electronics' => 'electronics',
                'Computers' => 'computers',
                'Books' => 'books',
                'HomeAndKitchen' => 'kitchen',
                'Fashion' => 'fashion',
                'Beauty' => 'beauty',
                'SportsAndOutdoors' => 'sports',
                'ToysAndGames' => 'toys-games',
                'Baby' => 'baby',
                'Automotive' => 'automotive'
            ]
        ];
        
        $country_map = $bestsellers_url_maps[$amazon_country] ?? $bestsellers_url_maps['com'];
        $url_segment = isset($country_map[$category]) ? $country_map[$category] : strtolower($category);
        $url = $base_url . '/gp/bestsellers/' . $url_segment . '/';
        
        return $this->fetchBestsellersFromUrl($url, $page, $imported_asins);
    }
    
    /**
     * Parse bestsellers HTML and extract product data
     */
    private function parseBestsellersHtml($response_body, $imported_asins) {
        if (!class_exists('simple_html_dom')) {
            require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
        }
        
        $html = new \simple_html_dom();
        $html->load($response_body);
        
        if (!$html) {
            return [];
        }
        
        // Find bestsellers using multiple selectors
        $bestseller_selectors = [
            '#zg-ordered-list li',
            '.zg-item-immersion',
            '.zg-item',
            '[data-asin]',
            '.a-cardui',
            'li[id*="item"]'
        ];
        
        $search_items = [];
        foreach ($bestseller_selectors as $selector) {
            $search_items = $html->find($selector);
            if (!empty($search_items)) {
                break;
            }
        }
        
        if (empty($search_items)) {
            return [];
        }
        
        $results = [];
        foreach ($search_items as $search_item) {
            $data_asin = $search_item->getAttribute('data-asin') ?? '';
            
            if (!$data_asin) {
                continue;
            }
            
            // Extract product data using existing logic
            $product_data = $this->extractBestsellerProductData($search_item, $imported_asins);
            if ($product_data) {
                $results[] = $product_data;
            }
        }
        
        return $results;
    }
    
    /**
     * Extract individual product data from HTML element
     */
    private function extractBestsellerProductData($search_item, $imported_asins) {
        $data_asin = $search_item->getAttribute('data-asin') ?? '';
        if (!$data_asin) {
            return null;
        }
        
        // Extract title
        $title_selectors = [
            'a[href*="/dp/"] span',
            '.a-link-normal span',
            '.p13n-sc-truncated',
            '.a-size-base-plus',
            'h2 a span',
            '.zg-item .a-link-normal'
        ];
        
        $title_element = null;
        foreach ($title_selectors as $selector) {
            $title_element = $search_item->find($selector, 0);
            if ($title_element && trim($title_element->plaintext)) {
                break;
            }
        }
        
        if (!$title_element) {
            return null;
        }
        
        // Extract product link
        $link_selectors = [
            'a[href*="/dp/"]',
            '.a-link-normal[href*="/dp/"]'
        ];
        
        $anchor = null;
        foreach ($link_selectors as $selector) {
            $anchor = $search_item->find($selector, 0);
            if ($anchor && $anchor->getAttribute('href') && strpos($anchor->getAttribute('href'), '/dp/') !== false) {
                break;
            }
        }
        
        if (!$anchor) {
            return null;
        }
        
        $base_url = no_api_active_country_url();
        $detailPageURL = $base_url . $anchor->getAttribute('href');
        
        // Extract price using the same method as working products-search-without-api
        $price_data = $this->fetchPriceFromHtml($search_item);
        
        // Extract image
        $image_element = $search_item->find('img', 0);
        $image_url = $image_element ? $image_element->getAttribute('src') : '';
        
        // Extract rating
        $rating_element = $search_item->find('.a-icon-alt', 0);
        $rating = 0;
        if ($rating_element) {
            $rating_text = $rating_element->getAttribute('alt') ?? '';
            preg_match('/(\d+\.?\d*)\s*out\s*of\s*5/', $rating_text, $matches);
            $rating = isset($matches[1]) ? floatval($matches[1]) : 0;
        }
        
        return [
            'ASIN' => $data_asin,
            'Title' => trim($title_element->plaintext),
            'DetailPageURL' => $detailPageURL,
            'RegularPrice' => isset($price_data['regular_price']) ? $price_data['regular_price'] : '',
            'SalePrice' => isset($price_data['sale_price']) ? $price_data['sale_price'] : '',
            'ImageURL' => $image_url,
            'Rating' => $rating,
            'ReviewCount' => 0, // Could be extracted if needed
            'IsImported' => in_array($data_asin, $imported_asins)
        ];
    }

    /**
     * Get bestsellers results from Amazon
     */
    function get_bestsellers_results($data) {
        $base_url = no_api_active_country_url();
        $category = $data['category'] ?? '';
        $page = $data['page'] ?? 1;
        $results_limit = $data['results_limit'] ?? get_option('ams_bestsellers_limit', '20'); // Use passed limit or saved setting
        $results_limit = intval($results_limit);
        
        // Get all imported ASINs once at the beginning for efficiency
        $ams_all_asin = ams_get_all_products_info_with_parent();
        $imported_asins = (!empty($ams_all_asin) && !empty($ams_all_asin['asin'])) ? $ams_all_asin['asin'] : array();
        
        // Simple cache with 5-minute expiration
        $cache_key = 'ams_bestsellers_' . md5($category . $page . $results_limit);
        $cached_results = get_transient($cache_key);
        if ($cached_results !== false) {
            // Update import status for cached results
            foreach ($cached_results as &$result) {
                $result['IsImported'] = in_array($result['ASIN'], $imported_asins);
            }
            return $cached_results;
        }
        
        // If no category selected (All Categories), use smart fallback strategy
        if (empty($category)) {
            return $this->getBestsellersWithFallback($base_url, $page, $results_limit, $imported_asins);
        }
        
        $start_time = microtime(true);
        
        $lists = array();
        $max_pages = 3;
        $total_results = 0;
        
        for ($current_page = $page; $current_page <= $max_pages; $current_page++) {
            // Build Amazon bestsellers URL
            if (!empty($category)) {
                // Get current Amazon country
                $amazon_country = get_option('ams_amazon_country', 'com');
                
                // Country-specific bestsellers URL mappings
                $bestsellers_url_maps = [
                    // United States (.com)
                    'com' => [
                        'Electronics' => 'electronics',
                        'Computers' => 'pc',
                        'Books' => 'books',
                        'HomeAndKitchen' => 'home-garden',
                        'Fashion' => 'fashion',
                        'Beauty' => 'beauty',
                        'SportsAndOutdoors' => 'sporting-goods',
                        'ToysAndGames' => 'toys-and-games',
                        'Baby' => 'baby-products',
                        'Automotive' => 'automotive',
                        'HealthPersonalCare' => 'hpc',
                        'MoviesAndTV' => 'movies-tv',
                        'Music' => 'popular',
                        'VideoGames' => 'videogames',
                        'Appliances' => 'appliances',
                        'ArtsAndCrafts' => 'arts-crafts',
                        'ToolsAndHomeImprovement' => 'hi',
                        'PetSupplies' => 'pet-supplies',
                        'OfficeProducts' => 'office-products'
                    ],
                    // Netherlands (.nl)
                    'nl' => [
                        'Electronics' => 'electronics',
                        'Computers' => 'computers',
                        'Books' => 'books',
                        'HomeAndKitchen' => 'kitchen',
                        'Fashion' => 'fashion',
                        'Beauty' => 'beauty',
                        'SportsAndOutdoors' => 'sports',
                        'ToysAndGames' => 'toys-games',
                        'Baby' => 'baby',
                        'Automotive' => 'automotive',
                        'HealthPersonalCare' => 'drugstore',
                        'MoviesAndTV' => 'dvd',
                        'Music' => 'popular',
                        'VideoGames' => 'videogames',
                        'Appliances' => 'appliances',
                        'ArtsAndCrafts' => 'arts-crafts',
                        'ToolsAndHomeImprovement' => 'diy',
                        'PetSupplies' => 'pet-supplies',
                        'OfficeProducts' => 'office-products'
                    ],
                    // Germany (.de)
                    'de' => [
                        'Electronics' => 'ce-de',
                        'Computers' => 'computers',
                        'Books' => 'books-intl-de',
                        'HomeAndKitchen' => 'kitchen',
                        'Fashion' => 'apparel',
                        'Beauty' => 'beauty',
                        'SportsAndOutdoors' => 'sports',
                        'ToysAndGames' => 'toys',
                        'Baby' => 'baby',
                        'Automotive' => 'automotive',
                        'HealthPersonalCare' => 'drugstore',
                        'MoviesAndTV' => 'dvd-de',
                        'Music' => 'popular',
                        'VideoGames' => 'videogames',
                        'Appliances' => 'appliances',
                        'ArtsAndCrafts' => 'arts-crafts-intl-de',
                        'ToolsAndHomeImprovement' => 'diy',
                        'PetSupplies' => 'pet-supplies-intl-de',
                        'OfficeProducts' => 'officeproducts-intl-de'
                    ],
                    // United Kingdom (.co.uk)
                    'co.uk' => [
                        'Electronics' => 'electronics',
                        'Computers' => 'computers',
                        'Books' => 'books',
                        'HomeAndKitchen' => 'kitchen',
                        'Fashion' => 'clothing',
                        'Beauty' => 'beauty',
                        'SportsAndOutdoors' => 'sports',
                        'ToysAndGames' => 'toys',
                        'Baby' => 'baby',
                        'Automotive' => 'automotive',
                        'HealthPersonalCare' => 'drugstore',
                        'MoviesAndTV' => 'dvd',
                        'Music' => 'popular',
                        'VideoGames' => 'videogames',
                        'Appliances' => 'appliances',
                        'ArtsAndCrafts' => 'arts-crafts',
                        'ToolsAndHomeImprovement' => 'diy',
                        'PetSupplies' => 'pet-supplies',
                        'OfficeProducts' => 'office-products'
                    ],
                    // France (.fr)
                    'fr' => [
                        'Electronics' => 'electronics',
                        'Computers' => 'computers',
                        'Books' => 'books-fr',
                        'HomeAndKitchen' => 'kitchen',
                        'Fashion' => 'apparel',
                        'Beauty' => 'beauty',
                        'SportsAndOutdoors' => 'sports',
                        'ToysAndGames' => 'toys',
                        'Baby' => 'baby',
                        'Automotive' => 'automotive',
                        'HealthPersonalCare' => 'drugstore',
                        'MoviesAndTV' => 'dvd-fr',
                        'Music' => 'popular',
                        'VideoGames' => 'videogames',
                        'Appliances' => 'appliances',
                        'ArtsAndCrafts' => 'arts-crafts-intl-fr',
                        'ToolsAndHomeImprovement' => 'diy',
                        'PetSupplies' => 'pet-supplies-intl-fr',
                        'OfficeProducts' => 'officeproducts-intl-fr'
                    ]
                ];
                
                // Get country-specific mapping or fallback to .com
                $country_map = $bestsellers_url_maps[$amazon_country] ?? $bestsellers_url_maps['com'];
                
                // Use mapped URL or fallback to lowercase
                $url_segment = isset($country_map[$category]) ? $country_map[$category] : strtolower($category);
                $url = $base_url . '/gp/bestsellers/' . $url_segment . '/';
            } else {
                $url = $base_url . '/gp/bestsellers/';
            }
            
            // Add pagination for bestsellers
            if ($current_page > 1) {
                $url .= '?pg=' . $current_page;
            }
            
            // Try direct request first, then fallback to scraping service
            $response_body = $this->getContentUsingCurl($url, null, true);

            if (empty($response_body)) {
                $response_body = executeScrapingService($url);
                if (empty($response_body)) {
                    break;
                }
            }
            
            if (!$response_body) {
                break;
            }
            
            // Parse HTML
            if (!class_exists('simple_html_dom')) {
                require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
            }
            $html = new \simple_html_dom();
            $html->load($response_body);
            
            if (!$html) {
                break;
            }
            
            // Find bestsellers using multiple selectors
            $bestseller_selectors = [
                '#zg-ordered-list li',           // Main bestsellers list
                '.zg-item-immersion',            // Individual bestseller items
                '.zg-item',                      // Alternative bestseller items
                '[data-asin]',                   // Any element with ASIN
                '.a-cardui',                     // Amazon card UI elements
                'li[id*="item"]'                 // List items with item ID
            ];
            
            $search_items = [];
            foreach ($bestseller_selectors as $selector) {
                $search_items = $html->find($selector);
                if (!empty($search_items)) {
                    break;
                }
            }
            
            if (empty($search_items)) {
                break;
            }
            $page_items = count($search_items);
            $total_results += $page_items;
            
            $processed_count = 0;
            $skipped_count = 0;
            
            foreach ($search_items as $search_item) {
                $data_asin = $search_item->getAttribute('data-asin') ?? '';
                
                if ($data_asin) {
                    $processed_count++;
                    
                    // Extract title using bestseller-specific selectors
                    $title_selectors = [
                        'a[href*="/dp/"]',
                        'a[href*="/dp/"] span',
                        '.a-link-normal',
                        '.a-link-normal span',
                        '.p13n-sc-truncated',
                        '.a-size-mini .a-link-normal',
                        '.a-size-base-plus',
                        'h2 a span',
                        'h2 span',
                        '.zg-item h3',
                        '.zg-item .a-link-normal',
                        '.s-size-mini span',
                        '.a-size-base span'
                    ];
                    
                    $title_element = null;
                    foreach ($title_selectors as $selector) {
                        $title_element = $search_item->find($selector, 0);
                        if ($title_element && trim($title_element->plaintext)) {
                            break;
                        }
                    }
                    
                    if (!$title_element) {
                        continue;
                    }
                    
                    // Extract product link using bestseller-specific selectors
                    $link_selectors = [
                        'a[href*="/dp/"]',
                        '.a-link-normal[href*="/dp/"]',
                        '.a-link-normal',
                        '.p13n-sc-truncated a',
                        '.zg-item a[href*="/dp/"]',
                        'h2 a'
                    ];
                    
                    $anchor = null;
                    foreach ($link_selectors as $selector) {
                        $anchor = $search_item->find($selector, 0);
                        if ($anchor && $anchor->getAttribute('href') && strpos($anchor->getAttribute('href'), '/dp/') !== false) {
                            break;
                        }
                    }
                    
                    $detailPageURL = $anchor ? $base_url . $anchor->getAttribute('href') : '';
                    if (strpos($detailPageURL, "/dp/") === false) {
                        continue;
                    }
                    
                    // Extract price using the same method as working products-search-without-api
                    $price_data = $this->fetchPriceFromHtml($search_item);
                    $regular_price = isset($price_data['regular_price']) ? $price_data['regular_price'] : 0;
                    $sale_price = isset($price_data['sale_price']) ? $price_data['sale_price'] : 0;
                    $currency = 'EUR'; // Default for bestsellers
                    

                    
                    // Extract image
                    $image_element = $search_item->find('img', 0);
                    $imageURL = '';
                    if ($image_element) {
                        $imageURL = $image_element->getAttribute('src') ?: $image_element->getAttribute('data-src');
                        if ($imageURL && strpos($imageURL, 'http') !== 0) {
                            $imageURL = 'https:' . $imageURL;
                        }
                    }
                    
                    // Extract rating and review count with multiple selectors
                    $rating = 0;
                    $review_count = 0;
                    $rating_selectors = [
                        '.a-icon-alt',
                        '.a-star-rating .a-icon-alt',
                        '.a-icon[class*="star"] .a-icon-alt',
                        '.cr-lighthouse-rating',
                        '.a-popover-trigger .a-icon-alt',
                        '.s-link-style .a-icon-alt',
                        '[aria-label*="out of 5"]',
                        '[title*="out of 5"]'
                    ];
                    
                    foreach ($rating_selectors as $selector) {
                        $rating_element = $search_item->find($selector, 0);
                        if ($rating_element) {
                            $rating_text = $rating_element->getAttribute('alt') ?: 
                                          $rating_element->getAttribute('aria-label') ?: 
                                          $rating_element->getAttribute('title') ?: 
                                          $rating_element->plaintext;
                            
                            // Support multiple rating formats (English, Dutch, German, etc.)
                            if (preg_match('/(\d+[.,]?\d*)\s*(out\s*of|van|von|sur|de)\s*5/i', $rating_text, $matches)) {
                                $rating_str = str_replace(',', '.', $matches[1]); // Convert comma to dot for parsing
                                $rating = floatval($rating_str);
                                break;
                            }
                        }
                    }
                    
                    // Extract review count
                    $review_count_selectors = [
                        'a[href*="#customerReviews"]',
                        'a[href*="reviews"]',
                        '.a-size-base.a-link-normal',
                        '.a-link-normal[href*="reviews"]',
                        'span.a-size-base',
                        '.cr-lighthouse-rating + a',
                        '.a-popover-trigger + a'
                    ];
                    
                    foreach ($review_count_selectors as $selector) {
                        $review_element = $search_item->find($selector, 0);
                        if ($review_element) {
                            $review_text = $review_element->plaintext;

                            
                            // Extract numbers from review text (support multiple formats)
                            // Pattern 1: "4,7 van 5 sterren  8.591" - extract the trailing number
                            if (preg_match('/van\s+5\s+sterren\s+(\d+(?:[.,]\d+)*)/i', $review_text, $matches)) {
                                $review_count_str = str_replace([',', '.'], '', $matches[1]); // "8.591" → "8591"
                                $review_count = intval($review_count_str);
                                break;
                            }
                            // Pattern 2: Standard formats like "1,234 reviews"
                            elseif (preg_match('/(\d+(?:[.,]\d+)*)\s*(?:reviews?|beoordelingen?|recensies?|bewertungen?|avis?)/i', $review_text, $matches)) {
                                $review_count_str = str_replace([',', '.'], '', $matches[1]);
                                $review_count = intval($review_count_str);
                                break;
                            }
                            // Pattern 3: Just numbers at the end of text
                            elseif (preg_match('/(\d+(?:[.,]\d+)*)\s*$/', trim($review_text), $matches)) {
                                $review_count_str = str_replace([',', '.'], '', $matches[1]);
                                $review_count = intval($review_count_str);
                                break;
                            }
                        }
                    }
                    
                    // Extract currency - simple approach for Dutch Amazon
                    $currency = 'EUR';
                    

                    
                    // Check if product is already imported (using pre-loaded list for efficiency)
                    $is_imported = in_array($data_asin, $imported_asins);
                    
                    $lists[] = array(
                        'ASIN' => $data_asin,
                        'Title' => trim($title_element->plaintext),
                        'DetailPageURL' => $detailPageURL,
                        'ImageURL' => $imageURL,
                        'RegularPrice' => $regular_price,
                        'SalePrice' => $sale_price,
                        'Currency' => $currency,
                        'Rating' => $rating,
                        'ReviewCount' => $review_count,
                        'IsBestseller' => true,
                        'IsImported' => $is_imported
                    );
                    
                    // Limit results per page
                    if (count($lists) >= $results_limit) {
                        break 2;
                    }
                }
            }
            
            // If we got fewer items than expected, we might be at the end
            if ($page_items < 10) {
                break;
            }
        }
        
        // Cache results for 5 minutes
        set_transient($cache_key, $lists, 5 * MINUTE_IN_SECONDS);
        
        return $lists;
    }

    /**
     * Validate if a text string is a valid brand name
     */
    private function isValidBrand($brand_text) {
        if (empty($brand_text)) return false;
        
        $brand_text = trim($brand_text);
        
        // Basic validation
        if (strlen($brand_text) < 2 || strlen($brand_text) > 50) return false;
        
        // Exclude common non-brand patterns
        $exclude_patterns = [
            '/^\d+$/',                          // Just numbers
            '/\$|€|£|¥|USD|EUR|GBP/',          // Prices/currencies
            '/\d+\s*(stars?|reviews?)/i',       // Ratings
            '/\d+%\s*(off|discount)/i',         // Discounts
            '/^(free|shipping|delivery)/i',     // Shipping terms
            '/^(new|used|refurbished)/i',       // Condition terms
            '/^(amazon|prime)/i',               // Amazon terms
            '/^(bestseller|choice)/i',          // Badge terms
        ];
        
        foreach ($exclude_patterns as $pattern) {
            if (preg_match($pattern, $brand_text)) {
                return false;
            }
        }
        
        return true;
    }

    /**
     * Extract brand from product title
     */
    private function extractBrandFromTitle($title) {
        if (empty($title)) return '';
        
        // Common brand patterns in titles
        $known_brands = [
            'Samsung', 'Apple', 'Sony', 'LG', 'Huawei', 'Xiaomi', 'OnePlus', 'Google', 'Microsoft',
            'HP', 'Dell', 'Lenovo', 'Asus', 'Acer', 'MSI', 'Razer', 'Corsair', 'Logitech',
            'Canon', 'Nikon', 'Panasonic', 'JBL', 'Bose', 'Sennheiser', 'Beats', 'Anker',
            'Philips', 'Braun', 'Oral-B', 'Gillette', 'Nike', 'Adidas', 'Puma', 'Under Armour'
        ];
        
        // Look for known brands at the beginning of the title
        foreach ($known_brands as $brand) {
            if (stripos($title, $brand) === 0 || stripos($title, $brand . ' ') !== false) {
                return $brand;
            }
        }
        
        // Try to extract first word if it looks like a brand
        $words = explode(' ', $title);
        $first_word = trim($words[0]);
        
        if ($this->isValidBrand($first_word) && ctype_alpha($first_word)) {
            return $first_word;
        }
        
        return '';
    }
}