<?php

if( !class_exists( 'Chart_Builder_Functions' ) ){

	/**
	 * Class Chart_Builder_Functions
	 * Class contains useful functions that uses in common
	 *
	 *
	 * Hooks used in the class
	 * There are not hooks yet
	 * @hooks           @actions
	 *                  @filters
	 *
	 *
	 * @param           $plugin_name
	 *
	 * @since           1.0.0
	 * @package         Chart_Builder
	 * @subpackage      Chart_Builder/includes
	 * @author          Chart Builder Team <info@ays-pro.com>
	 */
    class Chart_Builder_Functions {

        /**
         * The ID of this plugin.
         *
         * @since    1.0.0
         * @access   private
         * @var      string    $plugin_name    The ID of this plugin.
         */
        private $plugin_name;

		/**
         * The database table name
         *
         * @since    1.0.0
         * @access   private
         * @var      string    $db_table    The database table name
         */
        private $db_table;

	    /**
	     * The constructor of the class
	     *
	     * @since   1.0.0
	     * @param   $plugin_name
	     */
        public function __construct( $plugin_name ) {
			global $wpdb;
	        /**
	         * Assigning $plugin_name to the @plugin_name property
	         */
            $this->plugin_name = $plugin_name;
			$this->db_table = $wpdb->prefix . CHART_BUILDER_DB_PREFIX . "charts";
        }


	    /**
	     * Get instance of this class
	     *
	     * @access  public
	     * @since   1.0.0
	     * @param   $plugin_name
	     * @return  Chart_Builder_Functions
	     */
	    public static function get_instance( $plugin_name ){
		    return new self( $plugin_name );
	    }

	    /**
         * Date validation function
         *
         * @accept  two parameters
	     * @param   $date
	     * @param   string $format
	     *
	     * @return  bool
	     */
        public function validateDate( $date, $format = 'Y-m-d H:i:s' ){
            $d = DateTime::createFromFormat($format, $date);
            return $d && $d->format($format) == $date;
        }


	    /**
	     * Version compare function
	     *
	     * @accept  two parameters
	     * @param   $version1
	     * @param   $operator
	     * @param   $version2
	     *
	     * @return  bool|int
	     */
        public function versionCompare( $version1, $operator, $version2 ) {

            $_fv = intval ( trim ( str_replace ( '.', '', $version1 ) ) );
            $_sv = intval ( trim ( str_replace ( '.', '', $version2 ) ) );

            if (strlen ( $_fv ) > strlen ( $_sv )) {
                $_sv = str_pad ( $_sv, strlen ( $_fv ), 0 );
            }

            if (strlen ( $_fv ) < strlen ( $_sv )) {
                $_fv = str_pad ( $_fv, strlen ( $_sv ), 0 );
            }

            return version_compare ( ( string ) $_fv, ( string ) $_sv, $operator );
        }

        public function get_all_charts_count() {
			global $wpdb;
			$settings_obj = new Chart_Builder_Settings_DB_Actions($this->plugin_name);
			$where = str_replace("WHERE", "AND", Chart_Builder_DB_Actions::get_where_condition());
            $current_user_can_edit = $settings_obj->get_permission_for_editing_plugin();
			
			$sql = "SELECT COUNT(id) FROM " . $this->db_table . " WHERE `status`='published' " . $where;

            if ( ! $current_user_can_edit ) {
                $current_user = get_current_user_id();
                $sql .= "AND author_id = ".$current_user." ";
            }
			
			return intval( $wpdb->get_var( $sql ) );
        }


	    /**
	     * Gets the properties of the post type
	     *
	     * @access friendly
	     */
        public function fetch_post_type_properties() {
            $array = $this->get_post_type_properties( $_POST['post_type'] );
            $this->_sendResponse(
                array(
                    'success' => true,
                    'fields'  => $array,
                )
            );
            wp_die();
	    }

	    /**
	     * Gets the allowed types
	     *
	     * @access private
	     */
	    public function getAllowedTypes() {
		    return array(
			    'string',
			    'number',
			    'boolean',
			    'date',
			    'datetime',
			    'timeofday',
		    );
	    }

	    /**
	     * Gets the properties of the post type
	     *
	     * @access private
	     */
	    public function get_post_type_properties( $post_type ) {
		    $array = null;

		    $query = new WP_Query(
			    array(
				    'post_type'              => $post_type,
				    'no_rows_found'          => false,
				    'post_per_page'          => 1,
				    'orderby'                => 'post_date',
				    'order'                  => 'DESC',
				    'fields'                 => 'ids',
				    'update_post_meta_cache' => false,
				    'update_post_term_cache' => false,
			    )
		    );

		    $array = array(
			    'post_title',
			    'post_status',
			    'comment_count',
			    'post_date',
			    'post_modified',
		    );

		    if ( $query->have_posts() ) {
			    $id   = $query->posts[0];
			    $meta = get_post_meta( $id, '', true );
			    foreach ( $meta as $key => $values ) {
				    $array[] = $key;
			    }
		    }

		    return $array;
	    }

	    /**
	     * Gets all tables and their columns.
	     *
	     * @access public
	     * @return array
	     */
	    public function get_all_db_tables_column_mapping( $chart_id, $use_filter = true ) {
			global $wpdb;
		    $mapping    = array();
		    $tables     = $this->get_db_tables();
		    foreach ( $tables as $table ) {
		    	$query = $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $table ) );
				if ( ! $wpdb->get_var( $query ) == $table ) {
				    $cols   = $this->get_db_table_columns( $table, true );
				    $names  = wp_list_pluck( $cols, 'name' );
				    $mapping[ $table ] = $names;
				}
		    }

		    return $mapping;
	    }

	    /**
	     * Gets the tables in the database;
	     *
	     * @access public
	     * @return array
	     */
	    public function get_db_tables() {
		    global $wpdb;
		    $tables = get_transient( CHART_BUILDER_DB_PREFIX . 'db_tables' );
		    if ( $tables ) {
			    return $tables;
		    }
			$tables = array();

		    $sql    = $wpdb->get_col( 'SHOW TABLES', 0 );
		    foreach ( $sql as $table ) {
			    if ( empty( $prefix ) || 0 === strpos( $table, $wpdb->prefix ) ) {
				    $tables[] = $table;
			    }
		    }

		    set_transient( CHART_BUILDER_DB_PREFIX . 'db_tables', $tables, HOUR_IN_SECONDS );
		    return $tables;
	    }

	    /**
	     * Gets the column information for the table.
	     *
	     * @param string $table The table.
	     * @param bool   $prefix_with_table Whether to prefix column name with the name of the table.
	     * @access private
	     * @return array
	     */
	    private function get_db_table_columns( $table, $prefix_with_table = false ) {
		    global $wpdb;
		    $columns    = get_transient( CHART_BUILDER_DB_PREFIX . "db_{$table}_columns" );
		    if ( $columns ) {
			    return $columns;
		    }
		    $columns    = array();
		    // @codingStandardsIgnoreStart
		    $rows       = $wpdb->get_results( "SHOW COLUMNS IN `$table`", ARRAY_N );
		    // @codingStandardsIgnoreEnd
		    if ( $rows ) {
			    // n => numeric, d => date-ish, s => string-ish.
			    foreach ( $rows as $row ) {
				    $col        = ( $prefix_with_table ? "$table." : '' ) . $row[0];
				    $type       = $row[1];
				    if ( strpos( $type, 'int' ) !== false || strpos( $type, 'float' ) !== false ) {
					    $type   = 'n';
				    } elseif ( strpos( $type, 'date' ) !== false || strpos( $type, 'time' ) !== false ) {
					    $type   = 'd';
				    } else {
					    $type   = 's';
				    }
				    $columns[]  = array( 'name' => $col, 'type' => $type );
			    }
		    }

		    set_transient( CHART_BUILDER_DB_PREFIX . "db_{$table}_columns", $columns, DAY_IN_SECONDS );
		    return $columns;
	    }

	    /**
	     * Gets the dependent tables and then gets column information for all the tables.
	     *
	     * @param string $table The table.
	     * @access public
	     * @return array
	     */
	    public function get_table_columns( $table ) {
		    $columns    = array();
		    if ( ! $table ) {
			    return $columns;
		    }

		    $tables = array( $table );
		    $mapping = $this->get_db_table_mapping();
		    if ( array_key_exists( $table, $mapping ) ) {
			    $tables[] = $mapping[ $table ];
		    }
		    foreach ( $tables as $table ) {
			    $cols = $this->get_db_table_columns( $table, count( $tables ) > 1 );
			    $columns = array_merge( $columns, $cols );
		    }
		    return $columns;
	    }

	    /**
	     * Gets the relationship between tables in the database.
	     *
	     * @access public
	     * @return array
	     */
	    public function get_db_table_mapping() {
		    global $wpdb;
		    $mapping = get_transient( CHART_BUILDER_DB_PREFIX . 'db_table_mapping' );
		    if ( $mapping ) {
			    return $mapping;
		    }
		    // no need to provide x=>y and then y=>x as we will flip the array shortly.
		    $mapping = array(
			    $wpdb->prefix . 'posts' => $wpdb->prefix . 'postmeta',
			    $wpdb->prefix . 'users' => $wpdb->prefix . 'usermeta',
			    $wpdb->prefix . 'terms' => $wpdb->prefix . 'termmeta',
			    $wpdb->prefix . 'comments' => $wpdb->prefix . 'commentmeta',
		    );

		    $mapping += array_flip( $mapping );
		    set_transient( CHART_BUILDER_DB_PREFIX . 'db_table_mapping', $mapping, HOUR_IN_SECONDS );
		    return $mapping;
	    }

	    /**
	     * Creates HTML table from passed data
	     *
	     * @access public
	     * @return string
	     */
	    public function get_table_html( $headers, $rows, $table_id = 'results' ) {
		    ob_start();
		    ?>
		    <table cellspacing="0" width="100%" id="<?= $table_id ?>">
			    <thead>
			    <tr>
				    <?php
				    foreach ( $headers as $header ) {
                        if( empty( $header ) ){
                            continue;
                        }
					    echo '<th>' . $header . '</th>';
				    }
				    ?>
			    </tr>
			    </thead>
			    <tfoot>
			    </tfoot>
			    <tbody>
			    <?php
			    foreach ( $rows as $row ) {
				    if( empty( $row ) ){
					    continue;
				    }
				    echo '<tr>';
				    foreach ( $row as $r ) {
					    echo '<td>' . $r . '</td>';
				    }
				    echo '</tr>';
			    }
			    ?>
			    </tbody>
		    </table>
		    <?php
		    return ob_get_clean();
	    }

		/**
	     * Gets the the queries from quiz maker database tables.
	     *
	     * @access public
	     * @return array
	     */
		public function get_quiz_query ($query, $quiz_id = null, $quiz_cat_id = null, $quiz_question_cat_id = null, $user_id = null) {
			global $wpdb;
			$reports_table = $wpdb->prefix . 'aysquiz_reports';
			$quizes_table = $wpdb->prefix . 'aysquiz_quizes';
			$questions_table = $wpdb->prefix . 'aysquiz_questions';

			switch ($query) {
				case 'q1':
					$sql = "SELECT CAST(end_date AS date) AS `Date`, COUNT(id) AS `Count` 
							FROM ".$reports_table." 
							WHERE quiz_id = ".$quiz_id." 
							GROUP BY CAST(end_date AS date)";
					break;
				case 'q2':
					$sql = "SELECT CAST(end_date AS date) AS `Date`, COUNT(id) AS `Count` 
							FROM ".$reports_table." 
							WHERE user_id = ".$user_id." 
							GROUP BY CAST(end_date AS date)";
					break;
				case 'q3':
					$sql = "SELECT CAST(end_date AS date) AS `Date`, COUNT(id) AS `Count` 
							FROM ".$reports_table." 
							WHERE user_id = ".$user_id." AND quiz_id = ".$quiz_id." 
							GROUP BY CAST(end_date AS date)";
					break;
				case 'q4':
					$sql = "SELECT ".$quizes_table.".title AS `Quiz`, AVG(".$reports_table.".score) AS `Average` 
							FROM ".$reports_table." 
							LEFT JOIN ".$quizes_table." ON ".$reports_table.".quiz_id = ".$quizes_table.".id 
							WHERE ".$reports_table.".user_id = ".$user_id." 
							GROUP BY ".$quizes_table.".title";
					break;
				case 'q5':
					$sql = "SELECT ".$quizes_table.".title AS `Quiz`, COUNT(".$reports_table.".score) AS `Count` 
							FROM ".$reports_table." 
							LEFT JOIN ".$quizes_table." ON ".$reports_table.".quiz_id = ".$quizes_table.".id 
							WHERE ".$reports_table.".user_id = ".$user_id." 
							GROUP BY ".$quizes_table.".title";
					break;
				case 'q6':
					$sql = "SELECT score AS `Score` 
							FROM ".$reports_table." 
							WHERE user_id = ".$user_id." AND quiz_id = ".$quiz_id."";
					break;
				case 'q7':
					$sql = "SELECT ".$quizes_table.".title AS `Quiz`, AVG(".$reports_table.".score) AS `Average` 
							FROM ".$reports_table." 
							LEFT JOIN ".$quizes_table." ON ".$reports_table.".quiz_id = ".$quizes_table.".id 
							WHERE ".$reports_table.".user_id = ".$user_id." AND ".$quizes_table.".quiz_category_id = ".$quiz_cat_id." 
							GROUP BY ".$quizes_table.".title";
					break;
				case 'q8':
					$sql = "SELECT ".$quizes_table.".title AS `Quiz`, COUNT(".$reports_table.".score) AS `Count` 
							FROM ".$reports_table." 
							LEFT JOIN ".$quizes_table." ON ".$reports_table.".quiz_id = ".$quizes_table.".id 
							WHERE ".$reports_table.".user_id = ".$user_id." AND ".$quizes_table.".quiz_category_id = ".$quiz_cat_id." 
							GROUP BY ".$quizes_table.".title";
					break;
				case 'q9':
					$sql = "SELECT score AS `Score`, COUNT(user_id) AS `Count` 
							FROM ".$reports_table." 
							WHERE quiz_id = ".$quiz_id." 
							GROUP BY score";
					break;
				case 'q10':
					$sql = "SELECT (CASE WHEN score >= 0 AND score <= 25 THEN '0-25%'
										WHEN score > 25 AND score <= 50 THEN '26-50%'
										WHEN score > 50 AND score <= 75 THEN '51-75%'
										WHEN score > 75 AND score <= 100 THEN '76-100%'
								    END) AS 'Interval', COUNT(*) as 'Count'
	    					FROM ".$reports_table."
	    					WHERE quiz_id = ".$quiz_id."
	    					GROUP BY (CASE WHEN score >= 0 AND score <= 25 THEN '0-25%'
										WHEN score > 25 AND score <= 50 THEN '26-50%'
										WHEN score > 50 AND score <= 75 THEN '51-75%'
										WHEN score > 75 AND score <= 100 THEN '76-100%'
								    END)";
					break;
				case 'q11':
					$sql = "SELECT CAST(end_date AS date) as 'Date', SUM(CASE WHEN user_id > 0 THEN 1 ELSE 0 END) as 'Logged in', SUM(CASE WHEN user_id = 0 THEN 1 ELSE 0 END) as 'Guest' 
							FROM ".$reports_table." 
							WHERE quiz_id = ".$quiz_id." AND end_date > current_date - interval 7 day 
							GROUP BY CAST(end_date AS date)";
					break;
				case 'q12':
					$message = "";
					$sql = "SELECT question_ids FROM ".$quizes_table." WHERE id = ".$quiz_id."";
					$results = $wpdb->get_var( $sql);

					$questions = array();
					$questions_count = array();
					$correct_q = array();
					$not_correct_q = array();
					$not_ans_q = array();

					if($results != ''){
					    $sql = "SELECT id, question FROM ".$questions_table." WHERE id IN (".$results.")";
					    $questions_list = $wpdb->get_results($sql, "ARRAY_A");
					    foreach($questions_list as $k => $v){
					        $questions[$v['id']] = $v['question'];
					        $questions_count[$v['id']] = 0;
					        $correct_q[$v['id']] = 0;
					        $not_correct_q[$v['id']] = 0;
					        $not_ans_q[$v['id']] = 0;
					    }
					
					    $results = explode(",", $results);
					    $ordered = array();
					    foreach ($results as $key) {
					        if (array_key_exists($key, $correct_q)) {
					            $ordered[$key] = $correct_q[$key];
					            unset($correct_q[$key]);
					        }
					    }
					    $correct_q = $ordered + $correct_q;
					}

					$sql = "SELECT options FROM ".$reports_table." WHERE quiz_id = ".$quiz_id;

					$quiz_version = intval(get_plugin_data( WP_PLUGIN_DIR . "\quiz-maker\quiz-maker.php" )['Version']);
					if ($quiz_version >= 20) {
						$sql .= " AND `status` = 'finished'";
					}
					
					$reports = $wpdb->get_results( $sql, "ARRAY_A" );
					$result = array();
					if(! empty($reports)){
					    foreach ($reports as $key){
					        $reports = json_decode($key["options"]);
							
					        $questions_correctness = $reports->correctness;
					        if ( is_object( $questions_correctness ) && !empty( $questions_correctness ) ) {
					            foreach ($questions_correctness as $i => $v){
					                $q = (int) substr($i,12);
					                if(isset($correct_q[$q])) {
					                    if ($v) {
					                        $correct_q[$q]++;
					                    }
					                    $questions_count[$q]++;
					                }
					            }
					        }
							
							$user_answered = $reports->user_answered;
							if ( is_object( $user_answered ) && !empty( $user_answered ) ) {
					            foreach ($user_answered as $i => $v){
					                $q = (int) substr($i,12);
					                if(isset($correct_q[$q])) {
					                    if ($v == '') {
					                        $not_ans_q[$q]++;
					                    }
									}
					            }
					        }
					    }

						foreach ($questions_count as $q => $c) {
							$not_correct_q[$q] = $c - $correct_q[$q] - $not_ans_q[$q];
						}

						$result = array();
						foreach ($questions as $key => $question) {
							$row = array();
							$row['Question'] = $question;
							$row['Correct'] = strval($correct_q[$key]);
							$row['Not correct'] = strval($not_correct_q[$key]);
							$row['Not answered'] = strval($not_ans_q[$key]);

							$result[] = $row;
						}
					}
					
					if (empty($result)) {
						if ( !(isset($message) && $message != "") ) {
							$message = __( "There is no data for your query.", "chart-builder" );
						}
					}

					return array(
						'message' => $message,
						'result' => $result,
						'query' => $sql
					);
					break;
				case 'q13':
					$message = "";
					if ($quiz_id != '0' && $quiz_question_cat_id == '0') {
						$sql = "SELECT question_ids FROM ".$quizes_table." WHERE id = ".$quiz_id."";
						$results = $wpdb->get_var( $sql);
						if($results != ''){
							$sql = "SELECT id, question FROM ".$questions_table." WHERE id IN (".$results.")";
							$questions_list = $wpdb->get_results($sql, "ARRAY_A");
						} else {
							$questions_list = '';
						}
					} else if ($quiz_id == '0' && $quiz_question_cat_id != '0') {
						$sql = "SELECT id, question FROM ".$questions_table." WHERE category_id = ".$quiz_question_cat_id."";
						$questions_list = $wpdb->get_results($sql, "ARRAY_A");
						$results = array();
						foreach ($questions_list as $q => $v) {
							$results[] = $v['id'];
						}
						
						$sql = "SELECT id, question_ids FROM ".$quizes_table." WHERE published = 1";
						$quizes_q_list = $wpdb->get_results($sql, "ARRAY_A");
						$matches = array();
						foreach ($quizes_q_list as $key => $value) {
							$res = array_intersect(explode(",", $value['question_ids']), $results);
							if (!empty($res)) {
								$matches[] = $value['id'];
							}
						}
						$results = implode(",", $results);
					} else {
						$sql = "SELECT question_ids FROM ".$quizes_table." WHERE id = ".$quiz_id."";
						$results = $wpdb->get_var( $sql);
						if($results != ''){
							$sql = "SELECT id, question FROM ".$questions_table." WHERE id IN (".$results.") AND category_id = ".$quiz_question_cat_id."";
							$questions_list = $wpdb->get_results($sql, "ARRAY_A");
						} else {
							$questions_list = '';
						}
						$message = empty($questions_list) ? __( "The questions from chosen category aren't included in the selected quiz.", "chart-builder" ) : "";
					}
					
					$questions = array();
					$questions_count = array();
					$correct_q = array();
					$not_correct_q = array();
					$not_ans_q = array();

					if($questions_list != ''){
					    foreach($questions_list as $k => $v){
					        $questions[$v['id']] = $v['question'];
					        $questions_count[$v['id']] = 0;
					        $correct_q[$v['id']] = 0;
					        $not_correct_q[$v['id']] = 0;
					        $not_ans_q[$v['id']] = 0;
					    }
					
						$results = explode(",", $results);
						$ordered = array();
						foreach ($results as $key) {
							if (array_key_exists($key, $correct_q)) {
								$ordered[$key] = $correct_q[$key];
								unset($correct_q[$key]);
							}
						}
						$correct_q = $ordered + $correct_q;
					}
					
					$quiz_id = (isset($matches) && !is_null($matches)) ? implode(",", $matches) : $quiz_id.'';

					$sql = "SELECT options FROM ".$reports_table." WHERE quiz_id = ".$quiz_id;

					$quiz_version = intval(get_plugin_data( WP_PLUGIN_DIR . "\quiz-maker\quiz-maker.php" )['Version']);
					if ($quiz_version >= 20) {
						$sql .= " AND `status` = 'finished'";
					}

					$reports = $wpdb->get_results( $sql, "ARRAY_A" );
					$result = array();
					if(! empty($reports)){
					    foreach ($reports as $key){
					        $reports = json_decode($key["options"]);
							
					        $questions_correctness = $reports->correctness;
					        if ( is_object( $questions_correctness ) && !empty( $questions_correctness ) ) {
					            foreach ($questions_correctness as $i => $v){
					                $q = (int) substr($i,12);
					                if(isset($correct_q[$q])) {
					                    if ($v) {
					                        $correct_q[$q]++;
					                    }
					                    $questions_count[$q]++;
					                }
					            }
					        }
							
							$user_answered = $reports->user_answered;
							if ( is_object( $user_answered ) && !empty( $user_answered ) ) {
					            foreach ($user_answered as $i => $v){
					                $q = (int) substr($i,12);
					                if(isset($correct_q[$q])) {
					                    if ($v == '') {
					                        $not_ans_q[$q]++;
					                    }
									}
					            }
					        }
					    }

						foreach ($questions_count as $q => $c) {
							$not_correct_q[$q] = $c - $correct_q[$q] - $not_ans_q[$q];
						}

						foreach ($questions as $key => $question) {
							$row = array();
							$row['Question'] = $question;
							$row['Correct'] = strval($correct_q[$key]);
							$row['Not correct'] = strval($not_correct_q[$key]);
							$row['Not answered'] = strval($not_ans_q[$key]);

							$result[] = $row;
						}
					}

					if (empty($result)) {
						if ( !(isset($message) && $message != "") ) {
							$message = __( "There is no data for your query.", "chart-builder" );
						}
					}
					
					return array(
						'message' => $message,
						'result' => $result,
						'query' => $sql
					);
					break;
				case 'q14':
					$sql = "SELECT CASE WHEN user_id > 0 THEN 'Logged in' ELSE 'Guest' END AS 'User', COUNT(user_id) AS 'Count' 
							FROM ".$reports_table."
							WHERE end_date > current_date - interval 7 day
							GROUP BY User";
					break;
				default:
					break;
			}

			$result = $wpdb->get_results($sql, "ARRAY_A");
			$message = empty($result) ? __( "There is no data for your query.", "chart-builder" ) : "";
			return array(
				'message' => $message,
				'result' => $result,
				'query' => $sql
			);
		}

		public function get_woocommerce_query_data ($query) {
			global $wpdb;

			$message = "";
			$sql = "";
			$result = "";
			$processed_result = array();
			$wc_orders_table = $wpdb->prefix . "wc_orders";

			if($wpdb->get_var("SHOW TABLES LIKE '$wc_orders_table'") == $wc_orders_table) {
				if(( isset($query) && $query != "" ) ){
					// $days_count_in_month = cal_days_in_month(CAL_GREGORIAN,date('m'),date('Y'));
					$days_count_in_month = 30;
					if($days_count_in_month == 0){
						$message = __( "Something went wrong.", "chart-builder" );
					}
					
					// $start_date = date('Y-m-d',strtotime('-30 days')); // if choossing this,be sure sql queries are corrected
					$start_date = date('Y-m-d') . "' - INTERVAL 30 day";
					$end_date = date('Y-m-d');

					$date_format_month_day = array();
					for($i = 30; $i >= 0; $i--){ // >= 0 including today
						$date_format_month_day[] = date('Y-m-d',strtotime('-' . $i . ' days'));
					}
					
					switch (strtolower($query)){
						case "total_sale":
							$sql = "SELECT CAST(date_created_gmt AS DATE) AS DayOfMonth , currency,  SUM(total_amount) AS 'total_sale' from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) AND type = 'shop_order' AND CAST(date_created_gmt AS DATE) BETWEEN '" . $start_date ." AND '" . $end_date. "' GROUP BY currency, DayOfMonth ORDER BY DayOfMonth";
							$sql_currencies = "SELECT currency from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) GROUP BY currency";

							$result = $wpdb->get_results($sql, ARRAY_A);
							$result_currencies = $wpdb->get_results($sql_currencies, ARRAY_N);
							
							if (empty($result) || empty($result_currencies)) {
								if ( !(isset($message) && $message != "") ) {
									$message = __( "There is no data for your query.", "chart-builder" );
								}
							}
							else{
								$currencies_in_array['Day'] = 0;

								foreach($result_currencies as $index => $currency){
									$currencies_in_array[$currency[0]] = 0;
								}
								for ($i=0; $i <= 30; $i++) { 
									$currencies_in_array['Day'] = $date_format_month_day[$i];
									$processed_result[] = $currencies_in_array;
								}

								$index = 0;
								foreach($result as $key => $value){
									while(isset($processed_result[$index]) && $value['DayOfMonth'] != $processed_result[$index]['Day']){
										$index++;

									}
									if(isset($processed_result[$index])){
										$processed_result[$index][$value['currency']] = floatval($value['total_sale']);
									}
								}

								$processed_result = array_values($processed_result);
							}
							break;
							case "item_order":
								$sql = "SELECT CAST(date_created_gmt AS DATE) AS DayOfMonth , COUNT(*) AS 'item_order' from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) AND type = 'shop_order' AND CAST(date_created_gmt AS DATE) BETWEEN '" . $start_date ." AND '" . $end_date. "' GROUP BY DayOfMonth ORDER BY DayOfMonth";

								$result = $wpdb->get_results($sql, ARRAY_A);
								
								if (empty($result) ) {
									if ( !(isset($message) && $message != "") ) {
										$message = __( "There is no data for your query.", "chart-builder" );
									}
								}
								else{
									for ($i=0; $i <= 30; $i++) { 
										$key_values['Day'] = $date_format_month_day[$i];
										$key_values['Item Order'] = 0;
										$processed_result[] = $key_values;
									}
									$index = 0;
									foreach($result as $key => $value){
										
										while(isset($processed_result[$index]) && $value['DayOfMonth'] != $processed_result[$index]['Day']){
											$index++;
										}
										if(isset($processed_result[$index])){
											$processed_result[$index]['Item Order'] = floatval($value['item_order']);
										}
									}
								}
							break;
							case "refunds":
								$sql = "SELECT CAST(date_created_gmt AS DATE) AS DayOfMonth , COUNT(*) AS 'refunds' from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-refunded' ) ) . "' ) AND type = 'shop_order' AND CAST(date_created_gmt AS DATE) BETWEEN '" . $start_date ." AND '" . $end_date. "' GROUP BY DayOfMonth ORDER BY DayOfMonth";

								$result = $wpdb->get_results($sql, ARRAY_A);
								
								if (empty($result) ) {
									if ( !(isset($message) && $message != "") ) {
										$message = __( "There is no data for your query.", "chart-builder" );
									}
								}
								else{
									for ($i=0; $i <= 30; $i++) { 
										$key_values['Day'] = $date_format_month_day[$i];
										$key_values['Refunds'] = 0;
										$processed_result[] = $key_values;
									}
									$index = 0;
									foreach($result as $key => $value){
										
										while(isset($processed_result[$index]) && $value['DayOfMonth'] != $processed_result[$index]['Day']){
											$index++;
										}
										if(isset($processed_result[$index])){
											$processed_result[$index]['Refunds'] = floatval($value['refunds']);
										}
									}
								}
							break;
							case "avg_order_value": 
								$sql = "SELECT CAST(date_created_gmt AS DATE) AS DayOfMonth , currency,  AVG(total_amount) AS 'avg_order_value' from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) AND type = 'shop_order' AND CAST(date_created_gmt AS DATE) BETWEEN '" . $start_date ." AND '" . $end_date. "' GROUP BY currency, DayOfMonth ORDER BY DayOfMonth";
								$sql_currencies = "SELECT currency from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) GROUP BY currency";

								$result = $wpdb->get_results($sql, ARRAY_A);
								$result_currencies = $wpdb->get_results($sql_currencies, ARRAY_N);

								if (empty($result) || empty($result_currencies)) {
									if ( !(isset($message) && $message != "") ) {
										$message = __( "There is no data for your query.", "chart-builder" );
									}
								}
								else{
									$currencies_in_array['Day'] = 0;
									foreach($result_currencies as $index => $currency){
										$currencies_in_array[$currency[0]] = 0;
									}
									for ($i=0; $i <= 30; $i++) { 
										$currencies_in_array['Day'] = $date_format_month_day[$i];
										$processed_result[] = $currencies_in_array;
									}
					
									$index = 0;
									foreach($result as $key => $value){
										while(isset($processed_result[$index]) && $value['DayOfMonth'] != $processed_result[$index]['Day']){
											$index++;
										}
										if(isset($processed_result[$index])){
											$processed_result[$index][$value['currency']] = floatval($value['avg_order_value']);
										}
									}
								}
							break;
							case "total_tax": 
								$sql = "SELECT CAST(date_created_gmt AS DATE) AS DayOfMonth , currency,  SUM(tax_amount) AS 'total_tax' from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) AND type = 'shop_order' AND CAST(date_created_gmt AS DATE) BETWEEN '" . $start_date ." AND '" . $end_date. "' GROUP BY currency, DayOfMonth ORDER BY DayOfMonth";
								$sql_currencies = "SELECT currency from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) GROUP BY currency";
		
								$result = $wpdb->get_results($sql, ARRAY_A);
								$result_currencies = $wpdb->get_results($sql_currencies, ARRAY_N);
		
								if (empty($result) || empty($result_currencies)) {
									if ( !(isset($message) && $message != "") ) {
										$message = __( "There is no data for your query.", "chart-builder" );
									}
								}
								else{
									$currencies_in_array['Day'] = 0;
									foreach($result_currencies as $index => $currency){
										$currencies_in_array[$currency[0]] = 0;
									}
									for ($i=0; $i <= 30; $i++) { 
										$currencies_in_array['Day'] = $date_format_month_day[$i];
										$processed_result[] = $currencies_in_array;
									}
									$index = 0;
									foreach($result as $key => $value){
										while(isset($processed_result[$index]) && $value['DayOfMonth'] != $processed_result[$index]['Day']){
											$index++;
										}
										if(isset($processed_result[$index])){
											$processed_result[$index][$value['currency']] = floatval($value['total_tax']);
										}
									}
								}
								break;
							case "mixed": 
								$sql = "SELECT currency, SUM(tax_amount) AS 'total_tax' , AVG(total_amount) AS 'avg_order_value',  COUNT(*) AS 'item_order', SUM(total_amount) AS 'total_sale' from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) AND type = 'shop_order' AND CAST(date_created_gmt AS DATE) BETWEEN '" . $start_date ." AND '" . $end_date. "' GROUP BY currency";
								$sql_currencies = "SELECT currency from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-completed', 'wc-processing', 'wc-on-hold' ) ) . "' ) GROUP BY currency";
								$refund_sql = "SELECT currency, COUNT(*) AS 'refunds' from `" . $wpdb->prefix . "wc_orders` WHERE status IN ( '" . implode( "','", array( 'wc-refunded' ) ) . "' ) AND type = 'shop_order' AND CAST(date_created_gmt AS DATE) BETWEEN '" . $start_date ." AND '" . $end_date. "' GROUP BY currency";
		
								$result_currencies = $wpdb->get_results($sql_currencies, ARRAY_N);
								$result = $wpdb->get_results($sql, ARRAY_A);
								$refund_count_result = $wpdb->get_results($refund_sql, ARRAY_N);

								if (empty($result) || empty($result_currencies) ) {
									if ( !(isset($message) && $message != "") ) {
										$message = __( "There is no data for your query.", "chart-builder" );
									}
								}
								else{
									if( !empty($refund_count_result) ){
										for($i = 0; $i < count($refund_count_result); $i++){
											for($j = 0; $j < count($result); $j++){
												
												if($result[$j]['currency'] == $refund_count_result[$i][0]){
													$result[$j]['refunds'] = intval($refund_count_result[$i][1]);
												}
												else{
													if( !isset( $result[$j]['refunds'] ) ){
														$result[$j]['refunds'] = 0;
													}
												}
											}
										}
									}
									$axis_headers = array("total_tax","avg_order_value","item_order","total_sale","refunds");
									
									$processed_row = array();
									
									foreach ($axis_headers as $header_key => $header_value) {
										if($header_value == 'currency'){
											continue ;
										}
										$i = 0;
										$processed_row["Section"] = $header_value;
										while($i < count($result_currencies) && $i < count($result)){
											$checked_value = isset( $result[$i][$header_value] ) && $result[$i][$header_value] != "" ? floatval($result[$i][$header_value]) : 0;
											
											$processed_row[$result_currencies[$i][0]] = $checked_value;

											$i++;
										}
										$processed_result[] = $processed_row;
									}
								}
								break;
						default:
							break;
					}
				}
			}
			return array(
				'message' => $message,
				'result' => $processed_result,
				'query' => $sql
			);
		}
    }
}

if( ! function_exists( 'CBFunctions' ) ){
	/**
	 * Function for quick access to Chart_Builder_Functions class
	 *
	 * @since   1.0.0
	 * @return  Chart_Builder_Functions
	 */
	function CBFunctions(){

        static $instance = null;

        if( $instance === null ){
            $instance = Chart_Builder_Functions::get_instance( CHART_BUILDER_NAME );
        }

        if( $instance instanceof Chart_Builder_Functions ){
	        return $instance;
        }

		return Chart_Builder_Functions::get_instance( CHART_BUILDER_NAME );
	}
}