<?php
/**
 * The Event Link To plugin core class
 * 
 * @package Events_Link_To
 * @subpackage Events_Link_To/includes
 * @since 1.0.0
 */

class Plugin_Events_Link_To {
	var $LINK_ONOFF = 'rhcelt_events_link_onoff';
	var $LINK_URL = 'rhcelt_events_link_url';

	/**
	 * Status
	 * 
	 * @since 1.0.0
	 * @access private
	 * @var string
	 */
	private $version = '';

	/**
	 * Define the core functionality.
	 * 
	 * @since 1.0.0
	 */
	public function __construct()
	{
		add_action( 'plugins_loaded', array( $this, 'load_dependencies' ) );
		add_action( 'init', array( $this, 'init_rhc_metaboxes' ), 999 );
		add_action( 'post_type_link', array( $this, 'set_events_permalink' ), 10, 2 );
	}

	/**
	 * Load dependencies.
	 * 
	 * @since 1.0.0
	 */
	public function load_dependencies()
	{
		require_once RHCELT_PATH . 'includes/class-events-link-to-metabox.php';

		$this->textdomain();
	}

	/**
	 * Load textdomain.
	 * 
	 * @since 1.0.0
	 * @access private
	 */
	private function textdomain()
	{
		load_plugin_textdomain( 'rhcelt', false, RHCELT_PATH . 'languages/' );
	}

	/**
	 * Add Meta Boxes.
	 * 
	 * @since 1.0.0
	 */
	public function init_rhc_metaboxes()
	{
		if ( is_admin() ) {
			new rhcelt_events_link_to_metabox();
		}
	}

	/**
	 * Set Events permalink.
	 * 
	 * @since 1.0.0
	 * 
	 * @param string  $link The post's permalink.
	 * @param WP_Post $post The post in question.
	 * @return string $link The post's new permalink.
	 */
	public function set_events_permalink( $link, $post )
	{
		$enabled = $this->get_link_status( $post->ID );

		if ( $enabled ) {
			$meta_link = $this->get_link_url( $post->ID );

			if ( $meta_link ) {
				$link = esc_url( $meta_link );
			}
		}

		return $link;
	}

	/**
	 * Get Post Meta.
	 * 
	 * @since 1.0.0
	 * @access private
	 * 
	 * @param int $post_id Post ID.
	 * @param string $key Meta key.
	 * @return string|false The post meta, or false, if doesn't exist.
	 */
	private function get_post_meta( $post_id, $key )
	{
		if ( empty( $post_id ) || empty( $key ) ) return false;

		$post_id = (int) $post_id;
		$meta = get_post_meta( $post_id, $key, true );

		if ( $meta === '' ) return false;

		return $meta;
	}

	/**
	 * Get Link status.
	 * 
	 * @since 1.0.0
	 * @access private
	 * 
	 * @param int $post_id Post ID.
	 * @return int|false Link status or false on failure.
	 */
	private function get_link_status( $post_id )
	{
		if ( empty( $post_id ) ) return false;

		$post_id = (int) $post_id;
		$onoff = $this->get_post_meta( $post_id, $this->LINK_ONOFF );

		if ( $onoff === '' ) return false;

		return $onoff;
	}

	/**
	 * Get Link URL.
	 * 
	 * @since 1.0.0
	 * @access private
	 * 
	 * @param int $post_id Post ID.
	 * @return string|false Link url or false on failure.
	 */
	private function get_link_url( $post_id )
	{
		if ( empty( $post_id ) ) return false;

		$post_id = (int) $post_id;
		$url = $this->get_post_meta( $post_id, $this->LINK_URL );

		if ( $url === '' ) return false;

		return $url;
	}

	/**
	 * Get Link Target
	 * 
	 * @since 1.0.0
	 * @access private
	 * 
	 * @param int $post_id Post ID.
	 * @return int|false Open the link in a new tab or not, and false on failure.
	 */
	private function get_link_target( $post_id )
	{
		if ( empty( $post_id ) ) return false;

		return $this->get_post_meta( $post_id, $this->LINK_NEWTAB );
	}
}

?>