<?php
/**
 * Event Taxonomy Counter shortcodes
 * 
 * @package Event_Taxnonomy_Counter
 * @subpackage Event_Taxnonomy_Counter/includes
 * @since 1.0.0
 */

class Event_Taxnonomy_Counter_Shortcodes {
	/**
	 * Define the core functionality of the shortcodes.
	 * 
	 * @since 1.0.0
	 */
	public function __construct()
	{
		add_shortcode( 'rhc_event_taxonomy_counter', array( $this, 'handle_event_taxonomy_counter' ) );
	}

	/**
	 * Handle the Event Taxonomy Counter shortcode
	 * 
	 * @since 1.0.0
	 */
	public function handle_event_taxonomy_counter( $atts, $content = null )
	{
		global $rhcetc_framework;
		$wrapper_class = 'widget_event-taxonomy-counter';
		$container_class = 'taxonomy-counter-container';
		$modifier = 'etc'; // Event Taxonomy Counter

		extract( shortcode_atts( array(
				'post_type'   => '',
				'label_align' => 'left',
				'overlay'     => '',
				'icon'        => '',
				'calendar'    => '',
				'venue'       => '',
				'organizer'   => '',
				'taxonomy'    => '',
				'term'        => '',
				'upcoming'    => '',
			), $atts )
		);

		$html = '';

		if ( ! empty( $post_type ) ) {
			$post_type_obj = get_post_type_object( $post_type );
			$posts_count = wp_count_posts( $post_type );
			$post_type_label = ( $posts_count->publish > 1 ) ? $post_type_obj->labels->name : $post_type_obj->labels->singular_name;
			
			$html .= $this->get_event_counter_html( $post_type, $post_type_label, $posts_count->publish, $label_align, $icon, $overlay );
		}

		if ( ! empty( $calendar ) || ! empty( $venue) || ! empty( $organizer ) ) {
			$wrapper_class = 'widget_taxonomy-term-counter';
			$container_class = 'term-counter-container';
			$modifier = 'ttc-counter'; // Taxonomy Terms Counter

			if ( ! empty( $calendar ) && empty( $venue ) && empty( $organizer ) ) {
				$term_obj = get_term_by( 'slug', $calendar, RHC_CALENDAR );
				$term_count = $rhcetc_framework->count_terms_relation( RHC_CALENDAR, $calendar, $taxonomy, $term );

				$featured_content = $rhcetc_framework->get_featured_content_html( $term_obj->name );
				$html = $rhcetc_framework->get_counter_container_html( $featured_content, $term_count, $calendar, $container_class );

				$modifier .= ' ' . $rhcetc_framework->get_taxonomy_prefix( RHC_CALENDAR );
			}

			if ( ! empty( $venue ) && empty( $calendar ) && empty( $organizer ) ) {
				$term_obj = get_term_by( 'slug', $venue, RHC_VENUE );
				$term_count = $rhcetc_framework->count_terms_relation( RHC_VENUE, $venue, $taxonomy, $term );

				$featured_content = $rhcetc_framework->get_featured_content_html( $term_obj->name );
				$html = $rhcetc_framework->get_counter_container_html( $featured_content, $term_count, $venue, $container_class );

				$modifier .= ' ' . $rhcetc_framework->get_taxonomy_prefix( RHC_VENUE );
			}

			if ( ! empty( $organizer ) && empty( $venue ) && empty( $calendar ) ) {
				$term_obj = get_term_by( 'slug', $organizer, RHC_ORGANIZER );
				$term_count = $rhcetc_framework->count_terms_relation( RHC_ORGANIZER, $organizer, $taxonomy, $term );

				$featured_content = $rhcetc_framework->get_featured_content_html( $term_obj->name );
				$html = $rhcetc_framework->get_counter_container_html( $featured_content, $term_count, $organizer, $container_class );

				$modifier .= ' ' .$rhcetc_framework->get_taxonomy_prefix( RHC_ORGANIZER );
			}

		} elseif ( ! empty( $taxonomy ) ) {
			if ( $taxonomy === 'all' ) {
				$taxonomies = get_object_taxonomies( RHC_EVENTS );
			} else {
				$taxonomies = array_map( 'trim', explode( ',', $taxonomy ) );
			}

			foreach ( $taxonomies as $taxonomy ) {
				if ( taxonomy_exists( $taxonomy ) ) {
					$taxonomy_obj = get_taxonomy( $taxonomy );
					$terms_count = wp_count_terms( $taxonomy );
					$taxonomy_label = ( $terms_count > 1 ) ? $taxonomy_obj->labels->name : $taxonomy_obj->labels->singular_name;
					
					$html .= $this->get_event_counter_html( $taxonomy, $taxonomy_label, $terms_count, $label_align, $icon, $overlay );
				}
			}
		}

		if ( ! empty( $overlay ) ) {
			return $this->get_overlay_wrapper_html( $html, $content );
		}

		// Output styles.
		if ( ! did_action( 'event_taxonomy_styles' ) ) do_action( 'event_taxonomy_styles' );
		if ( ! did_action( 'calendar_terms_styles' ) ) do_action( 'calendar_terms_styles' );
		if ( ! did_action( 'venue_terms_styles' ) ) do_action( 'venue_terms_styles' );
		if ( ! did_action( 'organizer_terms_styles' ) ) do_action( 'organizer_terms_styles' );

		wp_enqueue_style( 'rhc-etc-animate' );
		wp_enqueue_style( 'tooltipster' );
		wp_enqueue_style( 'rhc-etc-style' );

		wp_enqueue_script( 'tooltipster' );
		wp_enqueue_script( 'rhc-etc-script' );

		return $rhcetc_framework->get_counter_box_html( $html, $wrapper_class, "$modifier-counter" );
	}

	/**
	 * Get HTML for the Event Taxonomy.
	 * 
	 * @param string $label Event Taxonomy Name.
	 * @param int|string $number Posts/Terms count.
	 * @param string $label_align Label alignment.
	 * 
	 * @since 1.0.0
	 * @access private
	 * 
	 * @return string $html HTML output
	 */
	private function get_event_counter_html( $name, $label, $number, $label_align, $icon, $overlay = null )
	{
		global $rhc_plugin, $rhcetc_framework;
		$html = '';
		$styles = '';

		if ( ! empty ( $overlay ) ) {
			$html .= '<div class="map-view-counter-content label-' . $label_align . '">';

			if ( $label_align !== 'right' ) {
				$html .= '<div class="label">' . $label . '</div>';
				$html .= '<div class="number">' . $number . '</div>';
			} else {
				$html .= '<div class="number">' . $number . '</div>';
				$html .= '<div class="label">' . $label . '</div>';
			}

			$html .= '</div>';
			} else {
				$has_icon = (int) $rhc_plugin->get_option( 'rhcetc-' . $name . '-icon', '0', true );

				if ( ! $icon || ! $has_icon ) {
					$featured_content = '<div class="label">' . $label . '</div>';
				} else {
					$featured_content = '<div class="icon"><i class="icon-' . $name . '"></i></div>';
				}

				// $html .= '<div class="rhcetc-counter-container taxonomy-counter-container ' . $name . '">';
				// $html .= '<div class="featured column">' . $featured_content . '</div>';
				// $html .= '<div class="number column">' . $number . '</div>';
				// $html .= '</div>';

				$title_attr = sprintf( __( '%s', 'rhcetc' ) , $label );
				$html .= $rhcetc_framework->get_counter_container_html( $featured_content, $number, $name, $title_attr );
		}

		return $html;
	}

	/**
	 * Get HTML overlay wrapper
	 * 
	 * @param string $content Content to wrap
	 * @param string $shortcode Shortcode content
	 * 
	 * @since 1.0.0
	 * @access private
	 * 
	 * @return string $overlay Overlay wrapped HTML content
	 */
	private function get_overlay_wrapper_html( $content, $shortcode )
	{
		global $rhc_plugin;
		$overlay = '';
		
		$timer_duration = (int) $rhc_plugin->get_option( 'rhcmvc-timer-duration', '15', true );
		$disappear_disable = (int) $rhc_plugin->get_option( 'rhcmvc-disappear-disable', '', true );
		$timer_animation_in = $rhc_plugin->get_option( 'rhcmvc-timer-animation-in', 'rhcetc-fadeIn', true );

		if ( ! $disappear_disable ) {
			$timer_animation_out = $rhc_plugin->get_option( 'rhcmvc-timer-animation-out', 'rhcetc-fadeOut', true );
		} else {
			$timer_animation_out = '';
		}
		
		$mobile_display = (int) $rhc_plugin->get_option( 'rhcmvc-mobile-display', '1', true );
		$mobile_disable = ( ! $mobile_display ) ? 'mvc-mobile-disalbed' : '';

		$overlay .= '<div class="rhcetc-map-view-counter">';
		$overlay .= '<div class="map-view-counter-box">';
		$overlay .= "<div class='fe-extrainfo-container mvc-container overlay rhcetc-animated $mobile_disable' data-timer='$timer_duration' data-appear='$timer_animation_in' data-disappear='$timer_animation_out'>$content</div>";
		$overlay .= do_shortcode( $shortcode );
		$overlay .= '</div>';
		$overlay .= '</div>';

		return $overlay;

	}
}


?>