<?php
/**
 * The plugin core class
 * 
 * @package Event_Taxnonomy_Counter
 * @subpackage Event_Taxnonomy_Counter/includes
 * @since 1.0.0
 */
class Plugin_Event_Taxonomy_Counter {
	/**
	 * The unique instance of the plugin.
	 *
	 * @since 	1.0.0
	 * @access 	private
	 * @var 	Plugin_Event_Taxonomy_Counter $instance
	 */
	private static $instance;

	/**
	 * Get an instance of our plugin.
	 *
	 * @since 	1.0.0
	 * @return 	Plugin_Event_Taxonomy_Counter An instance of the class.
	 */
	public static function getInstance()
	{
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	/**
	 * Define the core functionality of the plugin.
	 *
	 * @since 	1.0.0
	 * @access 	private
	 */
	private function __construct()
	{
		add_action( 'plugins_loaded', array( $this, 'load_dependencies' ) );
		add_action( 'widgets_init', array( $this, 'register_widgets' ) );
		//--- scripts and styles
		add_action( 'wp_enqueue_scripts', array( $this, 'admin_and_frontend_scripts') );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_and_frontend_scripts') );
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_only_scripts' ), 11 );
	}

	/**
	 * Load plugin dependencies.
	 *
	 * @since 	1.0.0
	 */
	public function load_dependencies()
	{
		require_once RHCETC_PATH . 'includes/class-event-taxonomy-counter-widget.php';
		require_once RHCETC_PATH . 'includes/class-event-taxonomy-counter-shortcodes.php';
		require_once RHCETC_PATH . 'includes/class-calendar-term-counter-widget.php';
		require_once RHCETC_PATH . 'includes/class-rhcetc-options.php';
		require_once RHCETC_PATH . 'includes/class-rhcetc-framework.php';

		require_once RHCETC_PATH . 'includes/class.rhc_etc_dashboard_widget.php';
		new rhc_etc_dashboard_widget();
		
		require_once RHC_PATH    . 'options-panel/class.PluginOptionsPanelModule.php';

		global $rhcetc_framework;
		$rhcetc_framework = new rhcetc_framework();

		$this->textdomain();
		$this->options();
		$this->shortcodes();
		$this->actions();
	}

	/**
	 * Register plugin widgets.
	 *
	 * @since 1.0.0
	 */
	public function register_widgets()
	{
		register_widget( 'Event_Taxonomy_Counter_Widget' );
		register_widget( 'Event_Calendar_Term_Widget' );
	}

	/**
	 * Load textdomain.
	 * 
	 * @since 1.0.0
	 * @access private
	 */
	private function textdomain()
	{
		load_plugin_textdomain( 'rhcetc', false, RHCETC_PATH . 'languages/' );
	}

	/**
	 * Admin and frontend styles and scripts.
	 *
	 * @since 1.0.0
	 */
	public function admin_and_frontend_scripts()
	{
		wp_register_style( 'rhc-etc-style', RHCETC_URL . 'css/rhc_etc_styles.css', array(), RHCETC_VERSION );
		wp_register_script( 'rhc-etc-script', RHCETC_URL . 'js/rhc_etc_main.js', array( 'jquery' ), RHCETC_VERSION, true );
	}

	/**
	 * Frontend styles and scripts.
	 *
	 * @since 1.0.0
	 */
	public function frontend_only_scripts()
	{
		wp_register_style( 'rhc-etc-animate', RHCETC_URL . 'css/animate.css' );
	}

	/**
	 * Load options.
	 * 
	 * @since 1.0.0
	 * @access private
	 * 
	 * @global object $rhc_plugin Options object.
	 */
	 private function options()
	 {
	 	global $rhc_plugin;

	 	$rhc_options = new rhcetc_options( 'rhcetc', 'manage_options', false );

		$settings = array(				
			'id'                     => 'rhcetc',
			'plugin_id'              => 'rhc',
			'capability'             => 'manage_options',
			'options_varname'        => $rhc_plugin->options_varname,
			'menu_id'                => 'rhc-cv-options',
			'page_title'             => __( 'Taxonomy Counter', 'rhccv' ),
			'menu_text'              => __( 'Taxonomy Counter', 'rhccv' ),
			'option_menu_parent'     => 'edit.php?post_type=' . RHC_EVENTS,
			'notification'           => (object) array(
				'plugin_version'         => RHC_VERSION,
				'plugin_code'            => 'RHC',
				'message'                => __( 'Calendar plugin update %s is available! <a href="%s">Please update now</a>', 'rhc' )
			),
			'theme'                  => false,
			'stylesheet'             => 'rhc-options',
			'extracss'               => 'rhccv-pop',
			'option_show_in_metabox' => true,
			'path'                   => RHC_PATH . 'options-panel/',
			'url'                    => RHC_URL . 'options-panel/',
			'pluginslug'             => RHC_SLUG,
			'api_url'                => 'http://plugins.righthere.com',
			'layout'                 => 'horizontal',
		);

		$plugin_options_panel_module = new PluginOptionsPanelModule( $settings );
	}

	/**
	 * Load actions.
	 * 
	 * @since 1.0.0
	 * @access private
	 */
	private function actions()
	{
		add_action( 'event_taxonomy_styles', array( $this, 'event_taxonomy_styles' ) );
		add_action( 'calendar_terms_styles', array( $this, 'calendar_terms_styles' ) );
		add_action( 'venue_terms_styles', array( $this, 'venue_terms_styles' ) );
		add_action( 'organizer_terms_styles', array( $this, 'organizer_terms_styles' ) );
	} 

	/**
	 * Load shortcodes.
	 * 
	 * @since 1.0.0
	 * @access private
	 */
	private function shortcodes()
	{
		$event_taxonomy_counter_shortcdes = new Event_Taxnonomy_Counter_Shortcodes();
	}

	/**
	 * Event Taxonomy styles.
	 * 
	 * @since 1.0.3
	 */
	public function event_taxonomy_styles()
	{
		$taxonomies = get_object_taxonomies( RHC_EVENTS );
		$styles = '';

		$this->taxonomy_styles( RHC_EVENTS );

		foreach ( $taxonomies as $taxonomy ) {
			$this->taxonomy_styles( $taxonomy );
		}

		echo "<style>$styles</style>";
	}

	/**
	 * Calendar terms styles.
	 * 
	 * @since 1.0.3
	 */
	public function calendar_terms_styles() {
		$this->terms_styles( RHC_CALENDAR );
	}
	
	/**
	 * Venue terms styles.
	 * 
	 * @since 1.0.3
	 */
	public function venue_terms_styles() {
		$this->terms_styles( RHC_VENUE );
	}

	/**
	 * Organizer terms styles.
	 * 
	 * @since 1.0.3
	 */
	public function organizer_terms_styles() {
		$this->terms_styles( RHC_ORGANIZER );
	}

	/**
	 * Taxonomy styles.
	 * 
	 * @since 1.0.3
	 */
	private function taxonomy_styles( $slug, $prefix = 'etc' )
	{
		global $rhc_plugin;
		$styles = '';

		$color_bg = $rhc_plugin->get_option( 'rhcetc-' . $slug . '-color', '', true );
		$styles .= '.' . $prefix . '-counter .' . $slug . ' {';
		$styles .= ( strlen( $color_bg ) > 1 ) ? 'background-color:' . $color_bg . ';': '';
		$styles .= '}';

		$color_txt = $rhc_plugin->get_option( 'rhcetc-' . $slug . '-text-color', '', true );
		$styles .= '.' . $prefix . '-counter .' . $slug . ' .column {';
		$styles .= ( strlen( $color_txt ) > 1 ) ? 'color:' . $color_txt . ';' : '';
		$styles .= '}';

		$color_icon = $rhc_plugin->get_option( 'rhcetc-' . $slug . '-icon-color', '', true );
		$styles .= '.' . $prefix . '-counter .' . $slug . ' .featured .icon {';
		$styles .= ( strlen( $color_icon ) > 1 ) ? 'color:' . $color_icon . ';' : '';
		$styles .= '}';

		echo "<style>$styles</style>";
	}

	/**
	 * Terms styles.
	 * 
	 * @param $taxonomy string Taxonomy name
	 * 
	 * @since 1.0.3
	 * @access private
	 */
	private function terms_styles( $taxonomy )
	{
		global $rhc_plugin;
		global $rhcetc_framework;

		$styles = '';
		$terms = get_terms( $taxonomy );

		if ( $terms ) {
			$prefix = $rhcetc_framework->get_taxonomy_prefix( $taxonomy );

			foreach ( $terms as $term) {
				$color_bg = $rhc_plugin->get_option( 'rhc' . $prefix . '-' . $term->slug . '-color', '', true );
				$styles .= '.' . $prefix . '-counter .' . $term->slug . ' {';
				$styles .= ( strlen( $color_bg ) > 1 ) ? 'background-color:' . $color_bg . ';': '';
				$styles .= '}';

				$color_txt = $rhc_plugin->get_option( 'rhc' . $prefix . '-' . $term->slug . '-text-color', '', true );
				$styles .= '.' . $prefix . '-counter .' . $term->slug . ' .column {';
				$styles .= ( strlen( $color_txt ) > 1 ) ? 'color:' . $color_txt . ';' : '';
				$styles .= '}';
			}
		}

		echo "<style>$styles</style>";
	}

}

?>