<?php
/*
Plugin Name: Buttons X - Tooltip
Plugin URI: https://www.button.sx
Description: Tooltip extension for Buttons X.
Version: 1.2
Author: Gautam Thapar
Author URI: http://gautamthapar.me/
License: Envato Split License
Text Domain: btnsx-tooltip
Domain Path: /languages
*/

// Make sure we don't expose any info if called directly
if ( !defined( 'ABSPATH' ) )
	exit;

define( 'BTNSX__TOOLTIP_VERSION', '1.2' );
define( 'BTNSX__TOOLTIP_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'BTNSX__TOOLTIP_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );

add_action( 'plugins_loaded', 'btnsx_tooltip_check_main_class' );

function btnsx_tooltip_check_main_class() {
	if( class_exists( 'Btnsx' )  ) {
		add_action( 'plugins_loaded', 'btnsx_tooltip_textdomain' );
		add_filter( 'btnsx_options_filter', 'btnsx_tooltip_options' );
		add_filter( 'btnsx_save_data_filter', 'btnsx_tooltip_data' );
		add_filter( 'btnsx_shortcode_attributes', 'btnsx_tooltip_button_shortcode_attributes' );
		add_filter( 'btnsx_output_button_attributes_filter', 'btnsx_tooltip_output', 10, 3 );
		add_action( 'wp_enqueue_scripts', 'btnsx_tooltip_public_scripts' );
		add_action( 'admin_enqueue_scripts', 'btnsx_tooltip_admin_scripts' );
	} else {
		add_action( 'admin_notices', 'btnsx_tooltip_dependency_notice' );
	}
}

/**
 * IMPORTANT :: We are using the same tooltip used by Buttons X so we do not require to load CSS and JS files.
 */

/**
 * Load plugin textdomain
 */
function btnsx_tooltip_textdomain() {
	load_plugin_textdomain( 'btnsx-tooltip', false, BTNSX__TOOLTIP_PLUGIN_URL . 'languages' );
}

/**
 * Buttons X - Options Filter
 */
function btnsx_tooltip_options( array $options ) {
	$meta = new Btnsx();
	global $post;
	$options[] = array(
		'icon_class'	=>	'fa fa-question-circle',
		'text'			=>	__( 'Tooltip', 'btnsx-tooltip' ),
		'elements'		=>	array(
		    array(
			    'type'			=>	'text',
			    'id'			=>	'btnsx_opt_tooltip_text',
			    'name'			=>	'btnsx_opt_tooltip_text',
			    'label'			=>	__( 'Tooltip Text', 'btnsx-tooltip' ),
			    'placeholder'	=>	' ',
			    'tooltip'		=>	__( 'Insert text for tooltip. HTML is allowed.', 'btnsx-tooltip' ),
			    'value' 		=> $meta->meta_values( $post->ID, array( 'field' => 'btnsx_tooltip', 'field2' => 'text' ) )
		    ),
		    array(
			    'type'			=>	'select',
			    'id'			=>	'btnsx_opt_tooltip_animation',
			    'name'			=>	'btnsx_opt_tooltip_animation',
			    'label'			=>	__( 'Tooltip Animation', 'btnsx-tooltip' ),
			    'options'		=>	array(
			    	'fade'			=>	__( 'Fade', 'btnsx-tooltip' ),
			    	'fall'			=>	__( 'Fall', 'btnsx-tooltip' ),
			    	'grow'			=>	__( 'Grow', 'btnsx-tooltip' ),
			    	'slide'			=>	__( 'Slide', 'btnsx-tooltip' ),
			    	'swing'			=>	__( 'Swing', 'btnsx-tooltip' ),
			    ),
			    'tooltip'		=>	__( 'Select tooltip animation type.', 'btnsx-tooltip' ),
			    'value' 		=> $meta->meta_values( $post->ID, array( 'field' => 'btnsx_tooltip', 'field2' => 'animation' ) )
		    ),
		    array(
			    'type'			=>	'select',
			    'id'			=>	'btnsx_opt_tooltip_position',
			    'name'			=>	'btnsx_opt_tooltip_position',
			    'label'			=>	__( 'Tooltip Position', 'btnsx-tooltip' ),
			    'options'		=>	array(
			    	'bottom'		=>	__( 'Bottom', 'btnsx-tooltip' ),
			    	'bottom-left'	=>	__( 'Bottom Left', 'btnsx-tooltip' ),
			    	'bottom-right'	=>	__( 'Bottom Right', 'btnsx-tooltip' ),
			    	'left'			=>	__( 'Left', 'btnsx-tooltip' ),
			    	'right'			=>	__( 'Right', 'btnsx-tooltip' ),
			    	'top'			=>	__( 'Top', 'btnsx-tooltip' ),
			    	'top-left'		=>	__( 'Top Left', 'btnsx-tooltip' ),
			    	'top-right'		=>	__( 'Top Right', 'btnsx-tooltip' ),
			    ),
			    'tooltip'		=>	__( 'Select tooltip position.', 'btnsx-tooltip' ),
			    'value' 		=> $meta->meta_values( $post->ID, array( 'field' => 'btnsx_tooltip', 'field2' => 'position', 'value' => 'top' ) )
		    ),
		    array(
			    'type'			=>	'select',
			    'id'			=>	'btnsx_opt_tooltip_theme',
			    'name'			=>	'btnsx_opt_tooltip_theme',
			    'label'			=>	__( 'Tooltip Theme', 'btnsx-tooltip' ),
			    'options'		=>	array(
			    	'tooltipster-default'	=>	__( 'Default', 'btnsx-tooltip' ),
			    	'tooltipster-light'		=>	__( 'Light', 'btnsx-tooltip' ),
			    	'tooltipster-noir'		=>	__( 'Noir', 'btnsx-tooltip' ),
			    	'tooltipster-punk'		=>	__( 'Punk', 'btnsx-tooltip' ),
			    	'tooltipster-shadow'	=>	__( 'Shadow', 'btnsx-tooltip' ),
			    	'tooltipster-btnsx'		=>	__( 'Buttons X', 'btnsx-tooltip' ),
			    ),
			    'tooltip'		=>	__( 'Select tooltip theme style.', 'btnsx-tooltip' ),
			    'value' 		=> $meta->meta_values( $post->ID, array( 'field' => 'btnsx_tooltip', 'field2' => 'theme', 'value' => 'tooltipster-light' ) )
		    ),
		)
	);

	return $options;
}

/**
 * Buttons X - Save Data Filter
 */
function btnsx_tooltip_data( array $data ) {
	
	$data['btnsx_tooltip'] = array(
		'text' 								=> sanitize_text_field( isset( $_POST['btnsx_opt_tooltip_text'] ) ? esc_html( $_POST['btnsx_opt_tooltip_text'] ) : '' ),
		'animation' 						=> sanitize_text_field( isset( $_POST['btnsx_opt_tooltip_animation'] ) ? $_POST['btnsx_opt_tooltip_animation'] : '' ),
		'position' 							=> sanitize_text_field( isset( $_POST['btnsx_opt_tooltip_position'] ) ? $_POST['btnsx_opt_tooltip_position'] : '' ),
		'theme' 							=> sanitize_text_field( isset( $_POST['btnsx_opt_tooltip_theme'] ) ? $_POST['btnsx_opt_tooltip_theme'] : '' ),
	);

	return $data;
}

/**
 * Buttons X - Button Short Code Attributes
 */
function btnsx_tooltip_button_shortcode_attributes( array $atts ) {
	
	$atts[] = array(
		'tooltip_text'								=>	'',
		'tooltip_animation'							=>	'',
		'tooltip_position'							=>	'',
		'tooltip_theme'								=>	''
	);

	return $atts;
}

/**
 * Buttons X - Button Attributes Filter
 */
function btnsx_tooltip_output( array $data, $options, $atts ) {
	
	extract( shortcode_atts( $options, $atts ) );
	
	$meta = get_post_meta( $id, 'btnsx', true );

	if( $tooltip_text === '' ){
		$tooltip_text = isset( $meta['btnsx_tooltip']['text'] ) ? $meta['btnsx_tooltip']['text'] : '';
	}
	if( $tooltip_animation === '' ){
		$tooltip_animation = isset( $meta['btnsx_tooltip']['animation'] ) ? $meta['btnsx_tooltip']['animation'] : '';
	}
	if( $tooltip_position === '' ){
		$tooltip_position = isset( $meta['btnsx_tooltip']['position'] ) ? $meta['btnsx_tooltip']['position'] : '';
	}
	if( $tooltip_theme === '' ){
		$tooltip_theme = isset( $meta['btnsx_tooltip']['theme'] ) ? $meta['btnsx_tooltip']['theme'] : '';
	}
	
	if( $tooltip_text != '' ){
		$data[] = array(
			'data-tooltip-text="' . $tooltip_text . '"',
			'data-tooltip-animation="' . $tooltip_animation . '"',
			'data-tooltip-position="' . $tooltip_position . '"',
			'data-tooltip-theme="' . $tooltip_theme . '"',
		);
	}

	return $data;
}

/**
 * Public Enqueue Scripts
 */
function btnsx_tooltip_public_scripts() {
	wp_enqueue_script(
		'tooltip',
		BTNSX__TOOLTIP_PLUGIN_URL . 'assets/js/tooltip-public.js',
		array( 'jquery' )
	);
	wp_enqueue_style( 'tooltip', BTNSX__TOOLTIP_PLUGIN_URL . 'assets/css/tooltip.min.css', array(), '0.1', 'all' );
}

/**
 * Admin Enqueue Scripts
 */
function btnsx_tooltip_admin_scripts() {
	$screen = get_current_screen();
	if( $screen->id === 'edit-buttons-x' ){
		wp_enqueue_script(
			'tooltip',
			BTNSX__TOOLTIP_PLUGIN_URL . 'assets/js/tooltip-admin.js',
			array( 'jquery' )
		);
	}
}

/**
 * Buttons X dependency notice
 */
function btnsx_tooltip_dependency_notice() {
	$class = 'update-nag';
	$url = 'https://www.button.sx';
	$message = sprintf( __( '<strong>Buttons X Tooltips</strong> extension is not really useful without <strong><a target="_blank" href="%s">Buttons X</a></strong> plugin.', 'btnsx-tooltip' ), $url );
    echo '<div class="' . $class . '"><p>' . $message . '</p></div>'; 
}