/**
 * @preserve jQuery DateTimePicker
 * @homepage http://xdsoft.net/jqplugins/datetimepicker/
 * @author Chupurnov Valeriy (<chupurnov@gmail.com>)
 */

/**
 * @param {jQuery} $
 */
var datetimepickerFactory = function ($) {
	'use strict';

	var default_options = {
		i18n: {
			ar: { // Arabic
				months: [
					"كانون الثاني", "شباط", "آذار", "نيسان", "مايو", "حزيران", "تموز", "آب", "أيلول", "تشرين الأول", "تشرين الثاني", "كانون الأول"
				],
				dayOfWeekShort: [
					"ن", "ث", "ع", "خ", "ج", "س", "ح"
				],
				dayOfWeek: ["الأحد", "الاثنين", "الثلاثاء", "الأربعاء", "الخميس", "الجمعة", "السبت", "الأحد"]
			},
			ro: { // Romanian
				months: [
					"Ianuarie", "Februarie", "Martie", "Aprilie", "Mai", "Iunie", "Iulie", "August", "Septembrie", "Octombrie", "Noiembrie", "Decembrie"
				],
				dayOfWeekShort: [
					"Du", "Lu", "Ma", "Mi", "Jo", "Vi", "Sâ"
				],
				dayOfWeek: ["Duminică", "Luni", "Marţi", "Miercuri", "Joi", "Vineri", "Sâmbătă"]
			},
			id: { // Indonesian
				months: [
					"Januari", "Februari", "Maret", "April", "Mei", "Juni", "Juli", "Agustus", "September", "Oktober", "November", "Desember"
				],
				dayOfWeekShort: [
					"Min", "Sen", "Sel", "Rab", "Kam", "Jum", "Sab"
				],
				dayOfWeek: ["Minggu", "Senin", "Selasa", "Rabu", "Kamis", "Jumat", "Sabtu"]
			},
			is: { // Icelandic
				months: [
					"Janúar", "Febrúar", "Mars", "Apríl", "Maí", "Júní", "Júlí", "Ágúst", "September", "Október", "Nóvember", "Desember"
				],
				dayOfWeekShort: [
					"Sun", "Mán", "Þrið", "Mið", "Fim", "Fös", "Lau"
				],
				dayOfWeek: ["Sunnudagur", "Mánudagur", "Þriðjudagur", "Miðvikudagur", "Fimmtudagur", "Föstudagur", "Laugardagur"]
			},
			bg: { // Bulgarian
				months: [
					"Януари", "Февруари", "Март", "Април", "Май", "Юни", "Юли", "Август", "Септември", "Октомври", "Ноември", "Декември"
				],
				dayOfWeekShort: [
					"Нд", "Пн", "Вт", "Ср", "Чт", "Пт", "Сб"
				],
				dayOfWeek: ["Неделя", "Понеделник", "Вторник", "Сряда", "Четвъртък", "Петък", "Събота"]
			},
			fa: { // Persian/Farsi
				months: [
					'فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور', 'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند'
				],
				dayOfWeekShort: [
					'یکشنبه', 'دوشنبه', 'سه شنبه', 'چهارشنبه', 'پنجشنبه', 'جمعه', 'شنبه'
				],
				dayOfWeek: ["یک‌شنبه", "دوشنبه", "سه‌شنبه", "چهارشنبه", "پنج‌شنبه", "جمعه", "شنبه", "یک‌شنبه"]
			},
			ru: { // Russian
				months: [
					'Январь', 'Февраль', 'Март', 'Апрель', 'Май', 'Июнь', 'Июль', 'Август', 'Сентябрь', 'Октябрь', 'Ноябрь', 'Декабрь'
				],
				dayOfWeekShort: [
					"Вс", "Пн", "Вт", "Ср", "Чт", "Пт", "Сб"
				],
				dayOfWeek: ["Воскресенье", "Понедельник", "Вторник", "Среда", "Четверг", "Пятница", "Суббота"]
			},
			uk: { // Ukrainian
				months: [
					'Січень', 'Лютий', 'Березень', 'Квітень', 'Травень', 'Червень', 'Липень', 'Серпень', 'Вересень', 'Жовтень', 'Листопад', 'Грудень'
				],
				dayOfWeekShort: [
					"Ндл", "Пнд", "Втр", "Срд", "Чтв", "Птн", "Сбт"
				],
				dayOfWeek: ["Неділя", "Понеділок", "Вівторок", "Середа", "Четвер", "П'ятниця", "Субота"]
			},
			en: { // English
				months: [
					"January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"
				],
				dayOfWeekShort: [
					"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
				],
				dayOfWeek: ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"]
			},
			el: { // Ελληνικά
				months: [
					"Ιανουάριος", "Φεβρουάριος", "Μάρτιος", "Απρίλιος", "Μάιος", "Ιούνιος", "Ιούλιος", "Αύγουστος", "Σεπτέμβριος", "Οκτώβριος", "Νοέμβριος", "Δεκέμβριος"
				],
				dayOfWeekShort: [
					"Κυρ", "Δευ", "Τρι", "Τετ", "Πεμ", "Παρ", "Σαβ"
				],
				dayOfWeek: ["Κυριακή", "Δευτέρα", "Τρίτη", "Τετάρτη", "Πέμπτη", "Παρασκευή", "Σάββατο"]
			},
			de: { // German
				months: [
					'Januar', 'Februar', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember'
				],
				dayOfWeekShort: [
					"So", "Mo", "Di", "Mi", "Do", "Fr", "Sa"
				],
				dayOfWeek: ["Sonntag", "Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag"]
			},
			nl: { // Dutch
				months: [
					"januari", "februari", "maart", "april", "mei", "juni", "juli", "augustus", "september", "oktober", "november", "december"
				],
				dayOfWeekShort: [
					"zo", "ma", "di", "wo", "do", "vr", "za"
				],
				dayOfWeek: ["zondag", "maandag", "dinsdag", "woensdag", "donderdag", "vrijdag", "zaterdag"]
			},
			tr: { // Turkish
				months: [
					"Ocak", "Şubat", "Mart", "Nisan", "Mayıs", "Haziran", "Temmuz", "Ağustos", "Eylül", "Ekim", "Kasım", "Aralık"
				],
				dayOfWeekShort: [
					"Paz", "Pts", "Sal", "Çar", "Per", "Cum", "Cts"
				],
				dayOfWeek: ["Pazar", "Pazartesi", "Salı", "Çarşamba", "Perşembe", "Cuma", "Cumartesi"]
			},
			fr: { //French
				months: [
					"Janvier", "Février", "Mars", "Avril", "Mai", "Juin", "Juillet", "Août", "Septembre", "Octobre", "Novembre", "Décembre"
				],
				dayOfWeekShort: [
					"Dim", "Lun", "Mar", "Mer", "Jeu", "Ven", "Sam"
				],
				dayOfWeek: ["dimanche", "lundi", "mardi", "mercredi", "jeudi", "vendredi", "samedi"]
			},
			es: { // Spanish
				months: [
					"Enero", "Febrero", "Marzo", "Abril", "Mayo", "Junio", "Julio", "Agosto", "Septiembre", "Octubre", "Noviembre", "Diciembre"
				],
				dayOfWeekShort: [
					"Dom", "Lun", "Mar", "Mié", "Jue", "Vie", "Sáb"
				],
				dayOfWeek: ["Domingo", "Lunes", "Martes", "Miércoles", "Jueves", "Viernes", "Sábado"]
			},
			th: { // Thai
				months: [
					'มกราคม', 'กุมภาพันธ์', 'มีนาคม', 'เมษายน', 'พฤษภาคม', 'มิถุนายน', 'กรกฎาคม', 'สิงหาคม', 'กันยายน', 'ตุลาคม', 'พฤศจิกายน', 'ธันวาคม'
				],
				dayOfWeekShort: [
					'อา.', 'จ.', 'อ.', 'พ.', 'พฤ.', 'ศ.', 'ส.'
				],
				dayOfWeek: ["อาทิตย์", "จันทร์", "อังคาร", "พุธ", "พฤหัส", "ศุกร์", "เสาร์", "อาทิตย์"]
			},
			pl: { // Polish
				months: [
					"styczeń", "luty", "marzec", "kwiecień", "maj", "czerwiec", "lipiec", "sierpień", "wrzesień", "październik", "listopad", "grudzień"
				],
				dayOfWeekShort: [
					"nd", "pn", "wt", "śr", "cz", "pt", "sb"
				],
				dayOfWeek: ["niedziela", "poniedziałek", "wtorek", "środa", "czwartek", "piątek", "sobota"]
			},
			pt: { // Portuguese
				months: [
					"Janeiro", "Fevereiro", "Março", "Abril", "Maio", "Junho", "Julho", "Agosto", "Setembro", "Outubro", "Novembro", "Dezembro"
				],
				dayOfWeekShort: [
					"Dom", "Seg", "Ter", "Qua", "Qui", "Sex", "Sab"
				],
				dayOfWeek: ["Domingo", "Segunda", "Terça", "Quarta", "Quinta", "Sexta", "Sábado"]
			},
			ch: { // Simplified Chinese
				months: [
					"一月", "二月", "三月", "四月", "五月", "六月", "七月", "八月", "九月", "十月", "十一月", "十二月"
				],
				dayOfWeekShort: [
					"日", "一", "二", "三", "四", "五", "六"
				]
			},
			se: { // Swedish
				months: [
					"Januari", "Februari", "Mars", "April", "Maj", "Juni", "Juli", "Augusti", "September",  "Oktober", "November", "December"
				],
				dayOfWeekShort: [
					"Sön", "Mån", "Tis", "Ons", "Tor", "Fre", "Lör"
				]
			},
			km: { // Khmer (ភាសាខ្មែរ)
				months: [
					"មករា​", "កុម្ភៈ", "មិនា​", "មេសា​", "ឧសភា​", "មិថុនា​", "កក្កដា​", "សីហា​", "កញ្ញា​", "តុលា​", "វិច្ឆិកា", "ធ្នូ​"
				],
				dayOfWeekShort: ["អាទិ​", "ច័ន្ទ​", "អង្គារ​", "ពុធ​", "ព្រហ​​", "សុក្រ​", "សៅរ៍"],
				dayOfWeek: ["អាទិត្យ​", "ច័ន្ទ​", "អង្គារ​", "ពុធ​", "ព្រហស្បតិ៍​", "សុក្រ​", "សៅរ៍"]
			},
			kr: { // Korean
				months: [
					"1월", "2월", "3월", "4월", "5월", "6월", "7월", "8월", "9월", "10월", "11월", "12월"
				],
				dayOfWeekShort: [
					"일", "월", "화", "수", "목", "금", "토"
				],
				dayOfWeek: ["일요일", "월요일", "화요일", "수요일", "목요일", "금요일", "토요일"]
			},
			it: { // Italian
				months: [
					"Gennaio", "Febbraio", "Marzo", "Aprile", "Maggio", "Giugno", "Luglio", "Agosto", "Settembre", "Ottobre", "Novembre", "Dicembre"
				],
				dayOfWeekShort: [
					"Dom", "Lun", "Mar", "Mer", "Gio", "Ven", "Sab"
				],
				dayOfWeek: ["Domenica", "Lunedì", "Martedì", "Mercoledì", "Giovedì", "Venerdì", "Sabato"]
			},
			da: { // Dansk
				months: [
					"Januar", "Februar", "Marts", "April", "Maj", "Juni", "Juli", "August", "September", "Oktober", "November", "December"
				],
				dayOfWeekShort: [
					"Søn", "Man", "Tir", "Ons", "Tor", "Fre", "Lør"
				],
				dayOfWeek: ["søndag", "mandag", "tirsdag", "onsdag", "torsdag", "fredag", "lørdag"]
			},
			no: { // Norwegian
				months: [
					"Januar", "Februar", "Mars", "April", "Mai", "Juni", "Juli", "August", "September", "Oktober", "November", "Desember"
				],
				dayOfWeekShort: [
					"Søn", "Man", "Tir", "Ons", "Tor", "Fre", "Lør"
				],
				dayOfWeek: ['Søndag', 'Mandag', 'Tirsdag', 'Onsdag', 'Torsdag', 'Fredag', 'Lørdag']
			},
			ja: { // Japanese
				months: [
					"1月", "2月", "3月", "4月", "5月", "6月", "7月", "8月", "9月", "10月", "11月", "12月"
				],
				dayOfWeekShort: [
					"日", "月", "火", "水", "木", "金", "土"
				],
				dayOfWeek: ["日曜", "月曜", "火曜", "水曜", "木曜", "金曜", "土曜"]
			},
			vi: { // Vietnamese
				months: [
					"Tháng 1", "Tháng 2", "Tháng 3", "Tháng 4", "Tháng 5", "Tháng 6", "Tháng 7", "Tháng 8", "Tháng 9", "Tháng 10", "Tháng 11", "Tháng 12"
				],
				dayOfWeekShort: [
					"CN", "T2", "T3", "T4", "T5", "T6", "T7"
				],
				dayOfWeek: ["Chủ nhật", "Thứ hai", "Thứ ba", "Thứ tư", "Thứ năm", "Thứ sáu", "Thứ bảy"]
			},
			sl: { // Slovenščina
				months: [
					"Januar", "Februar", "Marec", "April", "Maj", "Junij", "Julij", "Avgust", "September", "Oktober", "November", "December"
				],
				dayOfWeekShort: [
					"Ned", "Pon", "Tor", "Sre", "Čet", "Pet", "Sob"
				],
				dayOfWeek: ["Nedelja", "Ponedeljek", "Torek", "Sreda", "Četrtek", "Petek", "Sobota"]
			},
			cs: { // Čeština
				months: [
					"Leden", "Únor", "Březen", "Duben", "Květen", "Červen", "Červenec", "Srpen", "Září", "Říjen", "Listopad", "Prosinec"
				],
				dayOfWeekShort: [
					"Ne", "Po", "Út", "St", "Čt", "Pá", "So"
				]
			},
			hu: { // Hungarian
				months: [
					"Január", "Február", "Március", "Április", "Május", "Június", "Július", "Augusztus", "Szeptember", "Október", "November", "December"
				],
				dayOfWeekShort: [
					"Va", "Hé", "Ke", "Sze", "Cs", "Pé", "Szo"
				],
				dayOfWeek: ["vasárnap", "hétfő", "kedd", "szerda", "csütörtök", "péntek", "szombat"]
			},
			az: { //Azerbaijanian (Azeri)
				months: [
					"Yanvar", "Fevral", "Mart", "Aprel", "May", "Iyun", "Iyul", "Avqust", "Sentyabr", "Oktyabr", "Noyabr", "Dekabr"
				],
				dayOfWeekShort: [
					"B", "Be", "Ça", "Ç", "Ca", "C", "Ş"
				],
				dayOfWeek: ["Bazar", "Bazar ertəsi", "Çərşənbə axşamı", "Çərşənbə", "Cümə axşamı", "Cümə", "Şənbə"]
			},
			bs: { //Bosanski
				months: [
					"Januar", "Februar", "Mart", "April", "Maj", "Jun", "Jul", "Avgust", "Septembar", "Oktobar", "Novembar", "Decembar"
				],
				dayOfWeekShort: [
					"Ned", "Pon", "Uto", "Sri", "Čet", "Pet", "Sub"
				],
				dayOfWeek: ["Nedjelja","Ponedjeljak", "Utorak", "Srijeda", "Četvrtak", "Petak", "Subota"]
			},
			ca: { //Català
				months: [
					"Gener", "Febrer", "Març", "Abril", "Maig", "Juny", "Juliol", "Agost", "Setembre", "Octubre", "Novembre", "Desembre"
				],
				dayOfWeekShort: [
					"Dg", "Dl", "Dt", "Dc", "Dj", "Dv", "Ds"
				],
				dayOfWeek: ["Diumenge", "Dilluns", "Dimarts", "Dimecres", "Dijous", "Divendres", "Dissabte"]
			},
			'en-GB': { //English (British)
				months: [
					"January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"
				],
				dayOfWeekShort: [
					"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
				],
				dayOfWeek: ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"]
			},
			et: { //"Eesti"
				months: [
					"Jaanuar", "Veebruar", "Märts", "Aprill", "Mai", "Juuni", "Juuli", "August", "September", "Oktoober", "November", "Detsember"
				],
				dayOfWeekShort: [
					"P", "E", "T", "K", "N", "R", "L"
				],
				dayOfWeek: ["Pühapäev", "Esmaspäev", "Teisipäev", "Kolmapäev", "Neljapäev", "Reede", "Laupäev"]
			},
			eu: { //Euskara
				months: [
					"Urtarrila", "Otsaila", "Martxoa", "Apirila", "Maiatza", "Ekaina", "Uztaila", "Abuztua", "Iraila", "Urria", "Azaroa", "Abendua"
				],
				dayOfWeekShort: [
					"Ig.", "Al.", "Ar.", "Az.", "Og.", "Or.", "La."
				],
				dayOfWeek: ['Igandea', 'Astelehena', 'Asteartea', 'Asteazkena', 'Osteguna', 'Ostirala', 'Larunbata']
			},
			fi: { //Finnish (Suomi)
				months: [
					"Tammikuu", "Helmikuu", "Maaliskuu", "Huhtikuu", "Toukokuu", "Kesäkuu", "Heinäkuu", "Elokuu", "Syyskuu", "Lokakuu", "Marraskuu", "Joulukuu"
				],
				dayOfWeekShort: [
					"Su", "Ma", "Ti", "Ke", "To", "Pe", "La"
				],
				dayOfWeek: ["sunnuntai", "maanantai", "tiistai", "keskiviikko", "torstai", "perjantai", "lauantai"]
			},
			gl: { //Galego
				months: [
					"Xan", "Feb", "Maz", "Abr", "Mai", "Xun", "Xul", "Ago", "Set", "Out", "Nov", "Dec"
				],
				dayOfWeekShort: [
					"Dom", "Lun", "Mar", "Mer", "Xov", "Ven", "Sab"
				],
				dayOfWeek: ["Domingo", "Luns", "Martes", "Mércores", "Xoves", "Venres", "Sábado"]
			},
			hr: { //Hrvatski
				months: [
					"Siječanj", "Veljača", "Ožujak", "Travanj", "Svibanj", "Lipanj", "Srpanj", "Kolovoz", "Rujan", "Listopad", "Studeni", "Prosinac"
				],
				dayOfWeekShort: [
					"Ned", "Pon", "Uto", "Sri", "Čet", "Pet", "Sub"
				],
				dayOfWeek: ["Nedjelja", "Ponedjeljak", "Utorak", "Srijeda", "Četvrtak", "Petak", "Subota"]
			},
			ko: { //Korean (한국어)
				months: [
					"1월", "2월", "3월", "4월", "5월", "6월", "7월", "8월", "9월", "10월", "11월", "12월"
				],
				dayOfWeekShort: [
					"일", "월", "화", "수", "목", "금", "토"
				],
				dayOfWeek: ["일요일", "월요일", "화요일", "수요일", "목요일", "금요일", "토요일"]
			},
			lt: { //Lithuanian (lietuvių)
				months: [
					"Sausio", "Vasario", "Kovo", "Balandžio", "Gegužės", "Birželio", "Liepos", "Rugpjūčio", "Rugsėjo", "Spalio", "Lapkričio", "Gruodžio"
				],
				dayOfWeekShort: [
					"Sek", "Pir", "Ant", "Tre", "Ket", "Pen", "Šeš"
				],
				dayOfWeek: ["Sekmadienis", "Pirmadienis", "Antradienis", "Trečiadienis", "Ketvirtadienis", "Penktadienis", "Šeštadienis"]
			},
			lv: { //Latvian (Latviešu)
				months: [
					"Janvāris", "Februāris", "Marts", "Aprīlis ", "Maijs", "Jūnijs", "Jūlijs", "Augusts", "Septembris", "Oktobris", "Novembris", "Decembris"
				],
				dayOfWeekShort: [
					"Sv", "Pr", "Ot", "Tr", "Ct", "Pk", "St"
				],
				dayOfWeek: ["Svētdiena", "Pirmdiena", "Otrdiena", "Trešdiena", "Ceturtdiena", "Piektdiena", "Sestdiena"]
			},
			mk: { //Macedonian (Македонски)
				months: [
					"јануари", "февруари", "март", "април", "мај", "јуни", "јули", "август", "септември", "октомври", "ноември", "декември"
				],
				dayOfWeekShort: [
					"нед", "пон", "вто", "сре", "чет", "пет", "саб"
				],
				dayOfWeek: ["Недела", "Понеделник", "Вторник", "Среда", "Четврток", "Петок", "Сабота"]
			},
			mn: { //Mongolian (Монгол)
				months: [
					"1-р сар", "2-р сар", "3-р сар", "4-р сар", "5-р сар", "6-р сар", "7-р сар", "8-р сар", "9-р сар", "10-р сар", "11-р сар", "12-р сар"
				],
				dayOfWeekShort: [
					"Дав", "Мяг", "Лха", "Пүр", "Бсн", "Бям", "Ням"
				],
				dayOfWeek: ["Даваа", "Мягмар", "Лхагва", "Пүрэв", "Баасан", "Бямба", "Ням"]
			},
			'pt-BR': { //Português(Brasil)
				months: [
					"Janeiro", "Fevereiro", "Março", "Abril", "Maio", "Junho", "Julho", "Agosto", "Setembro", "Outubro", "Novembro", "Dezembro"
				],
				dayOfWeekShort: [
					"Dom", "Seg", "Ter", "Qua", "Qui", "Sex", "Sáb"
				],
				dayOfWeek: ["Domingo", "Segunda", "Terça", "Quarta", "Quinta", "Sexta", "Sábado"]
			},
			sk: { //Slovenčina
				months: [
					"Január", "Február", "Marec", "Apríl", "Máj", "Jún", "Júl", "August", "September", "Október", "November", "December"
				],
				dayOfWeekShort: [
					"Ne", "Po", "Ut", "St", "Št", "Pi", "So"
				],
				dayOfWeek: ["Nedeľa", "Pondelok", "Utorok", "Streda", "Štvrtok", "Piatok", "Sobota"]
			},
			sq: { //Albanian (Shqip)
				months: [
					"Janar", "Shkurt", "Mars", "Prill", "Maj", "Qershor", "Korrik", "Gusht", "Shtator", "Tetor", "Nëntor", "Dhjetor"
				],
				dayOfWeekShort: [
					"Die", "Hën", "Mar", "Mër", "Enj", "Pre", "Shtu"
				],
				dayOfWeek: ["E Diel", "E Hënë", "E Martē", "E Mërkurë", "E Enjte", "E Premte", "E Shtunë"]
			},
			'sr-YU': { //Serbian (Srpski)
				months: [
					"Januar", "Februar", "Mart", "April", "Maj", "Jun", "Jul", "Avgust", "Septembar", "Oktobar", "Novembar", "Decembar"
				],
				dayOfWeekShort: [
					"Ned", "Pon", "Uto", "Sre", "čet", "Pet", "Sub"
				],
				dayOfWeek: ["Nedelja","Ponedeljak", "Utorak", "Sreda", "Četvrtak", "Petak", "Subota"]
			},
			sr: { //Serbian Cyrillic (Српски)
				months: [
					"јануар", "фебруар", "март", "април", "мај", "јун", "јул", "август", "септембар", "октобар", "новембар", "децембар"
				],
				dayOfWeekShort: [
					"нед", "пон", "уто", "сре", "чет", "пет", "суб"
				],
				dayOfWeek: ["Недеља","Понедељак", "Уторак", "Среда", "Четвртак", "Петак", "Субота"]
			},
			sv: { //Svenska
				months: [
					"Januari", "Februari", "Mars", "April", "Maj", "Juni", "Juli", "Augusti", "September", "Oktober", "November", "December"
				],
				dayOfWeekShort: [
					"Sön", "Mån", "Tis", "Ons", "Tor", "Fre", "Lör"
				],
				dayOfWeek: ["Söndag", "Måndag", "Tisdag", "Onsdag", "Torsdag", "Fredag", "Lördag"]
			},
			'zh-TW': { //Traditional Chinese (繁體中文)
				months: [
					"一月", "二月", "三月", "四月", "五月", "六月", "七月", "八月", "九月", "十月", "十一月", "十二月"
				],
				dayOfWeekShort: [
					"日", "一", "二", "三", "四", "五", "六"
				],
				dayOfWeek: ["星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六"]
			},
			zh: { //Simplified Chinese (简体中文)
				months: [
					"一月", "二月", "三月", "四月", "五月", "六月", "七月", "八月", "九月", "十月", "十一月", "十二月"
				],
				dayOfWeekShort: [
					"日", "一", "二", "三", "四", "五", "六"
				],
				dayOfWeek: ["星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六"]
			},
			ug:{ // Uyghur(ئۇيغۇرچە)
				months: [
					"1-ئاي","2-ئاي","3-ئاي","4-ئاي","5-ئاي","6-ئاي","7-ئاي","8-ئاي","9-ئاي","10-ئاي","11-ئاي","12-ئاي"
				],
				dayOfWeek: [
					"يەكشەنبە", "دۈشەنبە","سەيشەنبە","چارشەنبە","پەيشەنبە","جۈمە","شەنبە"
				]
			},
			he: { //Hebrew (עברית)
				months: [
					'ינואר', 'פברואר', 'מרץ', 'אפריל', 'מאי', 'יוני', 'יולי', 'אוגוסט', 'ספטמבר', 'אוקטובר', 'נובמבר', 'דצמבר'
				],
				dayOfWeekShort: [
					'א\'', 'ב\'', 'ג\'', 'ד\'', 'ה\'', 'ו\'', 'שבת'
				],
				dayOfWeek: ["ראשון", "שני", "שלישי", "רביעי", "חמישי", "שישי", "שבת", "ראשון"]
			},
			hy: { // Armenian
				months: [
					"Հունվար", "Փետրվար", "Մարտ", "Ապրիլ", "Մայիս", "Հունիս", "Հուլիս", "Օգոստոս", "Սեպտեմբեր", "Հոկտեմբեր", "Նոյեմբեր", "Դեկտեմբեր"
				],
				dayOfWeekShort: [
					"Կի", "Երկ", "Երք", "Չոր", "Հնգ", "Ուրբ", "Շբթ"
				],
				dayOfWeek: ["Կիրակի", "Երկուշաբթի", "Երեքշաբթի", "Չորեքշաբթի", "Հինգշաբթի", "Ուրբաթ", "Շաբաթ"]
			},
			kg: { // Kyrgyz
				months: [
					'Үчтүн айы', 'Бирдин айы', 'Жалган Куран', 'Чын Куран', 'Бугу', 'Кулжа', 'Теке', 'Баш Оона', 'Аяк Оона', 'Тогуздун айы', 'Жетинин айы', 'Бештин айы'
				],
				dayOfWeekShort: [
					"Жек", "Дүй", "Шей", "Шар", "Бей", "Жум", "Ише"
				],
				dayOfWeek: [
					"Жекшемб", "Дүйшөмб", "Шейшемб", "Шаршемб", "Бейшемби", "Жума", "Ишенб"
				]
			},
			rm: { // Romansh
				months: [
					"Schaner", "Favrer", "Mars", "Avrigl", "Matg", "Zercladur", "Fanadur", "Avust", "Settember", "October", "November", "December"
				],
				dayOfWeekShort: [
					"Du", "Gli", "Ma", "Me", "Gie", "Ve", "So"
				],
				dayOfWeek: [
					"Dumengia", "Glindesdi", "Mardi", "Mesemna", "Gievgia", "Venderdi", "Sonda"
				]
			},
			ka: { // Georgian
				months: [
					'იანვარი', 'თებერვალი', 'მარტი', 'აპრილი', 'მაისი', 'ივნისი', 'ივლისი', 'აგვისტო', 'სექტემბერი', 'ოქტომბერი', 'ნოემბერი', 'დეკემბერი'
				],
				dayOfWeekShort: [
					"კვ", "ორშ", "სამშ", "ოთხ", "ხუთ", "პარ", "შაბ"
				],
				dayOfWeek: ["კვირა", "ორშაბათი", "სამშაბათი", "ოთხშაბათი", "ხუთშაბათი", "პარასკევი", "შაბათი"]
			}
		},

		ownerDocument: document,
		contentWindow: window,

		value: '',
		rtl: false,

		format:	'Y/m/d H:i',
		formatTime:	'H:i',
		formatDate:	'Y/m/d',

		startDate:	false, // new Date(), '1986/12/08', '-1970/01/05','-1970/01/05',
		step: 60,
		monthChangeSpinner: true,

		closeOnDateSelect: false,
		closeOnTimeSelect: true,
		closeOnWithoutClick: true,
		closeOnInputClick: true,
		openOnFocus: true,

		timepicker: true,
		datepicker: true,
		weeks: false,

		defaultTime: false,	// use formatTime format (ex. '10:00' for formatTime:	'H:i')
		defaultDate: false,	// use formatDate format (ex new Date() or '1986/12/08' or '-1970/01/05' or '-1970/01/05')

		minDate: false,
		maxDate: false,
		minTime: false,
		maxTime: false,
		minDateTime: false,

		disabledMinTime: false,
		disabledMaxTime: false,

		allowTimes: [],
		opened: false,
		initTime: true,
		inline: false,
		theme: '',
		touchMovedThreshold: 5,

		onSelectDate: function () {},
		onSelectTime: function () {},
		onChangeMonth: function () {},
		onGetWeekOfYear: function () {},
		onChangeYear: function () {},
		onChangeDateTime: function () {},
		onShow: function () {},
		onClose: function () {},
		onGenerate: function () {},

		withoutCopyright: true,
		inverseButton: false,
		hours12: false,
		next: 'xdsoft_next',
		prev : 'xdsoft_prev',
		dayOfWeekStart: 0,
		parentID: 'body',
		timeHeightInTimePicker: 25,
		timepickerScrollbar: true,
		todayButton: true,
		prevButton: true,
		nextButton: true,
		defaultSelect: true,

		scrollMonth: true,
		scrollTime: true,
		scrollInput: true,

		lazyInit: false,
		mask: false,
		validateOnBlur: true,
		allowBlank: true,
		yearStart: 1950,
		yearEnd: 2050,
		monthStart: 0,
		monthEnd: 11,
		style: '',
		id: '',
		fixed: false,
		roundTime: 'round', // ceil, floor
		className: '',
		weekends: [],
		highlightedDates: [],
		highlightedPeriods: [],
		allowDates : [],
		allowDateRe : null,
		disabledDates : [],
		disabledWeekDays: [],
		yearOffset: 0,
		beforeShowDay: null,

		enterLikeTab: true,
		showApplyButton: false
	};

	var dateHelper          = null,
		defaultDateHelper   = null,
		globalLocaleDefault = 'en',
		globalLocale        = 'en';

	var dateFormatterOptionsDefault = {
		meridiem: ['AM', 'PM']
	};

	var initDateFormatter = function(){
		var locale = default_options.i18n[globalLocale],
			opts   = {
				days: locale.dayOfWeek,
				daysShort: locale.dayOfWeekShort,
				months: locale.months,
				monthsShort: $.map( locale.months, function(n){ return n.substring( 0, 3 ) } )
		};

		if (typeof DateFormatter === 'function') {
			dateHelper = defaultDateHelper = new DateFormatter(
				{
					dateSettings: $.extend( {}, dateFormatterOptionsDefault, opts )
				}
			);
		}
	};

	var dateFormatters = {
		moment: {
			default_options:{
				format: 'YYYY/MM/DD HH:mm',
				formatDate: 'YYYY/MM/DD',
				formatTime: 'HH:mm',
			},
			formatter: {
				parseDate: function (date, format) {
					if (isFormatStandard( format )) {
						return defaultDateHelper.parseDate( date, format );
					}
					var d = moment( date, format );
					return d.isValid() ? d.toDate() : false;
				},

				formatDate: function (date, format) {
					if (isFormatStandard( format )) {
						return defaultDateHelper.formatDate( date, format );
					}
					return moment( date ).format( format );
				},

				formatMask: function(format){
					return format
						.replace( /Y{4}/g, '9999' )
						.replace( /Y{2}/g, '99' )
						.replace( /M{2}/g, '19' )
						.replace( /D{2}/g, '39' )
						.replace( /H{2}/g, '29' )
						.replace( /m{2}/g, '59' )
						.replace( /s{2}/g, '59' );
				},
			}
		}
	}

	// for locale settings
	$.datetimepicker = {
		setLocale: function(locale){
			var newLocale = default_options.i18n[locale] ? locale : globalLocaleDefault;
			if (globalLocale !== newLocale) {
				globalLocale = newLocale;
				// reinit date formatter
				initDateFormatter();
			}
		},

		setDateFormatter: function(dateFormatter) {
			if (typeof dateFormatter === 'string' && dateFormatters.hasOwnProperty( dateFormatter )) {
				var df = dateFormatters[dateFormatter];
				$.extend( default_options, df.default_options );
				dateHelper = df.formatter;
			} else {
				dateHelper = dateFormatter;
			}
		},
	};

	var standardFormats = {
		RFC_2822: 'D, d M Y H:i:s O',
		ATOM: 'Y-m-d\TH:i:sP',
		ISO_8601: 'Y-m-d\TH:i:sO',
		RFC_822: 'D, d M y H:i:s O',
		RFC_850: 'l, d-M-y H:i:s T',
		RFC_1036: 'D, d M y H:i:s O',
		RFC_1123: 'D, d M Y H:i:s O',
		RSS: 'D, d M Y H:i:s O',
		W3C: 'Y-m-d\TH:i:sP'
	}

	var isFormatStandard = function(format){
		return Object.values( standardFormats ).indexOf( format ) === -1 ? false : true;
	}

	$.extend( $.datetimepicker, standardFormats );

	// first init date formatter
	initDateFormatter();

	// fix for ie8
	if ( ! window.getComputedStyle) {
		window.getComputedStyle = function (el) {
			this.el               = el;
			this.getPropertyValue = function (prop) {
				var re = /(-([a-z]))/g;
				if (prop === 'float') {
					prop = 'styleFloat';
				}
				if (re.test( prop )) {
					prop = prop.replace(
						re,
						function (a, b, c) {
							return c.toUpperCase();
						}
					);
				}
				return el.currentStyle[prop] || null;
			};
			return this;
		};
	}
	if ( ! Array.prototype.indexOf) {
		Array.prototype.indexOf = function (obj, start) {
			var i, j;
			for (i = (start || 0), j = this.length; i < j; i += 1) {
				if (this[i] === obj) {
					return i; }
			}
			return -1;
		};
	}

	Date.prototype.countDaysInMonth = function () {
		return new Date( this.getFullYear(), this.getMonth() + 1, 0 ).getDate();
	};

	$.fn.xdsoftScroller = function (options, percent) {
		return this.each(
			function () {
				var timeboxparent = $( this ),
				pointerEventToXY  = function (e) {
					var out = {x: 0, y: 0},
						touch;
					if (e.type === 'touchstart' || e.type === 'touchmove' || e.type === 'touchend' || e.type === 'touchcancel') {
						touch = e.originalEvent.touches[0] || e.originalEvent.changedTouches[0];
						out.x = touch.clientX;
						out.y = touch.clientY;
					} else if (e.type === 'mousedown' || e.type === 'mouseup' || e.type === 'mousemove' || e.type === 'mouseover' || e.type === 'mouseout' || e.type === 'mouseenter' || e.type === 'mouseleave') {
						out.x = e.clientX;
						out.y = e.clientY;
					}
					return out;
				},
				timebox,
				parentHeight,
				height,
				scrollbar,
				scroller,
				maximumOffset     = 100,
				start             = false,
				startY            = 0,
				startTop          = 0,
				h1                = 0,
				touchStart        = false,
				startTopScroll    = 0,
				calcOffset        = function () {};

				if (percent === 'hide') {
					timeboxparent.find( '.xdsoft_scrollbar' ).hide();
					return;
				}

				if ( ! $( this ).hasClass( 'xdsoft_scroller_box' )) {
					timebox      = timeboxparent.children().eq( 0 );
					parentHeight = timeboxparent[0].clientHeight;
					height       = timebox[0].offsetHeight;
					scrollbar    = $( '<div class="xdsoft_scrollbar"></div>' );
					scroller     = $( '<div class="xdsoft_scroller"></div>' );
					scrollbar.append( scroller );

					timeboxparent.addClass( 'xdsoft_scroller_box' ).append( scrollbar );
					calcOffset = function calcOffset(event) {
						var offset = pointerEventToXY( event ).y - startY + startTopScroll;
						if (offset < 0) {
							offset = 0;
						}
						if (offset + scroller[0].offsetHeight > h1) {
							offset = h1 - scroller[0].offsetHeight;
						}
						timeboxparent.trigger( 'scroll_element.xdsoft_scroller', [maximumOffset ? offset / maximumOffset : 0] );
					};

					scroller
					.on(
						'touchstart.xdsoft_scroller mousedown.xdsoft_scroller',
						function (event) {
							if ( ! parentHeight) {
								timeboxparent.trigger( 'resize_scroll.xdsoft_scroller', [percent] );
							}

							startY         = pointerEventToXY( event ).y;
							startTopScroll = parseInt( scroller.css( 'margin-top' ), 10 );
							h1             = scrollbar[0].offsetHeight;

							if (event.type === 'mousedown' || event.type === 'touchstart') {
								if (options.ownerDocument) {
									$( options.ownerDocument.body ).addClass( 'xdsoft_noselect' );
								}
								$( [options.ownerDocument.body, options.contentWindow] ).on(
									'touchend mouseup.xdsoft_scroller',
									function arguments_callee() {
										$( [options.ownerDocument.body, options.contentWindow] ).off( 'touchend mouseup.xdsoft_scroller', arguments_callee )
										.off( 'mousemove.xdsoft_scroller', calcOffset )
										.removeClass( 'xdsoft_noselect' );
									}
								);
								$( options.ownerDocument.body ).on( 'mousemove.xdsoft_scroller', calcOffset );
							} else {
								touchStart = true;
								event.stopPropagation();
								event.preventDefault();
							}
						}
					)
						.on(
							'touchmove',
							function (event) {
								if (touchStart) {
									event.preventDefault();
									calcOffset( event );
								}
							}
						)
						.on(
							'touchend touchcancel',
							function () {
								touchStart     = false;
								startTopScroll = 0;
							}
						);

					timeboxparent
						.on(
							'scroll_element.xdsoft_scroller',
							function (event, percentage) {
								if ( ! parentHeight) {
									timeboxparent.trigger( 'resize_scroll.xdsoft_scroller', [percentage, true] );
								}
								percentage = percentage > 1 ? 1 : (percentage < 0 || isNaN( percentage )) ? 0 : percentage;

								scroller.css( 'margin-top', maximumOffset * percentage );

								setTimeout(
									function () {
										timebox.css( 'marginTop', -parseInt( (timebox[0].offsetHeight - parentHeight) * percentage, 10 ) );
									},
									10
								);
							}
						)
						.on(
							'resize_scroll.xdsoft_scroller',
							function (event, percentage, noTriggerScroll) {
								var percent, sh;
								parentHeight = timeboxparent[0].clientHeight;
								height       = timebox[0].offsetHeight;
								percent      = parentHeight / height;
								sh           = percent * scrollbar[0].offsetHeight;
								if (percent > 1) {
									scroller.hide();
								} else {
									scroller.show();
									scroller.css( 'height', parseInt( sh > 10 ? sh : 10, 10 ) );
									maximumOffset = scrollbar[0].offsetHeight - scroller[0].offsetHeight;
									if (noTriggerScroll !== true) {
										timeboxparent.trigger( 'scroll_element.xdsoft_scroller', [percentage || Math.abs( parseInt( timebox.css( 'marginTop' ), 10 ) ) / (height - parentHeight)] );
									}
								}
							}
						);

					timeboxparent.on(
						'mousewheel',
						function (event) {
							var top = Math.abs( parseInt( timebox.css( 'marginTop' ), 10 ) );

							top = top - (event.deltaY * 20);
							if (top < 0) {
								top = 0;
							}

							timeboxparent.trigger( 'scroll_element.xdsoft_scroller', [top / (height - parentHeight)] );
							event.stopPropagation();
							return false;
						}
					);

					timeboxparent.on(
						'touchstart',
						function (event) {
							start    = pointerEventToXY( event );
							startTop = Math.abs( parseInt( timebox.css( 'marginTop' ), 10 ) );
						}
					);

					timeboxparent.on(
						'touchmove',
						function (event) {
							if (start) {
								event.preventDefault();
								var coord = pointerEventToXY( event );
								timeboxparent.trigger( 'scroll_element.xdsoft_scroller', [(startTop - (coord.y - start.y)) / (height - parentHeight)] );
							}
						}
					);

					timeboxparent.on(
						'touchend touchcancel',
						function () {
							start    = false;
							startTop = 0;
						}
					);
				}
				timeboxparent.trigger( 'resize_scroll.xdsoft_scroller', [percent] );
			}
		);
	};

	$.fn.datetimepicker = function (opt, opt2) {
		var result     = this,
			KEY0       = 48,
			KEY9       = 57,
			_KEY0      = 96,
			_KEY9      = 105,
			CTRLKEY    = 17,
			DEL        = 46,
			ENTER      = 13,
			ESC        = 27,
			BACKSPACE  = 8,
			ARROWLEFT  = 37,
			ARROWUP    = 38,
			ARROWRIGHT = 39,
			ARROWDOWN  = 40,
			TAB        = 9,
			F5         = 116,
			AKEY       = 65,
			CKEY       = 67,
			VKEY       = 86,
			ZKEY       = 90,
			YKEY       = 89,
			ctrlDown   = false,
			options    = ($.isPlainObject( opt ) || ! opt) ? $.extend( true, {}, default_options, opt ) : $.extend( true, {}, default_options ),

			lazyInitTimer = 0,
			createDateTimePicker,
			destroyDateTimePicker,

			lazyInit = function (input) {
				input
					.on(
						'open.xdsoft focusin.xdsoft mousedown.xdsoft touchstart',
						function initOnActionCallback() {
					if (input.is( ':disabled' ) || input.data( 'xdsoft_datetimepicker' )) {
						return;
					}
						clearTimeout( lazyInitTimer );
						lazyInitTimer = setTimeout(
							function () {

								if ( ! input.data( 'xdsoft_datetimepicker' )) {
									createDateTimePicker( input );
								}
								input
								.off( 'open.xdsoft focusin.xdsoft mousedown.xdsoft touchstart', initOnActionCallback )
								.trigger( 'open.xdsoft' );
							},
							100
						);
						}
					);
			};

		createDateTimePicker  = function (input) {
			var datetimepicker   = $( '<div class="xdsoft_datetimepicker xdsoft_noselect"></div>' ),
				xdsoft_copyright = $( '<div class="xdsoft_copyright"><a target="_blank" href="http://xdsoft.net/jqplugins/datetimepicker/">xdsoft.net</a></div>' ),
				datepicker       = $( '<div class="xdsoft_datepicker active"></div>' ),
				month_picker     = $(
					'<div class="xdsoft_monthpicker"><button type="button" class="xdsoft_prev"></button><button type="button" class="xdsoft_today_button"></button>' +
					'<div class="xdsoft_label xdsoft_month"><span></span><i></i></div>' +
					'<div class="xdsoft_label xdsoft_year"><span></span><i></i></div>' +
					'<button type="button" class="xdsoft_next"></button></div>'
				),
				calendar         = $( '<div class="xdsoft_calendar"></div>' ),
				timepicker       = $( '<div class="xdsoft_timepicker active"><button type="button" class="xdsoft_prev"></button><div class="xdsoft_time_box"></div><button type="button" class="xdsoft_next"></button></div>' ),
				timeboxparent    = timepicker.find( '.xdsoft_time_box' ).eq( 0 ),
				timebox          = $( '<div class="xdsoft_time_variant"></div>' ),
				applyButton      = $( '<button type="button" class="xdsoft_save_selected blue-gradient-button">Save Selected</button>' ),

				monthselect      = $( '<div class="xdsoft_select xdsoft_monthselect"><div></div></div>' ),
				yearselect       = $( '<div class="xdsoft_select xdsoft_yearselect"><div></div></div>' ),
				triggerAfterOpen = false,
				XDSoft_datetime,

				xchangeTimer,
				timerclick,
				current_time_index,
				setPos,
				timer = 0,
				_xdsoft_datetime,
				forEachAncestorOf;

			if (options.id) {
				datetimepicker.attr( 'id', options.id );
			}
			if (options.style) {
				datetimepicker.attr( 'style', options.style );
			}
			if (options.weeks) {
				datetimepicker.addClass( 'xdsoft_showweeks' );
			}
			if (options.rtl) {
				datetimepicker.addClass( 'xdsoft_rtl' );
			}

			datetimepicker.addClass( 'xdsoft_' + options.theme );
			datetimepicker.addClass( options.className );

			month_picker
				.find( '.xdsoft_month span' )
				.after( monthselect );
			month_picker
				.find( '.xdsoft_year span' )
				.after( yearselect );

			month_picker
				.find( '.xdsoft_month,.xdsoft_year' )
				.on(
					'touchstart mousedown.xdsoft',
					function (event) {
						var select = $( this ).find( '.xdsoft_select' ).eq( 0 ),
						val        = 0,
						top        = 0,
						visible    = select.is( ':visible' ),
						items,
						i;

						month_picker
						.find( '.xdsoft_select' )
						.hide();
						if (_xdsoft_datetime.currentTime) {
							val = _xdsoft_datetime.currentTime[$( this ).hasClass( 'xdsoft_month' ) ? 'getMonth' : 'getFullYear']();
						}

						select[visible ? 'hide' : 'show']();
						for (items = select.find( 'div.xdsoft_option' ), i = 0; i < items.length; i += 1) {
							if (items.eq( i ).data( 'value' ) === val) {
								break;
							} else {
								top += items[0].offsetHeight;
							}
						}

						select.xdsoftScroller( options, top / (select.children()[0].offsetHeight - (select[0].clientHeight)) );
						event.stopPropagation();
						return false;
					}
				);

			var handleTouchMoved = function (event) {
				this.touchStartPosition = this.touchStartPosition || event.originalEvent.touches[0]
				var touchPosition       = event.originalEvent.touches[0]
				var xMovement           = Math.abs( this.touchStartPosition.clientX - touchPosition.clientX )
				var yMovement           = Math.abs( this.touchStartPosition.clientY - touchPosition.clientY )
				var distance            = Math.sqrt( xMovement * xMovement + yMovement * yMovement )
				if (distance > options.touchMovedThreshold) {
					this.touchMoved = true;
				}
			}

			month_picker
				.find( '.xdsoft_select' )
				.xdsoftScroller( options )
				.on(
					'touchstart mousedown.xdsoft',
					function (event) {
						this.touchMoved         = false;
						this.touchStartPosition = event.originalEvent.touches[0]
						event.stopPropagation();
						event.preventDefault();
					}
				)
				.on( 'touchmove', '.xdsoft_option', handleTouchMoved )
				.on(
					'touchend mousedown.xdsoft',
					'.xdsoft_option',
					function () {
						if ( ! this.touchMoved) {
							if (_xdsoft_datetime.currentTime === undefined || _xdsoft_datetime.currentTime === null) {
								_xdsoft_datetime.currentTime = _xdsoft_datetime.now();
							}

							var year = _xdsoft_datetime.currentTime.getFullYear();
							if (_xdsoft_datetime && _xdsoft_datetime.currentTime) {
								_xdsoft_datetime.currentTime[$( this ).parent().parent().hasClass( 'xdsoft_monthselect' ) ? 'setMonth' : 'setFullYear']( $( this ).data( 'value' ) );
							}

							$( this ).parent().parent().hide();

							datetimepicker.trigger( 'xchange.xdsoft' );
							if (options.onChangeMonth && $.isFunction( options.onChangeMonth )) {
								options.onChangeMonth.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ) );
							}

							if (year !== _xdsoft_datetime.currentTime.getFullYear() && $.isFunction( options.onChangeYear )) {
								options.onChangeYear.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ) );
							}
						}
					}
				);

			datetimepicker.getValue = function () {
				return _xdsoft_datetime.getCurrentTime();
			};

			datetimepicker.setOptions = function (_options) {
				var highlightedDates = {};

				options = $.extend( true, {}, options, _options );

				if (_options.allowTimes && $.isArray( _options.allowTimes ) && _options.allowTimes.length) {
					options.allowTimes = $.extend( true, [], _options.allowTimes );
				}

				if (_options.weekends && $.isArray( _options.weekends ) && _options.weekends.length) {
					options.weekends = $.extend( true, [], _options.weekends );
				}

				if (_options.allowDates && $.isArray( _options.allowDates ) && _options.allowDates.length) {
					options.allowDates = $.extend( true, [], _options.allowDates );
				}

				if (_options.allowDateRe && Object.prototype.toString.call( _options.allowDateRe ) === "[object String]") {
					options.allowDateRe = new RegExp( _options.allowDateRe );
				}

				if (_options.highlightedDates && $.isArray( _options.highlightedDates ) && _options.highlightedDates.length) {
					$.each(
						_options.highlightedDates,
						function (index, value) {
							var splitData = $.map( value.split( ',' ), $.trim ),
							exDesc,
							hDate         = new HighlightedDate( dateHelper.parseDate( splitData[0], options.formatDate ), splitData[1], splitData[2] ), // date, desc, style
							keyDate       = dateHelper.formatDate( hDate.date, options.formatDate );
							if (highlightedDates[keyDate] !== undefined) {
								exDesc = highlightedDates[keyDate].desc;
								if (exDesc && exDesc.length && hDate.desc && hDate.desc.length) {
									highlightedDates[keyDate].desc = exDesc + "\n" + hDate.desc;
								}
							} else {
								highlightedDates[keyDate] = hDate;
							}
						}
					);

					options.highlightedDates = $.extend( true, [], highlightedDates );
				}

				if (_options.highlightedPeriods && $.isArray( _options.highlightedPeriods ) && _options.highlightedPeriods.length) {
					highlightedDates = $.extend( true, [], options.highlightedDates );
					$.each(
						_options.highlightedPeriods,
						function (index, value) {
							var dateTest, // start date
							dateEnd,
							desc,
							hDate,
							keyDate,
							exDesc,
							style;
							if ($.isArray( value )) {
								dateTest = value[0];
								dateEnd  = value[1];
								desc     = value[2];
								style    = value[3];
							} else {
								var splitData = $.map( value.split( ',' ), $.trim );
								dateTest      = dateHelper.parseDate( splitData[0], options.formatDate );
								dateEnd       = dateHelper.parseDate( splitData[1], options.formatDate );
								desc          = splitData[2];
								style         = splitData[3];
							}

							while (dateTest <= dateEnd) {
								hDate   = new HighlightedDate( dateTest, desc, style );
								keyDate = dateHelper.formatDate( dateTest, options.formatDate );
								dateTest.setDate( dateTest.getDate() + 1 );
								if (highlightedDates[keyDate] !== undefined) {
									exDesc = highlightedDates[keyDate].desc;
									if (exDesc && exDesc.length && hDate.desc && hDate.desc.length) {
										highlightedDates[keyDate].desc = exDesc + "\n" + hDate.desc;
									}
								} else {
									highlightedDates[keyDate] = hDate;
								}
							}
						}
					);

					options.highlightedDates = $.extend( true, [], highlightedDates );
				}

				if (_options.disabledDates && $.isArray( _options.disabledDates ) && _options.disabledDates.length) {
					options.disabledDates = $.extend( true, [], _options.disabledDates );
				}

				if (_options.disabledWeekDays && $.isArray( _options.disabledWeekDays ) && _options.disabledWeekDays.length) {
					options.disabledWeekDays = $.extend( true, [], _options.disabledWeekDays );
				}

				if ((options.open || options.opened) && ( ! options.inline)) {
					input.trigger( 'open.xdsoft' );
				}

				if (options.inline) {
					triggerAfterOpen = true;
					datetimepicker.addClass( 'xdsoft_inline' );
					input.after( datetimepicker ).hide();
				}

				if (options.inverseButton) {
					options.next = 'xdsoft_prev';
					options.prev = 'xdsoft_next';
				}

				if (options.datepicker) {
					datepicker.addClass( 'active' );
				} else {
					datepicker.removeClass( 'active' );
				}

				if (options.timepicker) {
					timepicker.addClass( 'active' );
				} else {
					timepicker.removeClass( 'active' );
				}

				if (options.value) {
					_xdsoft_datetime.setCurrentTime( options.value );
					if (input && input.val) {
						input.val( _xdsoft_datetime.str );
					}
				}

				if (isNaN( options.dayOfWeekStart )) {
					options.dayOfWeekStart = 0;
				} else {
					options.dayOfWeekStart = parseInt( options.dayOfWeekStart, 10 ) % 7;
				}

				if ( ! options.timepickerScrollbar) {
					timeboxparent.xdsoftScroller( options, 'hide' );
				}

				if (options.minDate && /^[\+\-](.*)$/.test( options.minDate )) {
					options.minDate = dateHelper.formatDate( _xdsoft_datetime.strToDateTime( options.minDate ), options.formatDate );
				}

				if (options.maxDate && /^[\+\-](.*)$/.test( options.maxDate )) {
					options.maxDate = dateHelper.formatDate( _xdsoft_datetime.strToDateTime( options.maxDate ), options.formatDate );
				}

				if (options.minDateTime && /^\+(.*)$/.test( options.minDateTime )) {
					options.minDateTime = _xdsoft_datetime.strToDateTime( options.minDateTime ).dateFormat( options.formatDate );
				}

				applyButton.toggle( options.showApplyButton );

				month_picker
					.find( '.xdsoft_today_button' )
					.css( 'visibility', ! options.todayButton ? 'hidden' : 'visible' );

				month_picker
					.find( '.' + options.prev )
					.css( 'visibility', ! options.prevButton ? 'hidden' : 'visible' );

				month_picker
					.find( '.' + options.next )
					.css( 'visibility', ! options.nextButton ? 'hidden' : 'visible' );

				setMask( options );

				if (options.validateOnBlur) {
					input
						.off( 'blur.xdsoft' )
						.on(
							'blur.xdsoft',
							function () {
								if (options.allowBlank && ( ! $.trim( $( this ).val() ).length ||
									(typeof options.mask === "string" && $.trim( $( this ).val() ) === options.mask.replace( /[0-9]/g, '_' )))) {
									$( this ).val( null );
									datetimepicker.data( 'xdsoft_datetime' ).empty();
								} else {
									var d = dateHelper.parseDate( $( this ).val(), options.format );
									if (d) { // parseDate() may skip some invalid parts like date or time, so make it clear for user: show parsed date/time
										$( this ).val( dateHelper.formatDate( d, options.format ) );
									} else {
										var splittedHours = +([$( this ).val()[0], $( this ).val()[1]].join( '' )),
										splittedMinutes   = +([$( this ).val()[2], $( this ).val()[3]].join( '' ));

										// parse the numbers as 0312 => 03:12
										if ( ! options.datepicker && options.timepicker && splittedHours >= 0 && splittedHours < 24 && splittedMinutes >= 0 && splittedMinutes < 60) {
											$( this ).val(
												[splittedHours, splittedMinutes].map(
													function (item) {
														return item > 9 ? item : '0' + item;
													}
												).join( ':' )
											);
										} else {
											$( this ).val( dateHelper.formatDate( _xdsoft_datetime.now(), options.format ) );
										}
									}
									datetimepicker.data( 'xdsoft_datetime' ).setCurrentTime( $( this ).val() );
								}

								datetimepicker.trigger( 'changedatetime.xdsoft' );
								datetimepicker.trigger( 'close.xdsoft' );
							}
						);
				}
				options.dayOfWeekStartPrev = (options.dayOfWeekStart === 0) ? 6 : options.dayOfWeekStart - 1;

				datetimepicker
					.trigger( 'xchange.xdsoft' )
					.trigger( 'afterOpen.xdsoft' );
			};

			datetimepicker
				.data( 'options', options )
				.on(
					'touchstart mousedown.xdsoft',
					function (event) {
						event.stopPropagation();
						event.preventDefault();
						yearselect.hide();
						monthselect.hide();
						return false;
					}
				);

			//scroll_element = timepicker.find('.xdsoft_time_box');
			timeboxparent.append( timebox );
			timeboxparent.xdsoftScroller( options );

			datetimepicker.on(
				'afterOpen.xdsoft',
				function () {
					timeboxparent.xdsoftScroller( options );
				}
			);

			datetimepicker
				.append( datepicker )
				.append( timepicker );

			if (options.withoutCopyright !== true) {
				datetimepicker
					.append( xdsoft_copyright );
			}

			datepicker
				.append( month_picker )
				.append( calendar )
				.append( applyButton );

			$( options.parentID )
				.append( datetimepicker );

			XDSoft_datetime = function () {
				var _this = this;
				_this.now = function (norecursion) {
					var d = new Date(),
						date,
						time;

					if ( ! norecursion && options.defaultDate) {
						date = _this.strToDateTime( options.defaultDate );
						d.setFullYear( date.getFullYear() );
						d.setMonth( date.getMonth() );
						d.setDate( date.getDate() );
					}

					d.setFullYear( d.getFullYear() );

					if ( ! norecursion && options.defaultTime) {
						time = _this.strtotime( options.defaultTime );
						d.setHours( time.getHours() );
						d.setMinutes( time.getMinutes() );
						d.setSeconds( time.getSeconds() );
						d.setMilliseconds( time.getMilliseconds() );
					}
					return d;
				};

				_this.isValidDate = function (d) {
					if (Object.prototype.toString.call( d ) !== "[object Date]") {
						return false;
					}
					return ! isNaN( d.getTime() );
				};

				_this.setCurrentTime = function (dTime, requireValidDate) {
					if (typeof dTime === 'string') {
						_this.currentTime = _this.strToDateTime( dTime );
					} else if (_this.isValidDate( dTime )) {
						_this.currentTime = dTime;
					} else if ( ! dTime && ! requireValidDate && options.allowBlank && ! options.inline) {
						_this.currentTime = null;
					} else {
						_this.currentTime = _this.now();
					}

					datetimepicker.trigger( 'xchange.xdsoft' );
				};

				_this.empty = function () {
					_this.currentTime = null;
				};

				_this.getCurrentTime = function () {
					return _this.currentTime;
				};

				_this.nextMonth = function () {

					if (_this.currentTime === undefined || _this.currentTime === null) {
						_this.currentTime = _this.now();
					}

					var month = _this.currentTime.getMonth() + 1,
						year;
					if (month === 12) {
						_this.currentTime.setFullYear( _this.currentTime.getFullYear() + 1 );
						month = 0;
					}

					year = _this.currentTime.getFullYear();

					_this.currentTime.setDate(
						Math.min(
							new Date( _this.currentTime.getFullYear(), month + 1, 0 ).getDate(),
							_this.currentTime.getDate()
						)
					);
					_this.currentTime.setMonth( month );

					if (options.onChangeMonth && $.isFunction( options.onChangeMonth )) {
						options.onChangeMonth.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ) );
					}

					if (year !== _this.currentTime.getFullYear() && $.isFunction( options.onChangeYear )) {
						options.onChangeYear.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ) );
					}

					datetimepicker.trigger( 'xchange.xdsoft' );
					return month;
				};

				_this.prevMonth = function () {

					if (_this.currentTime === undefined || _this.currentTime === null) {
						_this.currentTime = _this.now();
					}

					var month = _this.currentTime.getMonth() - 1;
					if (month === -1) {
						_this.currentTime.setFullYear( _this.currentTime.getFullYear() - 1 );
						month = 11;
					}
					_this.currentTime.setDate(
						Math.min(
							new Date( _this.currentTime.getFullYear(), month + 1, 0 ).getDate(),
							_this.currentTime.getDate()
						)
					);
					_this.currentTime.setMonth( month );
					if (options.onChangeMonth && $.isFunction( options.onChangeMonth )) {
						options.onChangeMonth.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ) );
					}
					datetimepicker.trigger( 'xchange.xdsoft' );
					return month;
				};

				_this.getWeekOfYear = function (datetime) {
					if (options.onGetWeekOfYear && $.isFunction( options.onGetWeekOfYear )) {
						var week = options.onGetWeekOfYear.call( datetimepicker, datetime );
						if (typeof week !== 'undefined') {
							return week;
						}
					}
					var onejan = new Date( datetime.getFullYear(), 0, 1 );

					//First week of the year is th one with the first Thursday according to ISO8601
					if (onejan.getDay() !== 4) {
						onejan.setMonth( 0, 1 + ((4 - onejan.getDay() + 7) % 7) );
					}

					return Math.ceil( (((datetime - onejan) / 86400000) + onejan.getDay() + 1) / 7 );
				};

				_this.strToDateTime = function (sDateTime) {
					var tmpDate = [], timeOffset, currentTime;

					if (sDateTime && sDateTime instanceof Date && _this.isValidDate( sDateTime )) {
						return sDateTime;
					}

					tmpDate = /^([+-]{1})(.*)$/.exec( sDateTime );

					if (tmpDate) {
						tmpDate[2] = dateHelper.parseDate( tmpDate[2], options.formatDate );
					}

					if (tmpDate && tmpDate[2]) {
						timeOffset  = tmpDate[2].getTime() - (tmpDate[2].getTimezoneOffset()) * 60000;
						currentTime = new Date( (_this.now( true )).getTime() + parseInt( tmpDate[1] + '1', 10 ) * timeOffset );
					} else {
						currentTime = sDateTime ? dateHelper.parseDate( sDateTime, options.format ) : _this.now();
					}

					if ( ! _this.isValidDate( currentTime )) {
						currentTime = _this.now();
					}

					return currentTime;
				};

				_this.strToDate = function (sDate) {
					if (sDate && sDate instanceof Date && _this.isValidDate( sDate )) {
						return sDate;
					}

					var currentTime = sDate ? dateHelper.parseDate( sDate, options.formatDate ) : _this.now( true );
					if ( ! _this.isValidDate( currentTime )) {
						currentTime = _this.now( true );
					}
					return currentTime;
				};

				_this.strtotime = function (sTime) {
					if (sTime && sTime instanceof Date && _this.isValidDate( sTime )) {
						return sTime;
					}
					var currentTime = sTime ? dateHelper.parseDate( sTime, options.formatTime ) : _this.now( true );
					if ( ! _this.isValidDate( currentTime )) {
						currentTime = _this.now( true );
					}
					return currentTime;
				};

				_this.str         = function () {
					var format = options.format;
					if (options.yearOffset) {
						format = format.replace( 'Y', _this.currentTime.getFullYear() + options.yearOffset );
						format = format.replace( 'y', String( _this.currentTime.getFullYear() + options.yearOffset ).substring( 2, 4 ) );
					}
					return dateHelper.formatDate( _this.currentTime, format );
				};
				_this.currentTime = this.now();
			};

			_xdsoft_datetime = new XDSoft_datetime();

			applyButton.on(
				'touchend click',
				function (e) {//pathbrite
					e.preventDefault();
					datetimepicker.data( 'changed', true );
					_xdsoft_datetime.setCurrentTime( getCurrentValue() );
					input.val( _xdsoft_datetime.str() );
					datetimepicker.trigger( 'close.xdsoft' );
				}
			);
			month_picker
				.find( '.xdsoft_today_button' )
				.on(
					'touchend mousedown.xdsoft',
					function () {
						datetimepicker.data( 'changed', true );
						_xdsoft_datetime.setCurrentTime( 0, true );
						datetimepicker.trigger( 'afterOpen.xdsoft' );
					}
				).on(
					'dblclick.xdsoft',
					function () {
						var currentDate = _xdsoft_datetime.getCurrentTime(), minDate, maxDate;
						currentDate     = new Date( currentDate.getFullYear(), currentDate.getMonth(), currentDate.getDate() );
						minDate         = _xdsoft_datetime.strToDate( options.minDate );
						minDate         = new Date( minDate.getFullYear(), minDate.getMonth(), minDate.getDate() );
						if (currentDate < minDate) {
							return;
						}
						maxDate = _xdsoft_datetime.strToDate( options.maxDate );
						maxDate = new Date( maxDate.getFullYear(), maxDate.getMonth(), maxDate.getDate() );
						if (currentDate > maxDate) {
							return;
						}
						input.val( _xdsoft_datetime.str() );
						input.trigger( 'change' );
						datetimepicker.trigger( 'close.xdsoft' );
					}
				);
			month_picker
				.find( '.xdsoft_prev,.xdsoft_next' )
				.on(
					'touchend mousedown.xdsoft',
					function () {
						var $this = $( this ),
						timer     = 0,
						stop      = false;

						(function arguments_callee1(v) {
							if ($this.hasClass( options.next )) {
								_xdsoft_datetime.nextMonth();
							} else if ($this.hasClass( options.prev )) {
								_xdsoft_datetime.prevMonth();
							}
							if (options.monthChangeSpinner) {
								if ( ! stop) {
									timer = setTimeout( arguments_callee1, v || 100 );
								}
							}
						}(500));

						$( [options.ownerDocument.body, options.contentWindow] ).on(
							'touchend mouseup.xdsoft',
							function arguments_callee2() {
								clearTimeout( timer );
								stop = true;
								$( [options.ownerDocument.body, options.contentWindow] ).off( 'touchend mouseup.xdsoft', arguments_callee2 );
							}
						);
					}
				);

			timepicker
				.find( '.xdsoft_prev,.xdsoft_next' )
				.on(
					'touchend mousedown.xdsoft',
					function () {
						var $this = $( this ),
						timer     = 0,
						stop      = false,
						period    = 110;
						(function arguments_callee4(v) {
							var pheight = timeboxparent[0].clientHeight,
							height      = timebox[0].offsetHeight,
							top         = Math.abs( parseInt( timebox.css( 'marginTop' ), 10 ) );
							if ($this.hasClass( options.next ) && (height - pheight) - options.timeHeightInTimePicker >= top) {
								timebox.css( 'marginTop', '-' + (top + options.timeHeightInTimePicker) + 'px' );
							} else if ($this.hasClass( options.prev ) && top - options.timeHeightInTimePicker >= 0) {
								timebox.css( 'marginTop', '-' + (top - options.timeHeightInTimePicker) + 'px' );
							}
							/**
							 * Fixed bug:
							 * When using css3 transition, it will cause a bug that you cannot scroll the timepicker list.
							 * The reason is that the transition-duration time, if you set it to 0, all things fine, otherwise, this
							 * would cause a bug when you use jquery.css method.
							 * Let's say: * { transition: all .5s ease; }
							 * jquery timebox.css('marginTop') will return the original value which is before you clicking the next/prev button,
							 * meanwhile the timebox[0].style.marginTop will return the right value which is after you clicking the
							 * next/prev button.
							 *
							 * What we should do:
							 * Replace timebox.css('marginTop') with timebox[0].style.marginTop.
							 */
							timeboxparent.trigger( 'scroll_element.xdsoft_scroller', [Math.abs( parseInt( timebox[0].style.marginTop, 10 ) / (height - pheight) )] );
							period = (period > 10) ? 10 : period - 10;
							if ( ! stop) {
								timer = setTimeout( arguments_callee4, v || period );
							}
						}(500));
						$( [options.ownerDocument.body, options.contentWindow] ).on(
							'touchend mouseup.xdsoft',
							function arguments_callee5() {
								clearTimeout( timer );
								stop = true;
								$( [options.ownerDocument.body, options.contentWindow] )
								.off( 'touchend mouseup.xdsoft', arguments_callee5 );
							}
						);
					}
				);

			xchangeTimer = 0;
			// base handler - generating a calendar and timepicker
			datetimepicker
				.on(
					'xchange.xdsoft',
					function (event) {
						clearTimeout( xchangeTimer );
						xchangeTimer = setTimeout(
							function () {

								if (_xdsoft_datetime.currentTime === undefined || _xdsoft_datetime.currentTime === null) {
									_xdsoft_datetime.currentTime = _xdsoft_datetime.now();
								}

								var table   = '',
								start       = new Date( _xdsoft_datetime.currentTime.getFullYear(), _xdsoft_datetime.currentTime.getMonth(), 1, 12, 0, 0 ),
								i           = 0,
								j,
								today       = _xdsoft_datetime.now(),
								maxDate     = false,
								minDate     = false,
								minDateTime = false,
								hDate,
								day,
								d,
								y,
								m,
								w,
								classes     = [],
								customDateSettings,
								newRow      = true,
								time        = '',
								h,
								line_time,
								description;

								while (start.getDay() !== options.dayOfWeekStart) {
									start.setDate( start.getDate() - 1 );
								}

								table += '<table><thead><tr>';

								if (options.weeks) {
									table += '<th></th>';
								}

								for (j = 0; j < 7; j += 1) {
									table += '<th>' + options.i18n[globalLocale].dayOfWeekShort[(j + options.dayOfWeekStart) % 7] + '</th>';
								}

								table += '</tr></thead>';
								table += '<tbody>';

								if (options.maxDate !== false) {
									maxDate = _xdsoft_datetime.strToDate( options.maxDate );
									maxDate = new Date( maxDate.getFullYear(), maxDate.getMonth(), maxDate.getDate(), 23, 59, 59, 999 );
								}

								if (options.minDate !== false) {
									minDate = _xdsoft_datetime.strToDate( options.minDate );
									minDate = new Date( minDate.getFullYear(), minDate.getMonth(), minDate.getDate() );
								}

								if (options.minDateTime !== false) {
									minDateTime = _xdsoft_datetime.strToDate( options.minDateTime );
									minDateTime = new Date( minDateTime.getFullYear(), minDateTime.getMonth(), minDateTime.getDate(), minDateTime.getHours(), minDateTime.getMinutes(), minDateTime.getSeconds() );
								}

								while (i < _xdsoft_datetime.currentTime.countDaysInMonth() || start.getDay() !== options.dayOfWeekStart || _xdsoft_datetime.currentTime.getMonth() === start.getMonth()) {
									classes = [];
									i      += 1;

									day         = start.getDay();
									d           = start.getDate();
									y           = start.getFullYear();
									m           = start.getMonth();
									w           = _xdsoft_datetime.getWeekOfYear( start );
									description = '';

									classes.push( 'xdsoft_date' );

									if (options.beforeShowDay && $.isFunction( options.beforeShowDay.call )) {
										customDateSettings = options.beforeShowDay.call( datetimepicker, start );
									} else {
										customDateSettings = null;
									}

									if (options.allowDateRe && Object.prototype.toString.call( options.allowDateRe ) === "[object RegExp]") {
										if ( ! options.allowDateRe.test( dateHelper.formatDate( start, options.formatDate ) )) {
											classes.push( 'xdsoft_disabled' );
										}
									}

									if (options.allowDates && options.allowDates.length > 0) {
										if (options.allowDates.indexOf( dateHelper.formatDate( start, options.formatDate ) ) === -1) {
											classes.push( 'xdsoft_disabled' );
										}
									}

									if ((maxDate !== false && start > maxDate) || (minDateTime !== false && start < minDateTime) || (minDate !== false && start < minDate) || (customDateSettings && customDateSettings[0] === false)) {
										classes.push( 'xdsoft_disabled' );
									}

									if (options.disabledDates.indexOf( dateHelper.formatDate( start, options.formatDate ) ) !== -1) {
										classes.push( 'xdsoft_disabled' );
									}

									if (options.disabledWeekDays.indexOf( day ) !== -1) {
										classes.push( 'xdsoft_disabled' );
									}

									if (input.is( '[disabled]' )) {
										classes.push( 'xdsoft_disabled' );
									}

									if (customDateSettings && customDateSettings[1] !== "") {
										classes.push( customDateSettings[1] );
									}

									if (_xdsoft_datetime.currentTime.getMonth() !== m) {
										classes.push( 'xdsoft_other_month' );
									}

									if ((options.defaultSelect || datetimepicker.data( 'changed' )) && dateHelper.formatDate( _xdsoft_datetime.currentTime, options.formatDate ) === dateHelper.formatDate( start, options.formatDate )) {
										classes.push( 'xdsoft_current' );
									}

									if (dateHelper.formatDate( today, options.formatDate ) === dateHelper.formatDate( start, options.formatDate )) {
										classes.push( 'xdsoft_today' );
									}

									if (start.getDay() === 0 || start.getDay() === 6 || options.weekends.indexOf( dateHelper.formatDate( start, options.formatDate ) ) !== -1) {
										classes.push( 'xdsoft_weekend' );
									}

									if (options.highlightedDates[dateHelper.formatDate( start, options.formatDate )] !== undefined) {
										hDate = options.highlightedDates[dateHelper.formatDate( start, options.formatDate )];
										classes.push( hDate.style === undefined ? 'xdsoft_highlighted_default' : hDate.style );
										description = hDate.desc === undefined ? '' : hDate.desc;
									}

									if (options.beforeShowDay && $.isFunction( options.beforeShowDay )) {
										classes.push( options.beforeShowDay( start ) );
									}

									if (newRow) {
										table += '<tr>';
										newRow = false;
										if (options.weeks) {
											table += '<th>' + w + '</th>';
										}
									}

									table += '<td data-date="' + d + '" data-month="' + m + '" data-year="' + y + '"' + ' class="xdsoft_date xdsoft_day_of_week' + start.getDay() + ' ' + classes.join( ' ' ) + '" title="' + description + '">' +
									'<div>' + d + '</div>' +
									'</td>';

									if (start.getDay() === options.dayOfWeekStartPrev) {
										table += '</tr>';
										newRow = true;
									}

									start.setDate( d + 1 );
								}
								table += '</tbody></table>';

								calendar.html( table );

								month_picker.find( '.xdsoft_label span' ).eq( 0 ).text( options.i18n[globalLocale].months[_xdsoft_datetime.currentTime.getMonth()] );
								month_picker.find( '.xdsoft_label span' ).eq( 1 ).text( _xdsoft_datetime.currentTime.getFullYear() + options.yearOffset );

								// generate timebox
								time = '';
								h    = '';
								m    = '';

								line_time = function line_time(h, m) {
									var now          = _xdsoft_datetime.now(), optionDateTime, current_time,
									isALlowTimesInit = options.allowTimes && $.isArray( options.allowTimes ) && options.allowTimes.length;
									now.setHours( h );
									h = parseInt( now.getHours(), 10 );
									now.setMinutes( m );
									m              = parseInt( now.getMinutes(), 10 );
									optionDateTime = new Date( _xdsoft_datetime.currentTime );
									optionDateTime.setHours( h );
									optionDateTime.setMinutes( m );
									classes = [];
									if ((options.minDateTime !== false && options.minDateTime > optionDateTime) || (options.maxTime !== false && _xdsoft_datetime.strtotime( options.maxTime ).getTime() < now.getTime()) || (options.minTime !== false && _xdsoft_datetime.strtotime( options.minTime ).getTime() > now.getTime())) {
										classes.push( 'xdsoft_disabled' );
									} else if ((options.minDateTime !== false && options.minDateTime > optionDateTime) || ((options.disabledMinTime !== false && now.getTime() > _xdsoft_datetime.strtotime( options.disabledMinTime ).getTime()) && (options.disabledMaxTime !== false && now.getTime() < _xdsoft_datetime.strtotime( options.disabledMaxTime ).getTime()))) {
										classes.push( 'xdsoft_disabled' );
									} else if (input.is( '[disabled]' )) {
										classes.push( 'xdsoft_disabled' );
									}

									current_time = new Date( _xdsoft_datetime.currentTime );
									current_time.setHours( parseInt( _xdsoft_datetime.currentTime.getHours(), 10 ) );

									if ( ! isALlowTimesInit) {
										current_time.setMinutes( Math[options.roundTime]( _xdsoft_datetime.currentTime.getMinutes() / options.step ) * options.step );
									}

									if ((options.initTime || options.defaultSelect || datetimepicker.data( 'changed' )) && current_time.getHours() === parseInt( h, 10 ) && (( ! isALlowTimesInit && options.step > 59) || current_time.getMinutes() === parseInt( m, 10 ))) {
										if (options.defaultSelect || datetimepicker.data( 'changed' )) {
											classes.push( 'xdsoft_current' );
										} else if (options.initTime) {
											classes.push( 'xdsoft_init_time' );
										}
									}
									if (parseInt( today.getHours(), 10 ) === parseInt( h, 10 ) && parseInt( today.getMinutes(), 10 ) === parseInt( m, 10 )) {
										classes.push( 'xdsoft_today' );
									}
									time += '<div class="xdsoft_time ' + classes.join( ' ' ) + '" data-hour="' + h + '" data-minute="' + m + '">' + dateHelper.formatDate( now, options.formatTime ) + '</div>';
								};

								if ( ! options.allowTimes || ! $.isArray( options.allowTimes ) || ! options.allowTimes.length) {
									for (i = 0, j = 0; i < (options.hours12 ? 12 : 24); i += 1) {
										for (j = 0; j < 60; j += options.step) {
											h = (i < 10 ? '0' : '') + i;
											m = (j < 10 ? '0' : '') + j;
											line_time( h, m );
										}
									}
								} else {
									for (i = 0; i < options.allowTimes.length; i += 1) {
										h = _xdsoft_datetime.strtotime( options.allowTimes[i] ).getHours();
										m = _xdsoft_datetime.strtotime( options.allowTimes[i] ).getMinutes();
										line_time( h, m );
									}
								}

								timebox.html( time );

								opt = '';

								for (i = parseInt( options.yearStart, 10 ); i <= parseInt( options.yearEnd, 10 ); i += 1) {
									opt += '<div class="xdsoft_option ' + (_xdsoft_datetime.currentTime.getFullYear() === i ? 'xdsoft_current' : '') + '" data-value="' + i + '">' + (i + options.yearOffset) + '</div>';
								}
								yearselect.children().eq( 0 )
								.html( opt );

								for (i = parseInt( options.monthStart, 10 ), opt = ''; i <= parseInt( options.monthEnd, 10 ); i += 1) {
									opt += '<div class="xdsoft_option ' + (_xdsoft_datetime.currentTime.getMonth() === i ? 'xdsoft_current' : '') + '" data-value="' + i + '">' + options.i18n[globalLocale].months[i] + '</div>';
								}
								monthselect.children().eq( 0 ).html( opt );
								$( datetimepicker )
								.trigger( 'generate.xdsoft' );
							},
							10
						);
						event.stopPropagation();
					}
				)
				.on(
					'afterOpen.xdsoft',
					function () {
						if (options.timepicker) {
							var classType, pheight, height, top;
							if (timebox.find( '.xdsoft_current' ).length) {
								classType = '.xdsoft_current';
							} else if (timebox.find( '.xdsoft_init_time' ).length) {
								classType = '.xdsoft_init_time';
							}
							if (classType) {
								pheight = timeboxparent[0].clientHeight;
								height  = timebox[0].offsetHeight;
								top     = timebox.find( classType ).index() * options.timeHeightInTimePicker + 1;
								if ((height - pheight) < top) {
									top = height - pheight;
								}
								timeboxparent.trigger( 'scroll_element.xdsoft_scroller', [parseInt( top, 10 ) / (height - pheight)] );
							} else {
								timeboxparent.trigger( 'scroll_element.xdsoft_scroller', [0] );
							}
						}
					}
				);

			timerclick = 0;
			calendar
				.on(
					'touchend click.xdsoft',
					'td',
					function (xdevent) {
						xdevent.stopPropagation();  // Prevents closing of Pop-ups, Modals and Flyouts in Bootstrap
						timerclick += 1;
						var $this   = $( this ),
						currentTime = _xdsoft_datetime.currentTime;

						if (currentTime === undefined || currentTime === null) {
							_xdsoft_datetime.currentTime = _xdsoft_datetime.now();
							currentTime                  = _xdsoft_datetime.currentTime;
						}

						if ($this.hasClass( 'xdsoft_disabled' )) {
							return false;
						}

						currentTime.setDate( 1 );
						currentTime.setFullYear( $this.data( 'year' ) );
						currentTime.setMonth( $this.data( 'month' ) );
						currentTime.setDate( $this.data( 'date' ) );

						datetimepicker.trigger( 'select.xdsoft', [currentTime] );

						input.val( _xdsoft_datetime.str() );

						if (options.onSelectDate &&	$.isFunction( options.onSelectDate )) {
							options.onSelectDate.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ), xdevent );
						}

						datetimepicker.data( 'changed', true );
						datetimepicker.trigger( 'xchange.xdsoft' );
						datetimepicker.trigger( 'changedatetime.xdsoft' );
						if ((timerclick > 1 || (options.closeOnDateSelect === true || (options.closeOnDateSelect === false && ! options.timepicker))) && ! options.inline) {
							datetimepicker.trigger( 'close.xdsoft' );
						}
						setTimeout(
							function () {
								timerclick = 0;
							},
							200
						);
					}
				);

			timebox
				.on(
					'touchstart',
					'div',
					function (xdevent) {
						this.touchMoved = false;
					}
				)
				.on( 'touchmove', 'div', handleTouchMoved )
				.on(
					'touchend click.xdsoft',
					'div',
					function (xdevent) {
						if ( ! this.touchMoved) {
							xdevent.stopPropagation();
							var $this   = $( this ),
							currentTime = _xdsoft_datetime.currentTime;

							if (currentTime === undefined || currentTime === null) {
								_xdsoft_datetime.currentTime = _xdsoft_datetime.now();
								currentTime                  = _xdsoft_datetime.currentTime;
							}

							if ($this.hasClass( 'xdsoft_disabled' )) {
								return false;
							}
							currentTime.setHours( $this.data( 'hour' ) );
							currentTime.setMinutes( $this.data( 'minute' ) );
							datetimepicker.trigger( 'select.xdsoft', [currentTime] );

							datetimepicker.data( 'input' ).val( _xdsoft_datetime.str() );

							if (options.onSelectTime && $.isFunction( options.onSelectTime )) {
								options.onSelectTime.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ), xdevent );
							}
							datetimepicker.data( 'changed', true );
							datetimepicker.trigger( 'xchange.xdsoft' );
							datetimepicker.trigger( 'changedatetime.xdsoft' );
							if (options.inline !== true && options.closeOnTimeSelect === true) {
								datetimepicker.trigger( 'close.xdsoft' );
							}
						}
					}
				);

			datepicker
				.on(
					'mousewheel.xdsoft',
					function (event) {
						if ( ! options.scrollMonth) {
							return true;
						}
						if (event.deltaY < 0) {
							_xdsoft_datetime.nextMonth();
						} else {
							_xdsoft_datetime.prevMonth();
						}
						return false;
					}
				);

			input
				.on(
					'mousewheel.xdsoft',
					function (event) {
						if ( ! options.scrollInput) {
							return true;
						}
						if ( ! options.datepicker && options.timepicker) {
							current_time_index = timebox.find( '.xdsoft_current' ).length ? timebox.find( '.xdsoft_current' ).eq( 0 ).index() : 0;
							if (current_time_index + event.deltaY >= 0 && current_time_index + event.deltaY < timebox.children().length) {
								current_time_index += event.deltaY;
							}
							if (timebox.children().eq( current_time_index ).length) {
								timebox.children().eq( current_time_index ).trigger( 'mousedown' );
							}
							return false;
						}
						if (options.datepicker && ! options.timepicker) {
							datepicker.trigger( event, [event.deltaY, event.deltaX, event.deltaY] );
							if (input.val) {
								input.val( _xdsoft_datetime.str() );
							}
							datetimepicker.trigger( 'changedatetime.xdsoft' );
							return false;
						}
					}
				);

			datetimepicker
				.on(
					'changedatetime.xdsoft',
					function (event) {
						if (options.onChangeDateTime && $.isFunction( options.onChangeDateTime )) {
							var $input = datetimepicker.data( 'input' );
							options.onChangeDateTime.call( datetimepicker, _xdsoft_datetime.currentTime, $input, event );
							delete options.value;
							$input.trigger( 'change' );
						}
					}
				)
				.on(
					'generate.xdsoft',
					function () {
						if (options.onGenerate && $.isFunction( options.onGenerate )) {
							options.onGenerate.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ) );
						}
						if (triggerAfterOpen) {
							datetimepicker.trigger( 'afterOpen.xdsoft' );
							triggerAfterOpen = false;
						}
					}
				)
				.on(
					'click.xdsoft',
					function (xdevent) {
						xdevent.stopPropagation();
					}
				);

			current_time_index = 0;

			/**
			 * Runs the callback for each of the specified node's ancestors.
			 *
			 * Return FALSE from the callback to stop ascending.
			 *
			 * @param {DOMNode} node
			 * @param {Function} callback
			 * @returns {undefined}
			 */
			forEachAncestorOf = function (node, callback) {
				do {
					node = node.parentNode;

					if ( ! node || callback( node ) === false) {
						break;
					}
				} while (node.nodeName !== 'HTML');
			};

			/**
			 * Sets the position of the picker.
			 *
			 * @returns {undefined}
			 */
			setPos = function () {
				var dateInputOffset,
					dateInputElem,
					verticalPosition,
					left,
					position,
					datetimepickerElem,
					dateInputHasFixedAncestor,
					$dateInput,
					windowWidth,
					verticalAnchorEdge,
					datetimepickerCss,
					windowHeight,
					windowScrollTop;

				$dateInput      = datetimepicker.data( 'input' );
				dateInputOffset = $dateInput.offset();
				dateInputElem   = $dateInput[0];

				verticalAnchorEdge = 'top';
				verticalPosition   = (dateInputOffset.top + dateInputElem.offsetHeight) - 1;
				left               = dateInputOffset.left;
				position           = "absolute";

				windowWidth     = $( options.contentWindow ).width();
				windowHeight    = $( options.contentWindow ).height();
				windowScrollTop = $( options.contentWindow ).scrollTop();

				if ((options.ownerDocument.documentElement.clientWidth - dateInputOffset.left) < datepicker.parent().outerWidth( true )) {
					var diff = datepicker.parent().outerWidth( true ) - dateInputElem.offsetWidth;
					left     = left - diff;
				}

				if ($dateInput.parent().css( 'direction' ) === 'rtl') {
					left -= (datetimepicker.outerWidth() - $dateInput.outerWidth());
				}

				if (options.fixed) {
					verticalPosition -= windowScrollTop;
					left             -= $( options.contentWindow ).scrollLeft();
					position          = "fixed";
				} else {
					dateInputHasFixedAncestor = false;

					forEachAncestorOf(
						dateInputElem,
						function (ancestorNode) {
							if (ancestorNode === null) {
								return false;
							}

							if (options.contentWindow.getComputedStyle( ancestorNode ).getPropertyValue( 'position' ) === 'fixed') {
								dateInputHasFixedAncestor = true;
								return false;
							}
						}
					);

					if (dateInputHasFixedAncestor) {
						position = 'fixed';

						//If the picker won't fit entirely within the viewport then display it above the date input.
						if (verticalPosition + datetimepicker.outerHeight() > windowHeight + windowScrollTop) {
							verticalAnchorEdge = 'bottom';
							verticalPosition   = (windowHeight + windowScrollTop) - dateInputOffset.top;
						} else {
							verticalPosition -= windowScrollTop;
						}
					} else {
						if (verticalPosition + datetimepicker[0].offsetHeight > windowHeight + windowScrollTop) {
							verticalPosition = dateInputOffset.top - datetimepicker[0].offsetHeight + 1;
						}
					}

					if (verticalPosition < 0) {
						verticalPosition = 0;
					}

					if (left + dateInputElem.offsetWidth > windowWidth) {
						left = windowWidth - dateInputElem.offsetWidth;
					}
				}

				datetimepickerElem = datetimepicker[0];

				forEachAncestorOf(
					datetimepickerElem,
					function (ancestorNode) {
						var ancestorNodePosition;

						ancestorNodePosition = options.contentWindow.getComputedStyle( ancestorNode ).getPropertyValue( 'position' );

						if (ancestorNodePosition === 'relative' && windowWidth >= ancestorNode.offsetWidth) {
							left = left - ((windowWidth - ancestorNode.offsetWidth) / 2);
							return false;
						}
					}
				);

				datetimepickerCss = {
					position: position,
					left: left,
					top: '',  //Initialize to prevent previous values interfering with new ones.
					bottom: ''  //Initialize to prevent previous values interfering with new ones.
				};

				datetimepickerCss[verticalAnchorEdge] = verticalPosition;

				datetimepicker.css( datetimepickerCss );
			};

			datetimepicker
				.on(
					'open.xdsoft',
					function (event) {
						var onShow = true;
						if (options.onShow && $.isFunction( options.onShow )) {
							onShow = options.onShow.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ), event );
						}
						if (onShow !== false) {
							datetimepicker.show();
							setPos();
							$( options.contentWindow )
							.off( 'resize.xdsoft', setPos )
							.on( 'resize.xdsoft', setPos );

							if (options.closeOnWithoutClick) {
								$( [options.ownerDocument.body, options.contentWindow] ).on(
									'touchstart mousedown.xdsoft',
									function arguments_callee6() {
										datetimepicker.trigger( 'close.xdsoft' );
										$( [options.ownerDocument.body, options.contentWindow] ).off( 'touchstart mousedown.xdsoft', arguments_callee6 );
									}
								);
							}
						}
					}
				)
				.on(
					'close.xdsoft',
					function (event) {
						var onClose = true;
						month_picker
						.find( '.xdsoft_month,.xdsoft_year' )
						.find( '.xdsoft_select' )
						.hide();
						if (options.onClose && $.isFunction( options.onClose )) {
							onClose = options.onClose.call( datetimepicker, _xdsoft_datetime.currentTime, datetimepicker.data( 'input' ), event );
						}
						if (onClose !== false && ! options.opened && ! options.inline) {
							datetimepicker.hide();
						}
						event.stopPropagation();
					}
				)
				.on(
					'toggle.xdsoft',
					function () {
						if (datetimepicker.is( ':visible' )) {
							datetimepicker.trigger( 'close.xdsoft' );
						} else {
							datetimepicker.trigger( 'open.xdsoft' );
						}
					}
				)
				.data( 'input', input );

			timer = 0;

			datetimepicker.data( 'xdsoft_datetime', _xdsoft_datetime );
			datetimepicker.setOptions( options );

			function getCurrentValue() {
				var ct = false, time;

				if (options.startDate) {
					ct = _xdsoft_datetime.strToDate( options.startDate );
				} else {
					ct = options.value || ((input && input.val && input.val()) ? input.val() : '');
					if (ct) {
						ct = _xdsoft_datetime.strToDateTime( ct );
						if (options.yearOffset) {
							ct = new Date( ct.getFullYear() - options.yearOffset, ct.getMonth(), ct.getDate(), ct.getHours(), ct.getMinutes(), ct.getSeconds(), ct.getMilliseconds() );
						}
					} else if (options.defaultDate) {
						ct = _xdsoft_datetime.strToDateTime( options.defaultDate );
						if (options.defaultTime) {
							time = _xdsoft_datetime.strtotime( options.defaultTime );
							ct.setHours( time.getHours() );
							ct.setMinutes( time.getMinutes() );
						}
					}
				}

				if (ct && _xdsoft_datetime.isValidDate( ct )) {
					datetimepicker.data( 'changed', true );
				} else {
					ct = '';
				}

				return ct || 0;
			}

			function setMask(options) {

				var isValidValue = function (mask, value) {
						var reg = mask
							.replace( /([\[\]\/\{\}\(\)\-\.\+]{1})/g, '\\$1' )
							.replace( /_/g, '{digit+}' )
							.replace( /([0-9]{1})/g, '{digit$1}' )
							.replace( /\{digit([0-9]{1})\}/g, '[0-$1_]{1}' )
							.replace( /\{digit[\+]\}/g, '[0-9_]{1}' );
						return (new RegExp( reg )).test( value );
				},
					getCaretPos  = function (input) {
						try {
							if (options.ownerDocument.selection && options.ownerDocument.selection.createRange) {
								var range = options.ownerDocument.selection.createRange();
								return range.getBookmark().charCodeAt( 2 ) - 2;
							}
							if (input.setSelectionRange) {
								return input.selectionStart;
							}
						} catch (e) {
							return 0;
						}
					},
					setCaretPos  = function (node, pos) {
						node = (typeof node === "string" || node instanceof String) ? options.ownerDocument.getElementById( node ) : node;
						if ( ! node) {
							return false;
						}
						if (node.createTextRange) {
							var textRange = node.createTextRange();
							textRange.collapse( true );
							textRange.moveEnd( 'character', pos );
							textRange.moveStart( 'character', pos );
							textRange.select();
							return true;
						}
						if (node.setSelectionRange) {
							node.setSelectionRange( pos, pos );
							return true;
						}
						return false;
					};

				if (options.mask) {
					input.off( 'keydown.xdsoft' );
				}

				if (options.mask === true) {
					if (dateHelper.formatMask) {
						options.mask = dateHelper.formatMask( options.format )
					} else {
						options.mask = options.format
							.replace( /Y/g, '9999' )
							.replace( /F/g, '9999' )
							.replace( /m/g, '19' )
							.replace( /d/g, '39' )
							.replace( /H/g, '29' )
							.replace( /i/g, '59' )
							.replace( /s/g, '59' );
					}
				}

				if ($.type( options.mask ) === 'string') {
					if ( ! isValidValue( options.mask, input.val() )) {
						input.val( options.mask.replace( /[0-9]/g, '_' ) );
						setCaretPos( input[0], 0 );
					}

					input.on(
						'paste.xdsoft',
						function (event) {
							// couple options here
							// 1. return false - tell them they can't paste
							// 2. insert over current characters - minimal validation
							// 3. full fledged parsing and validation
							// let's go option 2 for now

							// fires multiple times for some reason

							// https://stackoverflow.com/a/30496488/1366033
							var clipboardData = event.clipboardData || event.originalEvent.clipboardData || window.clipboardData,
							pastedData        = clipboardData.getData( 'text' ),
							val               = this.value,
							pos               = this.selectionStart

							var valueBeforeCursor = val.substr( 0, pos );
							var valueAfterPaste   = val.substr( pos + pastedData.length );

							val  = valueBeforeCursor + pastedData + valueAfterPaste;
							pos += pastedData.length;

							if (isValidValue( options.mask, val )) {
								this.value = val;
								setCaretPos( this, pos );
							} else if ($.trim( val ) === '') {
								this.value = options.mask.replace( /[0-9]/g, '_' );
							} else {
								input.trigger( 'error_input.xdsoft' );
							}

							event.preventDefault();
							return false;
						}
					);

					input.on(
						'keydown.xdsoft',
						function (event) {
							var val = this.value,
							key     = event.which,
							pos     = this.selectionStart,
							selEnd  = this.selectionEnd,
							hasSel  = pos !== selEnd,
							digit;

							// only alow these characters
							if (((key >= KEY0 && key <= KEY9) ||
							(key >= _KEY0 && key <= _KEY9)) ||
							(key === BACKSPACE || key === DEL)) {

									// get char to insert which is new character or placeholder ('_')
									digit = (key === BACKSPACE || key === DEL) ? '_' :
								String.fromCharCode( (_KEY0 <= key && key <= _KEY9) ? key - KEY0 : key );

									// we're deleting something, we're not at the start, and have normal cursor, move back one
									// if we have a selection length, cursor actually sits behind deletable char, not in front
								if (key === BACKSPACE && pos && ! hasSel) {
									pos -= 1;
								}

								// don't stop on a separator, continue whatever direction you were going
								//   value char - keep incrementing position while on separator char and we still have room
								//   del char   - keep decrementing position while on separator char and we still have room
								while (true) {
									var maskValueAtCurPos        = options.mask.substr( pos, 1 );
									var posShorterThanMaskLength = pos < options.mask.length;
									var posGreaterThanZero       = pos > 0;
									var notNumberOrPlaceholder   = /[^0-9_]/;
									var curPosOnSep              = notNumberOrPlaceholder.test( maskValueAtCurPos );
									var continueMovingPosition   = curPosOnSep && posShorterThanMaskLength && posGreaterThanZero

									// if we hit a real char, stay where we are
									if ( ! continueMovingPosition) {
										break;
									}

									// hitting backspace in a selection, you can possibly go back any further - go forward
									pos += (key === BACKSPACE && ! hasSel) ? -1 : 1;

								}

								if (hasSel) {
									// pos might have moved so re-calc length
									var selLength = selEnd - pos

									// if we have a selection length we will wipe out entire selection and replace with default template for that range
									var defaultBlank                     = options.mask.replace( /[0-9]/g, '_' );
									var defaultBlankSelectionReplacement = defaultBlank.substr( pos, selLength );
									var selReplacementRemainder          = defaultBlankSelectionReplacement.substr( 1 ) // might be empty

									var valueBeforeSel      = val.substr( 0, pos );
									var insertChars         = digit + selReplacementRemainder;
									var charsAfterSelection = val.substr( pos + selLength );

									val = valueBeforeSel + insertChars + charsAfterSelection

								} else {
									var valueBeforeCursor  = val.substr( 0, pos );
									var insertChar         = digit;
									var valueAfterNextChar = val.substr( pos + 1 );

									val = valueBeforeCursor + insertChar + valueAfterNextChar
								}

								if ($.trim( val ) === '') {
									// if empty, set to default
									val = defaultBlank
								} else {
									// if at the last character don't need to do anything
									if (pos === options.mask.length) {
										event.preventDefault();
										return false;
									}
								}

								// resume cursor location
								pos += (key === BACKSPACE) ? 0 : 1;
								// don't stop on a separator, continue whatever direction you were going
								while (/[^0-9_]/.test( options.mask.substr( pos, 1 ) ) && pos < options.mask.length && pos > 0) {
									pos += (key === BACKSPACE) ? 0 : 1;
								}

								if (isValidValue( options.mask, val )) {
									this.value = val;
									setCaretPos( this, pos );
								} else if ($.trim( val ) === '') {
									this.value = options.mask.replace( /[0-9]/g, '_' );
								} else {
									input.trigger( 'error_input.xdsoft' );
								}
							} else {
								if (([AKEY, CKEY, VKEY, ZKEY, YKEY].indexOf( key ) !== -1 && ctrlDown) || [ESC, ARROWUP, ARROWDOWN, ARROWLEFT, ARROWRIGHT, F5, CTRLKEY, TAB, ENTER].indexOf( key ) !== -1) {
									return true;
								}
							}

							event.preventDefault();
							return false;
						}
					);
				}
			}

			_xdsoft_datetime.setCurrentTime( getCurrentValue() );

			input
				.data( 'xdsoft_datetimepicker', datetimepicker )
				.on(
					'open.xdsoft focusin.xdsoft mousedown.xdsoft touchstart',
					function () {
						if (input.is( ':disabled' ) || (input.data( 'xdsoft_datetimepicker' ).is( ':visible' ) && options.closeOnInputClick)) {
							return;
						}
						if ( ! options.openOnFocus) {
							return;
						}
						clearTimeout( timer );
						timer = setTimeout(
							function () {
								if (input.is( ':disabled' )) {
									return;
								}

								triggerAfterOpen = true;
								_xdsoft_datetime.setCurrentTime( getCurrentValue(), true );
								if (options.mask) {
									setMask( options );
								}
								datetimepicker.trigger( 'open.xdsoft' );
							},
							100
						);
					}
				)
				.on(
					'keydown.xdsoft',
					function (event) {
						var elementSelector,
						key = event.which;
						if ([ENTER].indexOf( key ) !== -1 && options.enterLikeTab) {
							elementSelector = $( "input:visible,textarea:visible,button:visible,a:visible" );
							datetimepicker.trigger( 'close.xdsoft' );
							elementSelector.eq( elementSelector.index( this ) + 1 ).focus();
							return false;
						}
						if ([TAB].indexOf( key ) !== -1) {
							datetimepicker.trigger( 'close.xdsoft' );
							return true;
						}
					}
				)
				.on(
					'blur.xdsoft',
					function () {
						datetimepicker.trigger( 'close.xdsoft' );
					}
				);
		};
		destroyDateTimePicker = function (input) {
			var datetimepicker = input.data( 'xdsoft_datetimepicker' );
			if (datetimepicker) {
				datetimepicker.data( 'xdsoft_datetime', null );
				datetimepicker.remove();
				input
					.data( 'xdsoft_datetimepicker', null )
					.off( '.xdsoft' );
				$( options.contentWindow ).off( 'resize.xdsoft' );
				$( [options.contentWindow, options.ownerDocument.body] ).off( 'mousedown.xdsoft touchstart' );
				if (input.unmousewheel) {
					input.unmousewheel();
				}
			}
		};
		$( options.ownerDocument )
			.off( 'keydown.xdsoftctrl keyup.xdsoftctrl' )
			.on(
				'keydown.xdsoftctrl',
				function (e) {
					if (e.keyCode === CTRLKEY) {
						ctrlDown = true;
					}
				}
			)
			.on(
				'keyup.xdsoftctrl',
				function (e) {
					if (e.keyCode === CTRLKEY) {
						ctrlDown = false;
					}
				}
			);

		this.each(
			function () {
				var datetimepicker = $( this ).data( 'xdsoft_datetimepicker' ), $input;
				if (datetimepicker) {
					if ($.type( opt ) === 'string') {
						switch (opt) {
							case 'show':
								$( this ).select().focus();
								datetimepicker.trigger( 'open.xdsoft' );
								break;
							case 'hide':
								datetimepicker.trigger( 'close.xdsoft' );
								break;
							case 'toggle':
								datetimepicker.trigger( 'toggle.xdsoft' );
								break;
							case 'destroy':
								destroyDateTimePicker( $( this ) );
								break;
							case 'reset':
								this.value = this.defaultValue;
								if ( ! this.value || ! datetimepicker.data( 'xdsoft_datetime' ).isValidDate( dateHelper.parseDate( this.value, options.format ) )) {
									datetimepicker.data( 'changed', false );
								}
								datetimepicker.data( 'xdsoft_datetime' ).setCurrentTime( this.value );
								break;
							case 'validate':
								$input = datetimepicker.data( 'input' );
								$input.trigger( 'blur.xdsoft' );
								break;
							default:
								if (datetimepicker[opt] && $.isFunction( datetimepicker[opt] )) {
									result = datetimepicker[opt]( opt2 );
								}
						}
					} else {
						datetimepicker
						.setOptions( opt );
					}
					return 0;
				}
				if ($.type( opt ) !== 'string') {
					if ( ! options.lazyInit || options.open || options.inline) {
						createDateTimePicker( $( this ) );
					} else {
						lazyInit( $( this ) );
					}
				}
			}
		);

		return result;
	};

	$.fn.datetimepicker.defaults = default_options;

	function HighlightedDate(date, desc, style) {
		"use strict";
		this.date  = date;
		this.desc  = desc;
		this.style = style;
	}
};
;(function (factory) {
	if ( typeof define === 'function' && define.amd ) {
		// AMD. Register as an anonymous module.
		define( ['jquery', 'jquery-mousewheel'], factory );
	} else if (typeof exports === 'object') {
		// Node/CommonJS style for Browserify
		module.exports = factory( require( 'jquery' ) );;
	} else {
		// Browser globals
		factory( jQuery );
	}
}(datetimepickerFactory));
