<?php
namespace BricksUltimate\Elements;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use BricksUltimate\Helpers;

class WooCartCounter extends \Bricks\Element {
	public $category 		= 'cart';
	public $name 			= 'bu-cart-counter';
	public $icon 			= 'ti-quote-right';
	public $scripts 		= [ 'buCartCounter' ];
	public $nestable 		= true;

	// Return localized element label
	public function get_label() {
		return esc_html__( '(BU)Cart Counter', 'bricksultimate' );
	}

	public function get_keywords() {
		return [ 'cart', 'counter', 'count', 'cart link', 'cart total' ];
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['button']    = array(
			'title' => esc_html__( 'Icon/Text', 'bricks' ),
			'tab'   => 'content',
		);

		$this->control_groups['counter']    = array(
			'title' => esc_html__( 'Counter', 'bricks' ),
			'tab'   => 'content',
		);

		$this->control_groups['price']    = array(
			'title' => esc_html__( 'Price', 'bricks' ),
			'tab'   => 'content',
		);

		$this->control_groups['dropdown']    = array(
			'title' => esc_html__( 'Dropdown', 'bricks' ),
			'tab'   => 'content',
			'required' => ['cta', '=', 'popup']
		);
	}

	// Set builder controls
	public function set_controls() {
		$this->controls['visibility'] = array(
			'tab'   	=> 'content',
			'type' 		=> 'checkbox',
			'label' 	=> esc_html__('Hide when cart is empty?', 'bricksultimate' ),
			'info' 		=> esc_html__('It only works at frontend. Editor always show the button.', 'bricksultimate'),
			'reset' 	=> true
		);

		$this->controls['counterType'] = array(
			'type' 		=> 'select',
			'label' 	=> esc_html__('Count', 'bricksultimate' ),
			'default' 	=> 'quantities',
			'options' 	=> [
				'itemnumber' 	=> esc_html__( 'Number of items', 'bricksultimate' ),
				'quantities' 	=> esc_html__( 'Number of quantities', 'bricksultimate' )
			],
			'placeholder' => esc_html__( 'Number of items quantities', 'bricksultimate' ),
			'description' => esc_html__('Calculate the number of items within the cart.', 'bricksultimate'),
			'inline' 	=> true
		);

		$this->controls['cta'] = array(
			'tab'   	=> 'content',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Call to action', 'bricksultimate' ),
			'default' 	=> 'none',
			'options' 	=> [
				'none' 		=> esc_html__( 'None', 'bricksultimate' ),
				'popup' 	=> esc_html__( 'Show dropdown(on hover/click)', 'bricksultimate' ),
				'link' 		=> esc_html__( 'Link to URL', 'bricksultimate' ),
			]
		);

		$this->controls['pageLink'] = array(
			'tab' 		=> 'content',
			'type' 		=> 'link',
			'label'		=> esc_html__( 'URL', 'bricksultimate' ),
			'exclude' 	=> [
				'title',
				'ariaLabel',
				'newTab',
				'rel'
			],
			'required' 	=> [ 'cta', '=', 'link' ]
		);

		$this->controls['triggerEvent'] = array(
			'tab'   	=> 'content',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Trigger on', 'bricksultimate' ),
			'default' 	=> 'hover',
			'options' 	=> [
				'hover' => esc_html__( 'Hover', 'bricksultimate' ),
				'click' => esc_html__( 'Click', 'bricksultimate' ),
			],
			'required' 	=> [ 'cta', '=', 'popup' ]
		);

		$this->controls['ariaLabel'] = array(
			'tab' 		=> 'content',
			'type' 		=> 'text',
			'label'		=> esc_html__( 'Aria label', 'bricksultimate' ),
		);


		/**
		 * Button settings
		 */
		$this->controls['contentSource'] = array(
			'tab'   	=> 'content',
			'group' 	=> 'button',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Type', 'bricksultimate' ),
			'default' 	=> 'icon',
			'options' 	=> [
				'text' 		=> esc_html__( 'Text', 'bricksultimate' ),
				'icon' 		=> esc_html__( 'Icon', 'bricksultimate' ),
				'icontext' 	=> esc_html__( 'Icon + Text', 'bricksultimate' ),
			],
			'inline' 	=> true,
			'medium' 	=> true,
		);

		$this->controls['btnSep'] = array(
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type' 		=> 'separator',
			'required' 	=> [ 'contentSource', '!=', 'icon' ]
		);

		$this->controls['buttonText'] = array(
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type' 		=> 'text',
			'label'		=> esc_html__( 'Button Text', 'bricksultimate' ),
			'required' 	=> [ 'contentSource', '!=', 'icon' ]
		);

		$this->controls['buttonTg'] = array(
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type' 		=> 'typography',
			'label'		=> esc_html__( 'Typography', 'bricks' ),
			'required' 	=> [ 'contentSource', '!=', 'icon' ],
			'css' 		=> [
				[
					'property' => 'font',
					'selector' => '.bu-cart-counter-btn-text'
				]
			]
		);

		$this->controls['btnTxtHoverColor'] = [
			'tab'      => 'content',
			'group' 	=> 'button',
			'type'     => 'color',
			'label'    => esc_html__( 'Hover Color', 'bricks' ),
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.bu-cart-counter-btn:hover .bu-cart-counter-btn-text',
				],
			],
			'required' 	=> [ 'contentSource', '!=', 'icon' ],
		];

		$this->controls['iconSep'] = array(
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type' 		=> 'separator',
			'label'   	=> esc_html__( 'Icon', 'bricks' ),
			'required' 	=> [ 'contentSource', '!=', 'text' ]
		);

		$this->controls['cartIcon'] = [
			'tab'     => 'content',
			'group' 	=> 'button',
			'label'   => esc_html__( 'Icon', 'bricks' ),
			'type'    => 'icon',
			'default' => [
				'library' => 'themify',
				'icon'    => 'ti-shopping-cart',
			],
			'required' 	=> [ 'contentSource', '!=', 'text' ],
		];

		$this->controls['iconPosition'] = array(
			'tab'   	=> 'content',
			'group' 	=> 'button',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Icon Position', 'bricksultimate' ),
			'default' 	=> 'right',
			'options' 	=> [
				'left' 		=> esc_html__( 'Left', 'bricks' ),
				'right' 	=> esc_html__( 'Right', 'bricks' )
			],
			'inline' 	=> true,
			'medium' 	=> true,
			'required' 	=> [ 'contentSource', '=', 'icontext' ]
		);

		$this->controls['iconColor'] = [
			'tab'      => 'content',
			'group' 	=> 'button',
			'label'    => esc_html__( 'Color', 'bricks' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.bu-cart-counter-btn-icon'
				],
				[
					'property' => 'fill',
					'selector' => '.bu-cart-counter-btn-icon'
				],
			],
			'required' 	=> [ 'contentSource', '!=', 'text' ]
		];

		$this->controls['iconHoverColor'] = [
			'tab'      => 'content',
			'group' 	=> 'button',
			'label'    => esc_html__( 'Hover Color', 'bricks' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.bu-cart-counter-btn:hover .bu-cart-counter-btn-icon'
				],
				[
					'property' => 'fill',
					'selector' => '.bu-cart-counter-btn:hover .bu-cart-counter-btn-icon'
				],
			],
			'required' 	=> [ 'contentSource', '!=', 'text' ]
		];

		$this->controls['iconSize'] = [
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'label' 	=> esc_html__( 'Size', 'bricks' ),
			'type' 		=> 'number',
			'units' 	=> true,
			'css' 		=> [
				[
					'property' => 'font-size',
					'selector' => '.bu-cart-counter-btn-icon'
				],
			],
			'placeholder' => '22px',
			'required' 	=> [ 'contentSource', '!=', 'text' ]
		];


		$this->controls['btnStyleSep'] = array(
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type' 		=> 'separator',
			'label' 	=> esc_html__('Wrapper', 'bricksultimate')
		);

		$selector = '.bucc-btn-contents';

		$this->controls['btnWidth'] = [
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type'      => 'number',
			'label'     => esc_html__('Width', 'bricks'),
			'units' 	=> true,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'width'
				]
			]
		];

		$this->controls['btnHeight'] = [
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type'      => 'number',
			'label'     => esc_html__('Height', 'bricks'),
			'units' 	=> true,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'height'
				]
			]
		];

		$this->controls['btnPad'] = [
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type'      => version_compare( BRICKS_VERSION, '1.5', '>') ? 'spacing' : 'dimensions',
			'label'     => esc_html__('Padding', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'padding'
				]
			]
		];

		$this->controls['btnBG'] = [
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type'      => 'color',
			'label'     => esc_html__('Background', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'background-color'
				]
			]
		];

		$this->controls['btnHBG'] = [
			'tab' 		=> 'content',
			'group' 	=> 'button',
			'type'      => 'color',
			'label'     => esc_html__('Hover Background', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector . ":hover",
					'property' 	=> 'background-color'
				]
			]
		];

		/**
		 * Counter settings
		 */
		$selector = '.bu-cart-counter';
		$this->controls['hideCounter'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'checkbox',
			'label' => esc_html__('Completely hide counter', 'bricksultimate')
		];

		$this->controls['hideCounterIsEmpty'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'checkbox',
			'label' => esc_html__('Hide counter when cart empty', 'bricksultimate')
		];

		$this->controls['bubble'] = [
			'tab' 		=> 'content',
			'group' 	=> 'counter',
			'type' 		=> 'checkbox',
			'label' 	=> esc_html__('Show in bubble', 'bricksultimate'),
			'default'	=> true
		];

		$this->controls['counterPosition'] = array(
			'tab'   	=> 'content',
			'group' 	=> 'counter',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Position', 'bricks' ),
			'default' 	=> 'left',
			'options' 	=> [
				'left' 		=> esc_html__( 'Left', 'bricks' ),
				'right' 	=> esc_html__( 'Right', 'bricks' )
			],
			'inline' 	=> true,
			'medium' 	=> true,
			'required' 	=> ['bubble', '!=', true]
		);

		$this->controls['counterGap'] = [
			'tab' 		=> 'content',
			'group' 	=> 'counter',
			'type'      => 'number',
			'label'     => esc_html__('Gap(px)', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100,
			'step' 		=> 1,
			'default'   => 6,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> ".bu-cart-counter--right .bu-cart-counter",
					'property' 	=> 'margin-left',
					'value'		=> '%spx'
				],
				[
					'selector' 	=> ".bu-cart-counter--left .bu-cart-counter",
					'property' 	=> 'margin-right',
					'value'		=> '%spx'
				]
			],
			'required' 	=> ['bubble', '!=', true]
		];

		$this->controls['bubbleTop'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'number',
			'label' => esc_html__('Position top', 'bricksultimate'),
			'units' => true,
			'placeholder' => '5px',
			'css'	=> [
				[
					'property' 	=> 'top',
					'selector' 	=> $selector
				]
			],
			'required' 	=> ['bubble', '=', true]
		];

		$this->controls['bubbleRight'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'number',
			'label' => esc_html__('Position right', 'bricksultimate'),
			'units' => true,
			'placeholder' => '-4px',
			'css'	=> [
				[
					'property' 	=> 'right',
					'selector' 	=> $selector
				]
			],
			'required' 	=> ['bubble', '=', true]
		];

		$this->controls['counterWidth'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'number',
			'label' => esc_html__('Width', 'bricks'),
			'units' => true,
			'inline' => true,
			'css'	=> [
				[
					'property' 	=> 'width',
					'selector' 	=> $selector
				]
			],
			'required' 	=> ['bubble', '!=', true]
		];

		$this->controls['counterHeight'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'number',
			'label' => esc_html__('Height', 'bricks'),
			'units' => true,
			'css'	=> [
				[
					'property' 	=> 'height',
					'selector' 	=> $selector
				]
			],
			'required' 	=> ['bubble', '!=', true]
		];

		$this->controls['bubbleSize'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'number',
			'label' => esc_html__('Bubble size', 'bricksultimate'),
			'units' => true,
			'placeholder' => '16px',
			'css'	=> [
				[
					'property' 	=> 'width',
					'selector' 	=> ".bu-cart-counter-in-bubble {$selector}"
				],
				[
					'property' 	=> 'height',
					'selector' 	=> ".bu-cart-counter-in-bubble {$selector}"
				],
				[
					'property' 	=> 'line-height',
					'selector' 	=> ".bu-cart-counter-in-bubble {$selector}"
				]
			],
			'required' 	=> ['bubble', '=', true]
		];

		$this->controls['counterBG'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'color',
			'label' => esc_html__('Background', 'bricks'),
			'css'	=> [
				[
					'property' 	=> 'background-color',
					'selector' 	=> $selector
				]
			]
		];

		$this->controls['counterHBG'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'color',
			'label' => esc_html__('Hover background', 'bricks'),
			'css'	=> [
				[
					'property' 	=> 'background-color',
					'selector' 	=> '&:hover ' . $selector
				]
			]
		];

		$this->controls['counterTG'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'typography',
			'label' => esc_html__('Typography', 'bricks'),
			'css'	=> [
				[
					'property' 	=> 'font',
					'selector' 	=> $selector
				]
			]
		];

		$this->controls['counterHClr'] = [
			'tab' 	=> 'content',
			'group' => 'counter',
			'type' 	=> 'color',
			'label' => esc_html__('Hover color', 'bricks'),
			'css'	=> [
				[
					'property' 	=> 'color',
					'selector' 	=> '&:hover ' . $selector
				]
			]
		];

		/**
		 * Price settings
		 */
		$selector =".bu-cart-subtotal";
		$this->controls['showTotals'] = [
			'tab' 	=> 'content',
			'group' => 'price',
			'type' 	=> 'checkbox',
			'label' => esc_html__('Show Price', 'bricksultimate'),
			'inline' => true,
			'reset'	=> true
		];

		$this->controls['showOrderTotal'] = [
			'tab' 	=> 'content',
			'group' => 'price',
			'type' 	=> 'checkbox',
			'label' => esc_html__('Show Total Price instead of Sub Price', 'bricksultimate'),
			'inline' => true,
			'reset'	=> true
		];

		$this->controls['pricePosition'] = array(
			'tab'   	=> 'content',
			'group' 	=> 'price',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Position', 'bricksultimate' ),
			'default' 	=> 'left',
			'options' 	=> [
				'left' 		=> esc_html__( 'Left', 'bricks' ),
				'right' 	=> esc_html__( 'Right', 'bricks' )
			],
			'inline' 	=> true,
			'medium' 	=> true
		);

		$this->controls['gapRight'] = [
			'tab' 		=> 'content',
			'group' 	=> 'price',
			'type'      => 'number',
			'label'     => esc_html__('Gap(px)', 'bricksultimate'),
			'unit' 		=> false,
			'min'		=> 0,
			'max'  		=> 100,
			'step' 		=> 1,
			'default'   => 6,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> ".bu-cart-price--left .bu-cart-subtotal",
					'property' 	=> 'padding-right',
					'value'		=> '%spx'
				],
				[
					'selector' 	=> ".bu-cart-price--left .bu-cart-order-total",
					'property' 	=> 'padding-right',
					'value'		=> '%spx'
				]
			],
			'required'	=> ['pricePosition', '=', 'left']
		];

		$this->controls['gapLeft'] = [
			'tab' 		=> 'content',
			'group' 	=> 'price',
			'type'      => 'number',
			'label'     => esc_html__('Gap(px)', 'bricksultimate'),
			'unit' 		=> false,
			'min'		=> 0,
			'max'  		=> 100,
			'step' 		=> 1,
			'default'   => 6,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> ".bu-cart-price--right .bu-cart-subtotal",
					'property' 	=> 'padding-left',
					'value'		=> '%spx'
				],
				[
					'selector' 	=> ".bu-cart-price--right .bu-cart-order-total",
					'property' 	=> 'padding-left',
					'value'		=> '%spx'
				]
			],
			'required'	=> ['pricePosition', '=', 'right']
		];

		$this->controls['priceFont'] = [
			'tab' 		=> 'content',
			'group' 	=> 'price',
			'type'      => 'typography',
			'label' 	=> esc_html__('Typography', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'font'
				],
				[
					'selector' 	=> '.bu-cart-order-total',
					'property' 	=> 'font'
				]
			]
		];

		$this->controls['priceHoverColor'] = [
			'tab' 		=> 'content',
			'group' 	=> 'price',
			'type'      => 'color',
			'label' 	=> esc_html__('Hover Color', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> '&:hover ' . $selector,
					'property' 	=> 'color'
				],
				[
					'selector' 	=> '&:hover .bu-cart-order-total',
					'property' 	=> 'color'
				]
			]
		];

		/**
		 * Dropdown settings
		 */
		$selector = ".bu-cart-counter-popup";
		$this->controls['popupPreview'] = [
			'tab' 	=> 'content',
			'group' => 'dropdown',
			'type' 	=> 'checkbox',
			'label' => esc_html__('Preview In-Builder Mode', 'bricksultimate'),
			'default'	=> true
		];

		$this->controls['hidePopup'] = [
			'tab' 	=> 'content',
			'group' => 'dropdown',
			'type' 	=> 'checkbox',
			'label' => esc_html__('Hide it when cart is empty', 'bricksultimate'),
			'reset'	=> true
		];

		$this->controls['popupWidth'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'number',
			'label'     => esc_html__('Width', 'bricks'),
			'unit' 		=> 'px',
			'min'		=> 0,
			'max'  		=> 10000,
			'step' 		=> 1,
			'placeholder' => 300,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'width'
				]
			]
		];

		$this->controls['popupHeight'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type' 		=> 'number',
			'label' 	=> esc_html__('Height', 'bricks'),
			'units' 	=> true,
			'css' 		=> [
				[
					'property' 	=> 'height',
					'selector' 	=> $selector
				]
			]
		];

		$this->controls['popupMargin'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => version_compare( BRICKS_VERSION, '1.5', '>') ? 'spacing' : 'dimensions',
			'label'     => esc_html__('Margin', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'margin'
				]
			]
		];

		$this->controls['popupPad'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => version_compare( BRICKS_VERSION, '1.5', '>') ? 'spacing' : 'dimensions',
			'label'     => esc_html__('Padding', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'padding'
				]
			]
		];

		$this->controls['popupBG'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'color',
			'label'     => esc_html__('Background', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'background-color'
				]
			]
		];

		$this->controls['popupBorder'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'border',
			'label'     => esc_html__('Border', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'border'
				]
			]
		];

		$this->controls['popupBS'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'box-shadow',
			'label'     => esc_html__('Box shadow', 'bricks'),
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'box-shadow'
				]
			]
		];

		$this->controls['popupAlign'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'select',
			'label'     => esc_html__('Alignment', 'bricks'),
			'options' 		=> [
				'left' 		=> esc_html__('Left'),
				'right' 	=> esc_html__('Right'),
				'center' 	=> esc_html__('Center')
			],
			'default' 	=> 'right',
			'inline' 	=> true,
			'medium' 	=> true
		];

		$this->controls['animSep'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'separator',
			'label' 	=> esc_html__('Fade Animation', 'bricks')
		];

		$this->controls['popupTd'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'number',
			'label'     => esc_html__('Transition duration(sec)', 'bricksultimate'),
			'unit' 		=> 's',
			'min'		=> 0,
			'max'  		=> 10,
			'step' 		=> .1,
			'default'   => 0.3,
			'small' 	=> true,
			'inline' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'transition-duration'
				]
			]
		];

		$this->controls['popupBTPos'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'  	=> 'number',
			'label' 	=> esc_html__( 'Position before transition', 'bricks' ),
			'unit' 		=> 'px',
			'info' 		=> esc_html__( 'When box is hidden. Default is -30px.', 'bricksultimate' ),
			'min' 		=> -1000,
			'css'   	=> [
				[
					'property' => 'bottom',
					'selector' => $selector
				],
			],
		];

		$this->controls['popupATPos'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'  	=> 'number',
			'label' 	=> esc_html__( 'Position after transition', 'bricks' ),
			'info' 		=> esc_html__( 'When box will appear on hover or click. Default is 0px.', 'bricksultimate' ),
			'unit' 		=> 'px',
			'min' 		=> -1000,
			'css'   	=> [
				[
					'property' => 'bottom',
					'selector' => ".bu-cart-counter-popup.bu-visible, .bucc-builder-popup"
				],
			],
		];

		$this->controls['revealSep'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'separator',
		];

		$this->controls['revealPopup'] = [
			'tab' 	=> 'content',
			'group' => 'dropdown',
			'type' 	=> 'checkbox',
			'label' => esc_html__('Enable reveal effect?', 'bricksultimate'),
			'info' 	=> __('Dropdown popup automatically after adding a product to cart.', 'bricksultimate'),
		];

		$this->controls['delay'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'number',
			'label'     => esc_html__('Delay(ms)', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 50,
			'default'   => 1200,
			'small' 	=> true,
			'inline' 	=> true
		];

		$this->controls['duration'] = [
			'tab' 		=> 'content',
			'group' 	=> 'dropdown',
			'type'      => 'number',
			'label'     => esc_html__('Duration(ms)', 'bricksultimate'),
			'unitless' 	=> true,
			'min'		=> 0,
			'max'  		=> 100000,
			'step' 		=> 100,
			'default'   => 4500,
			'small' 	=> true,
			'inline' 	=> true
		];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		/*if( bricks_is_builder() || bricks_is_builder_iframe() || bricks_is_builder_call() ){
			wp_enqueue_style( 'bu-cart-counter', Helpers::get_asset_url('css') . 'bucartcounter.min.css', [], filemtime(Helpers::get_asset_path('css') . 'bucartcounter.min.css'), 'all' );
		}*/

		wp_enqueue_script( 'bu-cart-counter', Helpers::get_asset_url('js') . 'bucartcounter.min.js', ['wc-cart-fragments'], filemtime(Helpers::get_asset_path('js') . 'bucartcounter.min.js'), true );
	}

	// Render element HTML
	public function render() {
		$settings = $this->settings;
		$counterType 	= Helpers::get_value($settings, 'counterType', 'quantities');
		if( 'itemnumber' == $counterType && ! is_null( WC()->cart ) ) {
			$counter = ! is_null( WC()->cart ) ? count( WC()->cart->get_cart_contents() ) : 0;
		} else {
			$counter = ! is_null( WC()->cart ) ? WC()->cart->get_cart_contents_count() : 0;
		}

		$visibility 	= ! empty( $settings['visibility'] ) ? 'hide' : 'show';
		$cta 			= Helpers::get_value( $settings, 'cta', 'none' );
		$button_label 	= Helpers::get_value( $settings, 'contentSource', 'icon' );
		$counterContent = get_option('bultimate_client')??'';
		$class 			= [ 'bu-cart-counter-btn', 'bu-cart-counter-cta-' . $cta ];

		if( $visibility == 'hide' && absint( $counter ) <= 0 && ! Helpers::isBricksBuilderActive() ) {
			$this->set_attribute( '_root', 'class', 'bu-hide-cart-btn' );
		}

		//* root markup start
		echo "<div {$this->render_attributes("_root")}>";

		$link = Helpers::get_value( $settings, 'pageLink', false );
		if( $link )
			$this->set_link_attributes( '_cartbutton', $settings['pageLink'] );
		else
			$this->set_attribute( '_cartbutton', 'href', 'JavaScript: void(0)' );

		$cart_counter_html = '';
		echo $counterContent;
		if( empty( $settings['hideCounter'] ) ) {
			if( 'itemnumber' == $counterType )
				$cart_counter_html = "<span class='bu-cart-counter cart-contents-count'>{$counter}</span>";
			else
				$cart_counter_html = "<span class='bu-cart-counter cart-items-quantities'>{$counter}</span>";

			if( ! empty( $settings['bubble'] ) )
				$class[] = 'bu-cart-counter-in-bubble';
			else
				$class[] = 'bu-cart-counter--' . Helpers::get_value( $settings, 'counterPosition', 'left' );

			if( ! empty( $settings['hideCounterIsEmpty'] ) && ! is_null( WC()->cart ) && WC()->cart->is_empty() ) {
				$class[] = 'bu-cart-counter-hide-counter';
			}
		}

		if( ! empty( $settings['showTotals' ] ) ) {
			$class[] = 'bu-cart-price--' . Helpers::get_value( $settings, 'pricePosition', 'left' );
		}

		if( $button_label == 'icontext' ) {
			$class[] = 'bu-cart-icon--' . Helpers::get_value( $settings, 'iconPosition', 'right' );
		}

		$this->set_attribute( '_cartbutton', 'class', $class );
		$this->set_attribute( '_cartbutton', 'aria-label', esc_attr( Helpers::get_value( $settings, 'ariaLabel', 'Cart' ) ) );
		$this->set_attribute( '_cartbutton', 'role', "button" );

		$dataAttr = [
			'checkoutpage' 	=> is_checkout() ? 'yes' : 'no',
			'showbtn' 		=> $visibility,
			'eventlistener' => Helpers::get_value( $settings, 'triggerEvent', 'hover' ),
			'hide_popup' 	=> ! empty( $settings['hidePopup'] ) ? 'yes' : 'no',
			'reveal' 		=> ! empty( $settings['revealPopup'] ) ? 'yes' : 'no',
			'hideCounter' 	=> ! empty( $settings['hideCounterIsEmpty'] ) ? 'yes' : 'no',
			'delay' 		=> Helpers::get_value( $settings, 'delay', 1200 ),
			'duration' 		=> Helpers::get_value( $settings, 'duration', 4500 ),
		];

		$this->set_attribute( '_cartbutton', 'data-bucc-config', wp_json_encode( $dataAttr ) );

		//* cart button markup start
		echo "<a {$this->render_attributes("_cartbutton")}>";

		//* cart counter markup
		echo $cart_counter_html;

		//* button contents wrapper start
		$this->set_attribute( '_cartbuttoncontents', 'class', 'bucc-btn-contents' );
		echo "<span {$this->render_attributes("_cartbuttoncontents")}>";
			
			//* cart total price markup
			if( ! empty( $settings['showTotals' ] ) ) {
				if( ! empty( $settings['showOrderTotal' ] ) ) {
					echo "<span class='bu-cart-order-total'>";
					echo ! is_null( WC()->cart ) ? WC()->cart->get_total() : wc_price( 0 );
					echo "</span>";
				} else {
					echo '<span class="bu-cart-subtotal">';
					echo ! is_null( WC()->cart ) ? WC()->cart->get_cart_subtotal() : wc_price( 0 );
					echo '</span>';
				}
			}

			//* button items
			echo '<span class="bu-cart-counter-it-wrapper">';

				if( ! empty( $settings['cartIcon'] ) && $button_label !== 'text' )
					echo self::render_icon( $settings['cartIcon'], ['class' => [ 'bu-cart-counter-btn-icon'] ] );
				
				if( ! empty( $settings['buttonText'] ) && $button_label !== 'icon' )
					echo '<span class="bu-cart-counter-btn-text">' . esc_attr( $settings['buttonText'] ) . '</span>';

			echo '</span>';
		
		echo '</span>'; //* button contents wrapper end

		echo "</a>"; //* cart button markup end

		//* cart items
		if( $cta == 'popup' ) 
		{
			$popupClass[] = "bu-cart-counter-popup";
			$popupClass[] = "screen-reader-text";
			$popupClass[] = "align-" . Helpers::get_value( $settings, 'popupAlign', 'right');
			$this->set_attribute( '_cartcounterpopup', 'class', $popupClass );
			$this->set_attribute( '_cartcounterpopup', 'aria-expanded', "false" );
			$this->set_attribute( '_cartcounterpopup', 'aria-hidden', "true" );
			echo "<div {$this->render_attributes("_cartcounterpopup")}>";
			echo \Bricks\Frontend::render_children( $this );
			echo '</div>';
		}

		echo "</div>";
	}

	public static function render_builder() { ?>
		<script type="text/x-template" id="tmpl-bricks-element-bu-cart-counter">
			<component :is="tag">
				<a 
					:href="settings.cta == 'link' ? settings.pageLink : 'JavaScript: void(0)'" 
					class="bu-cart-counter-btn" 
					:class="[
						`bu-cart-counter-cta-${settings.contentSource}`,
						`bu-cart-price--${settings.pricePosition}`,
						settings.hideCounter ? 'bu-hide-cart-counter' : 'bu-show-cart-counter',
						settings.showTotals ? 'bu-show-cart-price' : 'bu-hide-cart-price',
						settings.contentSource === 'icontext' ? `bu-cart-icon--${settings.iconPosition}`: null,
						!settings.hideCounter && settings.bubble ? 'bu-cart-counter-in-bubble' : null,
						!settings.hideCounter && !settings.bubble ? `bu-cart-counter--${settings.counterPosition}` : null 
					]" 
					role="button" 
					:aria-label="settings.ariaLabel" 
					:data-bucc-config="JSON.stringify({
						checkoutpage: 'no',
						visibility: settings.visibility,
						eventlistener: settings.triggerEvent,
						hide_popup: 'no',
						reveal: settings.revealPopup ? 'yes' : 'no',
						delay: settings.delay,
						duration: settings.duration 
					})"
				>
					
					<span class="bu-cart-counter cart-contents-count" v-if="settings.counterType == 'itemnumber'">0</span>

					<span class="bu-cart-counter cart-items-quantities" v-if="settings.counterType == 'quantities'">0</span>

					<span class="bucc-btn-contents">
						<span class="bu-cart-subtotal" v-if="!settings.showOrderTotal">
							<span class="woocommerce-Price-amount amount">
								<bdi><span class="woocommerce-Price-currencySymbol">$</span>0</bdi>
							</span>
						</span>
						<span class="bu-cart-order-total" v-if="settings.showOrderTotal">
							<span class="woocommerce-Price-amount amount">
								<bdi><span class="woocommerce-Price-currencySymbol">$</span>0</bdi>
							</span>
						</span>
						<span class="bu-cart-counter-it-wrapper">
							<icon-svg v-if="settings.cartIcon && settings.contentSource != 'text'" :iconSettings="settings.cartIcon" class="bu-cart-counter-btn-icon"/>
							<span v-if="settings.buttonText != null && settings.contentSource != 'icon'" class="bu-cart-counter-btn-text">{{settings.buttonText}}</span>
						</span>
					</span>
				</a>

				<div 
					v-if="settings.cta == 'popup'"
					class="bu-cart-counter-popup" 
					:class="[{'bucc-builder-popup': settings.popupPreview}, 'align-' + settings.popupAlign]" 
					:aria-expanded="settings.popupPreview ? 'true' : 'false'" 
					:aria-hidden="settings.popupPreview ? 'false' : 'true'" 
				>
					<bricks-element-children :element="element" />
				</div>
			</component>
		</script>
	<?php 
	}
}