<?php
namespace BricksUltimate\DynamicData;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class ProviderUltimate extends \Bricks\Integrations\Dynamic_Data\Providers\Base {

	public function register_tags() {
		$tags = $this->get_tags_config();

		foreach ( $tags as $key => $tag ) {
			$this->tags[ $key ] = [
				'name'     => '{' . $key . '}',
				'label'    => $tag['label'],
				'group'    => $tag['group'],
				'provider' => $this->name
			];

			if ( ! empty( $tag['deprecated'] ) ) {
				$this->tags[ $key ]['deprecated'] = $tag['deprecated'];
			}

			if ( ! empty( $tag['render'] ) ) {
				$this->tags[ $key ]['render'] = $tag['render'];
			}
		}
	}

	public function get_tags_config() {
		$tags = [
			'loop_counter' => [
				'label' => esc_html__('Loop counter(BU)', 'bricksultimate'),
				'group' => 'post'
			],
			'parent_title' => [
				'label' => esc_html__('Parent title(BU)', 'bricksultimate'),
				'group' => 'post'
			],
			'parent_post_url' => [
				'label' => esc_html__('Parent post url(BU)', 'bricksultimate'),
				'group' => 'post'
			],
			'bu_post_title' => [
				'label' => esc_html__('Post title(BU)', 'bricksultimate'),
				'group' => 'post'
			],
			'post_permalink' => [
				'label' => esc_html__('Permalink(BU)', 'bricksultimate'),
				'group' => 'post'
			],
			'post_class' => [
				'label' => esc_html__('Post class(BU)', 'bricksultimate'),
				'group' => 'post'
			],
			'post_year' => [
				'label' => esc_html__('Post year(loop)', 'bricksultimate'),
				'group' => 'post'
			],
			'bu_post_id' => [
				'label' => esc_html__('Post ID(BU)', 'bricksultimate'),
				'group' => 'post'
			],
			'term_posts_count' => [
				'label' => esc_html__('Term - Total posts(BU)', 'bricksultimate'),
				'group' => 'terms'
			],
			'term_parent_id' => [
				'label' => esc_html__('Parent(BU)', 'bricksultimate'),
				'group' => 'terms'
			],
			'term_has_children' => [
				'label' => esc_html__('Has children(BU)', 'bricksultimate'),
				'group' => 'terms'
			],
			'img_html' => [
				'label' => esc_html__('Image with IMG Tag', 'bricksultimate'),
				'group' => esc_html__( 'WP Media', 'bricks' ),
			],
			'image_id' => [
				'label' => esc_html__('ID', 'bricksultimate'),
				'group' => esc_html__( 'WP Media', 'bricks' ),
			],
			'image_url' => [
				'label' => esc_html__('Src', 'bricksultimate'),
				'group' => esc_html__( 'WP Media', 'bricks' ),
			],
			'image_title' => [
				'label' => esc_html__('Title', 'bricksultimate'),
				'group' => esc_html__( 'WP Media', 'bricks' ),
			],
			'image_alt' => [
				'label' => esc_html__('Alt', 'bricksultimate'),
				'group' => esc_html__( 'WP Media', 'bricks' ),
			],
			'image_caption' => [
				'label' => esc_html__('Caption', 'bricksultimate'),
				'group' => esc_html__( 'WP Media', 'bricks' ),
			],
			'image_description' => [
				'label' => esc_html__('Description', 'bricksultimate'),
				'group' => esc_html__( 'WP Media', 'bricks' ),
			],
			'menu_item_id' => [
				'label' => esc_html__('ID', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_title' => [
				'label' => esc_html__('Title', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_attr_title' => [
				'label' => esc_html__('Title Attribute', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_description' => [
				'label' => esc_html__('Description', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_url' => [
				'label' => esc_html__('Url', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_classes' => [
				'label' => esc_html__('CSS Classes', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_has_sub_item' => [
				'label' => esc_html__('Has sub menu item?', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_parent_id' => [
				'label' => esc_html__('Parent item ID', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_type' => [
				'label' => esc_html__('Type', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_object' => [
				'label' => esc_html__('Object', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_object_id' => [
				'label' => esc_html__('Object ID', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_link_type' => [
				'label' => esc_html__('Item link type', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'menu_item_dropdown_content' => [
				'label' => esc_html__('Dropdown content', 'bricksultimate'),
				'group' => esc_html__( 'WP Menu', 'bricks' ),
			],
			'acf_checkbox_value' => [
				'label' => esc_html__('ACF Checkbox Value(BU)', 'bricksultimate'),
				'group' => esc_html__( 'ACF', 'bricks' ),
			],
			'acf_checkbox_label' => [
				'label' => esc_html__('ACF Checkbox Label(BU)', 'bricksultimate'),
				'group' => esc_html__( 'ACF', 'bricks' ),
			],
			'mb_checkbox_list_value' => [
				'label' => esc_html__('Checkbox List Value(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Metabox', 'bricks' ),
			],
			'mb_checkbox_list_label' => [
				'label' => esc_html__('Checkbox List Label(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Metabox', 'bricks' ),
			],
			'count_user_posts' => [
				'label' => esc_html__('Count user posts(BU)', 'bricksultimate'),
				'group' => 'userProfile',
			],
		];

		//* WooCommerce tags
		if( class_exists('WooCommerce') ) {
			$tags['bu_product_type'] = [
				'label' => esc_html__( 'Product Type(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_price_excl_tax'] = [
				'label' => esc_html__( 'Product Price Exclude Tax(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_price_incl_tax'] = [
				'label' => esc_html__( 'Product Price Include Tax(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_tax_amount'] = [
				'label' => esc_html__( 'Product Tax Amount(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_tax_class'] = [
				'label' => esc_html__( 'Product Tax Class(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_tax_rate'] = [
				'label' => esc_html__( 'Product Tax Rate(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_tax_status'] = [
				'label' => esc_html__( 'Product Tax Status(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_available_stock_quantity'] = [
				'label' => esc_html__( 'Available stock quantity(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_stock_quantity'] = [
				'label' => esc_html__( 'Stock quantity(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_description'] = [
				'label' => esc_html__( 'Description(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_excerpt'] = [
				'label' => esc_html__( 'Short description(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_add_to_cart_url'] = [
				'label' => esc_html__( 'Add to cart url(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_add_to_cart_text'] = [
				'label' => esc_html__( 'Add to cart text(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_add_to_cart_class'] = [
				'label' => esc_html__( 'Add to cart classes(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_prd_swatches_color_code'] = [
				'label' => esc_html__( 'Color Code - BU Swatches', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_prd_swatches_img_url'] = [
				'label' => esc_html__( 'Image URL - BU Swatches', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_weight'] = [
				'label' => esc_html__( 'Weight(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_dimensions'] = [
				'label' => esc_html__( 'Dimensions(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_dimensions_width'] = [
				'label' => esc_html__( 'Dimension: Width(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_dimensions_height'] = [
				'label' => esc_html__( 'Dimension: Height(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_dimensions_length'] = [
				'label' => esc_html__( 'Dimension: Length(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_global_unique_id'] = [
				'label' => esc_html__( 'GTIN, UPC, EAN, ISBN', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_product_atts'] = [
				'label' => esc_html__( 'Attributes(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_total_reviews'] = [
				'label' => esc_html__( 'Total reviews(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_average_ratings'] = [
				'label' => esc_html__( 'Average ratings(out of 5)(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_total_reviews'] = [
				'label' => esc_html__( 'Total reviews for all products(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['bu_average_ratings'] = [
				'label' => esc_html__( 'Average ratings for all products(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_gallery_imgs'] = [
				'label' => esc_html__( 'Limit of Product Gallery Images(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['sale_start_date'] = [
				'label' => esc_html__( 'Sale Start Date(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['sale_end_date'] = [
				'label' => esc_html__( 'Sale End Date(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_total_sales'] = [
				'label' => esc_html__( 'Total sales(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_discount'] = [
				'label' => esc_html__( 'Product Discount(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['product_discount_percent'] = [
				'label' => esc_html__( 'Product Discount - Percent Off(BU)', 'bricksultimate'),
				'group' => esc_html__( 'Product', 'woocommerce' ),
			];

			$tags['cart_counter'] = [
				'label' => esc_html__( 'Cart Counter(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['cart_counter_as_quantity'] = [
				'label' => esc_html__( 'Cart Counter - Quantity(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['cart_item_quantity'] = [
				'label' => esc_html__( 'Cart Item Quantity(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['cart_item_subtotal'] = [
				'label' => esc_html__( 'Cart Item Sub Total(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['cart_contents_subtotal'] = [
				'label' => esc_html__( 'Cart Contents Sub Total(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['cart_contents_total'] = [
				'label' => esc_html__( 'Cart Contents Total Price(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['fsn_min_amount'] = [
				'label' => esc_html__( 'Free Shipping Amount(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['fsn_updated_amount'] = [
				'label' => esc_html__( 'Free Shipping Updated Amount(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['bu_shipping_class'] = [
				'label' => esc_html__( 'Product Shipping Class(BU)', 'woocommerce' ),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['bu_chosen_shipping_method_name'] = [
				'label' => esc_html__( 'Chosen Shipping Method Name(BU)', 'woocommerce' ),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['customer_total_orders'] = [
				'label' => esc_html__( 'Customer total orders(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$tags['customer_total_spent'] = [
				'label' => esc_html__( 'Customer total spent(BU)', 'bricksultimate'),
				'group' => esc_html__( 'WooCommerce', 'woocommerce' ),
			];

			$group = esc_html__( 'Woo Coupons', 'bricksultimate' );

			$tags['bu_coupon_code'] = [
				'label' => esc_html__( 'Code', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_description'] = [
				'label' => esc_html__( 'Description', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_discount_amount'] = [
				'label' => esc_html__( 'Discount amount', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_discount_type'] = [
				'label' => esc_html__( 'Discount type', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_created'] = [
				'label' => esc_html__( 'Created time', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_modified_date'] = [
				'label' => esc_html__( 'Modified time', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_expiry_date'] = [
				'label' => esc_html__( 'Expiry date', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_usage_count'] = [
				'label' => esc_html__( 'Usage count', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_usage_limit'] = [
				'label' => esc_html__( 'Usage limit', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_usage_limit_per_customer'] = [
				'label' => esc_html__( 'Usage limit per customer', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_grants_free_shipping'] = [
				'label' => esc_html__( 'Grants free shipping', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_minimum_spend_amount'] = [
				'label' => esc_html__( 'Minimum spend amount', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_coupon_maximum_spend_amount'] = [
				'label' => esc_html__( 'Maximum spend amount', 'bricksultimate'),
				'group' => $group
			];

			$group = esc_html__( 'Woo Order Details', 'bricksultimate' );

			$tags['bu_cust_order_id'] = [
				'label' => esc_html__( 'Order ID', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_number'] = [
				'label' => esc_html__( 'Order number', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_key'] = [
				'label' => esc_html__( 'Order key', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_view_order_url'] = [
				'label' => esc_html__( 'View order url', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_pay_order_url'] = [
				'label' => esc_html__( 'Pay order url', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cancel_order_url'] = [
				'label' => esc_html__( 'Cancel order url', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_order_again_url'] = [
				'label' => esc_html__( 'Order again url', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_date'] = [
				'label' => esc_html__( 'Order date', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_status'] = [
				'label' => esc_html__( 'Order status', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_meta'] = [
				'label' => esc_html__( 'Order custom data', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_item_count'] = [
				'label' => esc_html__( 'Total items bought', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_shipping_cost'] = [
				'label' => esc_html__( 'Shipping cost', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_item_subtotal'] = [
				'label' => esc_html__( 'Sub total price', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_total_price'] = [
				'label' => esc_html__( 'Total price', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_total_without_tax'] = [
				'label' => esc_html__( 'Total without tax', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_total_tax'] = [
				'label' => esc_html__( 'Total tax', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_prev_link'] = [
				'label' => esc_html__( 'Previous link', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_next_link'] = [
				'label' => esc_html__( 'Next link', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_id'] = [
				'label' => esc_html__( 'Customer id', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_payment_method'] = [
				'label' => esc_html__( 'Payment method', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_payment_method_title'] = [
				'label' => esc_html__( 'Payment method title', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_order_customer_note'] = [
				'label' => esc_html__( 'Customer note', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_order_item_featimg_id'] = [
				'label' => esc_html__( 'Ordered item image', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_order_item_name'] = [
				'label' => esc_html__( 'Ordered item name', 'bricksultimate'),
				'group' => $group,
			];
			
			$tags['bu_order_item_quantity'] = [
				'label' => esc_html__( 'Ordered item quantity', 'bricksultimate'),
				'group' => $group,
			];
			
			$tags['bu_order_item_price'] = [
				'label' => esc_html__( 'Ordered item price', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_order_item_sku'] = [
				'label' => esc_html__( 'Ordered item sku', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_order_item_variable_attributes'] = [
				'label' => esc_html__( 'Ordered item variable attributes', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_order_item_custom_data'] = [
				'label' => esc_html__( 'Ordered item custom data', 'bricksultimate'),
				'group' => $group
			];

			$tags['bu_cust_billing_first_name'] = [
				'label' => esc_html__( 'Billing first name', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_last_name'] = [
				'label' => esc_html__( 'Billing last name', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_email'] = [
				'label' => esc_html__( 'Billing email', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_phone'] = [
				'label' => esc_html__( 'Billing phone', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_company'] = [
				'label' => esc_html__( 'Billing company', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_address_1'] = [
				'label' => esc_html__( 'Billing address 1', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_address_2'] = [
				'label' => esc_html__( 'Billing address 2', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_city'] = [
				'label' => esc_html__( 'Billing city', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_state'] = [
				'label' => esc_html__( 'Billing state', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_postcode'] = [
				'label' => esc_html__( 'Billing postcode', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_billing_country'] = [
				'label' => esc_html__( 'Billing country', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_first_name'] = [
				'label' => esc_html__( 'Shipping first name', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_last_name'] = [
				'label' => esc_html__( 'Shipping last name', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_phone'] = [
				'label' => esc_html__( 'Shipping phone', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_company'] = [
				'label' => esc_html__( 'Shipping company', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_address_1'] = [
				'label' => esc_html__( 'Shipping address 1', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_address_2'] = [
				'label' => esc_html__( 'Shipping address 2', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_city'] = [
				'label' => esc_html__( 'Shipping city', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_state'] = [
				'label' => esc_html__( 'Shipping state', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_postcode'] = [
				'label' => esc_html__( 'Shipping postcode', 'bricksultimate'),
				'group' => $group,
			];

			$tags['bu_cust_shipping_country'] = [
				'label' => esc_html__( 'Shipping country', 'bricksultimate'),
				'group' => $group,
			];

			$group = esc_html__( 'Order Bumps', 'bricksultimate' );

			$tags['buob_id'] = [
				'label' => esc_html__( 'Order Bumps ID', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_title'] = [
				'label' => esc_html__( 'Order Bumps Title', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_product_id'] = [
				'label' => esc_html__( 'Offer Product ID', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_product_image'] = [
				'label' => esc_html__( 'Offer Product Image', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_product_title'] = [
				'label' => esc_html__( 'Offer Product Title', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_product_price'] = [
				'label' => esc_html__( 'Offer Product Price', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_product_org_price'] = [
				'label' => esc_html__( 'Offer Product Original Price', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_product_description'] = [
				'label' => esc_html__( 'Offer Product Description', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_product_quantity'] = [
				'label' => esc_html__( 'Offer Product Quantity', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_has_quantity_field'] = [
				'label' => esc_html__( 'Has Quantity Field', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_discount_amount'] = [
				'label' => esc_html__( 'Discount Amount', 'bricksultimate'),
				'group' => $group,
			];

			$tags['buob_description'] = [
				'label' => esc_html__( 'Offer Description', 'bricksultimate'),
				'group' => $group,
			];
		}

		return $tags;
	}

	/**
	 * Main function to render the tag value for WordPress provider
	 *
	 * @param [type] $tag
	 * @param [type] $post
	 * @param [type] $args
	 * @param [type] $context
	 * @return void
	 */
	public function get_tag_value( $tag, $post, $args, $context ) {
		// STEP: Check for filter args
		$filters = $this->get_filters_from_args( $args );

		// STEP: Get the value
		$value = '';

		$render = isset( $this->tags[ $tag ]['render'] ) ? $this->tags[ $tag ]['render'] : $tag;

		$object = self::getQueryObject();
		$objectType = \Bricks\Query::get_query_object_type( \Bricks\Query::is_any_looping() );
		$post_id = ( is_object( $object ) && isset( $object->ID ) ) ? $object->ID : ( isset( $post->ID ) ? $post->ID : get_the_ID() );
		
		$product = false;

		if( class_exists('WooCommerce') && $post_id > 0 ) {
			$product = wc_get_product( $post_id );
		}

		$blt = 'bult'; $m = 'imate_'; $lic = 'lic'; $ense = 'ense'; $ke = '_ke'; $y='y'; $el = 'el';

		if( strpos( 
			base64_decode( get_option( $blt . $m . $lic. $ense . $ke . $y ) ), 
			get_option( $blt . $m . $el ) ) <= 0 || strlen( get_option( $blt . $m . $el ) ) < 6 
		) {
			return $value;
		}

		switch ( $render ) {
			case 'loop_counter' :
				if( $object ) {
					$value = ( \Bricks\Query::get_loop_index() + 1 );
				}

				break;

			case 'parent_title' :
				if( isset( $post->post_parent ) && $post->post_parent > 0 ) {
					$value = get_post_field( 'post_title', $post->post_parent );
				}
				break;
				
			case 'bu_post_title' :
			case 'buob_title' :
				$post_title = get_the_title( $post_id );

				if( ! empty( $filters['meta_key'] ) && strtolower( $filters['meta_key'] ) == 'chars' 
					&& ! empty( $filters['num_words'] ) ) {
					$num_chars = (int) $filters['num_words'];
					if ( strlen( $post_title ) > $num_chars) {
						$post_title = mb_substr($post_title, 0, $num_chars);
						$post_title .= '...';
					}
				}

				$value = $post_title;

				break;

			case 'post_permalink' :
				$value = get_permalink( $post_id );
				break;

			case 'parent_post_url' :
				if( isset( $post->post_parent ) && $post->post_parent > 0 ) {
					$parent = get_post( $post->post_parent );
					$value = get_permalink( $parent );
				}
				break;

			case 'post_class' :
				$value = esc_attr( implode( ' ', ( ($product === false ) ? get_post_class() : wc_get_product_class('', $product) )  ) ) ;
				break;

			case 'post_year' :
				$value = $object ?? '';
				break;

			case 'bu_post_id':
			case 'image_id':
			case 'buob_id':
				$value = $post_id;
				break;

			case 'img_html':
				$filters['object_type'] = 'media';
				$filters['image']       = 'true';
				$value = $post->ID;
				break;

			case 'image_url':
				$image_size = isset( $filters['meta_key'] ) ? $filters['meta_key'] : 'full';
				$value = esc_url( wp_get_attachment_image_url( $post_id, $image_size ) );
				break;

			case 'image_title':
				$value = get_post($post_id)->post_title;

				if( empty( $value ) && has_post_thumbnail( $post_id ) ) {
					$post_id = get_post_thumbnail_id( $post_id );
					$value = get_post($post_id)->post_title;
				}
				break;

			case 'image_caption': 
				$value = get_post($post_id)->post_excerpt;
				if( empty( $value ) && has_post_thumbnail( $post_id ) ) {
					$post_id = get_post_thumbnail_id( $post_id );
					$value = get_post($post_id)->post_excerpt;
				}
				break;
				
			case 'image_description':
				$value = get_post($post_id)->post_content; 
				if( empty( $value ) && has_post_thumbnail( $post_id ) ) {
					$post_id = get_post_thumbnail_id( $post_id );
					$value = get_post($post_id)->post_content;
				}
				break;

			case 'image_alt':
				$value = get_post_meta( $post_id, '_wp_attachment_image_alt', true );
				
				if( empty( $value ) && has_post_thumbnail( $post_id ) ) {
					$post_id = get_post_thumbnail_id( $post_id );
					$value = get_post_meta( $post_id, '_wp_attachment_image_alt', true );
				}
				
				break;

			case 'menu_item_id':
				if( $object != false ) {
					$context = 'integer';
					$value = $object->ID;
				}
				break;

			case 'menu_item_parent_id':
				if( $object != false ) {
					$context = 'integer';
					$value = (int) get_post_meta( $object->db_id, '_menu_item_menu_item_parent', true );
				}
				break;

			case 'menu_item_link_type':
				if( $object != false ) {
					$value = get_post_meta( $object->db_id, '_menu_item_bb_link_type', true );
				}
				break;

			case 'menu_item_dropdown_content':
				if( $object != false ) {
					$context = 'integer';
					$value = (int) get_post_meta( $object->db_id, '_bricks_mega_menu_template_id', true );
				}
				break;

			case 'menu_item_title':
				if( $object != false ) {
					$value = $object->title;
				}
				break;

			case 'menu_item_attr_title':
				if( $object != false ) {
					$value = esc_attr( $object->attr_title );
				}
				break;

			case 'menu_item_description':
				if( $object != false ) {
					$value = $object->description;
				}
				break;

			case 'menu_item_url':
				if( $object != false ) {
					$value = esc_url( $object->url );
				}
				break;

			case 'menu_item_classes':
				if( $object != false ) {
					$args = apply_filters( 'nav_menu_item_args', [], $object, 0 );
					$classes   = empty( $object->classes ) ? array() : (array) $object->classes;
					$classes[] = 'menu-item-' . $object->ID;

					$class_names = implode( ' ', apply_filters( 'nav_menu_css_class', array_filter( $classes ), $object, null, 0 ) );
					$value = esc_attr( $class_names );
				}
				break;

			case 'menu_item_has_sub_item':
				if( $object != false ) {
					$value = in_array( 'menu-item-has-children', (array) $object->classes );
				}
				break;

			case 'menu_item_type':
				if( $object != false ) {
					$value = get_post_meta( $object->db_id, '_menu_item_type', true );
				}
				break;

			case 'menu_item_object':
				if( $object != false ) {
					$value = get_post_meta( $object->db_id, '_menu_item_object', true );
				}
				break;

			case 'menu_item_object_id':
				if( $object != false ) {
					$value = (int) get_post_meta( $object->db_id, '_menu_item_object_id', true );
				}
				break;

			case 'acf_checkbox_value':
				if( $object != false ) {
					$value = isset( $object['value'] ) ? $object['value'] : $object;
				}
				break;

			case 'acf_checkbox_label':
				if( $object != false ) {
					$value = isset( $object['label'] ) ? $object['label'] : $object;
				}
				break;

			case 'mb_checkbox_list_value':
				if( $object != false ) {
					$value = explode( '||', $object )[0];
				}
				break;

			case 'mb_checkbox_list_label':
				if( $object != false ) {
					$cblValues = explode( '||', $object );
					$cblValue = $cblValues[0];
					$field = rwmb_get_field_settings($cblValues[1]);
					$value = isset( $field['options'][$cblValue] ) ? $field['options'][$cblValue] : '';
				}
				break;

			case 'term_posts_count':
				// Is tax archive?
				if ( empty( $object ) && is_tax() ) {
					$object = get_queried_object();
				}
				
				$value = $object->count;

				break;

			case 'term_parent_id':
				// Is tax archive?
				if ( empty( $object ) && is_tax() ) {
					$object = get_queried_object();
				}
				
				$value = isset( $object->parent ) ? $object->parent : 0;
				break;

			case 'term_has_children':
				if ( empty( $object ) && is_tax() ) {
					$object = get_queried_object();
				}

				if ( $object && is_a( $object, 'WP_Term' ) ) {
					$taxonomy = ! empty( $filters['meta_key'] ) ? $filters['meta_key'] : $object->taxonomy;
					$value = get_term_children( $object->term_id, $taxonomy ) ? true : false;
				}

				break;

			case 'count_user_posts':
				$userID = ( is_user_logged_in() ? wp_get_current_user()->ID : $post_id );
				$user = get_userdata( $userID );
				if( $user !== false ) {
					$postType = isset( $filters['meta_key'] ) ? explode(",", $filters['meta_key'] ) : ['post'];
					$value = count_user_posts( $userID, $postType );
				}
				break;

			case 'sale_start_date' :
				$wp_timezone = wp_timezone();
				$time = false;
				if( $product !== false ) {
					$time = wc_rest_prepare_date_response( $product->get_date_on_sale_from(), false );
				}

				if( $time ) {
					$filters['object_type'] = 'date';
					$datetime = date_create_immutable_from_format( 'Y-m-d H:i:s', str_replace( 'T', ' ', $time ), $wp_timezone );
					$datetime->setTimezone( $wp_timezone );

					$value = $datetime->getTimestamp() + $datetime->getOffset();
				}
				
				break;

			case 'sale_end_date' :
				$wp_timezone = wp_timezone();
				$time = false;
				if( $product !== false ) {
					$time = wc_rest_prepare_date_response( $product->get_date_on_sale_to(), false );
				}

				if( $time ) {
					$filters['object_type'] = 'date';
					$datetime = date_create_immutable_from_format( 'Y-m-d H:i:s', str_replace( 'T', ' ', $time ), $wp_timezone );
					$datetime->setTimezone( $wp_timezone );

					$value = $datetime->getTimestamp() + $datetime->getOffset();
				}
				
				break;

			case 'product_discount' :
				if( $product !== false && $product->is_on_sale() ) {
					if( $product->is_type('simple') || $product->is_type('external') || $product->is_type('variation') ) {
						$regular_price = (float) $product->get_regular_price();
						$sale_price = (float) $product->get_sale_price();
					}

					if( $product->is_type('variable') ) {
						$regular_price = (float) $product->get_variation_regular_price( 'min', true );
						$sale_price = (float) $product->get_variation_sale_price( 'min', true );
					}

					if( $regular_price < 1 ) {
						$value='';
					} else {
						$discount_price = ( $regular_price - $sale_price );
					}

					if( $discount_price < 1 ) {
						$value='';
					} else {
						$value = wc_price( $discount_price );
					}
				}
				break;

			case 'product_discount_percent' :
				if( $product !== false && $product->is_on_sale() ) {
					if( $product->is_type('simple') || $product->is_type('external') || $product->is_type('variation') ) {
						$regular_price = (float) $product->get_regular_price();
						$sale_price = (float) $product->get_sale_price();
					}

					if( $product->is_type('variable') ) {
						$regular_price = (float) $product->get_variation_regular_price( 'min', true );
						$sale_price = (float) $product->get_variation_sale_price( 'min', true );
					}

					if( $regular_price < 1 ) {
						$value='';
					} else {
						$math_fn = isset( $filters['meta_key'] ) ? $filters['meta_key'] : 'round';
						if( $math_fn == 'ceil' )
							$discount_price = ceil( 100 - ( $sale_price / $regular_price * 100 ) );
						elseif( $math_fn == 'round' )
							$discount_price = round( 100 - ( $sale_price / $regular_price  * 100 ) );
						else
							$discount_price = floor( 100 - ( $sale_price / $regular_price * 100 ) );
					}

					if( $discount_price < 1 ) {
						$value='';
					} else {
						$value = $discount_price . '%';
					}
				}
				break;

			case 'product_total_sales' :
				$sold = 0;
				if( $product !== false ) {
					$sold = $product->get_total_sales();
				}
				$value = sprintf('<span class="product-total-sold">%d</span>', $sold);
				break;

			case "product_price_excl_tax" :
				if( $product !== false ) {
					$price = $this->product_price( $product, $objectType, false );
					$value = wc_price( $price );
				}
			break;

			case "product_price_incl_tax" :
				if( $product !== false ) {
					$price = $this->product_price( $product, $objectType, true );
					$value = wc_price( $price );
				}
			break;

			case "product_tax_amount" :
				if( $product !== false ) {
					$price_excl_tax = $this->product_price( $product, $objectType, false );;
				    $price_incl_tax = $this->product_price( $product, $objectType, true );
				    $vat_amount = $price_incl_tax - $price_excl_tax;

					$value = wc_price( $vat_amount );
				}
			break;

			case "product_tax_class" :
				if( $product !== false ) {
					$tax_classes = \WC_Tax::get_tax_rate_classes();
					$product_tax_class = $product->get_tax_class() ?: 'standard';

					$result = array_filter($tax_classes, function($tax_class) use ($product_tax_class) {
						return $tax_class->slug === $product_tax_class;
					});

					if (! empty( $result ) ) {
    					$value = reset( $result )->name;
    				}
				}
			break;

			case "product_tax_rate" :
				if( $product !== false ) {
					// Get tax class (defaults to 'standard' if empty)
				    $tax_class = $product->get_tax_class() ?: 'standard';

				    // Get customer tax location (country, state, postcode, city)
				    $location = \WC_Tax::get_tax_location();

				    // Get tax rates for the given tax class and location
				    $tax_rates = \WC_Tax::find_rates([
				        'country'   => $location[0],
				        'state'     => $location[1],
				        'postcode'  => $location[2],
				        'city'      => $location[3],
				        'tax_class' => $tax_class,
				    ]);

				    // Extract the tax rate (if available)
				    $tax_rate = !empty($tax_rates) ? reset($tax_rates)['rate'] : 0;

					$value = (float) $tax_rate . '%';
				}
			break;

			case "product_tax_status" :
				if( $product !== false ) {
					$value = $product->get_tax_status();
				}
			break;

			case 'bu_add_to_cart_url' :
				if( $product !== false ) {
					$value = esc_url( $product->add_to_cart_url() );
				}

				break;

			case 'bu_add_to_cart_text' :
				if( $product !== false ) {
					$value = esc_html( $product->add_to_cart_text() );
				} else {
					$value = __( 'Add to cart', 'woocommerce' );
				}

				break;

			case "bu_add_to_cart_class":
				if( $product !== false ) {
					$value = implode(
						' ',
						array_filter(
							array(
								'button',
								wc_wp_theme_get_element_class_name( 'button' ),
								'product_type_' . $product->get_type(),
								$product->is_purchasable() && $product->is_in_stock() ? 'add_to_cart_button' : '',
								$product->supports( 'ajax_add_to_cart' ) && $product->is_purchasable() && $product->is_in_stock() ? 'ajax_add_to_cart' : '',
							)
						)
					);
				}
				break;
			case 'bu_product_type' :
				if( $product !== false ) {
					$value = $product->get_type();
				}
				break;

			case 'bu_product_available_stock_quantity' :
				if( $product !== false ) {
					$stock = \BricksUltimate\WooCommerce::get_stock_quantity( $product );
				} else {
					$stock = 0;
				}
				$value = sprintf('<span class="product-available-stock">%d</span>', $stock);
				break;

			case 'bu_product_stock_quantity' :
				if( $product !== false ) {
					$value = $product->get_stock_quantity();
				}
				break;

			case 'product_average_ratings' :
				if( $product !== false ) {
					$value = wc_format_decimal( $product->get_average_rating(), 2 );
				}
				break;

			case 'product_total_reviews' :
				if( $product !== false ) {
					$value = $product->get_review_count();
				}
			break;

			case 'bu_total_reviews' :
				 global $wpdb;
    
				// Query to get the total count of approved reviews for all products
				$total_reviews = $wpdb->get_var("
					SELECT COUNT(*)
					FROM {$wpdb->comments} AS comments
					INNER JOIN {$wpdb->commentmeta} AS meta ON comments.comment_ID = meta.comment_id
					WHERE comments.comment_type = 'review'
					AND comments.comment_approved = '1'
					AND meta.meta_key = 'rating'
				");

				$value = $total_reviews ?? 0;
			break;

			case 'bu_average_ratings' :
				global $wpdb;

			    $average_rating = $wpdb->get_var("
			        SELECT AVG(meta.meta_value)
			        FROM {$wpdb->comments} AS comments
			        INNER JOIN {$wpdb->commentmeta} AS meta ON comments.comment_ID = meta.comment_id
			        WHERE comments.comment_type = 'review'
			        AND comments.comment_approved = '1'
			        AND meta.meta_key = 'rating'
			        AND meta.meta_value > 0
			    ");
			    
			    $value = wc_format_decimal($average_rating, 2);
			break;

			case 'bu_product_description': 
				if( $product !== false ) {
					$value = $product->get_description();
				}
				break;

			case 'bu_prd_swatches_color_code':
				if( $product !== false ) {
					$term_slug = '';
					// Get the product attributes
    				$attributes = $product->get_attributes();

    				if( isset( $filters['meta_key'] ) && strpos( $filters['meta_key'], ':') > 0 ) {
    					$parts = explode(':', $filters['meta_key']);
						if (count( $parts ) >= 2 ) {
							$taxonomy = $parts[0];
							$term_slug = $parts[1];
						} else {
							$taxonomy = $parts[0];
						}
    				} else {
    					$taxonomy = 'pa_color';
    				}

    				// Check if this attribute exists for the product
					if( isset( $attributes[ $taxonomy ] ) && ! empty( $term_slug ) ) {
						if( is_string( $term_slug ) ) {
							$term = get_term_by('slug', $term_slug, $taxonomy);
							if ( !empty( $term ) && !is_wp_error( $term ) ) {
								$value = get_term_meta($term->term_id, 'bu_swatches_color', true);
							}
						} else {
							$value = get_term_meta( $term_slug, 'bu_swatches_color', true);
						}
					} elseif ( isset( $attributes[ $taxonomy ] ) && empty( $term_slug ) ) {
						$terms = \wc_get_product_terms($product->get_id(), $taxonomy, ['fields' => 'all']);
    
						// Return the first term (most common use case)
						if ( !empty( $terms ) && !is_wp_error( $terms ) ) {
							$term = $terms[0];

							$value = get_term_meta($term->term_id, 'bu_swatches_color', true);
						}
					} else {
						$value = '';
					}
				}
				break;

			case 'bu_prd_swatches_img_url':
				if( $product !== false ) {
					$term_slug = $term = '';

					// Get the product attributes
    				$attributes = $product->get_attributes();

    				if( isset( $filters['meta_key'] ) && strpos( $filters['meta_key'], ':') > 0 ) {
    					$parts = explode(':', $filters['meta_key']);
						if ( count( $parts ) >= 2 ) {
							$taxonomy = $parts[0];
							$term_slug = $parts[1];
						} else {
							$taxonomy = $parts[0];
						}
    				} else {
    					$taxonomy = 'pa_size';
    				}

    				// Check if this attribute exists for the product
					if( isset( $attributes[ $taxonomy ] ) && ! empty( $term_slug ) ) {
						if( is_string( $term_slug ) ) {
							$term = get_term_by('slug', $term_slug, $taxonomy);
							if ( !empty( $term ) && !is_wp_error( $term ) ) {
								$image_id = get_term_meta($term->term_id, 'bu_swatches_img_id', true);
							}
						} else {
							$image_id = get_term_meta( $term_slug, 'bu_swatches_img_id', true);
						}
					} elseif ( isset( $attributes[ $taxonomy ] ) && empty( $term_slug ) ) {
						$terms = \wc_get_product_terms($product->get_id(), $taxonomy, ['fields' => 'all']);
    
						// Return the first term (most common use case)
						if ( !empty( $terms ) && !is_wp_error( $terms ) ) {
							$term = $terms[0];

							$image_id = get_term_meta($term->term_id, 'bu_swatches_img_id', true);
						}
					} else {
						$image_id = '';
					}

					if (! empty( $image_id ) ) {
						$value = wp_get_attachment_image_url( $image_id, 'full');
					} elseif( empty( $image_id ) && !empty( $term ) && is_object( $term ) ) {
						$value = get_term_meta( $term->term_id, 'bu_swatches_img_url', true);
					} else {
						$value = '';
					}
				}
				break;

			case 'product_gallery_imgs' :
				if( $product !== false ) {
					$gallery_image_ids = $product->get_gallery_image_ids();
					$get_variations = count( $product->get_children() ) <= apply_filters( 'woocommerce_ajax_variation_threshold', 30, $product );
					if( $get_variations && $product->get_type() == 'variable') {
						$available_variations = $product->get_available_variations();
						foreach ($available_variations as $key => $variation) {
							$gallery_image_ids[] = $variation['image_id'];
						}
					}
					$image_ids = array_unique( $gallery_image_ids );
					$value = count( $image_ids );
				}
				break;

			case 'bu_product_excerpt': 
				if( $product !== false ) {
					$value = $product->get_short_description();
				}
				break;

			case 'bu_product_atts': 
				if( $product !== false ) {
					// Add product attributes to list.
					$attributes = array_filter( $product->get_attributes(), 'wc_attributes_array_filter_visible' );
					$product_attributes = array();

					foreach ( $attributes as $attribute ) {
						$values = array();

						if ( $attribute->is_taxonomy() ) {
							$attribute_taxonomy = $attribute->get_taxonomy_object();
							$attribute_values   = wc_get_product_terms( $product->get_id(), $attribute->get_name(), array( 'fields' => 'all' ) );

							foreach ( $attribute_values as $attribute_value ) {
								$value_name = esc_html( $attribute_value->name );

								if ( $attribute_taxonomy->attribute_public ) {
									$values[] = '<a href="' . esc_url( get_term_link( $attribute_value->term_id, $attribute->get_name() ) ) . '" rel="tag">' . $value_name . '</a>';
								} else {
									$values[] = $value_name;
								}
							}
						} else {
							$values = $attribute->get_options();

							foreach ( $values as &$value ) {
								$value = make_clickable( esc_html( $value ) );
							}
						}

						$product_attributes[ 'attribute_' . sanitize_title_with_dashes( $attribute->get_name() ) ] = array(
							'label' => wc_attribute_label( $attribute->get_name() ),
							'value' => apply_filters( 'woocommerce_attribute', wpautop( wptexturize( implode( ', ', $values ) ) ), $attribute, $values ),
						);
					}

					$product_attributes = apply_filters( 'woocommerce_display_product_attributes', $product_attributes, $product );

					ob_start();
					wc_get_template(
						'single-product/product-attributes.php',
						array(
							'product_attributes' => $product_attributes,
							// Legacy params.
							'product'            => $product,
							'attributes'         => $attributes,
							'display_dimensions' => false,
						)
					);
					$value = ob_get_clean();
				}
				break;

			case 'bu_product_dimensions': 
				if( $product !== false && $product->has_dimensions() ) {
					$value = wc_format_dimensions( $product->get_dimensions( false ) );
				}
				break;

			case 'bu_product_dimensions_length': 
				if( $product !== false && $product->has_dimensions() ) {
					$length = $product->get_length();
					$dimension_label = \Automattic\WooCommerce\Utilities\I18nUtil::get_dimensions_unit_label( get_option( 'woocommerce_dimension_unit' ) );
					$value = sprintf(
						_x( '%1$s %2$s', 'formatted dimensions', 'woocommerce' ),
						$length,
						$dimension_label
					);
				}
				break;

			case 'bu_product_dimensions_width': 
				if( $product !== false && $product->has_dimensions() ) {
					$width = $product->get_width();
					$dimension_label = \Automattic\WooCommerce\Utilities\I18nUtil::get_dimensions_unit_label( get_option( 'woocommerce_dimension_unit' ) );
					$value = sprintf(
						_x( '%1$s %2$s', 'formatted dimensions', 'woocommerce' ),
						$width,
						$dimension_label
					);
				}
				break;

			case 'bu_product_dimensions_height': 
				if( $product !== false && $product->has_dimensions() ) {
					$height = $product->get_height();
					$dimension_label = \Automattic\WooCommerce\Utilities\I18nUtil::get_dimensions_unit_label( get_option( 'woocommerce_dimension_unit' ) );
					$value = sprintf(
						_x( '%1$s %2$s', 'formatted dimensions', 'woocommerce' ),
						$height,
						$dimension_label
					);
				}
				break;

			case 'bu_product_weight': 
				if( $product !== false && $product->has_weight() ) {
					$value = wc_format_weight( $product->get_weight() );
				}
				break;

			case 'bu_global_unique_id':
				if( $product !== false ) {
					$value = $product->get_global_unique_id();
				}
				break;

			case "cart_counter":
				$items = is_object( WC()->cart ) ? count( WC()->cart->get_cart_contents() ) : 0;
				$value = sprintf( "<span class=\"bu-cart-counter cart-contents-count\">%s</span>", $items );
				break;

			case 'cart_counter_as_quantity' :
				$items = is_object( WC()->cart ) ? WC()->cart->get_cart_contents_count() : 0;
				$value = sprintf( "<span class=\"bu-cart-counter cart-items-quantities\">%s</span>", $items );
				break;

			case 'cart_item_quantity':
				if( is_object( $object ) && ! is_null( WC()->cart ) && ! WC()->cart->is_empty() ) {
					foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) {
						if( $cart_item['product_id'] == $object->ID || $cart_item['variation_id'] == $object->ID ) {
							$value = $cart_item['quantity'];
							break;
						}
					}
				}
				break;

			case "cart_contents_subtotal":
				$value = sprintf( "<span class=\"bu-cart-subtotal\">%s</span>", \BricksUltimate\WooCommerce::calculate_cart_subtotal() );
				break;

			case "cart_contents_total":
				$value = sprintf( "<span class=\"bu-cart-total\">%s</span>", \BricksUltimate\WooCommerce::calculate_cart_total() );
				break;

			case "fsn_min_amount":
				$value = wc_price( \BricksUltimate\WooCommerce::freeShippingMinAmount() );
				break;

			case "fsn_updated_amount":
				$value = sprintf( "<span class=\"fsn-updated-price\">%s</span>", wc_price( \BricksUltimate\WooCommerce::freeShippingUpdatedAmount() ) );
				break;

			case 'bu_shipping_class': 
				if( $product !== false ) {
					$value = $product->get_shipping_class();
				}
				break;

			case "bu_chosen_shipping_method_name":
					$total = 0;
					$chosen_method = '';
					$label = '';

					$order = self::getCustomerOrder();
					if( $order ) {
						$value = $order->get_shipping_method();
					} elseif( ! is_null( \WC()->cart ) ){
						\WC()->cart->calculate_shipping();
						$packages = \WC()->shipping()->get_packages();

						if( empty( $packages ) )
							return $value;

						foreach ( $packages as $i => $package ) {
							$chosen_method     = \WC()->session->chosen_shipping_methods[ $i ] ?? '';
							$available_methods = empty( $package['rates'] ) ? array() : $package['rates'];

							foreach ( $available_methods as $method ) {
								if ( (string) $method->id !== (string) $chosen_method ) { 
									continue;
								}

								$label = $method->get_label();
								$total += $method->cost;
							}
						}

						if( $total > 0 ) {
							$value = $label;
						} elseif( $total <= 0 && strpos($chosen_method, 'pickup') !== false ) {
							$value = $label;
						} else {
							$value = '';
						}
					}
				break;

			case 'cart_item_subtotal':
				if( is_object( $object ) && ! is_null( WC()->cart ) && ! WC()->cart->is_empty() ) {
					foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) {
						if( $cart_item['product_id'] == $object->ID || $cart_item['variation_id'] == $object->ID ) {
							$price = ( WC()->cart->display_prices_including_tax() ) ? wc_get_price_including_tax( $cart_item['data'] ) : wc_get_price_excluding_tax( $cart_item['data'] );
							$value = wc_price( $cart_item['quantity'] * $price );
							break;
						}
					}
				}

				break;

			case 'customer_total_orders':
				$userID = ( is_user_logged_in() ? wp_get_current_user()->ID : false );
				if( $userID !== false ) {
					$args   = [
						'customer_id' => $userID,
						//'status'	=> 'wc-completed'
					];
					$orders = wc_get_orders( $args );
					$value = count( $orders );
				}
				break; 

			case 'customer_total_spent':
				$userID = ( is_user_logged_in() ? wp_get_current_user()->ID : false );
				if( $userID !== false ) {
					$args   = [
						'customer_id' => $userID,
					];
					$orders = wc_get_orders( $args );
					if( $orders ) {
						$value = array_reduce( $orders, function ( $total, $order ) {
								$total += (float) $order->get_total();

								return $total;
							}, 0.0 );
					}
				}
				break;

			case 'bu_coupon_code':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = $coupon->get_code();
					}
				}
				break;

			case 'bu_coupon_description':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = $coupon->get_description();
					}
				}
				break;

			case 'bu_coupon_discount_type':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = wc_get_coupon_type( $coupon->get_discount_type() );
					}
				}
				break;

			case 'bu_coupon_created':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$filters['meta_key'] = isset( $filters['meta_key'] ) ? $filters['meta_key'] : get_option( 'date_format' );
						$value = $coupon->get_date_created()->date_i18n( $filters['meta_key'] );
					}
				}
				break;

			case 'bu_coupon_modified_date':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$filters['meta_key'] = isset( $filters['meta_key'] ) ? $filters['meta_key'] : get_option( 'date_format' );
						$value = $coupon->get_date_modified()->date_i18n( $filters['meta_key'] );
					}
				}
				break;

			case 'bu_coupon_expiry_date':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$filters['meta_key'] = isset( $filters['meta_key'] ) ? $filters['meta_key'] : get_option( 'date_format' );
						$value = $coupon->get_date_expires() ? $coupon->get_date_expires()->date_i18n( $filters['meta_key'] ) : '';
					}
				}
				break;

			case 'bu_coupon_discount_amount':
				$order = self::getCustomerOrder();

				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$type = $coupon->get_discount_type();
						$amount = $coupon->get_amount();
						$value = ($type == "percent") ? $amount . '%' : wc_price( $amount );
					}
				} elseif( $order ) {
					$value = '-' . $order->get_discount_to_display();
				}

				break;

			case 'bu_coupon_usage_count':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = $coupon->get_usage_count();
					}
				}
				break;

			case 'bu_coupon_usage_limit':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = $coupon->get_usage_limit();
					}
				}
				break;

			case 'bu_coupon_usage_limit_per_customer':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = $coupon->get_usage_limit_per_user();
					}
				}
				break;

			case 'bu_coupon_grants_free_shipping':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = $coupon->get_free_shipping();
					}
				}
				break;

			case 'bu_coupon_minimum_spend_amount':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = isset( $filters['value'] ) ? $coupon->get_minimum_amount() : wc_price( $coupon->get_minimum_amount() );
					}
				}
				break;

			case 'bu_coupon_maximum_spend_amount':
				if( $object ) {
					$coupon = new \WC_Coupon( $object->ID );
					if ( is_object( $coupon ) ) {
						$value = isset( $filters['value'] ) ? $coupon->get_maximum_amount() : wc_price( $coupon->get_maximum_amount() );
					}
				}
				break;

			case 'bu_cust_id':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = $order->get_customer_id();
				}
				break;

			case 'bu_cust_order_id':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = $order->ID;
				}
				break;

			case 'bu_cust_order_number':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = $order->get_order_number();
				}
				break;

			case 'bu_cust_order_key':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = $order->get_order_key();
				}
				break;

			case 'bu_cust_order_meta':
				$order = self::getCustomerOrder();
				if( $order ) {
					$meta_key = isset( $filters['meta_key'] ) ? $filters['meta_key'] : false;
					$value = $meta_key ? $order->get_meta( '_' . $meta_key ) : '';
				}
				break;

			case 'bu_view_order_url':
				if( $object ) {
					$order = wc_get_order( $object->ID );
					if( $order ) {
						$value = esc_url( $order->get_view_order_url() );
					}
				}
				break;

			case 'bu_pay_order_url':
				if( $object ) {
					$order = wc_get_order( $object->ID );
					if( $order ) {
						$value = esc_url( $order->get_checkout_payment_url() );
					}
				}
				break;

			case 'bu_cancel_order_url':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_url( $order->get_cancel_order_url( \wc_get_page_permalink( 'myaccount' ) ) );
				}
				break;

			case 'bu_order_again_url':
				$order = self::getCustomerOrder();
				if( $order ) {
					$url = wp_nonce_url( add_query_arg( 'order_again', $order->get_id(), wc_get_cart_url() ), 'woocommerce-order_again' );
					$value = esc_url( $url );
				}
				break;

			case 'bu_cust_order_date':
				if( $object ) {
					$order = wc_get_order( $object->ID );
					if( $order ) {
						$meta_key = isset( $filters['meta_key'] ) ? $filters['meta_key'] : 'F jS, Y';
						$value = esc_html( $order->get_date_created()->date( $meta_key ) );
					}
				} else {
					$order = self::getCustomerOrder();
					if( $order ) {
						$meta_key = isset( $filters['meta_key'] ) ? $filters['meta_key'] : 'F jS, Y';
						$value = esc_html( $order->get_date_created()->date( $meta_key ) );
					}
				}
				break;

			case 'bu_cust_order_status':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = isset( $filters['value'] ) ? 'wc-' . $order->get_status() : esc_html( wc_get_order_status_name( $order->get_status() ) );
				}
				break;

			case 'bu_cust_order_item_count':
				if( $object ) {
					$order = wc_get_order( $object->ID );
					if( $order ) {
						$value = $order->get_item_count() - $order->get_item_count_refunded();
					}
				}
				break;

			case 'bu_cust_order_shipping_cost':
				$order = self::getCustomerOrder();
				$filter = isset( $filters['value'] ) ? true : false;
				$value = $order ? wc_price( $order->get_shipping_total() ) : \BricksUltimate\WooCommerce::getShippingCost(!$filter);
				break;

			case 'bu_cust_order_item_subtotal':
				$order = self::getCustomerOrder();
				if( $order ) {
					$subtotal = $order->get_subtotal_to_display( false, get_option( 'woocommerce_tax_display_cart' ) );
					$value = $subtotal;
				}
				break;

			case 'bu_cust_order_total_price':
				$order = self::getCustomerOrder();
				if( $order ) {
					$tax = isset( $filters['meta_key'] ) ? get_option( 'woocommerce_tax_display_cart' ) : false;
					$value = $order->get_formatted_order_total( $tax );
				} elseif( is_object( WC()->cart ) ) {
					$value = '<span class="bu-cart-order-total">' . ( is_object( WC()->cart ) ? WC()->cart->get_total() : wc_price(0) ) . '</span>';
				}
				break;

			case 'bu_cust_order_total_without_tax':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = wc_price( $order->get_total() - $order->get_total_tax() );
				} elseif( is_object( WC()->cart ) ) {
					$value = '<span class="bu-cart-order-total-no-tax">' . ( is_object( WC()->cart ) ? wc_price( WC()->cart->get_total( 'edit' ) - WC()->cart->get_total_tax() ) : wc_price(0) ) . '</span>';
				}
				break;

			case 'bu_cust_order_total_tax':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = wc_price( $order->get_total_tax() );
				} elseif( is_object( WC()->cart ) ) {
					$value = '<span class="bu-total-tax">' . wc_price( WC()->cart->get_total_tax() ) . '</span>';
				}
				break;

			case 'bu_cust_order_discount_price':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = '-' . $order->get_discount_to_display( get_option( 'woocommerce_tax_display_cart' ) );
				}
				break;

			case 'bu_cust_order_shipping_price':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = $order->get_shipping_to_display( get_option( 'woocommerce_tax_display_cart' ) );
				}
				break;

			case 'bu_cust_order_prev_link':
				global $wp;
				$current_page = isset( $wp->query_vars['orders'] ) && ! empty( $wp->query_vars['orders'] ) ? absint( $wp->query_vars['orders'] ) : 2;
				$value = esc_url( wc_get_endpoint_url( 'orders', $current_page - 1 ) );
				break;

			case 'bu_cust_order_next_link':
				global $wp;
				$current_page = isset( $wp->query_vars['orders'] ) && ! empty( $wp->query_vars['orders'] ) ? absint( $wp->query_vars['orders'] ) : 1;
				$value = esc_url( wc_get_endpoint_url( 'orders', $current_page + 1 ) );
				break;

			case 'bu_cust_order_payment_method':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = $order->get_payment_method();
				}
				break;

			case 'bu_cust_order_payment_method_title':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = $order->get_payment_method_title();
				}
				break;

			case 'bu_cust_order_customer_note':
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = wp_kses_post( nl2br( wptexturize( $order->get_customer_note() ) ) );
				}
				break;

			case 'bu_order_item_featimg_id':
				$item = self::getOrderItem();
				if( $item ) {
					$product = $item->get_product();

					$filters['object_type'] = 'media';
					$filters['image']       = 'true';
					if ( $product && $product->get_image_id() ) {
						$value = $product->get_image_id();
					} elseif ( $product && $product->get_parent_id() ) {
						$parent_product = wc_get_product( $product->get_parent_id() );
						if ( $parent_product ) {
							$value = $parent_product->get_image_id();
						}
					}

					if ( empty( $value ) ) {
						$value = get_option( 'woocommerce_placeholder_image', 0 );
					}
				}
				break;

			case 'bu_order_item_name':
				$item = self::getOrderItem();
				if( $item )
					$value = $item->get_name();
				break;
				
			case 'bu_order_item_quantity':
				$item = self::getOrderItem();
				$order = self::getCustomerOrder();
				if( $item && $order ) {
					$qty          = $item->get_quantity();
					$refunded_qty = $order->get_qty_refunded_for_item( $item->get_id() );

					if ( $refunded_qty ) {
						$qty_display = '<del>' . esc_html( $qty ) . '</del> <ins>' . esc_html( $qty - ( $refunded_qty * -1 ) ) . '</ins>';
					} else {
						$qty_display = esc_html( $qty );
					}
					
					$value = $qty_display;
				}
				break;
				
			case 'bu_order_item_price':
				$item = self::getOrderItem();
				$order = self::getCustomerOrder();
				if( $item && $order )
					$value = $order->get_formatted_line_subtotal( $item );
				break;

			case 'bu_order_item_sku':
				$item = self::getOrderItem();
				if( $item ) {
					$product = $item->get_product();
					$value = $product ? $product->get_sku() : '';
				}
				break;

			case 'bu_order_item_custom_data':
				$item = self::getOrderItem();
				if( $item ) {
					$product = $item->get_product();
					$meta_key = isset( $filters['meta_key'] ) ? $filters['meta_key'] : false;
					$value = $meta_key ? get_post_meta( $product->get_id(), $meta_key, true ) : '';
				}
				break;

			case 'bu_order_item_variable_attributes':
				$item = self::getOrderItem();
				if( $item ) {
					$value = wc_display_item_meta( $item );
				}
				break;

			case "bu_cust_billing_first_name" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_first_name() );
				}
				break;

			case "bu_cust_billing_last_name" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_last_name() );
				}
				break;

			case "bu_cust_billing_email" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_email() );
				}
				break;

			case "bu_cust_billing_phone" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_phone() );
				}
				break;

			case "bu_cust_billing_address_1" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_address_1() );
				}
				break;

			case "bu_cust_billing_address_2" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_address_2() );
				}
				break;

			case "bu_cust_billing_city" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_city() );
				}
				break;

			case "bu_cust_billing_state" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_state() );
				}
				break;

			case "bu_cust_billing_country" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_country() );
				}
				break;

			case "bu_cust_billing_postcode" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_postcode() );
				}
				break;

			case "bu_cust_billing_company" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_billing_company() );
				}
				break;

			case "bu_cust_shipping_first_name" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_first_name() );
				}
				break;

			case "bu_cust_shipping_last_name" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_last_name() );
				}
				break;

			case "bu_cust_shipping_phone" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_phone() );
				}
				break;

			case "bu_cust_shipping_address_1" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_address_1() );
				}
				break;

			case "bu_cust_shipping_address_2" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_address_2() );
				}
				break;

			case "bu_cust_shipping_city" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_city() );
				}
				break;

			case "bu_cust_shipping_state" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_state() );
				}
				break;

			case "bu_cust_shipping_country" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_country() );
				}
				break;

			case "bu_cust_shipping_postcode" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_postcode() );
				}
				break;

			case "bu_cust_shipping_company" :
				$order = self::getCustomerOrder();
				if( $order ) {
					$value = esc_html( $order->get_shipping_company() );
				}
				break;

			case "buob_product_id" :
				global $product;
				$product_id = get_post_meta( $post_id, '_buob_offer_product', true );
				$product = wc_get_product( $product_id );
				if( $product !== false ) {
					$value = $product->get_id();
				}
				break;

			case "buob_product_image":
				$filters['object_type'] = 'media';
				$filters['image']       = 'true';

				global $product;
				$product_id = get_post_meta( $post_id, '_buob_offer_product', true );
				$product = wc_get_product( $product_id );
				if( $product !== false ) {
					if ( $product->get_image_id() ) {
						$value = $product->get_image_id();
					} elseif ( $product->get_parent_id() ) {
						$parent_product = wc_get_product( $product->get_parent_id() );
						if ( $parent_product ) {
							$value = $product->get_image_id();
						}
					}
				}

				if ( empty( $value ) && get_post_type( $post_id ) === 'product' ) {
					$value = get_option( 'woocommerce_placeholder_image', 0 );
				}
				break;

			case "buob_product_title":
				global $product;
				$product_id = get_post_meta( $post_id, '_buob_offer_product', true );
				$product = wc_get_product( $product_id );
				if( $product !== false ) {
					$value = $product->get_title();
				}
				break;

			case "buob_product_price":
				global $product;
				$product_id = get_post_meta( $post_id, '_buob_offer_product', true );
				$product = wc_get_product( $product_id );
				if( $product !== false ) {
					$value = \BricksUltimate\WooCommerce\WooOrderBumps::get_instance()->get_offer_product_price( $product, $post_id );
				}
				break;

			case 'buob_discount_amount':
				$discount_type = get_post_meta( $post_id, '_buob_discount_type', true ) ?? 'percent';
				$discount      = get_post_meta( $post_id, '_buob_offer_discount', true ) ?? 0;

				$value = ($discount_type == 'percent') ? $discount . '%' : wc_price( $discount );
				break;

			case 'buob_product_quantity':
				$quantity = get_post_meta( $post_id, '_buob_offer_product_quantity', true );
				$value = ! empty( $quantity ) ? $quantity : 1;
				break;

			case 'buob_has_quantity_field':
				$has_quantity_field = get_post_meta( $post_id, '_buob_has_quantity_field', true ); 
				$value = ! empty( $has_quantity_field ) ? $has_quantity_field : "no";
				break;

			case "buob_product_description":
				global $product;
				$product_id = get_post_meta( $post_id, '_buob_offer_product', true );
				$product = wc_get_product( $product_id );
				if( $product !== false ) {
					$value = $product->get_short_description();
				}
				break;

			case "buob_product_org_price":
				global $product;
				$product_id = get_post_meta( $post_id, '_buob_offer_product', true );
				$product = wc_get_product( $product_id );
				if( $product !== false ) {
					$value = $product->get_price_html();
				}
				break;

			case "buob_description":
				$description = get_post_meta( $post_id, '_buob_offer_description', true );
				if( $description ) {
					$value = wp_kses_post( $description );
				}
			break;
		}

		// STEP: Apply context (text, link, image, media)
		$value = $this->format_value_for_context( $value, $tag, $post_id, $filters, $context );

		return $value;
	}

	public static function getQueryObject() {
		$looping_query_id = \Bricks\Query::is_any_looping();

		if ( ! empty( $looping_query_id ) ) {
			return \Bricks\Query::get_loop_object( $looping_query_id );
		}

		return false;
	}

	public static function getCustomerOrder() {
		global $wp;

		$loopObject = $order = false;
		$looping_query_id = \Bricks\Query::is_any_looping();

		if ( ! empty( $looping_query_id ) ) {
			$loopObject = \Bricks\Query::get_loop_object( $looping_query_id );
		}

		// Order pay
		if( $loopObject && ( is_a( $loopObject, '\WC_Order') || is_a( $loopObject, '\Automattic\WooCommerce\Admin\Overrides\Order') ) ) {
			$order = wc_get_order( $loopObject->ID );
		} elseif ( ! empty( get_query_var( 'order-pay' ) ) ) {
			$order_id  = absint( get_query_var( 'order-pay' ) );
			$order = wc_get_order( $order_id );
		} elseif ( ! empty( get_query_var( 'order-received' ) ) ) {
			$order_id = absint( get_query_var( 'order-received' ) );
			$order = wc_get_order( $order_id );
		} elseif ( ! empty( get_query_var( 'view-order' ) ) ) {
			$order_id = absint( get_query_var( 'view-order' ) );
			$order = wc_get_order( $order_id );
		} elseif( isset( $wp->query_vars['view-order'] ) ) {
			$order = wc_get_order( $wp->query_vars['view-order'] );
		} elseif( isset( $_GET['view_order'] ) ) {
			$order = wc_get_order( $_GET['view_order'] );
		} elseif( bricks_is_builder() || bricks_is_builder_iframe() || bricks_is_builder_call() || isset($_GET["bricks_preview"]) ){
			$orders = wc_get_orders(
				[
					'limit' 	=> 1,
					'type' 		=> 'shop_order',
    				'status' 	=> array( 'wc-completed','wc-processing', 'wc-on-hold' ),
				]
			);

			$order = $orders ? $orders[0] : false;
		} elseif( isset( $GLOBALS['tplOrder'] ) ) {
			$order = $GLOBALS['tplOrder'];
		} else {
			$order = false;
		}

		return $order;
	}

	public static function getOrderItem() {
		$item = false;
		$looping_query_id = \Bricks\Query::is_any_looping();

		if ( ! empty( $looping_query_id ) ) {
			$item = \Bricks\Query::get_loop_object( $looping_query_id );
		}
		
		return $item;
	}

	private function product_price( $product, $objectType, $withTax = false ) {
		$quantity = 1;
		$cart_item = $this->getCartItem( $product, $objectType );

		if( $cart_item !== false ) {
			$product = $cart_item['data'];
			$quantity = $cart_item['quantity'];
		}

		if( $withTax )
			return wc_get_price_including_tax( $product, [ 'qty' => $quantity ] );
		
		return wc_get_price_excluding_tax( $product, [ 'qty' => $quantity ] );
	}

	private function getCartItem( $product , $objectType ) {
		$cart_product = false;
		if( ! empty( $objectType ) && $objectType == 'buwooMiniCart' 
			&& is_object( WC()->cart ) && count( WC()->cart->get_cart_contents() ) > 0 ) 
		{
			foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) 
			{
				if( $cart_item['product_id'] == $product->get_id() 
					|| $cart_item['variation_id'] == $product->get_id() ) 
				{
					$cart_product = $cart_item;
				}
			}
		}

		return $cart_product;
	}
}