<?php
/**
 * Boom Bar test page class
 *
 * This class manages the page that allows you to test conditionals
 *
 * @package IT_Boom_Bar
 * @since 1.1.0
*/
class IT_Boom_Bar_Test_Conditionals_Page {

	/**
	 * The name WordPress uses to associate to the page.
	 * @since 1.1.0
	*/
	var $page_var = 'it-boombar-test-conditionals';

	/**
	 * Settings data as found in storage or as modified on form submission in the event of an error.
	 * @since 1.1.0
	*/
	var $options = array();

	/**
	 * Error messages produced while processing form submissions.
	 * @since 1.1.0
	*/
	var $errors = array();

	/**
	 * Status messages produced while processing form submissions.
	 * @since 1.1.0
	*/
	var $messages = array();


	/**
	 * Class constructor
	 *
	 * @uses add_action()
	 * @since 1.1.0
	 * @return null
	*/
	function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
	}

	/**
	 * Add the settings page to the Dashboard menu.
	 *
	 * @uses add_submenu_page()
	 * @since 1.1.0
	 * @return null
	*/
	function add_admin_menu() {
		$page_title = _x( 'Test Visitor Conditions', 'page title', 'it-l10n-boombar' );
		$menu_title = _x( 'Preview Bars', 'menu title', 'it-l10n-boombar' );

		add_submenu_page( 'edit.php?post_type=it_boom_bar', $page_title, $menu_title, 'edit_posts', $this->page_var, array( $this, 'index' ) );
	}

	/**
	 * Save settings when appropriate and show the settings page.
	 *
	 * @since 1.1.0
	 * @return null
	*/
	function index() {
		$this->options = $GLOBALS['it_boom_bar']->get_plugin_options();

		if ( isset( $_POST['submit'] ) )
			$this->save_settings();

		$this->show_settings();
	}

	/**
	 * Save submitted form values.
	 *
	 * @since 1.1.0
	 * @return null
	*/
	function save_settings() {
		check_admin_referer( $this->page_var );


		$errors = array();
		$messages = array();

		if ( empty( $errors ) ) {
			$GLOBALS['it_boom_bar']->update_plugin_options( $this->options );
			
			$messages[] = __( 'Settings saved.', 'it-l10n-boombar' );
		} else {
			$errors[] = _n( 'Due to the error, the settings were not saved.', 'Due to the errors, the settings were not saved.', count( $errors ), 'it-l10n-boombar' );
		}


		$this->errors = array_merge( $this->errors, $errors );
		$this->messages = array_merge( $this->messages, $messages );
	}

	/**
	 * Show the settings page form.
	 *
	 * @since 1.1.0
	 * @return null
	*/
	function show_settings() {
		$action_url = admin_url( 'edit.php?post_type=it_boom_bar' ) . '&page=' . $this->page_var;


	?>
	<style type="text/css">
		#adminmenuback { z-index: -1; }
	</style>
	<div class="wrap">
		<div id="icon-edit" class="icon32 icon32-posts-it_boom_bar">
			<br />
		</div>

		<h2><?php _e( 'Test Visitor Conditions', 'it-l10n-boombar' ); ?></h2>

		<?php
			foreach ( $this->messages as $message )
				echo "<div class=\"updated fade\"><p><strong>$message</strong></p></div>\n";

			foreach ( $this->errors as $error )
				echo "<div class=\"error\"><p><strong>$error</strong></p></div>\n";

			$content_id         = empty( $_POST['content_id'] ) ? '' : $_POST['content_id'];
			$user_authenticated = empty( $_POST['user_authenticated'] ) || ( $_POST['user_authenticated'] != 'true' ) ? false : true;
			$first_time_visitor = empty( $_POST['first_time_visitor'] ) || ( $_POST['first_time_visitor'] != 'true' ) ? false : true;
			$timestamp          = empty( $_POST['timestamp'] ) ? date( 'U' ) : $_POST['timestamp'] / 1000;
			$day_of_week        = ! isset( $_POST['day_of_week'] ) ? date( 'w', current_time( 'timestamp' ) ) : $_POST['day_of_week'];
			$referral           = empty( $_POST['referral'] ) ? '' : $_POST['referral'];
		?>

		<p>
			<?php
			$string = "It is possible to display different bars to site visitors based on the Display Conditions of the bar and the attributes of the site visitor. eg: First time visitors, referral source, etc. This tool allows you to imitate different viewing conditions to ensure the proper bar is displayed.";
			printf( __( $string, 'it-l10n-boombar' ), 'BoomBar' );
			?>
		</p>
		<form action="" method="post">
			<table class="form-table">
				<tbody>
					<tr>
						<th scope="row">
							<label for="content_id"><?php _e( 'What Page or Post do you want to test for?', 'it-l10n-boombar' ); ?></label>
						</th>
						<td>
							<select name="content_id">
								<?php echo $this->get_page_post_select_options( $content_id ); ?>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="user_authenticated"><?php _e( 'Is the user logged in?', 'it-l10n-boombar' ); ?></label>
						</th>
						<td>
							<select name="user_authenticated">
								<option value="false" <?php selected( false, $user_authenticated ); ?>><?php _e( 'No', 'it-l10n-boombar' ); ?></option>
								<option value="true" <?php selected( true, $user_authenticated ); ?>><?php _e( 'Yes', 'it-l10n-boombar' ); ?></option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="first_time_visitor"><?php _e( 'Is the user a first time visitor?', 'it-l10n-boombar' ); ?></label>
						</th>
						<td>
							<select name="first_time_visitor">
								<option value="false" <?php selected( false, $first_time_visitor ); ?>><?php _e( 'No', 'it-l10n-boombar' ); ?></option>
								<option value="true" <?php selected( true, $first_time_visitor ); ?>><?php _e( 'Yes', 'it-l10n-boombar' ); ?></option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="timestamp-display"><?php _e( 'Date / Time of visit. Leave blank for current time.', 'it-l10n-boombar' ); ?></label>
						</th>
						<td>
							<input type="text" id="timestamp-display" class="medium-text" value="<?php esc_attr_e( date( 'Y-m-d', $timestamp ) ); ?>" />
							<input type="hidden" id="timestamp" name="timestamp" value="<?php esc_attr_e( $timestamp * 1000 ); ?>" />
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="day_of_week"><?php _e( 'Day of Week? This will override the day of the timestamp.', 'it-l10n-boombar' ); ?></label>
						</th>
						<td>
							<select name="day_of_week">
								<option value="0" <?php selected( 0, $day_of_week ); ?>><?php _e( 'Sunday', 'it-l10n-boombar' ); ?></option>
								<option value="1" <?php selected( 1, $day_of_week ); ?>><?php _e( 'Monday', 'it-l10n-boombar' ); ?></option>
								<option value="2" <?php selected( 2, $day_of_week ); ?>><?php _e( 'Tuesday', 'it-l10n-boombar' ); ?></option>
								<option value="3" <?php selected( 3, $day_of_week ); ?>><?php _e( 'Wednesday', 'it-l10n-boombar' ); ?></option>
								<option value="4" <?php selected( 4, $day_of_week ); ?>><?php _e( 'Thursday', 'it-l10n-boombar' ); ?></option>
								<option value="5" <?php selected( 5, $day_of_week ); ?>><?php _e( 'Friday', 'it-l10n-boombar' ); ?></option>
								<option value="6" <?php selected( 6, $day_of_week ); ?>><?php _e( 'Saturday', 'it-l10n-boombar' ); ?></option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="referral"><?php _e( 'Referring URL', 'it-l10n-boombar' ); ?></label>
						</th>
						<td>
							<input class="regular-text" placeholder="http://google.com" type="text" name="referral" value="<?php esc_attr_e( $referral ); ?>" />
						</td>
					</tr>
				</tbody>
			</table>
			<p><input class="button button-primary" type="submit" value="<?php esc_attr_e( 'Test Site Visitor Condition', 'it-l10n-boombar' ); ?>" /></p>
		</form>
		<script type="text/javascript">
			var _boombar = _boombar || [];
			_boombar.push(['host', '<?php echo esc_js( site_url( '', is_ssl() ? 'https' : 'http' ) ); ?>']);
			<?php if ( ! empty( $content_id ) ) : ?>
				_boombar.push(['content_id', '<?php echo esc_js( $content_id ); ?>']);
			<?php endif; ?>
			_boombar.push(['user_authenticated', '<?php echo esc_js( $user_authenticated ); ?>']);
			_boombar.push(['first_time_visitor', '<?php echo esc_js( $first_time_visitor ); ?>']);
			<?php if ( ! empty( $timestamp ) ) : ?>
				_boombar.push(['timestamp', '<?php echo esc_js( $timestamp ); ?>']);
			<?php endif; ?>
			<?php if ( ! empty( $day_of_week ) ) : ?>
				_boombar.push(['day_of_week', '<?php echo esc_js( $day_of_week ); ?>']);
			<?php endif; ?>
			<?php if ( ! empty( $referral ) ) : ?>
				_boombar.push(['referral', '<?php echo esc_js( esc_url( $referral ) ); ?>']);
			<?php endif; ?>
			(function(window, document) {
				var loader = function() {
					var bb = document.createElement('script'); bb.type = 'text/javascript'; bb.async = true;
					bb.src = '<?php echo esc_js( site_url( '?bb-embed=1', is_ssl() ? 'https' : 'http' ) ); ?>';
					var s = document.getElementsByTagName('script')[0]; s.parentNode.insertBefore(bb, s);
				};
				window.addEventListener ? window.addEventListener("load", loader, false) : window.attachEvent("onload", loader);
			})(window, document);
		</script>
	</div>
	<?php
	}

	function get_page_post_select_options( $selected ) {
		$posts = get_posts( array( 'post_type' => 'post', 'post_status' => 'publish', 'posts_per_page' => -1 ) );
		$pages = get_posts( array( 'post_type' => 'page', 'post_status' => 'publish', 'posts_per_page' => -1 ) );

		$options = '<option value="0">' . __( 'Unspecified', 'it-l10n-boombar' ) . '</option>';
		$options .= '<optgroup label="' . esc_attr( 'Pages' ) . '">';
		foreach( (array) $pages as $page ) {
			$options .= '<option value="' . esc_attr( $page->ID ) . '" ' . selected( $selected, $page->ID, false ) . '>' . get_the_title( $page->ID ) . '</option>';
		}
		$options .= '</optgroup>';

		$options .= '<optgroup label="' . esc_attr( 'Posts' ) . '">';
		foreach( (array) $posts as $post ) {
			$options .= '<option value="' . esc_attr( $post->ID ) . '" ' . selected( $selected, $post->ID, false ) . '>' . get_the_title( $post->ID ) . '</option>';
		}
		$options .= '</optgroup>';
		
		return $options;
	}
}
