<?php
/**
 * Bars Class - Determines what bars should be displayed on this page view
 *
 * @package IT_Boom_Bar
 * @since 0.1
*/
if ( ! class_exists( 'IT_Boom_Bar_Bars' ) ) {
	class IT_Boom_Bar_Bars {

		/**
		 * @var array $current_conditions A multi-dimentional array of the current pageview's conditions
		 * @since 0.1
		*/
		var $current_conditions;

		/**
		 * @var array $visible_bars An array of objects containing all published bars
		 * @since 0.1
		*/
		var $visible_bars = array();

		/**
		 * @var array $current_bars An array of objects containing all bars to be shown on current pageview
		 * @since 0.1
		*/
		var $current_bars = array();

		/**
		 * @var boolean $default_is_single_override
		 * @since 1.2.0
		*/
		var $default_is_single_override = false;

		/**
		 * @var int $top_margin Offset for the top margin
		 * @since 0.1
		*/
		var $top_margin = 0;

		var $is_forced_id = false;

		/**
		 * Class Constructor. Add's correct actions / filters based on where we are in WordPress
		 *
		 * @uses is_admin()
		 * @uses add_action()
		 * @uses add_filter()
		 * @since 0.1
		 * @return null
		*/
		function __construct( $embeded_bar=false ) {
			// Abandon automated load if embedding a bar
			if ( ! empty( $embeded_bar ) ) {
				return;
			}

			add_action( 'init', array( $this, 'register_bars_post_type' ) );
			if ( ! is_admin() ) {
				add_action( 'template_redirect', array( $this, 'load_current_bars' ) );
			} else {
				add_filter( 'manage_it_boom_bar_posts_columns', array( $this, 'filter_admin_columns' ) );
				add_action( 'manage_it_boom_bar_posts_custom_column' , array( $this, 'add_admin_column_data' ), 10, 2 );
				add_filter( 'post_row_actions', array( $this, 'remove_quick_edit' ) );
				add_filter( 'screen_layout_columns', array( $this, 'modify_add_edit_page_layout' ) );
				add_filter( 'pre_get_posts', array( $this, 'set_default_sort' ) );

				// Force user settings to 1 column view for add / edit bars
				add_filter( 'get_user_option_screen_layout_it_boom_bar', '__return_true' );
			}
		}

		/**
		 * Register's our custom post type with WP
		 *
		 * @uses register_post_type()
		 * @since 0.1
		 * @return null
		*/
		function register_bars_post_type() {
			register_post_type( 'it_boom_bar', array(
				'labels' => array(
					'name'               => _x( 'Bars', 'post type general name', 'it-l10n-boombar' ),
					'singular_name'      => _x( 'Bar', 'singular name', 'it-l10n-boombar' ),
					'menu_name'          => _x( 'Boom Bar' , 'name in admin menu', 'it-l10n-boombar' ),
					'name_admin_bar'     => _x( 'Boom Bar', 'add new from admin bar', 'it-l10n-boombar' ),
					'all_items'          => _x( 'Existing Bars', 'existing bars', 'it-l10n-boombar' ),
					'add_new'            => _x( 'Add New Bar', 'add new bar', 'it-l10n-boombar' ),
					'add_new_item'       => _x( 'Add New Bar', 'add new page title', 'it-l10n-boombar' ),
					'edit_item'          => _x( 'Edit Bar', 'edit bar', 'it-l10n-boombar' ),
					'view_item'          => _x( 'View Bar', 'view bar', 'it-l10n-boombar' ),
					'not_found_in_trash' => _x( 'No Bars found in trash', 'no bars in trash', 'it-l10n-boombar' ),
				),
				'public'               => false,
				'show_ui'              => true,
				'capability_type'      => 'post',
				'map_meta_cap'         => true,
				'hierarchical'         => false,
				'rewrite'              => false,
				'query_var'            => false,
				'delete_with_user'     => false,
				'supports'             => array( 'title', ),
				'menu_position'        => 100,
				'register_meta_box_cb' => array( $this, 'add_metaboxes' ),
			) );
		}

		/**
		 * Determines correct bars to display on current page
		 *
		 * @uses IT_Boom_Bar_Bars::build_current_conditions()
		 * @uses IT_Boom_Bar_Bars::set_visible_bars()
		 * @uses IT_Boom_Bar_Bars::set_current_bars()
		 * @since 0.1
		 * @return null
		*/
		function load_current_bars() {
			// Grab the current page conditionals
			$this->build_current_conditions();
			$this->set_visible_bars();
			$this->set_current_bars();
		}

		/**
		 * Sets current conditions for this page view
		 *
		 * @todo Use WP time localization functions for when object
		 * @uses IT_Boom_Bar_Bars::get_default_bar_for_pageview()
		 * @uses is_user_logged_in()
		 * @uses IT_Boom_Bar_Bars::is_first_time_visitor()
		 * @uses IT_Boom_Bar_Bars::is_referral()
		 * @since 0.1
		 * @return null
		*/
		function build_current_conditions( $forced_conditions=false ) {
			global $current_user;

			// Forced conditon defaults
			$fc_defaults = array(
				'bar'                => false,
				'content_id'         => false,
				'user_authenticated' => false,
				'first_time_visitor' => false,
				'timestamp'          => current_time( 'timestamp' ),
				'day_of_week'        => date( 'w', current_time( 'timestamp' ) ),
				'referral'           => false,
			);
			$forced_referral = false;
			if ( ! empty( $forced_conditions ) ) {
				$forced_conditions = wp_parse_args( $forced_conditions, $fc_defaults );
				$forced_referral   = $forced_conditions['referral'];
				// Set default bar for forced conditions
				if ( empty($forced_conditions['bar'] ) && ! empty( $forced_conditions['content_id'] ) ) {
					$forced_conditions['bar'] = $this->get_default_bar_for_pageview( $forced_conditions['content_id'] );
				}
				// Clean up string data for booleans
				if ( $forced_conditions['user_authenticated'] === 'false' ) { $forced_conditions['user_authenticated'] = false; }
				if ( $forced_conditions['user_authenticated'] === 'true' ) { $forced_conditions['user_authenticated'] = true; }
				if ( $forced_conditions['first_time_visitor'] === 'false' ) { $forced_conditions['first_time_visitor'] = false; }
				if ( $forced_conditions['first_time_visitor'] === 'true' ) { $forced_conditions['first_time_visitor'] = true; }
			}

			$conditions = new stdClass();
			$conditions->who = new stdClass();
			$conditions->when = new stdClass();
			$conditions->referer = new stdClass();

			$conditions->default_bar              = isset( $forced_conditions['bar'] ) ? $forced_conditions['bar'] : $this->get_default_bar_for_pageview();
			$user_authenticated                   = isset( $forced_conditions['user_authenticated'] ) ? $forced_conditions['user_authenticated'] : is_user_logged_in();
			$conditions->who->everyone            = ! $user_authenticated;
			$conditions->who->authenticated_users = $user_authenticated;
			$conditions->who->first_time_visitor  = isset( $forced_conditions['first_time_visitor'] ) ? $forced_conditions['first_time_visitor'] : $this->is_first_time_visitor();
			$conditions->who->returning_visitors  = ! $conditions->who->first_time_visitor;
			$conditions->when->startdate          = isset( $forced_conditions['timestamp'] ) ? $forced_conditions['timestamp'] : current_time( 'timestamp' );
			$conditions->when->enddate            = $conditions->when->startdate;
			$conditions->when->dayofweek          = isset( $forced_conditions['day_of_week'] ) ? $forced_conditions['day_of_week'] : date( 'w', $conditions->when->startdate );
			$conditions->referer->google          = $this->is_referral( array( 'google.' ), $forced_referral );
			$conditions->referer->twitter         = $this->is_referral( array( 'twitter.', 't.co' ), $forced_referral );
			$conditions->referer->facebook        = $this->is_referral( array( 'facebook.' ), $forced_referral );
			$conditions->referer->linkedin        = $this->is_referral( array( 'linkedin.', 't.co' ), $forced_referral );
			$conditions->referer->bing            = $this->is_referral( array( 'bing.' ), $forced_referral );

			if ( ! empty( $forced_conditions['bar'] ) ) {
				$this->is_forced_id = $forced_conditions['bar'];
			}
			$this->current_conditions             = $conditions;
		}

		/**
		 * Grabs the default bar for a frontend pageview
		 *
		 * If this is a singular pageview, see if there is an override to the default bar
		 * Otherwise, use the default bar, if one exists
		 * If one exists, but user has closed it and cookie is still valid, return 0
		 *
		 * @uses is_singular()
		 * @uses get_post_meta()
		 * @since 0.2
		 * @return boolean|int False if no bar. Bar ID if we found one.
		*/
		function get_default_bar_for_pageview( $forced_post_id=false ) {
			global $it_boom_bar, $post;

			if ( empty( $post->ID ) && empty( $forced_post_id ) ) {
				if ( is_callable( 'is_shop' ) && is_shop() ) {
					$is_shop = true;
					$forced_post_id = wc_get_page_id( 'shop' );
				} else {
					return;
				}
			}

			$post_id = empty( $forced_post_id ) ? $post->ID : $forced_post_id;

			// Maybe update the default bar
			if ( ! empty( $post_id ) && ( is_singular( get_post_type( $post_id ) ) || ! empty( $is_shop ) ) && $bar = get_post_meta( $post_id, '_it_boom_bar_pt_override', true ) ) {
				$default = $bar;
				$this->default_is_single_override = $bar;
			} else {
				$default = $it_boom_bar->options['default_bar'];
			}

			// Return 0 if current user has cookie saying they closed the bar
			if ( ! empty( $_COOKIE['it_boom_bar_' . $default] ) && empty( $forced_post_id ) ) {
				$default = 0;
			}

			return $default;
		}

		/**
		 * Reads cookies to determine if they've been here before
		 *
		 * Not technically 'first time' visitor. more like, in recent history.
		 * If it is first time, lets set the cookie.
		 *
		 * @since 0.1
		 * @return boolean True if first time visitor. False if not.
		*/
		function is_first_time_visitor() {
			$time        = current_time( 'timestamp' );
			$twenty4ago  = $time - DAY_IN_SECONDS;
			$first_visit = empty( $_COOKIE['it_boombar_first_visit'] ) ? $time : $_COOKIE['it_boombar_first_visit'];

			// Set cookie if it doesn't exist
			if ( empty( $_COOKIE['it_boombar_first_visit'] ) ) {
				$this->set_first_visit_cookie( $first_visit );
			}

			// Return true if first visit was within last 24 hours
			if ( $first_visit >= $twenty4ago ) {
				return true;
			}

			// Return false
			return false;
		}

		/**
		 * Set COOKIE
		 *
		 * @since 1.1.21
		 *
		 * @param int $first_visit unix timestamp of first visit time
		 * @return void
		 */
		function set_first_visit_cookie( $first_visit ) {
			setcookie( 'it_boombar_first_visit', $first_visit, strtotime( '+1 years' ), COOKIEPATH, COOKIE_DOMAIN );
		}

		/**
		 * Returns referral is from a specific domain
		 *
		 * Use HTTP_REFERER for simple solution
		 *
		 * @since 0.1
		 * @return boolean True if from domain. False if not.
		*/
		function is_referral( $sources, $referral=false ) {

			// If forced, alwasy use that
			$referer = empty( $referral ) && ! empty( $_SERVER['HTTP_REFERER'] ) ? parse_url( $_SERVER['HTTP_REFERER'], PHP_URL_HOST ) . parse_url( $_SERVER['HTTP_REFERER'], PHP_URL_PATH ) : $referral;

			// Is referer relative and not forced?
			if ( empty( $referral ) && $this->is_url_relative( $referer ) ) {
				$referer = $this->get_cookie_referral_url();
			} else if ( empty( $referral ) && ! $this->is_url_relative( $referer ) ) {
				$cookie = $this->get_cookie_referral_url();
				if ( $cookie !== $referer && ! empty( $referer ) ) {
					$this->set_referral_cookie_url( $referer );
				}
			}

			// Return false if referer is empty after all of the above.
			if ( empty( $referer ) ) {
				return false;
			}

			// Loop through our sources and see if we have a match
			foreach( (array) $sources as $source ) {
				if ( false !== strpos( $referer, $source ) ) {
					return true;
				}
			}

			return false;
		}

		/**
		 * Set the referral saved in the cookie
		 *
		 * @since 1.2.0
		 *
		 * @param value
		 * @return void
		 */
		function set_referral_cookie_url( $referer ) {
			setcookie( 'it_boombar_referral', $referer, strtotime( '+1 years' ), COOKIEPATH, COOKIE_DOMAIN );
		}

		/**
		 * Get the REFERRAL SAVED IN THE COOKIE
		 *
		 * @since 1.2.0
		 *
		 * @return mixed false or http string
		 */
		function get_cookie_referral_url() {
			// Grab referral saved in the cookie
			return empty( $_COOKIE['it_boombar_referral'] ) ? false : $_COOKIE['it_boombar_referral'];
		}

		/**
		 * Compares passed string with siteurl to see if it's the same
		 *
		 * @since 1.2.0
		 *
		 * @param $string url
		 * @return boolean
		 */
		function is_url_relative( $url ) {
			$url_host = parse_url( $url, PHP_URL_HOST );
			$url_path = parse_url( $url, PHP_URL_PATH );
			$url      = trailingslashit( $url_host . $url_path );
			$site_url = get_option( 'siteurl' );
			$siteurl  = trailingslashit( parse_url( $site_url, PHP_URL_HOST ) . parse_url( $site_url, PHP_URL_PATH ) );

			return ( substr( $url, 0, strlen( $siteurl ) ) === $siteurl );
		}


		/**
		 * Queiries the DB for all visible bars and sets the associated property
		 *
		 * @uses get_posts()
		 * @since 0.1
		 * @return null
		*/
		function set_visible_bars() {
			global $it_boom_bar;

			// Return the override if available
			if ( ! empty( $this->default_is_single_override ) ) {
				if ( -1 == $this->default_is_single_override ) {
					$this->visible_bars = array();
				} else {
					$this->visible_bars = array( get_post( $this->default_is_single_override ) );
				}
				return;
			}

			// Return the default bar if forced through
			if ( ! empty( $this->is_forced_id ) ) {
				$this->visible_bars = array( get_post( $this->is_forced_id ) );
				return;
			}

			// Grab global bars
			$default_params = array(
				'post_type'      => 'it_boom_bar',
				'post_status'    => 'publish',
				'orderby'        => 'menu_order',
				'order'          => 'ASC',
				'posts_per_page' => -1,
			);

			// Query DB for visible bar
			$this->visible_bars = get_posts( $default_params );

			foreach( $this->visible_bars as $key => $bar ) {

				// Should be it_boom_bar from above query but in the event a plugin modifies the query,
				// do a sanity check here
				if ( 'it_boom_bar' !== get_post_type( $bar->ID ) ) {
					unset( $this->visible_bars[$key] );
					continue;
				}

				// Remove any that aren't global options
				$meta = get_post_meta( $bar->ID, '_it_boom_bar', true );
				if ( ! empty( $meta['bar_global'] ) && 'no' == $meta['bar_global'] ) {
					unset( $this->visible_bars[$key] );
					continue;
				}

				// Make sure after start date and before end date if set.
				$bb_meta   = get_post_meta( $bar->ID, '_it_boom_bar', true );
				$startdate = empty( $bb_meta['conditions']['when']['startdate'] ) ? false : $bb_meta['conditions']['when']['startdate'];
				$enddate   = empty( $bb_meta['conditions']['when']['enddate'] ) ? false : $bb_meta['conditions']['when']['enddate'];
				$current   = current_time( 'timestamp' );
				if ( ( ! empty( $startdate ) && $startdate > $current ) || ( ! empty( $enddate ) && $enddate < $current ) ) {
					unset( $this->visible_bars[$key] );
					continue;
				}
			}

			// Hacky sanity check. Nothing will be displayed if this fires
			if ( ! array( $this->visible_bars ) )
				$this->visible_bars = array();
		}

		/**
		 * Filters the visible bars array for current conditions
		 *
		 * Since 0.2 - only one bar will be returned.
		 *
		 * @since 0.1
		 * @return null
		*/
		function set_current_bars() {
			$visible_bars = $this->visible_bars;
			$bar_set      = false;

			foreach( $visible_bars as $id => $post_obj ) {
				$bar = new IT_Boom_Bar_Single_Bar( $post_obj );

				if ( $this->is_current_bar( $bar ) && empty( $bar_set ) ) {
					if ( empty( $bar->closable ) || $bar->closable == 'no' || ( $bar->closable == 'yes' && empty( $_COOKIE['it_boom_bar_' . $bar->id] ) ) ) {
						$this->current_bars[$bar->id]	= $bar;
						$bar_set = true;
						unset( $bar );
					}
				}
			}
		}

		/**
		 * Validates that a given bar object matches the current conditons
		 *
		 * @param object $bar An instance of IT_Boom_Bar_Single_Bar
		 * @uses is_user_logged_in()
		 * @since 0.1
		 * @return boolean True if the user can see the referenced bar. False if it can't.
		*/
		function is_current_bar( $bar ) {

			// For forced bars by id
			if ( ! empty( $this->is_forced_id ) ) {
				return true;
			}

			$current_user_can_see_bar = false;

			// If bar is login and user is logged in, don't show it
			if ( 'login' == $bar->type && is_user_logged_in() )
				return false;

			// It doesn't matter who you are if we have a limited timeframe. Blacklist outside that timeframe
			if ( ! empty( $bar->conditions['when']['startdate'] ) && ( mysql2date( 'U', $bar->conditions['when']['startdate'] . '00:01:01' ) > $this->current_conditions->when->startdate ) )
				return false;
			if ( ! empty( $bar->conditions['when']['enddate'] ) && ( mysql2date( 'U', $bar->conditions['when']['enddate'] . '23:59:00' ) < $this->current_conditions->when->enddate ) )
				return false;

			// If the current day of the week wasn't checked, don't show the bar. Only exceptions is if no days of the week are checked.
			if ( ! empty( $bar->conditions['when']['dayofweek'] ) && is_array( $bar->conditions['when']['dayofweek'] ) ) {
				if ( ! array_key_exists( $this->current_conditions->when->dayofweek, $bar->conditions['when']['dayofweek'] ) )
					return false;
			}
			/** If we made it here, we aren't going to hide the bar based on when it is being shown **/

			// It's not very conditional if everyone was checked...
			if ( ! empty( $bar->conditions['who']['everyone'] ) )
				$current_user_can_see_bar = true;

			// If All Authenticated Users is checked and the visitor IS logged in, they can see the bar
			if ( ! empty( $bar->conditions['who']['authenticated'] ) && $this->current_conditions->who->authenticated_users )
				$current_user_can_see_bar = true;

			// If All UNauthenticated Users is checked and the visitor is NOT logged in, they can see the bar
			if ( ! empty( $bar->conditions['who']['unauthenticated'] ) && ! $this->current_conditions->who->authenticated_users )
				$current_user_can_see_bar = true;

			// If First Time Visitors is checked and the visitor is here for the first time, they can see the bar
			if ( ! empty( $bar->conditions['who']['first_time_visitors'] ) && $this->current_conditions->who->first_time_visitor )
				$current_user_can_see_bar = true;

			// If Returning Visitors is checked and the visitor is has been here before, they can see the bar
			if ( ! empty( $bar->conditions['who']['returning_visitors'] ) && $this->current_conditions->who->returning_visitors )
				$current_user_can_see_bar = true;

			// We should probably not prevent it from being seen if no 'who' conditions were checked
			if ( empty( $bar->conditions['who'] ) && empty( $bar->conditions['referer'] ) )
				$current_user_can_see_bar = true;

			// If coming from Google and Google is checked, set as true
			if ( ! empty( $bar->conditions['referer']['google'] ) && $this->current_conditions->referer->google ) {
				$current_user_can_see_bar = true;
			}

			// If coming from Twitter and Twitter is checked, set as true
			if ( ! empty( $bar->conditions['referer']['twitter'] ) && $this->current_conditions->referer->twitter ) {
				$current_user_can_see_bar = true;
			}

			// If coming from Facebook and Facebook is checked, set as true
			if ( ! empty( $bar->conditions['referer']['facebook'] ) && $this->current_conditions->referer->facebook) {
				$current_user_can_see_bar = true;
			}

			// If coming from LinkedIn and LinkedIn is checked, set as true
			if ( ! empty( $bar->conditions['referer']['linkedin'] ) && $this->current_conditions->referer->linkedin ) {
				$current_user_can_see_bar = true;
			}

			// If coming from Bing and Bing is checked, set as true
			if ( ! empty( $bar->conditions['referer']['bing'] ) && $this->current_conditions->referer->bing ) {
				$current_user_can_see_bar = true;
			}

			return $current_user_can_see_bar;
		}

		/**
		 * Metabox callback for it_boom_bar post type
		 *
		 * Hooked from the register_post_type. All metaboxes get added/removed from in here
		 *
		 * @uses remove_meta_box()
		 * @uses add_meta_box()
		 * @since 0.1
		 * @return null
		*/
		function add_metaboxes() {
			// new metaboxes must be whitelisted in restrict_third_party_meta_boxes() in classes/admin.php
			remove_meta_box( 'submitdiv', null, 'side' );
			add_meta_box( 'it_boom_bar_type', __( 'Bar Content', 'it-l10n-boombar' ), array( $this, 'add_bar_type_form_fields'), 'it_boom_bar', 'normal', 'high' );
			add_meta_box( 'it_boom_bar_settings', __( 'General Settings', 'it-l10n-boombar' ), array( $this, 'add_bar_settings_form_fields'), 'it_boom_bar', 'normal', 'high' );
			add_meta_box( 'it_boom_bar_settings_display', __( 'Display Conditions', 'it-l10n-boombar' ), array( $this, 'add_bar_display_conditions_form_fields'), 'it_boom_bar', 'normal', 'high' );
			add_meta_box( 'it_boom_bar_css', __( 'Custom CSS', 'it-l10n-boombar' ), array( $this, 'add_custom_css_field'), 'it_boom_bar', 'normal', 'high' );
			add_meta_box( 'submitdiv', __( 'Publish' ), array( $this, 'add_publish_meta_box' ), 'it_boom_bar', 'normal' );
		}

		/**
		 * Bar Settings fields
		 *
		 * Includes views/ file with HTML field elements
		 *
		 * @since 0.1
		 * @return null
		*/
		function add_bar_settings_form_fields() {
			global $it_boom_bar, $it_boom_bar_admin;
			require_once( $it_boom_bar->_pluginPath . '/views/add-edit-settings-fields.php' );
		}

		/**
		 * Display Conditions fields
		 *
		 * Includes views/ file with HTML field elements
		 *
		 * @since 1.2.0
		 * @return null
		*/
		function add_bar_display_conditions_form_fields() {
			global $it_boom_bar, $it_boom_bar_admin;
			require( $it_boom_bar->_pluginPath . '/views/add-edit-display-fields.php' );
		}

		/**
		 * Bar Type fields
		 *
		 * Includes views/ file with HTML field elements
		 *
		 * @since 0.1
		 * @return null
		*/
		function add_bar_type_form_fields() {
			global $it_boom_bar, $it_boom_bar_admin;
			require_once( $it_boom_bar->_pluginPath . '/views/add-edit-type-fields.php' );
		}

		/**
		 * Customized Publish Meta Box
		 *
		 * Includes the customize publish meta box
		 * @since 0.1
		 * @return null
		*/
		function add_publish_meta_box( $post ) {
			global $it_boom_bar;
			require_once( $it_boom_bar->_pluginPath . '/views/publish-meta-box.php' );
		}

		/**
		 * Custom CSS Field
		 *
		 * Includes views/ file with HTML field elements
		 *
		 * @since 0.1
		 * @return null
		*/
		function add_custom_css_field( $post ) {
			global $it_boom_bar;
			require_once( $it_boom_bar->_pluginPath . '/views/add-edit-css-fields.php' );
		}

		/**
		 * Returns the current value for a field in the add / edit bar screen
		 *
		 * 1) Is there a value already set
		 * 2) If not, what's the default
		 * 3) Also - take empty into consideration
		 *
		 * @uses get_post_status()
		 * @uses IT_Boom_Bar::get_plugin_options()
		 * @since 0.1
		 * @return string Value of bar setting
		*/
		function get_settings_value( $setting ) {
			global $it_boom_bar;

			if ( is_array( $setting ) ) {
				$existing_value = $it_boom_bar->admin->current_bar->{$setting[0]};
				$real_setting = $setting[0];
				unset( $setting[0] );
				foreach ( $setting as $key ) {
					if ( isset( $existing_value[$key] ) )
						$existing_value = $existing_value[$key];
					else
						$existing_value = false;
				}
				$setting = $real_setting;
			} else {
				if ( 'priority' == $setting  && empty( $it_boom_bar->admin->current_bar->priority ) ) { $it_boom_bar->admin->current_bar->priority = 1; }
				$existing_value = empty( $it_boom_bar->admin->current_bar->$setting ) ? '' : $it_boom_bar->admin->current_bar->$setting;
			}

			if( empty( $existing_value ) ) {
				if ( 'publish' == get_post_status( $it_boom_bar->admin->current_bar->id ) ) {
					$existing_value = '';
				} else {
					$existing_value = $it_boom_bar->get_plugin_options( 'bar_' . $setting );
				}
			}

			return $existing_value;
		}

		/**
		 * Modifies the columns that appear in the Existing Bars table
		 *
		 * @param array $columns Existing columns passed to us by the WP hook
		 * @since 0.1
		 * @return array Associated array of columns
		*/
		function filter_admin_columns( $columns ) {
			unset( $columns['date'] );
			unset( $columns['builder_layout'] );
			$columns['type'] = __( 'Content Type', 'it-l10n-boombar' );
			$columns['priority'] = __( 'Bar Priority', 'it-l10n-boombar' );
			$columns['global'] = __( 'Global Bar', 'it-l10n-boombar' );
			return $columns;
		}

		/**
		 * Populates our custom data in the cells for our custom columns
		 *
		 * @uses IT_Boom_Bar_Bars::get_bar_types()
		 * @since 0.1
		 * @return string Data for cell
		*/
		function add_admin_column_data( $name, $post_id ) {
			global $it_boom_bar;
			$options = $it_boom_bar->get_plugin_options();
			include_once( $it_boom_bar->_pluginPath . '/classes/single-bar.php' );
			if ( $bar = new IT_Boom_Bar_Single_Bar( $post_id ) ) {
				switch( $name ) {
					case 'type' :
						$types = $this->get_bar_types();
						echo empty( $types[$bar->type] ) ? 'Draft' : $types[$bar->type];
						break;
					case 'global' :
						$value  = empty( $bar->global ) ? __( 'No', 'it-l10n-boombar' ) : ucwords( $bar->global );
						echo $value;
						break;
					case 'priority' :
						echo empty( $bar->priority ) ? '' : $bar->priority;
						break;
					default:
					break;
				}
			}
		}

		/**
		 * An assoc array of bar types
		 *
		 * @since 0.1
		 * @return array Bar Types
		*/
		function get_bar_types() {
			$types = array(
				'text'  => __( 'Custom Text', 'it-l10n-boombar' ),
				'tweet' => __( 'Latest Tweet', 'it-l10n-boombar' ),
				'login' => __( 'Log In', 'it-l10n-boombar' ),
			);
			return $types;
		}

		/**
		 * NOT CURRENTLY USED
		 *
		 * Sets the default sort by priority for Existing Bars view in WP Admin
		 * @since 0.1
		*/
		function set_default_sort( $query ) {
			global $current_screen;

			if ( empty( $current_screen->id ) || 'edit-it_boom_bar' != $current_screen->id )
				return;

			if ( ! get_query_var( 'orderby' ) )
				$query->set( 'orderby', 'menu_order' );
			if ( ! get_query_var( 'order' ) )
				$query->set( 'order', 'asc' );
		}

		/**
		 * Hooked filter to remove quick edit for Existing Bars
		 *
		 * @param array $actions Array of action available to be preformd on each existing bar
		 * @since 0.2
		 * @return array Filtered array of actions
		*/
		function remove_quick_edit( $actions ) {
			global $current_screen;

			if ( ! empty( $current_screen->parent_file ) && 'edit.php?post_type=it_boom_bar' == $current_screen->parent_file )
				unset( $actions['inline boombar-hide-if-no-js'] );
			return $actions;
		}

		/**
		 * Set the max columns option for the add / edit bar page.
		 *
		 * @param $columns Existing array of how many colunns to show for a post type
		 * @since 0.2
		 * @return arra Filtered array
		*/
		function modify_add_edit_page_layout( $columns ) {
			$columns['it_boom_bar'] = 1;
			return $columns;
		}
	}
}
