<?php
/**
 * Boom Bar admin class.
 *
 * This class manages the admin side of the plugin
 *
 * @package IT_Boom_Bar
 * @since 0.1
*/
if ( ! class_exists( 'IT_Boom_Bar_Admin' ) ) {
	class IT_Boom_Bar_Admin {

		/**
		 * Stores errors generated by user actions
		 * @since 0.1
		*/
		var $errors      = array();

		/**
		 * @ Stores a single IT_Boom_Bar_Single_Bar instance
		 * @since 0.1
		*/
		var $current_bar = false;

		/**
		 * Class constructor
		 *
		 * @uses add_action()
		 * @since 0.1
		 * @return null
		*/
		function __construct() {
			global $it_boom_bar;

			add_action( 'admin_init', array( $this, 'save_pt_overrides' ) );
			add_action( 'admin_init', array( $this, 'maybe_do_gdpr_nag' ) );
			add_action( 'wp_ajax_it-boombar-dismiss-gdpr-notice', array( $this, 'dismiss_gdpr_nag' ) );
			add_action( 'admin_menu', array( $this, 'hook_admin_menu' ) );
			add_filter( 'post_updated_messages', array( $this, 'post_updated_messages' ) );

			// Set current bar property
			add_action( 'load-post.php', array( $this, 'set_current_bar' ) );
			add_action( 'load-post-new.php', array( $this, 'set_current_bar' ) );

			add_action( 'save_post', array( $this, 'save_bar_data' ), 10, 2 );

			// Register metaboxes for post types
			add_action( 'add_meta_boxes', array( $this, 'register_override_meta_boxes' ) );

			// Whitelist Core and Boom Bar metaboxes
			add_action( 'admin_head', array( $this, 'restrict_third_party_meta_boxes' ), 9999 );

			// Privacy Policy Hooks
			add_action( 'admin_init', array( $this, 'add_privacy_policy_content' ) );
		}

		/**
		 * Sets the current bar property when on new or edit bar page in admin
		 *
		 * @uses get_post()
		 * @uses IT_Boom_Bar_Single_Bar
		 * @since 0.1
		 * @return null
		*/
		function set_current_bar( $post ) {
			global $current_screen;

			// Abort if not the correct post type
			if ( 'it_boom_bar' != $current_screen->post_type )
				return;

			// Establish the post ID. Borrowed from wp-admin/post.php
			if ( isset( $_GET['post'] ) )
				$post_id = $post_ID = (int) $_GET['post'];
			elseif ( isset( $_POST['post_ID'] ) )
				$post_id = $post_ID = (int) $_POST['post_ID'];
			else
				$post_id = $post_ID = 0;

			$post = $post_type = $post_type_object = null;

			if ( $post_id )
				$post = get_post( $post_id );

			$this->current_bar = new IT_Boom_Bar_Single_Bar( $post );
		}

		/**
		 * Add's the link to settings page to the admin menu
		 *
		 * Also provides callback for admin page and enqueues admin CSS
		 *
		 * @todo Look at moving add_action calls for enqueue functions somewhere else so I can use $current_screen
		 * @todo Break out builder and error into their own methods and own hooks
		 * @uses add_action()
		 * @uses add_filter()
		 * @since 0.1
		 * @return null
		*/
		function hook_admin_menu() {
			global $it_boom_bar, $pagenow;

			// Enqueue admin scripts and styles.
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );

			// Is this a new Boom Bar screen?
			$is_boom_bar_bar_page = ( ! empty( $_GET['post_type'] ) && 'it_boom_bar' == $_GET['post_type'] ) ? true : false;
			// Is this an edit Boom Bar screen?
			$is_boom_bar_bar_page = ( $is_boom_bar_bar_page || ( 'post.php' == $pagenow && ! empty( $_GET['post'] ) && 'it_boom_bar' == get_post_type( $_GET['post'] ) ) ) ? true : false;

			if ( in_array( $pagenow, array( 'post-new.php', 'post.php' ) ) && $is_boom_bar_bar_page ) {
				add_action( 'admin_print_styles-' . $pagenow, array( $this, 'enqueue_admin_bars_styles_scripts' ) );

				// Load the preview of the bar
				add_action( 'admin_footer', array( $this, 'add_bar_preview' ) );

				// Remove builder Custom Layout meta-box
				if ( 'builder' == strtolower( get_option( 'template' ) ) )
					add_filter( 'builder_layout_filter_non_layout_post_types', array( $this, 'remove_builder_custom_layout_box' ) );
			}

			if ( 'edit.php' === $pagenow && ! empty( $_GET['post_type'] ) && ! empty( $_GET['page'] ) && $_GET['post_type'] === 'it_boom_bar' && $_GET['page'] === 'it-boombar-test-conditionals' ) {
				add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_bar_preview_styles_scripts' ) );
			}

			// Register errors
			if ( $is_boom_bar_bar_page )
				add_action( 'admin_head', array( $this, 'register_errors' ) );
		}

		/**
		 * Enqueue global admin styles
		 *
		 * @since 0.2
		 * @uses wp_enqueue_style()
		 * @return null
		*/
		function enqueue_admin_styles() {
			global $it_boom_bar, $wp_version;
			
			if ( $wp_version <= '3.7' ) {
				wp_enqueue_style( $it_boom_bar->_slug . '-admin-global', $it_boom_bar->_pluginURL . '/styles/admin-global-pre-3_8.css' );
			} else {
				wp_enqueue_style( $it_boom_bar->_slug . '-admin-global', $it_boom_bar->_pluginURL . '/styles/admin-global.css' );
			}
		}

		/**
		 * Enqueues styles for Add / Edit bars screen
		 *
		 * @uses wp_enqueue_style()
		 * @uses wp_enqueue_script()
		 * @since 0.2
		 * @return null
		 */
		function enqueue_admin_bars_styles_scripts() {
			global $it_boom_bar;
			wp_enqueue_style( $it_boom_bar->_slug . '-admin-bars', $it_boom_bar->_pluginURL . '/styles/admin-bars.css' );
			wp_enqueue_script( $it_boom_bar->_slug . '-admin-bars', $it_boom_bar->_pluginURL . '/js/admin-bars.js' );
			wp_enqueue_script( $it_boom_bar->_slug . '-colorpicker', $it_boom_bar->_pluginURL . '/js/colorpicker/colorpicker.js' );
			wp_enqueue_script( $it_boom_bar->_slug . '-datepicker', $it_boom_bar->_pluginURL . '/js/datepicker/datepicker.js', array( 'jquery', 'jquery-ui-datepicker' ) );
			wp_enqueue_style( $it_boom_bar->_slug . '-colorpicker', $it_boom_bar->_pluginURL . '/js/colorpicker/colorpicker.css' );
			wp_enqueue_style( $it_boom_bar->_slug . '-datepicker', $it_boom_bar->_pluginURL . '/js/datepicker/datepicker.css' );
		}

		/**
		 * Enqueue JS and CSS for Preview Page
		 *
		 * @since 1.2.0
		 *
		 * @return void
		 */
		function enqueue_admin_bar_preview_styles_scripts() {
			global $it_boom_bar;
			wp_enqueue_script( $it_boom_bar->_slug . '-datepicker', $it_boom_bar->_pluginURL . '/js/datepicker/datepicker.js', array( 'jquery', 'jquery-ui-datepicker' ) );
			wp_enqueue_style( $it_boom_bar->_slug . '-datepicker', $it_boom_bar->_pluginURL . '/js/datepicker/datepicker.css' );
		}

		/**
		 * Save boom bar overrides for specifc posts
		 *
		 * @uses update_post_meta()
		 * @since 0.2
		 * @return null
		*/
		function save_pt_overrides() {
			$value = empty( $_POST['it_boom_bar_pt_override'] ) ? 0 : $_POST['it_boom_bar_pt_override'];
			if ( ! empty( $_POST['it_boom_bar_update_pt_override'] ) )
				update_post_meta( $_POST['post_ID'], '_it_boom_bar_pt_override', $_POST['it_boom_bar_pt_override'] );
		}

		/**
		 * Prints any messages
		 *
		 * @uses get_post_meta()
		 * @since 0.1
		 * @return null
		*/
		function register_errors() {
			global $post, $notice;

			// Detect / inject possible add/edit bar errors
			if ( ! empty( $post->post_type ) && 'it_boom_bar' == $post->post_type && $bar_errors = get_post_meta( $post->ID, '_it_boom_bar_errors') )
				$this->errors = array_merge( $this->errors, $bar_errors );

			// Print messages / errors
			if ( ! empty( $this->errors ) ) {
				$errors = implode( '<br />', $this->errors );

				// Append to WP Notices
				$notice .= $errors;
			}
		}

		/**
		 * Loads the JS in the footer for the bar preview
		 *
		 * @since 1.2.0
		 * @return void
		 */
		function add_bar_preview() {
			global $post;
			$status = get_post_status( $post );
			if ( ! in_array( $status, array( 'publish', 'draft', 'future' ) ) ) {
				return;
			}
			?>
			<script type="text/javascript">
				var _boombar = _boombar || [];
				_boombar.push(['host', '<?php echo esc_js( site_url( '', is_ssl() ? 'https' : 'http' ) ); ?>']);
				_boombar.push(['bar', '<?php echo esc_js( $post->ID ); ?>']);
				(function(window, document) {
					var loader = function() {
						var bb = document.createElement('script'); bb.type = 'text/javascript'; bb.async = true;
						bb.src = '<?php echo esc_js( site_url( '?bb-embed=1', is_ssl() ? 'https' : 'http' ) ); ?>';
						var s = document.getElementsByTagName('script')[0]; s.parentNode.insertBefore(bb, s);
					};
					window.addEventListener ? window.addEventListener("load", loader, false) : window.attachEvent("onload", loader);
				})(window, document);
			</script>
			<style type="text/css">
				#adminmenuback { z-index: -1; }
			</style>
			<?php	
		}

		/**
		 * Returns the option elements for the bar type dropdown
		 * If $selected param matches one of $values in the foreach, it will be marked as selected
		 *
		 * @todo Pull from custom taxonomies
		 * @param bool|string $selected False or selected value of field
		 * @uses IT_Boom_Bar_Bars::get_bar_types()
		 * @uses esc_attr()
		 * @uses selected()
		 * @since 0.1
		 * @return string HTML option element for a select field
		*/
		function get_bar_type_select_options( $selected=false ) {
			global $it_boom_bar;
			$options = $it_boom_bar->bars->get_bar_types();
			$html = '';
			foreach( $options as $value => $label ) {
				$html .= '<option value="' . esc_attr( $value ) . '"' .  selected( $selected, $value, false) . '>' . esc_attr( $label ) . '</option>';
			}
			return $html;
		}

        /**
         * Returns the option elements for the bar type dropdown
         * If $selected param matches one of $values in the foreach, it will be marked as selected
         *
         * @param bool|string $selected False or selected value of field
         * @uses esc_attr()
         * @uses selected()
         * @since 0.1
         * @return string HTML option element for a select field
        */
        function get_color_scheme_select_options( $selected=false ) {
            global $it_boom_bar;
            $options = array(
                'lime'     => __( 'Lime Green', 'it-l10n-boombar' ),
                'silver'   => __( 'Silver', 'it-l10n-boombar' ),
                'light'    => __( 'Light', 'it-l10n-boombar' ),
                'inset'    => __( 'Dark Inset', 'it-l10n-boombar' ),
                'flatdark' => __( 'Dark Flat', 'it-l10n-boombar' ),
				'custom'   => __( 'Custom', 'it-l10n-boombar' ),
				'seasonal' => __( 'Seasonal', 'it-l10n-boombar' ),
            );
            $options = apply_filters( 'it_boom_bar_color_schemes', $options );
            $html = '';
            foreach( $options as $value => $label ) {
                $html .= '<option value="' . esc_attr( $value ) . '"' .  selected( $selected, $value, false) . '>' . esc_attr( $label ) . '</option>';
            }
            return $html;
        }

		/**
		 * Returns the option elements for the cookie exp select
		 *
		 * If $selected param matches one of $values in the foreach, it will be marked as selected
		 *
		 * @param string $selected Selected value of field
		 * @uses esc_attr()
		 * @uses selected()
		 * @since 0.2
		 * @return string HTML option element for a select field
		*/
		function get_cookie_exp_select_options( $selected='1*24*60*60*1000' ) {
			$options = array(
				'-1'           => __( 'until next page load', 'it-l10n-boombar' ),
				'10000'        => __( 'for 10 Seconds [for testing]', 'it-l10n-boombar' ),
				'3600000'      => __( 'for 1 Hour', 'it-l10n-boombar' ),
				'86400000'     => __( 'for 1 Day', 'it-l10n-boombar' ),
				'604800000'    => __( 'for 1 Week', 'it-l10n-boombar' ),
				'630720000000' => __( 'forever', 'it-l10n-boombar' ),
			);
			$html = '';
			foreach( $options as $value => $label ) {
				$html .= '<option value="' . esc_attr( $value ) . '"' .  selected( $selected, $value, false) . '>' . esc_attr( $label ) . '</option>';
			}
			return $html;
		}
		/**
		 * Returns the option elements for a yes / no select
		 *
		 * If $selected param matches one of $values in the foreach, it will be marked as selected
		 *
		 * @param string $selected Selected value of field
		 * @uses esc_attr()
		 * @uses selected()
		 * @since 0.1
		 * @return string HTML option element for a select field
		*/
		function get_yesno_select_options( $selected='yes' ) {
			$selected = ( $selected === 'yes' ) ? $selected : 'no';
			$options = array(
				'yes' => __( 'Yes', 'it-l10n-boombar' ),
				'no'  => __( 'No', 'it-l10n-boombar' ),
			);
			$html = '';
			foreach( $options as $value => $label ) {
				$html .= '<option value="' . esc_attr( $value ) . '"' .  selected( $selected, $value, false) . '>' . esc_attr( $label ) . '</option>';
			}
			return $html;
		}

		/**
		 * Returns the option elements for the default bar setting on add/edit bars page.
		 *
		 * If $selected param matches one of $values in the foreach, it will be marked as selected
		 *
		 * @param string $selected Selected value of field
		 * @uses esc_attr()
		 * @uses selected()
		 * @since 0.2
		 * @return string HTML option element for a select field
		*/
		function get_default_bar_select_options( $selected='no' ) {
			global $post;

			$options = array(
				'no'      => __( 'No', 'it-l10n-boombar' ),
				$post->ID => __( 'Yes', 'it-l10n-boombar' ),
			);
			$html = '';
			foreach( $options as $value => $label ) {
				$html .= '<option value="' . esc_attr( $value ) . '"' .  selected( $selected, $value, false) . '>' . esc_attr( $label ) . '</option>';
			}
			return $html;
		}

		/**
		 * Returns options for Google Fonts for specific bar
		 *
		 * If $selected param matches one of $values in the foreach, it will be marked as selected
		 *
		 * @param string $selected Selected value of field
		 * @uses esc_attr()
		 * @uses selected()
		 * @since 0.2
		 * @return string HTML option element for a select field
		*/
		function get_font_for_bar( $selected ) {
			global $it_boom_bar;
			$html = '<option value="0" ' . selected( $selected, 0, false ) . '>' . __( 'Use Theme Styles', 'it-l10n-boombar' ) . '</option>';

			foreach ( (array) $it_boom_bar->get_google_fonts() as $font => $args ) {
				$html .= '<option value="' . esc_attr( $font ) . '" ' . selected( $selected, $font, false ) . '>' . esc_attr( $font ) . '</option>';
			}
			return $html;
		}

		/**
		 * Update general settings
		 *
		 * This originally updated multiple settings on a general settings admin page
		 * It now updates default bar when chosen from add/edit bar admin page
		 *
		 * @uses IT_Boom_Bar::update_plugin_options()
		 * @since 0.1
		 * @return null
		*/
		function update_general_settings() {
			global $it_boom_bar;
			$options['default_bar'] = empty( $_POST['default_bar'] ) ? false : (int) $_POST['default_bar'];
			$it_boom_bar->update_plugin_options( $options );
		}

		/**
		 * Saves bar data on Publish / Update
		 *
		 * If everything validates, this function creates / updates a WP post type and then sets post_meta
		 * It also syncs the WP post_order with the priority field and possibly updates the plugin's default_bar option.
		 *
		 * @param int $id ID of Bar post type
		 * @param object $obj Post type Object
		 * @uses check_admin_referer()
		 * @uses IT_Boom_Bar_Admin::validate_bar_data()
		 * @uses update_post_meta()
		 * @uses remove_action()
		 * @uses add_action()
		 * @uses wp_update_post()
		 * @uses IT_Boom_Bar::update_plugin_options()
		 * @since 0.1
		 * @return null
		*/
		function save_bar_data( $id, $obj ) {
			global $it_boom_bar;

			// Exit if not dealing with our post type or if we're in a recurssive loop
			if ( empty( $obj->post_type ) || 'it_boom_bar' != $obj->post_type )
				return;

			// Bail if we're doing autosave
			if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
				return;
			}

			// Bail if we're doing any AJAX at all
			if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
				return;
			}

			// Make sure we have a POST Request
			if ( empty( $_POST ) )
				return;

			// Check nonce
			check_admin_referer('it-boom_bar','it_boom_bar-new_edit_bar' );

			// Set Postmeta data
			$pm_params = array(
				'bar_type', 'bar_color_scheme', 'bar_bg_color', 'bar_text_color', 'bar_link_color', 'bar_border_color', 'bar_closable', 'bar_cookie_exp', 'bar_location', 'bar_position', 'bar_links_target', 'bar_font', 'bar_priority', 'bar_conditions', 'bar_custom_css', 'bar_text', 'bar_link_text', 'bar_link_url', 'bar_twitter_un', 'bar_global',
			);

			// Validate data
			$this->validate_bar_data();

			// Save validated data
			foreach( $pm_params as $param ) {
				if ( isset( $_POST[$param] ) )
					$new_values[$param] = $_POST[$param];
			}

			update_post_meta( $obj->ID, '_it_boom_bar', $new_values );

			// Sync priority with menu_order
			$obj->menu_order = $_POST['bar_priority'];

			// Break tweet cache if needed
			if ( 'tweet' == $new_values['bar_type'] && ! empty( $new_values['bar_twitter_un'] ) )
				delete_transient( 'it_boom_bar-latest_tweet-' . esc_attr( $new_values['bar_twitter_un'] ) );

			// Set back to draft if errors are present
			$obj->post_status = empty( $this->errors ) ? $obj->post_status : 'draft';

			remove_action( 'save_post', array( $this, 'save_bar_data' ) );
			wp_update_post( $obj );
			add_action( 'save_post', array( $this, 'save_bar_data' ) );

			// Process default bar option
			$default_bar = absint( $_POST['default_bar'] );
			if ( ! empty( $default_bar ) || ( empty( $default_bar ) && $_POST['post_ID'] == $it_boom_bar->options['default_bar'] ) )
				$it_boom_bar->update_plugin_options( array( 'default_bar' => $default_bar ) );
		}

		/**
		 * Validates data on add / edit bar save
		 *
		 * If data doesn't pass validation, add to errors property so that we can abandon update and add to postmeta to alert
		 * user on page reload.
		 *
		 * @todo Refactor error messages. This feels hacky.
		 * @uses delete_post_meta()
		 * @uses add_post_meta()
		 * @since 0.2
		 * @return null
		 */
		function validate_bar_data() {
			global $post;
			$errors = array();

			// Delete existing errors
			delete_post_meta( $post->ID, '_it_boom_bar_errors' );

			// Validate all required fields are present
			if ( empty( $_POST['post_title'] ) )
				$errors[] = __( 'The bar title cannot be left empty', 'it-l10n-boombar' );
			if ( 'tweet' == $_POST['bar_type'] && empty( $_POST['bar_twitter_un'] ) )
				$errors[] = __( 'Please provide a username for the Twitter account', 'it-l10n-boombar' );
			if ( 'text' == $_POST['bar_type'] && empty( $_POST['bar_text'] ) )
				$errors[] = __( 'The bar text cannot be left blank.', 'it-l10n-boombar' );
			if ( ! empty( $_POST['bar_link_text'] ) && empty( $_POST['bar_link_url'] ) )
				$errors[] = __( 'The bar link cannot have a blank URL.', 'it-l10n-boombar' );
			if ( empty( $_POST['bar_priority'] ) || $_POST['bar_priority'] < 1 )
				$errors[] = __( "The bar's priority must be a number greater than 0", 'it-l10n-boombar' );

			if ( ! empty( $errors ) ) {
				foreach ( $errors as $error ) {
					add_post_meta( $post->ID, '_it_boom_bar_errors', $error );
				}
				$this->errors = $errors;
			}
		}

		/**
		 * Registers my meta boxes for all public post types
		 *
		 * @uses get_post_types()
		 * @uses add_meta_box()
		 * @since 0.2
		 * @return null
		*/
		function register_override_meta_boxes() {
			if ( $visible_post_types = get_post_types( array( 'public' => true, 'show_in_nav_menus' => true ) ) ) {
				foreach( $visible_post_types as $type ) {
					add_meta_box( 'it_boom_bar_override', __( 'Boom Bar', 'it-l10n-boombar' ), array( $this, 'post_type_override_mbs' ), $type, 'side', 'default' );
				}
			}
		}

		/**
		 * This method prints the content of the Boom Bar default override on public post type's Add / Edit views in admin
		 *
		 * @param object $post WordPress Post Object
		 * @uses get_post_meta()
		 * @uses get_posts()
		 * @uses get_post()
		 * @uses selected()
		 * @uses esc_attr_e()
		 * @since 0.2
		 * @return null
		*/
		function post_type_override_mbs( $post ) {
			global $it_boom_bar;

			// Grab this post's Boom Bar. If none is specified, use the default.
			$default_bar = $it_boom_bar->options['default_bar'];
			if ( ! $current_bar = get_post_meta( $post->ID, '_it_boom_bar_pt_override', true ) )
				$current_bar = 0;

			$params = array(
				'post_type'      => 'it_boom_bar',
				'post_status'    => 'publish',
				'orderby'        => 'menu_order',
				'order'          => 'ASC',
				'posts_per_page' => -1,
			);

			$valid_bars = get_posts( $params );

			if ( $default_bar && $default_bar_obj = get_post( $default_bar ) )
				$default_bar_title = $default_bar_obj->post_title;
			else
				$default_bar_title = __( 'No Default Bar', 'it-l10n-boombar' );

			?>
			<p class="description"><?php _e( 'Use the below select box to override the default Boom Bar for this page.', 'it-l10n-boombar' ); ?></p>

			<select name="it_boom_bar_pt_override">
				<option value="-1" <?php selected( -1, $current_bar ); ?>><?php _e( 'No Boom Bar', 'it-l10n-boombar' ); ?></option>
				<option value="0" <?php selected( 0, $current_bar ); ?>><?php _e( 'Default Boom Bar', 'it-l10n-boombar' ); ?></option>
				<?php foreach( $valid_bars as $bar ) { ?>
				<option value="<?php esc_attr_e( $bar->ID ); ?>" <?php selected( $bar->ID, $current_bar ); ?>><?php esc_attr_e( $bar->post_title ); ?></option>
				<?php } ?>
			</select>
			<input type="hidden" name="it_boom_bar_update_pt_override" value="1" />
			<?php
		}

		/**
		 * Filters the messages displayed when editing / updating a bar
		 *
		 * If update was successful, replace 'post' verbage with 'Boom Bar'.
		 * If there was an error, prevent the message from displaying
		 *
		 * @param array $messages Existin messages
		 * @uses get_home_url()
		 * @uses esc_url()
		 * @uses date_i18n()
		 * @since 0.2
		 * @return array An Array of messages
		*/
		function post_updated_messages( $messages ) {
			global $post;
			$boom_bar_messages = array(
				0 => '', // Unused. Messages start at index 1.
				1 => sprintf( __('Bar updated. <a href="%s">View bar</a>', 'it-l10n-boombar' ), esc_url( get_home_url() ) ),
				2 => __( 'Custom field updated.' ),
				3 => __( 'Custom field deleted.' ),
				4 => __( 'Bar updated.', 'it-l10n-boombar' ),
				5 => isset($_GET['revision']) ? sprintf( __( 'Bar restored to revision from %s', 'it-l10n-boombar' ), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false,
				6 => sprintf( __( 'Bar published. <a href="%s">View bar</a>', 'it-l10n-boombar' ), esc_url( get_home_url() ) ),
				7 => __( 'Bar saved.', 'it-l10n-boombar' ),
				8 => __( 'Bar submitted.', 'it-l10n-boombar' ),
				9 => sprintf( __('Bar scheduled for: <strong>%1$s</strong>.', 'it-l10n-boombar' ), date_i18n( __( 'M j, Y @ G:i' ), strtotime( $post->post_date ) ) ),
				10 => __('Bar draft updated.', 'it-l10n-boombar' ),
			);

			// Clear success messages if we find an error
			if ( get_post_meta( $post->ID, '_it_boom_bar_errors', true ) ) {
				foreach ( $boom_bar_messages as $key => $value ) {
					$boom_bar_messages[$key] = '';
				}
			}

			$messages['it_boom_bar'] = $boom_bar_messages;
			return $messages;
		}

		/**
		 * Add it_boom_bar post type to Builder blacklist for Custom Layouts meta box
		 *
		 * @param array $post_types An array of post types that will not include the builder custom layout
		 * @since 0.2
		 * @return array Array of post types
		*/
		function remove_builder_custom_layout_box( $post_types ) {
			$post_types[] = 'it_boom_bar';
			return $post_types;
		}

		/**
		 * Restrict 3rd party meta boxes
		 *
		 * @since 0.2
		 * @return null
		*/
		function restrict_third_party_meta_boxes() {
			global $wp_meta_boxes;

			$it_boom_bar_mbs = array(
				'it_boom_bar_type',
				'it_boom_bar_settings',
				'it_boom_bar_settings_display',
				'it_boom_bar_css',
				'submitdiv',
			);

			$wp_meta_boxes['it_boom_bar']['side']['core'] = array();
			$wp_meta_boxes['it_boom_bar']['side']['high'] = array();
			$wp_meta_boxes['it_boom_bar']['side']['default'] = array();
			$wp_meta_boxes['it_boom_bar']['side']['low'] = array();

			if ( ! empty ( $wp_meta_boxes['it_boom_bar']['normal'] ) ) {
				// Whitelist Normal / High
				if ( ! empty ( $wp_meta_boxes['it_boom_bar']['normal']['high'] ) ) {
					foreach( (array) $wp_meta_boxes['it_boom_bar']['normal']['high'] as $metabox => $values ) {
						if ( ! in_array( $metabox, $it_boom_bar_mbs ) )
								unset( $wp_meta_boxes['it_boom_bar']['normal']['high'][$metabox] );
					}
				}

				// Whitelist Normal / Default
				if ( ! empty ( $wp_meta_boxes['it_boom_bar']['normal']['default'] ) ) {
					foreach( (array) $wp_meta_boxes['it_boom_bar']['normal']['default'] as $metabox => $values ) {
						if ( ! in_array( $metabox, $it_boom_bar_mbs ) )
							unset( $wp_meta_boxes['it_boom_bar']['normal']['default'][$metabox] );
					}
				}
			}
		}

		/**
		 * Adds content to the example privacy statment for the site
		 *
		 * @since 1.2.13
		 * @return null
		 */
		function add_privacy_policy_content() {
			if ( ! function_exists( 'wp_add_privacy_policy_content' ) ) {
				return;
			}

			$heading = '<div class="wp-suggested-text"><h2>' . __( 'What personal data we collect and why we collect it', 'it-l10n-boombar' ) . '</h2>';
			$content = sprintf( __( "%sCookies%s %s%sSuggested text:%s The first time you visit this site, a browser cookie is generated to identify you as a new visitor. If you were refered to this site by another site, the referal URL is also tracked in a cookie. This allows us to create customized messaging and notices for each of our site visitors. This data is retained in your web browser until cleared by you or your browser.%s%s", 'it-l10n-boombar' ), '<h3>', '</h3>', '<p>', '<strong class="privacy-policy-tutorial">', '</strong>', '</p>', '</div>' );

			wp_add_privacy_policy_content( 'iThemes BoomBar', wp_kses_post( wpautop( $heading . $content, false ) ) );
		}


		/**
		 * Shows the GDPR Warning to admin if all conditions are met
		 *
		 * Conditions:
		 * - is_admin returns true
		 * - current user can activate_plugins
		 * - message hasn't been dismissed previously
		 *
		 * @since 1.2.13
		 * @return null
		 */
		function maybe_do_gdpr_nag() {
			if ( ! is_admin() || ! current_user_can( 'activate_plugins' ) ) {
				return;
			}

			if ( true != get_option( 'it_boombar_gdpr_warning_dismissed' ) ) {
				add_action( 'admin_notices', array( $this, 'display_gdpr_nag' ) );
				return;
			}
		}

		/**
		 * Displays the dismissable GDPR nag
		 *
		 * @since 1.2.13
		 *
		 * @return void
		 */
		function display_gdpr_nag() {
			$priv_url     = '<a href="' . admin_url( 'tools.php?wp-privacy-policy-guide#wp-privacy-policy-guide-ithemes-boombar' ) . '">';
			$privacy_link = function_exists( 'wp_add_privacy_policy_content' ) ? __( sprintf( 'BoomBar has given you example text for your privacy statement %shere%s. ', $priv_url, '</a>' ) ) : '';
			$gdpr_url     = '<a href="https://gdpr-info.eu/" target="_blank">';
			$wpplugin_url = '<a href="' . admin_url( 'plugin-install.php?s=cookie+notice&tab=search&type=term' ) . '">';
			?>
			<div class="notice notice-warning is-dismissible it-boombar-gdpr-notice">
				<p><?php _e( sprintf( '%sGDPR Notice%sYou have activated the iThemes BoomBar plugin. This plugin uses cookies to track your site visitors. In order to be %sGDRP%s compliant, this site will need to opt users in to accepting cookies as well as notifiy them about this behavior in your Privacy Statement. %sThere are multiple plugins on the WordPress %splugin repository%s  to help you opt-in users to your cookie policy.', '<h2>', '</h2>', $gdpr_url, '</a>', $privacy_link, $wpplugin_url, '</a>' ), 'it-l10n-boombar' ); ?></p>
			</div>
			<?php
			wp_enqueue_script( 'it-boombar-dismiss-gdpr-nag', plugins_url( 'boombar/js/gdpr-notice.js' ), array( 'jquery' ), false, true );
		}


		/**
		 * Admin Ajax Hook to dismiss gdpr nag
		 *
		 * @since 1.2.13
		 *
		 * @return void
		 */
		function dismiss_gdpr_nag() {
			update_option( 'it_boombar_gdpr_warning_dismissed', true );
		}
	}
}
