<?php
/*
 *	Admin Settings Page
 *
 * 	@author		Kerry Kline
 * 	@copyright	Copyright (c) 2017, Kerry Kline
 * 	@link		http://www.bnecreative.com
 *	@package	BNE Testimonials Pro
 *	@package	CMB2
 *	@since 		v2.5
 *
*/

// Exit if accessed directly
if ( !defined('ABSPATH')) exit;



class BNE_Testimonials_Settings {
	

	/*
 	 * 	Option key used in the Database
 	 *	 
 	 * 	@var string
 	*/
	private $options_key = 'bne_testimonials_settings';
	
	
	/*
 	 * 	Metabox ID assigned using CMB2
 	 *	 
 	 * 	@var string
 	*/
	private $metabox_id = 'bne_testimonials_settings_metabox';


	/*
	 * 	Constructor
 	 *
	 * 	@since 	v2.5
	 *
	*/
	function __construct() {
		
		// Plugin action Links
		add_filter( 'plugin_action_links_' . BNE_TESTIMONIALS_BASENAME, array( $this, 'add_plugin_action_links' ) );
		
		// Register the Metabox
		add_action( 'cmb2_admin_init', array( $this, 'register_settings' ) );
		
		// Hook before Metabox
		add_action( "cmb2_before_options-page_form_{$this->metabox_id}", array( $this, 'before_metabox' ), 10, 2 );
		
		// Hook After Metabox
		add_action( "cmb2_after_options-page_form_{$this->metabox_id}", array( $this, 'after_metabox' ), 10, 2 );
				
	}


	/*
	 * 	Register our admin plugin action links
 	 *
	 * 	@since 	v2.5
	 *
	*/
	function add_plugin_action_links( $links ) {
		$links[] = '<a href="'.esc_url( get_admin_url( null, 'edit.php?post_type=bne_testimonials&page='.$this->options_key) ).'">'.esc_html__( 'Settings', 'bne-testimonials' ).'</a>';
		$links[] = '<a href="'.esc_url( get_admin_url( null, 'edit.php?post_type=bne_testimonials&page=bne-testimonial-help') ).'">'.esc_html__( 'Help', 'bne-testimonials' ).'</a>';
		return $links;
	}
	

	/*
	 * 	Register our admin page links
 	 *
	 * 	@since 	v2.5
	 *
	*/
	function wrapper() {
		?>
		<div class="wrap cmb2-options-page option-<?php echo $this->options_key; ?>">
			
			<!-- Clear Cache -->
			<form method="post" class="clear-cache-form">
				<input type="hidden" name="action" value="bne_testimonials_api_delete_transients" />
				<input type="hidden" name="transient" value="all" />
				<?php wp_nonce_field( 'bne_testimonials_api_delete_transients' ); ?>
				<input type="submit" class="button button-primary" value="<?php _e( 'Clear API Cache', 'bne-testimonials' ); ?>" />
			</form>

			<h2><?php echo esc_html__( 'BNE Testimonials Pro Settings', 'bne-testimonials' ); ?></h2>

			<!-- Main Settings -->
			<?php cmb2_metabox_form( $this->metabox_id, $this->options_key, array( 'save_button' => esc_html__( 'Save Changes', 'bne-testimonials' ) ) ); ?>

		</div>
		<?php

	}



	/*
	 * 	Register the Settings Metabox and Settings Page
	 *	Hooks into CMB2 to automatically register the page and options metabox
	 *
	 * 	@since 	v2.5
	 *
	*/
	function register_settings() {
	
		add_action( "cmb2_save_options-page_fields_{$this->metabox_id}", array( $this, 'settings_notices' ), 10, 2 );
		
		/**
		 * Registers options page menu item and form.
		 */
		$cmb_options = new_cmb2_box( array(
			'id'			=> 	$this->metabox_id,
			'title'			=> 	esc_html__( 'BNE Testimonials Pro Settings', 'bne-testimonials' ),
			'object_types'	=> 	array( 'options-page' ),
			'option_key'	=> 	$this->options_key,
			'menu_title'	=> 	esc_html__( 'Settings', 'bne-testimonials' ),
			'parent_slug'	=> 	'edit.php?post_type=bne_testimonials',
			'capability'	=> 	'manage_options',
			'save_button'	=> 	esc_html__( 'Save Changes', 'bne-testimonials' ),
			'display_cb'	=>	array( $this, 'wrapper' )
			//'classes'		=>	'bne-cmb-wrapper',
		) );
		
		
		// == SECTION: General Settings/Styles == /
		$cmb_options->add_field( array(
			'name' 		=> 	esc_html__( 'General Settings', 'bne-testimonials' ),
			'desc' 		=> 	esc_html__( 'Define the general options for BNE Testimonials.', 'bne-testimonials' ),
			'id'   		=> 	'section_general',
			'type' 		=> 	'title',
		) );


		// Enable API
		$cmb_options->add_field( array(
			'name' 		=> 	esc_html__( 'Gravatar Emails', 'bne-testimonials' ),
			'desc' 		=> 	esc_html__( 'Enable / Disable', 'bne-testimonials' ),
			'id'   		=> 	'enable_gravatars',
			'type' 		=> 	'checkbox',
			'after_field'	=>	'<p class="cmb2-metabox-description">'.esc_html__( 'When a testimonial thumbnail is not used, but an email address is entered, try to use their Gravatar image.', 'bne-testimonials' ).'</p>',
		) );



		// == SECTION: Form == /
		$cmb_options->add_field( array(
			'name' 		=> 	esc_html__( 'Submit Form Notification', 'bne-testimonials' ),
			'desc' 		=> 	esc_html__( "Form labels and functionality are set within the shortcode directly. The below options are for the form notification email. If you find that receiving the notification is going into spam or not being delivered, make sure the from email address reflects this website domain and that this server is allowed to send emails using the from email address according to your domain's DNS records.", 'bne-testimonials' ),
			'id'   		=> 	'section_form',
			'type' 		=> 	'title',
		) );


		// From Email
		$cmb_options->add_field( array(
			'name' 		=> 	esc_html__( 'From Email', 'bne-testimonials' ),
			'desc' 		=> 	esc_html__( 'Set the from email address. Defaults to the site admin.', 'bne-testimonials' ),
			'id'   		=> 	'form_email',
			'type' 		=> 	'text_email',
			'default'	=>	get_bloginfo( 'admin_email' )
		) );


		// Subject
		$cmb_options->add_field( array(
			'name' 		=> 	esc_html__( 'Subject', 'bne-testimonials' ),
			'desc' 		=> 	esc_html__( 'The subject line for the email notification.', 'bne-testimonials' ),
			'id'   		=> 	'form_subject',
			'type' 		=> 	'text',
			'default'	=>	'You have a new Testimonial'
		) );


		// Message
		$merge_tags = array('{field_list}','{name}','{email}','{tagline}','{website}','{rating}','{message}','{terms}');
		$cmb_options->add_field( array(
			'name' 		=> 	esc_html__( 'Message', 'bne-testimonials' ),
			'desc' 		=> 	esc_html__( 'Customize the text that is displayed within the email notification. The below merge tags can be used which will dynamically insert the matching field data entered from the form.', 'bne-testimonials' ).'<br><br>Available Merge Tags:<br>'.implode(", ",$merge_tags),
			'id'   		=> 	'form_message',
			'type' 		=> 	'wysiwyg',
			'options' 	=> array(
				'media_buttons' => 	false,
				'textarea_rows' =>	8,
				'teeny' 		=>	true,
			),
			'default_cb'	=>	array( $this, 'form_message_default_cb' )
		) );



		// == SECTION: API == /
		$cmb_options->add_field( array(
			'name' 		=> 	esc_html__( 'API Settings', 'bne-testimonials' ),
			'desc' 		=> 	esc_html__( 'Configure API keys and settings. Yelp API Key is already included within the plugin.', 'bne-testimonials' ),
			'id'   		=> 	'section_api',
			'type' 		=> 	'title',
		) );



		// Enable WP API
		$cmb_options->add_field( array(
			'name' 		=> 	esc_html__( 'BNE Testimonials Source API', 'bne-testimonials' ),
			'desc' 		=> 	esc_html__( 'Enable / Disable', 'bne-testimonials' ),
			'id'   		=> 	'enable_api',
			'type' 		=> 	'checkbox',
			'after_field'	=>	'<p class="cmb2-metabox-description">'.esc_html__( 'When enabled, public testimonials will be added to the WordPress REST API for this website. This allows other WordPress websites to display testimonials from this website by using the [bne_testimonials_api] shortcode. Only the source website needs to enable the Source API. API platforms below do not require this to be enabled and are not included.', 'bne-testimonials' ).'</p>',
		) );

	
		// Google Key
		$cmb_options->add_field( array(
			'name' 			=> 	esc_html__( 'Google Places API', 'bne-testimonials' ),
			'desc' 			=> 	sprintf( 
									esc_html__( 'An API key is required to interact with Google Places and can be managed through the %sGoogle API Console%3$s. To create a key, %sfollow these steps%3$s.', 'bne-testimonials'),
									'<a href="https://developers.google.com/places/web-service/" rel="noopener" target="_blank">',
									'<a href="http://docs.bnecreative.com/articles/plugins/google-testimonials-api/" rel="noopener" target="_blank">',
									'</a>'
								),
			'id'   			=> 	'google_places_api_key',
			'type' 			=> 	'text',
		) );


		// Facebook Key
		$cmb_options->add_field( array(
			'name' 			=> 	esc_html__( 'Facebook Graph API', 'bne-testimonials' ),
			'desc' 			=> 	sprintf( 
									esc_html__( 'An Access Token key is required to interact with the %sFacebook Graph API%3$s. You must be signed into Facebook when requesting a token and you must be an administrator of the page you want pull reviews from. To create a token, %sfollow these steps%3$s.', 'bne-testimonials'),
									'<a href="https://developers.facebook.com/tools-and-support/" rel="noopener" target="_blank">',
									'<a href="http://docs.bnecreative.com/articles/plugins/facebook-testimonials-api/" rel="noopener" target="_blank">',
									'</a>'
								),
			'id'   			=> 	'facebook_api_key',
			'type' 			=> 	'text',
		) );


		// Yelp Key
/*
		$cmb_options->add_field( array(
			'name' 			=> 	esc_html__( 'Yelp API', 'bne-testimonials' ),
			'desc' 			=> 	sprintf( 
									esc_html__( 'An API key is required to interact with the Yelp API and can be created at the %sYelp Developers site%3$s. To create a key, %sfollow these steps%3$s.', 'bne-testimonials' ),
									'<a href="https://www.yelp.com/developers/v3/manage_app" rel="noopener" target="_blank">',
									'<a href="http://docs.bnecreative.com/articles/plugins/yelp-testimonials-api/" rel="noopener" target="_blank">',
									'</a>'
								),
			'id'   			=> 	'yelp_api_key',
			'type' 			=> 	'text',
		) );
*/


		// Yellow Pages Key
		$cmb_options->add_field( array(
			'name' 			=> 	esc_html__( 'Yellow Pages (YP) API', 'bne-testimonials' ),
			'desc' 			=> 	sprintf( 
									esc_html__( 'An API key is required to interact with the Yellow Pages API and can be created at the %sYP Publisher website%3$s. To create a key, %sfollow these steps%3$s.', 'bne-testimonials' ),
									'<a href="https://publisher.yp.com" rel="noopener" target="_blank">',
									'<a href="http://docs.bnecreative.com/articles/plugins/yellow-pages-testimonials-api/" rel="noopener" target="_blank">',
									'</a>'
								),
			'id'   			=> 	'yp_api_key',
			'type' 			=> 	'text',
		) );


	} // END FIELDS



	/*
	 * 	Form Message Default callback
	 *
	 * 	@since 	v2.5
	 *
	*/
	function form_message_default_cb() {
		echo '<p>You have received a new testimonial at <a href="'.esc_url( get_bloginfo( 'url' ) ).'">'.get_bloginfo( 'name' ).'</a> from the below person. This testimonial is not published yet and needs to be approved.</p>';
		echo '{field_list}';
	}



	/*
	 * 	Register settings notices for display
	 *
	 * 	$object_id 	int 	Option key
	 * 	$updated 	array 	Array of updated fields
	 * 	@return 	void
	 *
	 * 	@since 	v2.5
	 *
	*/
	function settings_notices( $object_id, $updated ) {
		if ( $object_id !== $this->options_key || empty( $updated ) ) {
			return;
		}
		add_settings_error( $this->options_key . '-notices', '', esc_html__( 'Settings updated.', 'bne-testimonials' ), 'updated' );
		settings_errors( $this->options_key . '-notices' );
	}



	/*
	 * 	Before Metabox Hook
	 *	Adds content before the CMB2 metabox
	 *
	 * 	@since 	v2.5
	 *
	*/
	function before_metabox( $cmb_id, $object_id ) {
		settings_errors( 'bne_testimonials_clear_cache_notice' );
		?>
		<style type="text/css">
			.clear-cache-form { float: right; margin-bottom: 20px; }
			.bne-cmb-options-footer {
				margin-top: 10px;
				text-align: right;
			}
			.bne-cmb-options-wrapper .button { background: #fff; }
			.bne-cmb-options-wrapper .cmb-th label { padding: 0; }
			.bne-cmb-options-wrapper .cmb2-metabox-description { color: #888; max-width: 90%; }
			.bne-cmb-options-wrapper label .cmb2-metabox-description { font-style: normal; }
			.bne-cmb-options-wrapper .cmb-type-checkbox label .cmb2-metabox-description { color: #444; }
			.bne-cmb-options-wrapper input[type="text"],
			.bne-cmb-options-wrapper input[type="email"] { width: 35em; }
			
			@media (max-width: 480px ) {
				.bne-cmb-options-wrapper .cmb-type-group .cmb-repeat-group-field {
					padding-top: 15px;
				}
				.bne-cmb-options-wrapper .cmb-type-group .cmb-th, 
				.bne-cmb-options-wrapper .cmb-type-group .cmb-th + .cmb-td {
					float: none;
					width: 100%;
				}
			}
		</style>
		<div class="bne-cmb-options-wrapper">
		<?php
	}
	


	/*
	 * 	After Metabox Hook
	 *	Adds content after the CMB2 metabox
	 *
	 * 	@since 	v2.5
	 *
	*/
	function after_metabox( $cmb_id, $object_id ) {
		?>
			<div class="bne-cmb-options-footer">v<?php echo BNE_TESTIMONIALS_VERSION; ?> (<a href="http://updates.bnecreative.com/changelogs/?product=testimonials-pro" target="_blank" rel="noopener">changelog</a>) Created by <a href="https://www.bnecreative.com" target="_blank" rel="noopener">BNE Creative</a>.</div>
		</div><!-- .bne-cmb-options-wrapper (end) -->
		<?php
	}

} // End Class
$BNE_Testimonials_Settings = new BNE_Testimonials_Settings();