<?php
/*
 *	Register Shortcode Generator Custom Post Type
 *
 * 	@author		Kerry Kline
 * 	@link		http://www.bnecreative.com
 *	@package	BNE Testimonials Pro
 *
*/

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit; 


class BNE_Testimonials_Pro_SG_CPT {


	/*
	 *	Post type name
	 *
	 *	@var string $post_type_name Holds the name of the post type.
	 *
	*/
	public $post_type_name = 'bne_testimonials_sg';
	

    /*
     * 	Constructor
     *
     *	@since 		v2.0
     *	@updated 	v2.8.2
     *
    */
	function __construct() {

		
	    /*
	     * 	Hook into BNE_CPT and register a new custom post type.
	     *	Translation is taken care of in the BNE_CPT class.
	     *
	    */
		$cpt = new BNE_CPT( $this->post_type_name, array(
			'singular'		=> 	esc_html__( 'Testimonial Shortcode', 'bne-testimonials' ),
			'plural'		=> 	esc_html__( 'Testimonial Shortcodes', 'bne-testimonials' ),
			'slug' 			=> 	'testimonial_generator',
			'args'			=>	array(
				'labels'		=>	array(
					'menu_name' 	=> esc_html__( 'Shortcode Generator', 'bne-testimonials' ),
					'all_items'		=> esc_html__( 'Shortcode Generator', 'bne-testimonials' ),
				),
				'supports'				=>	array( 'title' ),
				'menu_icon'				=>	'dashicons-id-alt',
			    'public'             	=> 	false,
			    'publicly_queryable' 	=> 	false,
			    'show_ui'            	=> 	true,
			    'show_in_menu'  		=>	'edit.php?post_type=bne_testimonials',
			    'show_in_nav_menus'  	=> 	false,
			    'show_in_admin_bar'  	=> 	false,
			    'hierarchical'       	=> 	false,
			    'has_archive'        	=> 	false,
			),
		));



	    // Post List Columns
		add_filter( "manage_edit-{$this->post_type_name}_columns", array( $this, 'columns' ), 10, 1 );
		add_action( "manage_{$this->post_type_name}_posts_custom_column",  array( $this, 'column_content' ), 10, 2 );

		
		// Post Admin
		add_action( 'edit_form_after_title', array( $this, 'form_after_title' ) );

				
		// CPT CMB2 Fields
		add_action( 'cmb2_admin_init', array( $this, 'cmb2_metabox' ) );
		add_action( 'cmb2_before_post_form_type_options', array( $this, 'cmb2_metabox_scripts' ), 10, 2 );
		
	}




	/*
	 *	Add/Remove Columns on Post List Admin Screen
	 *
     *	@since 		v2.0
     *	@updated 	v2.5
	 *
	*/
	function columns( $columns ) {
		
		// Remove Columns
		unset( $columns['date'] );
	
		// Add Columns
	    $columns['shortcode'] =	esc_html__( 'Reusable Shortcode', 'bne-testimonials' );
	    $columns['alt_shortcode'] =	esc_html__( 'Designed Shortcode', 'bne-testimonials' );
	
	    return $columns;
	
	}




	/*
	 *	Add Content to Columns on Post List Admin Screen
	 *
     *	@since 		v2.0
     *	@updated 	v2.8
	 *
	*/
	function column_content( $column, $post_id ) {
		
		// Generated Shortcode
		if( $column === 'shortcode' ) {
			echo '<textarea readonly rows="2" style="width: 100%; font-size: 12px;">[bne_testimonials custom="'.$post_id.'"]</textarea>';
		}
	
		// Generated Shortcode
		if( $column === 'alt_shortcode' ) {
			echo '<textarea readonly rows="2" style="width: 100%; font-size: 12px;">'.get_post_meta( $post_id, '_bne_testimonials_sg_shortcode', true ).'</textarea>';
		}

	}




	/*
	 *	Admin after_title for Shortcode Generator
	 *
     *	@since 		v2.0
     *	@updated 	v2.8
	 *
	*/
	function form_after_title() {
	    $screen = get_current_screen();
	
	    if( $this->post_type_name == $screen->post_type ) {
	    	echo '<p class="description">';
	    		echo esc_html__( 'The Shortcode Generator is optional, but allows you to save pre-defined configurations to be re-used throughout your website including with the v2 Testimonial Widget.', 'bne-testimonials' );
	    	echo '</p>';
	    }
	}
	


	
	/*
	 *	Meta Box
	 *
     *	@since 		v2.0
     *	@updated 	v2.8.2
	 *
	*/
	function cmb2_metabox() {
	
		// Prefix the field ID's
		$prefix = '_bne_testimonials_sg_';
	
		// Testimonial List (available posts)
		$testimonials_list = array();
		$testimonials_list_obj = get_posts( 'post_type=bne_testimonials&posts_per_page=-1&sort_column=post_parent,name' );
		$testimonials_list[''] = esc_html__( 'Disabled', 'bne-testimonials' );
		foreach( $testimonials_list_obj as $testimonial ) {
			$testimonials_list[$testimonial->ID] = $testimonial->post_title. ' (ID: '.$testimonial->ID.')';
		}
	
		// Testimonial Category List
		$testimonial_terms = array();
		$testimonial_terms_obj = get_terms( array( 'taxonomy' => 'bne-testimonials-taxonomy', 'hide_empty' => false ) );
		//$testimonial_terms['Disabled'] = esc_html__( 'Disabled', 'bne-testimonials' );
		foreach( $testimonial_terms_obj as $testimonial_term ) {
			$testimonial_terms[$testimonial_term->slug] = $testimonial_term->name;
		}

		// Testimonial Category List (FORM ID)
		$testimonial_form_terms = array();
		$testimonial_form_terms_obj = get_terms( array( 'taxonomy' => 'bne-testimonials-taxonomy', 'hide_empty' => false ) );
		foreach( $testimonial_form_terms_obj as $testimonial_form_term ) {
			$testimonial_form_terms[$testimonial_form_term->term_id] = $testimonial_form_term->name;
		}


		/* ============================================ */
		/* == Type Meta Box == */
		/* ============================================ */
		$cmb_type = new_cmb2_box( array(
			'id'            =>	'type_options',
			'title'         =>  esc_html__( 'Shortcode Type', 'bne-testimonials' ),
			'object_types'  =>  array( 'bne_testimonials_sg' ),
			'classes'		=>	'bne-cmb-wrapper',
		) );
		
		// Type
		$cmb_type->add_field( array(
			'name'    		=>	esc_html__( 'Choose the shortcode to build', 'bne-testimonials' ),
			'id'      		=>	$prefix.'type',
			'type'    		=>	'select',
			'options'		=>	array(
				''				=>	esc_html__( 'Select a shortcode', 'bne-testimonials' ),
				'local'			=>	esc_html__( 'Local', 'bne-testimonials' ),
				'api'			=>	esc_html__( 'Review API', 'bne-testimonials' ),
				'badge'			=>	esc_html__( 'Badge', 'bne-testimonials' ),
				'badge_api'		=>	esc_html__( 'Badge Review API', 'bne-testimonials' ),
				'stats'			=>	esc_html__( 'Stats', 'bne-testimonials' ),
				'form'			=>	esc_html__( 'Submission Form', 'bne-testimonials' ),
			),
		) );





		/* ============================================ */
		/* == Query Meta Box == */
		/* ============================================ */
		$cmb_query = new_cmb2_box( array(
			'id'            =>	'query_options',
			'title'         =>  esc_html__( 'Query Options', 'bne-testimonials' ),
			'object_types'  =>  array( 'bne_testimonials_sg' ),
			'classes'		=>	'bne-cmb-wrapper',
		) );



		/* == API ONLY OPTIONS == */
		// source (API)
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'API Source', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The API sources to display (yelp, facebook, google, yp). If more than one, separate each with a comma.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'api_source',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'style'						=>	'width: 100%;',
				'required'					=>	'required',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'api', 'badge_api' ) ),
			)
		) );

		// id (API)
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'API ID', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The ID of the review profile to display based on the source above. If more than one, separate each with a comma. To find your ID, review the help page or support docs for the appropriate source.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'api_id',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'style'						=>	'width: 100%;',
				'required'					=>	'required',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'api', 'badge_api' ) ),
			)
		) );

		// message_required
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'Require Review Message', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Only include testimonials with a message. (Ex: Facebook does not require users to enter a message with their rating)', 'bne-testimonials' ),
			'id'      		=>	$prefix.'message_required',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'false'			=>	esc_html__( 'No (default)', 'bne-testimonials' ),
				'true'			=>	esc_html__( 'Yes', 'bne-testimonials' ),
			),
			'default'		=>	'false',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'api' ) ),
			)
		) );



		/* == Local/General OPTIONS == */
		// Limit
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'Number of Testimonials?', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Determines how many testimonials to include in this display. API Platforms may limit what is returned.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'post',
			'type'    		=>	'text',
			'default'		=>	'20',
			'attributes'	=>	array(
				'required'		=>	'required',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// rating_min
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'Minimum Rating', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Only include testimonials with this rating or higher.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'rating_min',
			'type'    		=>	'select',
			'options'		=>	array(
				''				=>	esc_html__( 'All ratings', 'bne-testimonials' ),
				'2'				=>	esc_html__( '2 or higher', 'bne-testimonials' ),
				'3'				=>	esc_html__( '3 or higher', 'bne-testimonials' ),
				'4'				=>	esc_html__( '4 or higher', 'bne-testimonials' ),
				'5'				=>	esc_html__( '5 or higher', 'bne-testimonials' ),
			),
			'default'		=>	'',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> $prefix.'type',
				'data-conditional-value' 	=> wp_json_encode( array( 'local', 'api' ) ),
			)
		) );

		// id (local)
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'Single Testimonial', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'If you only want to display only one and a specific testimonial.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'id',
			'type'    		=>	'select',
			'options'		=>	$testimonials_list,
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local' ) ),
			)
		) );
	
		// category (local)
		if( $testimonial_terms ) {
			$cmb_query->add_field( array(
				'name'    		=>	esc_html__( 'Category', 'bne-testimonials' ),
				'desc'    		=>	esc_html__( 'Show testimonials from the above checked categories only. Leave unchecked to show all. Not available with API Platforms.', 'bne-testimonials' ),
				'id'      		=>	$prefix.'category',
				'type'    		=>	'multicheck',
				'select_all_button' => false,
				'options'		=>	$testimonial_terms,
				'attributes'	=>	array(
					// Conditional Logic
					'data-conditional-id'    	=> 	$prefix.'type',
					'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api', 'badge', 'stats' ) ),
				)
			) );
		}
	
		// order
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'Order', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Displays the testimonials based on "order by" option below in either an ascending or descending direction.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'order',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'desc'			=>	esc_html__( 'Descending (default)', 'bne-testimonials' ),
				'asc'			=>	esc_html__( 'Ascending', 'bne-testimonials' ),
			),
			'default'		=>	'desc',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'stats' ) ),
			)
		) );
	
		// orderby
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'Order By', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Arranges the testimonials this pattern based on their initial order. API Platforms can only use date or random. Mixed APIs will be ordered based on their source position or can use random.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'orderby',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'date'			=>	esc_html__( 'Publish Date (default)', 'bne-testimonials' ),
				'rand'			=>	esc_html__( 'Random Order', 'bne-testimonials' ),
				'title'			=>	esc_html__( 'Name', 'bne-testimonials' ),
			),
			'default_cb'	=>	array( $this, 'return_default_date' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// pagination
		$cmb_query->add_field( array(
			'name'    		=>	esc_html__( 'Pagination', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'If using the List or Masonry Displays, you can add pagination to the bottom of the testimonial display. Requires "Number of Testimonials" to not be "-1". ', 'bne-testimonials' ),
			'id'      		=>	$prefix.'pagination',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide (default)', 'bne-testimonials' ),
			),
			'default'		=>	'false',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local' ) ),
			)
		) );


	
		/* == FORM OPTIONS == */

		// category_id
		if( $testimonial_form_terms ) {
			$cmb_query->add_field( array(
				'name'    		=>	esc_html__( 'Categories', 'bne-testimonials' ),
				'desc'    		=>	esc_html__( '(optional) Does not require category="true"; however, if true, the category dropdown field will list only these categories. If using more than one category, you should set category="true" to allow the user to choose the category.', 'bne-testimonials' ),
				'id'      		=>	$prefix.'form_category_id',
				'type'    		=>	'multicheck',
				'select_all_button' => false,
				'options'		=>	$testimonial_form_terms,
	
				'attributes'	=>	array(
					// Conditional Logic
					'data-conditional-id'    	=> 	$prefix.'type',
					'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
				)
			) );
		}



	
		/* ========================================== == */
		/* == Display Meta Box == */
		/* ========================================== == */
		$cmb_display = new_cmb2_box( array(
			'id'            =>	'display_options',
			'title'         =>  esc_html__( 'Display Options', 'bne-testimonials' ),
			'object_types'  =>  array( 'bne_testimonials_sg' ),
			'classes'		=>	'bne-cmb-wrapper',
		) );


	
		/* == Local/General OPTIONS == */
		// layout
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Layout', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Choose the display layout to use.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'layout',
			'type'    		=>	'select',
			'options'		=>	array(
				''				=>	esc_html__( 'Select Layout', 'bne-testimonials' ),
				'list'			=>	esc_html__( 'List', 'bne-testimonials' ),
				'masonry'		=>	esc_html__( 'Masonry Grid', 'bne-testimonials' ),
				'slider'		=>	esc_html__( 'Slider', 'bne-testimonials' ),
				'thumbs'		=>	esc_html__( 'Thumbnail Slider', 'bne-testimonials' ),
			),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// theme
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Theme', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Choose a theme to use with the layout.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'theme',
			'type'    		=>	'select',
			'options'		=>	array(
				'default'		=>	esc_html__( 'Default Theme (default)', 'bne-testimonials' ),
				'bubble'		=>	esc_html__( 'Bubble Theme', 'bne-testimonials' ),
				'cards-h'		=>	esc_html__( 'Cards "Horizontal" Theme', 'bne-testimonials' ),
				'cards-v'		=>	esc_html__( 'Cards "Vertical" Theme', 'bne-testimonials' ),
				'simple'		=>	esc_html__( 'Simple Theme (minimal styling)', 'bne-testimonials' ),
			),
			'default'		=>	'default',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=>	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// columns
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Columns', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Determines the max number of columns to use for the Masonry, List, and Slider layouts.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'columns',
			'type'    		=>	'select',
			'options'		=>	array(
				''			=>	esc_html__( 'Default', 'bne-testimonials' ),
				'1'			=>	esc_html__( '1 Column', 'bne-testimonials' ),
				'2'			=>	esc_html__( '2 Columns', 'bne-testimonials' ),
				'3'			=>	esc_html__( '3 Columns', 'bne-testimonials' ),
				'4'			=>	esc_html__( '4 Columns', 'bne-testimonials' ),
			),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );

		// arrangement
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Arrangement', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Choose the arrangement of the testimonial content. With the Thumbnail Slider this will also include the placement of the thumbnail navigation.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'arrangement',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'1'				=>	esc_html__( 'Author then Message (default)', 'bne-testimonials' ),
				'2'				=>	esc_html__( 'Message then Author', 'bne-testimonials' ),
			),
			'default'		=>	'1',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// alignment
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Alignment', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Choose the alignment of the testimonial content. Note: Testimonial message will only align left or center.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'alignment',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'left'			=>	esc_html__( 'Left Aligned (default)', 'bne-testimonials' ),
				'center'		=>	esc_html__( 'Center Aligned', 'bne-testimonials' ),
				'right'			=>	esc_html__( 'Right Aligned', 'bne-testimonials' ),
			),
			'default'		=>	'left',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> $prefix.'type',
				'data-conditional-value' 	=> wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// image_style
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Image Style', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Choose the image style of the testimonial profile image.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'image_style',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'square'		=>	esc_html__( 'Square (default)', 'bne-testimonials' ),
				'circle'		=>	esc_html__( 'Circle', 'bne-testimonials' ),
				'flat-square'	=>	esc_html__( 'Flat Square', 'bne-testimonials' ),
				'flat-circle'	=>	esc_html__( 'Flat Circle', 'bne-testimonials' ),
			),
			'default'		=>	'square',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
		
		// name
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Name', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the testimonial name.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'name',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
		
		// image
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Image', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the testimonial image.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'image',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// tagline
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Tagline', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the testimonial tagline.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'tagline',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// website
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Website', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the testimonial website.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'website',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// rating
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Rating', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the testimonial rating (stars).', 'bne-testimonials' ),
			'id'      		=>	$prefix.'rating',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// readmore
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Read More', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Truncate (shorten) the testimonial message with a expandable link. Note: Yelp API only provides a fixed truncated message.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'readmore',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'false'			=>	esc_html__( 'False (default)', 'bne-testimonials' ),
				'true'			=>	esc_html__( 'True ', 'bne-testimonials' ),
			),
			'default'		=>	'false',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );

		// readmore_limit
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Read More Limit', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The number of characters to truncate (shorten) the testimonial message followed by the read more link.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'readmore_limit',
			'type'    		=>	'text',
			'default'		=>	'250',
			'attributes'	=>	array(
				'type'						=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );

		// readmore_text
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Read More Text', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The text used for the read more link to show the remainder testimonial message.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'readmore_text',
			'type'    		=>	'text',
			'default'		=>	'Read More',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );

		// readmore_less
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Read Less Text', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The text used for the read less link to hide the remainder testimonial message.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'readless_text',
			'type'    		=>	'text',
			'default'		=>	'Read Less',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );



		/* == BADGE Options == */
		// width
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Badge Width', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The max width of the badge.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_width',
			'type'    		=>	'text',
			'default'		=>	'400px',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// branding
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Branding', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The branding styles the badge to use the review provider\'s logo, color scheme and styled frame.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_branding',
			'type'    		=>	'select',
			'options'		=>	array(
				'local'			=>	esc_html__( 'Local (default non-styled)', 'bne-testimonials' ),
				'facebook'		=>	esc_html__( 'Facebook', 'bne-testimonials' ),
				'google'		=>	esc_html__( 'Google', 'bne-testimonials' ),
				'yp'			=>	esc_html__( 'Yellow Pages', 'bne-testimonials' ),
				'yelp'			=>	esc_html__( 'Yelp', 'bne-testimonials' ),
			),
			'default'		=>	'local',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge' ) ),
			)
		) );

		// image
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Image', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Include an image such as a logo or product image. Not available with single Badge API or selected branding.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_image',
			'type'    		=>	'file',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// image_size
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Image Size', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The width of the image using pixels. Not available with single Badge API or selected branding.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_image_size',
			'type'    		=>	'text',
			'default'		=>	'60px',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// image_align
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Image Align', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Align the image around the review content.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_image_align',
			'type'    		=>	'select',
			'options'		=>	array(
				'left'			=>	esc_html__( 'Left Align', 'bne-testimonials' ),
				'center'		=>	esc_html__( 'Center Align', 'bne-testimonials' ),
				'right'			=>	esc_html__( 'Right Align', 'bne-testimonials' ),
			),
			'default'		=>	'left',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// rating
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Rating', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'When not defined, this will be the calculated average of all local testimonial ratings. Only testimonials that have a rating will be included.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_rating',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'type'			=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge' ) ),
			)
		) );

		// total
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Total', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'When not defined, this will be the total of all local testimonials that have a rating. Only testimonials that have a rating will be included. Google API Note: Their API does not provide a total value; therefore, you can set it here.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_total',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'type'			=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// heading
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Heading', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Text displayed as a heading.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_heading',
			'type'    		=>	'text',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// rating_name
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Rating Name Label', 'bne-testimonials' ),
			'desc'    		=>	__( 'The label used to describe the rating type. Example: 4.5 <code>Stars</code> - Based on 305 User Reviews.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_rating_name',
			'type'    		=>	'text',
			'default'		=>	'Stars',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// before_total
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Before Total Label', 'bne-testimonials' ),
			'desc'    		=>	__( 'Text displayed between the rating and total numbers. Example: 4.5 Stars <code>- Based on</code> 305 User Reviews', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_before_total',
			'type'    		=>	'text',
			'default'		=>	'- Based on',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// after_total
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'After Total Label', 'bne-testimonials' ),
			'desc'    		=>	__( 'Text displayed after the total number. Example: 4.5 Stars - Based on 305 <code>User Reviews</code>', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_after_total',
			'type'    		=>	'text',
			'default'		=>	'User Reviews',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// link
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Review Link', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The URL where the review total is linked to - typically a review page of all published reviews on your site.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_link',
			'type'    		=>	'text',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );

		// link_title
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Link Title', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The alt label used for the above link.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_link_title',
			'type'    		=>	'text',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );



		/* == Stats OPTIONS == */
		// layout
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Layout', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The display used to present the statistics.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_layout',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'bar'			=>	esc_html__( 'Bar', 'bne-testimonials' ),
				'face'			=>	esc_html__( 'Face', 'bne-testimonials' ),
			),
			'default'		=>	'bar',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'stats' ) ),
			)
		) );

		// label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Label Position', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( '(Bar layout only) The position of the labels within the bar graph.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_label',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'above'			=>	esc_html__( 'Above', 'bne-testimonials' ),
				'side'			=>	esc_html__( 'Side', 'bne-testimonials' ),
			),
			'default'		=>	'above',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'stats' ) ),
			)
		) );

		// label_text
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Statistic Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The text used for the statistic label. Ex: 5 <code>Stars</code> or 234 <code>Reviews</code>.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_label_text',
			'type'    		=>	'text',
			'default'		=>	'Star',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'stats' ) ),
			)
		) );

		// total_1
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( '1 Rating Total', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( '(optional) Allows manually setting the total count of 1 star ratings instead of using the calculated total.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_total_1',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'type'						=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'stats' ) ),
			)
		) );

		// total_2
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( '2 Rating Total', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( '(optional) Allows manually setting the total count of 2 star ratings instead of using the calculated total.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_total_2',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'type'						=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'stats' ) ),
			)
		) );

		// total_3
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( '3 Rating Total', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( '(optional) Allows manually setting the total count of 3 star ratings instead of using the calculated total.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_total_3',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'type'						=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'stats' ) ),
			)
		) );

		// total_4
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( '4 Rating Total', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( '(optional) Allows manually setting the total count of 4 star ratings instead of using the calculated total.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_total_4',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'type'						=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'stats' ) ),
			)
		) );

		// total_5
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( '5 Rating Total', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( '(optional) Allows manually setting the total count of 5 star ratings instead of using the calculated total.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_total_5',
			'type'    		=>	'text',
			'attributes'	=>	array(
				'type'						=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'stats' ) ),
			)
		) );




		/* == Slider Options == */
		// nav
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Nav', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the slider pagination navigation.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'nav',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'slider' ) ),
			)
		) );
	
		// arrows
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Arrows', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the slider next/previous arrows.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'arrows',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'slider' ) ),
			)
		) );
	
		// pause
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Pause', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'When hovered with a mouse, should the slider pause or continue on?', 'bne-testimonials' ),
			'id'      		=>	$prefix.'pause',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Yes (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'No', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'slider', 'thumbs' ) ),
			)
		) );
		
		// smooth
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Smooth Height', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Should the height of the slider adjust with the height of each testimonial or should the height of the slider be based on the tallest testimonial?', 'bne-testimonials' ),
			'id'      		=>	$prefix.'smooth',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Yes (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'No', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'slider', 'thumbs' ) ),
			)
		) );
	
		// animation
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Animation', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Use a fade or slide animation between each testimonial.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'animation',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'slide'			=>	esc_html__( 'Slide (default)', 'bne-testimonials' ),
				'fade'			=>	esc_html__( 'Fade', 'bne-testimonials' ),
			),
			'default'		=>	'slide',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'slider', 'thumbs' ) ),
			)
		) );
	
		// animation_speed
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Animation Speed', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The time in milliseconds the slide animation occurs.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'animation_speed',
			'type'    		=>	'text',
			'default'		=>	'700',
			'before_field'	=>	'<div class="bne-input-append">'.esc_html__( 'milliseconds', 'bne-testimonials' ).'</div>',
			'classes'		=>	'has-input-wrap',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'slider', 'thumbs' ) ),
			)
		) );
	
		// speed
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Speed', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The time in milliseconds that each slide is displayed.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'speed',
			'type'    		=>	'text',
			'default'		=>	'7000',
			'before_field'	=>	'<div class="bne-input-append">'.esc_html__( 'milliseconds', 'bne-testimonials' ).'</div>',
			'classes'		=>	'has-input-wrap',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'slider', 'thumbs' ) ),
			)
		) );
	
	

		/* == FORM OPTIONS == */
		// category
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Category Field', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'If true, a dropdown field listing testimonial categories will show.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_category',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide (default)', 'bne-testimonials' ),
			),
			'default'		=>	'false',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// image
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Image Field', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the image upload field.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_image',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// tagline
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Tagline Field', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the tagline field.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_tagline',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// website
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Website Field', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the website field.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_website',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// rating
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Rating Field', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the rating field. Select "Stars" to use a visualized star rating field rather than a select field.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_rating',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show as a select field (default)', 'bne-testimonials' ),
				'stars'			=>	esc_html__( 'Show as a stars field', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// terms
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Terms Field', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Show or hide the terms/consent field. Set this to true for GDPR compliance.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_terms',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'Show', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'Hide (default)', 'bne-testimonials' ),
			),
			'default'		=>	'false',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// name_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Name Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the name field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_name_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_name' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// email_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Email Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the email field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_email_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_email' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// message_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Message Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the message field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_message_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_testimonial' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// image_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Image Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the photo field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_image_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_photo' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );
	
		// category_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Category Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the category field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_category_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_category' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// tagline_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Tagline Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the tagline field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_tagline_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_tagline' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// website_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Website Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the website field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_website_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_website' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// rating_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Rating Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the star rating field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_rating_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_rating' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// terms_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Terms Field Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the terms/consent field label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_terms_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_terms' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// submit_label
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Submit Button Label', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the submit button label.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_submit_label',
			'type'    		=>	'text',
			'default'		=>	bne_testimonials_get_local( 'form_submit' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// success_message
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Success Message', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the success message after the form is submitted.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_success_message',
			'type'    		=>	'textarea',
			'default'		=>	bne_testimonials_get_local( 'form_success_message' ),
			'attributes'	=>	array(
				'rows'						=> '3',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// error_message
		$cmb_display->add_field( array(
			'name'    		=>	esc_html__( 'Failed Message', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows changing the error message after the form is submitted.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_error_message',
			'type'    		=>	'textarea',
			'default'		=>	bne_testimonials_get_local( 'form_error_message' ),
			'attributes'	=>	array(
				'rows'						=> '3',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );







		/* ============================================ */
		/* == Style Meta Box == */
		/* ============================================ */
		$cmb_styles = new_cmb2_box( array(
			'id'            =>	'style_options',
			'title'         =>  esc_html__( 'Style Options', 'bne-testimonials' ),
			'object_types'  =>  array( 'bne_testimonials_sg' ),
			'classes'		=>	'bne-cmb-wrapper',
		) );
	
	
		/* == Local/General Options == */
		// author_bg_color
		$cmb_styles->add_field( array(
			'name'    		=>	esc_html__( 'Author Background Color', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Style the author area background color of the testimonial.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'author_bg_color',
			'type'    		=>	'colorpicker',	
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// author_color
		$cmb_styles->add_field( array(
			'name'    		=>	esc_html__( 'Author Font Color', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Style the author area font color of the testimonial.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'author_color',
			'type'    		=>	'colorpicker',	
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// message_bg_color
		$cmb_styles->add_field( array(
			'name'    		=>	esc_html__( 'Message Background Color', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Style the message area background color of the testimonial.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'message_bg_color',
			'type'    		=>	'colorpicker',	
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// message_color
		$cmb_styles->add_field( array(
			'name'    		=>	esc_html__( 'Message Font Color', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Style the message area font color of the testimonial.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'message_color',
			'type'    		=>	'colorpicker',	
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );
	
		// rating_color
		$cmb_styles->add_field( array(
			'name'    		=>	esc_html__( 'Rating / Star Color', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Style the rating (stars) color of the testimonial display. Not available with branded APIs.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'rating_color',
			'type'    		=>	'colorpicker',	
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'local', 'api', 'badge', 'badge_api' ) ),
			)
		) );
	

		/* == BADGE Options == */
		// accent
		$cmb_styles->add_field( array(
			'name'    		=>	esc_html__( 'Accent Color', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The color used on the top border.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'badge_accent',
			'type'    		=>	'colorpicker',	
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'badge', 'badge_api' ) ),
			)
		) );
		
	
		/* == Stats OPTIONS == */
		// color
		$cmb_styles->add_field( array(
			'name'    		=>	esc_html__( 'Bar color', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( '(Bar layout only) The color of the bar graphs.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'stats_color',
			'type'    		=>	'colorpicker',
			'default'		=>	'#337ab7',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=>	wp_json_encode( array( 'stats' ) ),
			)
		) );




		/* ============================================ */
		/* == Other Meta Box == */
		/* ============================================ */

		$cmb_other_options = new_cmb2_box( array(
			'id'            =>	'other_options',
			'title'         =>  esc_html__( 'Other Options', 'bne-testimonials' ),
			'object_types'  =>  array( 'bne_testimonials_sg' ),
			//'priority'     	=> 	'low',
			'classes'		=>	'bne-cmb-wrapper',
		) );
	


		/* == FORM OPTIONS == */
		// notify
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Email Notification', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'If true, an email notification will be sent to the email address below.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_notify',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'True (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'False', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// email
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Email To Address', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Requires notify option to be true. Defaults to the site admin email address. Allows changing the email address of where notifications are sent after a successful form submission.' ),
			'id'      		=>	$prefix.'form_email',
			'type'    		=>	'text',
			'default'		=>	get_bloginfo( 'admin_email' ),
			'classes'		=>	'has-input-wrap',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// redirect_url
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Redirect URL', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'If successful, you can have the user redirect to a custom thank you page. Leave blank to reload the current page with a success message.' ),
			'id'      		=>	$prefix.'form_redirect_url',
			'type'    		=>	'text',
			'before_field'	=>	'<div class="bne-input-prepend">'.esc_html__( 'URL', 'bne-testimonials' ).'</div>',
			'classes'		=>	'has-input-wrap',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// auto_hide
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Auto Hide', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'If true, the form will be hidden after a successful submission.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_auto_hide',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'True', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'False (default)', 'bne-testimonials' ),
			),
			'default'		=>	'false',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'type',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'form' ) ),
			)
		) );

		// auto_fill
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Auto Fill', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'If true, the form will auto populate the current logged in user\'s display name and email address.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'form_auto_fill',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'True', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'False (default)', 'bne-testimonials' ),
			),
			'default'		=>	'false',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=>  $prefix.'type',
				'data-conditional-value' 	=>  wp_json_encode( array( 'form' ) ),
			)
		) );

		// btn_class
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Submit Button Class', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Adds specific classes to the submit button for CSS customizations or theme styles.' ),
			'id'      		=>	$prefix.'form_btn_class',
			'type'    		=>	'text',
			'classes'		=>	'has-input-wrap',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> $prefix.'type',
				'data-conditional-value' 	=> wp_json_encode( array( 'form' ) ),
			)
		) );



		/* == LIST / MASONRY OPTIONS == */
		// expand
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Expand Hidden Testimonials', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'List and Masonry layouts only. Allows grouping extra testimonials in a hidden container that will expand when clicked. This number will set how many testimonials are shown before the hidden group. If using columns, this should match the number of columns used.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'expand',
			'type'    		=>	'text',
			'default'		=>	'',
			'attributes'	=>	array(
				'type'			=>	'number',
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'list', 'masonry' ) ),
			)
		) );

		// expand_more
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Expand More Text', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The text used for the expand more link to show the hidden testimonials.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'expand_more',
			'type'    		=>	'text',
			'default'		=>	'Show More',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'list', 'masonry' ) ),
			)
		) );

		// expand_less
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Expand Less Text', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The text used for the expand less link to hide the hidden testimonials.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'expand_less',
			'type'    		=>	'text',
			'default'		=>	'Show Less',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=> 	$prefix.'layout',
				'data-conditional-value' 	=> 	wp_json_encode( array( 'list', 'masonry' ) ),
			)
		) );


		/* == API OPTIONS == */
		// cache
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Cache', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'The initial response will be cached for 1 week (7 days) and future calls will use the cache stored locally until reset after expiration. API providers may limit the number of request and for a faster website and API response, make sure to set this to true after you are finished testing.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'cache',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'True (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'False', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=>	$prefix.'type',
				'data-conditional-value' 	=>	wp_json_encode( array( 'api', 'badge_api' ) ),
			)
		) );


		/* == GLOBAL OPTIONS == */

		// schema
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Schema', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Allows completely turning off the schema markup if not needed or is displayed elsewhere on the page.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'schema',
			'type'    		=>	'radio_inline',
			'options'		=>	array(
				'true'			=>	esc_html__( 'True (default)', 'bne-testimonials' ),
				'false'			=>	esc_html__( 'False', 'bne-testimonials' ),
			),
			'default'		=>	'true',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=>	$prefix.'type',
				'data-conditional-value' 	=>	wp_json_encode( array( 'local', 'api', 'badge', 'badge_api' ) ),
			)
		) );

		// schema_type
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Schema Type', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'This is the "itemReviewed" property in schema. If the reviews are about a business, this should be set to "LocalBusiness" or a more closely related term such as "Restaurant", "Book", "Product", etc.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'schema_type',
			'type'    		=>	'text',
			'default'		=>	'Product',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=>	$prefix.'type',
				'data-conditional-value' 	=>	wp_json_encode( array( 'local', 'api', 'badge', 'badge_api' ) ),
			)
		) );

		// schema_name
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Schema Name', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'This should be the name of the item being reviewed such as the business or product name. Defaults to the Site name.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'schema_name',
			'type'    		=>	'text',
			'default'		=>	get_bloginfo( 'name' ),
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=>	$prefix.'type',
				'data-conditional-value' 	=>	wp_json_encode( array( 'local', 'api', 'badge', 'badge_api' ) ),
			)
		) );

		// schema_image
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Schema Image', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Some Schema Types require an image. For example if using LocalBusiness as a type.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'schema_image',
			'type'    		=>	'file',
			'attributes'	=>	array(
				// Conditional Logic
				'data-conditional-id'    	=>	$prefix.'type',
				'data-conditional-value' 	=>	wp_json_encode( array( 'local', 'api' ) ),
			)
		) );


		// class
		$cmb_other_options->add_field( array(
			'name'    		=>	esc_html__( 'Classes', 'bne-testimonials' ),
			'desc'    		=>	esc_html__( 'Include an additional CSS classname for this testimonial display. This allows you to further customize this shortcode with custom CSS.', 'bne-testimonials' ),
			'id'      		=>	$prefix.'class',
			'type'    		=>	'text',
			'classes'		=>	'has-input-wrap',
		) );
	
	


		/* == ======================================== == */
		/* == Shortcode Display Meta Box == */
		/* == ======================================== == */
		
		$cmb_generated_shortcode = new_cmb2_box( array(
			'id'            =>	'shortcode_metabox',
			'title'         =>  esc_html__( 'Testimonial Shortcodes', 'bne-testimonials' ),
			'object_types'  =>  array( 'bne_testimonials_sg' ),
			'priority'     	=> 	'low',
			//'context'		=>	'side', // Using JS to move it to the side due to it resetting the query from do_shortcode
			'classes'		=>	'bne-cmb-wrapper',
		) );

		$cmb_generated_shortcode->add_field( array(
			'name'    		=>	'',
			'id'      		=>	'_bne_testimonials_sg_shortcode',
			'type'    		=>	'title',
			'before_field'	=>	'bne_testimonials_sg_create_shortcode',
		) );


	} // END CMB FIELDS
	


	
	/*
	 *	Default Value "date" Callback
	 *
	 *	The "default" param for cmb2 may call an actual
	 *	function, in this case, "date" is a php function,
	 *	so we need to make a unique callback for the return
	 *	value.
	 *
	 *	@since 	v2.0
	 *
	*/
	function return_default_date() {
	    return 'date';
	}


	/*
	 *	Meta Box CSS / JS
	 *	Injects needed scripts/styles for our fields
	 *
     *	@since 		v2.0
     *	@updated 	v2.8.2
	 *
	*/
	function cmb2_metabox_scripts( $post_id, $cmb ) {
		
		// Init Conditional Class
		$cmb2_conditionals = new CMB2_Conditionals();
		
		?>
		<!-- Admin Form CSS -->
		<style type="text/css">
			#minor-publishing { display: none; }
			.cmb2-wrap.bne-cmb-wrapper .cmb2-metabox .cmb-row > .cmb-td { min-height: 90px; }
			#shortcode_metabox .testimonial-image { width: 75px !important; height: 75px !important; }
			.testimonial-stats.layout-face .rating-summary-inner { min-width: 50px !important; }
			.form-star-rating label { display: inline-block !important; }
		</style>
		
		<!-- Admin Form JS -->
		<script type="text/javascript">
			jQuery(document).ready(function($) { 
				$('.bne-cmb-wrapper .has-input-wrap input').wrap('<div class="bne-input-wrap"></div>');
				$('#shortcode_metabox').insertAfter('#submitdiv');

				$( ".bne-testimonial-form-wrapper input, .bne-testimonial-form-wrapper select, .bne-testimonial-form-wrapper textarea" ).each( function() {
					$(this).attr("disabled", true);
				});
			});
		</script>	
		<?php
	}
	

} // END Class
$BNE_Testimonials_Pro_SG_CPT = new BNE_Testimonials_Pro_SG_CPT();



/*
 *	Testimonial Shortcode Metabox Content
 *
 *	Queries the current post and builds the shortcode
 *	for the user. Then once the post is saved, creates a
 *	meta_key to retrieve the shortcode for the Widget and
 *	other scenarios.
 *
 *	@since		v2.0
 *	@updated	v2.8.3
 *
*/
function bne_testimonials_sg_create_shortcode( $field_args, $field ) {
	
	if( isset( $_GET['post'] ) ) {
		$post_id = $_GET['post'];
	} else {
		echo '<p>'.esc_html__( 'Publish first to view the shortcode created.', 'bne-testimonials').'</p>';
		return;
	}

	// Type				
	$type = get_post_meta( $post_id, '_bne_testimonials_sg_type', true );

	// Query Args
	$limit = get_post_meta( $post_id, '_bne_testimonials_sg_post', true );
	$id = get_post_meta( $post_id, '_bne_testimonials_sg_id', true );
	$category = get_post_meta( $post_id, '_bne_testimonials_sg_category', true );
	$order = get_post_meta( $post_id, '_bne_testimonials_sg_order', true );
	$orderby = get_post_meta( $post_id, '_bne_testimonials_sg_orderby', true );
	$pagination = get_post_meta( $post_id, '_bne_testimonials_sg_pagination', true );
	$message_required = get_post_meta( $post_id, '_bne_testimonials_sg_message_required', true );
	$rating_min = get_post_meta( $post_id, '_bne_testimonials_sg_rating_min', true );
	
	// Display Args
	$layout = get_post_meta( $post_id, '_bne_testimonials_sg_layout', true );
	$theme = get_post_meta( $post_id, '_bne_testimonials_sg_theme', true );
	$columns = get_post_meta( $post_id, '_bne_testimonials_sg_columns', true );
	$arragnement = get_post_meta( $post_id, '_bne_testimonials_sg_arrangement', true );
	$alignment = get_post_meta( $post_id, '_bne_testimonials_sg_alignment', true );
	$image_style = get_post_meta( $post_id, '_bne_testimonials_sg_image_style', true );
	$name = get_post_meta( $post_id, '_bne_testimonials_sg_name', true );
	$image = get_post_meta( $post_id, '_bne_testimonials_sg_image', true );
	$tagline = get_post_meta( $post_id, '_bne_testimonials_sg_tagline', true );
	$website = get_post_meta( $post_id, '_bne_testimonials_sg_website', true );
	$rating = get_post_meta( $post_id, '_bne_testimonials_sg_rating', true );
	$readmore = get_post_meta( $post_id, '_bne_testimonials_sg_readmore', true );
	$readmore_limit = get_post_meta( $post_id, '_bne_testimonials_sg_readmore_limit', true );
	$readmore_text = get_post_meta( $post_id, '_bne_testimonials_sg_readmore_text', true );
	$readless_text = get_post_meta( $post_id, '_bne_testimonials_sg_readless_text', true );

	// Styles Args
	$author_bg_color = get_post_meta( $post_id, '_bne_testimonials_sg_author_bg_color', true );
	$author_color = get_post_meta( $post_id, '_bne_testimonials_sg_author_color', true );
	$message_bg_color = get_post_meta( $post_id, '_bne_testimonials_sg_message_bg_color', true );
	$message_color = get_post_meta( $post_id, '_bne_testimonials_sg_message_color', true );
	$rating_color = get_post_meta( $post_id, '_bne_testimonials_sg_rating_color', true );
	
	// Slider Args
	$nav = get_post_meta( $post_id, '_bne_testimonials_sg_nav', true );
	$arrows = get_post_meta( $post_id, '_bne_testimonials_sg_arrows', true );
	$pause = get_post_meta( $post_id, '_bne_testimonials_sg_pause', true );
	$smooth = get_post_meta( $post_id, '_bne_testimonials_sg_smooth', true );
	$animation = get_post_meta( $post_id, '_bne_testimonials_sg_animation', true );
	$animation_speed = get_post_meta( $post_id, '_bne_testimonials_sg_animation_speed', true );
	$speed = get_post_meta( $post_id, '_bne_testimonials_sg_speed', true );
		
	// Other Args
	$expand = get_post_meta( $post_id, '_bne_testimonials_sg_expand', true );
	$expand_more = get_post_meta( $post_id, '_bne_testimonials_sg_expand_more', true );
	$expand_less = get_post_meta( $post_id, '_bne_testimonials_sg_expand_less', true );
	$class = get_post_meta( $post_id, '_bne_testimonials_sg_class', true );
	
	// API
	$api_id = get_post_meta( $post_id, '_bne_testimonials_sg_api_id', true );
	$api_source = get_post_meta( $post_id, '_bne_testimonials_sg_api_source', true );
	$cache = get_post_meta( $post_id, '_bne_testimonials_sg_cache', true );
	
	// Badge
	$badge_width = get_post_meta( $post_id, '_bne_testimonials_sg_badge_width', true );
	$badge_branding = get_post_meta( $post_id, '_bne_testimonials_sg_badge_branding', true );
	$badge_image = get_post_meta( $post_id, '_bne_testimonials_sg_badge_image', true );
	$badge_image_size = get_post_meta( $post_id, '_bne_testimonials_sg_badge_image_size', true );
	$badge_image_align = get_post_meta( $post_id, '_bne_testimonials_sg_badge_image_align', true );
	$badge_rating = get_post_meta( $post_id, '_bne_testimonials_sg_badge_rating', true );
	$badge_total = get_post_meta( $post_id, '_bne_testimonials_sg_badge_total', true );
	$badge_accent = get_post_meta( $post_id, '_bne_testimonials_sg_badge_accent', true );
	$badge_heading = get_post_meta( $post_id, '_bne_testimonials_sg_badge_heading', true );
	$badge_rating_name = get_post_meta( $post_id, '_bne_testimonials_sg_badge_rating_name', true );
	$badge_before_total = get_post_meta( $post_id, '_bne_testimonials_sg_badge_before_total', true );
	$badge_after_total = get_post_meta( $post_id, '_bne_testimonials_sg_badge_after_total', true );
	$badge_link = get_post_meta( $post_id, '_bne_testimonials_sg_badge_link', true );
	$badge_link_title = get_post_meta( $post_id, '_bne_testimonials_sg_badge_link_title', true );
	
	// Stats
	$stats_layout = get_post_meta( $post_id, '_bne_testimonials_sg_stats_layout', true );
	$stats_color = get_post_meta( $post_id, '_bne_testimonials_sg_stats_color', true );
	$stats_label = get_post_meta( $post_id, '_bne_testimonials_sg_stats_label', true );
	$stats_label_text = get_post_meta( $post_id, '_bne_testimonials_sg_stats_label_text', true );
	$stats_total_1 = get_post_meta( $post_id, '_bne_testimonials_sg_stats_total_1', true );
	$stats_total_2 = get_post_meta( $post_id, '_bne_testimonials_sg_stats_total_2', true );
	$stats_total_3 = get_post_meta( $post_id, '_bne_testimonials_sg_stats_total_3', true );
	$stats_total_4 = get_post_meta( $post_id, '_bne_testimonials_sg_stats_total_4', true );
	$stats_total_5 = get_post_meta( $post_id, '_bne_testimonials_sg_stats_total_5', true );
	
	// Schema
	$schema = get_post_meta( $post_id, '_bne_testimonials_sg_schema', true );
	$schema_type = get_post_meta( $post_id, '_bne_testimonials_sg_schema_type', true );
	$schema_name = get_post_meta( $post_id, '_bne_testimonials_sg_schema_name', true );
	$schema_image = get_post_meta( $post_id, '_bne_testimonials_sg_schema_image', true );

	// Form
	$form_notify = get_post_meta( $post_id, '_bne_testimonials_sg_form_notify', true );
	$form_email = get_post_meta( $post_id, '_bne_testimonials_sg_form_email', true );
	$form_redirect_url = get_post_meta( $post_id, '_bne_testimonials_sg_form_redirect_url', true );
	$form_auto_hide = get_post_meta( $post_id, '_bne_testimonials_sg_form_auto_hide', true );
	$form_auto_fill = get_post_meta( $post_id, '_bne_testimonials_sg_form_auto_fill', true );
	$form_btn_class = get_post_meta( $post_id, '_bne_testimonials_sg_form_btn_class', true );
	$form_category = get_post_meta( $post_id, '_bne_testimonials_sg_form_category', true );
	$form_category_id = get_post_meta( $post_id, '_bne_testimonials_sg_form_category_id', true );
	$form_image = get_post_meta( $post_id, '_bne_testimonials_sg_form_image', true );
	$form_tagline = get_post_meta( $post_id, '_bne_testimonials_sg_form_tagline', true );
	$form_website = get_post_meta( $post_id, '_bne_testimonials_sg_form_website', true );
	$form_rating = get_post_meta( $post_id, '_bne_testimonials_sg_form_rating', true );
	$form_terms = get_post_meta( $post_id, '_bne_testimonials_sg_form_terms', true );
	$form_name_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_name_label', true );
	$form_email_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_email_label', true );
	$form_message_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_message_label', true );
	$form_image_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_image_label', true );
	$form_category_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_category_label', true );
	$form_tagline_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_tagline_label', true );
	$form_website_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_website_label', true );
	$form_rating_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_rating_label', true );
	$form_terms_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_terms_label', true );
	$form_submit_label = get_post_meta( $post_id, '_bne_testimonials_sg_form_submit_label', true );
	$form_success_message = get_post_meta( $post_id, '_bne_testimonials_sg_form_success_message', true );
	$form_error_message = get_post_meta( $post_id, '_bne_testimonials_sg_form_error_message', true );



	// Shortcode
	$shortcode = '[bne_testimonials';
	if( 'api' == $type ) {
		$shortcode .= '_api';
	} elseif( 'badge' == $type ) {
		$shortcode .= '_badge';
	} elseif( 'badge_api' == $type ) {
		$shortcode .= '_badge api="true"';
	} elseif( 'stats' == $type ) {
		$shortcode .= '_stats';
	} elseif( 'form' == $type ) {
		$shortcode .= '_form';
	}


	// Local or API
	if( !$type || $type == 'local' || $type == 'api' ) {
		$options = array(
			'source'			=>	( 'api' == $type ) ? $api_source : '',
			'id'				=>	( 'api' == $type && $api_id ) ? $api_id : $id,
			'limit'				=>	( 'api' == $type && '-1' == $limit ) ? '20' : $limit,
			'order'				=>	( 'local' == $type ) ? $order : '',
			'orderby'			=>	$orderby,
			'category'			=>	( is_array( $category ) ) ? implode( ',', $category ) : $category,
			'pagination'		=>	( 'local' == $type && ( 'list' == $layout || 'masonry' == $layout ) ) ? $pagination : '',
			'rating_min'		=>	$rating_min,
			'message_required'	=>	$message_required,
			'layout'			=>	$layout,
			'theme'				=>	$theme,
			'columns'			=>	( 'thumbs' != $layout ) ? $columns : '',
			'arrangement'		=>	$arragnement,
			'alignment'			=>	$alignment,
			'image_style'		=>	$image_style,
			'name'				=>	$name,
			'image'				=>	$image,
			'tagline'			=>	$tagline,
			'website'			=>	$website,
			'rating'			=>	$rating,
			'readmore'			=>	$readmore,
			'readmore_limit'	=>	( 'true' == $readmore ) ? $readmore_limit : '',
			'readmore_text'		=>	( 'true' == $readmore ) ? $readmore_text : '',
			'readless_text'		=>	( 'true' == $readmore ) ? $readless_text : '',
			'expand'			=>	( (int) $expand && ( 'list' == $layout || 'masonry' == $layout ) ) ? $expand : '',
			'expand_more'		=>	( (int) $expand && ( 'list' == $layout || 'masonry' == $layout ) ) ? $expand_more : '',
			'expand_less'		=>	( (int) $expand && ( 'list' == $layout || 'masonry' == $layout ) ) ? $expand_less : '',
			'author_bg_color'	=>	$author_bg_color,
			'author_color'		=>	$author_color,
			'message_bg_color'	=>	$message_bg_color,
			'message_color'		=>	$message_color,
			'rating_color'		=>	$rating_color,
			'nav'				=>	( $layout == 'slider' || $layout == 'thumbs' ) ? $nav : '',
			'arrows'			=>	( $layout == 'slider' || $layout == 'thumbs' ) ? $arrows : '',
			'pause'				=>	( $layout == 'slider' || $layout == 'thumbs' ) ? $pause : '',
			'smooth'			=>	( $layout == 'slider' || $layout == 'thumbs' ) ? $smooth : '',
			'animation'			=>	( $layout == 'slider' || $layout == 'thumbs' ) ? $animation : '',
			'animation_speed'	=>	( $layout == 'slider' || $layout == 'thumbs' ) ? $animation_speed : '',
			'speed'				=>	( $layout == 'slider' || $layout == 'thumbs' ) ? $speed : '',
		);
		
	} // END Local / API


	// Badge / API
	elseif( $type == 'badge' || $type == 'badge_api' ) {
		$options = array(
			'source'			=>	( 'badge_api' == $type ) ? $api_source : '',
			'id'				=>	( 'badge_api' == $type && $api_id ) ? $api_id : $id,
			'category'			=>	( 'badge' == $type && is_array( $category ) ) ? implode( ',', $category ) : '',
			'width'				=>	$badge_width,
			'branding'			=>	( 'badge' == $type ) ? $badge_branding : '',
			'image'				=>	$badge_image,
			'image_size'		=>	$badge_image_size,
			'image_align'		=>	$badge_image_align,
			'rating'			=>	( 'badge' == $type ) ? $badge_rating : '',
			'total'				=>	( 'badge' == $type ) ? $badge_total : '',
			'heading'			=>	$badge_heading,
			'rating_name'		=>	$badge_rating_name,
			'before_total'		=>	$badge_before_total,
			'after_total'		=>	$badge_after_total,
			'link'				=>	$badge_link,
			'link_title'		=>	$badge_link_title,
			'rating_color'		=>	$rating_color,
			'accent'			=>	$badge_accent,

		);
		
	} // END Badge

	// Stats
	elseif( $type == 'stats' ) {
		$options = array(
			'order'				=>	$order,
			'category'			=>	( is_array( $category ) ) ? implode( ',', $category ) : '',
			'layout'			=>	$stats_layout,
			'label'				=>	$stats_label,
			'label_text'		=>	$stats_label_text,
			'total_1'			=>	$stats_total_1,
			'total_2'			=>	$stats_total_2,
			'total_3'			=>	$stats_total_3,
			'total_4'			=>	$stats_total_4,
			'total_5'			=>	$stats_total_5,
			'color'				=>	$stats_color,
		);
		
	} // END Stats

	// Form
	elseif( $type == 'form' ) {
		$options = array(
			'notify'			=>	$form_notify,
			'email'				=>	$form_email,
			'redirect_url'		=>	$form_redirect_url,
			'auto_hide'			=>	$form_auto_hide,
			'auto_fill'			=>	$form_auto_fill,
			'btn_class'			=>	$form_btn_class,
			'category'			=>	$form_category,
			'category_id'		=>	(is_array( $form_category_id ) ) ? implode( ',',$form_category_id ) : '',
			'image'				=>	$form_image,
			'tagline'			=>	$form_tagline,
			'website'			=>	$form_website,
			'rating'			=>	$form_rating,
			'terms'				=>	$form_terms,
			'name_label'		=>	$form_name_label,
			'email_label'		=>	$form_email_label,
			'message_label'		=>	$form_message_label,
			'image_label'		=>	$form_image_label,
			'category_label'	=>	$form_category_label,
			'tagline_label'		=>	$form_tagline_label,
			'website_label'		=>	$form_website_label,
			'rating_label'		=>	$form_rating_label,
			'terms_label'		=>	$form_terms_label,
			'submit_label'		=>	$form_submit_label,
			'success_message'	=>	$form_success_message,
			'error_message'		=>	$form_error_message,
		);
		
	} // END Stats


	// Other
	$options['cache'] = ( 'api' == $type || 'badge_api' == $type ) ? $cache : '';
	$options['class'] = $class;
	
	// Schema
	if( 'stats' != $type && 'form' != $type ) {
		$options['schema'] = $schema;
		if( 'true' == $schema ) {
			$options['schema_type'] = $schema_type;
			$options['schema_name'] = $schema_name;
			if( $type == 'local' || $type == 'api' ) {
				$options['schema_image'] = $schema_image;
			}
		}
	}



	// Remove empty options
	$options = array_filter($options);

	// Output Shortcode
	$output = $shortcode;
	foreach( $options as $index => $option ) {
		$output .= ' '.$index.'="'.$option.'"';
	}
	$output .= ']';
		
			
	// Example
	wp_enqueue_style( 'bne-testimonials-css', BNE_TESTIMONIALS_URI . '/assets/css/bne-testimonials.min.css', '', BNE_TESTIMONIALS_VERSION, 'all' );
	wp_enqueue_script( 'flexslider', BNE_TESTIMONIALS_URI . '/assets/js/flexslider.min.js', array('jquery'), '2.2.2', true );
	$example_display = str_replace( ']', ' columns="1" limit="1"]', $output );
	if( $layout == 'thumbs' || $layout == 'slider' ) {
		$example_display = str_replace( 'limit="1"', ' limit="2"', $example_display );
	}
	?>
		
	<h4 style="margin-bottom: 5px; margin-top:0;"><?php echo esc_html__( 'Display:', 'bne-testimonials' ); ?></h4>
	<p style="margin-top: 0; font-size: 13px;"><?php echo esc_html__( 'Press update to view any changes made.', 'bne-testimonials' ); ?></p>
	<?php echo do_shortcode($example_display); ?>

	<hr style="margin: 15px auto;">
	<h4 style="margin-bottom: 5px; margin-top:0;"><?php echo esc_html__( 'Reusable Shortcode:', 'bne-testimonials' ); ?></h4>
	<p style="margin-top: 0; font-size: 13px;"><?php echo esc_html__( 'This shortcode will always reflect this testimonial build. Any changes to the design will always carry over to this shortcode.', 'bne-testimonials' ); ?></p>
	<textarea readonly rows="1" style="width: 100%; font-size: 12px;">[bne_testimonials custom="<?php echo $post_id; ?>"]</textarea>

	<hr style="margin: 15px auto;">
	<h4 style="margin-bottom: 5px; margin-top:0px;"><?php echo esc_html__( 'Designed Shortcode', 'bne-testimonials' ); ?>:</h4>
	<p style="margin-top: 0; font-size: 13px;"><?php echo esc_html__( 'This is the extended version allowing you to further adjust after placing it in your page. It will not reflect any future changes from this build after you add it to your page.', 'bne-testimonials' ); ?></p>
	<textarea readonly rows="6" style="width: 100%; font-size: 12px;"><?php echo $output; ?></textarea>
	<?php

	
	/*
	 *	Update Shortcode Results $meta_key
	 *
	 *	@note: This may change in later versions. We went this route
	 *	due to not being able to define the shortcode reference with
	 *	a hidden CMB2 field. Therefore, let's create the key on publish
	 *	and update the field when the post is updated.
	 *
	*/
	update_post_meta( $post_id, '_bne_testimonials_sg_shortcode', $output );

	
}