<?php
/*
 *	API Class and Functions
 *
 * 	@author		Kerry Kline
 * 	@link		http://www.bnecreative.com
 *	@package	BNE Testimonials Pro
 *
*/

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit; 




/*
 *	Testimonials Rest API
 *
 *	Extends the WP_REST_Controller to include
 *	a custom route for BNE Testimonials.
 *
 *	@note		Replaces BNE_Testimonials_API class introduced in v2.3
 *				which allows the bne_testimonials post type to remain private.
 *
 *	@since		v2.6
 *	@updated	v2.7.1
 *
*/
if( class_exists( 'WP_REST_Controller' ) ) {
	class BNE_Testimonials_REST_API extends WP_REST_Controller {
	
		// Namespace within WP Rest API
		var $name_space = 'bnetestimonials/v';
		
		// Current Version
		var $version = '1';
		
		/*
		 * 	Register API Route
		*/
		public function register_routes() {
			
			// Set the namespace and version
			$namespace = $this->name_space . $this->version;
			
			// Register and create the endpoint
			register_rest_route( $namespace, '/reviews', array(
				'methods'	=> WP_REST_Server::READABLE,
				'callback'	=> array( $this, 'get_testimonials' ),
			) );
		}
		
		/*
		 * 	Hook into Rest API INIT with our routes
		*/
		public function hook_rest_server() {
			
			// Check to see if enabled to be public
			$options = get_option('bne_testimonials_settings');
			if( 'on' == isset( $options['enable_api'] ) ) {
				add_action( 'rest_api_init', array( $this, 'register_routes' ) );
			}
		}
		
		/*
		 * 	Get Testimonials
		 *	
		 *	$request	object	The initial API Request
		 *	returns		array	The final information pertaining to testimonials.
		 *
		*/
		public function get_testimonials( WP_REST_Request $request ) {
			
			// Query Args
			$query_args = array(
				'post_type'			=>	'bne_testimonials',
				'posts_per_page'	=>	$request->get_param( 'limit' ) ?: 20,
				'include'			=>	$request->get_param( 'id' ),
				'order'				=>	$request->get_param( 'order' ),
				'orderby'			=>	$request->get_param( 'orderby' ),
			);
		
			// Category Argument
			if( $request->get_param( 'category' ) ) {
				$query_args['tax_query'] = array(
			        array(
			            'taxonomy' 	=> 	'bne-testimonials-taxonomy',
			            'field' 	=> 	'id',
			            'terms' 	=> 	explode( ',', $request->get_param( 'category' ) )
			        )
			    );
			}
			
			// Min Rating Argument
			if( $request->get_param( 'rating_min' ) ) {
			    $query_args['meta_query'] = array(
			        array(
			            'key' 			=> 	'rating',
			            'value' 		=> 	$request->get_param( 'rating_min' ),
			            'compare' 		=>	'>='
			        )
			    );
			}
				
			// Query testimonials
			$testimonials = get_posts( $query_args );
	
			
			// Throw an error if we come up empty
			if( empty( $testimonials) ) {
				return new WP_Error( 'Not Found', esc_html( 'No testimonials to show based on API query.', 'bne-testimonials' ) );
			}
		

			/*
			 * 	Total / Rating calculations
			 *
			 * 	Based on query, get the total reviews with
			 *	ratings and then calculate the average.
			 *	Then add it to the $api_data array.
			 *
			*/
			$total = 0;
			$ratings = array();
			foreach( $testimonials as $testimonial ) {
				$rating = get_post_meta( $testimonial->ID, 'rating', true );
				if( $rating ) { 
					$total++;
				}
				$ratings[] = $rating;
			}
			
			// Add all ratings
			$sum_rating = array_sum( $ratings );
			
			// Get average based on total testimonials
			$average = $sum_rating / $total;
	
			// Format average to 1 decimal point.
			$rating = number_format( $average, 1 );
	
			// Setup the base data for the API response
			$api_data = array(
				'name' 			=>	get_bloginfo('name'),
				'url' 			=>	get_bloginfo('url'),
				'rating'		=>	$rating,
				'rating_count'	=>	$total,
			);
				

			/*
			 * 	Testimonials Loop
			 *
			 * 	For each testimonial, grab their data and apply it
			 *	to the $api_data array as a new entry.
			 *
			*/
			foreach( $testimonials as $testimonial ) {
				
				$data = array(
					'id'				=>	$testimonial->ID,
					'slug'				=>	$testimonial->post_name,
					'name'				=>	$testimonial->post_title,
					'date'				=>	$testimonial->post_date,
					'message'			=>	$testimonial->post_content,
					'image'				=>	wp_get_attachment_image_src( get_post_thumbnail_id( $testimonial->ID ), 'thumbnail' ),
					'image_full'		=>	wp_get_attachment_image_src( get_post_thumbnail_id( $testimonial->ID ), 'full' ),
					'tagline'			=>	get_post_meta( $testimonial->ID, 'tagline', true ),
					'website'			=>	esc_url( get_post_meta( $testimonial->ID, 'website-url', true ) ),
					'email'				=>	null,
					'rating'			=>	(int) get_post_meta( $testimonial->ID, 'rating', true ),
					'category'			=>	get_the_terms( $testimonial->ID, 'bne-testimonials-taxonomy' )
				);

	
				/*
				 *	Dev - Allow developers to alter the returned data
				 *
				 * 	Due to privacy concerns, the email address above is not included even
				 *	if the source website has enabled gravatar support. With that said,
				 *	the email field, and other data, can manually be added using the below filter.
				 *
				*/
				$api_data['reviews'][] = apply_filters( 'bne_testimonials_api_get_data', $data, $testimonial->ID );
	
			}
			
			// Return complete API response
			return $api_data;
		
		}
	
	}
	$BNE_Testimonials_REST_API = new BNE_Testimonials_REST_API();
	$BNE_Testimonials_REST_API->hook_rest_server();
	
}






/*
 *	Branding Assets
 *
 *	Pre-defined assets for supported branding sources.
 *	Depending on the source, these assets may be required when
 *	using the API.
 *
 *	$source		string	The source API type
 *	return		$brands[$source], only if it's available
 *	
 *	@since 		v2.4.1
 *
*/
function bne_testimonials_api_branding_assets( $source = null ) {
	
	// Resets
	$brands_defaults = array(
		'img'			=>	'',
		'alt' 			=> 	'',
		'width' 		=> 	'',
		'height' 		=> 	'',
		'badge'			=>	'',
		'badge_width'	=>	'60px',
		'rating_color'	=>	'',
	);
	
	$brands = apply_filters( 'bne_testimonials_api_branding_assets', array(
		
		'facebook' => array(
			'img'			=>	BNE_TESTIMONIALS_URI.'/assets/images/branding-facebook.png',
			'alt' 			=> 	'Facebook',
			'width' 		=> 	'67',
			'height' 		=> 	'13',
			'badge'			=>	BNE_TESTIMONIALS_URI.'/assets/images/branding-facebook-badge.png',
			'badge_width'	=>	'50px',
			'rating_color'	=>	'#4080FF',
		),
		
		'google' => array(
			'img'			=>	BNE_TESTIMONIALS_URI.'/assets/images/branding-google.png',
			'alt' 			=> 	'Powered by Google',
			'width' 		=> 	'100',
			'height' 		=> 	'13',
			'badge'			=>	BNE_TESTIMONIALS_URI.'/assets/images/branding-google-badge.png',
			'badge_width'	=>	'50px',
			'rating_color'	=>	'#E7711C',
		),
		
		'yelp' => array(
			'img'			=>	BNE_TESTIMONIALS_URI.'/assets/images/branding-yelp.png',
			'alt' 			=> 	'Yelp',
			'width' 		=> 	'50',
			'height' 		=> 	'26',
			'badge'			=>	BNE_TESTIMONIALS_URI.'/assets/images/branding-yelp-badge.png',
			'badge_width'	=>	'50px',
			'rating_color'	=>	'#fff',
		),
		
		'yp' => array(
			'img'			=>	BNE_TESTIMONIALS_URI.'/assets/images/branding-yp.png',
			'alt' 			=> 	'Yellow Pages',
			'width' 		=> 	'100',
			'height' 		=> 	'22',
			'badge'			=>	BNE_TESTIMONIALS_URI.'/assets/images/branding-yp-badge.png',
			'badge_width'	=>	'50px',
		)
		
	) );
	
	if( array_key_exists( $source, $brands ) ) {
		$brands[$source] = wp_parse_args( $brands[$source], $brands_defaults );
		return $brands[$source];
	}
	
}



/*
 *	API Set Testimonial fields
 *
 *	Loops through each of the Testimonials and assign the correct
 *	meta from the JSON API response
 *
 *	$testimonial	array			The assigned data for each individual testimonial
 *	$source			string			The source API type
 *	return			$testimonial
 *	
 *	@since 		v2.3
 *	@updated 	v2.8.4
 *
*/
function bne_testimonaials_api_set_fields( $testimonial, $source = null ) {
		
	// Rand ID
	$rand_id = rand(1,1000);
	
	// Blank Thumb
	$mystery_man = BNE_TESTIMONIALS_URI.'/assets/images/blank-thumb2.jpg';

	// Resets
	$defaults = array(
		'name' 		=> 	'',
		'message'	=>	'',
		'image' 	=> 	$mystery_man,
		'tagline' 	=> 	'',
		'website' 	=> 	'',
		'email'		=>	'',
		'rating' 	=> 	'',
		'date'		=>	'',
		'id'		=>	$source.$rand_id,
	);
		

	// BNE Testimonials Review Meta (WP Local API v1)
	if( $source == 'local' ) {
		$testimonial = array(
			'name' 		=> 	$testimonial['name'],
			'message'	=>	$testimonial['message'],
			'image' 	=> 	$testimonial['image'][0],
			'tagline' 	=> 	$testimonial['tagline'],
			'website' 	=> 	$testimonial['website'],
			'email' 	=> 	$testimonial['email'],
			'rating' 	=> 	$testimonial['rating'],
			'date'		=>	$testimonial['date'],
			'id'		=>	$testimonial['id'],
		);

		
	// Facebook Review Meta
	} elseif( $source == 'facebook' ) {
		$testimonial = array(
			'name' 				=> 	$testimonial['reviewer']['name'],
			'message'			=>	( isset( $testimonial['review_text'] ) ) ? $testimonial['review_text']: '',
			'image' 			=> 	'https://graph.facebook.com/'.$testimonial['reviewer']['id'].'/picture',
			'tagline' 			=> 	esc_html__( 'Facebook Review','bne-testimonials' ),
			'website' 			=> 	$testimonial['url'],
			'rating' 			=> 	( isset( $testimonial['rating'] ) ) ? $testimonial['rating'] : '',
			'recommendation'	=>	( isset( $testimonial['recommendation_type'] ) ) ? $testimonial['recommendation_type'] : '',
			'date'				=>	$testimonial['created_time'],
			'id'				=>	'f-'.$testimonial['reviewer']['id'],
		);
		// Adjust the return size of the image
		if( isset( $testimonial['image'] ) ) {
			$testimonial['image'] .= '?width=300&height=300';
		}


	// Google Review Meta
	} elseif( $source == 'google' ) {
		$testimonial = array(
			'name' 		=> 	$testimonial['author_name'],
			'message'	=>	$testimonial['text'],
			'image' 	=> 	( isset( $testimonial['profile_photo_url'] ) ) ? $testimonial['profile_photo_url'] : $mystery_man,
			'tagline' 	=> 	esc_html__( 'Google Review','bne-testimonials' ),
			'website' 	=> 	( isset( $testimonial['author_url'] ) ) ? $testimonial['author_url'] : null,
			'rating' 	=> 	$testimonial['rating'],
			'date'		=>	date('Y-m-d', $testimonial['time'] ),
			'id'		=>	'g-'.$testimonial['time'],
		);
		// Adjust the return size of the image
		if( isset( $testimonial['image'] ) ) {
			$testimonial['image'] .= '?sz=150';
		}


	// Yelp Review Meta
	} elseif( $source == 'yelp' ) {
		$testimonial = array(
			'name' 		=> 	$testimonial['user']['name'],
			'message'	=>	$testimonial['text'],
			'image' 	=> 	( !empty( $testimonial['user']['image_url'] ) ) ? $testimonial['user']['image_url'] : $mystery_man,
			'tagline' 	=> 	esc_html__( 'Yelp Review','bne-testimonials' ),
			'website' 	=> 	$testimonial['url'],
			'rating' 	=> 	$testimonial['rating'],
			'date'		=>	$testimonial['time_created'],
			'id'		=>	'y-'.$testimonial['id'],
		);
		// Adjust the return size of the image
		if( isset( $testimonial['image'] ) ) {
			$testimonial['image'] = str_replace('o.jpg', '300s.jpg', $testimonial['image']);
		}


	// Yellow Pages
	} elseif( $source == 'yp' ) {
		$testimonial = array(
			'name' 		=> 	$testimonial['reviewer'],
			'message'	=>	$testimonial['reviewBody'],
			//'image' 	=> 	null,
			'tagline' 	=> 	esc_html__( 'YP Review', 'bne-testimonials' ),
			'website' 	=> 	$testimonial['url'],
			'rating' 	=> 	$testimonial['rating'],
			'date'		=>	$testimonial['reviewDate'],
			'id'		=>	'yp-'.strtotime($testimonial['reviewDate'])
		);
	}
	
	// Dev Support to assign fields
	$testimonial = apply_filters( 'bne_testimonials_api_fields', $testimonial, $source  );
	
	// Parse against the defaults.
	$testimonial = wp_parse_args( $testimonial, $defaults );
	
	return $testimonial;
}



/*
 *	API Branding Placement
 *
 *	Some API's require their branding. Since
 *	there are no taglines available, this replaces that area
 *	with the providers logo that links to the review.
 *	
 *	$output			string			Existing markup and data.
 *	$atts			array			Shortcode Options
 *	return			$output
 *
 *	@since 			v2.4
 *	@updated 		v2.7
 *
*/
function bne_testimonials_api_tagline_branding( $output, $atts ) {
	if( isset( $atts['source'] ) && isset( $atts['branding'] ) ) {
		
		if( 'true' == $atts['branding'] && 'local' != $atts['source'] ) {
			
			// Get Brand Assets
			$brand_assets = bne_testimonials_api_branding_assets( $atts['source'] );
						
			// Build out the new tagline/website with branding
			if( $brand_assets && $brand_assets['img'] ) {
				$output = '<span class="testimonial-website">';
					if( $atts['api']['website'] ) {
						$output .= '<a href="'.esc_url( $atts['api']['website'] ).'" target="_blank" title="'.$atts['api']['tagline'].'" rel="nofollow">';
					}
					
					$output .= sprintf( '<img src="%s" alt="%s" class="branding-logo" width="%s" height="%s" />',
						$brand_assets['img'],
						$brand_assets['alt'],
						$brand_assets['width'],
						$brand_assets['height']
					);
					
					if( $atts['api']['website'] ) {
						$output .= '</a>';
					}
				$output .= '</span>';
	
			}
						
		}
	
	}
	
	return $output;

}
// Using priority 15 to make sure branding sticks but still allowing user to alter.
add_filter( 'bne_testimonials_tagline_and_website', 'bne_testimonials_api_tagline_branding', 15, 2 );




/*
 *	API Request
 *
 *	This function handles the API request and cache state
 *
 *	$source		string		The API provider (local, google, facebook, yp, yelp)
 *	$key		string		The API Key if different from settings.
 *	$url		string		Local WP URL for BNE Testimonials API
 *	$id			string		The Google, Yelp, Yellow Pages, or Facebook page ID
 *	$cache		boolan		Should we fetch a new response or use cache
 *	$query_args	array		The shortcode query arguments
 *	
 *	@since 		v2.4
 *	@updated 	v2.8.2
 *
*/
function bne_testimonials_api_request( $source, $key, $url, $id, $cache, $query_args ) {

	
	// Prefix cache - Make cache relate only to current plugin version
	$hash_version = BNE_TESTIMONIALS_VERSION;
	$hash_prefix = "bne_testimonials_api_{$hash_version}_{$source}_";

	
	// API Options
	$options = get_option( 'bne_testimonials_settings' );
	$key = apply_filters( 'bne_testimonials_api_key', $key, $source, $id );
	
	// API Defaults
	$api_data = array();
	$api_data_defaults = array(
		'api_url' 			=>	'',
		'api_remote_args'	=>	'',
		'api_hash'			=>	''
	);


	// BNE Tesimonails API
	if( $source == 'local' ) {
		$api_data['api_url'] = add_query_arg( $query_args, esc_url( trailingslashit( $url ).'wp-json/bnetestimonials/v1/reviews' ) );
	}
	

	// Facebook
	if( $source == 'facebook' ) {
		if( empty( $key ) && isset( $options['facebook_api_key'] ) ) {
			$key = $options['facebook_api_key'];
		}
		$query_args = array(
			'access_token'	=>	esc_html( $key ),
			'fields'		=>	'name,link,rating_count,overall_star_rating'
		);
		$api_data['api_url'] = add_query_arg( $query_args, esc_url( "https://graph.facebook.com/v3.1/{$id}" ) );
	}


	// Google API
	if( $source == 'google' ) {
		if( empty( $key ) && isset( $options['google_places_api_key'] ) ) {
			$key = $options['google_places_api_key'];
		}
		$query_args = array(
			'key'		=>	esc_html( $key ),
			'placeid'	=>	esc_html( $id )
		);
		$api_data['api_url'] = add_query_arg( $query_args, 'https://maps.googleapis.com/maps/api/place/details/json' );
	}


	// Yelp API
	if( $source == 'yelp' ) {

		/*
		 * 	Yelp API Key
		 *
		 * 	Starting March 1, 2018, Yelp will no longer require a token as an
		 *	authorization and moved to using an API Key. As before, we have 
		 *	included our Key here and removed the token endpoint request.
		 *
		 *	@depreciated bne_testimonials_yelp_token()
		*/
		if( empty( $key ) ) {
			if( isset( $options['yelp_api_key'] ) ) {
				$key = esc_html( $options['yelp_api_key'] );
			} else {
				$key = '4YGzj8d_Q8IINYq54ZhQAvHZG5lDlUSOqvoBFBh79T7KZBApN7LyEVGR2L7cpO7p5_SDDY46-MA0LoezZEIxtKdIZ5oz30rP13i4JWC2c8emdXg1U0856Iaql6CsWHYx';
			}
		}		
		
		$api_data['api_remote_args'] = array(
			'user-agent' 	=> 	'',
			'headers' 		=> 	array(
				'Authorization'	=>	"Bearer {$key}",
			),
		);
		$api_data['api_url'] = add_query_arg( $api_data['api_remote_args'], esc_url( "https://api.yelp.com/v3/businesses/{$id}/reviews" ) );
	}


	// Yellow Pages (YP)
	if( $source == 'yp') {
		if( empty( $key ) && isset( $options['yp_api_key'] ) ) {
			$key = $options['yp_api_key'];
		}
		$query_args = array(
			'listingid'	=>	$id,
			'key'		=>	$key,
			'format'	=>	'json'
		);
		$api_data['api_url'] = add_query_arg( $query_args, 'http://pubapi.yp.com/search-api/search/devapi/details' );
	}


	// Allow Developers to add new sources:
	$api_data = apply_filters( 'bne_testimonials_api_source_args', $api_data, $source, $id, $key );


	// Parse $api_data with defaults
	$api_data = wp_parse_args( $api_data, $api_data_defaults );


	// Set Source/ID Hash for caching the response
	$api_data['api_hash'] = $hash_prefix.md5( $api_data['api_url'] );



	/*
	 * 	Testimonial Remote Get Request
	 *
	 *	First check if there is a cache version of the API,
	 *	if not, request a new set.
	 *
	*/
	$testimonials = get_transient( $api_data['api_hash'] );
	if( $testimonials === false || 'false' == $cache ) {

		// Request new Data
		$request = wp_remote_get( $api_data['api_url'], $api_data['api_remote_args'] );
		$body = wp_remote_retrieve_body( $request );

		// Remove UTF-8 BOM if present, json_decode() does not like it.
		if( substr( $body, 0, 3 ) == pack( "CCC", 0xEF, 0xBB, 0xBF ) ) {
			$body = substr( $body, 3 );
		}
		
		$api_response = json_decode( $body, true );


		/*
		 * 	Catch any errors from the API request
		 *
		*/
		$error_message = null;
		// WP Error
		if( is_wp_error( $request ) ) {
			$error_message = $request->get_error_message();
		}
		// WP Route Error
		elseif( !empty( $api_response['message'] ) ) {
			if( $source == 'local' ) {
				$error_message = $api_response['message'];
			}
		}
		// Facebook Error
		elseif( $source == 'facebook' ) {
			if( !empty( $api_response['error'] ) ) {
				$error_message = $api_response['error']['message'];
			}
		}
		// Google Key Error
		elseif( $source == 'google' ) {
			if( isset( $api_response['error_message'] ) ) {
				$error_message = $api_response['error_message'];
			} elseif( isset( $api_response['status'] ) ) {
				if(  $api_response['status'] == 'INVALID_REQUEST' )  {
					$error_message = 'The PlaceID is invalid.';
				}
			}
		}
		// Yelp Error
		elseif( $source == 'yelp' ) {
			if( !empty( $api_response['error'] ) ) {
				$error_message = $api_response['error']['description'];
			}
		}
		// YP Error
		elseif( $source == 'yp' ) {
			if( !empty( $api_response['result']['metaProperties']['message'] ) ) {
				$error_message = $api_response['result']['metaProperties']['message'];
			} elseif( '0' == $api_response['listingsDetailsResult']['metaProperties']['listingCount'] ) {
				$error_message = 'Listing ID is invalid';
			}
		}
		// Generic
		elseif( !is_array($api_response) ) {
			$error_message = 'API Not available.';
		}
		if( $error_message ) {
			return ucfirst($source).' API Error: '.$error_message;
		}
		
		

		/*
		 * 	Testimonials Data Structure
		 *
		 *	Arrange the testimonial data based
		 *	on source API reponses.
		 *
		*/
		
		// Default Array Assignments
		$testimonials = array();
		$response_defaults = array(
			'name'		=>	'',
			'icon'		=>	'',
			'rating'	=>	'',
			'total'		=>	'',
			'url'		=>	'',
			'reviews'	=>	array(),
		);

		// BNE Tesimonials API (WP Local v1)
		if( $source == 'local' ) {
			$testimonials = array(
				'name'		=>	$api_response['name'],
				'rating'	=>	$api_response['rating'],
				'total'		=>	$api_response['rating_count'],
				'url'		=>	$api_response['url'],
				'reviews'	=>	$api_response['reviews'],
			);
		}
		

		// Facebook
		elseif( $source == 'facebook' ) {
						
			/* 
			 *	To get more than 25 reviews we need to do an additional request
			 * 	to the page's ratings endpoint with a different set of query arguments.
			 *	Developers can also use the below filter to adjust API query.
			*/
			$query_args = apply_filters( 'bne_testimonials_api_facebook_args', array( 
				'access_token' 	=>	esc_html( $key ),
				'limit'			=>	200
			) );
			$reviews_url = add_query_arg( $query_args, "https://graph.facebook.com/v3.1/{$id}/ratings" );
			$reviews_request = wp_remote_get( $reviews_url );
			$reviews_body = wp_remote_retrieve_body( $reviews_request );
			$reviews_data = json_decode( $reviews_body, true );
			$reviews_data = $reviews_data['data'];
			
			$testimonials = array(
				'name'		=>	$api_response['name'],
				'rating'	=>	$api_response['overall_star_rating'],
				'total'		=>	$api_response['rating_count'],
				'url'		=>	$api_response['link'],
				'reviews'	=>	( isset( $reviews_data ) ) ? $reviews_data : '',
			);
			
			// Inject the page URL since Facebook doesn't provide a direct URL to the review
			if( $testimonials['reviews'] ) {
				foreach( $testimonials['reviews'] as $key => $value ) {
					$testimonials['reviews'][$key]['url'] = $api_response['link'].'reviews/';
				}
			}
			
		}


		// Google Places
		elseif( $source == 'google' ) {
			$testimonials = array(
				'name'		=>	$api_response['result']['name'],
				'rating'	=>	$api_response['result']['rating'],
				'total'		=>	'', // Total is not provided with the API
				'url'		=>	$api_response['result']['url'],
				'reviews'	=>	$api_response['result']['reviews'],
			);
		}

	
		// Yelp
		elseif( $source == 'yelp' ) {
	
			// We need to do one more request for Business Information
			$business_request = wp_remote_get( "https://api.yelp.com/v3/businesses/{$id}", $api_data['api_remote_args'] );
			$business_body = wp_remote_retrieve_body( $business_request );
			$business_data = json_decode( $business_body, true );

			$testimonials = array(
				'name'		=>	$business_data['name'],
				'rating'	=>	$business_data['rating'],
				'total'		=>	$business_data['review_count'],
				'url'		=>	$business_data['url'],
				'reviews'	=>	$api_response['reviews'],
			);
		}
		
	
		// Yellow Pages
		if( $source == 'yp') {
	
			// Main Business Information
			$business_data = $api_response['listingsDetailsResult']['listingsDetails']['listingDetail'][0];
			$testimonials = array(
				'name'		=>	$business_data['businessName'],
				'rating'	=>	$business_data['averageRating'],
				'total'		=>	$business_data['ratingCount'],
				'url'		=>	$business_data['attribution'],
			);
	
			// We need to do one more request for the "reviews" endpoint
			$query_args = array(
				'listingid'	=>	$id,
				'key'		=>	$key,
				'format'	=>	'json'
			);
			$reviews_url = add_query_arg( $query_args, 'http://pubapi.yp.com/search-api/search/devapi/reviews' );
			$reviews_request = wp_remote_get( $reviews_url );
			$review_body = wp_remote_retrieve_body( $reviews_request );
			$reviews_data = json_decode( $review_body, true );
			$testimonials['reviews'] = $reviews_data['ratingsAndReviewsResult']['reviews']['review'];
			
			// Inject the page URL
			foreach($testimonials['reviews'] as $key => $value ) {
				$testimonials['reviews'][$key]['url'] = $business_data['attribution'].'#reviews';
			}

		}


		/*
		 * 	Dev - Allow developers to define response data from API request.
		 *	This is in addition to also using bne_testimonials_api_source_args to define the source
		 *
		*/
		$testimonials = apply_filters( 'bne_testimonials_api_source_response', $testimonials, $source, $api_response );


		// Parse final array with Defaults
		$testimonials = wp_parse_args( $testimonials, $response_defaults );
	
		
		// Cache results in the local DB
		if( 'true' == $cache ) {
			$day = 60 * 60 * 24; 		// 1 Day
			$week = 60 * 60 * 168; 		// 1 Week
			$expires = apply_filters( 'bne_testimonials_cache_length', $week );	// Allow Devs to change expiration
			set_transient( $api_data['api_hash'], $testimonials, $expires );
		}

	} // END API Query


	return $testimonials;

}



/*
 *	Clear BNE Testimonials Cache 
 *
 *	Since API respones are stored locally in the database,
 *	there may be a need to clear it manually if testing or a
 *	token needs to be cleared prior to expiration.
 *
 *	@since v2.4.3
 *
*/
function bne_testimonials_delete_cache() {

	if( empty( $_REQUEST['action'] ) || empty( $_REQUEST['_wpnonce'] ) ) {
		return;
	}

	if( !current_user_can( 'manage_options' ) ) {
		return;
	}
	
	if( !wp_verify_nonce( $_REQUEST['_wpnonce'], 'bne_testimonials_api_delete_transients' ) ) {
		return;
	}
	
    global $wpdb;
	
    $prefix = $wpdb->esc_like( '_transient_bne_testimonials_api_' );
 
    // Build up our SQL query
    $sql = "SELECT `option_name` FROM $wpdb->options WHERE `option_name` LIKE '%s'";
 
    // Execute our query
    $transients = $wpdb->get_results( $wpdb->prepare( $sql, $prefix . '%' ), ARRAY_A );
 
    // If if looks good, continue on
    if( $transients && !is_wp_error( $transients ) ) {

	    if( !isset( $transients ) ) {
	        return false;
	    }
	 
	    // If we get a string key passed in, might as well use it correctly
	    if( is_string( $transients ) ) {
	        $transients = array( array( 'option_name' => $transients ) );
	    }
	 
	    // If its not an array, we can't do anything
	    if( !is_array( $transients ) ) {
	        return false;
	    }
	 
	    $results = array();
	 
	    // Loop through our transients
	    foreach( $transients as $transient ) {
	 
	        if( is_array( $transient ) ) {
	 
	            // If we have an array, grab the first element
	            $transient = current( $transient );
	        }
	 
	        // Remove that sucker
	        $results[$transient] = delete_transient( str_replace( '_transient_', '', $transient ) );
	    }
	 
		add_settings_error(
			'bne_testimonials_clear_cache_notice',
			esc_attr( 'settings_updated' ),
			esc_html__( 'Successfully cleared the API cache', 'bne-testimonials' ),
			'updated'
		);
	    
    }

}
add_action( 'admin_init', 'bne_testimonials_delete_cache' );