<?php
/*
 * Plugin Name: BNE Testimonials Pro
 * Version: 2.8.7
 * Plugin URI: https://www.bnecreative.com/products/testimonials-wordpress-pro/
 * Description: Easily display Testimonials on any page, post, or widget area. Display the testimonials as a traditional list, slider, or masonry, and includes frontend submission.
 * Author: پرشین اسکریپت
 * Author URI: https://www.persianscript.ir
 * Requires at least: 4.7
 * Text Domain: bne-testimonials
 * License: GPL2

    Copyright (C) 2013-2018 BNE Creative

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2,
    as published by the Free Software Foundation.

    You may NOT assume that you can use any other version of the GPL.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    The license for this software can likely be found here:
    http://www.gnu.org/licenses/gpl-2.0.html

*/


// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;


class BNE_Testimonials_Pro {

	
    /*
     * 	Constructor
     *
     *	@since 		v2.3
     *	@updated	v2.5
     *
    */
	function __construct() {
		
		// Set Constants
		define( 'BNE_TESTIMONIALS_VERSION', '2.8.7' );
		define( 'BNE_TESTIMONIALS_DIR', dirname( __FILE__ ) );
		define( 'BNE_TESTIMONIALS_URI', plugins_url( '', __FILE__ ) );
		define( 'BNE_TESTIMONIALS_BASENAME', plugin_basename( __FILE__ ) );
		
		// Textdomain
		add_action( 'plugins_loaded', array( $this, 'text_domain' ) );
		
		// Setup Includes / Files
		add_action( 'after_setup_theme', array( $this, 'setup' ) );
		
		// Scripts 
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ), 11, 1 );
		
		// Deactivate Legacy Widgets
		add_action( 'widgets_init', array( $this, 'deactivate_widgets_1x' ), 12 );
		
		// Privacy Notice
		add_action( 'admin_init', array($this, 'privacy_policy_content' ), 20 );
		
		// Admin Notices
		//add_action( 'admin_notices', array( $this, 'notice' ) );
		
		
	}


	/*
	 * 	Informational Notice
	 *
	*/
/*
	function notice() {
		$notice_id = 'bne_testimonials_notice2_7';
		$dismissed = get_option( $notice_id );
		if( $dismissed == 'notified' || !current_user_can('manage_options' ) ) {
			return;
		}
		if( isset( $_GET[$notice_id] ) && $_GET[$notice_id] ) {
			if( $_GET[$notice_id] == 'close_notice' ) {
				update_option( $notice_id, 'notified' );
				return;
			}
		}
		$help_url = 'http://docs.bnecreative.com/articles/plugins/testimonials-pro/#api_docs';
		$close = add_query_arg( $notice_id, 'close_notice' );
		?>
		<div class="notice notice-info is-dismissible">
			<p>
			<?php echo sprintf( 
				'<strong>New in BNE Testimonials:</strong> '.esc_html__( 'Reviews from API platforms can now be combined into a single display, including badges, instead of using multiple shortcodes. %3$s%1$s%5$s | %4$s%2$s%5$s', 'bne-testimonials' ),
				esc_html__( 'Learn More', 'bne-testimonials' ),
				esc_html__( 'Close', 'bne-testimonials' ),
				'<a href="'.esc_url( 'https://www.bnecreative.com/blog/bne-testimonials-pro-v2-7/' ).'" target="_blank">',
				'<a href="'.esc_url( $close ).'">',
				'</a>'
				
			); ?>
			</p>
		</div>
		<?php
	}
*/



	/*
	 * 	Textdomain for Localization
	 *
	 * 	@since		v1.8
	 * 	@updated	v2.3
	*/
	function text_domain() {
		load_plugin_textdomain( 'bne-testimonials', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
	}



	/*
	 *	Plugin Setup
	 *	
	 * 	@since 		v1.0
	 * 	@updated 	v2.6
	 *
	*/
	function setup() {
		
		// Libraries
		include_once( BNE_TESTIMONIALS_DIR . '/includes/lib/bne-cpt.php' );

		// Admin Only Pages
		if( is_admin() ) {
			
			// CMB
			include_once( BNE_TESTIMONIALS_DIR . '/includes/lib/cmb2/init.php' );

			// CMB Addons
			include_once( BNE_TESTIMONIALS_DIR . '/includes/lib/cmb2-conditionals/cmb2-conditionals.php' );
			
			// Admin Help Page
			include_once( BNE_TESTIMONIALS_DIR . '/includes/help/help.php' );
		
		}

		// Settings
		include_once( BNE_TESTIMONIALS_DIR . '/includes/settings.php' );

		// API
		include_once( BNE_TESTIMONIALS_DIR . '/includes/api.php' );

		// Support
		include_once( BNE_TESTIMONIALS_DIR . '/includes/locals.php' );
		include_once( BNE_TESTIMONIALS_DIR . '/includes/form-functions.php' );

		// CPT
		include_once( BNE_TESTIMONIALS_DIR . '/includes/cpt-main.php' );
		include_once( BNE_TESTIMONIALS_DIR . '/includes/cpt-generator.php' );

		// Shortcodes
		include_once( BNE_TESTIMONIALS_DIR . '/includes/shortcode-display.php' );
		include_once( BNE_TESTIMONIALS_DIR . '/includes/shortcode-display-api.php' );
		include_once( BNE_TESTIMONIALS_DIR . '/includes/shortcode-form.php' );
		include_once( BNE_TESTIMONIALS_DIR . '/includes/shortcode-badge.php' );
		include_once( BNE_TESTIMONIALS_DIR . '/includes/shortcode-stats.php' );

		
		/*
		 *	Plugin Update Function (BNE Server)
		 *	Allows updating the plugin from the WP Dashboard.
		 *
		*/
		require( BNE_TESTIMONIALS_DIR .'/includes/lib/update-checker/plugin-update-checker.php' );
		$BNE_testimonials_pro_update = Puc_v4_Factory::buildUpdateChecker(
			'https://updates.bnecreative.com/?action=get_metadata&slug=bne-testimonials-pro',
			__FILE__,
			'bne-testimonials-pro'
		);
		
		
		/*
		 *	Thumbnail Support
		 *
		 *	Because some themes will selectively choose what post types
		 *	can use post-thumbnails, we will first remove support to
		 *	basically reset the option, then we will add it back.
		 *
		 *	This may seem link backwards thinking, but works.
		 *	
		*/
		remove_theme_support( 'post-thumbnails' );
		if( !current_theme_supports( 'post-thumbnails' ) ) {
			add_theme_support( 'post-thumbnails' );
		}


		// v2.0 Widget
		include_once( BNE_TESTIMONIALS_DIR . '/includes/widgets.php' );
		
		
		
		/*
		 *	v1.x legacy Shortcodes and Widgets
		 *
		 *	Prior to v2.0, Testimonials used a customizable filter set
		 *	to output the data. This allowed devs to re-arrange and customize
		 *	how the testimonials are displayed. With v2.0+ we have removed all
		 *	of this in favor of pre-defined themes. To prevent confusion and
		 *	disruption of customizations, we have moved these functions here
		 *	and can only be called using the legacy shortcode varients and widgets.
		 *
		 *	Shortcodes using this structure:
		 *	[bne_testimonials_list]
		 *	[bne_testimonials_slider]
		 *	[bne_testimonials_masonry]
		 *
		*/
		
		// Planned Migration Step 2 of 3
		include_once( BNE_TESTIMONIALS_DIR . '/includes/legacy/migrate.php' );
		
		//include_once( BNE_TESTIMONIALS_DIR . '/includes/legacy/testimonial-output.php' );
		//include_once( BNE_TESTIMONIALS_DIR . '/includes/legacy/shortcode-list.php' );
		//include_once( BNE_TESTIMONIALS_DIR . '/includes/legacy/shortcode-masonry.php' );
		//include_once( BNE_TESTIMONIALS_DIR . '/includes/legacy/shortcode-slider.php' );	
		
		include_once( BNE_TESTIMONIALS_DIR . '/includes/legacy/widget-list.php' );
		include_once( BNE_TESTIMONIALS_DIR . '/includes/legacy/widget-slider.php' );

	}




	/*
	 *	Remove v1.x legacy Widgets
	 *
	 *	If the site still has the List Widget or Slider Widget activated,
	 *	let's continue to use them. However, if they do not, then lets
	 *	remove them for future users.
	 *
	 *	@since 		v2.0
	 *	@updated 	v2.3
	 *
	*/
	function deactivate_widgets_1x() {
	
		// List Widget Check
		if( !is_active_widget( false, false, 'bne_testimonials_list_widget') ) {
			unregister_widget( 'bne_testimonials_list_widget' );
		}
		
		// Slider Widget Check
		if( !is_active_widget( false, false, 'bne_testimonials_slider_widget') ) {
			unregister_widget( 'bne_testimonials_slider_widget' );
		}	
	}
	



	/*
	 *	Register frontend CSS and JS
	 *
	 *	@since 		v1.0
	 *	@updated 	v2.4.2
	 *
	*/
	function frontend_scripts() {
		
		$min = ( defined('WP_DEBUG') && true === WP_DEBUG ) ? '' : '.min';
		
		// CSS
		wp_register_style( 'bne-testimonials-css', BNE_TESTIMONIALS_URI . '/assets/css/bne-testimonials'.$min.'.css', '', BNE_TESTIMONIALS_VERSION, 'all' );
		
		// Masonry JS
		wp_register_script( 'masonry', BNE_TESTIMONIALS_URI . '/assets/js/masonry.min.js', array('jquery'), '3.1.5', true );
		
		// Curtail JS
		wp_register_script( 'curtail', BNE_TESTIMONIALS_URI . '/assets/js/jquery.curtail.min.js', array('jquery'), '1.1.2', true );
		
		// Check if we're on a BNE WordPress Theme...
		if( !defined('BNE_FRAMEWORK_VERSION') ) {
			// Flexslider
			wp_register_script( 'flexslider', BNE_TESTIMONIALS_URI . '/assets/js/flexslider.min.js', array('jquery'), '2.2.2', true );
		}
	
		// Load the plugin CSS
		wp_enqueue_style( 'bne-testimonials-css');
	
	}



	/*
	 *	
	 *	Register Admin CSS and JS
	 *
	 *	@since 		v1.0
	 *	@updated 	v2.3
	 *
	*/
	function admin_scripts( $hook ) {
		
		global $post;
		
		// Post Types to check against
		$cpt_slugs = array(
			'bne_testimonials',
			'bne_testimonials_sg',
		);
		
		// Check if we're on a post new or edit admin screen.
		if( $hook == 'post-new.php' || $hook == 'post.php' ) {
			
			// Crosscheck with our Post Type list.
			if( in_array( $post->post_type, $cpt_slugs ) ) {     
				
				// Finally, check if we're not on a BNE theme as this is already available from there.
				if( !defined('BNE_FRAMEWORK_VERSION') ) {
					wp_enqueue_style( 'bne-cmb-admin-css', BNE_TESTIMONIALS_URI . '/assets/css/bne-cmb-admin.css', '', BNE_TESTIMONIALS_VERSION, 'all'  );
				}
			
			}
		}
	}


	/*
	 * 	GDPR Privacy Statement
	 *
	 *	@todo
	 *
	*/
	function privacy_policy_content() {
		if( !function_exists( 'wp_add_privacy_policy_content' ) ) {
			return;
		}

		$content = '<h2>'.__( 'Review & Testimonial Submission', 'bne-testimonials' ).'</h2>';
		$content .= '<p>'.__( 'When visitors leave reviews on this website, we collect the data shown in the form. An anonymized string created from your email address (also called a hash) may be provided to the Gravatar service to see if you are using it. The Gravatar service privacy policy is available here: https://automattic.com/privacy/. After approval of your review, your profile picture either from your supplied image upload or from the Gravatar service, may be visible to the public in the context of your review.', 'bne-testimonials' ).'</p>';
		
		wp_add_privacy_policy_content( 
			__( 'BNE Testimonials', 'bne-testimonials' ), 
			wp_kses_post( wpautop( $content, false ) )
		);
	}



	
	
} // END Class

// Initiate the Class
$BNE_Testimonials_Pro = new BNE_Testimonials_Pro();



/*
 * 	Get Template Parts
 *
 *	Returns the template part needed to display in the testimonial.
 *
 *	$part		string			The filename of the template part.
 *	$atts		string|array	Any needed attributes to pass.
 *	$return		string			Either return or echo the part.
 *	$wrapper	string			The wrapping container - div, h1, h3, h4, p, etc.
 *
 *	@since 		v2.3
*/
function bne_testimonials_get_template( $part, $atts, $api = null, $return = 'return', $wrapper = null ) {
	return include( BNE_TESTIMONIALS_DIR . "/includes/templates/$part.php" );
}



/*
 * 	Get Testimonial Gravatar
 *
 *	Queries Gravatar and checks if an image is set for the email. If it is, get the URL.
 *
 *	$email		string			Email address to check against
 *	$size		int				Gravatar returned sized
 *	$gravatar	string			Default/fallback Image
 *
 *	@since v2.5
*/
function bne_testimonials_get_gravatar( $email = null, $size = '150', $gravatar = null ) {
	$options = get_option('bne_testimonials_settings');
	if( isset( $options['enable_gravatars'] ) ) {
		if( $options['enable_gravatars'] == 'on' && $email ) {
			$hash = md5( stripslashes( sanitize_email( $email ) ) );
			$resp = wp_remote_head( "https://www.gravatar.com/avatar/{$hash}?d=404" );
			if( !is_wp_error( $resp ) && $resp['response']['code'] == 200 ) {
				$gravatar = get_avatar_url( $email, array( 'size' => $size ) );
			}
		}
	}
	
	return $gravatar;

}



/*
 * 	Get Schema Tags
 *
 *	Assigns the appropriate schema tag/markup in the testimonial
 *	areas
 *
 *	$enabled	string			Check if schema is enabled from shortcode
 *	$part		string			The schema tag to return
 *	$data		string|array	The schema data to use if defined
 *
 *	@since 	v2.8.7
*/
function bne_testimonials_get_schema( $enabled, $part, $data = null ) {
	
	// Bail if schema is turned off!
	if( 'true' != $enabled ) {
		return '';
	}
	
	// Schema Header
	// Template: shortcode
	if( 'header' == $part ) {
		$schema = 'itemscope itemtype="http://schema.org/Review"';
	}

	// Schema Type (Product, LocalBusiness, etc)
	// Template: single
	if( 'type' == $part ) {
		$schema = '<meta itemprop="datePublished" content="'.$data['date'].'">';
		$schema .= '<span itemprop="itemReviewed" itemscope itemtype="http://schema.org/'.$data['type'].'">';
			$schema .= '<meta itemprop="name" content="'.$data['name'].'">';
			$schema .= '<meta itemprop="image" content="'.$data['image'].'">';
			// Allow Devs to add additional schema tags for their particular type (pricerange, telephone, address, etc)
			$schema .= apply_filters( 'bne_testimonials_schema_additional_tags', '' );
		$schema .= '</span>';
	}
	
	// Review Body
	// Template: message
	if( 'message' == $part ) {
		$schema = 'itemprop="reviewBody"';
	}

	// Author Info
	// Template: name
	if( 'author' == $part ) {
		$schema = '<span itemprop="author" itemscope itemtype="http://schema.org/Person">';
			$schema .= '<meta itemprop="name" content="'.esc_attr( $data['name'] ).'">';
		$schema .= '</span>';
	}

	// Rating Type
	// Template: rating
	if( 'rating_header' == $part ) {
		$schema = 'itemprop="reviewRating" itemscope="" itemtype="http://schema.org/'.$data.'"';
	}
	
	// Rating
	// Template: rating
	if( 'rating' == $part ) {
		$schema = '<meta itemprop="worstRating" content="1">';
		$schema .= '<meta itemprop="ratingValue" content="'.$data['rating'].'">';
		$schema .= '<meta itemprop="bestRating" content="5">';
		if( 'true' == $data['aggregate'] ) {
			$schema .= '<meta itemprop="ratingCount" content="'.$data['count'].'">';
		}
	}
	
	// Return tags
	return $schema;
}