<?php
/**
 * Remove Registrant Meeting
 *
 * @package     AutomatorWP\Integrations\Zoom\Actions\Remove_Registrant_Meeting
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Zoom_Remove_Registrant_Meeting extends AutomatorWP_Integration_Action {

    public $integration = 'zoom';
    public $action = 'zoom_remove_registrant_meeting';

    /**
     * Registrant Email
     *
     * @since 1.0.0
     *
     * @var string $registrant_email
     */
    public $registrant_email = '';

    /**
     * Registrant ID
     *
     * @since 1.0.0
     *
     * @var string $registrant_id
     */
    public $registrant_id = '';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Remove user from meeting', 'automatorwp-pro' ),
            'select_option'     => __( 'Remove user from <strong>meeting</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Registrant. %2$s: Meeting. */
            'edit_label'        => sprintf( __( 'Remove %1$s from %2$s', 'automatorwp-pro' ), '{registrant}', '{meeting}' ),
            /* translators: %1$s: Registrant. %2$s: Meeting. */
            'log_label'         => sprintf( __( 'Remove %1$s from %2$s', 'automatorwp-pro' ), '{registrant}', '{meeting}' ),
            'options'           => array(
                'registrant' => array(
                    'from' => 'registrant_email',
                    'default' => __( 'registrant', 'automatorwp-pro' ),
                    'fields' => array(
                        'registrant_email' => array(
                            'name' => __( 'Email:', 'automatorwp-pro' ),
                            'desc' => __( 'The registrant email.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required' => true
                        ),
                    )
                ),
                'meeting' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'meeting',
                    'option_default'    => __( 'meeting', 'automatorwp-pro' ),
                    'placeholder'       => __( 'Select a meeting', 'automatorwp-pro' ),
                    'name'              => __( 'Meeting:', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_zoom_get_meetings',
                    'options_cb'        => 'automatorwp_zoom_options_cb_meetings',
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->registrant_email = '';
        $this->registrant_id = '';
        $this->result = '';

        // Shorthand
        $meeting_id = $action_options['meeting'];
        $registrant_email = $action_options['registrant_email'];

        // Bail if empty meeting to assign
        if( empty( $meeting_id ) ) {
            $this->result = __( 'Could not add registrant to meeting, missing meeting', 'automatorwp-pro' );
            return;
        }

        $params = automatorwp_zoom_get_request_parameters( 'meetings' );

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            $this->result = __( 'Could not add registrant to meeting, Zoom authentication failed', 'automatorwp-pro' );
            return;
        }

        $this->registrant_email = $registrant_email;

        $this->registrant_id = automatorwp_zoom_get_meeting_registrant_id( $meeting_id,  $this->registrant_email );

        if( $this->registrant_id === false ) {
            $this->result = __( 'Could not add registrant to meeting, registrant was not registered on this meeting', 'automatorwp-pro' );
            return;
        }

        $params['method'] = 'PUT';
        $params['body'] = json_encode( array(
            'action'      => 'cancel',
            'registrants' => array(
                array(
                    'id'    => $this->registrant_id,
                    'email' => $this->registrant_email,
                ),
            ),
        ) );

        // Setup the URL
        $url = 'https://api.zoom.us/v2/meetings/' . $meeting_id . '/registrants/status';

        // Execute the request
        $response = wp_remote_post( $url, $params );

        if ( is_wp_error( $response ) ) {
            $this->result = sprintf( __( 'Could not remove registrant from meeting, error received: %1$s', 'automatorwp-pro' ), $response->get_error_message() );
            return;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $body = json_decode( $response['body'], true, 512, JSON_BIGINT_AS_STRING );

        if ( $response_code === 201 || $response_code === 204 ) {
            $this->result = __( 'Registrant successfully removed', 'automatorwp-pro' );
        } else {
            $this->result = sprintf( __( 'Could not remove registrant from meeting, error received: %1$s', 'automatorwp-pro' ), $body['message'] );
        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' || $object->type !== $this->action ) {
            return;
        }

        $params = automatorwp_zoom_get_request_parameters( 'meetings' );

        // Warn user if the authorization has not been setup from settings
        if( $params === false ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">Zoom Meetings settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-zoom'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/zoom/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['registrant_email'] = $this->registrant_email;
        $log_meta['registrant_id'] = $this->registrant_id;
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['registrant_email'] = array(
            'name' => __( 'Registrant Email:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['registrant_id'] = array(
            'name' => __( 'Registrant ID:', 'automatorwp-pro' ),
            'desc' => __( 'Unique identifier assigned to the registrant.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Zoom_Remove_Registrant_Meeting();