<?php
/**
 * Watch Video
 *
 * @package     AutomatorWP\Integrations\Youtube\Triggers\Watch_Video
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Youtube_Watch_Video extends AutomatorWP_Integration_Trigger {

    public $integration = 'youtube';
    public $trigger = 'youtube_watch_video';

    public function hooks() {

        // Custom trigger replacements
        add_filter( 'automatorwp_parse_automation_item_label_replacements', array( $this, 'item_label_replacements' ), 10, 5 );

        // Custom html on automations UI
        add_action( 'automatorwp_automation_ui_after_item_label', array( $this, 'after_item_label' ) );

        // Custom title replacement
        add_filter( 'automatorwp_clone_trigger_youtube_watch_video_title', array( $this, 'item_label_clone_replacements' ), 10, 3 );

        parent::hooks();
    }

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User watches a Youtube video', 'automatorwp-pro' ),
            'select_option'     => __( 'User watches <strong>a Youtube video</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Youtube shortcode. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User watches %1$s %2$s time(s)', 'automatorwp-pro' ), '{youtube}', '{times}' ),
            /* translators: %1$s: Youtube shortcode. */
            'log_label'         => sprintf( __( 'User watches %1$s', 'automatorwp-pro' ), '{youtube}' ),
            'action'            => 'automatorwp_youtube_video_watched',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 6,
            'options'           => array(
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Parses item title replacements
     *
     * @since 1.0.0
     *
     * @param string    $item_title      The item title
     * @param int       $old_item_id     The old item ID
     * @param int       $new_item_id     The new item ID
     *
     * @return array
     */
    public function item_label_clone_replacements( $item_title, $old_item_id, $new_item_id ) {
    
        // ID replacement
        $item_title = str_replace($old_item_id, $new_item_id, $item_title);
    
        return $item_title;
        
    }

    /**
     * Parses trigger label replacements
     *
     * @since 1.0.0
     *
     * @param array     $replacements   The replacements to apply
     * @param stdClass  $object         The trigger object
     * @param string    $item_type      The item type (trigger|action)
     * @param string    $label          The label to parse
     * @param string    $context        The context this function is executed
     *
     * @return array
     */
    public function item_label_replacements( $replacements, $object, $item_type, $label, $context ) {

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $replacements;
        }

        // Add the youtube replacement
        $replacements['{youtube}'] = '[automatorwp_youtube trigger="' . $object->id . '"]';

        // On edit context, prevent to grab
        if( $context === 'edit' ) {
            $replacements['{youtube}'] = '<span class="automatorwp-no-grab">' . $replacements['{youtube}'] . '</span>';
        }

        return $replacements;

    }

    /**
     * Custom html on automations UI
     *
     * @since 1.0.0
     *
     * @param stdClass $object The trigger/action object
     */
    public function after_item_label( $object ) {

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return;
        }

        $attributes = array(
            'trigger'   => __( 'The trigger identifier. Leave the same value as provided by the trigger.', 'automatorwp-pro' ),
            'url'       => __( 'The Youtube video url or ID.', 'automatorwp-pro' ),
            'width'     => __( 'The player width (in pixels). By default: 640.', 'automatorwp-pro' ),
            'height'    => __( 'The player height (in pixels). By default: 360.', 'automatorwp-pro' ),
            'autoplay'  => __( 'Automatically play the video. Accepts: "yes" or "no". By default: no.', 'automatorwp-pro' ),
            'controls'  => __( 'Show player controls. Accepts: "yes" or "no". By default: yes.', 'automatorwp-pro' ),
        );

        ?>
        <a href="#" class="automatorwp-no-grab" style="margin-top: 5px; cursor: pointer;"
           onclick="jQuery(this).next().slideToggle('fast'); return false;"
        ><?php _e( 'Shortcode attributes', 'automatorwp-pro' ); ?></a>
        <div class="automatorwp-youtube-shortcode-attributes automatorwp-no-grab" style="display: none;">
            <ul>
                <?php foreach( $attributes as $attr => $desc ) : ?>
                    <li>
                        <strong><?php echo $attr; ?></strong> - <span style="color: #757575;"><?php echo $desc; ?></span>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>

        <?php

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $trigger_id     The trigger ID
     * @param string    $video_id       The video ID
     * @param int       $user_id        The user ID
     * @param int       $post_id        The post ID
     * @param int       $seconds        Seconds watched
     * @param int       $duration       The video duration
     */
    public function listener( $trigger_id, $video_id, $user_id, $post_id, $seconds, $duration ) {

        if( $user_id === 0 ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'trigger_id'    => $trigger_id,
            'post_id'       => $post_id,
            'video_id'      => $video_id,
            'seconds'       => $seconds,
            'duration'      => $duration,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if automation is not received
        if( ! isset( $event['trigger_id'] ) ) {
            return false;
        }

        $trigger_id = absint( $event['trigger_id'] );

        // Don't deserve if trigger doesn't match with the trigger option
        if( absint( $trigger->id ) !== $trigger_id ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_Youtube_Watch_Video();