<?php
/**
 * Get Application
 *
 * @package     AutomatorWP\Integrations\WP_Job_Manager\Triggers\Get_Application
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WP_Job_Manager_Get_Application extends AutomatorWP_Integration_Trigger {

    public $integration = 'wp_job_manager';
    public $trigger = 'wp_job_manager_get_application';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User receives an application to a job of a type', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>receives an application</strong> to a job of a type', 'automatorwp-pro' ),
            /* translators: %1$s: Term title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User receives an application to a job of %1$s %2$s time(s)', 'automatorwp-pro' ), '{term}', '{times}' ),
            /* translators: %1$s: Term title. */
            'log_label'         => sprintf( __( 'User receives an application to a job of %1$s', 'automatorwp-pro' ), '{term}' ),
            'action'            => 'new_job_application',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'term' => automatorwp_utilities_term_option( array(
                    'name'              => __( 'Type:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any type', 'automatorwp-pro' ),
                    'taxonomy'          => 'job_listing_type',
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags( __( 'Job', 'automatorwp-pro' ) ),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $application_id
     * @param int $job_id
     */
    public function listener( $application_id, $job_id ) {

        $post = get_post( $job_id );
        $user_id = $post->post_author;
        $terms_ids = automatorwp_get_term_ids( $job_id, 'job_listing_type' );

        automatorwp_trigger_event( array(
            'trigger'           => $this->trigger,
            'user_id'           => $user_id,
            'post_id'           => $job_id,
            'application_id'    => $application_id,
            'terms_ids'         => $terms_ids,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post and terms IDs are not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['terms_ids'] ) ) {
            return false;
        }

        // Don't deserve if term doesn't match with the trigger option
        if( ! automatorwp_terms_matches( $event['terms_ids'], $trigger_options['term'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_WP_Job_Manager_Get_Application();