<?php
/**
 * User Is In Level
 *
 * @package     AutomatorWP\Integrations\WishList_Member\Filters\User_Is_In_Level
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WishList_Member_User_Is_In_Level_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'wishlist_member';
    public $filter = 'wishlist_member_user_is_in_level';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User is in level', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>is in level</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Level. */
            'edit_label'        => sprintf( __( '%1$s', 'automatorwp-pro' ), '{level}'  ),
            /* translators: %1$s: Level. */
            'log_label'         => sprintf( __( '%1$s', 'automatorwp-pro' ), '{level}' ),
            'options'           => array(
                'level' => array(
                    'from' => 'level',
                    'fields' => array(
                        'level' => array(
                            'name' => __( 'Level:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => 'automatorwp_wishlist_member_levels_options_cb',
                            'default' => 'any'
                        )
                    )
                ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        global $WishListMemberInstance;

        // Shorthand
        $level_option = $filter_options['level'];
        $user_levels = wlmapi_get_member_levels( $user_id );

        // Bail if wrong configured
        if( empty( $level_option ) ) {
            $this->result = __( 'Filter not passed. Level option has not been configured.', 'automatorwp-pro' );
            return false;
        }

        $level_data = wlmapi_get_level( $level_option );

        if ($level_option !== 'any') {
            $level_name = $level_data['level']['name'];
        } else{
            $level_name = 'any level';
        }

        if ( $level_option === 'any' ) {
            // Don't deserve if user is not in level
            if( empty( $user_levels ) ) {
                $this->result = sprintf( __( 'Filter not passed. User is not in level "%1$s".', 'automatorwp-pro' ),
                    $level_name
                );
                return false;
            }
        } else{
            // Don't deserve if user is not in this level
            if( ! key_exists( $level_option, $user_levels ) ) {
           
                $this->result = sprintf( __( 'Filter not passed. User is not in level "%1$s".', 'automatorwp-pro' ),
                    $level_name
                );
                return false;
            }
            
        }

        $this->result = sprintf( __( 'Filter passed. User is in level "%1$s".', 'automatorwp-pro' ),
            $level_name
        );

        return $deserves_filter;
    }

}

new AutomatorWP_WishList_Member_User_Is_In_Level_Filter();