<?php
/**
 * Functions
 *
 * @package     AutomatorWP\WhatsApp\Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

function automatorwp_whatsapp_get_url() {
    
    return 'https://graph.facebook.com/v23.0/';

}

/**
 * Helper function to get the WhatsApp API parameters
 *
 * @since 1.0.0
 *
 * @return array|false
 */
function automatorwp_whatsapp_get_api() {

    $url = automatorwp_whatsapp_get_url();
    $token = automatorwp_whatsapp_get_option( 'token', '' );
    $phone_id = automatorwp_whatsapp_get_option( 'phone_id', '' );
    $business_id = automatorwp_whatsapp_get_option( 'business_id', '' );


    if( empty( $token ) ) {
        return false;
    }

    return array(
        'url' => $url,
        'token' => $token,
        'phone_id' => $phone_id,
        'business_id' => $business_id
    );

}

/**
 * Get WhatsApp webhook URL.
 *
 * @since 1.0.0
 *
 * @return string
 */
function automatorwp_whatsapp_get_webhook_url() {

    $prefix = 'automatorwp_whatsapp_';
    $settings = get_option( 'automatorwp_settings' );

    $webhook_url = ( isset( $settings[$prefix . 'webhook'] ) ) ? $settings[$prefix . 'webhook'] : '';

    // Check if webhook URL exists
    if ( empty( $webhook_url ) ){

        $slug = strtolower( wp_generate_password( 8, false ) );
        $webhook_url = get_rest_url() . 'whatsapp/webhooks/' . $slug;

        $settings[$prefix . 'webhook'] = $webhook_url;
        $settings[$prefix . 'slug'] = $slug;

        // Update settings
        update_option( 'automatorwp_settings', $settings );

    }

    return $webhook_url;

}

/**
 * Get WhatsApp webhook slug.
 *
 * @since 1.0.0
 *
 * @return string
 */

function automatorwp_whatsapp_get_webhook_slug() {

    $prefix = 'automatorwp_whatsapp_';
    $settings = get_option( 'automatorwp_settings' );

    $slug = $settings[$prefix . 'slug'];

    // Check if webhook slug exists
    if ( empty( $slug ) ){

        $slug = strtolower( wp_generate_password( 8, false ) );
        $webhook_url = get_rest_url() . 'whatsapp/webhooks/' . $slug;

        $settings[$prefix . 'webhook'] = $webhook_url;
        $settings[$prefix . 'slug'] = $slug;

        // Update settings
        update_option( 'automatorwp_settings', $settings );

    }

    return $slug;

}

/**
 * Warn about the WhatsApp webhook.
 *
 * @since 1.0.0
 *
 * 
 */
function automatorwp_whatsapp_trigger_notice() {
    $webhook_url = automatorwp_whatsapp_get_webhook_url();

    // Warn user about copy the webhook url at WhatsApp
    ?>
    <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
        <?php echo sprintf(
            __( 'It is required to configure a webhook in your WhatsApp Business account to make this trigger work.', 'automatorwp' ),
            admin_url( 'admin.php?page=automatorwp_settings&tab=opt-tab-whatsapp' )
        ); ?>
        <?php echo sprintf(
            __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp' ),
            'https://automatorwp.com/docs/whatsapp/'
        ); ?> |
        <?php echo __( '<a href="#" class="automatorwp-view-webhook">View Webhook</a>', 'automatorwp' ); ?>
    </div>

    <div class="automatorwp-option-form-container" style="display:none">
        <div class="cmb2-wrap form-table automatorwp-form automatorwp-option-form">
            <div id="cmb2-metabox-webhook_form" class="cmb2-metabox cmb-field-list">
                <div class="cmb-row cmb-type-text cmb2-id-url table-layout automatorwp-webhooks-url" data-fieldtype="text">
                    <div class="cmb-th">
                        <label for="url"><?php echo __('URL:', 'automatorwp'); ?></label>
                    </div>
                    <div class="cmb-td">
                        <input type="text" class="regular-text" name="" id="url" value=<?php echo $webhook_url; ?> readonly data-option="url">
                        <p class="cmb2-metabox-description"><?php echo __('Copy this URL in your WhatsApp Business account.', 'automatorwp'); ?></p>
                    </div>
                </div>
            </div>
        </div>
        <button type="button" class="button button-primary automatorwp-cancel-option-form"><?php echo __('Hide Webhook', 'automatorwp'); ?></button>
    </div>

    <?php
}

/**
 * Send message
 *
 * @since 1.0.0
 * 
 * @param string    $phone                    Phone number
 * @param string    $message_content          The content of the message
 */
function automatorwp_whatsapp_send_message( $phone, $message_content ) {
    
    $api = automatorwp_whatsapp_get_api();
    
    if( ! $api ) {
        return;
    }
        
    $body = array(
        "messaging_product" => "whatsapp",
        "recipient_type" => "individual",
        "to" => $phone,
        "type" => "text",
        "text" => array(
            "preview_url" => true,
            "body" => $message_content,
        )
    );
    
    $response = wp_remote_post( $api['url'] . $api['phone_id'] . '/messages', array(
        'headers' => array(
            'Authorization' => 'Bearer ' . $api["token"],
            'Content-Type'  => 'application/json'
        ),
        'body'    => json_encode( $body ),
    ));
 
    return $response;

}

/**
 * Send message template
 *
 * @since 1.0.0
 * 
 * @param string    $phone    Phone number
 * @param string    $template The template name for the message
 */
function automatorwp_whatsapp_send_message_template( $phone, $template ) {
    
    $api = automatorwp_whatsapp_get_api();
    $template_lang = automatorwp_whatsapp_get_template_language( $template );

    if( ! $api ) {
        return;
    }
        
    $body = array(
        "messaging_product" => "whatsapp",
        "to" => $phone,
        "type" => "template",
        "template" => array(
            "name" => $template,
            "language" => array(
                "code" => $template_lang
            )
        )
    );
    
    $response = wp_remote_post( $api['url'] . $api['phone_id'] . '/messages', array(
        'headers' => array(
            'Authorization' => 'Bearer ' . $api["token"],
            'Content-Type'  => 'application/json'
        ),
        'body'    => json_encode( $body ),
    ));
    
    return $response;
}



/**
 * Get templates from WhatsApp
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_whatsapp_get_templates( ) {

    $api = automatorwp_whatsapp_get_api();
    $templates = array();

    if( ! $api ) {
        return $templates;
    }

    $response = wp_remote_get( $api["url"].$api["business_id"].'/message_templates', array(
        'headers' => array(
            'Authorization' => 'Bearer ' . $api["token"],
            'Accept' => 'application/json',
            'Content-Type'  => 'application/json'
        )
    ) );
    
    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
    
    // Bail if not response
    if ( !isset ( $response["data"] ) ) {
        return $templates;
    }

    foreach ( $response["data"] as $template ){

        $templates[] = array(
            'name'  => $template['name'],
            'language'  => $template['language'],
        );
        
    }

    return $templates;

}

/**
 * Get template from WhatsApp
 *
 * @since 1.0.0
 * 
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_whatsapp_options_cb_template( $field ) {
    
    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any template', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );
    
    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }
    
        foreach( $value as $template ) {
            
            // Skip option none
            if( $template === $none_value ) {
                continue;
            }

            $options[$template] = automatorwp_whatsapp_get_template_name( $template );
        }
    }

    return $options;
}


/**
* Get the template name
*
* @since 1.0.0
* 
* @param string $template
*
* @return array
*/
function automatorwp_whatsapp_get_template_name( $template_s ) {

    if( empty($template_s) ) {
        return '';
    }

    $templates = automatorwp_whatsapp_get_templates();
   
    $template_name = '';

    foreach( $templates as $template ) {

        if( $template['name'] === $template_s  ) {
            $template_name = $template['name'];
            break;
        }
    }

    return $template_name;

}


/**
* Get the template language
*
* @since 1.0.0
* 
* @param string $template
*
* @return array
*/
function automatorwp_whatsapp_get_template_language( $template_s ) {

    if( empty( $template_s ) ) {
        return '';
    }

    $templates = automatorwp_whatsapp_get_templates();
   
    $template_lang = '';

    foreach( $templates as $template ) {

        if( $template['name'] === $template_s  ) {
            $template_lang = $template['language'];
            break;
        }
    }

    return $template_lang;

}

/**
 * Sanitize phone number
 * 
 * @param string $prefix  The country code
 * @param string $phone_number  The phone number
 *
 * @since 1.0.0
 *
 * @return string|false|WP_Error
 */
function automatorwp_whatsapp_validate_phone_number( $prefix, $phone_number ) {

    // Filter the Numbers from String
    $prefix = preg_replace('/[^0-9]/', '', $prefix);
    $phone = preg_replace('/[^0-9]/', '', $phone_number);

    return $prefix . $phone;

}