<?php
/**
 * Send Tweet
 *
 * @package     AutomatorWP\Integrations\Twitter\Actions\Send_Tweet
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined('ABSPATH')) exit;

class AutomatorWP_Twitter_Send_Tweet extends AutomatorWP_Integration_Action {

    public $integration = 'twitter';
    public $action = 'twitter_send_tweet';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register(){
            
        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Post a tweet', 'automatorwp-pro'),
            'select_option'     => __( 'Post a <strong>tweet</strong>', 'automatorwp-pro'),
            /* translators: %1$s: Tweet. */
            'edit_label'        => sprintf( __( 'Post %1$s', 'automatorwp-pro' ), '{tweet}' ),
            /* translators: %1$s: Tweet. */
            'log_label'         => sprintf( __( 'Post %1$s', 'automatorwp-pro' ) , '{tweet}' ),
            'options'           => array(
                'tweet' => array(
                    'default'   => __( 'a tweet','automatorwp-pro' ),
                    'fields'    => array(
                        'tweet' => array(
                            'name'              => __( 'Tweet:', 'automatorwp-pro' ),
                            'desc'              => '<strong style="font-size: 14px;"><span class="automatorwp-twitter-char-count">0</span>/280</strong>',
                            'type'              => 'textarea',
                            'required'          => true,
                            'attributes'        => array(
                                'rows' => '5'
                            ),
                        ),
                    ),
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation){
        
        $this->result = '';
        $this->tweet = '';

        // Bail if empty tweet to send
        if( empty( $action_options['tweet'] ) ) {
            $this->result = __( 'There is not tweet to send', 'automatorwp-pro' );
            return;
        }

        $tweet = wpautop( $action_options['tweet'] );
		$tags_to_clean = array( '<p>', '</p>', '<br />', '<br/>', '<br>', '<b>', '</b>', '<i>', '</i>' );
		$tweet = str_replace( $tags_to_clean, '', $tweet);

        // Get credentials
        $consumer_key = automatorwp_twitter_get_option( 'consumer_key' );
        $consumer_secret = automatorwp_twitter_get_option( 'consumer_secret' );
        $access_token = automatorwp_twitter_get_option( 'access_token' );
        $access_token_secret = automatorwp_twitter_get_option( 'access_token_secret' );
        $access_valid = automatorwp_twitter_get_option( 'access_valid' );

        if( $access_valid && $consumer_key !== '' && $consumer_secret !== '' && $access_token !== '' && $access_token_secret !== '' ) {

            $oauth_header = automatorwp_twitter_get_oauth_header( $consumer_key, $consumer_secret, $access_token, $access_token_secret );
            $response = automatorwp_twitter_send_tweet( $oauth_header, $tweet );

            $this->result = sprintf( __( '%s', 'automatorwp-pro' ), $response );
        
        }

    }
    
    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );
        
        parent::hooks();
    }
    
    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        // Get credentials
        $consumer_key = automatorwp_twitter_get_option( 'consumer_key' );
        $consumer_secret = automatorwp_twitter_get_option( 'consumer_secret' );
        $access_token = automatorwp_twitter_get_option( 'access_token' );
        $access_token_secret = automatorwp_twitter_get_option( 'access_token_secret' );
        $access_valid = automatorwp_twitter_get_option( 'access_valid' );


        // Warn user if the authorization has not been setup from settings
        if(!$access_valid || $consumer_key == '' || $consumer_secret == '' || $access_token == '' || $access_token_secret == '') : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                        __( 'You need to configure the <a href="%s" target="_blank">Twitter settings</a> to get this action to work.', 'automatorwp-pro' ),
                        get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-twitter'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/twitter/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['tweet'] = $this->tweet;
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

   /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['tweet'] = array(
            'name' => __( 'Tweet:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }
}

new AutomatorWP_Twitter_Send_Tweet();