<?php
/**
 * Send SMS
 *
 * @package     AutomatorWP\Integrations\Twilio\Actions\Send_SMS
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Twilio_Send_SMS extends AutomatorWP_Integration_Action {

    public $integration = 'twilio';
    public $action = 'twilio_send_sms';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Send SMS to phone number(s)', 'automatorwp-pro' ),
            'select_option'     => __( 'Send <strong>SMS</strong> to phone number(s)', 'automatorwp-pro' ),
            /* translators: %1$s: Contact. */
            'edit_label'        => sprintf( __( 'Send %1$s to %2$s', 'automatorwp-pro' ), '{sms}', '{phone_number}' ),
            /* translators: %1$s: Contact. */
            'log_label'         => sprintf( __( 'Send %1$s to %2$s', 'automatorwp-pro' ), '{sms}', '{phone_number}' ),
            'options'           => array(
                'sms' => array(
                    'default' => __( 'SMS', 'automatorwp-pro' ),
                    'fields' => array(
                        'sms' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'The SMS content.', 'automatorwp-pro' ),
                            'type' => 'textarea',
                        ),
                    )
                ),
                'phone_number' => array(
                    'from' => 'phone_number',
                    'default' => __( 'phone number(s)', 'automatorwp-pro' ),
                    'fields' => array(
                        'phone_number' => array(
                            'name' => __( 'Phone Number(s):', 'automatorwp-pro' ),
                            'desc' => __( 'Single or comma-separated list of phone numbers that will receive this SMS.', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $sms = $action_options['sms'];
        $phone_number = $action_options['phone_number'];

        $phone_numbers = explode( ',', $phone_number );

        $this->result = '';

        foreach( $phone_numbers as $phone_number ) {

            // Send the SMS
            $result = automatorwp_twilio_send_sms( $phone_number, $sms );

            // Store result per phone number
            $this->result .= $phone_number . ': ' . $result['message'] . '<br>';

        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Twilio_Send_SMS();